unit LCDScreen;

////////////////////////////////////////////////////////////////////////////////
//
//  TLCDScreen / TLCDAnimator v3.0 (22/jan/02)
//  written by Jacques VOIRIN
//  E-Mail: jacques.voirin@iname.com
//
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//
//   This is the main unit for TLCDScreen and TLCDAnimator components.
//
////////////////////////////////////////////////////////////////////////////////


interface

{$R LCDScreen.res}

uses
  Windows, Classes, Graphics, Controls, Dialogs, ExtCtrls, Math, SysUtils,
  Forms;


const MaxSizeCodeInstructionArray = 255; //Removing dynamic Arrays...

type
  TLCDAnimator = class;

  TCodeInstruction = record
                       Word: string[25];
                       Param: Integer;
                       end;

  CodeInstructionArrayBuilder = array[1..MaxSizeCodeInstructionArray] of TCodeInstruction; //Removing dynamic Arrays...

  TEndCodeEvent = procedure(Sender: TLCDAnimator) of object;

  TResetMode = (rmDisplay, rmCode, rmDisplayAndCode);

  TLineExecutedEvent = procedure(Sender: TLCDAnimator; LineNumber: Word) of object;

  TPixelSize = (pixCustom, pix1x1, pix2x2, pix3x3, pix4x4, pix5x5, pix6x6,
                pix7x7, pix8x8, pix9x9, pix10x10, pix11x11, pix12x12, pix13x13,
                pix14x14, pix15x15, pix16x16);

  TPixelShape = (psSquare, psRound);

  TLCDBorder = (bsLowered, bsNone, bsRaised);

  TSpecialEffect = (spBlink, spBold, spInverse, spItalic, spUnderline, spStrike);

  TSpecialEffects = set of TSpecialEffect;

  TOneChar = record
                TheChar:Char;
                SpEff: Byte;
                end;

  TAnimMode = (amDynamic, amStatic);

  TDisplayMode = (dmText, dmBitmap, dmBoth);

  TBitmapCopyMode = (cmNotTransparent, cmTransparent);


////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimator
//
////////////////////////////////////////////////////////////////////////////////

  TLCDAnimator = class(TComponent)
  private
    FCode: TStrings;
    FCodeErrorFound: Boolean;
    FCurrentLine: SmallInt;
    FOnEndCode: TEndCodeEvent;
    FOnLineExecuted: TLineExecutedEvent;
    function  GetAbout: string;
    procedure SetAbout(Value: string);
    procedure SetCurrentLine(Value: SmallInt);

  protected
    procedure SetCode(Value: TStrings);

  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;

  published
    property About: string read GetAbout write SetAbout stored False;
    property Code: TStrings read FCode write SetCode;
    property CodeErrorFound: Boolean read FCodeErrorFound write FCodeErrorFound;
    property CurrentLine: SmallInt read FCurrentLine write SetCurrentLine;
    property OnEndCode: TEndCodeEvent read FOnEndCode write FOnEndCode;
    property OnLineExecuted: TLineExecutedEvent read FOnLineExecuted write FOnLineExecuted;
  end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDScreen
//
////////////////////////////////////////////////////////////////////////////////

  TLCDScreen = class(TGraphicControl)
  private
    P: ^CodeInstructionArrayBuilder;      //Removing dynamic Arrays...
    Display: array [0..255,0..255] of TOneChar; { Internal image of the screen        }
    PixHRef, PixVRef: Integer;            { Internal offset in Pixel unit used for animation routines }
    FNbXPixels, FNbYPixels: Word;         { Internal size in dots of the component    }
    FLinesImage: TBitmap;                 { Internal black&white image of Flines      }
    FWidth, FHeight: Integer;             { Internal height and width of the control  }
    FTimer: TTimer;                       { Internal Timer used                       }
    FTrueOnColor : TColor;                { Internal color for OnPixels               }
    FTrueOffColor: TColor;                { Internal color for OffPixels              }
    FBlinkingStatus: Boolean;             { Internal Blinking On/Off flag             }
    FAnimationDelay: Cardinal;            { In milliseconds; the smaller, the faster  }
    FAnimationEnabled: Boolean;           { Enable or disable the animation           }
    FAnimationRepeating: Boolean;         { What to do when all code lines had been processed }
    FBitmap: TBitmap;                     { The component's bitmap                    }
    FBitmapCopyMode: TBitmapCopyMode;     { Set bitmap transparent or not             }
    FBitmapAnimMode: TAnimMode;           { Bitmap is static or dynamic               }
    FBitmapXOffset: ShortInt;             { Value in Pixel for bitmap horz correction }
    FBitmapYOffset: ShortInt;             { Value in Pixel for bitmap vertical correction }
    FBorderSpace: ShortInt;               { Distance from the LCD border              }
    FBorderStyle: TLCDBorder;             { Border around the the component           }
    FColor: TColor;                       { LCD color                                 }
    FDisplayMode: TDisplayMode;           { Text only, Bitmap only or both displayed  }
    FFont: TFont;                         { The Font used in TLCDScreen               }
    FIntensity: ShortInt;                 { [-128..127] percentage of ligth           }
    FLCDAnimator: TLCDAnimator;           { The TLCDAnimator associated with          }
    FLines: TStringList;                  { The component's multilines text           }
    FLinesAnimMode: TAnimMode;            { Lines are static or dynamic               }
    FLinesXOffset: ShortInt;              { Value in Pixel for lines horz  correction }
    FLinesYOffset: ShortInt;              { Value in Pixel for lines vertical correction }
    FPixelHeight: ShortInt;               { Pixel height                              }
    FPixelOffColor: TColor;               { LCD pixel OFF color                       }
    FPixelSize: TPixelSize;               { Size of a LCD pixel (in screen pixels)    }
    FPixelShape: TPixelShape;             { Shape of a LCD pixel                      }
    FPixelSpacing: ShortInt;              { Space between each pixel in the matrix    }
    FPixelWidth: ShortInt;                { Pixel widht                               }
    FSpecialEffects: TSpecialEffects;     { Enable or disable special effects         }

    function  GetAbout: string;
    procedure SetAbout(Value: string);
    procedure SetAnimationEnabled(Value: Boolean);
    procedure SetAnimationRepeating(Value: Boolean);
    procedure SetAnimationDelay(Value: Cardinal);
    procedure SetBitmap(Value: TBitmap);
    procedure SetBitmapAnimMode(Value: TAnimMode);
    procedure SetBorderSpace(Value: ShortInt);
    procedure SetBorderStyle(Value: TLCDBorder);
    procedure SetBitmapCopyMode(Value: TBitmapCopyMode);
    procedure SetBitmapXOffset(Value: ShortInt);
    procedure SetBitmapYOffset(Value: ShortInt);
    procedure SetColor(Value: TColor);
    procedure SetDisplayMode(Value: TDisplayMode);
    procedure SetFont(Value: TFont);
    procedure SetIntensity(Value: ShortInt);
    procedure SetLCDAnimator(Value: TLCDAnimator);
    procedure SetLinesAnimMode(Value: TAnimMode);
    procedure SetLinesXOffset(Value: ShortInt);
    procedure SetLinesYOffset(Value: ShortInt);
    procedure SetPixelSize(Value: TPixelSize);
    procedure SetPixelShape(Value: TPixelShape);
    procedure SetPixelSpacing(Value: ShortInt);
    procedure SetPixelOffColor(Value: TColor);
    procedure SetPixelWidth(Value: ShortInt);
    procedure SetPixelHeight(Value: ShortInt);
    procedure SetSpecialEffects(Value: TSpecialEffects);

    procedure DrawDisplayCharacters(BitMap: TBitmap);
    procedure FontOnChange(Sender: TObject);
    procedure SetCorrectSize;
    procedure UpdateFLinesImage;
    procedure UpdateTrueColors;

  protected
    procedure HorzScroll(Value: ShortInt);
    procedure LinesOnChange(Sender: TObject);
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure Paint; override;
    procedure SetLines(Value: TStringList);
    procedure TimerOnTimer(Sender: TObject);
    procedure VertScroll(Value: ShortInt);


  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    procedure   Reset(Value: TResetMode);
    property    Canvas;

  published
    property About: string read GetAbout write SetAbout stored False;
    property AnimationDelay: Cardinal read FAnimationDelay write SetAnimationDelay;
    property AnimationEnabled: Boolean read FAnimationEnabled write SetAnimationEnabled;
    property AnimationRepeating: Boolean read FAnimationRepeating write SetAnimationRepeating;
    property Bitmap: TBitmap read FBitmap write SetBitmap;
    property BitmapCopyMode: TBitmapCopyMode read FBitmapCopyMode write SetBitmapCopyMode;
    property BitmapAnimMode: TAnimMode read FBitmapAnimMode write SetBitmapAnimMode;
    property BitmapXOffset: ShortInt read FBitmapXOffset write SetBitmapXOffset;
    property BitmapYOffset: ShortInt read FBitmapYOffset write SetBitmapYOffset;
    property BorderSpace: ShortInt read FBorderSpace write SetBorderSpace;
    property BorderStyle: TLCDBorder read FBorderStyle write SetBorderStyle;
    property Color: TColor read FColor write SetColor;
    property DisplayMode: TDisplayMode read FDisplayMode write SetDisplayMode;
    property Enabled;
    property Font: TFont read FFont write SetFont;
    property Intensity: ShortInt read FIntensity write SetIntensity;
    property LCDAnimator: TLCDAnimator read FLCDAnimator write SetLCDAnimator;
    property Lines: TStringList read FLines write SetLines;
    property LinesAnimMode: TAnimMode read FLinesAnimMode write SetLinesAnimMode;
    property LinesXOffset: ShortInt read FLinesXOffset write SetLinesXOffset;
    property LinesYOffset: ShortInt read FLinesYOffset write SetLinesYOffset;
    property PixelHeight: ShortInt read FPixelHeight write SetPixelHeight;
    property PixelOff: TColor read FPixelOffColor write SetPixelOffColor;
    property PixelShape: TPixelShape read FPixelShape write SetPixelShape;
    property PixelSize: TPixelSize read FPixelSize write SetPixelSize;
    property PixelSpacing: ShortInt read FPixelSpacing write SetPixelSpacing;
    property PixelWidth: ShortInt read FPixelWidth write SetPixelWidth;
    property ShowHint;
    property SpecialEffects: TSpecialEffects read FSpecialEffects write SetSpecialEffects;
    property Visible;

    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

const
  startinverse_tag   = '<inv>';
  stopinverse_tag    = '</inv>';
  startblinking_tag  = '<bl>';
  stopblinking_tag   = '</bl>';
  startunderline_tag = '<u>';
  stopunderline_tag  = '</u>';
  startstrike_tag    = '<s>';
  stopstrike_tag     = '</s>';
  startbold_tag      = '<b>';
  stopbold_tag       = '</b>';
  startitalic_tag    = '<i>';
  stopitalic_tag     = '</i>';

  LCDTag: array[0..11]of string = ( startinverse_tag,   stopinverse_tag,
                                    startblinking_tag,  stopblinking_tag,
                                    startunderline_tag, stopunderline_tag,
                                    startstrike_tag,    stopstrike_tag,
                                    startbold_tag,      stopbold_tag,
                                    startitalic_tag,    stopitalic_tag);

function CountLCDTag(str: String): Byte;
function NbOfThings(tempcode: string; Thing: Char): Integer;
function RemoveLCDTag(str: String): String;

procedure Register;

implementation

////////////////////////////////////////////////////////////////////////////////
//
//  General Intrest Routines. Used for CodeEditor synthax analysis.
//
////////////////////////////////////////////////////////////////////////////////

function NbOfThings(tempcode: string; Thing: Char): Integer;
var i: Integer;
begin
  i := 0;
  while Pos(Thing, tempcode) <> 0
  do begin
       try Delete(tempcode, 1, Pos(Thing, tempcode)); except; end;
       Inc(i);
       end;
  NbOfThings := i;
end;


////////////////////////////////////////////////////////////////////////////////
//
// General Intrest Routines. Count number of LCDTags in a String.
//
////////////////////////////////////////////////////////////////////////////////

function CountLCDTag(str: String): Byte;
var i, col, nbtag: Byte;
begin
  nbtag := 0;
  if Length(str) <> 0
  then for i := 0 to 11
       do for col := 0 to Length(str) - 3
          do if Copy(str, col, Length(LCDTag[i])) = LCDTag[i]
             then Inc(nbtag);

  CountLCDTag := nbtag;
end;


////////////////////////////////////////////////////////////////////////////////
//
// General Intrest Routines. Remove LCDTags from a String.
//
////////////////////////////////////////////////////////////////////////////////

function RemoveLCDTag(str: String): String;
var i: Byte;
begin
  if Length(str) <> 0
  then for i := 0 to 11
       do str := StringReplace(str, LCDTag[i], '', [rfReplaceAll, rfIgnoreCase]);

  RemoveLCDTag := str;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Create and initialize component TLCDScreen.
//
////////////////////////////////////////////////////////////////////////////////

constructor TLCDScreen.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csOpaque] - [csAcceptsControls, csSetCaption];

  FLines := TStringList.Create;

  FFont := TFont.Create;

  FLinesImage := TBitmap.Create;
  FLinesImage.PixelFormat := pf1bit;

  FPixelSize := pix2x2;
  FPixelSpacing := 1;
  FBorderSpace := 3;
  Width := 145;
  Height := 65;
  FBorderStyle := bsLowered;
  FColor := clInactiveBorder;
  FPixelOffColor := $00AAAAAA;
  FIntensity := 127;
  UpdateTrueColors;
  PixVRef := 0;
  PixHRef :=0;
  FLinesXOffset := 0;
  FLinesYOffset := 0;
  FBitmapXOffset := 0;
  FBitmapYOffset := 0;

  FTimer := TTimer.Create(Self);
  FTimer.Interval := 250;
  FTimer.OnTimer := TimerOnTimer;
  FTimer.Enabled := False;

  FAnimationDelay := 251;
  FAnimationEnabled := False;
  FAnimationRepeating := False;

  FLines.OnChange := LinesOnChange;
  FFont.OnChange := FontOnChange;

  FBitmap := TBitmap.Create;
  FDisplayMode := dmText;

  SetCorrectSize;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Remove component TLCDScreen.
//
////////////////////////////////////////////////////////////////////////////////

destructor TLCDScreen.Destroy;
begin
  FBitmap.Destroy;
  FLinesImage.Destroy;
  FTimer.Destroy;
  FLines.Destroy;
  FFont.Destroy;
  inherited Destroy;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set Width and Height correct.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetCorrectSize;
begin
  if PixelSize <> pixCustom
  then begin  { Predefined width selected - make square pixels. }
         FPixelWidth := Ord(FPixelSize);
         FPixelHeight := FPixelWidth;
         end;

  FNbXPixels := Trunc((Width  - 2 * FBorderSpace + FPixelSpacing) / (FPixelWidth + FPixelSpacing));
  FNbYPixels := Trunc((Height - 2 * FBorderSpace + FPixelSpacing) / (FPixelHeight + FPixelSpacing));

  FWidth  := 2 * FBorderSpace + FNbXPixels * FPixelWidth + (FNbXPixels - 1) * FPixelSpacing;
  FHeight := 2 * FBorderSpace + FNbYPixels * FPixelHeight + (FNbYPixels - 1) * FPixelSpacing;
  Width  := FWidth;
  Height := FHeight;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Repaint the component.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.Paint;
var tempBitMap, tempBitmap2: TBitMap;
begin
  if Visible   { Draw only real graphics if display is marked visible. }
  then begin
         tempBitmap := TBitMap.Create;
         tempBitmap.Height := Height;
         tempBitmap.Width  := Width;

         with tempBitmap.Canvas
         do begin                   { Border drawing on tempBitmap. }
              Brush.Color := FColor;
              FillRect(ClipRect);
              if FBorderStyle <> bsNone
              then begin
                     case FBorderStyle of
                        bsRaised:  Pen.Color := clBtnShadow;
                        bsLowered: Pen.Color := clBtnHighlight;
                       end;
                     PolyLine([Point(Width - 1, 0), Point(Width - 1, Height - 1), Point(-1, Height - 1)]);

                     case FBorderStyle of
                        bsRaised:  Pen.Color := clBtnHighlight;
                        bsLowered: Pen.Color := clBtnShadow;
                        end;
                     PolyLine([Point(Width - 1, 0), Point(0, 0), Point(0, Height - 1)]);
                     end;
              end;

         if Enabled
         then begin
                tempBitmap2 := TBitMap.Create;
                tempBitmap2.Height := Height - 2 * FBorderSpace;
                tempBitmap2.Width  := Width  - 2 * FBorderSpace;
                with tempBitmap2.Canvas
                do begin
                     Brush.Color := FColor;
                     FillRect(ClipRect);
                     end;

                { Characters drawing on tempBitmap2.
                  The use of bitmap2 is justified to avoid flickering. }

                 DrawDisplayCharacters(tempBitmap2);

                { Copy characters from tempBitmap2 to tempBitmap bitmap. }

                BitBlt(tempBitmap.Canvas.Handle,
                       FBorderSpace, FBorderSpace,
                       Width - 2 * FBorderSpace, Height - 2 * FBorderSpace,
                       tempBitMap2.Canvas.Handle, 0, 0, srcCopy);

                tempBitmap2.Free;
                end;

         { Copy characters + border from tempBitmap to Window bitmap. }

          BitBlt(Canvas.Handle, 0, 0, Width, Height, tempBitMap.Canvas.Handle, 0, 0, srcCopy);

         tempBitMap.Free;
         end;
end;



////////////////////////////////////////////////////////////////////////////////
//
// Drawing routine using FLinesImage and/or FBitmap.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.DrawDisplayCharacters(BitMap: TBitMap);
var
  i, j, z, lx, ly: Integer;
  w1, w2, maxwidth, maxheight: Integer;
  onbitmap, offbitmap, offlinebitmap, displaybitmap: TBitMap;
  PByteRow: PByteArray;
  EightPixels: Byte;
  PixRect: TRect;

  procedure DrawOnePixel(var b: TBitmap; PixelShape: TPixelShape;
                       Color1, Color2: TColor);
  begin
    with b.Canvas
    do begin
         Brush.Color := Color2;
         FillRect(Rect(0,0,b.Width, b.Height));

         if (b.Height = 1) and (b.Width = 1) then Pixels[0, 0] := Color1
         else begin
                Pen.Color := Color1;
                Brush.Color := Color1;
                case PixelShape of
                  psSquare:  Rectangle(b.Canvas.ClipRect); { Standard square pixels}
                  psRound:   Ellipse(b.Canvas.ClipRect);   { Round pixels }
                  end;
                end;
         end;
  end;


begin
  SetCorrectSize;

  displaybitmap := TBitmap.Create;
  displaybitmap.Width := FNbXPixels;
  displaybitmap.Height := FNbYPixels;
  displaybitmap.PixelFormat := pf1bit;

  { Create a b&w image of the text and/or bitmap using animation parameters (VRef, Href). }
  case FDisplayMode of
    dmText: begin
              if pixHRef + FLinesXOffset < -FLinesImage.Width  then pixHRef := pixHref + FLinesImage.Width + FNbXPixels;
              if pixHRef + FLinesXOffset > FNbXPixels          then pixHref := pixHref - FLinesImage.Width - FNbXPixels;// - FLinesXOffset;
              if pixVRef + FLinesYOffset < -FLinesImage.Height then pixVRef := pixVref + FLinesImage.Height + FNbYPixels;
              if pixVRef + FLinesYOffset > FNbYPixels          then pixVref := pixVref - FLinesImage.Height - FNbYPixels;// - FLinesXOffset;

              if FLinesAnimMode = amDynamic
              then
              displaybitmap.Canvas.CopyRect(Rect(pixHRef + FLinesXOffset, pixVRef + FLinesYOffset,
                                                 FLinesImage.Width + pixHRef + FLinesXOffset,
                                                 FLinesImage.Height + pixVRef + FLinesYOffset),
                                            FLinesImage.Canvas,
                                            Rect(0, 0, FLinesImage.Width, FLinesImage.Height))
              else
              displaybitmap.Canvas.CopyRect(Rect(FLinesXOffset, FLinesYOffset,
                                                 FLinesImage.Width + FLinesXOffset,
                                                 FLinesImage.Height + FLinesYOffset),
                                            FLinesImage.Canvas,
                                            Rect(0, 0, FLinesImage.Width, FLinesImage.Height))
              end;
  dmBitmap: begin
              if pixHRef + FBitmapXOffset < -FBitmap.Width  then pixHRef := pixHref + FBitmap.Width + FNbXPixels;
              if pixHRef + FBitmapXOffset > FNbXPixels      then pixHref := pixHref - FBitmap.Width - FNbXPixels;// - FBitmapXOffset;
              if pixVRef + FBitmapYOffset < -FBitmap.Height then pixVRef := pixVref + FBitmap.Height + FNbYPixels;
              if pixVRef + FBitmapYOffset > FNbYPixels      then pixVref := pixVref - FBitmap.Height - FNbYPixels;// - FBitmapXOffset;

              if FBitmapAnimMode = amDynamic
              then
              displaybitmap.Canvas.CopyRect(Rect(pixHRef + FBitmapXOffset, pixVRef + FBitmapYOffset,
                                                 FBitmap.Width + pixHRef + FBitmapXOffset,
                                                 FBitmap.Height + pixVRef + FBitmapYOffset),
                                            FBitmap.Canvas,
                                            Rect(0, 0, FBitmap.Width, FBitmap.Height))
              else
              displaybitmap.Canvas.CopyRect(Rect(FBitmapXOffset, FBitmapYOffset,
                                                 FBitmap.Width + FBitmapXOffset,
                                                 FBitmap.Height + FBitmapYOffset),
                                            FBitmap.Canvas,
                                            Rect(0, 0, FBitmap.Width, FBitmap.Height));
              end
       else begin {dmBoth}
              maxwidth := max(FLinesImage.Width, FBitmap.Width);
              maxheight := max(FLinesImage.Height, FBitmap.Height);

              if (pixHRef + FLinesXOffset < -maxwidth) or (pixHRef + FBitmapXOffset < -maxwidth)
              then pixHRef := pixHref + maxwidth + FNbXPixels;
              if (pixHRef + FLinesXOffset > FNbXPixels) or (pixHRef + FBitmapXOffset > FNbXPixels)
              then pixHref := pixHref - maxwidth - FNbXPixels;
              if (pixVRef + FLinesYOffset < -maxheight) or (pixVRef + FBitmapYOffset < -maxheight)
              then pixVRef := pixVref + maxheight + FNbYPixels;
              if (pixVRef + FLinesYOffset > FNbYPixels) or (pixVRef + FBitmapYOffset > FNbYPixels)
              then pixVref := pixVref - maxheight - FNbYPixels;

              if FLinesAnimMode = amDynamic
              then
              displaybitmap.Canvas.CopyRect(Rect(pixHRef + FLinesXOffset, pixVRef + FLinesYOffset,
                                                 FLinesImage.Width + pixHRef + FLinesXOffset,
                                                 FLinesImage.Height + pixVRef + FLinesYOffset),
                                            FLinesImage.Canvas,
                                            Rect(0, 0, FLinesImage.Width, FLinesImage.Height))
              else
              displaybitmap.Canvas.CopyRect(Rect(FLinesXOffset, FLinesYOffset,
                                                 FLinesImage.Width + FLinesXOffset,
                                                 FLinesImage.Height + FLinesYOffset),
                                            FLinesImage.Canvas,
                                            Rect(0, 0, FLinesImage.Width, FLinesImage.Height));

              if FBitmapCopyMode = cmTransparent then displaybitmap.Canvas.CopyMode := cmSrcAnd;

              if FBitmapAnimMode = amDynamic
              then
              displaybitmap.Canvas.CopyRect(Rect(pixHRef + FBitmapXOffset, pixVRef + FBitmapYOffset,
                                                 FBitmap.Width + pixHRef + FBitmapXOffset,
                                                 FBitmap.Height + pixVRef + FBitmapYOffset),
                                            FBitmap.Canvas,
                                            Rect(0, 0, FBitmap.Width, FBitmap.Height))
              else
              displaybitmap.Canvas.CopyRect(Rect(FBitmapXOffset, FBitmapYOffset,
                                                 FBitmap.Width + FBitmapXOffset,
                                                 FBitmap.Height + FBitmapYOffset),
                                            FBitmap.Canvas,
                                            Rect(0, 0, FBitmap.Width, FBitmap.Height));
              end;
       end;

  
  { Create onand offbitmap, representing one on/off pixel.
    offlinebitmap is used for speed optimization: the idea is that, globally,
    there are more off pixels than on ones. }

  PixRect := Rect(0, 0, FPixelWidth, FPixelHeight);

  onbitmap := TBitmap.Create;
  onbitmap.Width :=FPixelWidth;
  onbitmap.Height :=FPixelHeight;
  DrawOnePixel(onbitmap, FPixelShape, FTrueOnColor, FTrueOffColor);

  offbitmap := TBitmap.Create;
  offbitmap.Width :=FPixelWidth;
  offbitmap.Height := FPixelHeight;
  DrawOnePixel(offbitmap, FPixelShape, FTrueOffColor, FTrueOffColor);

  offlinebitmap := TBitmap.Create;
  offlinebitmap.Width :=Bitmap.Width;
  offlinebitmap.Height :=FPixelHeight;
  offlinebitmap.Canvas.Brush.Color := Color;
  offlinebitmap.Canvas.FillRect(offlinebitmap.Canvas.ClipRect);

  offlinebitmap.Canvas.CopyRect(PixRect, offbitmap.Canvas, PixRect);

  w1 := FPixelWidth + FPixelSpacing;
  w2 := 2 * w1 - FPixelSpacing;
  repeat
    offlinebitmap.Canvas.CopyRect(Rect(w1, 0, w2, FPixelHeight), offLinebitmap.Canvas, Rect(0, 0, w1 - FPixelSpacing, FPixelHeight));
    w1 := 2 * w1;
    w2 := 2 * w1 - FPixelSpacing;
  until w1 >= bitmap.Width;

  ly:=0;
  for j := 0 to displaybitmap.Height - 1
  do begin

       { Draw a offpixels line. }

       Bitmap.Canvas.CopyRect(Rect(0, ly, Bitmap.Width, ly + FPixelHeight),offlinebitmap.Canvas, offlinebitmap.Canvas.ClipRect);

       { 8 per 8 pixels analysis: only switch on the needed pixels. }

       PByteRow := pByteArray(displaybitmap.Scanline[j]);
       lx := 7*(FPixelWidth + FPixelSpacing);
       for i := 0 to (displaybitmap.Width div 7) - 1
       do begin
            EightPixels := PByteRow[i];
            if EightPixels <> 255
            then begin
                   for z := 7 downto 0
                   do begin
                        if (EightPixels mod 2) = 0
                        then BitMap.Canvas.CopyRect(rect(lx , ly, lx + FPixelWidth, ly + FPixelHeight), onbitmap.Canvas, PixRect);
                        EightPixels := EightPixels shr 1;
                        lx := lx - FPixelWidth - FPixelSpacing;
                        end;
                   lx := lx + 16*(FPixelWidth + FPixelSpacing);
                   end
            else lx := lx + 8*(FPixelWidth + FPixelSpacing);
            end;
       ly := ly + FPixelHeight + FPixelSpacing;
       end;

  displaybitmap.Free;
  offlinebitmap.Free;
  offbitmap.Free;
  onbitmap.Free;

end;


////////////////////////////////////////////////////////////////////////////////
//
// Change border style.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetBorderStyle(Value: TLCDBorder);
begin
  if Value <> FBorderStyle
  then begin
         FBorderStyle := Value;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Change shape of LCD pixels.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetPixelShape(Value: TPixelShape);
begin
  if Value <> FPixelShape
  then begin
         FPixelShape := Value;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Change pixel spacing (distance between the pixels in the LCD).
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetPixelSpacing(Value: ShortInt);
begin
  if Value <> FPixelSpacing
  then begin
         FPixelSpacing := Value;
         if not FAnimationEnabled then Paint;
         end;
end;



////////////////////////////////////////////////////////////////////////////////
//
// Change LCD pixel size.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetPixelSize(Value: TPixelSize);
begin
  if Value <> FPixelSize
  then begin
         FPixelSize := Value;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set space between the border and character array.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetBorderSpace(Value: ShortInt);
begin
  if Value <> FBorderSpace
  then begin
         FBorderSpace := Value;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set the Special effect enabled or disabled.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetSpecialEffects(Value: TSpecialEffects);
begin
  if Value <> FSpecialEffects
  then begin
         FSpecialEffects := Value;
         UpdateFLinesImage;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set pixel width.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetPixelWidth(Value: ShortInt);
begin
  if FPixelSize = pixCustom
  then if Value <> FPixelWidth
       then if Value < 1
            then MessageDlg('Value must be between 1 and 255.', mtError, [mbOk], 0)
            else begin
                   FPixelWidth := Value;
                   if not FAnimationEnabled then Paint;
                   end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set pixel height.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetPixelHeight(Value: ShortInt);
begin
  if FPixelSize = pixCustom
  then if Value <> FPixelHeight
       then if Value < 1
            then MessageDlg('Value must be between 1 and 255.', mtError, [mbOk], 0)
            else begin
                   FPixelHeight := Value;
                   if not FAnimationEnabled then Paint;
                   end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set component color.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetColor(Value: TColor);
begin
  if Value <> FColor
  then begin
         FColor := Value;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set pixel OFF color.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetPixelOffColor(Value: TColor);
begin
  if Value <> FPixelOffColor
  then begin
         FPixelOffColor := Value;
         UpdateTrueColors;
         if not FAnimationEnabled then Paint;
         end;
end;


///////////////////////////////////////////////////////////////////////////////
//
// Set Font color's intensity ie the color of ONPixels.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetIntensity(Value: ShortInt);
begin
  if Value <> FIntensity
  then begin
         FIntensity := Value;
         UpdateTrueColors;
         if not FAnimationEnabled then Paint;
         end;
end;


///////////////////////////////////////////////////////////////////////////////
//
// Set DisplayMode (only text, only bitmap, both).
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetDisplayMode(Value: TDisplayMode);
begin
  if Value <> FDisplayMode
  then begin
         FDisplayMode := Value;
         if not FAnimationEnabled then Paint;
         end;
end;

////////////////////////////////////////////////////////////////////////////////
//
// Change font.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetFont(Value: TFont);
begin
  if Value <> FFont
  then FFont.Assign(Value);
end;


procedure TLCDScreen.FontOnChange;
begin
  UpdateTrueColors;
  UpdateFLinesImage;
  if not FAnimationEnabled then Paint;
end;


///////////////////////////////////////////////////////////////////////////////
//
// Calculate TrueOnColor.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.UpdateTrueColors;
var
  DeltaRed, DeltaGreen, DeltaBlue: Integer;
  R, G, B: Byte;
begin
  DeltaRed := GetRValue(ColorToRGB(FPixelOffColor)) - GetRValue(ColorToRGB(FFont.Color));
  DeltaGreen := GetGValue(ColorToRGB(FPixelOffColor)) - GetGValue(ColorToRGB(FFont.Color));
  DeltaBlue := GetBValue(ColorToRGB(FPixelOffColor)) - GetBValue(ColorToRGB(FFont.Color));
  R := MulDiv(255 - Abs(FIntensity), DeltaRed, 255);
  G := MulDiv(255 - Abs(FIntensity), DeltaGreen, 255);
  B := MulDiv(255 - Abs(FIntensity), DeltaBlue, 255);

  if FIntensity >= 0
  then begin
         FTrueOnColor := RGB(R, G, B);
         FTrueOffColor := FPixelOffColor;
         end
  else begin
         FTrueOnColor := FPixelOffColor;
         FTrueOffColor := RGB(R, G, B);
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set FLines strings.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetLines(Value: TStringList);
begin
  FLines.Assign(Value);
end;


////////////////////////////////////////////////////////////////////////////////
//
// Update Display with new FLines value.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.LinesOnChange(Sender: TObject);
var line, col, offset: Byte;
    tempstr, istag: String;
    tempsp: LongWord;
    underline, strike ,bold, italic , blink, inverse, isitatag: Boolean;
begin
  Fillchar(Display, sizeof(Display), 0);

  underline := false;
  strike := false;
  bold := false;
  italic := false;
  blink := false;
  inverse := false;

  { Lines per lines, then char per char analysis of Flines.
    Update Display area using the tags.
    Each Display.Char is affected with its Display.SpeEff font effect. }

  for line := 0 to FLines.Count - 1
  do begin
       tempstr := FLines[line];
       if Length(tempstr) <> 0
       then begin
             col := 1;
             offset := 0;
             while tempstr[col+offset] <> Char(0)
             do begin
               if tempstr[col+offset] <> '<'
                  then begin
                         Display[line,col].TheChar := tempstr[col+offset];
                         tempsp := 0;
                         if underline then tempsp := 1;
                         if strike    then tempsp := tempsp + 2;
                         if bold      then tempsp := tempsp + 4;
                         if italic    then tempsp := tempsp + 8;
                         if blink     then tempsp := tempsp + 16;
                         if inverse   then tempsp := tempsp + 32;
                         Display[line, col].SpEff := tempsp;
                         end

                  else begin
                         isitatag := false;
                         istag := Copy(tempstr, col+offset, 3);

                         if istag = startunderline_tag // '<u>'
                         then begin
                                underline := true;
                                inc(offset, 3);
                                dec(col);
                                isitatag := true;
                                end;
                         if istag = startstrike_tag // '<s>'
                         then begin
                                strike := true;
                                inc(offset, 3);
                                dec(col);
                                isitatag := true;
                                end;
                         if istag = startbold_tag // '<b>'
                         then begin
                                bold := true;
                                inc(offset, 3);
                                dec(col);
                                isitatag := true;
                                end;
                         if istag = startitalic_tag // '<i>'
                         then begin
                                italic := true;
                                inc(offset, 3);
                                dec(col);
                                isitatag := true;
                                end;

                         istag := Copy(tempstr, col+offset, 4);

                         if istag = startblinking_tag // '<bl>'
                         then begin
                                blink := true;
                                inc(offset, 4);
                                dec(col);
                                isitatag := true;
                                end;
                         if istag = stopunderline_tag // '</u>'
                         then begin
                                underline := false;
                                inc(offset, 4);
                                dec(col);
                                isitatag := true;
                                end;

                         if istag = stopstrike_tag // '</s>'
                         then begin
                                strike := false;
                                inc(offset, 4);
                                dec(col);
                                isitatag := true;
                                end;

                         if istag = stopbold_tag // '</b>'
                         then begin
                                bold := false;
                                inc(offset, 4);
                                dec(col);
                                isitatag := true;
                                end;
                         if istag = stopitalic_tag // '</i>'
                         then begin
                                italic := false;
                                inc(offset, 4);      
                                dec(col);
                                isitatag := true;
                                end;

                         istag := Copy(tempstr, col+offset, 5);

                         if istag = startinverse_tag // '<inv>'
                         then begin
                                inverse := true;
                                inc(offset, 5);
                                dec(col);
                                isitatag := true;
                                end;
                         if istag = stopblinking_tag // '</bl>'
                         then begin
                                blink := false;
                                inc(offset, 5);
                                dec(col);
                                isitatag := true;
                                end;

                         istag := Copy(tempstr, col+offset, 6);

                         if istag = stopinverse_tag // '</inv>'
                         then begin
                                inverse := false;
                                inc(offset, 6);
                                dec(col);
                                isitatag := true;
                                end;

                         if not isitatag    { special code for '<' char; }
                         then begin
                                Display[line,col].TheChar := tempstr[col+offset];
                                tempsp := 0;
                                if underline then tempsp := 1;
                                if strike then tempsp := tempsp + 2;
                                if bold then tempsp := tempsp + 4;
                                if italic then tempsp := tempsp + 8;
                                if blink then tempsp := tempsp + 16;
                                if inverse then tempsp := tempsp + 32;
                                Display[line, col].SpEff := tempsp;
                                end;
                         end;
                 Inc(col);
                end;
             end;
       end;
  UpdateFLinesImage;
  if not FAnimationEnabled then Paint;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Change LineXOffset.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetLinesXOffset(Value: ShortInt);
begin
  if Value <> FLinesXOffset
  then begin
         FLinesXOffset := Value;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Change LineYOffset.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetLinesYOffset(Value: ShortInt);
begin
  if Value <> FLinesYOffset
  then begin
         FLinesYOffset := Value;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Change LineAnimMode (static or dynamic).
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetLinesAnimMode(Value: TAnimMode);
begin
  if Value <> FLinesAnimMode
  then begin
         FLinesAnimMode := Value;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Update FLinesImage the b&w image of FLines.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.UpdateFLinesImage;
var line, col, ypos, oldwidth: Integer;
  str:string;
  strbitmap: TBitmap;
begin
  FLinesImage.Height := 0;
  FLinesImage.Width := 0;

  strbitmap := TBitmap.Create;
  strbitmap.Width := 0;
  strbitmap.Canvas.Font.Assign(FFont);

  ypos := 0;

  { Lines per lines, then char per char analysis using Display.Char affected off
    its Display.SpeEff font effect. }

  for line := 0 to FLines.Count - 1
  do begin
       col := 1;
       strbitmap.Width := 0;
       strbitmap.Height := 0;


       while Display[line, col].TheChar <> #0
       do begin
            strbitmap.Canvas.Font.Color  := clBlack;
            strbitmap.Canvas.Brush.Color := clWhite;    strbitmap.Canvas.Font.Style := [];
            str := Display[line, col].TheChar;
            if Display[line, col].SpEff <> 0
            then begin
                 if (Display[line, col].SpEff and 1 = 1) and (spUnderline in FSpecialEffects)
                 then strbitmap.Canvas.Font.Style := strbitmap.Canvas.Font.Style + [fsUnderline]
                 else strbitmap.Canvas.Font.Style := strbitmap.Canvas.Font.Style - [fsUnderline];

                 if (Display[line, col].SpEff and 2 <> 0) and (spStrike in FSpecialEffects)
                 then strbitmap.Canvas.Font.Style := strbitmap.Canvas.Font.Style + [fsStrikeOut]
                 else strbitmap.Canvas.Font.Style := strbitmap.Canvas.Font.Style - [fsStrikeOut];

                 if (Display[line, col].SpEff and 4 <> 0) and (spBold in FSpecialEffects)
                 then strbitmap.Canvas.Font.Style := strbitmap.Canvas.Font.Style + [fsBold]
                 else strbitmap.Canvas.Font.Style := strbitmap.Canvas.Font.Style - [fsBold];

                 if (Display[line, col].SpEff and 8 <> 0) and (spItalic in FSpecialEffects)
                 then strbitmap.Canvas.Font.Style := strbitmap.Canvas.Font.Style + [fsItalic]
                 else strbitmap.Canvas.Font.Style := strbitmap.Canvas.Font.Style - [fsItalic];

                 if (((Display[line, col].SpEff and 16) <> 0) and (spBlink in FSpecialEffects) and FBlinkingStatus)
                     or
                    (((Display[line, col].SpEff and 32) <> 0) and (spInverse in FSpecialEffects))
                 then begin
                        strbitmap.Canvas.Font.Color  := clWhite;
                        strbitmap.Canvas.Brush.Color := clBlack;
                        end;
                 end;

            oldwidth := strbitmap.Width;
            strbitmap.Width := strbitmap.Width + strbitmap.Canvas.TextWidth(str);;
            strbitmap.Height := max(strbitmap.Height, strbitmap.Canvas.TextHeight(str));
            strbitmap.Canvas.TextOut(oldwidth,0,str);
            inc(col);
            end;

       if strbitmap.Height = 0 then strbitmap.Height := Abs(FFont.Height);

       FLinesImage.Height := FLinesImage.Height + strbitmap.Height;
       FLinesImage.Width := max(FLinesImage.Width, strbitmap.Width);
       FLinesImage.Canvas.CopyRect(Rect(0, ypos, strbitmap.Width, ypos + strbitmap.Height),
                                  strbitmap.Canvas,
                                  strbitmap.Canvas.ClipRect);

       ypos := ypos + strbitmap.Height;
       end;

  strbitmap.Free;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set FBitmap.
//
////////////////////////////////////////////////////////////////////////////////


procedure TLCDScreen.SetBitmap(Value: TBitmap);
begin
  if Value <> FBitmap
  then begin
         FBitmap.Assign(Value);
         FBitmap.Monochrome := True;
         Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Change BitmapCopyMode (Transparent or not).
//
////////////////////////////////////////////////////////////////////////////////


procedure TLCDScreen.SetBitmapCopyMode(Value: TBitmapCopyMode);
begin
  if Value <> FBitmapCopyMode
  then begin
         FBitmapCopyMode := Value;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Change BitmapXOffset.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetBitmapXOffset(Value: ShortInt);
begin
  if Value <> FBitmapxOffset
  then begin
         FBitmapxOffset := Value;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Change BitmapYOffset.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetBitmapYOffset(Value: ShortInt);
begin
  if Value <> FBitmapyOffset
  then begin
         FBitmapyOffset := Value;
         if not FAnimationEnabled then Paint;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Change BitmapAnimMode (static or dynamic).
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetBitmapAnimMode(Value: TAnimMode);
begin
  if Value <> FBitmapAnimMode
  then begin
         FBitmapAnimMode := Value;
         if not FAnimationEnabled then Paint;
         end;
end;

////////////////////////////////////////////////////////////////////////////////
//
// Set LCDAnimator.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetLCDAnimator(Value: TLCDAnimator);
var tmr: TModalResult;
begin
  if Value <> FLCDAnimator
  then begin
         if Value <> nil
         then begin
                tmr := mrIgnore;

                if Value.CodeErrorFound and (csDesigning in ComponentState)
                then tmr := MessageDlg('Code synthax error(s) detected in this TLCDAnimator.' +
                                       #13 +#10 + #13 + #10 + 'Continue anyway?',
                                       mtWarning, [mbAbort, mbIgnore], 0);
                if tmr = mrIgnore
                then FTimer.OnTimer := TimerOnTimer;
                end;
         FLCDAnimator := Value;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Notification routine when a TLCDAnimator is removed and was linked to a TLCDScreen.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and Assigned(FLCDAnimator) and (AComponent = FLCDAnimator)
  then FLCDAnimator := nil;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set Animation Speed.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetAnimationDelay(Value: Cardinal);
begin
  if Value <> FAnimationDelay
  then begin
         FAnimationDelay := Value;
         FTimer.Interval := Value;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set Animation Active.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetAnimationEnabled(Value: Boolean);
begin
  if Value <> FAnimationEnabled
  then begin
         FAnimationEnabled := Value;
         FTimer.Enabled := Value;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set Cycling Animation.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.SetAnimationRepeating(Value: Boolean);
begin
  if Value <> FAnimationRepeating
  then FAnimationRepeating := Value;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Reset Method.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.Reset(Value: TResetMode);
begin
  case Value of
    rmDisplay: begin
                 PixVRef  := 0;
                 PixHRef  := 0;

                 LinesOnChange(Self);
                 end;

    rmCode   : FLCDAnimator.CurrentLine := 0;

    else begin {rmDisplayAndCode}
           PixVRef  := 0;
           PixHRef  := 0;

           LinesOnChange(Self);
           FLCDAnimator.CurrentLine := 0;
           end;
    end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// General Animation routine.
// Decode CurrentLine and execute it.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.TimerOnTimer(Sender: TObject);
var tempcode: string;
    i: Integer;

    ////////////
        function ExtractWord(Value: String): String;
        begin
           ExtractWord := Trim(Copy(Value, 1, Pos('(', Value) - 1));
        end;
    ////////////
        function ExtractParam(Value: String): Integer;
        begin
          ExtractParam := StrtoInt(Copy(Value, Length(ExtractWord(Value)) + 2,
                                   Pos(')', Value) - Length(ExtractWord(Value)) -2));
        end;
    ////////////
        procedure ExtractCode(tempcode: string; var CodeInstructions: array of TCodeInstruction;
                              NbOfElement: Byte);
        var i,n: Integer;
            s,m: string;
        begin
          tempcode := Trim(LowerCase(Copy(tempcode, 2, Length(tempcode) - 2)));
          for i := 0 to NbOfElement - 1
          do begin
               s := Trim(Copy(tempcode, 1, Pos(';', tempcode)));
               m := ExtractWord(s);
               n := ExtractParam(s);
               CodeInstructions[i].Word := m;
               CodeInstructions[i].Param := n;
               Delete(tempcode, 1, Length(s));
               tempcode := Trim(tempcode);
               end;
          end;
    ////////////

begin
  if FLCDAnimator <> nil
  then begin
         if FLCDAnimator.Code.Count <> 0 then tempcode  :=  FLCDAnimator.Code[FLCDAnimator.CurrentLine];
         P := AllocMem(NbOfThings(tempcode, ';') * SizeOf(TCodeInstruction)); //Removing dynamic Arrays...
         ExtractCode(tempcode, P^, NbOfThings(tempcode, ';'));

         for i := 1 to NbOfThings(tempcode, ';')
         do begin
              if P^[i].Word = 'horzscroll' then HorzScroll(P^[i].Param);
              if P^[i].Word = 'vertscroll' then VertScroll(P^[i].Param);
              if P^[i].Word = 'setintensity' then SetIntensity(P^[i].Param);
              if P^[i].Word = 'animationdelay' then SetAnimationDelay(P^[i].Param);
              if P^[i].Word = 'gotoline' then FLCDAnimator.CurrentLine := Min(P^[i].Param, FLCDAnimator.Code.Count);
              if P^[i].Word = 'resetdisplay' then Reset(rmDisplay);
              end;
         if Assigned(FLCDAnimator.FOnLineExecuted) then FLCDAnimator.FOnLineExecuted(FLCDAnimator, FLCDAnimator.CurrentLine);

         if FLCDAnimator.CurrentLine = FLCDAnimator.Code.Count - 1
         then begin
                FLCDAnimator.CurrentLine := 0;
                if not FAnimationRepeating then SetAnimationEnabled(False);
                if Assigned(FLCDAnimator.FOnEndCode) then FLCDAnimator.FOnEndCode(FLCDAnimator);
                end
         else FLCDAnimator.CurrentLine := FLCDAnimator.CurrentLine + 1;
         FreeMem(P); //Removing dynamic Arrays...

         FBlinkingStatus := not FBlinkingStatus;
         if (spBlink in FSpecialEffects) and (FDisplayMode <> dmBitmap)
         then UpdateFLinesImage;

         Paint;
         end;
  end;


////////////////////////////////////////////////////////////////////////////////
//
// HorzScroll Routine.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.HorzScroll(Value: ShortInt);
begin
  pixHRef := pixHRef + Value;
end;


////////////////////////////////////////////////////////////////////////////////
//
// VertScroll Routine.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreen.VertScroll(Value: ShortInt);
begin
  pixVRef := pixVRef + Value;
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDScreen About routines.
//
////////////////////////////////////////////////////////////////////////////////

function TLCDScreen.GetAbout: string;
begin
  GetAbout := 'About LCDAnimator';
end;


procedure TLCDScreen.SetAbout(Value: string);
begin
  // just for syntax
end;


////////////////////////////////////////////////////////////////////////////////
//
// Create and initialize component TLCDAnimator.
//
////////////////////////////////////////////////////////////////////////////////

constructor TLCDAnimator.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCode := TStringList.Create;
  FCodeErrorFound := False;
  FCurrentLine := 0;
end;


///////////////////////////////////////////////////////////////////////////////
//
// Remove component TLCDAnimator.
//
////////////////////////////////////////////////////////////////////////////////

destructor TLCDAnimator.Destroy;
begin
  FCode.Free;
  inherited Destroy;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set FCode strings.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDAnimator.SetCode(Value: TStrings);
begin
  FCode.Assign(Value);
end;


////////////////////////////////////////////////////////////////////////////////
//
// Set FCurrentLine.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDAnimator.SetCurrentLine(Value: SmallInt);
begin
  if Value <> FCurrentLine
  then begin
         if Value > Code.Count
         then Value := Max(0, Code.Count - 1);
         FCurrentLine := Value;
         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimator About routines.
//
////////////////////////////////////////////////////////////////////////////////

function TLCDAnimator.GetAbout: string;
begin
  GetAbout := 'About LCDAnimator';
end;


procedure TLCDAnimator.SetAbout(Value: string);
begin
  // just for syntax
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDScreen and TLCDAnimator registration.
//
////////////////////////////////////////////////////////////////////////////////

procedure Register;
begin
  RegisterComponents('LCDScreen', [TLCDScreen, TLCDAnimator]);
end;


end.




