unit LCDAnimatorEditor;

////////////////////////////////////////////////////////////////////////////////
//
//  TLCDScreen / TLCDAnimator v3.0 (22/jan/02)
//  written by Jacques VOIRIN
//  E-Mail: jacques.voirin@iname.com
//
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//
//  This unit contain all the routines for the TLCDAnimatorCodeEditor:
//   - registering the editor and associate it to TLCDAnimator.Code
//   - the internal code for its form TLCDAnimatorCodeEditorForm.
//
//  LCDAnimatorEditor is only use at design-time.
//  
////////////////////////////////////////////////////////////////////////////////


interface

uses
  Windows, SysUtils, Classes, Controls, Forms, StdCtrls, Grids, Graphics, Menus,
  Dialogs, ExtCtrls, Messages,
 
    {$IFDEF  VER140}
      DesignIntf, DesignEditors,
    {$ELSE}
      DsgnIntf,
    {$ENDIF}

  LCDScreen;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimatorEditor is the TComponentEditor which generate the custom context
// menu when TLCDAnimator is left-clicked and then display the LCDAnimatorCodeEditorForm.
//
////////////////////////////////////////////////////////////////////////////////
type
  TLCDAnimatorEditor = class(TComponentEditor)
  public
    procedure ExecuteVerb(Index: Integer); override;
    function  GetVerb(Index: Integer): String; override;
    function  GetVerbCount: Integer; override;
  end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimatorCodeEditorProperty is the TPropertyEditor in relation with 'Code'
// property of TLCDAnimator. Clicking on '...' in the Object Inspector display
// the LCDAnimatorCodeEditorForm.
//
////////////////////////////////////////////////////////////////////////////////

  TLCDAnimatorCodeEditorProperty = class(TPropertyEditor)
    procedure Edit; override;
    function  GetValue: String; override;
    function  GetAttributes: TPropertyAttributes; override;
    private
  end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimatorCodeEditorForm is the CodeEditor form, which allow to write
// the code for the animation.
//
////////////////////////////////////////////////////////////////////////////////

  TLCDAnimatorCodeEditorForm = class(TForm)
    CancelButton: TButton;
    OkButton: TButton;
    InsertButton: TButton;
    AddButton: TButton;
    HorzScrollButton: TButton;
    VertScrollButton: TButton;
    LineBeginningButton: TButton;
    LineEndingButton: TButton;
    InstructionEndingButton: TButton;
    SetIntensityButton: TButton;
    AnimationDelayButton: TButton;
    ResetDisplayButton: TButton;
    PanelOnButton: TButton;
    PanelOffButton: TButton;
    GotoLineButton: TButton;
    BlankButton: TButton;
    DeleteButton: TButton;
    CodeCellPUM: TPopupMenu;
    EditMenu: TMenuItem;
    N1: TMenuItem;
    DeleteMenu: TMenuItem;
    DeleteAllMenu: TMenuItem;
    SynthaxPanel: TPanel;
    SynthaxLabel2: TLabel;
    SynthaxLabel1: TLabel;
    SynthaxLabel3: TLabel;
    InsertMenu: TMenuItem;
    AddMenu: TMenuItem;
    N2: TMenuItem;
    Bevel1: TBevel;
    N3: TMenuItem;
    SyntaxAnalysisMenu: TMenuItem;
    CodeStringGrid: TStringGrid;
    Bevel2: TBevel;
    Bevel3: TBevel;

    procedure FormActivate(Sender: TObject);
    procedure InsertLineClick(Sender: TObject);
    procedure WordClick(Sender: TObject);
    procedure DeleteLineClick(Sender: TObject);
    procedure DeleteAllClick(Sender: TObject);
    procedure CodeStringGridDrawCell(Sender: TObject; ACol, ARow: Integer;
      Rect: TRect; State: TGridDrawState);
    procedure AddLineClick(Sender: TObject);
    procedure CodeStringGridSelectCell(Sender: TObject; ACol,
      ARow: Integer; var CanSelect: Boolean);
    procedure EditMenuClick(Sender: TObject);
    procedure SyntaxAnalysisMenuClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure CodeStringGridDblClick(Sender: TObject);
  end;


procedure Register;


var
  LCDAnimatorCodeEditorForm: TLCDAnimatorCodeEditorForm;


implementation

{$R *.DFM}

////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimatorEditor
//
// Result := 1; Only one personalized entry in the context menu.
//
////////////////////////////////////////////////////////////////////////////////

function  TLCDAnimatorEditor.GetVerbCount: Integer;
begin
 Result := 1;
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimatorEditor
//
// Result := 'LCDAnimator Code Editor'; The caption of the personalized entry
//                                      in the context menu.
//
////////////////////////////////////////////////////////////////////////////////

function  TLCDAnimatorEditor.GetVerb(Index: Integer): String;
begin
  Result := 'LCDAnimator Code Editor';
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimatorEditor
//
// This is te code displaying the editor form when 'LCDAnimator Code Editor' in
// the context menu is clicked or twhen he component is double-clicked.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDAnimatorEditor.ExecuteVerb(Index: Integer);
var F: TLCDAnimatorCodeEditorForm;
    i: Byte;
begin
  F := TLCDAnimatorCodeEditorForm.Create(Application);
  F.Caption := 'TLCDAnimator Code Editor - ' + (Component as TLCDAnimator).Name;

  try
    F.CodeStringGrid.RowCount := (Component as TLCDAnimator).Code.Count;

  { Load Code lines from TLCDAnimator to CodeStringGrid }
    if (Component as TLCDAnimator).Code.Count <> 0 
    then for i := 0 to F.CodeStringGrid.RowCount - 1
         do F.CodeStringGrid.Cells[1, i] := (Component as TLCDAnimator).Code[i];

    if F.ShowModal = mrOK  //Display the Editor form
    then begin
           (Component as TLCDAnimator).Code.Clear;

  { UnLoad Code lines from CodeStringGrid to TLCDAnimator }
           if F.CodeStringGrid.RowCount <> 0
           then for i := 0 to F.CodeStringGrid.RowCount - 1
                do (Component as TLCDAnimator).Code.Append(F.CodeStringGrid.Cells[1, i]);

           try Designer.Modified; except end;
           end;

  { Check if a code synthax error had been detected and update CodeErrorFond prop.}
    if F.Tag = 0 then (Component as TLCDAnimator).CodeErrorFound := False
                 else (Component as TLCDAnimator).CodeErrorFound := True;
  finally
    F.Free;
  end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimatorCodeEditorProperty
//
// Result := '(TStrings)'; The 'Code' caption displayed in the Object Inspector.
//
////////////////////////////////////////////////////////////////////////////////

function TLCDAnimatorCodeEditorProperty.GetValue: String;
begin
  Result := '(TStrings)';
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimatorCodeEditorProperty
//
// Result := [paDialog]; Indicates that the '...' button in the Object Inspector
//                       will bring up a dialog form.
//
////////////////////////////////////////////////////////////////////////////////

function TLCDAnimatorCodeEditorProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimatorCodeEditorProperty
//
// The code displaying the editor form when the '...' button in the Object
// Inspector is clicked.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDAnimatorCodeEditorProperty.Edit;
var F: TLCDAnimatorCodeEditorForm;
    C: TPersistent;
    i: Integer;
begin
  C := GetComponent(0);
  F := TLCDAnimatorCodeEditorForm.Create(Application);
  F.Caption := 'TLCDAnimator Code Editor - ' + (C as TLCDAnimator).Name;

  try
    F.CodeStringGrid.RowCount := (C as TLCDAnimator).Code.Count;

  { Load Code lines from TLCDAnimator to CodeStringGrid }
    if (C as TLCDAnimator).Code.Count <> 0
    then for i := 0 to F.CodeStringGrid.RowCount - 1
         do F.CodeStringGrid.Cells[1, i] := (C as TLCDAnimator).Code[i];

    if F.ShowModal = mrOk
    then begin
           (C as TLCDAnimator).Code.Clear;

  { UnLoad Code lines from CodeStringGrid to TLCDAnimator }
           if F.CodeStringGrid.RowCount <> 0
           then for i := 0 to F.CodeStringGrid.RowCount - 1
                do (C as TLCDAnimator).Code.Append(F.CodeStringGrid.Cells[1, i]);

           try Designer.Modified; except end;
           end;

  { Check if a code synthax error had been detected and update CodeErrorFond prop.}
    if F.Tag = 0 then (C as TLCDAnimator).CodeErrorFound := False
                 else (C as TLCDAnimator).CodeErrorFound := True;
  finally
    F.Free;
  end;
end;



////////////////////////////////////////////////////////////////////////////////
//
// Begining of the routines for the Form Editor.
//
////////////////////////////////////////////////////////////////////////////////


procedure TLCDAnimatorCodeEditorForm.FormActivate(Sender: TObject);
begin
  CodeStringGrid.ColWidths[0] := 45;
  CodeStringGrid.ColWidths[1] := 430;
end;


procedure TLCDAnimatorCodeEditorForm.CodeStringGridSelectCell(
  Sender: TObject; ACol, ARow: Integer; var CanSelect: Boolean);
begin
   with CodeStringGrid do
  begin
    Cells[1, Selection.Top] :=  Trim(Cells[1, Selection.Top]);
    Invalidate;
    end;
end;


procedure TLCDAnimatorCodeEditorForm.InsertLineClick(Sender: TObject);
var i: Integer;
begin
  with CodeStringGrid do
  begin
    RowCount := RowCount + 1;

    for i := RowCount downto Selection.Top + 1
    do Cells[1, i] := Cells[1, i - 1];

    Cells[1, Selection.Top] := '';
    SetFocus;
    Perform(WM_Keydown, vk_F2,0);
    end;
end;


procedure TLCDAnimatorCodeEditorForm.AddLineClick(Sender: TObject);
var i: Integer;
begin
  with CodeStringGrid do
  begin
    RowCount := RowCount + 1;

    for i := RowCount downto Selection.Top + 2
    do Cells[1, i] := Cells[1, i - 1];

    Perform(WM_Keydown, vk_Down,0);
    Cells[1, Selection.Top] := '';
    SetFocus;
    CodeStringGrid.Perform(WM_Keydown, vk_F2,0);
    end;
end;


procedure TLCDAnimatorCodeEditorForm.DeleteLineClick(Sender: TObject);
var i:Integer;
begin
  with CodeStringGrid do
  begin
    for i := Selection.Top to RowCount - 1
    do Cells[1, i] := Cells[1, i + 1];

    RowCount := RowCount - 1;

   // Invalidate;
    end;
end;


procedure TLCDAnimatorCodeEditorForm.DeleteAllClick(Sender: TObject);
begin
  if MessageDlg('Delete all code lines?', mtConfirmation, [mbOK, mbCancel], 0) = mrOK
  then begin
         CodeStringGrid.RowCount := 0;
         CodeStringGrid.Cells[1, 0] := '';
         end;
end;


procedure TLCDAnimatorCodeEditorForm.WordClick(Sender: TObject);
begin
  with CodeStringGrid do
  begin
    SetFocus;
    Cells[1, Selection.Top] := Cells[1, Selection.Top] + (Sender as TButton).Caption;
    Perform(WM_Keydown, vk_F2,0);
    Perform(WM_Keydown, vk_End,0);
    end;
end;


function CodeSynthaxAnalysis(var Caption: String): TColor;
var n1, n2, n3: Byte;
begin
  CodeSynthaxAnalysis := clRed;
  Caption := Trim(Caption);
  n1 := NbOfThings(Caption, ';');
  n2 := NbOfThings(Caption, '(');
  n3 := NbOfThings(Caption, ')');

  if Caption = ''
  then begin
         CodeSynthaxAnalysis := clYellow;
         Caption := 'Nul line';
         end
  else if Copy(Caption, 1, 1) <> '['
       then Caption := 'Error of ['
       else if Copy(Caption, Length(Caption), 1) <> ']'
            then Caption := 'Error of ]'
            else if not((n1 = n2) and (n2 = n3))
                 then begin
                        if n2 <> n3 then Caption := 'Error of ( )'
                                    else if n1 <> n2 then  Caption := 'Error of ;';
                        end
                 else begin
                        CodeSynthaxAnalysis := clLime;
                        Caption := 'Ok'
                        end;
end;


procedure TLCDAnimatorCodeEditorForm.CodeStringGridDrawCell(Sender: TObject;
  ACol, ARow: Integer; Rect: TRect; State: TGridDrawState);
var str: String;
begin
  if gdFixed in State
  then with CodeStringGrid do
  begin
    str := CodeStringGrid.Cells[1, ARow];
    Canvas.Font.Color := CodeSynthaxAnalysis(str);

    if not SyntaxAnalysisMenu.Checked
    then str := IntToStr(ARow + 1) + ' / ' + IntToStr(RowCount);

    Canvas.TextOut((ColWidths[0] - Canvas.TextWidth(str)) div 2, Rect.Top + 2, str);
    end;
end;


procedure TLCDAnimatorCodeEditorForm.EditMenuClick(Sender: TObject);
begin
  CodeStringGrid.Perform(WM_Keydown, vk_F2,0);
end;


procedure TLCDAnimatorCodeEditorForm.SyntaxAnalysisMenuClick(
  Sender: TObject);
begin
  SyntaxAnalysisMenu.Checked := not SyntaxAnalysisMenu.Checked;

  if SyntaxAnalysisMenu.Checked then CodeStringGrid.ColWidths[0] := 70
                                else CodeStringGrid.ColWidths[0] := 45;
end;


procedure TLCDAnimatorCodeEditorForm.CodeStringGridDblClick(
  Sender: TObject);
var ACol, ARow: Integer;
begin
  CodeStringGrid.MouseToCell(Mouse.CursorPos.X - CodeStringGrid.ClientOrigin.X,
                             Mouse.CursorPos.Y - CodeStringGrid.ClientOrigin.Y, ACol, ARow);
  if ACol = 0 then SyntaxAnalysisMenuClick(Sender);
end;


procedure TLCDAnimatorCodeEditorForm.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
var i, NbErr: Integer;
    str: String;
    tmr: TModalResult;
begin
  NbErr := 0;
  tmr := mrIgnore;

  for i := 0 to CodeStringGrid.RowCount - 1
  do begin
       str := CodeStringGrid.Cells[1, i];
       if CodeSynthaxAnalysis(str) = clRed then Inc(NbErr);
       end;

  if NbErr <> 0 then tmr := MessageDlg('Code synthax error(s) detected.' + #13 + #10 +
                                       #13 + #10 + 'Close Editor anyway?',
                                       mtWarning, [mbAbort, mbIgnore], 0);
  Tag := NbErr;

  if tmr = mrIgnore then CanClose := True
                    else CanClose := False;
end;


////////////////////////////////////////////////////////////////////////////////
//
// End of the routines for the Form Editor.
//
////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
//
// Registration of the context menu and 'Code  [...]' in the ObjectInspector.
//
////////////////////////////////////////////////////////////////////////////////

procedure Register;
begin
  RegisterComponentEditor(TLCDAnimator,TLCDAnimatorEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TLCDAnimator, 'Code', TLCDAnimatorCodeEditorProperty);
end;


end.
