{***************************************************************
 *
 * Unit Name: AJBFormCaption
 * Purpose  : Specify a TWinControl to act as a Caption Bar
 * Author   : Andrew Baylis
 * Date     : 6/07/2001
 * History  : Ver 1.00
              Ver 2.00 Allowed any descendant of TControl to be used
              as a captionbar
              Ver 2.10 Checked for mouseevent over another control
              that lies on the captionbar - Fires OnFoundControl Event
              Ver 2.20 Changed Window resize method to use MoveWindow
              rather than window regions - this leads to better functionality
              when resizing the form onscreen.
 *
 ****************************************************************}

unit AJBFormCaption;

interface

uses Classes, messages, Controls, Forms, Windows, sysutils;

type
   //Set Handled to False to let control receive it normally, True to send
   //to caption bar
   TFoundContEvent = procedure(const AControl: TControl; var TranslateToNC: Boolean) of object;

   TAJBFormCaption = class(TComponent)
   private
      FActive: Boolean;
      FForm: TWinControl;
      FHideWindowCaption: Boolean;
      FNewProc: Pointer;
      FOldProc: Pointer;
      FWinControl: TWinControl;
      FCaptionControl: TControl;
      FCheckBounds: Boolean;
      FOnFoundControl: TFoundContEvent;
      procedure Activate;
      procedure DeActivate;
      procedure HookedMsgProc(var Msg: TMessage);
      procedure HookWindow;
      procedure SetActive(Value: boolean);
      procedure SetHideWindowCaption(Value: Boolean);
      procedure UnhookWindow;
      procedure SetCaptionControl(Value: TControl);
   protected
      procedure Loaded; override;
      procedure ProcessCaptionBar(Hide: Boolean);
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
   published
      property Active: Boolean read FActive write SetActive;
      property HideWindowCaption: Boolean read FHideWindowCaption write
      SetHideWindowCaption;
      property CaptionControl: TControl read FCaptionControl write SetCaptionControl;
      property OnFoundControl: TFoundContEvent read FOnFoundControl write
      FOnFoundControl;
   end;

procedure Register;

implementation

procedure register;
begin
   RegisterComponents('AJB', [TAJBFormCaption]);
end;

procedure TAJBFormCaption.Activate;
begin
   if not (csDesigning in ComponentState) and (Owner is TControl) then
   begin
    // If the property WinControl is empty try underlying form
      if not assigned(FWinControl) then
         FWinControl := FForm;
      HookWindow;

      ProcessCaptionBar(FHideWindowCaption);
   end;
end;

procedure TAJBFormCaption.DeActivate;
begin
   if FActive then
   begin
      UnHookWindow;
      ProcessCaptionBar(False);
      FActive := False;
   end;
end;

procedure TAJBFormCaption.HookedMsgProc(var Msg: TMessage);
var
   Handled: boolean;
   r: TRect;
   x, y: Integer;
   c: TControl;
begin
   Handled := False;
   if (Msg.Msg >= WM_MOUSEMOVE) and (Msg.Msg <= WM_MBUTTONDBLCLK) and
      (FForm <> nil) then
   begin
      Handled := True;
      x := Msg.LParamLo;
      y := Msg.LParamHi;
      if FCheckBounds then // captionbar is TGraphicControl descendant
      begin
         r := FCaptionControl.BoundsRect;
         if (r.Left <= x) and (r.Right >= x) and (r.Top <= y) and (r.Bottom >= y) then
            Handled := true
         else
            Handled := False;
      end;

      if Handled then
      begin
         c := FWinControl.ControlAtPos(Point(x, y), False);
         if (c <> nil) and (c <> FCaptionControl) and
            Assigned(FOnFoundControl) then FOnFoundControl(c, Handled);
      end;

      if (Handled) then
      begin
    // change mouse message to NC Mouse message
         Postmessage(FForm.Handle, Msg.Msg - $0160, HTCAPTION, Msg.lParam);
         Msg.Result := 0;
      end;
   end;
  // If the event isn't handled yet call old WndProc
   if not Handled then
   begin
    // Must set Msg.Result
      Msg.Result := CallWindowProc(FOldProc, FWinControl.Handle, Msg.Msg, Msg.WParam, Msg.LParam);
   end;
end;

procedure TAJBFormCaption.HookWindow;
begin
   if not (csDesigning in ComponentState) and (Owner is TControl) then
   begin
      if assigned(FWinControl) and (FOldProc = nil) then
      begin
         FOldProc := Pointer(GetWindowLong(FWinControl.Handle, GWL_WNDPROC));
         SetWindowLong(FWinControl.Handle, GWL_WNDPROC, Integer(FNewProc));
      end;
   end;
end;

procedure TAJBFormCaption.SetActive(Value: boolean);
begin
   if FActive <> Value then
   begin
      if not (csLoading in ComponentState) then
         if Value then
            Activate
         else
            DeActivate;
      FActive := Value;
   end;
end;

procedure TAJBFormCaption.SetHideWindowCaption(Value: Boolean);
begin
   FHideWindowCaption := Value;
   if (FActive) and (not (csLoading in ComponentState)) then
      ProcessCaptionBar(Value);
end;

procedure TAJBFormCaption.UnhookWindow;
begin
   if not (csDesigning in ComponentState) and (Owner is TControl) then
   begin
      if assigned(FWinControl) and (not (csDestroying in FWinControl.ComponentState)) then
      begin
         if (FOldProc <> nil) then
            SetWindowLong(FWinControl.Handle, GWL_WNDPROC, Integer(FOldProc));
         FOldProc := nil;
      end
   end;
end;

procedure TAJBFormCaption.Loaded;
begin
   inherited Loaded;
   if FActive then
      Activate;
end;

procedure TAJBFormCaption.ProcessCaptionBar(Hide: Boolean);
var
   winStyle: Longint;
   cRect, wRect: TRect;
begin
   if not (csDesigning in ComponentState) and (Owner is TControl) and
      (FForm <> nil) and not (csDestroying in FForm.ComponentState) then
   begin
      winStyle := GetWindowLong(FForm.Handle, GWL_STYLE);
      GetWindowRect(FForm.Handle, wRect);
      GetClientRect(FForm.Handle, cRect);
      if Hide and ((winStyle and WS_CAPTION) > 0) then
      begin
         winStyle := winStyle and (not WS_CAPTION);
         wRect.Bottom := wRect.Top + cRect.Bottom;
      end;
      if (not Hide) and ((winStyle and WS_CAPTION) > 0) then
      begin
         winStyle := winStyle or WS_CAPTION;
         wRect.Bottom := wRect.Bottom + GetSystemMetrics(SM_CYCAPTION);
      end;
      SetWindowLong(FForm.Handle, GWL_STYLE, winStyle);
      MoveWindow(FForm.Handle, wRect.Left, wRect.Top,
         (wRect.Right - wRect.Left), (wRect.Bottom - wRect.Top), True);
   end;
end;

constructor TAJBFormCaption.Create(AOwner: TComponent);
var
   tc: TWinControl;
begin
   inherited Create(AOwner);
   FWinControl := nil;
   FCaptionControl := nil;
   FCheckBounds := False;
   FHideWindowCaption := False;
   FActive := False;
   tc := TWinControl(AOwner);
   while (tc <> nil) and (not (tc is TForm)) do
      tc := tc.Parent;
   if (tc <> nil) then
      FForm := tc
   else
      FForm := nil;
  // Make FNewProc a pointer to the new WndProc
   FNewProc := MakeObjectInstance(HookedMsgProc);
   FOnFoundControl := nil;
end;

destructor TAJBFormCaption.Destroy;
begin
   DeActivate;
   FreeObjectInstance(FNewProc);
   inherited Destroy;
end;

procedure TAJBFormCaption.Notification(AComponent: TComponent;
   Operation: TOperation);
begin
   if (Operation = opRemove) and (AComponent = FCaptionControl) then
   begin
      DeActivate;
      FWinControl := nil;
      FCaptionControl := nil;
      FCheckBounds := False;
   end;
   inherited;
end;

procedure TAJBFormCaption.SetCaptionControl(Value: TControl);
begin
   FCaptionControl := Value;
   if (Value = nil) then
   begin
      FWinControl := nil;
      FCaptionControl := nil;
      Exit;
   end;
   if Value is TWinControl then
   begin
      FWinControl := TWinControl(Value);
      FCheckBounds := False;
   end
   else
   begin
      FWinControl := FForm;
      FCheckBounds := True;
   end;
end;

end.

