{$INCLUDE cHeader.inc}
unit cTypes;

{                                                                              }
{                    Base classes for data types v0.05 (L0)                    }
{                                                                              }
{        This unit is copyright  2000 by David Butler (david@e.co.za)         }
{                                                                              }
{                  This unit is part of Delphi Fundamentals.                   }
{                    It's original file name is cTypes.pas                     }
{                      It was generated 5 Jul 2000 02:10.                      }
{                                                                              }
{                I invite you to use this unit, free of charge.                }
{        I invite you to distibute this unit, but it must be for free.         }
{             I also invite you to contribute to its development,              }
{             but do not distribute a modified copy of this file.              }
{       Send modifications, suggestions and bug reports to david@e.co.za       }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   2000/06/07  v0.01  Base classes (AIntegerArray, ASet) moved from           }
{                      cDataStruct.                                            }
{   2000/06/08  v0.02  Added AObjectArray.                                     }
{                      Started moving cType from L1 to L0, revising it on the  }
{                      way. Implementations storing simple types go to L0,     }
{                      and those that store ATypes stay in L1.                 }
{   2000/06/09  v0.03  Added AStack base classes.                              }
{                      473 lines interface. 1220 lines implementation.         }
{   2000/06/14  v0.04  Converted to a template file.                           }
{                      359 lines interface. 661 lines implementation.          }
{   2000/06/16  v0.05  Added ADictionary.                                      }
{                      Added AQueue.                                           }
{                                                                              }

interface

uses
  // Delphi units
  SysUtils,
  Classes,

  // Delphi Fundamentals (L0)
  cUtils,
  cStreams;




{                                                                              }
{ AType                                                                        }
{   Abstract base class for data structures. Provides an interface for common  }
{   data operators such as assigning, comparing and duplicating.               }
{                                                                              }
{   Duplicate creates a new instance of the structure and copies the content.  }
{   Clear sets instance content ("value") to an empty/initialized/zero state.  }
{   The streaming functions provides interface to stream the content.          }
{   IsEqual compares content.                                                  }
{   Compare is the ranking function (for sorting and searching).               }
{   Assign calls AssignTo if no overriden implementation provided.             }                       
{                                                                              }
type
  EType = class (Exception);
  AType = class
    protected
    Function  GetAsString : String; virtual;
    Procedure SetAsString (const S : String); virtual;
    Procedure AssignTo (const Dest : AType); virtual;

    public
    class Function CreateInstance : AType; virtual; abstract;
    Property  AsString : String read GetAsString write SetAsString;
    Function  Duplicate : AType; virtual;
    Procedure Assign (const Source : AType); virtual;
    Function  IsEqual (const V : AType) : Boolean; virtual;
    Function  Compare (const V : AType) : TCompareResult; virtual;
    Procedure Clear; virtual; abstract;
    Function  IsEmpty : Boolean; virtual; abstract;
    Procedure StreamOut (const S : TExStream); virtual;
    Procedure StreamIn (const S : TExStream); virtual; 
  end;



{                                                                              }
{ ACollection                                                                  }
{   IsHomogeneous returns True if all entries are of the same type.            }
{                                                                              }
type
  EIterator = class (Exception);
  AIterator = class
    Procedure Next; virtual; abstract;
    Function  EOF : Boolean; virtual; abstract;
    Procedure Delete; virtual; abstract;
    Function  Duplicate : AIterator; virtual; abstract;
  end;
  ACollection = class (AType)
    protected
    Function  GetCount : Integer; virtual;
    Procedure SetCount (const NewCount : Integer); virtual; abstract;

    public
    { ACollection interface }
    class Function IsOrdered : Boolean; virtual; abstract;
    class Function IsHomogeneous : Boolean; virtual; abstract;

    Function  Iterate : AIterator; virtual; abstract;
    Property  Count : Integer read GetCount;
  end;


{                                                                              }
{ AArray                                                                       }
{   Base class for arrays (ordered list).                                      }
{   Sort, Reverse and Clear have default implementations.                      }
{                                                                              }
type
  EArray = class (Exception);
  AArray = class (ACollection)
    protected
    Function  GetSorted : Boolean; virtual; abstract;
    Procedure SetSorted (const Sorted : Boolean); virtual; abstract;

    public
    { AType implementations }
    Procedure Clear; override;

    { ACollection implementations }
    class Function IsOrdered : Boolean; override;

    { AArray interface                                                         }
    Property  Count read GetCount write SetCount;
    Function  DuplicateRange (const LoIdx, HiIdx : Integer) : AArray; virtual; abstract;
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); virtual; abstract;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; virtual; abstract;
    Procedure Sort; virtual;
    Property  Sorted : Boolean read GetSorted write SetSorted;
    Procedure ReverseOrder; virtual;
    Procedure Delete (const Idx : Integer; const Count : Integer = 1); virtual; abstract;
    Procedure Insert (const Idx : Integer; const Count : Integer = 1); virtual; abstract;
    Function  Add (const V : AArray) : Integer; virtual; abstract;
  end;


{                                                                              }
{ AIntegerArray                                                                }
{   Base class for an array of Integers.                                       }
{                                                                              }
type
  EIntegerArray = class (Exception);
  AIntegerArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Integer); virtual; abstract;
    Function  GetItem (const Idx : Integer) : Integer; virtual; abstract;
    Function  GetRange (const LoIdx, HiIdx : Integer) : IntegerArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : IntegerArray); virtual;
    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const V : AType); override;
    Function  IsEqual (const V : AType) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  Add (const V : AArray) : Integer; reintroduce; overload; override;

    { AIntegerArray interface                                                      }
    Property  Item [const Idx : Integer] : Integer read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : IntegerArray read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : Integer); virtual;
    Procedure FillInc (const Idx, Count : Integer; const StartValue : Integer = 0; const Increment : Integer = 1); virtual;
    Function  Add (const Value : Integer) : Integer; reintroduce; overload; virtual;
    Function  Add (const V : IntegerArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : Integer; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;



{                                                                              }
{ AInt64Array                                                                  }
{   Base class for an array of Int64s.                                         }
{                                                                              }
type
  EInt64Array = class (Exception);
  AInt64Array = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Int64); virtual; abstract;
    Function  GetItem (const Idx : Integer) : Int64; virtual; abstract;
    Function  GetRange (const LoIdx, HiIdx : Integer) : Int64Array; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : Int64Array); virtual;
    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const V : AType); override;
    Function  IsEqual (const V : AType) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  Add (const V : AArray) : Integer; reintroduce; overload; override;

    { AInt64Array interface                                                        }
    Property  Item [const Idx : Integer] : Int64 read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : Int64Array read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : Int64); virtual;
    Procedure FillInc (const Idx, Count : Integer; const StartValue : Int64 = 0; const Increment : Int64 = 1); virtual;
    Function  Add (const Value : Int64) : Integer; reintroduce; overload; virtual;
    Function  Add (const V : Int64Array) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : Int64; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;



{                                                                              }
{ ASingleArray                                                                 }
{   Base class for an array of Singles.                                        }
{                                                                              }
type
  ESingleArray = class (Exception);
  ASingleArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Single); virtual; abstract;
    Function  GetItem (const Idx : Integer) : Single; virtual; abstract;
    Function  GetRange (const LoIdx, HiIdx : Integer) : SingleArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : SingleArray); virtual;
    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const V : AType); override;
    Function  IsEqual (const V : AType) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  Add (const V : AArray) : Integer; reintroduce; overload; override;

    { ASingleArray interface                                                       }
    Property  Item [const Idx : Integer] : Single read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : SingleArray read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : Single); virtual;
    Procedure FillInc (const Idx, Count : Integer; const StartValue : Single = 0; const Increment : Single = 1); virtual;
    Function  Add (const Value : Single) : Integer; reintroduce; overload; virtual;
    Function  Add (const V : SingleArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : Single; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;



{                                                                              }
{ ADoubleArray                                                                 }
{   Base class for an array of Doubles.                                        }
{                                                                              }
type
  EDoubleArray = class (Exception);
  ADoubleArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Double); virtual; abstract;
    Function  GetItem (const Idx : Integer) : Double; virtual; abstract;
    Function  GetRange (const LoIdx, HiIdx : Integer) : DoubleArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : DoubleArray); virtual;
    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const V : AType); override;
    Function  IsEqual (const V : AType) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  Add (const V : AArray) : Integer; reintroduce; overload; override;

    { ADoubleArray interface                                                       }
    Property  Item [const Idx : Integer] : Double read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : DoubleArray read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : Double); virtual;
    Procedure FillInc (const Idx, Count : Integer; const StartValue : Double = 0; const Increment : Double = 1); virtual;
    Function  Add (const Value : Double) : Integer; reintroduce; overload; virtual;
    Function  Add (const V : DoubleArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : Double; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;



{                                                                              }
{ AExtendedArray                                                               }
{   Base class for an array of Extendeds.                                      }
{                                                                              }
type
  EExtendedArray = class (Exception);
  AExtendedArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Extended); virtual; abstract;
    Function  GetItem (const Idx : Integer) : Extended; virtual; abstract;
    Function  GetRange (const LoIdx, HiIdx : Integer) : ExtendedArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : ExtendedArray); virtual;
    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const V : AType); override;
    Function  IsEqual (const V : AType) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  Add (const V : AArray) : Integer; reintroduce; overload; override;

    { AExtendedArray interface                                                     }
    Property  Item [const Idx : Integer] : Extended read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : ExtendedArray read GetRange write SetRange;
    Procedure Fill (const Idx, Count : Integer; const Value : Extended); virtual;
    Procedure FillInc (const Idx, Count : Integer; const StartValue : Extended = 0; const Increment : Extended = 1); virtual;
    Function  Add (const Value : Extended) : Integer; reintroduce; overload; virtual;
    Function  Add (const V : ExtendedArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : Extended; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;




{                                                                              }
{ AStringArray                                                                 }
{   Base class for an array of Strings.                                        }
{                                                                              }
type
  EStringArray = class (Exception);
  AStringArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : String); virtual; abstract;
    Function  GetItem (const Idx : Integer) : String; virtual; abstract;
    Function  GetRange (const LoIdx, HiIdx : Integer) : StringArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : StringArray); virtual;
    Function  GetAsStringList : TStringList; virtual;
    Procedure SetAsStringList (const V : TStringList); virtual;
    Function  GetAsString : String; override;
    Procedure SetAsString (const S : String); override;

    public
    { AType implementations                                                    }
    Procedure Assign (const V : AType); override;
    Function  IsEqual (const V : AType) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  Add (const V : AArray) : Integer; reintroduce; overload; override;

    { AStringArray interface                                                   }
    Property  Item [const Idx : Integer] : String read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : StringArray read GetRange write SetRange;
    Property  AsStringList : TStringList read GetAsStringList write SetAsStringList;
    Procedure Fill (const Idx, Count : Integer; const Value : String = ''); virtual;
    Function  Add (const Value : String) : Integer; reintroduce; overload; virtual;
    Function  Add (const V : StringArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : String; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;



{                                                                              }
{ ABitArray                                                                    }
{   Base class for a bit array (a set) implementations.                        }
{   Bits are defined as False at initialization.                               }
{   Find finds Count consequetive bits set to Value, starting at (and          }
{     including) Start. It returns the index of the leftmost bit or -1 if      }
{     not found.                                                               }
{                                                                              }
type
  EBitArray = class (Exception);
  ABitArray = class (AArray)
    protected
    Function  GetBit (const Idx : Integer) : Boolean; virtual; abstract;
    Procedure SetBit (const Idx : Integer; const Value : Boolean); virtual; abstract;
    Function  GetRangeAsCardinal (const Idx : Integer) : Cardinal; virtual;
    Procedure SetRangeAsCardinal (const Idx : Integer; const Value : Cardinal); virtual;

    public
    Property  Bit [const Idx : Integer] : Boolean read GetBit write SetBit; default;
    Property  RangeAsCardinal [const Idx : Integer] : Cardinal read GetRangeAsCardinal write SetRangeAsCardinal;
    Function  CompareRange (const LoIdx, HiIdx : Integer; const Value : Boolean) : Boolean; virtual;
    Procedure Fill (const Idx, Count : Integer; const Value : Boolean); virtual;
    Procedure Assign (const V : AType); override;
    Function  IsEqual (const V : AType) : Boolean; override;
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  Add (const Value : Boolean) : Integer; reintroduce; overload; virtual;
    Procedure Invert; virtual;
    Function  Find (const Value : Boolean = False; const Start : Integer = 0;
              const FindForward : Boolean = True) : Integer; virtual;
    Function  FindRange (const Value : Boolean = False; const Start : Integer = 0;
              const Count : Integer = 1; const FindForward : Boolean = True) : Integer; virtual;
  end;



{                                                                              }
{ AObjectArray                                                                 }
{   Base class for an array of Objects.                                        }
{                                                                              }
type
  EObjectArray = class (Exception);
  AObjectArray = class (AArray)
    protected
    Procedure SetItem (const Idx : Integer; const Value : TObject); virtual; abstract;
    Function  GetItem (const Idx : Integer) : TObject; virtual; abstract;
    Function  GetRange (const LoIdx, HiIdx : Integer) : ObjectArray; virtual;
    Procedure SetRange (const LoIdx, HiIdx : Integer; const V : ObjectArray); virtual;
    Function  GetAsString : String; override;

    public
    { AType implementations                                                    }
    Procedure Assign (const V : AType); override;
    Function  IsEqual (const V : AType) : Boolean; override;

    { AArray implementations                                                   }
    Procedure ExchangeItems (const Idx1, Idx2 : Integer); override;
    Function  CompareItems (const Idx1, Idx2 : Integer) : TCompareResult; override;
    Function  Add (const V : AArray) : Integer; reintroduce; overload; override;

    { AObjectArray interface                                                   }
    Property  Item [const Idx : Integer] : TObject read GetItem write SetItem; default;
    Property  Range [const LoIdx, HiIdx : Integer] : ObjectArray read GetRange write SetRange;
    Function  Add (const Value : TObject) : Integer; reintroduce; overload; virtual;
    Function  Add (const V : ObjectArray) : Integer; reintroduce; overload; virtual;
    Function  PosNext (const Find : TObject; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer; virtual;
  end;


{                                                                              }
{ ADictionary                                                                  }
{                                                                              }
type
  ADictionaryIterator = class (AIterator)
    protected
    Function  GetKey : String; virtual; abstract;
    Procedure SetKey (const Key : String); virtual; abstract;

    public
    Property Key : String read GetKey write SetKey;
  end;
  ADictionary = class (ACollection)
    protected
    Function  GetCreateOnGet : Boolean; virtual; abstract;
    Procedure SetCreateOnGet (const CreateOnGet : Boolean); virtual; abstract;
    Function  GetCreateOnSet : Boolean; virtual; abstract;
    Procedure SetCreateOnSet (const CreateOnSet : Boolean); virtual; abstract;

    public
    Procedure Delete (const Key : String); virtual; abstract;
    Function  HasKey (const Key : String) : Boolean; virtual; abstract;
    Property  CreateOnGet : Boolean read GetCreateOnGet write SetCreateOnGet;
    Property  CreateOnSet : Boolean read GetCreateOnSet write SetCreateOnSet;
  end;



{                                                                              }
{ AIntegerDictionary                                                           }
{                                                                              }
type
  AIntegerDictionaryIterator = class (ADictionaryIterator)
    protected
    Function  GetValue : Integer; virtual; abstract;
    Procedure SetValue (const Value : Integer); virtual; abstract;

    public
    Property Value : Integer read GetValue write SetValue;
  end;
  AIntegerDictionary = class (ADictionary)
    protected
    Function  GetItem (const Key : String) : Integer; virtual; abstract;
    Procedure SetItem (const Key : String; const Value : Integer); virtual; abstract;

    public
    Property  Item [const Key : String] : Integer read GetItem write SetItem; default;
    Procedure Add (const Key : String; const Value : Integer); virtual; abstract;
  end;



{                                                                              }
{ AInt64Dictionary                                                             }
{                                                                              }
type
  AInt64DictionaryIterator = class (ADictionaryIterator)
    protected
    Function  GetValue : Int64; virtual; abstract;
    Procedure SetValue (const Value : Int64); virtual; abstract;

    public
    Property Value : Int64 read GetValue write SetValue;
  end;
  AInt64Dictionary = class (ADictionary)
    protected
    Function  GetItem (const Key : String) : Int64; virtual; abstract;
    Procedure SetItem (const Key : String; const Value : Int64); virtual; abstract;

    public
    Property  Item [const Key : String] : Int64 read GetItem write SetItem; default;
    Procedure Add (const Key : String; const Value : Int64); virtual; abstract;
  end;



{                                                                              }
{ ASingleDictionary                                                            }
{                                                                              }
type
  ASingleDictionaryIterator = class (ADictionaryIterator)
    protected
    Function  GetValue : Single; virtual; abstract;
    Procedure SetValue (const Value : Single); virtual; abstract;

    public
    Property Value : Single read GetValue write SetValue;
  end;
  ASingleDictionary = class (ADictionary)
    protected
    Function  GetItem (const Key : String) : Single; virtual; abstract;
    Procedure SetItem (const Key : String; const Value : Single); virtual; abstract;

    public
    Property  Item [const Key : String] : Single read GetItem write SetItem; default;
    Procedure Add (const Key : String; const Value : Single); virtual; abstract;
  end;



{                                                                              }
{ ADoubleDictionary                                                            }
{                                                                              }
type
  ADoubleDictionaryIterator = class (ADictionaryIterator)
    protected
    Function  GetValue : Double; virtual; abstract;
    Procedure SetValue (const Value : Double); virtual; abstract;

    public
    Property Value : Double read GetValue write SetValue;
  end;
  ADoubleDictionary = class (ADictionary)
    protected
    Function  GetItem (const Key : String) : Double; virtual; abstract;
    Procedure SetItem (const Key : String; const Value : Double); virtual; abstract;

    public
    Property  Item [const Key : String] : Double read GetItem write SetItem; default;
    Procedure Add (const Key : String; const Value : Double); virtual; abstract;
  end;



{                                                                              }
{ AExtendedDictionary                                                          }
{                                                                              }
type
  AExtendedDictionaryIterator = class (ADictionaryIterator)
    protected
    Function  GetValue : Extended; virtual; abstract;
    Procedure SetValue (const Value : Extended); virtual; abstract;

    public
    Property Value : Extended read GetValue write SetValue;
  end;
  AExtendedDictionary = class (ADictionary)
    protected
    Function  GetItem (const Key : String) : Extended; virtual; abstract;
    Procedure SetItem (const Key : String; const Value : Extended); virtual; abstract;

    public
    Property  Item [const Key : String] : Extended read GetItem write SetItem; default;
    Procedure Add (const Key : String; const Value : Extended); virtual; abstract;
  end;



{                                                                              }
{ AStringDictionary                                                            }
{                                                                              }
type
  AStringDictionaryIterator = class (ADictionaryIterator)
    protected
    Function  GetValue : String; virtual; abstract;
    Procedure SetValue (const Value : String); virtual; abstract;

    public
    Property Value : String read GetValue write SetValue;
  end;
  AStringDictionary = class (ADictionary)
    protected
    Function  GetItem (const Key : String) : String; virtual; abstract;
    Procedure SetItem (const Key : String; const Value : String); virtual; abstract;

    public
    Property  Item [const Key : String] : String read GetItem write SetItem; default;
    Procedure Add (const Key : String; const Value : String); virtual; abstract;
  end;





{                                                                              }
{ AStack                                                                       }
{                                                                              }
type
  AStack = class (ACollection)
    { AType interface                                                          }
    Function  IsEmpty : Boolean; override;
  end;



{                                                                              }
{ AIntegerStack                                                                }
{                                                                              }
type
  AIntegerStack = class (AStack)
    Procedure Push (const V : Integer); overload; virtual; abstract;
    Procedure Push (const V : IntegerArray); overload; virtual; abstract;
    Function  Pop : Integer; overload; virtual; abstract;
    Function  Pop (const N : Integer) : IntegerArray; overload; virtual; abstract;
  end;



{                                                                              }
{ AInt64Stack                                                                  }
{                                                                              }
type
  AInt64Stack = class (AStack)
    Procedure Push (const V : Int64); overload; virtual; abstract;
    Procedure Push (const V : Int64Array); overload; virtual; abstract;
    Function  Pop : Int64; overload; virtual; abstract;
    Function  Pop (const N : Integer) : Int64Array; overload; virtual; abstract;
  end;



{                                                                              }
{ ASingleStack                                                                 }
{                                                                              }
type
  ASingleStack = class (AStack)
    Procedure Push (const V : Single); overload; virtual; abstract;
    Procedure Push (const V : SingleArray); overload; virtual; abstract;
    Function  Pop : Single; overload; virtual; abstract;
    Function  Pop (const N : Integer) : SingleArray; overload; virtual; abstract;
  end;



{                                                                              }
{ ADoubleStack                                                                 }
{                                                                              }
type
  ADoubleStack = class (AStack)
    Procedure Push (const V : Double); overload; virtual; abstract;
    Procedure Push (const V : DoubleArray); overload; virtual; abstract;
    Function  Pop : Double; overload; virtual; abstract;
    Function  Pop (const N : Integer) : DoubleArray; overload; virtual; abstract;
  end;



{                                                                              }
{ AExtendedStack                                                               }
{                                                                              }
type
  AExtendedStack = class (AStack)
    Procedure Push (const V : Extended); overload; virtual; abstract;
    Procedure Push (const V : ExtendedArray); overload; virtual; abstract;
    Function  Pop : Extended; overload; virtual; abstract;
    Function  Pop (const N : Integer) : ExtendedArray; overload; virtual; abstract;
  end;



{                                                                              }
{ AStringStack                                                                 }
{                                                                              }
type
  AStringStack = class (AStack)
    Procedure Push (const V : String); overload; virtual; abstract;
    Procedure Push (const V : StringArray); overload; virtual; abstract;
    Function  Pop : String; overload; virtual; abstract;
    Function  Pop (const N : Integer) : StringArray; overload; virtual; abstract;
  end;



{                                                                              }
{ AObjectStack                                                                 }
{                                                                              }
type
  AObjectStack = class (AStack)
    Procedure Push (const V : TObject); overload; virtual; abstract;
    Procedure Push (const V : ObjectArray); overload; virtual; abstract;
    Function  Pop : TObject; overload; virtual; abstract;
    Function  Pop (const N : Integer) : ObjectArray; overload; virtual; abstract;
  end;





{                                                                              }
{ ALinkedList                                                                  }
{   Base class for linked lists                                                }
{                                                                              }
type
  ALinkedListIterator = class (AIterator)
    Procedure InsertBefore; overload; virtual; abstract;
    Procedure InsertAfter; overload; virtual; abstract;
  end;
  ALinkedList = class (ACollection)
    Function  First : ALinkedListIterator; virtual; abstract;
    Function  Last : ALinkedListIterator; virtual; abstract;
    Function  GetCount : Integer; override;
    Procedure AddInFront; overload; virtual;
    Procedure AddToBack; overload; virtual;
    Procedure DeleteFirst; virtual;
  end;
  ADoublyLinkedListIterator = class (ALinkedListIterator)
    Procedure Previous; virtual; abstract;
  end;
  ADoublyLinkedList = class (ACollection)
    Function  First : ADoublyLinkedListIterator; virtual; abstract;
    Function  Last : ADoublyLinkedListIterator; virtual; abstract;
    Function  GetCount : Integer; override;
    Procedure AddInFront; overload; virtual;
    Procedure AddToBack; overload; virtual;
    Procedure DeleteFirst; virtual;
    Procedure DeleteLast; virtual;
  end;


{                                                                              }
{ AIntegerLinkedList                                                           }
{                                                                              }
type
  ALinkedIntegerListIterator = class (ALinkedListIterator)
    protected
    Function  GetValue : Integer; virtual; abstract;
    Procedure SetValue (const Value : Integer); virtual; abstract;

    public
    Property  Value : Integer read GetValue write SetValue;
    Procedure InsertBefore (const Value : Integer); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : Integer); reintroduce; overload; virtual; abstract;
  end;
  ALinkedIntegerList = class (ALinkedList)
    Procedure AddInFront (const Value : Integer); reintroduce; overload; virtual; abstract;
    Procedure AddToBack (const Value : Integer); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedIntegerListIterator = class (ADoublyLinkedListIterator)
    protected
    Function  GetValue : Integer; virtual; abstract;
    Procedure SetValue (const Value : Integer); virtual; abstract;

    public
    Property  Value : Integer read GetValue write SetValue;
    Procedure InsertBefore (const Value : Integer); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : Integer); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedIntegerList = class (ADoublyLinkedList)
    Procedure AddToBack (const Value : Integer); reintroduce; overload; virtual; abstract;
    Procedure AddInFront (const Value : Integer); reintroduce; overload; virtual; abstract;
  end;



{                                                                              }
{ AInt64LinkedList                                                             }
{                                                                              }
type
  ALinkedInt64ListIterator = class (ALinkedListIterator)
    protected
    Function  GetValue : Int64; virtual; abstract;
    Procedure SetValue (const Value : Int64); virtual; abstract;

    public
    Property  Value : Int64 read GetValue write SetValue;
    Procedure InsertBefore (const Value : Int64); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : Int64); reintroduce; overload; virtual; abstract;
  end;
  ALinkedInt64List = class (ALinkedList)
    Procedure AddInFront (const Value : Int64); reintroduce; overload; virtual; abstract;
    Procedure AddToBack (const Value : Int64); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedInt64ListIterator = class (ADoublyLinkedListIterator)
    protected
    Function  GetValue : Int64; virtual; abstract;
    Procedure SetValue (const Value : Int64); virtual; abstract;

    public
    Property  Value : Int64 read GetValue write SetValue;
    Procedure InsertBefore (const Value : Int64); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : Int64); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedInt64List = class (ADoublyLinkedList)
    Procedure AddToBack (const Value : Int64); reintroduce; overload; virtual; abstract;
    Procedure AddInFront (const Value : Int64); reintroduce; overload; virtual; abstract;
  end;



{                                                                              }
{ ASingleLinkedList                                                            }
{                                                                              }
type
  ALinkedSingleListIterator = class (ALinkedListIterator)
    protected
    Function  GetValue : Single; virtual; abstract;
    Procedure SetValue (const Value : Single); virtual; abstract;

    public
    Property  Value : Single read GetValue write SetValue;
    Procedure InsertBefore (const Value : Single); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : Single); reintroduce; overload; virtual; abstract;
  end;
  ALinkedSingleList = class (ALinkedList)
    Procedure AddInFront (const Value : Single); reintroduce; overload; virtual; abstract;
    Procedure AddToBack (const Value : Single); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedSingleListIterator = class (ADoublyLinkedListIterator)
    protected
    Function  GetValue : Single; virtual; abstract;
    Procedure SetValue (const Value : Single); virtual; abstract;

    public
    Property  Value : Single read GetValue write SetValue;
    Procedure InsertBefore (const Value : Single); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : Single); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedSingleList = class (ADoublyLinkedList)
    Procedure AddToBack (const Value : Single); reintroduce; overload; virtual; abstract;
    Procedure AddInFront (const Value : Single); reintroduce; overload; virtual; abstract;
  end;



{                                                                              }
{ ADoubleLinkedList                                                            }
{                                                                              }
type
  ALinkedDoubleListIterator = class (ALinkedListIterator)
    protected
    Function  GetValue : Double; virtual; abstract;
    Procedure SetValue (const Value : Double); virtual; abstract;

    public
    Property  Value : Double read GetValue write SetValue;
    Procedure InsertBefore (const Value : Double); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : Double); reintroduce; overload; virtual; abstract;
  end;
  ALinkedDoubleList = class (ALinkedList)
    Procedure AddInFront (const Value : Double); reintroduce; overload; virtual; abstract;
    Procedure AddToBack (const Value : Double); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedDoubleListIterator = class (ADoublyLinkedListIterator)
    protected
    Function  GetValue : Double; virtual; abstract;
    Procedure SetValue (const Value : Double); virtual; abstract;

    public
    Property  Value : Double read GetValue write SetValue;
    Procedure InsertBefore (const Value : Double); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : Double); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedDoubleList = class (ADoublyLinkedList)
    Procedure AddToBack (const Value : Double); reintroduce; overload; virtual; abstract;
    Procedure AddInFront (const Value : Double); reintroduce; overload; virtual; abstract;
  end;



{                                                                              }
{ AExtendedLinkedList                                                          }
{                                                                              }
type
  ALinkedExtendedListIterator = class (ALinkedListIterator)
    protected
    Function  GetValue : Extended; virtual; abstract;
    Procedure SetValue (const Value : Extended); virtual; abstract;

    public
    Property  Value : Extended read GetValue write SetValue;
    Procedure InsertBefore (const Value : Extended); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : Extended); reintroduce; overload; virtual; abstract;
  end;
  ALinkedExtendedList = class (ALinkedList)
    Procedure AddInFront (const Value : Extended); reintroduce; overload; virtual; abstract;
    Procedure AddToBack (const Value : Extended); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedExtendedListIterator = class (ADoublyLinkedListIterator)
    protected
    Function  GetValue : Extended; virtual; abstract;
    Procedure SetValue (const Value : Extended); virtual; abstract;

    public
    Property  Value : Extended read GetValue write SetValue;
    Procedure InsertBefore (const Value : Extended); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : Extended); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedExtendedList = class (ADoublyLinkedList)
    Procedure AddToBack (const Value : Extended); reintroduce; overload; virtual; abstract;
    Procedure AddInFront (const Value : Extended); reintroduce; overload; virtual; abstract;
  end;



{                                                                              }
{ AStringLinkedList                                                            }
{                                                                              }
type
  ALinkedStringListIterator = class (ALinkedListIterator)
    protected
    Function  GetValue : String; virtual; abstract;
    Procedure SetValue (const Value : String); virtual; abstract;

    public
    Property  Value : String read GetValue write SetValue;
    Procedure InsertBefore (const Value : String); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : String); reintroduce; overload; virtual; abstract;
  end;
  ALinkedStringList = class (ALinkedList)
    Procedure AddInFront (const Value : String); reintroduce; overload; virtual; abstract;
    Procedure AddToBack (const Value : String); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedStringListIterator = class (ADoublyLinkedListIterator)
    protected
    Function  GetValue : String; virtual; abstract;
    Procedure SetValue (const Value : String); virtual; abstract;

    public
    Property  Value : String read GetValue write SetValue;
    Procedure InsertBefore (const Value : String); reintroduce; overload; virtual; abstract;
    Procedure InsertAfter (const Value : String); reintroduce; overload; virtual; abstract;
  end;
  ADoublyLinkedStringList = class (ADoublyLinkedList)
    Procedure AddToBack (const Value : String); reintroduce; overload; virtual; abstract;
    Procedure AddInFront (const Value : String); reintroduce; overload; virtual; abstract;
  end;




{                                                                              }
{ AIntegerQueue                                                                }
{                                                                              }
type
  AIntegerQueue = class (ACollection)
    Procedure Add (const V : Integer); overload; virtual; abstract;
    Procedure Add (const V : IntegerArray); overload; virtual;
    Function  Remove : Integer; overload; virtual; abstract;
    Function  Remove (const N : Integer) : IntegerArray; overload; virtual;
  end;



{                                                                              }
{ AInt64Queue                                                                  }
{                                                                              }
type
  AInt64Queue = class (ACollection)
    Procedure Add (const V : Int64); overload; virtual; abstract;
    Procedure Add (const V : Int64Array); overload; virtual;
    Function  Remove : Int64; overload; virtual; abstract;
    Function  Remove (const N : Integer) : Int64Array; overload; virtual;
  end;



{                                                                              }
{ ASingleQueue                                                                 }
{                                                                              }
type
  ASingleQueue = class (ACollection)
    Procedure Add (const V : Single); overload; virtual; abstract;
    Procedure Add (const V : SingleArray); overload; virtual;
    Function  Remove : Single; overload; virtual; abstract;
    Function  Remove (const N : Integer) : SingleArray; overload; virtual;
  end;



{                                                                              }
{ ADoubleQueue                                                                 }
{                                                                              }
type
  ADoubleQueue = class (ACollection)
    Procedure Add (const V : Double); overload; virtual; abstract;
    Procedure Add (const V : DoubleArray); overload; virtual;
    Function  Remove : Double; overload; virtual; abstract;
    Function  Remove (const N : Integer) : DoubleArray; overload; virtual;
  end;



{                                                                              }
{ AExtendedQueue                                                               }
{                                                                              }
type
  AExtendedQueue = class (ACollection)
    Procedure Add (const V : Extended); overload; virtual; abstract;
    Procedure Add (const V : ExtendedArray); overload; virtual;
    Function  Remove : Extended; overload; virtual; abstract;
    Function  Remove (const N : Integer) : ExtendedArray; overload; virtual;
  end;



{                                                                              }
{ AStringQueue                                                                 }
{                                                                              }
type
  AStringQueue = class (ACollection)
    Procedure Add (const V : String); overload; virtual; abstract;
    Procedure Add (const V : StringArray); overload; virtual;
    Function  Remove : String; overload; virtual; abstract;
    Function  Remove (const N : Integer) : StringArray; overload; virtual;
  end;





{                                                                              }
{ AAllocator                                                                   }
{                                                                              }
type
  AAllocator = class (TStreamer)
    Function  Get (const Size : Integer) : Int64; virtual; abstract;
    Procedure Release (const Offset : Int64); virtual; abstract;
    Procedure Resize (var Offset : Int64; const NewSize : Integer); virtual; abstract;
    Function  BlockSize (const Offset : Int64) : Integer; virtual; abstract;

    Procedure Read (const Offset : Int64; var Buf); virtual; abstract;
    Procedure Write (const Offset : Int64; const Buf); virtual; abstract;

    Procedure Pack; virtual; abstract;
  end;



implementation

uses
  // Delphi units
  Math,

  // Delphi Fundamentals (L0)
  cStrings;



{                                                                              }
{ AType                                                                        }
{                                                                              }
Function AType.GetAsString : String;
  Begin
    raise EType.Create (ClassName + ' cannot convert to string');
  End;

Procedure AType.SetAsString (const S : String);
  Begin
    raise EType.Create (ClassName + ' cannot convert from string');
  End;

Function AType.Duplicate : AType;
  Begin
    Result := CreateInstance;
    Result.Assign (self);
  End;

Procedure AType.StreamOut (const S : TExStream);
  Begin
    S.Write (AsString);
  End;

Procedure AType.StreamIn (const S : TExStream);
  Begin
    raise EType.Create (ClassName + ' can not stream in');
  End;

Procedure AType.Assign (const Source : AType);
  Begin
    try
      Source.AssignTo (self);
    except
      raise EType.Create (ClassName + ' can not assign from ' + Source.ClassName);
    end;
  End;

Procedure AType.AssignTo (const Dest : AType);
  Begin
    raise EType.Create (ClassName + ' can not assign to ' + Dest.ClassName);
  End;

Function AType.IsEqual (const V : AType) : Boolean;
  Begin
    raise EType.Create (ClassName + ' can not compare with ' + V.ClassName);
  End;

Function AType.Compare (const V : AType) : TCompareResult;
  Begin
    raise EType.Create (ClassName + ' can not compare with ' + V.ClassName);
  End;


{                                                                              }
{ ACollection                                                                  }
{                                                                              }
Function ACollection.GetCount : Integer;
var I : AIterator;
  Begin
    I := Iterate;
    try
      Result := 0;
      While not I.EOF do
        begin
          Inc (Result);
          I.Next;
        end;
    finally
      FreeAndNil (I);
    end;
  End;

{                                                                              }
{ ALinkedList                                                                  }
{                                                                              }
Function ALinkedList.GetCount : Integer;
var I : ALinkedListIterator;
  Begin
    I := First;
    try
      Result := 0;
      While not I.EOF do
        begin
          Inc (Result);
          I.Next;
        end;
    finally
      FreeAndNil (I);
    end;
  End;

Procedure ALinkedList.AddInFront;
var I : ALinkedListIterator;
  Begin
    I := First;
    try
      I.InsertBefore;
    finally
      FreeAndNil (I);
    end;
  End;

Procedure ALinkedList.AddToBack;
var I : ALinkedListIterator;
  Begin
    I := Last;
    try
      I.InsertAfter;
    finally
      FreeAndNil (I);
    end;
  End;

Procedure ALinkedList.DeleteFirst;
var I : ALinkedListIterator;
  Begin
    I := First;
    try
      I.Delete;
    finally
      FreeAndNil (I);
    end;
  End;



{                                                                              }
{ ADoublyLinkedList                                                            }
{                                                                              }
Function ADoublyLinkedList.GetCount : Integer;
var I : ADoublyLinkedListIterator;
  Begin
    I := First;
    try
      Result := 0;
      While not I.EOF do
        begin
          Inc (Result);
          I.Next;
        end;
    finally
      FreeAndNil (I);
    end;
  End;

Procedure ADoublyLinkedList.AddInFront;
var I : ADoublyLinkedListIterator;
  Begin
    I := First;
    try
      I.InsertBefore;
    finally
      FreeAndNil (I);
    end;
  End;

Procedure ADoublyLinkedList.AddToBack;
var I : ADoublyLinkedListIterator;
  Begin
    I := Last;
    try
      I.InsertAfter;
    finally
      FreeAndNil (I);
    end;
  End;

Procedure ADoublyLinkedList.DeleteFirst;
var I : ADoublyLinkedListIterator;
  Begin
    I := First;
    try
      I.Delete;
    finally
      FreeAndNil (I);
    end;
  End;




Procedure ADoublyLinkedList.DeleteLast;
var I : ADoublyLinkedListIterator;
  Begin
    I := Last;
    try
      I.Delete;
    finally
      FreeAndNil (I);
    end;
  End;


{                                                                              }
{ AArray                                                                       }
{                                                                              }
Procedure AArray.Clear;
  Begin
    Count := 0;
  End;

class Function AArray.IsOrdered : Boolean;
  Begin
    Result := True;
  End;

Procedure AArray.Sort;

  Procedure QuickSort (L, R : Integer);
  var I, J : Integer;
      M    : Integer;
    Begin
      Repeat
        I := L;
        J := R;
        M := (L + R) shr 1;
        Repeat
          While CompareItems (I, M) = crLess do
            Inc (I);
          While CompareItems (J, M) = crGreater do
            Dec (J);
          if I <= J then
            begin
              ExchangeItems (I, J);
              Inc (I);
              Dec (J);
            end;
        Until I > J;
        if L < J then
          QuickSort (L, J);
        L := I;
      Until I >= R;
    End;

var I : Integer;
  Begin
    I := Count;
    if I > 0 then
      QuickSort (0, I - 1);
  End;

Procedure AArray.ReverseOrder;
var I, L : Integer;
  Begin
    L := Count;
    For I := 1 to L div 2 do
      ExchangeItems (I - 1, L - I);
  End;



{                                                                              }
{ AIntegerArray                                                                }
{                                                                              }
Procedure AIntegerArray.Assign (const V : AType);
var I, L : Integer;
  Begin
    if not (V is AIntegerArray) then
      raise EIntegerArray.Create ('AIntegerArray can not assign from ' + V.ClassName);
    L := AArray (V).Count;
    Count := L;
    For I := 0 to L - 1 do
      Item [I] := AIntegerArray (V) [I];
  End;


Function AIntegerArray.IsEqual (const V : AType) : Boolean;
var I, L : Integer;
  Begin
    if not (V is AIntegerArray) then
      raise EIntegerArray.Create ('AIntegerArray can not compare with ' + V.ClassName);

    L := AArray (V).Count;
    if Count <> L then
      begin
        Result := False;
        exit;
      end;
    For I := 0 to L - 1 do
      if Item [I] <> AIntegerArray (V) [I] then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Procedure AIntegerArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Integer;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function AIntegerArray.Add (const Value : Integer) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function AIntegerArray.Add (const V : IntegerArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function AIntegerArray.Add (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if not (V is AIntegerArray) then
      raise EIntegerArray.Create ('AIntegerArray can not append ' + V.ClassName);

    Result := Count;
    L := V.Count;
    Count := Result + L;
    For I := 0 to L - 1 do
      Item [Result + I] := AIntegerArray (V) [I];
  End;

Function AIntegerArray.PosNext (const Find : Integer; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Integer;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function AIntegerArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : Integer;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Procedure AIntegerArray.Fill (const Idx, Count : Integer; const Value : Integer);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Procedure AIntegerArray.FillInc (const Idx, Count : Integer; const StartValue : Integer; const Increment : Integer);
var I : Integer;
    J : Integer;
  Begin
    J := StartValue;
    For I := Idx to Idx + Count - 1 do
      begin
        Item [I] := J;
        J := J + Increment;
      end;
  End;

Function AIntegerArray.GetRange (const LoIdx, HiIdx : Integer) : IntegerArray;
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := H - L  + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [L + I] := Item [I];
  End;

Procedure AIntegerArray.SetRange (const LoIdx, HiIdx : Integer; const V : IntegerArray);
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := Min (Length (V), H - L  + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Function AIntegerArray.GetAsString : String;
var I : Integer;
  Begin
    Result := '(';
    For I := 0 to Count - 1 do
      Result := Result + Cond (I > 0, ',', '') + IntToStr (Item [I]);
    Result := Result + ')';
  End;

Procedure AIntegerArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if (L < 2) or (S [1] <> '(') or (S [L] <> ')') then
      raise EIntegerArray.Create ('Invalid string format');

    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := 0 else
          Item [L - 1] := StrToInt (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;



{                                                                              }
{ AInt64Array                                                                  }
{                                                                              }
Procedure AInt64Array.Assign (const V : AType);
var I, L : Integer;
  Begin
    if not (V is AInt64Array) then
      raise EInt64Array.Create ('AInt64Array can not assign from ' + V.ClassName);
    L := AArray (V).Count;
    Count := L;
    For I := 0 to L - 1 do
      Item [I] := AInt64Array (V) [I];
  End;


Function AInt64Array.IsEqual (const V : AType) : Boolean;
var I, L : Integer;
  Begin
    if not (V is AInt64Array) then
      raise EInt64Array.Create ('AInt64Array can not compare with ' + V.ClassName);

    L := AArray (V).Count;
    if Count <> L then
      begin
        Result := False;
        exit;
      end;
    For I := 0 to L - 1 do
      if Item [I] <> AInt64Array (V) [I] then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Procedure AInt64Array.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Int64;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function AInt64Array.Add (const Value : Int64) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function AInt64Array.Add (const V : Int64Array) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function AInt64Array.Add (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if not (V is AInt64Array) then
      raise EIntegerArray.Create ('AInt64Array can not append ' + V.ClassName);

    Result := Count;
    L := V.Count;
    Count := Result + L;
    For I := 0 to L - 1 do
      Item [Result + I] := AInt64Array (V) [I];
  End;

Function AInt64Array.PosNext (const Find : Int64; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Int64;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function AInt64Array.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : Int64;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Procedure AInt64Array.Fill (const Idx, Count : Integer; const Value : Int64);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Procedure AInt64Array.FillInc (const Idx, Count : Integer; const StartValue : Int64; const Increment : Int64);
var I : Integer;
    J : Int64;
  Begin
    J := StartValue;
    For I := Idx to Idx + Count - 1 do
      begin
        Item [I] := J;
        J := J + Increment;
      end;
  End;

Function AInt64Array.GetRange (const LoIdx, HiIdx : Integer) : Int64Array;
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := H - L  + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [L + I] := Item [I];
  End;

Procedure AInt64Array.SetRange (const LoIdx, HiIdx : Integer; const V : Int64Array);
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := Min (Length (V), H - L  + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Function AInt64Array.GetAsString : String;
var I : Integer;
  Begin
    Result := '(';
    For I := 0 to Count - 1 do
      Result := Result + Cond (I > 0, ',', '') + IntToStr (Item [I]);
    Result := Result + ')';
  End;

Procedure AInt64Array.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if (L < 2) or (S [1] <> '(') or (S [L] <> ')') then
      raise EInt64Array.Create ('Invalid string format');

    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := 0 else
          Item [L - 1] := StrToInt (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;



{                                                                              }
{ ASingleArray                                                                 }
{                                                                              }
Procedure ASingleArray.Assign (const V : AType);
var I, L : Integer;
  Begin
    if not ((V is ASingleArray) or (V is ADoubleArray) or (V is AExtendedArray) or
            (V is AIntegerArray) or (V is AInt64Array)) then
      raise EExtendedArray.Create ('ASingleArray can not assign from ' + V.ClassName);

    L := AArray (V).Count;
    Count := L;
    if V is ASingleArray then
      For I := 0 to L - 1 do
        Item [I] := ASingleArray (V) [I] else
    if V is ADoubleArray then
      For I := 0 to L - 1 do
        Item [I] := ADoubleArray (V) [I] else
    if V is AExtendedArray then
      For I := 0 to L - 1 do
        Item [I] := AExtendedArray (V) [I] else
    if V is AIntegerArray then
      For I := 0 to L - 1 do
        Item [I] := AIntegerArray (V) [I] else
      For I := 0 to L - 1 do
        Item [I] := AInt64Array (V) [I]
  End;


Function ASingleArray.IsEqual (const V : AType) : Boolean;
var I, L : Integer;
  Begin
    if not (V is ASingleArray) then
      raise ESingleArray.Create ('ASingleArray can not compare with ' + V.ClassName);

    L := AArray (V).Count;
    if Count <> L then
      begin
        Result := False;
        exit;
      end;
    For I := 0 to L - 1 do
      if Item [I] <> ASingleArray (V) [I] then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Procedure ASingleArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Single;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function ASingleArray.Add (const Value : Single) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function ASingleArray.Add (const V : SingleArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function ASingleArray.Add (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if not (V is ASingleArray) then
      raise EIntegerArray.Create ('ASingleArray can not append ' + V.ClassName);

    Result := Count;
    L := V.Count;
    Count := Result + L;
    For I := 0 to L - 1 do
      Item [Result + I] := ASingleArray (V) [I];
  End;

Function ASingleArray.PosNext (const Find : Single; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Single;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function ASingleArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : Single;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Procedure ASingleArray.Fill (const Idx, Count : Integer; const Value : Single);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Procedure ASingleArray.FillInc (const Idx, Count : Integer; const StartValue : Single; const Increment : Single);
var I : Integer;
    J : Single;
  Begin
    J := StartValue;
    For I := Idx to Idx + Count - 1 do
      begin
        Item [I] := J;
        J := J + Increment;
      end;
  End;

Function ASingleArray.GetRange (const LoIdx, HiIdx : Integer) : SingleArray;
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := H - L  + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [L + I] := Item [I];
  End;

Procedure ASingleArray.SetRange (const LoIdx, HiIdx : Integer; const V : SingleArray);
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := Min (Length (V), H - L  + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Function ASingleArray.GetAsString : String;
var I : Integer;
  Begin
    Result := '(';
    For I := 0 to Count - 1 do
      Result := Result + Cond (I > 0, ',', '') + FloatToStr (Item [I]);
    Result := Result + ')';
  End;

Procedure ASingleArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if (L < 2) or (S [1] <> '(') or (S [L] <> ')') then
      raise ESingleArray.Create ('Invalid string format');

    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := 0.0 else
          Item [L - 1] := StrToFloat (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;



{                                                                              }
{ ADoubleArray                                                                 }
{                                                                              }
Procedure ADoubleArray.Assign (const V : AType);
var I, L : Integer;
  Begin
    if not ((V is ASingleArray) or (V is ADoubleArray) or (V is AExtendedArray) or
            (V is AIntegerArray) or (V is AInt64Array)) then
      raise EExtendedArray.Create ('ADoubleArray can not assign from ' + V.ClassName);

    L := AArray (V).Count;
    Count := L;
    if V is ASingleArray then
      For I := 0 to L - 1 do
        Item [I] := ASingleArray (V) [I] else
    if V is ADoubleArray then
      For I := 0 to L - 1 do
        Item [I] := ADoubleArray (V) [I] else
    if V is AExtendedArray then
      For I := 0 to L - 1 do
        Item [I] := AExtendedArray (V) [I] else
    if V is AIntegerArray then
      For I := 0 to L - 1 do
        Item [I] := AIntegerArray (V) [I] else
      For I := 0 to L - 1 do
        Item [I] := AInt64Array (V) [I]
  End;


Function ADoubleArray.IsEqual (const V : AType) : Boolean;
var I, L : Integer;
  Begin
    if not (V is ADoubleArray) then
      raise EDoubleArray.Create ('ADoubleArray can not compare with ' + V.ClassName);

    L := AArray (V).Count;
    if Count <> L then
      begin
        Result := False;
        exit;
      end;
    For I := 0 to L - 1 do
      if Item [I] <> ADoubleArray (V) [I] then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Procedure ADoubleArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Double;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function ADoubleArray.Add (const Value : Double) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function ADoubleArray.Add (const V : DoubleArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function ADoubleArray.Add (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if not (V is ADoubleArray) then
      raise EIntegerArray.Create ('ADoubleArray can not append ' + V.ClassName);

    Result := Count;
    L := V.Count;
    Count := Result + L;
    For I := 0 to L - 1 do
      Item [Result + I] := ADoubleArray (V) [I];
  End;

Function ADoubleArray.PosNext (const Find : Double; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Double;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function ADoubleArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : Double;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Procedure ADoubleArray.Fill (const Idx, Count : Integer; const Value : Double);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Procedure ADoubleArray.FillInc (const Idx, Count : Integer; const StartValue : Double; const Increment : Double);
var I : Integer;
    J : Double;
  Begin
    J := StartValue;
    For I := Idx to Idx + Count - 1 do
      begin
        Item [I] := J;
        J := J + Increment;
      end;
  End;

Function ADoubleArray.GetRange (const LoIdx, HiIdx : Integer) : DoubleArray;
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := H - L  + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [L + I] := Item [I];
  End;

Procedure ADoubleArray.SetRange (const LoIdx, HiIdx : Integer; const V : DoubleArray);
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := Min (Length (V), H - L  + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Function ADoubleArray.GetAsString : String;
var I : Integer;
  Begin
    Result := '(';
    For I := 0 to Count - 1 do
      Result := Result + Cond (I > 0, ',', '') + FloatToStr (Item [I]);
    Result := Result + ')';
  End;

Procedure ADoubleArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if (L < 2) or (S [1] <> '(') or (S [L] <> ')') then
      raise EDoubleArray.Create ('Invalid string format');

    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := 0.0 else
          Item [L - 1] := StrToFloat (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;



{                                                                              }
{ AExtendedArray                                                               }
{                                                                              }
Procedure AExtendedArray.Assign (const V : AType);
var I, L : Integer;
  Begin
    if not ((V is ASingleArray) or (V is ADoubleArray) or (V is AExtendedArray) or
            (V is AIntegerArray) or (V is AInt64Array)) then
      raise EExtendedArray.Create ('AExtendedArray can not assign from ' + V.ClassName);

    L := AArray (V).Count;
    Count := L;
    if V is ASingleArray then
      For I := 0 to L - 1 do
        Item [I] := ASingleArray (V) [I] else
    if V is ADoubleArray then
      For I := 0 to L - 1 do
        Item [I] := ADoubleArray (V) [I] else
    if V is AExtendedArray then
      For I := 0 to L - 1 do
        Item [I] := AExtendedArray (V) [I] else
    if V is AIntegerArray then
      For I := 0 to L - 1 do
        Item [I] := AIntegerArray (V) [I] else
      For I := 0 to L - 1 do
        Item [I] := AInt64Array (V) [I]
  End;


Function AExtendedArray.IsEqual (const V : AType) : Boolean;
var I, L : Integer;
  Begin
    if not (V is AExtendedArray) then
      raise EExtendedArray.Create ('AExtendedArray can not compare with ' + V.ClassName);

    L := AArray (V).Count;
    if Count <> L then
      begin
        Result := False;
        exit;
      end;
    For I := 0 to L - 1 do
      if Item [I] <> AExtendedArray (V) [I] then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Procedure AExtendedArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Extended;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function AExtendedArray.Add (const Value : Extended) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function AExtendedArray.Add (const V : ExtendedArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function AExtendedArray.Add (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if not (V is AExtendedArray) then
      raise EIntegerArray.Create ('AExtendedArray can not append ' + V.ClassName);

    Result := Count;
    L := V.Count;
    Count := Result + L;
    For I := 0 to L - 1 do
      Item [Result + I] := AExtendedArray (V) [I];
  End;

Function AExtendedArray.PosNext (const Find : Extended; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Extended;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function AExtendedArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : Extended;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Procedure AExtendedArray.Fill (const Idx, Count : Integer; const Value : Extended);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;

Procedure AExtendedArray.FillInc (const Idx, Count : Integer; const StartValue : Extended; const Increment : Extended);
var I : Integer;
    J : Extended;
  Begin
    J := StartValue;
    For I := Idx to Idx + Count - 1 do
      begin
        Item [I] := J;
        J := J + Increment;
      end;
  End;

Function AExtendedArray.GetRange (const LoIdx, HiIdx : Integer) : ExtendedArray;
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := H - L  + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [L + I] := Item [I];
  End;

Procedure AExtendedArray.SetRange (const LoIdx, HiIdx : Integer; const V : ExtendedArray);
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := Min (Length (V), H - L  + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Function AExtendedArray.GetAsString : String;
var I : Integer;
  Begin
    Result := '(';
    For I := 0 to Count - 1 do
      Result := Result + Cond (I > 0, ',', '') + FloatToStr (Item [I]);
    Result := Result + ')';
  End;

Procedure AExtendedArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if (L < 2) or (S [1] <> '(') or (S [L] <> ')') then
      raise EExtendedArray.Create ('Invalid string format');

    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := 0.0 else
          Item [L - 1] := StrToFloat (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;



{                                                                              }
{ AStringArray                                                                 }
{                                                                              }
Procedure AStringArray.Assign (const V : AType);
var I, L : Integer;
  Begin
    if not (V is AStringArray) then
      raise EStringArray.Create ('AStringArray can not assign from ' + V.ClassName);
    L := AArray (V).Count;
    Count := L;
    For I := 0 to L - 1 do
      Item [I] := AStringArray (V) [I];
  End;


Function AStringArray.IsEqual (const V : AType) : Boolean;
var I, L : Integer;
  Begin
    if not (V is AStringArray) then
      raise EStringArray.Create ('AStringArray can not compare with ' + V.ClassName);

    L := AArray (V).Count;
    if Count <> L then
      begin
        Result := False;
        exit;
      end;
    For I := 0 to L - 1 do
      if Item [I] <> AStringArray (V) [I] then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Procedure AStringArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : String;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function AStringArray.Add (const Value : String) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function AStringArray.Add (const V : StringArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function AStringArray.Add (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if not (V is AStringArray) then
      raise EIntegerArray.Create ('AStringArray can not append ' + V.ClassName);

    Result := Count;
    L := V.Count;
    Count := Result + L;
    For I := 0 to L - 1 do
      Item [Result + I] := AStringArray (V) [I];
  End;

Function AStringArray.PosNext (const Find : String; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : String;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else // linear search
      begin
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;

Function AStringArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : String;
  Begin
    I := Item [Idx1];
    J := Item [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Procedure AStringArray.Fill (const Idx, Count : Integer; const Value : String);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Item [I] := Value;
  End;


Function AStringArray.GetRange (const LoIdx, HiIdx : Integer) : StringArray;
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := H - L  + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [L + I] := Item [I];
  End;

Procedure AStringArray.SetRange (const LoIdx, HiIdx : Integer; const V : StringArray);
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := Min (Length (V), H - L  + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I];
  End;

Function AStringArray.GetAsString : String;
var I : Integer;
  Begin
    Result := '(';
    For I := 0 to Count - 1 do
      Result := Result + Cond (I > 0, ',', '') + UnquoteText (Item [I]);
    Result := Result + ')';
  End;

Procedure AStringArray.SetAsString (const S : String);
var F, G, L, C : Integer;
  Begin
    L := Length (S);
    if (L < 2) or (S [1] <> '(') or (S [L] <> ')') then
      raise EStringArray.Create ('Invalid string format');

    L := 0;
    F := 2;
    C := Length (S);
    While F < C do
      begin
        G := 0;
        While (F + G < C) and (S [F + G] <> ',') do
          Inc (G);
        Inc (L);
        Count := L;
        if G = 0 then
          Item [L - 1] := '' else
          Item [L - 1] := QuoteText (Copy (S, F, G));
        Inc (F, G + 1);
      end;
  End;

Function AStringArray.GetAsStringList : TStringList;
var I : Integer;
  Begin
    Result := TStringList.Create;
    For I := 0 to Count - 1 do
      Result.Add (Item [I]);
  End;

Procedure AStringArray.SetAsStringList (const V : TStringList);
var I, C : Integer;
  Begin
    C := V.Count;
    Count := C;
    For I := 0 to C - 1 do
      Item [I] := V [I];
  End;


  
{                                                                              }
{ ABitArray                                                                    }
{                                                                              }
Procedure ABitArray.Fill (const Idx, Count : Integer; const Value : Boolean);
var I : Integer;
  Begin
    For I := Idx to Idx + Count - 1 do
      Bit [I] := Value;
  End;

Function ABitArray.CompareRange (const LoIdx, HiIdx : Integer; const Value : Boolean) : Boolean;
var I : Integer;
  Begin
    For I := LoIdx to HiIdx do
      if Bit [I] <> Value then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Procedure ABitArray.Assign (const V : AType);
var I, L : Integer;
  Begin
    if not (V is ABitArray) then
      raise EBitArray.Create ('ABitArray can not assign from ' + V.ClassName);

    L := AArray (V).Count;
    Count := L;
    For I := 0 to L - 1 do
      Bit [I] := ABitArray (V) [I];
  End;

Function ABitArray.IsEqual (const V : AType) : Boolean;
var I, L : Integer;
  Begin
    if not (V is ABitArray) then
      raise EInt64Array.Create ('ABitArray can not compare with ' + V.ClassName);

    L := AArray (V).Count;
    if Count <> L then
      begin
        Result := False;
        exit;
      end;
    For I := 0 to L - 1 do
      if Bit [I] <> ABitArray (V) [I] then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Procedure ABitArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : Boolean;
  Begin
    I := Bit [Idx1];
    Bit [Idx1] := Bit [Idx2];
    Bit [Idx2] := I;
  End;

Function ABitArray.Add (const Value : Boolean) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Bit [Result] := Value;
  End;

Function ABitArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
var I, J : Boolean;
  Begin
    I := Bit [Idx1];
    J := Bit [Idx2];
    if I < J then
      Result := crLess else
    if I > J then
      Result := crGreater else
      Result := crEqual;
  End;

Procedure ABitArray.Invert;
var I : Integer;
  Begin
    For I := 0 to Count - 1 do
      Bit [I] := not Bit [I];
  End;

Function ABitArray.Find (const Value : Boolean; const Start : Integer; const FindForward : Boolean) : Integer;
var I, C : Integer;
  Begin
    I := Start;
    if FindForward then
      begin
        C := Count;
        While (I < C) and (Bit [I] <> Value) do
          Inc (I);
        if I = C then
          Result := -1 else
          Result := I;
      end else
      begin
        While (I >= 0) and (Bit [I] <> Value) do
          Dec (I);
        Result := I;
      end;
  End;

Function ABitArray.FindRange (const Value : Boolean; const Start : Integer; const Count : Integer; const FindForward : Boolean) : Integer;
var I, C, F : Integer;
  Begin
    I := Start;
    if FindForward then
      begin
        C := self.Count;
        F := 0;
        While (I + F < C) and (F < Count) do
          if Bit [I + F] = Value then
            Inc (F) else
            begin
              Inc (I, F + 1);
              F := 0;
            end;
        if F < Count then
          Result := -1 else
          Result := I;
      end else
      begin
        F := 0;
        While (I - F >= 0) and (F < Count) do
          if Bit [I - F] = Value then
            Inc (F) else
            begin
              Dec (I, F + 1);
              F := 0;
            end;
        if F < Count then
          Result := -1 else
          Result := I - F + 1;
      end;
  End;

Function ABitArray.GetRangeAsCardinal (const Idx : Integer) : Cardinal;
var I : Integer;
  Begin
    Result := 0;
    For I := 0 to BitsPerCardinal - 1 do
      if Bit [Idx + I] then
        Result := Result or (1 shl I);
  End;

Procedure ABitArray.SetRangeAsCardinal (const Idx : Integer; const Value : Cardinal);
var I : Integer;
    C : Cardinal;
  Begin
    C := 1;
    For I := 0 to BitsPerCardinal - 1 do
      begin
        if Value and C <> 0 then
          Bit [Idx + I] := True else
          Bit [Idx + I] := False;
        C := C shl 1;
      end;
  End;



{                                                                              }
{ AObjectArray                                                                 }
{                                                                              }
Procedure AObjectArray.Assign (const V : AType);
var I, L : Integer;
  Begin
    if not (V is AObjectArray) then
      raise EObjectArray.Create ('AObjectArray can not assign from ' + V.ClassName);

    L := AArray (V).Count;
    Count := L;
    For I := 0 to L - 1 do
      Item [I] := AObjectArray (V) [I];
  End;

Function AObjectArray.IsEqual (const V : AType) : Boolean;
var I, L : Integer;
  Begin
    if not (V is AObjectArray) then
      raise EObjectArray.Create ('AObjectArray can not compare with ' + V.ClassName);

    L := AArray (V).Count;
    if Count <> L then
      begin
        Result := False;
        exit;
      end;
    For I := 0 to L - 1 do
      if Item [I] = AObjectArray (V) [I] then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

Procedure AObjectArray.ExchangeItems (const Idx1, Idx2 : Integer);
var I : TObject;
  Begin
    I := Item [Idx1];
    Item [Idx1] := Item [Idx2];
    Item [Idx2] := I;
  End;

Function AObjectArray.Add (const Value : TObject) : Integer;
  Begin
    Result := Count;
    Count := Result + 1;
    Item [Result] := Value;
  End;

Function AObjectArray.Add (const V : ObjectArray) : Integer;
  Begin
    Result := Count;
    Count := Result + Length (V);
    Range [Result, Count - 1] := V;
  End;

Function AObjectArray.Add (const V : AArray) : Integer;
var I, L : Integer;
  Begin
    if not (V is AObjectArray) then
      raise EObjectArray.Create ('AObjectArray can not append ' + V.ClassName);

    Result := Count;
    L := V.Count;
    Count := Result + L;
    For I := 0 to L - 1 do
      Item [Result + I] := AObjectArray (V) [I];
  End;

Function AObjectArray.PosNext (const Find : TObject; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I : Integer;
  Begin
    For I := Max (PrevPos + 1, 0) to Count - 1 do
      if Find = Item [I] then
        begin
          Result := I;
          exit;
        end;
    Result := -1;
  End;

Function AObjectArray.CompareItems (const Idx1, Idx2 : Integer) : TCompareResult;
  Begin
    Result := cUtils.Compare (Item [Idx1], Item [Idx2]);
  End;

Function AObjectArray.GetRange (const LoIdx, HiIdx : Integer) : ObjectArray;
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := H - L  + 1;
    SetLength (Result, C);
    For I := 0 to C - 1 do
      Result [L + I] := Item [I];
  End;

Procedure AObjectArray.SetRange (const LoIdx, HiIdx : Integer; const V : ObjectArray);
var I, L, H, C : Integer;
  Begin
    L := Max (0, LoIdx);
    H := Min (Count - 1, HiIdx);
    C := Min (Length (V), H - L  + 1);
    For I := 0 to C - 1 do
      Item [L + I] := V [I] as AType;
  End;

Function AObjectArray.GetAsString : String;
var I : Integer;
  Begin
    Result := '(';
    For I := 0 to Count - 1 do
      Result := Result + Cond (I > 0, ',', '') + '$' + IntToHex (Integer (Item [I]), Sizeof (Integer) * 2);
    Result := Result + ')';
  End;



{                                                                              }
{ AStack                                                                       }
{                                                                              }
Function AStack.IsEmpty : Boolean;
  Begin
    Result := Count = 0;
  End;



{                                                                              }
{ AIntegerQueue                                                                }
{                                                                              }
Procedure AIntegerQueue.Add (const V : IntegerArray); 
var I : Integer;
  Begin
    For I := 0 to Length (V) - 1 do
      Add (V [I]);
  End;

Function AIntegerQueue.Remove (const N : Integer) : IntegerArray;
var I : Integer;
  Begin
    SetLength (Result, N);
    For I := 0 to N - 1 do
      Result [I] := Remove;
  End;



{                                                                              }
{ AInt64Queue                                                                  }
{                                                                              }
Procedure AInt64Queue.Add (const V : Int64Array); 
var I : Integer;
  Begin
    For I := 0 to Length (V) - 1 do
      Add (V [I]);
  End;

Function AInt64Queue.Remove (const N : Integer) : Int64Array;
var I : Integer;
  Begin
    SetLength (Result, N);
    For I := 0 to N - 1 do
      Result [I] := Remove;
  End;



{                                                                              }
{ ASingleQueue                                                                 }
{                                                                              }
Procedure ASingleQueue.Add (const V : SingleArray); 
var I : Integer;
  Begin
    For I := 0 to Length (V) - 1 do
      Add (V [I]);
  End;

Function ASingleQueue.Remove (const N : Integer) : SingleArray;
var I : Integer;
  Begin
    SetLength (Result, N);
    For I := 0 to N - 1 do
      Result [I] := Remove;
  End;



{                                                                              }
{ ADoubleQueue                                                                 }
{                                                                              }
Procedure ADoubleQueue.Add (const V : DoubleArray); 
var I : Integer;
  Begin
    For I := 0 to Length (V) - 1 do
      Add (V [I]);
  End;

Function ADoubleQueue.Remove (const N : Integer) : DoubleArray;
var I : Integer;
  Begin
    SetLength (Result, N);
    For I := 0 to N - 1 do
      Result [I] := Remove;
  End;



{                                                                              }
{ AExtendedQueue                                                               }
{                                                                              }
Procedure AExtendedQueue.Add (const V : ExtendedArray); 
var I : Integer;
  Begin
    For I := 0 to Length (V) - 1 do
      Add (V [I]);
  End;

Function AExtendedQueue.Remove (const N : Integer) : ExtendedArray;
var I : Integer;
  Begin
    SetLength (Result, N);
    For I := 0 to N - 1 do
      Result [I] := Remove;
  End;



{                                                                              }
{ AStringQueue                                                                 }
{                                                                              }
Procedure AStringQueue.Add (const V : StringArray); 
var I : Integer;
  Begin
    For I := 0 to Length (V) - 1 do
      Add (V [I]);
  End;

Function AStringQueue.Remove (const N : Integer) : StringArray;
var I : Integer;
  Begin
    SetLength (Result, N);
    For I := 0 to N - 1 do
      Result [I] := Remove;
  End;





end.




AIterator
    Procedure First; virtual; abstract;
    Procedure Last; virtual; abstract;







