{$INCLUDE cHeader.inc}
unit cStreams;

{                                                                              }
{                       Streams and Streamers v0.29 (L0)                       }
{                                                                              }
{        This unit is copyright  2000 by David Butler (david@e.co.za)         }
{                                                                              }
{                  This unit is part of Delphi Fundamentals.                   }
{                   It's original file name is cStreams.pas                    }
{                      It was generated 5 Jul 2000 02:10.                      }
{                                                                              }
{                I invite you to use this unit, free of charge.                }
{        I invite you to distibute this unit, but it must be for free.         }
{             I also invite you to contribute to its development,              }
{             but do not distribute a modified copy of this file.              }
{       Send modifications, suggestions and bug reports to david@e.co.za       }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   01/03/99  v0.01   Initial version of TextParser (Skip, Extract and Match   }
{                     methods)                                                 }
{   11/05/99  v0.02   Whitespace functions                                     }
{   20/06/99  v0.03   Added MatchString                                        }
{                     Added Line, Col, LineDelim properties                    }
{                     199 lines interface, 207 lines implementation            }
{   26/06/99  v0.10   Moved AStream from cDataStruct unit.                     }
{                     Added ATBinaryStream. Added ReadPacked functions.        }
{                     Added ATTextStream from TTextParser in unit cTextParser. }
{                     Added Peek, Skip, OctetsRemaing and OctetsAvailable to   }
{                     AStream.                                                 }
{                     Added ExtractPacked functions                            }
{                     Moved from cTextParser and renamed TTextParser to        }
{                     ATTextStream. It now inherits from AStream to allow all  }
{                     derived streams to use it.                               }
{   03/07/99  v0.11   Added MetaCharacter support (machine dependant charsets),}
{                     WhiteSpace functions are now implemented using these.    }
{                     Added MetaString support.                                }
{   17/07/99  v0.12   Added Col property.                                      }
{   03/11/99  v0.13   Revised AStream documentation.                           }
{   04/11/99  v0.14   Overloaded methods in ATBinaryStream. Added Int64.       }
{   05/11/99  v0.15   Removed MetaCharacter support.                           }
{   06/11/99  v0.16   Revisited whole unit.                                    }
{                     Overloaded methods.                                      }
{                     Added general functions to ATStream.                     }
{                     Renamed TStringStream to TMemoryStream.                  }
{                     433 lines interface, 933 lines implementation.           }
{   09/11/99  v0.17   Moved WritePacked's implementations to cStrings unit as  }
{                     functions called Pack.                                   }
{   26/12/99  v0.18   Bug fixes. Added SelfTest procedure.                     }
{   03/01/00  v0.19   Removed FLineDelim. A line delimiter is now defined as   }
{                     EOF or either or both of CR and LF.                      }
{   08/02/00  v0.20   Added ATRecordStream.                                    }
{   27/02/00  v0.21   AStream now derives from Delphi's TStream.               }
{                     Revisited whole unit.                                    }
{                     521 lines interface, 1468 lines implementation.          }
{   05/03/00  v0.22   Renamed TMemoryStream to TStringStream.                  }
{   12/03/00  v0.23   Added TEOLType to ATTextStream.                          }
{   23/04/00  v0.24   Added TSocketStream for use in cHTTPServer.              }
{   08/05/00  v0.25   Cleaned up unit. Merged ATStream, ATBinaryStream,        }
{                     ATTextStream into AStream.                               }
{                     Removed ATRecordStream from TStream and created          }
{                     TFixedLenRecordStreamer.                                 }
{                     405 lines interface, 1307 lines implementation.          }
{   30/05/00  v0.26   Split TStream into TStream (abstract), TExStream         }
{                     (default implementations) and various TStreamer classes, }
{                     including TTextParser (back to v0.01 :-)                 }
{   01/06/00  v0.27   Renamed TFixedLenRecordStreamer to TRecordStreamer.      }
{                     Added TIntegerArrayStreamer from TStreamArray in         }
{                     cDataStructs and added TExtendedArrayStreamer,           }
{                     TShortStringArrayStreamer and TBitArrayStreamer.         }
{   08/06/00  v0.28   Converted TBinaryStreamer methods to plain functions.    }
{                     595 lines interface, 1862 lines implementation.          }
{   14/06/00  v0.29   Converted to template.                                   }
{                     Tmpl: 550 lines interface, 1663 lines implementation.    }
{                     Source: 640 lines interface, 2026 lines implementation.  }
{                                                                              }
interface

uses
  Windows,              // GetTickCount
  SysUtils,             // Exception, fmxxx constants
  Classes,              // TStream, TFileStream
  scktcomp,             // TWinSocketStream

  // Delphi Fundamentals (L0)
  cUtils;






{                                                                              }
{ TStream                                                                      }
{   Abstract base class for stream implementations.                            }
{                                                                              }
{   EOF is False if more octects may become available.                         }
{                                                                              }
{   Read blocks until EOF or until the data is available. It raises an         }
{   exception if the data can not be read (eg because of a time-out).          }
{                                                                              }
type
  EStream = class (Exception);
  TOctet = Byte;
  TStream = class (Classes.TStream)
    protected
    // Classes.TStream implementations
    Procedure SetSize (NewSize : Integer); override;
    Function  Seek (Offset : Integer; Origin : Word) : Integer; override;

    // Abstract properties
    Function  GetStreamSize : Int64; virtual; abstract;
    Procedure SetStreamSize (const NewSize : Int64); virtual; abstract;
    Function  GetPosition : Int64; virtual; abstract;
    Procedure SetPosition (const NewPosition : Int64); virtual; abstract;

    public
    // Stream attributes
    Function  SequentialOnly : Boolean; virtual; abstract;
    Function  SizeUnknown : Boolean; virtual; abstract;

    // Content attributes
    Property  Position : Int64 read GetPosition write SetPosition;
    Property  Size : Int64 read GetStreamSize write SetStreamSize;
    Function  Remaining : Int64; virtual; abstract;
    Function  EOF : Boolean; virtual; abstract;

    // Content access
    Function  Read (var Buffer; Count : Longint) : Longint; reintroduce; overload; override;
    Function  Write (const Buffer; Count : Longint) : Longint; reintroduce; overload; override;
  end;



{                                                                              }
{ TExStream (Extended Stream)                                                  }
{   Methods have default implementations using above abstract methods.         }
{   Stream implementations should override these with more efficient           }
{   implementations.                                                           }
{                                                                              }
{   Locate returns the position (relative to the current position) of the      }
{   first occurance of Find in the stream. It does not change the current      }
{   position. Returns -1 if Find could not be found.                           }
{                                                                              }
type
  TExStream = class (TStream)
    protected
    Function  GetAsString : String; virtual;
    Procedure SetAsString (const S : String); virtual;

    public
    Function  Duplicate : TExStream; virtual; abstract;

    Function  Read : TOctet; overload; virtual;
    Function  Read (const Count : Integer) : String; overload; virtual;
    Procedure Write (const S : TOctet); overload; virtual;
    Procedure Write (const S : String); overload; virtual;
    Procedure Write (const S : Classes.TStream); overload; virtual;

    Property  AsString : String read GetAsString write SetAsString;
    Procedure Assign (const S : Classes.TStream); virtual;

    Function  Remaining : Int64; override;
    Function  EOF : Boolean; override;

    Procedure Skip; overload; virtual;
    Procedure Skip (const Count : Integer); overload; virtual;

    Function  Peek : TOctet; overload; virtual;
    Function  Peek (const Count : Integer) : String; overload; virtual;

    Procedure Reset; virtual;
    Procedure Truncate; virtual;
    Procedure Append; overload; virtual;

    Function  Extract (const S : String) : String; virtual;
    Function  Match (const S : String) : Boolean; overload; virtual;
    Function  Match (const Pos, Length : Int64) : Boolean; overload; virtual;
    Function  Match (const S : Classes.TStream; const Length : Int64) : Boolean; overload; virtual;
    Function  IsEqual (const S : Classes.TStream) : Boolean; overload; virtual;
    Function  IsEqual (const S : String) : Boolean; overload; virtual;

    Function  Locate (const Ch : CharSet) : Int64; overload; virtual;
    Function  Locate (const S : String) : Int64; overload; virtual;

    Procedure CopyFrom (const Offset : Int64; const Count : Integer); virtual;
    Procedure Delete (const Count : Integer); virtual;
    Procedure Insert (const Count : Integer); virtual;
  end;



{                                                                              }
{ Binary streamer functions                                                    }
{   Implements functionality to read/write binary data types from.to streams.  }
{                                                                              }
Function  ReadPackedShortInt (const Stream : TExStream) : ShortInt;
Function  ReadPackedWord (const Stream : TExStream) : Word;
Function  ReadPackedSmallInt (const Stream : TExStream) : SmallInt;
Function  ReadPackedInteger (const Stream : TExStream) : Integer;
Function  ReadPackedInt64 (const Stream : TExStream) : Int64;
Function  ReadPackedBoolean (const Stream : TExStream) : Boolean;
Function  ReadPackedSingle (const Stream : TExStream) : Single;
Function  ReadPackedDouble (const Stream : TExStream) : Double;
Function  ReadPackedExtended (const Stream : TExStream) : Extended;
Function  ReadPackedCurrency (const Stream : TExStream) : Currency;
Function  ReadPackedDateTime (const Stream : TExStream) : TDateTime;
Function  ReadPackedString (const Stream : TExStream) : String;
Function  ReadPackedShortString (const Stream : TExStream) : ShortString;

Procedure WritePackedShortInt (const Stream : TExStream; const D : ShortInt);
Procedure WritePackedWord (const Stream : TExStream; const D : Word);
Procedure WritePackedSmallInt (const Stream : TExStream; const D : SmallInt);
Procedure WritePackedInteger (const Stream : TExStream; const D : Integer);
Procedure WritePackedInt64 (const Stream : TExStream; const D : Int64);
Procedure WritePackedBoolean (const Stream : TExStream; const D : Boolean);
Procedure WritePackedSingle (const Stream : TExStream; const D : Single);
Procedure WritePackedDouble (const Stream : TExStream; const D : Double);
Procedure WritePackedExtended (const Stream : TExStream; const D : Extended);
Procedure WritePackedCurrency (const Stream : TExStream; const D : Currency);
Procedure WritePackedDateTime (const Stream : TExStream; const D : TDateTime);
Procedure WritePackedString (const Stream : TExStream; const D : String);
Procedure WritePackedShortString (const Stream : TExStream; const D : ShortString);



{                                                                              }
{ TFileStream                                                                  }
{                                                                              }
type
  TFileStreamMode = Set of (fsmCreate,
                            fsmDeleteOnClose,
                            fsmReadOnlyAccess,
                            fsmShareAllowWrite,
                            fsmShareDenyRead);
type
  TFileStream = class (TExStream)
    protected
    FFileStream    : Classes.TFileStream;
    FFileCreated   : Boolean;
    FDeleteOnClose : Boolean;
    FFilePath      : String;
    FFileMode      : TFileStreamMode;

    Function  GetStreamSize : Int64; override;
    Procedure SetStreamSize (const NewSize : Int64); override;
    Function  GetFileDate : TDateTime;
    Procedure SetFileDate (const D : TDateTime);

    public
    Constructor Create (const FilePath : String; const Mode : TFileStreamMode);
    Destructor Destroy; override;

    Function  Duplicate : TExStream; override;
    Function  SequentialOnly : Boolean; override;
    Function  SizeUnknown : Boolean; override;
    Function  Read : TOctet; overload; override;
    Function  Read (const Count : Integer) : String; overload; override;
    Function  Read (var Buf; Count : Integer) : Integer; overload; override;
    Function  Write (const Buf; Count : Integer) : LongInt; overload; override;
    Procedure Write (const S : TOctet); overload; override;
    Procedure Write (const S : String); overload; override;
    Procedure Skip (const Count : Integer); overload; override;
    Procedure Skip; overload; override;
    Function  Peek : TOctet; overload; override;
    Function  Peek (const Count : Integer) : String; overload; override;
    Function  GetPosition : Int64; override;
    Procedure SetPosition (const NewPosition : Int64); override;
    Procedure Append; override;
    Function  Remaining : Int64; override;
    Function  EOF : Boolean; override;
    Procedure Truncate; override;
    Function  Locate (const Ch : CharSet) : Int64; overload; override;
    Function  Locate (const S : String) : Int64; overload; override;

    Property  DeleteOnClose : Boolean read FDeleteOnClose write FDeleteOnClose;
    Property  FileDate : TDateTime read GetFileDate write SetFileDate;
  end;



{                                                                              }
{ TStringStream                                                                }
{   A implementation of a stream that stores/retrieves data using a Delphi     }
{   String.                                                                    }
{                                                                              }
type
  TStringStream = class (TExStream)
    protected
    FData : String;
    FPos  : Integer;

    Function GetStreamSize : Int64; override;
    Procedure SetStreamSize (const NewSize : Int64); override;

    public
    Constructor CreateEx (const S : String);

    Function  Duplicate : TExStream; override;
    Procedure Skip (const Count : Integer); overload; override;
    Procedure Skip; overload; override;
    Function  Read : TOctet; overload; override;
    Function  Read (const Count : Integer) : String; overload; override;
    Function  Read (var Buf; Count : Integer) : Integer; override;
    Function  Peek : TOctet; overload; override;
    Function  Peek (const Count : Integer) : String; overload; override;
    Function  GetPosition : Int64; override;
    Procedure SetPosition (const NewPosition : Int64); override;
    Procedure Append; override;
    Function  Remaining : Int64; override;
    Function  EOF : Boolean; override;
    Procedure Truncate; override;
    Function  Write (const Buf; Count : Integer) : LongInt; overload; override;
    Procedure Write (const Data : TOctet); overload; override;
    Procedure Write (const Data : String); overload; override;
    Function  Locate (const Find : String) : Int64; override;
    Function  SequentialOnly : Boolean; override;
    Function  SizeUnknown : Boolean; override;
  end;



{                                                                              }
{ TSocketStream                                                                }
{                                                                              }
type
  ESocketStream = class (Exception);
  TSocketStream = class (TExStream)
    protected
    FClient      : TClientSocket;
    FSocket      : TCustomWinSocket;
    FSockStream  : TWinSocketStream;
    FBytesRead   : Integer;
    FPeekBuf     : String;
    FTimeOut     : Integer;
    FCloseSocket : Boolean;

    Procedure FillPeekBuf (const Len : Integer);

    public
    Constructor Create (const Host : String; const Port : Integer;
                const TimeOut : Integer; const CloseSocketOnDestroy : Boolean = True);
    Constructor CreateEx (const Socket : TCustomWinSocket; const TimeOut : Integer;
                const CloseSocketOnDestroy : Boolean = True);
    Destructor Destroy; override;

    Procedure Skip (const Count : Integer); overload; override;
    Procedure Skip; overload; override;
    Function  Read (var Buf; Count : Integer) : Integer; override;
    Function  Read : TOctet; overload; override;
    Function  Read (const Count : Integer) : String; overload; override;
    Function  Peek : TOctet; overload; override;
    Function  Peek (const Count : Integer) : String; overload; override;
    Function  GetPosition : Int64; override;
    Procedure SetPosition (const NewPosition : Int64); override;
    Procedure Append; override;
    Function  EOF : Boolean; override;
    Procedure Write (const Data : TOctet); overload; override;
    Procedure Write (const Data : String); overload; override;
    Function  Write (const Buf; Count : Integer) : LongInt; overload; override;

    Procedure CloseSocket;
    Property  CloseSocketOnDestroy : Boolean read FCloseSocket write FCloseSocket;
  end;



{                                                                              }
{ TStreamer                                                                    }
{                                                                              }
type
  TStreamer = class
    protected
    FStream : TExStream;

    public
    Constructor Create (const Stream : TExStream);
    Destructor Destroy; override;
    Procedure Reset; virtual;
    Property  Stream : TExStream read FStream;
  end;



{                                                                              }
{ TTextParser                                                                  }
{   Implements functionality to parse text from a stream.                      }
{                                                                              }
{   The Skip functions return True if any characters were skipped; and exit    }
{     without raising an exception if EOF is reached.                          }
{   Match functions return True if next characters match, without moving the   }
{     current position. If the match overruns EOF then the function returns    }
{     False.                                                                   }
{   The Extract functions return the extracted text, unchanged. It returns ''  }
{     if a match could not be made.                                            }
{   Function Extract : Char raises an exception if called on EOF.              }
{   MatchText, ExtractText and MatchTextLine look at the case/locale sensitive }
{     properties.                                                              }
{   ExtractKeyword uses MatchText to match from a list of possible keywords.   }
{     The keyword must be followed by Delimiter or EOF. It returns the index   }
{     of the extracted keyword, or -1 if none matched.                         }
{   ExtractQuotedText extracts text enclosed in a pair Quote.                  }
{     If AllowQuotedQuotes then quotes can be included inside the text by      }
{     using two consequetive quotes, eg for Quote='"', "A""BC" extracts A"BC   }
{     if ExtractLiteral=False and "A""BC" if ExtractLiteral=True.              }
{                                                                              }
type
  TEOLType = Set of (eol_CR, eol_LF, eol_CRLF, eol_LFCR, eol_EOF);
  TTextParser = class (TStreamer)
    protected
    FLineNr              : Integer;
    FLastNewLine         : Integer;
    FEOLType             : TEOLType;
    FWhiteSpace          : CharSet;
    FTextCaseSensitive   : Boolean;
    FTextLocaleSensitive : Boolean;

    Function  GetCol : Integer;

    public
    Constructor Create (const Stream : TExStream);

    Function  EOF : Boolean;

    Procedure Skip;
    Function  Match (const Ch : Char) : Boolean; overload;
    Function  Match (const S : String) : Boolean; overload;
    Function  Extract : Char; overload;
    Function  Extract (const S : String) : String; overload;

    Function  SkipChar (const Ch : CharSet) : Boolean;
    Function  MatchChar (const Ch : CharSet) : Boolean;
    Function  ExtractChar (const Ch : CharSet) : String;

    Function  SkipChars (const Ch : CharSet) : Boolean;
    Function  ExtractChars (const Ch : CharSet) : String;

    Function  SkipWhiteSpace : Boolean;
    Function  MatchWhiteSpace : Boolean;
    Function  ExtractWhiteSpace : String;

    Function  MatchText (const Txt : String) : Boolean;
    Function  ExtractText (const Txt : String) : String;

    Function  SkipEOL : Boolean;
    Function  MatchEOL : Boolean;
    Function  ExtractEOL : String;

    Function  SkipLine : Boolean;
    Function  MatchLine (const S : String) : Boolean;
    Function  MatchTextLine (const S : String) : Boolean;
    Function  ExtractLine : String;

    Function  SkipTo (const Delimiter : String) : Boolean;
    Function  ExtractTo (const Delimiter : String; const SkipDelimiter : Boolean = False) : String; overload;
    Function  ExtractTo (const Delimiter : CharSet; const SkipDelimiter : Boolean = False) : String; overload;
    Function  ExtractToEOF : String;

    Function  MatchKeyword (const Keywords : StringArray; const Delimiter : CharSet) : Integer;
    Function  ExtractKeyword (const Keywords : StringArray; const Delimiter : CharSet) : String;

    Function  ExtractQuoted (const Quote : Char; const AllowQuotedQuote : Boolean) : String;
    Function  ExtractNestedPair (const OpenQuote, CloseQuote : String;
              const LiteralsOpenQuote, LiteralsCloseQuote : StringArray) : String;

    Property  TextCaseSensitive : Boolean read FTextCaseSensitive write FTextCaseSensitive;
    Property  TextLocaleSensitive : Boolean read FTextLocaleSensitive write FTextLocaleSensitive;
    Property  WhiteSpace : CharSet read FWhiteSpace write FWhiteSpace;
    Property  EOLType : TEOLType read FEOLType write FEOLType;
    Property  Line : Integer read FLineNr;
    Property  Col : Integer read GetCol;

    Function  GetBookmark : Int64;
    Procedure GotoBookmark (const B : Int64);
  end;



{                                                                              }
{ TRecordStreamer                                                              }
{   Implements functionality to access fixed-length records in streams.        }
{                                                                              }
type
  TRecordStreamer = class (TStreamer)
    protected
    FRecordSize   : Integer;
    FRecordOffset : Int64;

    Function  GetRecordCount : Int64;
    Procedure SetRecordCount (const NewRecordCount : Int64);
    Procedure SetRecordSize (const RecordSize : Integer);
    Procedure Seek (const Idx : Int64);

    public
    Constructor Create (const Stream : TExStream; const RecordSize : Integer = 1;
                const FirstRecordOffset : Int64 = 0);

    Procedure Read (const Idx : Int64; const Count : Integer; var Buf);
    Procedure Write (const Idx : Int64; const Count : Integer; const Buf);
    Procedure Insert (const Idx : Int64; const Count : Integer);
    Procedure Delete (const Idx : Int64; const Count : Integer);
    Procedure Clear (const Idx : Int64; const Count : Integer);

    Property RecordSize : Integer read FRecordSize write FRecordSize;
    Property RecordOffset : Int64 read FRecordOffset write FRecordOffset;
    Property RecordCount : Int64 read GetRecordCount write SetRecordCount;
  end;

  TIntegerArrayStreamer = class (TRecordStreamer)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Integer);
    Function  GetItem (const Idx : Integer) : Integer;
    Function  GetAsIntegerArray : IntegerArray;
    Procedure SetAsIntegerArray (const V : IntegerArray);

    public
    Constructor Create (const Stream : TExStream);

    Property  Item [const Idx : Integer] : Integer read GetItem write SetItem; default;
    Property  Count : Int64 read GetRecordCount write SetRecordCount;
    Procedure Fill (const Value : Integer);
    Procedure FillInc (const StartValue : Integer = 0; const Increment : Integer = 1);
    Function  PosNext (const Find : Integer; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer;
    Property  AsIntegerArray : IntegerArray read GetAsIntegerArray write SetAsIntegerArray;
  end;


  TInt64ArrayStreamer = class (TRecordStreamer)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Int64);
    Function  GetItem (const Idx : Integer) : Int64;
    Function  GetAsInt64Array : Int64Array;
    Procedure SetAsInt64Array (const V : Int64Array);

    public
    Constructor Create (const Stream : TExStream);

    Property  Item [const Idx : Integer] : Int64 read GetItem write SetItem; default;
    Property  Count : Int64 read GetRecordCount write SetRecordCount;
    Procedure Fill (const Value : Int64);
    Procedure FillInc (const StartValue : Int64 = 0; const Increment : Int64 = 1);
    Function  PosNext (const Find : Int64; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer;
    Property  AsInt64Array : Int64Array read GetAsInt64Array write SetAsInt64Array;
  end;


  TSingleArrayStreamer = class (TRecordStreamer)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Single);
    Function  GetItem (const Idx : Integer) : Single;
    Function  GetAsSingleArray : SingleArray;
    Procedure SetAsSingleArray (const V : SingleArray);

    public
    Constructor Create (const Stream : TExStream);

    Property  Item [const Idx : Integer] : Single read GetItem write SetItem; default;
    Property  Count : Int64 read GetRecordCount write SetRecordCount;
    Procedure Fill (const Value : Single);
    Procedure FillInc (const StartValue : Single = 0; const Increment : Single = 1);
    Function  PosNext (const Find : Single; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer;
    Property  AsSingleArray : SingleArray read GetAsSingleArray write SetAsSingleArray;
  end;


  TDoubleArrayStreamer = class (TRecordStreamer)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Double);
    Function  GetItem (const Idx : Integer) : Double;
    Function  GetAsDoubleArray : DoubleArray;
    Procedure SetAsDoubleArray (const V : DoubleArray);

    public
    Constructor Create (const Stream : TExStream);

    Property  Item [const Idx : Integer] : Double read GetItem write SetItem; default;
    Property  Count : Int64 read GetRecordCount write SetRecordCount;
    Procedure Fill (const Value : Double);
    Procedure FillInc (const StartValue : Double = 0; const Increment : Double = 1);
    Function  PosNext (const Find : Double; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer;
    Property  AsDoubleArray : DoubleArray read GetAsDoubleArray write SetAsDoubleArray;
  end;


  TExtendedArrayStreamer = class (TRecordStreamer)
    protected
    Procedure SetItem (const Idx : Integer; const Value : Extended);
    Function  GetItem (const Idx : Integer) : Extended;
    Function  GetAsExtendedArray : ExtendedArray;
    Procedure SetAsExtendedArray (const V : ExtendedArray);

    public
    Constructor Create (const Stream : TExStream);

    Property  Item [const Idx : Integer] : Extended read GetItem write SetItem; default;
    Property  Count : Int64 read GetRecordCount write SetRecordCount;
    Procedure Fill (const Value : Extended);
    Procedure FillInc (const StartValue : Extended = 0; const Increment : Extended = 1);
    Function  PosNext (const Find : Extended; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer;
    Property  AsExtendedArray : ExtendedArray read GetAsExtendedArray write SetAsExtendedArray;
  end;


  TShortStringArrayStreamer = class (TRecordStreamer)
    protected
    FStringLen : Byte;

    Procedure SetItem (const Idx : Integer; const Value : ShortString);
    Function  GetItem (const Idx : Integer) : ShortString;

    public
    Constructor Create (const Stream : TExStream; const StringLen : Byte = 255);

    Property  StringLen : Byte read FStringLen;
    Property  Item [const Idx : Integer] : ShortString read GetItem write SetItem; default;
    Property  Count : Int64 read GetRecordCount write SetRecordCount;
    Procedure Fill (const Value : ShortString = '');
    Function  PosNext (const Find : ShortString; const PrevPos : Integer = -1;
              const IsSortedAscending : Boolean = False) : Integer;
  end;

  TBitArrayStreamer = class (TRecordStreamer)
    protected
    Procedure SetItem (const Idx : Int64; const Value : Boolean);
    Function  GetItem (const Idx : Int64) : Boolean;
    Procedure SetCount (const NewCount : Int64);
    Function  GetCount : Int64;

    public
    Constructor Create (const Stream : TExStream);

    Property  Item [const Idx : Int64] : Boolean read GetItem write SetItem; default;
    Property  Count : Int64 read GetCount write SetCount;
    Procedure Fill (const Value : Boolean = False);
    Procedure FillRange (const LoIdx, HiIdx : Int64; const Value : Boolean);
    Function  CompareRange (const LoIdx, HiIdx : Int64; const Value : Boolean) : Boolean;
    Procedure Invert;
    Function  Find (const Value : Boolean; const Start : Integer = 0;
              const FindForward : Boolean = True) : Integer;
    Function  FindRange (const Value : Boolean; const Start : Integer = 0;
              const Count : Integer = 1; const FindForward : Boolean = True) : Integer;
  end;



implementation



uses
  // System units
  Math,                 // Max

  // Delphi Fundamentals (L0)
  cStrings;             // Pack



{                                                                              }
{ TStream                                                                      }
{                                                                              }
Function TStream.Read (var Buffer; Count : Longint) : Longint;
  Begin
    raise EAbstractError.Create ('Method TStream.Read not implemented.');
  End;

Function TStream.Write (const Buffer; Count : Longint) : Longint;
  Begin
    raise EAbstractError.Create ('Method TStream.Write not implemented.');
  End;

Function TStream.Seek (Offset : Integer; Origin : Word) : Integer;
  Begin
    Case Origin of
      soFromBeginning : if Offset >= 0 then
                          SetPosition (Offset) else
                          raise EStream.Create ('Seek before bof');
      soFromCurrent   : SetPosition (GetPosition + Offset);
      soFromEnd       : SetPosition (GetStreamSize + Offset);
    end;
    Result := GetPosition;
  End;

Procedure TStream.SetSize (NewSize : Integer);
  Begin
    SetStreamSize (NewSize);
  End;



{                                                                              }
{ TExStream                                                                    }
{                                                                              }
Function TExStream.Read : TOctet;
  Begin
    if Read (Result, Sizeof (Result)) <> Sizeof (Result) then
      raise EReadError.Create ('Stream read error.');
  End;

Function TExStream.Read (const Count : Integer) : String;
  Begin
    SetLength (Result, Count);
    if Count > 0 then
      SetLength (Result, Read (Result [1], Count));
  End;

Procedure TExStream.Write (const S : TOctet);
  Begin
    if Write (S, Sizeof (S)) <> Sizeof (S) then
      raise EWriteError.Create ('Stream write error.');
  End;

Procedure TExStream.Write (const S : String);
var L : Integer;
  Begin
    L := Length (S);
    if L > 0 then
      if Write (S [1], L) <> L then
        raise EWriteError.Create ('Stream write error.');
  End;

Procedure TExStream.Write (const S : Classes.TStream);
const CopyBufferSize = 32768;
var Buffer  : String;
    F, P, L : Int64;
  Begin
    L := S.Size;
    P := S.Seek (0, soFromBeginning);
    Repeat
      F := Min (CopyBufferSize, L - P);
      if F > 0 then
        begin
          SetLength (Buffer, F);
          S.ReadBuffer (Buffer [1], F);
          Write (Buffer);
          Inc (P, F);
        end;
    Until F = 0;
  End;

Procedure TExStream.Assign (const S : Classes.TStream);
  Begin
    Position := 0;
    Write (S);
    Truncate;
  End;

Function TExStream.Remaining : Int64;
  Begin
    Result := Size - Position;
  End;

Function TExStream.EOF : Boolean;
  Begin
    Result := Size = Position;
  End;

Procedure TExStream.Truncate;
  Begin
    Size := Position;
  End;

Procedure TExStream.Append;
  Begin
    Position := Size;
  End;

Procedure TExStream.Skip;
  Begin
    if SequentialOnly then
      Read else
      Position := Position + 1;
  End;

Procedure TExStream.Skip (const Count : Integer);
  Begin
    if SequentialOnly then
      Read (Count) else
      Position := Min (Size, Position + Count);
  End;

Function TExStream.GetAsString : String;
  Begin
    Position := 0;
    Result := Read (Size);
  End;

Procedure TExStream.SetAsString (const S : String);
  Begin
    Position := 0;
    Write (S);
    Truncate;
  End;

Procedure TExStream.Reset;
  Begin
    Position := 0;
  End;

Function TExStream.Peek : TOctet;
var P : Int64;
  Begin
    if SequentialOnly then
      raise EStream.Create ('Peek not supported.');

    P := Position;
    Result := Read;
    Position := P;
  End;

Function TExStream.Peek (const Count : Integer) : String;
var P : Int64;
  Begin
    if SequentialOnly then
      raise EStream.Create ('Peek not supported.');

    P := Position;
    Result := Read (Count);
    Position := P;
  End;

Function TExStream.Extract (const S : String) : String;
const CompareBufferSize = 1024;
var I, R, L : Integer;
    P       : Int64;
    Buffer  : String;
  Begin
    if SequentialOnly then
      raise EStream.Create ('Match not supported.');

    Result := '';
    if Remaining < Length (S) then
      exit;

    P := Position;
    L := Length (S);
    R := L;
    While R > 0 do
      begin
        I := Min (R, CompareBufferSize);
        Buffer := Read (I);
        if not cStrings.Match (Buffer, S, L - R + 1) then
          begin
            Position := P;
            exit;
          end;
        Dec (R, I);
      end;

    Result := S;
  End;

Function TExStream.Match (const S : String) : Boolean;
var P : Int64;
  Begin
    if SequentialOnly then
      raise EStream.Create ('Match not supported.');

    P := Position;
    Result := Extract (S) <> '';
    if Result then
      Position := P;
  End;

Function TExStream.Match (const Pos, Length : Int64) : Boolean;
const CompareBufferSize = 1024;
var I, R, P : Int64;
    Buffer  : String;
  Begin
    if SequentialOnly then
      raise EStream.Create ('Match not supported.');

    Result := False;
    if Remaining < Length then
      exit;

    R := Length;
    P := Position;
    While R > 0 do
      begin
        I := Min (R, CompareBufferSize);
        Buffer := Read (I);
        Position := Pos + Length - R;
        if Buffer <> Read (I) then
          begin
            Position := P;
            exit;
          end;
        Dec (R, I);
        Position := P + Length - R;
      end;

    Position := P;
    Result := True;
  End;

Function TExStream.Match (const S : Classes.TStream; const Length : Int64) : Boolean;
const CompareBufferSize = 1024;
var P, I, R : Int64;
    Buffer  : String;
  Begin
    if SequentialOnly then
      raise EStream.Create ('Match not supported.');

    Result := False;
    if (Remaining < Length) or (S.Size - S.Position < Length) then
      exit;

    P := Position;
    R := Length;
    While R > 0 do
      begin
        I := Min (R, CompareBufferSize);
        SetLength (Buffer, I);
        S.ReadBuffer (Buffer [1], I);
        if Read (I) <> Buffer then
          begin
            Position := P;
            exit;
          end;
        Dec (R, I);
      end;

    Position := P;
    Result := True;
  End;

Function TExStream.IsEqual (const S : Classes.TStream) : Boolean;
var L1, L2 : Int64;
  Begin
    if SequentialOnly then
      raise EStream.Create ('Match not supported.');

    L1 := Size;
    L2 := S.Size;
    if (L1 >= 0) and (L2 >= 0) and (L1 <> L2) then
      Result := False else
      begin
        Position := 0;
        S.Seek (0, soFromBeginning);
        Result := Match (S, L1);
      end;
  End;

Function TExStream.IsEqual (const S : String) : Boolean;
  Begin
    if SequentialOnly then
      raise EStream.Create ('Match not supported.');

    if Length (S) <> Remaining then
      Result := False else
      begin
        Position := 0;
        Result := Match (S);
      end;
  End;

Function TExStream.Locate (const Ch : CharSet) : Int64;
var P : Int64;
  Begin
    if SequentialOnly then
      raise EStream.Create ('Locate not supported.');

    P := Position;
    While not EOF do
      if Char (Read) in Ch then
        begin
          Result := Position - P;
          Position := P;
          exit;
        end;
    Position := P;
    Result := -1;
  End;

Function TExStream.Locate (const S : String) : Int64;
var P, I : Int64;
    L    : Integer;
  Begin
    if SequentialOnly then
      raise EStream.Create ('Locate not supported.');

    L := Length (S);
    P := Position;
    I := 0;
    While not EOF do
      begin
        if Peek (L) = S then
          begin
            Position := P;
            Result := I;
            exit;
          end;
        Inc (I);
        Position := P + I;
      end;
    Position := P;
    Result := -1;
  End;

Procedure TExStream.CopyFrom (const Offset : Int64; const Count : Integer);
const BufSize = 16384;
var S, D   : Int64;
    L, C   : Integer;
    Buffer : String;
  Begin
    if SequentialOnly then
      raise EStream.Create ('CopyFrom not supported.');

    S := Offset;
    D := Position;
    L := Min (Count, Size - S);
    While L > 0 do
      begin
        C := Min (BufSize, L);
        SetLength (Buffer, C);
        Position := S;
        Read (Buffer [1], C);
        Position := D;
        Write (Buffer [1], C);
        Dec (L, C);
      end;
  End;

Procedure TExStream.Delete (const Count : Integer);
var D, S : Int64;
  Begin
    D := Position + Count;
    S := Size;
    CopyFrom (D, S - D);
    Position := S - Count;
    Truncate;
  End;

Procedure TExStream.Insert (const Count : Integer);
var P : Int64;
  Begin
    P := Position;
    Position := P + Count;
    CopyFrom (P, Size - P);
  End;



{                                                                              }
{ TFileStream                                                                  }
{                                                                              }
Constructor TFileStream.Create (const FilePath : String; const Mode : TFileStreamMode);
var M : Word;
    CreateFile : Boolean;
  Begin
    inherited Create;

    FFilePath := FilePath;
    FFileMode := Mode;

    FDeleteOnClose := fsmDeleteOnClose in Mode;

    CreateFile := (fsmCreate in Mode) and not FileExists (FilePath);

    if CreateFile then
      M := fmCreate else
      if fsmReadOnlyAccess in Mode then
        M := fmOpenRead else
        M := fmOpenReadWrite;
    M := M or fmShareCompat;
    if not (fsmShareAllowWrite in Mode) then
      M := M or fmShareDenyWrite;
    if fsmShareDenyRead in Mode then
      M := M or fmShareDenyRead;

    FFileStream := Classes.TFileStream.Create (FilePath, M);
    if not CreateFile and (fsmCreate in Mode) then
      FFileStream.Size := 0;
    FFileCreated := CreateFile;
  End;

Destructor TFileStream.Destroy;
  Begin
    FreeAndNil (FFileStream);
    if FFileCreated and FDeleteOnClose then
      DeleteFile (FFilePath);
    inherited Destroy;
  End;

Function TFileStream.Duplicate : TExStream;
  Begin
    Result := TFileStream.Create (FFilePath, FFileMode);
  End;

Function TFileStream.SequentialOnly : Boolean;
  Begin
    Result := False;
  End;

Function TFileStream.SizeUnknown : Boolean;
  Begin
    Result := False;
  End;

Function TFileStream.Read : TOctet;
  Begin
    FFileStream.ReadBuffer (Result, Sizeof (Result));
  End;

Function TFileStream.Read (var Buf; Count : Integer) : Integer;
  Begin
    if Count <= 0 then
      Result := 0 else
      Result := FFileStream.Read (Buf, Min (Count, Remaining));
  End;

Function TFileStream.Read (const Count : Integer) : String;
var I : Integer;
  Begin
    if Count <= 0 then
      Result := '' else
      begin
        I := Min (Count, Remaining);
        SetLength (Result, I);
        if I > 0 then
          SetLength (Result, FFileStream.Read (Result [1], I));
      end;
  End;

Procedure TFileStream.Write (const S : TOctet);
  Begin
    FFileStream.WriteBuffer (S, Sizeof (S));
  End;

Function TFileStream.Write (const Buf; Count : Integer) : LongInt;
  Begin
    FFileStream.WriteBuffer (Buf, Count);
    Result := Count;
  End;

Procedure TFileStream.Write (const S : String);
  Begin
    FFileStream.WriteBuffer (S [1], Length (S));
  End;

Function TFileStream.Remaining : Int64;
  Begin
    Result := FFileStream.Size - FFileStream.Position;
  End;

Function TFileStream.EOF : Boolean;
  Begin
    Result := not (FFileStream.Position < FFileStream.Size);
  End;

Procedure TFileStream.Skip;
  Begin
    if EOF then
      raise EStream.Create ('Skip past EOF');
    FFileStream.Seek (FFileStream.Position + 1, soFromBeginning);
  End;

Procedure TFileStream.Skip (const Count : Integer);
  Begin
    FFileStream.Seek (Min (FFileStream.Position + Count, FFileStream.Size), soFromBeginning);
  End;

Function TFileStream.GetPosition : Int64;
  Begin
    Result := FFileStream.Position;
  End;

Function TFileStream.GetStreamSize : Int64;
  Begin
    Result := FFileStream.Size;
  End;

Procedure TFileStream.SetStreamSize (const NewSize : Int64);
  Begin
    FFileStream.Size := NewSize;
  End;

Procedure TFileStream.SetPosition (const NewPosition : Int64);
  Begin
    FFileStream.Position := NewPosition;
  End;

Function TFileStream.Peek : TOctet;
var P : Integer;
  Begin
    P := FFileStream.Position;
    FFileStream.Read (Result, Sizeof (Result));
    FFileStream.Seek (P, soFromBeginning);
  End;

Function TFileStream.Peek (const Count : Integer) : String;
var P : Integer;
  Begin
    P := FFileStream.Position;
    Result := Read (Count);
    FFileStream.Seek (P, soFromBeginning);
  End;

Procedure TFileStream.Truncate;
  Begin
    FFileStream.Size := FFileStream.Position;
  End;

Procedure TFileStream.Append;
  Begin
    FFileStream.Seek (0, soFromEnd);
  End;

Function TFileStream.Locate (const S : String) : Int64;

  Function ReadBuf (const Len : Integer) : String;
    Begin
      SetLength (Result, Len);
      SetLength (Result, FFileStream.Read (Result [1], Len));
    End;

const BufSize = 1024;

var Buf, B : String;
    P, I   : Integer;

  Begin
    P := FFileStream.Position;
    try
      Buf := ReadBuf (Length (S));
      Result := PosNext (S, Buf);
      if Result > 0 then
        exit;
      I := 0;
      While FFileStream.Position < FFileStream.Size do
        begin
          B := ReadBuf (BufSize);
          Buf := Buf + B;
          Result := I + PosNext (S, Buf);
          if Result > I then
            exit;
          System.Delete (Buf, 1, Length (B));
          Inc (I, Length (B));
        end;

      Result := -1;
    finally
      FFileStream.Seek (P, soFromBeginning);
    end;
  End;

Function TFileStream.Locate (const Ch : CharSet) : Int64;
const BufSize = 1024;
var Buf  : String;
    P, I : Integer;
  Begin
    P := FFileStream.Position;
    try
      While not EOF do
        begin
          I := Min (Remaining, BufSize);
          SetLength (Buf, I);
          SetLength (Buf, FFileStream.Read (Buf [1], I));

          I := PosNext (Ch, Buf);
          if I > 0 then
            begin
              Result := FFileStream.Position - Length (Buf) + I - 1;
              exit;
            end;
        end;
      Result := -1;
    finally
      FFileStream.Seek (P, soFromBeginning);
    end;
  End;

Function TFileStream.GetFileDate : TDateTime;
  Begin
    Result := FileDateToDateTime (FileGetDate (FFileStream.Handle));
  End;

Procedure TFileStream.SetFileDate (const D : TDateTime);
  Begin
    FileSetDate (FFileStream.Handle, DateTimeToFileDate (D));
  End;



{                                                                              }
{ TStringStream                                                                }
{                                                                              }
Constructor TStringStream.CreateEx (const S : String);
  Begin
    inherited Create;
    FData := S;
  End;

Function TStringStream.SequentialOnly : Boolean;
  Begin
    Result := False;
  End;

Function TStringStream.SizeUnknown : Boolean;
  Begin
    Result := False;
  End;

Function TStringStream.Duplicate : TExStream;
  Begin
    Result := TStringStream.CreateEx (FData);
  End;

Function TStringStream.Read : TOctet;
  Begin
    if FPos = Length (FData) then
      raise EStream.Create ('Read past EOF');
    Inc (FPos);
    Result := TOctet (FData [FPos]);
  End;

Function TStringStream.Read (var Buf; Count : Integer) : Integer;
var L : Integer;
  Begin
    L := Length (FData);
    if FPos + Count > L then
      Result := L - FPos else
      Result := Count;
    Move (FData [FPos + 1], Buf, Result);
    Inc (FPos, Result);
  End;

Function TStringStream.Read (const Count : Integer) : String;
var C, L : Integer;
  Begin
    L := Length (FData);
    if FPos + Count > L then
      C := L - FPos else
      C := Count;
    Result := Copy (FData, FPos + 1, C);
    Inc (FPos, C);
  End;

Procedure TStringStream.Write (const Data : TOctet);
  Begin
    if FPos = Length (FData) then
      FData := FData + Char (Data) else
      FData [FPos] := Char (Data);
    Inc (FPos);
  End;

Function TStringStream.Write (const Buf; Count : Integer) : LongInt;
  Begin
    if FPos + Count > Length (FData) then
      SetLength (FData, FPos + Count);
    Move (Buf, FData [FPos + 1], Count);
    Inc (FPos, Count);
    Result := Count;
  End;

Procedure TStringStream.Write (const Data : String);
  Begin
    if FPos + Length (Data) > Length (FData) then
      SetLength (FData, FPos + Length (Data));
    Move (Data [1], FData [FPos + 1], Length (Data));
    Inc (FPos, Length (Data));
  End;

Function TStringStream.EOF : Boolean;
  Begin
    Result := FPos >= Length (FData);
  End;

Procedure TStringStream.Skip;
  Begin
    if EOF then
      raise EStream.Create ('Skip past EOF');
    Inc (FPos);
  End;

Procedure TStringStream.Skip (const Count : Integer);
  Begin
    if FPos + Count > Length (FData) then
      FPos := Length (FData) else
      Inc (FPos, Count);
  End;

Function TStringStream.Peek : TOctet;
  Begin
    if EOF then
      raise EStream.Create ('Peek past EOF');
    Result := TOctet (FData [FPos + 1]);
  End;

Function TStringStream.Peek (const Count : Integer) : String;
  Begin
    Result := Copy (FData, FPos + 1, Count);
  End;

Function TStringStream.GetPosition : Int64;
  Begin
    Result := FPos;
  End;

Procedure TStringStream.SetPosition (const NewPosition : Int64);
  Begin
    if NewPosition > Length (FData) then
      raise EStream.Create ('Seek past EOF');
    FPos := NewPosition;
  End;

Procedure TStringStream.Append;
  Begin
    FPos := Length (FData);
  End;

Function TStringStream.Remaining : Int64;
  Begin
    Result := Length (FData) - FPos;
  End;

Function TStringStream.GetStreamSize : Int64;
  Begin
    Result := Length (FData);
  End;

Procedure TStringStream.SetStreamSize (const NewSize : Int64);
  Begin
    SetLength (FData, NewSize);
  End;

Procedure TStringStream.Truncate;
  Begin
    SetLength (FData, FPos);
  End;

Function TStringStream.Locate (const Find : String) : Int64;
var I : Integer;
  Begin
    I := PosNext (Find, FData, FPos);
    if I = 0 then
      Result := -1 else
      Result := I - FPos;
  End;



{                                                                              }
{ TSocketStream                                                                }
{                                                                              }
Constructor TSocketStream.CreateEx (const Socket : TCustomWinSocket; const TimeOut : Integer; const CloseSocketOnDestroy : Boolean);
  Begin
    inherited Create;
    FSocket := Socket;
    FTimeOut := TimeOut;
    FSockStream := TWinSocketStream.Create (FSocket, TimeOut);
    FBytesRead := 0;
    FPeekBuf := '';
    FCloseSocket := CloseSocketOnDestroy;
  End;

Constructor TSocketStream.Create (const Host : String; const Port : Integer; const TimeOut : Integer; const CloseSocketOnDestroy : Boolean);
  Begin
    inherited Create;
    FClient := TClientSocket.Create (nil);
    FClient.Host := Host;
    FClient.Port := Port;
    FClient.ClientType := ctBlocking;
    FClient.Open;
    FSocket := FClient.Socket;
    FTimeOut := TimeOut;
    FSockStream := TWinSocketStream.Create (FSocket, TimeOut);
    FBytesRead := 0;
    FPeekBuf := '';
    FCloseSocket := CloseSocketOnDestroy;
  End;

Destructor TSocketStream.Destroy;
  Begin
    if FCloseSocket then
      try
        CloseSocket;
      except end;
    FreeAndNil (FClient);
    FreeAndNil (FSockStream);
    inherited Destroy;
  End;

Procedure TSocketStream.CloseSocket;
  Begin
    if Assigned (FSocket) then
      FSocket.Close;
  End;

Function TSocketStream.EOF : Boolean;
  Begin
    Result := not FSocket.Connected;
  End;

procedure TSocketStream.Append;
  Begin
  End;

Procedure TSocketStream.FillPeekBuf (const Len : Integer);
var L : Integer;
    S : String;
  Begin
    L := Len - Length (FPeekBuf);
    if L <= 0 then
      exit;
    FSockStream.WaitForData (FTimeOut);
    SetLength (S, L);
    SetLength (S, FSockStream.Read (S [1], L));
    if S <> '' then
      FPeekBuf := FPeekBuf + S;
  End;

Function TSocketStream.Peek : TOctet;
  Begin
    FillPeekBuf (Sizeof (Result));
    Result := TOctet (FPeekBuf [1]);
  End;

Function TSocketStream.Peek (const Count : Integer) : String;
  Begin
    FillPeekBuf (Count);
    Result := CopyLeft (FPeekBuf, Count);
  End;

Function TSocketStream.Read (var Buf; Count : Integer) : Integer;
type ByteBuffer = Array [0..2140000000] of Byte;
var Buffer : ByteBuffer absolute Buf;
    I : Integer;
  Begin
    if FPeekBuf <> '' then
      begin
        I := Min (Count, Length (FPeekBuf));
        Move (FPeekBuf [1], Buffer [1], I);
        System.Delete (FPeekBuf, 1, I);
        Dec (Count, I);
      end else
      I := 0;
    Result := I + FSockStream.Read (Buffer [I], Count);
  End;

Function TSocketStream.Read (const Count : Integer) : String;
  Begin
    FillPeekBuf (Count);
    Result := CopyLeft (FPeekBuf, Count);
    System.Delete (FPeekBuf, 1, Count);
    Inc (FBytesRead, Count);
  End;

Function TSocketStream.Read : TOctet;
  Begin
    FillPeekBuf (Sizeof (Result));
    if Length (FPeekBuf) = 0 then
      raise ESocketStream.Create ('Time out reading from socket.');
      
    Result := TOctet (FPeekBuf [1]);
    System.Delete (FPeekBuf, 1, 1);
    Inc (FBytesRead);
  End;

Function TSocketStream.GetPosition : Int64;
  Begin
    Result := FBytesRead;
  End;

Procedure TSocketStream.SetPosition (const NewPosition : Int64);
var P : Int64;
  Begin
    P := Position;
    if NewPosition >= P then
      Read (NewPosition - P) else
      raise EStream.Create ('TSocketStream does not support seeking backwards.');
  End;

Procedure TSocketStream.Skip;
  Begin
    Read;
  End;

Procedure TSocketStream.Skip (const Count : Integer);
  Begin
    Read (Count);
  End;

Function TSocketStream.Write (const Buf; Count : Integer) : LongInt;
const SendBlockSize = 2048;
type ByteBuffer = Array [1..2140000000] of Byte;
var I, L   : Integer;
    Buffer : ByteBuffer absolute Buf;
  Begin
    if Count <= 0 then
      begin
        Result := 0;
        exit;
      end;
    I := 1;
    While (I <= Count) and FSocket.Connected do
      begin
        L := Min (Count - I + 1, SendBlockSize);
        L := FSockStream.Write (Buffer [I], L);
        if L = 0 then
          break;
        Inc (I, L);
      end;
    Result := I - 1;
  End;

Procedure TSocketStream.Write (const Data : String);
var I, L : Integer;
  Begin
    L := Length (Data);
    I := Write (Data [1], L);
    if I < L then
      raise EWriteError.Create ('Could not write to socket');
  End;

Procedure TSocketStream.Write (const Data : TOctet);
  Begin
    FSockStream.WriteBuffer (Data, 1);
  End;



{                                                                              }
{ TStreamer                                                                    }
{                                                                              }
Constructor TStreamer.Create (const Stream : TExStream);
  Begin
    inherited Create;
    FStream := Stream;
  End;

Destructor TStreamer.Destroy;
  Begin
    FreeAndNil (FStream);
    inherited Destroy;
  End;

Procedure TStreamer.Reset;
  Begin
    FStream.Reset;
  End;



{                                                                              }
{ TBinaryStreamer                                                              }
{                                                                              }
{   Uses internal Delphi formats:                                              }
{     Integer       4 byte integer (LSB first)                                 }
{     Int64         8 byte integer                                             }
{     Single        4 byte FPU Single                                          }
{     Double        8 byte FPU Double                                          }
{     Extended      10 byte FPU Extended                                       }
{     Currency      8 byte FPU fixed point                                     }
{     Boolean       1 byte, 0 = False                                          }
{     DateTime      8 byte FPU Double                                          }
{     String        Integer length + String                                    }
{     ShortString   Byte length + String                                       }
{                                                                              }
Function ReadPackedShortInt (const Stream : TExStream) : ShortInt;
  Begin
    Stream.Read (Result, Sizeof (Result));
  End;

Function ReadPackedWord (const Stream : TExStream) : Word;
  Begin
    Stream.Read (Result, Sizeof (Result));
  End;

Function ReadPackedSmallInt (const Stream : TExStream) : SmallInt;
  Begin
    Stream.Read (Result, Sizeof (Result));
  End;

Function ReadPackedInteger (const Stream : TExStream) : Integer;
  Begin
    Stream.Read (Result, Sizeof (Result));
  End;

Function ReadPackedInt64 (const Stream : TExStream) : Int64;
  Begin
    Stream.Read (Result, Sizeof (Result));
  End;

Function ReadPackedSingle (const Stream : TExStream) : Single;
  Begin
    Stream.Read (Result, Sizeof (Result));
  End;

Function ReadPackedDouble (const Stream : TExStream) : Double;
  Begin
    Stream.Read (Result, Sizeof (Result));
  End;

Function ReadPackedExtended (const Stream : TExStream) : Extended;
  Begin
    Stream.Read (Result, Sizeof (Result));
  End;

Function ReadPackedCurrency (const Stream : TExStream) : Currency;
  Begin
    Stream.Read (Result, Sizeof (Result));
  End;

Function ReadPackedBoolean (const Stream : TExStream) : Boolean;
  Begin
    Stream.Read (Result, Sizeof (Result));
  End;

Function ReadPackedDateTime (const Stream : TExStream) : TDateTime;
  Begin
    Stream.Read (Result, Sizeof (Result));
  End;

Function ReadPackedString (const Stream : TExStream) : String;
var L : Integer;
  Begin
    L := ReadPackedInteger (Stream);
    SetLength (Result, L);
    if L > 0 then
      Stream.Read (Result [1], L);
  End;

Function ReadPackedShortString (const Stream : TExStream) : ShortString;
var L : Byte;
  Begin
    L := Stream.Read;
    SetLength (Result, L);
    if L > 0 then
      Stream.Read (Result [1], L);
  End;

Procedure WritePackedShortInt (const Stream : TExStream; const D : ShortInt);
  Begin
    Stream.Write (D, Sizeof (D));
  End;

Procedure WritePackedWord (const Stream : TExStream; const D : Word);
  Begin
    Stream.Write (D, Sizeof (D));
  End;

Procedure WritePackedSmallInt (const Stream : TExStream; const D : SmallInt);
  Begin
    Stream.Write (D, Sizeof (D));
  End;

Procedure WritePackedInteger (const Stream : TExStream; const D : Integer);
  Begin
    Stream.Write (D, Sizeof (D));
  End;

Procedure WritePackedInt64 (const Stream : TExStream; const D : Int64);
  Begin
    Stream.Write (D, Sizeof (D));
  End;

Procedure WritePackedSingle (const Stream : TExStream; const D : Single);
  Begin
    Stream.Write (D, Sizeof (D));
  End;

Procedure WritePackedDouble (const Stream : TExStream; const D : Double);
  Begin
    Stream.Write (D, Sizeof (D));
  End;

Procedure WritePackedExtended (const Stream : TExStream; const D : Extended);
  Begin
    Stream.Write (D, Sizeof (D));
  End;

Procedure WritePackedCurrency (const Stream : TExStream; const D : Currency);
  Begin
    Stream.Write (D, Sizeof (D));
  End;

Procedure WritePackedBoolean (const Stream : TExStream; const D : Boolean);
  Begin
    Stream.Write (D, Sizeof (D));
  End;

Procedure WritePackedDateTime (const Stream : TExStream; const D : TDateTime);
  Begin
    Stream.Write (D, Sizeof (D));
  End;

Procedure WritePackedString (const Stream : TExStream; const D : String);
var L : Integer;
  Begin
    L := Length (D);
    WritePackedInteger (Stream, L);
    Stream.Write (D [1], L);
  End;

Procedure WritePackedShortString (const Stream : TExStream; const D : ShortString);
  Begin
    Stream.Write (D [0], Length (D) + 1);
  End;



{                                                                              }
{ TTextParser                                                                  }
{                                                                              }
Constructor TTextParser.Create (const Stream : TExStream);
  Begin
    inherited Create (Stream);
    FEOLType := [eol_CRLF, eol_EOF];
    FLineNr := 1;
    FLastNewLine := 0;
    FWhiteSpace := cs_WhiteSpace;
    FTextCaseSensitive := False;
    FTextLocaleSensitive := False;
  End;

Function TTextParser.EOF : Boolean;
  Begin
    Result := FStream.EOF;
  End;


Function TTextParser.GetCol : Integer;
  Begin
    Result := FStream.GetPosition - FLastNewLine + 1;
  End;

{ Skip                                                                         }
Procedure TTextParser.Skip;
  Begin
    FStream.Skip;
  End;

Function TTextParser.SkipChar (const Ch : CharSet) : Boolean;
  Begin
    Result := not EOF and (Char (FStream.Peek) in Ch);
    if Result then
      FStream.Skip;
  End;

Function TTextParser.SkipChars (const Ch : CharSet) : Boolean;
  Begin
    Result := SkipChar (Ch);
    if Result then
      Repeat
      Until not SkipChar (Ch);
  End;

Function TTextParser.SkipWhiteSpace : Boolean;
  Begin
    Result := SkipChars (cs_WhiteSpace)
  End;

Function TTextParser.SkipLine : Boolean;
  Begin
    While not MatchEOL do
      Skip;
    Result := SkipEOL;
  End;

Function TTextParser.ExtractEOL : String;
  Begin
  End;

Function TTextParser.SkipEOL : Boolean;
var Ch     : Char;
    CR, LF : Boolean;
    S      : String;
  Begin
    Result := (eol_EOF in FEOLType) and EOF;
    if not Result then
      begin
        CR := eol_CRLF in FEOLType;
        LF := eol_LFCR in FEOLType;
        if CR or LF then
          begin
            S := FStream.Read (2);
            Result := CR and (S = ASCII_CR + ASCII_LF) or
                      LF and (S = ASCII_LF + ASCII_CR);
            if not Result then
              begin
                Result := (eol_CR in FEOLType) and (S [1] = ASCII_CR) or
                          (eol_LF in FEOLType) and (S [1] = ASCII_LF);
                if Result then
                  FStream.Position := FStream.Position - 1 else
                  FStream.Position := FStream.Position - 2;
              end;
          end else
          begin
            CR := eol_CR in FEOLType;
            LF := eol_LF in FEOLType;
            if CR or LF then
              begin
                Ch := Char (FStream.Read);
                Result := CR and (Ch = ASCII_CR) or
                          LF and (Ch = ASCII_LF);
                if not Result then
                  FStream.Position := FStream.Position - 1;
              end;
          end;
      end;

    if Result then
      begin
        Inc (FLineNr);
        FLastNewLine := FStream.GetPosition;
      end;
  End;

{ Match                                                                        }
Function TTextParser.MatchEOL : Boolean;
var CR, LF : Boolean;
    Ch     : Char;
    S      : String;
  Begin
    Result := (eol_EOF in FEOLType) and EOF;
    if not Result then
      begin
        CR := eol_CR in FEOLType;
        LF := eol_LF in FEOLType;
        if CR or LF then
          begin
            Ch := Char (FStream.Peek);
            Result := CR and (Ch = ASCII_CR) or
                      LF and (Ch = ASCII_LF);
          end;

        if not Result then
          begin
            CR := eol_CRLF in FEOLType;
            LF := eol_LFCR in FEOLType;
            if CR or LF then
              begin
                S := FStream.Peek (2);
                Result := CR and (S = ASCII_CR + ASCII_LF) or
                          LF and (S = ASCII_LF + ASCII_CR);
              end;
          end;
      end;
  End;

Function TTextParser.MatchChar (const Ch : CharSet) : Boolean;
  Begin
    Result := not EOF and (Char (FStream.Peek) in Ch);
  End;

Function TTextParser.MatchWhiteSpace : Boolean;
  Begin
    Result := MatchChar (FWhiteSpace);
  End;

Function TTextParser.Match (const Ch : Char) : Boolean;
  Begin
    Result := not EOF and (Char (FStream.Peek) = Ch);
  End;

Function TTextParser.Match (const S : String) : Boolean;
var L, R : Integer;
  Begin
    L := Length (S);
    R := FStream.Remaining;
    if (R >= 0) and (R < L) then
      begin
        Result := False;
        exit;
      end;

    Result := FStream.Peek (L) = S;
  End;

Function TTextParser.MatchText (const Txt : String) : Boolean;
var L, R : Integer;
  Begin
    L := Length (Txt);
    R := FStream.Remaining;
    if (R >= 0) and (R < L) then
      begin
        Result := False;
        exit;
      end;

    Result := AnsiCompareText (Txt, FStream.Peek (L)) = 0;
  End;

Function TTextParser.MatchLine (const S : String) : Boolean;
var T : String;
    P : Int64;
  Begin
    P := FStream.Position;
    T := Extract (S);
    Result := (T = S) and MatchEOL;
    if Result then
      SkipEOL else
      FStream.Position := P;
  End;

Function TTextParser.MatchTextLine (const S : String) : Boolean;
var T : String;
    P : Int64;
  Begin
    P := FStream.Position;
    T := Extract (S);
    Result := (AnsiCompareText (T, S) = 0) and MatchEOL;
    if Result then
      SkipEOL else
      FStream.Position := P;
  End;

{ Extract                                                                      }
Function TTextParser.ExtractChar (const Ch : CharSet) : String;
  Begin
    if MatchChar (Ch) then
      Result := Extract else
      Result := '';
  End;

Function TTextParser.Extract : Char;
  Begin
    Result := Char (FStream.Read);
  End;

Function TTextParser.Extract (const S : String) : String;
  Begin
    if Match (S) then
      begin
        FStream.Skip (Length (S));
        Result := S;
      end else
      Result := '';
  End;

Function TTextParser.ExtractText (const Txt : String) : String;
  Begin
    if MatchText (Txt) then
      Result := FStream.Read (Length (Txt)) else
      Result := '';
  End;

Function TTextParser.ExtractToEOF : String;
  Begin
    Result := '';
    While not EOF do
      Result := Result + FStream.Read (Max (1, FStream.Remaining));
  End;

Function TTextParser.ExtractWhiteSpace : String;
  Begin
    Result := ExtractChars (FWhiteSpace);
  End;

{ ExtractDelim                                                                 }
Function TTextParser.ExtractChars (const Ch : CharSet) : String;
  Begin
    Result := '';
    While not EOF and (Char (FStream.Peek) in Ch) do
      Result := Result + Char (FStream.Read);
  End;

Function TTextParser.SkipTo (const Delimiter : String) : Boolean;
var I : Int64;
  Begin
    I := FStream.Locate (Delimiter);
    Result := I >= 0;
    if I > 0 then
      FStream.Skip (I);
  End;

Function TTextParser.ExtractTo (const Delimiter : CharSet; const SkipDelimiter : Boolean = False) : String;
  Begin
    Result := '';
    While not EOF and not (Char (FStream.Peek) in Delimiter) do
      Result := Result + Char (FStream.Read);
    if SkipDelimiter then
      Skip;
  End;

Function TTextParser.ExtractTo (const Delimiter : String; const SkipDelimiter : Boolean = False) : String;
var R : Boolean;
  Begin
    Result := '';
    R := False;
    While not EOF and not R do
      begin
        R := FStream.Peek (Length (Delimiter)) = Delimiter;
        if not R then
          Result := Result + Char (FStream.Read);
      end;
    if SkipDelimiter and R then
      FStream.Skip (Length (Delimiter));
  End;

Function TTextParser.ExtractQuoted (const Quote : Char; const AllowQuotedQuote : Boolean) : String;
var Fin   : Boolean;
  Begin
    if Match (Quote) then
      begin
        Extract;
        Result := '';
        Repeat
          Result := Result + ExtractTo ([Quote]);
          if EOF then
            raise EStream.Create ('No closing quote (' + Quote + ' expected)');
          Skip;
          Fin := not AllowQuotedQuote or not Match (Quote);
          if not Fin then
            begin
              Result := Result + Quote;
              Skip;
            end;
        Until Fin;
      end else
      Result := '';
  End;

Function TTextParser.ExtractNestedPair (const OpenQuote, CloseQuote : String; const LiteralsOpenQuote, LiteralsCloseQuote : StringArray) : String;
  Begin
  End;

Function TTextParser.ExtractLine : String;
  Begin
    Result := '';
    While not MatchEOL do
      Result := Result + Extract;
    SkipEOL;
  End;

Function TTextParser.MatchKeyword (const Keywords : StringArray; const Delimiter : CharSet) : Integer;
var I, L : Integer;
    S    : String;
  Begin
    For I := 0 to High (Keywords) do
      if MatchText (Keywords [I]) then
        begin
          S := FStream.Peek (Length (Keywords [I]) + 1);
          L := Length (S);
          if (L = Length (Keywords [I])) or (Delimiter = []) or (S [L] in Delimiter) then
            begin
              Result := I;
              exit;
            end;
        end;
    Result := -1;
  End;

Function TTextParser.ExtractKeyword (const Keywords : StringArray; const Delimiter : CharSet) : String;
var I : Integer;
  Begin
    I := MatchKeyword (Keywords, Delimiter);
    if I = -1 then
      Result := '' else
      Result := FStream.Read (Length (Keywords [I]));
  End;

Function TTextParser.GetBookmark : Int64;
  Begin
    Result := FStream.Position;
  End;

Procedure TTextParser.GotoBookmark (const B : Int64);
  Begin
    FStream.Position := B;
  End;



{                                                                              }
{ TRecordStreamer                                                              }
{                                                                              }
Constructor TRecordStreamer.Create (const Stream : TExStream; const RecordSize : Integer; const FirstRecordOffset : Int64);
  Begin
    if Stream.SequentialOnly then
      raise EStream.Create ('RecordStreamer must have a random access stream.');
    inherited Create (Stream);
    FRecordSize := RecordSize;
    FRecordOffset := FirstRecordOffset;
  End;

Function TRecordStreamer.GetRecordCount : Int64;
  Begin
    Result := (FStream.Size - FRecordOffset) div FRecordSize;
  End;

Procedure TRecordStreamer.SetRecordCount (const NewRecordCount : Int64);
  Begin
    FStream.SetStreamSize (FRecordOffset + NewRecordCount * FRecordSize);
  End;

Procedure TRecordStreamer.SetRecordSize (const RecordSize : Integer);
  Begin
    if RecordSize <= 0 then
      raise EStream.Create ('Invalid record size') else
      FRecordSize := RecordSize;
  End;

Procedure TRecordStreamer.Seek (const Idx : Int64);
  Begin
    FStream.Position := FRecordOffset + Idx * FRecordSize;
  End;

Procedure TRecordStreamer.Read (const Idx : Int64; const Count : Integer; var Buf);
  Begin
    Seek (Idx);
    FStream.Read (Buf, Count * FRecordSize);
  End;

Procedure TRecordStreamer.Write (const Idx : Int64; const Count : Integer; const Buf);
  Begin
    Seek (Idx);
    FStream.Write (Buf, Count * FRecordSize);
  End;

Procedure TRecordStreamer.Insert (const Idx : Int64; const Count : Integer);
  Begin
    Seek (Idx);
    FStream.Insert (Count * FRecordSize);
  End;

Procedure TRecordStreamer.Delete (const Idx : Int64; const Count : Integer);
  Begin
    Seek (Idx);
    FStream.Delete (Count * FRecordSize);
  End;

Procedure TRecordStreamer.Clear (const Idx : Int64; const Count : Integer);
const BufSize = 1024;
var Buffer  : String;
    L, C, I : Integer;
  Begin
    C := Min (Count, BufSize);
    L := FRecordSize * C;
    SetLength (Buffer, L);
    FillChar (Buffer [1], L, #0);
    For I := 0 to Count div C do
      Write (Idx + I * C, C, Buffer [1]);
  End;

{                                                                              }
{ TIntegerArrayStreamer                                                        }
{                                                                              }
Constructor TIntegerArrayStreamer.Create (const Stream : TExStream);
  Begin
    inherited Create (Stream, Sizeof (Integer));
  End;

Procedure TIntegerArrayStreamer.SetItem (const Idx : Integer; const Value : Integer);
  Begin
    Write (Idx, 1, Value);
  End;

Function TIntegerArrayStreamer.GetItem (const Idx : Integer) : Integer;
  Begin
    Read (Idx, 1, Result);
  End;

Procedure TIntegerArrayStreamer.Fill (const Value : Integer);
const BlockSize = 1024;
var C, I : Integer;
    RC   : Int64;
    A    : IntegerArray;
  Begin
    RC := RecordCount;
    C := Min (RC, BlockSize);
    A := DupInteger (Value, C);
    For I := 0 to (RC - 1) div C do
      Write (I * C, Min (C, RC - I * C), A [0]);
  End;

Procedure TIntegerArrayStreamer.FillInc (const StartValue : Integer; const Increment : Integer);
const BlockSize = 1024;
var C, I : Integer;
    RC   : Int64;
    A    : IntegerArray;
  Begin
    RC := RecordCount;
    C := Min (RC, BlockSize);
    For I := 0 to (RC - 1) div C do
      begin
        A := RangeInteger (StartValue + I * C * Increment, C, Increment);
        Write (I * C, Min (C, RC - I * C), A [0]);
      end;
  End;

Function TIntegerArrayStreamer.GetAsIntegerArray : IntegerArray;
var L : Integer;
  Begin
    L := Count;
    SetLength (Result, L);
    Read (0, L, Result [0]);
  End;

Procedure TIntegerArrayStreamer.SetAsIntegerArray (const V : IntegerArray);
var L : Integer;
  Begin
    L := Length (V);
    Count := L;
    Write (0, L, V [0]);
  End;

Function TIntegerArrayStreamer.PosNext (const Find : Integer; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Integer;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else
      begin // linear search
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;



{                                                                              }
{ TInt64ArrayStreamer                                                          }
{                                                                              }
Constructor TInt64ArrayStreamer.Create (const Stream : TExStream);
  Begin
    inherited Create (Stream, Sizeof (Int64));
  End;

Procedure TInt64ArrayStreamer.SetItem (const Idx : Integer; const Value : Int64);
  Begin
    Write (Idx, 1, Value);
  End;

Function TInt64ArrayStreamer.GetItem (const Idx : Integer) : Int64;
  Begin
    Read (Idx, 1, Result);
  End;

Procedure TInt64ArrayStreamer.Fill (const Value : Int64);
const BlockSize = 1024;
var C, I : Integer;
    RC   : Int64;
    A    : Int64Array;
  Begin
    RC := RecordCount;
    C := Min (RC, BlockSize);
    A := DupInt64 (Value, C);
    For I := 0 to (RC - 1) div C do
      Write (I * C, Min (C, RC - I * C), A [0]);
  End;

Procedure TInt64ArrayStreamer.FillInc (const StartValue : Int64; const Increment : Int64);
const BlockSize = 1024;
var C, I : Integer;
    RC   : Int64;
    A    : Int64Array;
  Begin
    RC := RecordCount;
    C := Min (RC, BlockSize);
    For I := 0 to (RC - 1) div C do
      begin
        A := RangeInt64 (StartValue + I * C * Increment, C, Increment);
        Write (I * C, Min (C, RC - I * C), A [0]);
      end;
  End;

Function TInt64ArrayStreamer.GetAsInt64Array : Int64Array;
var L : Integer;
  Begin
    L := Count;
    SetLength (Result, L);
    Read (0, L, Result [0]);
  End;

Procedure TInt64ArrayStreamer.SetAsInt64Array (const V : Int64Array);
var L : Integer;
  Begin
    L := Length (V);
    Count := L;
    Write (0, L, V [0]);
  End;

Function TInt64ArrayStreamer.PosNext (const Find : Int64; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Int64;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else
      begin // linear search
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;



{                                                                              }
{ TSingleArrayStreamer                                                         }
{                                                                              }
Constructor TSingleArrayStreamer.Create (const Stream : TExStream);
  Begin
    inherited Create (Stream, Sizeof (Single));
  End;

Procedure TSingleArrayStreamer.SetItem (const Idx : Integer; const Value : Single);
  Begin
    Write (Idx, 1, Value);
  End;

Function TSingleArrayStreamer.GetItem (const Idx : Integer) : Single;
  Begin
    Read (Idx, 1, Result);
  End;

Procedure TSingleArrayStreamer.Fill (const Value : Single);
const BlockSize = 1024;
var C, I : Integer;
    RC   : Int64;
    A    : SingleArray;
  Begin
    RC := RecordCount;
    C := Min (RC, BlockSize);
    A := DupSingle (Value, C);
    For I := 0 to (RC - 1) div C do
      Write (I * C, Min (C, RC - I * C), A [0]);
  End;

Procedure TSingleArrayStreamer.FillInc (const StartValue : Single; const Increment : Single);
const BlockSize = 1024;
var C, I : Integer;
    RC   : Int64;
    A    : SingleArray;
  Begin
    RC := RecordCount;
    C := Min (RC, BlockSize);
    For I := 0 to (RC - 1) div C do
      begin
        A := RangeSingle (StartValue + I * C * Increment, C, Increment);
        Write (I * C, Min (C, RC - I * C), A [0]);
      end;
  End;

Function TSingleArrayStreamer.GetAsSingleArray : SingleArray;
var L : Integer;
  Begin
    L := Count;
    SetLength (Result, L);
    Read (0, L, Result [0]);
  End;

Procedure TSingleArrayStreamer.SetAsSingleArray (const V : SingleArray);
var L : Integer;
  Begin
    L := Length (V);
    Count := L;
    Write (0, L, V [0]);
  End;

Function TSingleArrayStreamer.PosNext (const Find : Single; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Single;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else
      begin // linear search
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;



{                                                                              }
{ TDoubleArrayStreamer                                                         }
{                                                                              }
Constructor TDoubleArrayStreamer.Create (const Stream : TExStream);
  Begin
    inherited Create (Stream, Sizeof (Double));
  End;

Procedure TDoubleArrayStreamer.SetItem (const Idx : Integer; const Value : Double);
  Begin
    Write (Idx, 1, Value);
  End;

Function TDoubleArrayStreamer.GetItem (const Idx : Integer) : Double;
  Begin
    Read (Idx, 1, Result);
  End;

Procedure TDoubleArrayStreamer.Fill (const Value : Double);
const BlockSize = 1024;
var C, I : Integer;
    RC   : Int64;
    A    : DoubleArray;
  Begin
    RC := RecordCount;
    C := Min (RC, BlockSize);
    A := DupDouble (Value, C);
    For I := 0 to (RC - 1) div C do
      Write (I * C, Min (C, RC - I * C), A [0]);
  End;

Procedure TDoubleArrayStreamer.FillInc (const StartValue : Double; const Increment : Double);
const BlockSize = 1024;
var C, I : Integer;
    RC   : Int64;
    A    : DoubleArray;
  Begin
    RC := RecordCount;
    C := Min (RC, BlockSize);
    For I := 0 to (RC - 1) div C do
      begin
        A := RangeDouble (StartValue + I * C * Increment, C, Increment);
        Write (I * C, Min (C, RC - I * C), A [0]);
      end;
  End;

Function TDoubleArrayStreamer.GetAsDoubleArray : DoubleArray;
var L : Integer;
  Begin
    L := Count;
    SetLength (Result, L);
    Read (0, L, Result [0]);
  End;

Procedure TDoubleArrayStreamer.SetAsDoubleArray (const V : DoubleArray);
var L : Integer;
  Begin
    L := Length (V);
    Count := L;
    Write (0, L, V [0]);
  End;

Function TDoubleArrayStreamer.PosNext (const Find : Double; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Double;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else
      begin // linear search
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;



{                                                                              }
{ TExtendedArrayStreamer                                                       }
{                                                                              }
Constructor TExtendedArrayStreamer.Create (const Stream : TExStream);
  Begin
    inherited Create (Stream, Sizeof (Extended));
  End;

Procedure TExtendedArrayStreamer.SetItem (const Idx : Integer; const Value : Extended);
  Begin
    Write (Idx, 1, Value);
  End;

Function TExtendedArrayStreamer.GetItem (const Idx : Integer) : Extended;
  Begin
    Read (Idx, 1, Result);
  End;

Procedure TExtendedArrayStreamer.Fill (const Value : Extended);
const BlockSize = 1024;
var C, I : Integer;
    RC   : Int64;
    A    : ExtendedArray;
  Begin
    RC := RecordCount;
    C := Min (RC, BlockSize);
    A := DupExtended (Value, C);
    For I := 0 to (RC - 1) div C do
      Write (I * C, Min (C, RC - I * C), A [0]);
  End;

Procedure TExtendedArrayStreamer.FillInc (const StartValue : Extended; const Increment : Extended);
const BlockSize = 1024;
var C, I : Integer;
    RC   : Int64;
    A    : ExtendedArray;
  Begin
    RC := RecordCount;
    C := Min (RC, BlockSize);
    For I := 0 to (RC - 1) div C do
      begin
        A := RangeExtended (StartValue + I * C * Increment, C, Increment);
        Write (I * C, Min (C, RC - I * C), A [0]);
      end;
  End;

Function TExtendedArrayStreamer.GetAsExtendedArray : ExtendedArray;
var L : Integer;
  Begin
    L := Count;
    SetLength (Result, L);
    Read (0, L, Result [0]);
  End;

Procedure TExtendedArrayStreamer.SetAsExtendedArray (const V : ExtendedArray);
var L : Integer;
  Begin
    L := Length (V);
    Count := L;
    Write (0, L, V [0]);
  End;

Function TExtendedArrayStreamer.PosNext (const Find : Extended; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : Extended;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else
      begin // linear search
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;




{                                                                              }
{ TShortStringArrayStreamer                                                    }
{                                                                              }
Constructor TShortStringArrayStreamer.Create (const Stream : TExStream; const StringLen : Byte = 255);
  Begin
    inherited Create (Stream, StringLen + 1);
  End;

Procedure TShortStringArrayStreamer.SetItem (const Idx : Integer; const Value : ShortString);
  Begin
    Write (Idx, 1, Value [0]);
  End;

Function TShortStringArrayStreamer.GetItem (const Idx : Integer) : ShortString;
  Begin
    Read (Idx, 1, Result [0]);
  End;

Procedure TShortStringArrayStreamer.Fill (const Value : ShortString);
var I : Integer;
  Begin
    For I := 0 to RecordCount - 1 do
      Write (I, 1, Value [0]);
  End;
Function TShortStringArrayStreamer.PosNext (const Find : ShortString; const PrevPos : Integer; const IsSortedAscending : Boolean) : Integer;
var I, L, H : Integer;
    D       : ShortString;
  Begin
    if IsSortedAscending then // binary search
      begin
        if Max (PrevPos + 1, 0) = 0 then // find first
          begin
            L := 0;
            H := Count - 1;
            Repeat
              I := (L + H) div 2;
              D := Item [I];
              if D = Find then
                begin
                  While (I > 0) and (Item [I - 1] = Find) do
                    Dec (I);
                  Result := I;
                  exit;
                end else
              if D > Find then
                H := I - 1 else
                L := I + 1;
            Until L > H;
            Result := -1;
          end else // find next
          if PrevPos >= Count - 1 then
            Result := -1 else
            if Item [PrevPos + 1] = Find then
              Result := PrevPos + 1 else
              Result := -1;
      end else
      begin // linear search
        For I := Max (PrevPos + 1, 0) to Count - 1 do
          if Item [I] = Find then
            begin
              Result := I;
              exit;
            end;
        Result := -1;
      end;
  End;




{                                                                              }
{ TBitArrayStreamer                                                            }
{                                                                              }
type
  TBitArrayHeader = packed record
    Count : Int64;
  end;

Function GetBitArrayHeader (const S : TExStream) : TBitArrayHeader;
  Begin
    S.Position := 0;
    S.Read (Result, Sizeof (Result));
  End;

Constructor TBitArrayStreamer.Create (const Stream : TExStream);
  Begin
    inherited Create (Stream, Sizeof (Byte), Sizeof (TBitArrayHeader));
  End;

Procedure TBitArrayStreamer.SetCount (const NewCount : Int64);
  Begin
    SetRecordCount ((NewCount + 7) div 8);
    FStream.Position := 0;
    FStream.Write (NewCount, Sizeof (Int64));
  End;

Function TBitArrayStreamer.GetCount : Int64;
  Begin
    FStream.Position := 0;
    FStream.Read (Result, Sizeof (Int64));
  End;

Procedure TBitArrayStreamer.SetItem (const Idx : Int64; const Value : Boolean);
var B, C : Byte;
    I    : Int64;
  Begin
    I := Idx div 8;
    C := 1 shl (Idx mod 8);
    Read (I, 1, B);
    if Value then
      B := B or C else
      B := B and not C;
    Write (I, 1, B);
  End;

Function TBitArrayStreamer.GetItem (const Idx : Int64) : Boolean;
var B : Byte;
  Begin
    Read (Idx div 8, 1, B);
    Result := B or (1 shl (Idx mod 8)) <> 0;
  End;

Procedure TBitArrayStreamer.FillRange (const LoIdx, HiIdx : Int64; const Value : Boolean);
const BlockSize = 1024;
var C, I  : Integer;
    RC, J : Int64;
    A     : ByteArray;
    H, L  : Int64;
  Begin
    H := Min (HiIdx, Count);
    L := Max (0, LoIdx);
    RC := H - L + 1;
    C := Min (RC, BlockSize);
    if Value then
      A := DupByte ($FF, C) else
      A := DupByte ($00, C);
    For I := 0 to (RC - 1) div C do
      begin
        J := I * C;
        Write (L + J, Min (C, RC - J), A [0]);
      end;
  End;

Procedure TBitArrayStreamer.Fill (const Value : Boolean);
  Begin
    FillRange (0, MaxInt64, Value);
  End;

Function TBitArrayStreamer.CompareRange (const LoIdx, HiIdx : Int64; const Value : Boolean) : Boolean;
  Begin
  End;

Procedure TBitArrayStreamer.Invert;
  Begin
  End;

Function TBitArrayStreamer.Find (const Value : Boolean; const Start : Integer; const FindForward : Boolean) : Integer;
  Begin
  End;

Function TBitArrayStreamer.FindRange (const Value : Boolean; const Start : Integer; const Count : Integer; const FindForward : Boolean) : Integer;
  Begin
  End;

end.

