unit cSelfTest;

interface

Procedure DoSelfTest;

implementation

uses
  Classes,
  SysUtils,
  Dialogs,
  Controls,

  cUtils,
  cDateTime,
  cStrings,
  cStreams,
  cMaths;

const
  ShowIndividualErrors = False;

type
  TUnitTestProcedure = Procedure;
  TObjectTestProcedure = Procedure (const X : TObject);
  ETestResults = class (Exception);

var
  TestReports    : TStringList;
  TestStats      : String;

  FLastAssertion : String;
  FTestUnit      : String;
  FTestObject    : TObject;
  FErrorCount,
  FAssertCount,
  FObjTestCount,
  FUnitCount     : Integer;

Procedure SetUnitStats;
  Begin
    if FErrorCount = 0 then
      TestStats := TestStats + 'PASS' else
      TestStats := TestStats + IntToStr (FErrorCount) + ' errors in ';
    TestStats := TestStats + ' unit ' + FTestUnit;
    if FAssertCount > 0 then
      TestStats := TestStats + ', ' + IntToStr (FAssertCount) + ' assertions made';
    if FObjTestCount > 0 then
      TestStats := TestStats + ', ' + IntToStr (FObjTestCount) + ' object tests made';
    TestStats := TestStats + '.'#13;
    FObjTestCount := 0;
    FAssertCount := 0;
    FErrorCount := 0;
  End;

Procedure ShowError (const E : Exception);
var S, T, L : String;
  Begin
    Inc (FErrorCount);

    S := 'Self-test failed in ' + FTestUnit + ':'#13#13;
    L := '* ' + FTestUnit;

    S := S + E.ClassName + ' exception raised ' + Cond (FLastAssertion = '', 'before any assertions were made',
             'after assertion ' + FLastAssertion) + '.'#13#13;

    if Assigned (FTestObject) then
      begin
        T := FTestObject.ClassName;
        {$IFDEF L1}
        if FTestObject is AType then
          try
            T := T + ' (''' + AType (FTestObject).AsString + ''')';
          except end;
        {$ENDIF}
        S := S + 'Class ' + T + #13#13;
        L := L + '.' + T;
      end;

    S := S + E.Message;
    L := L + '  ' + E.Message + Cond (FLastAssertion = '', '', ' after assertion ' + FLastAssertion);

    TestReports.Add (L);
    if ShowIndividualErrors then
      if MessageDlg (S, mtError, [mbOk, mbCancel], 0) = mrCancel then
        Halt;
  End;

Procedure DoUnitTest (const TestUnit : String; const P : TUnitTestProcedure);
  Begin
    Inc (FUnitCount);
    if FTestUnit <> '' then
      SetUnitStats;

    try
      FLastAssertion := '';
      FTestUnit := TestUnit;
      P;
    except
      on E : Exception do ShowError (E);
    end;
  End;

Procedure DoObjectTest (const O : TObject; const P : TObjectTestProcedure);
  Begin
    Inc (FObjTestCount);

    try
      FTestObject := O;
      P (O);
    except
      on E : Exception do ShowError (E);
    end;
    FTestObject := nil;
  End;

Procedure CheckAssertion (const Assertion : Boolean; const Code : String);
  Begin
    Inc (FAssertCount);

    if not Assertion then
      ShowError (EAssertionFailed.Create ('Assertion ' + Code + ' failed')) else
      FLastAssertion := Code;
  End;



{                                                                              }
{ cUtil                                                                        }
{                                                                              }
Procedure cUtilSelfTest;
var S, T : StringArray;
    I, J : IntegerArray;
    F, G : Integer;
    U, V : String;
  Begin
    CheckAssertion (Cond (True, 1, 2) = 1,              '#Cond.1');
    CheckAssertion (Cond (False, 1, 2) = 2,             '#Cond.2');
    CheckAssertion (Cond (True, 1.1, 2.2) = 1.1,        '#Cond.3');
    CheckAssertion (Cond (False, 1.1, 2.2) = 2.2,       '#Cond.4');
    CheckAssertion (Cond (True, '1', '2') = '1',        '#Cond.5');
    CheckAssertion (Cond (False, '1', '2') = '2',       '#Cond.6');

    F := $12345678;
    G := $55555555;
    Swap (F, G);
    CheckAssertion (F = $55555555,                      'IntSwap.1');
    CheckAssertion (G = $12345678,                      'IntSwap.2');
    U := '12345678';
    V := '55555555';
    Swap (U, V);
    CheckAssertion (U = '55555555',                     'StrSwap.1');
    CheckAssertion (V = '12345678',                     'StrSwap.2');

    S := nil;
    For F := 1 to 100 do
      begin
        Append (S, IntToStr (F));
        CheckAssertion (Length (S) = F,                 '#StringArray.Append.1.' + IntToStr (F));
        CheckAssertion (S [F - 1] = IntToStr (F),       '#StringArray.Append.2.' + IntToStr (F));
      end;
    T := Copy (S);
    Append (S, T);
    For F := 1 to 100 do
      CheckAssertion (S [F + 99] = IntToStr (F),        '#StringArray.Append.3.' + IntToStr (F));
    CheckAssertion (PosNext ('60', S) = 59,             '#StringArray.PosNext.1');
    CheckAssertion (PosNext ('60', T) = 59,             '#StringArray.PosNext.2');
    CheckAssertion (PosNext ('60', S, 59) = 159,        '#StringArray.PosNext.3');
    CheckAssertion (PosNext ('60', T, 59) = -1,         '#StringArray.PosNext.4');
    CheckAssertion (PosNext ('60', T, -1, True) = 59,   '#StringArray.PosNext.5');
    CheckAssertion (PosNext ('60', T, 59, True) = -1,   '#StringArray.PosNext.6');
    For F := 1 to 100 do
      begin
        Remove (S, PosNext (IntToStr (F), S), 1);
        CheckAssertion (Length (S) = 200 - F,           '#StringArray.Remove.1.' + IntToStr (F));
      end;
    For F := 99 downto 0 do
      begin
        Remove (S, PosNext (IntToStr (F xor 3 + 1), S), 1);
        CheckAssertion (Length (S) = F,                 '#StringArray.Remove.2.' + IntToStr (F));
      end;

    I := nil;
    For F := 1 to 100 do
      begin
        Append (I, F);
        CheckAssertion (Length (I) = F,                 '#IntegerArray.Append.1.' + IntToStr (F));
        CheckAssertion (I [F - 1] = F,                  '#IntegerArray.Append.2.' + IntToStr (F));
      end;
    J := Copy (I);
    Append (I, J);
    For F := 1 to 100 do
      CheckAssertion (I [F + 99] = F,                   '#IntegerArray.Append.3.' + IntToStr (F));
    CheckAssertion (PosNext (60, I) = 59,               '#IntegerArray.PosNext.1');
    CheckAssertion (PosNext (60, J) = 59,               '#IntegerArray.PosNext.2');
    CheckAssertion (PosNext (60, I, 59) = 159,          '#IntegerArray.PosNext.3');
    CheckAssertion (PosNext (60, J, 59) = -1,           '#IntegerArray.PosNext.4');
    CheckAssertion (PosNext (60, J, -1, True) = 59,     '#IntegerArray.PosNext.5');
    CheckAssertion (PosNext (60, J, 59, True) = -1,     '#IntegerArray.PosNext.6');
    For F := 1 to 100 do
      begin
        Remove (I, PosNext (F, I), 1);
        CheckAssertion (Length (I) = 200 - F,           '#IntegerArray.Remove.1.' + IntToStr (F));
      end;
    For F := 99 downto 0 do
      begin
        Remove (I, PosNext (F xor 3 + 1, I, -1, True), 1);
        CheckAssertion (Length (I) = F,                 '#IntegerArray.Remove.2.' + IntToStr (F));
      end;

    S := AsStringArray (['C', 'A', 'B', 'E', 'D']);
    CheckAssertion (IsEqual (S, StrToStringArray (StringArrayToStr (S))), '#ArrayToStr.1');
    I := AsIntegerArray ([3, 1, 2, 5, 4]);
    CheckAssertion (IsEqual (I, StrToIntegerArray (IntegerArrayToStr (I))), '#ArrayToStr.2');

    Sort (S);
    CheckAssertion (S [0] = 'A', '#StringSort.1');
    CheckAssertion (S [1] = 'B', '#StringSort.2');
    CheckAssertion (S [2] = 'C', '#StringSort.3');
    CheckAssertion (S [3] = 'D', '#StringSort.4');
    CheckAssertion (S [4] = 'E', '#StringSort.5');

    Sort (I);
    CheckAssertion (I [0] = 1, '#IntegerSort.1');
    CheckAssertion (I [1] = 2, '#IntegerSort.2');
    CheckAssertion (I [2] = 3, '#IntegerSort.3');
    CheckAssertion (I [3] = 4, '#IntegerSort.4');
    CheckAssertion (I [4] = 5, '#IntegerSort.5');
  End;



{                                                                              }
{ cDateTime                                                                    }
{                                                                              }
Procedure cDateTimeSelfTest;
var D1, D2 : TDateTime;
    I      : Integer;
  Begin
    CheckAssertion (DayOfYear (2000, 1, 1) = 1,          '#DayOfYear.1');
    CheckAssertion (DayOfYear (2000, 12, 31) = 366,      '#DayOfYear.2');
    CheckAssertion (DaysInMonth (2000, 2) = 29,          '#DaysInMonth.1');
    CheckAssertion (DaysInMonth (1999, 2) = 28,          '#DaysInMonth.2');
    CheckAssertion (DaysInMonth (2000, 12) = 31,         '#DaysInMonth.3');
    CheckAssertion (DaysInYear (2000) = 366,             '#DaysInYear.1');
    CheckAssertion (DaysInYear (1999) = 365,             '#DaysInYear.2');

    CheckAssertion (DiffDays (EncodeDate (2000, 1, 31), EncodeDate (2000, 3, 1)) = 30,   '#Diff.1');
    CheckAssertion (DiffDays (EncodeDate (2000, 3, 1), EncodeDate (2000, 1, 31)) = -30,  '#Diff.2');
    CheckAssertion (DiffDays (EncodeDate (2000, 1, 1), EncodeDate (2001, 1, 1)) = 366,   '#Diff.3');
    CheckAssertion (DiffMonths (EncodeDate (1999, 1, 5), EncodeDate (1999, 2, 4)) = 0,   '#Diff.4');
    CheckAssertion (DiffMonths (EncodeDate (1999, 1, 5), EncodeDate (1999, 2, 5)) = 1,   '#Diff.5');
    CheckAssertion (DiffMonths (EncodeDate (1999, 1, 5), EncodeDate (1999, 3, 6)) = 2,   '#Diff.6');
    CheckAssertion (DiffMonths (EncodeDate (1999, 1, 5), EncodeDate (1998, 12, 6)) = 0,  '#Diff.7');
    CheckAssertion (DiffMonths (EncodeDate (1999, 1, 5), EncodeDate (1998, 12, 5)) = -1, '#Diff.8');
    CheckAssertion (DiffYears (EncodeDate (1999, 1, 5), EncodeDate (2001, 1, 4)) = 1,    '#Diff.9');
    CheckAssertion (DiffYears (EncodeDate (1999, 1, 5), EncodeDate (2001, 1, 5)) = 2,    '#Diff.10');
    CheckAssertion (DiffYears (EncodeDate (1999, 1, 5), EncodeDate (1998, 1, 6)) = 0,    '#Diff.11');
    CheckAssertion (DiffYears (EncodeDate (1999, 1, 5), EncodeDate (1998, 1, 5)) = -1,   '#Diff.12');

    D1 := EncodeDate (2000, 1, 1) + EncodeTime (0, 0, 0, 0);
    D2 := EncodeDate (2000, 12, 31) + EncodeTime (23, 59, 59, 999);
    CheckAssertion (DiffHours (D1, D2) = 366*24 - 1,                            '#Diff.2.1');
    CheckAssertion (DiffMinutes (D1, D2) = 366*24*60 - 1,                       '#Diff.2.2');
    CheckAssertion (DiffSeconds (D1, D2) = 366*24*60*60 - 1,                    '#Diff.2.3');
    CheckAssertion (Abs (DiffMilliseconds (D1, D2) - 31622399999) <= 10,        '#Diff.2.4');

    CheckAssertion (ANSIToDateTime (2000366) = EncodeDate (2000, 12, 31),        '#ANSI.1');
    CheckAssertion (DateTimeToANSI (EncodeDate (2000, 12, 31)) = 2000366,        '#ANSI.2');
    CheckAssertion (ISOIntegerToDateTime (20001231) = EncodeDate (2000, 12, 31), '#ISO.1');
    CheckAssertion (DateTimeToISOInteger (EncodeDate (2000, 12, 31)) = 20001231, '#ISO.2');

    CheckAssertion (IsEqual (GoodFriday (2000), EncodeDate (2000, 4, 21)),      '#GoodFriday');

    Randomize;
    For I := 1 to 1000 do
      begin
        try
          D1 := EncodeDate (1900 + Random (200), Random (12) + 1, Random (31) + 1);
        except
          continue;
        end;
        CheckAssertion (IsEqual (ANSIToDateTime (DateTimeToANSI (D1)), D1),             '#Conv.ANSI.' + IntToStr (I));
        CheckAssertion (IsEqual (ISOIntegerToDateTime (DateTimeToISOInteger (D1)), D1), '#Conv.ISO.' + IntToStr (I));
      end;
  End;



{                                                                              }
{ cStrings                                                                     }
{                                                                              }
Procedure cStringsSelfTest;
var S1 : String;
    Ch : Char;
  Begin
    CheckAssertion (CopyRange ('1234567890', 5, 7) = '567',             '#Copy.1');
    CheckAssertion (CopyRange ('1234567890', 1, 1) = '1',               '#Copy.2');
    CheckAssertion (CopyRange ('1234567890', 0, 11) = '1234567890',     '#Copy.3');
    CheckAssertion (CopyRange ('1234567890', 7, 4) = '',                '#Copy.4');
    CheckAssertion (CopyFrom ('1234567890', 8) = '890',                 '#Copy.5');
    CheckAssertion (CopyFrom ('1234567890', 11) = '',                   '#Copy.6');
    CheckAssertion (CopyFrom ('1234567890', 0) = '1234567890',          '#Copy.7');
    CheckAssertion (CopyLeft ('1234567890', 3) = '123',                 '#Copy.8');
    CheckAssertion (CopyLeft ('1234567890', 11) = '1234567890',         '#Copy.9');
    CheckAssertion (CopyLeft ('1234567890', 0) = '',                    '#Copy.10');
    CheckAssertion (CopyRight ('1234567890', 3) = '890',                '#Copy.11');
    CheckAssertion (CopyRight ('1234567890', 11) = '1234567890',        '#Copy.12');
    CheckAssertion (CopyRight ('1234567890', 0) = '',                   '#Copy.13');

    CheckAssertion (CopyBefore ('1234543210', '4', True) = '123',               '#CopyBefore.1');
    CheckAssertion (CopyBefore ('1234543210', '4', False) = '123',              '#CopyBefore.2');
    CheckAssertion (CopyBefore ('1234543210', '6', True) = '1234543210',        '#CopyBefore.3');
    CheckAssertion (CopyBefore ('1234543210', '6', False) = '',                 '#CopyBefore.4');
    CheckAssertion (CopyBefore ('1234543210', ['2', '4'], True) = '1',          '#CopyBefore.5');
    CheckAssertion (CopyBefore ('1234543210', ['2', '4'], False) = '1',         '#CopyBefore.6');
    CheckAssertion (CopyBefore ('1234543210', ['6', 'a'], True) = '1234543210', '#CopyBefore.7');
    CheckAssertion (CopyBefore ('1234543210', ['6', 'a'], False) = '',          '#CopyBefore.8');
    CheckAssertion (CopyAfter ('1234543210', '4', True) = '543210',             '#CopyAfter.1');
    CheckAssertion (CopyAfter ('1234543210', '4', False) = '543210',            '#CopyAfter.2');
    CheckAssertion (CopyAfter ('1234543210', '6', True) = '1234543210',         '#CopyAfter.3');
    CheckAssertion (CopyAfter ('1234543210', '6', False) = '',                  '#CopyAfter.4');
    CheckAssertion (CopyAfter ('1234543210', ['4', '5'], True) = '543210',      '#CopyAfter.5');
    CheckAssertion (CopyAfter ('1234543210', ['4', '5'], False) = '543210',     '#CopyAfter.6');
    CheckAssertion (CopyAfter ('1234543210', ['6', 'a'], True) = '1234543210',  '#CopyAfter.7');
    CheckAssertion (CopyAfter ('1234543210', ['6', 'a'], False) = '',           '#CopyAfter.8');
    CheckAssertion (CopyFrom ('1234543210', '4') = '4543210',                   '#CopyFrom.1');
    CheckAssertion (CopyFrom ('1234543210', '6') = '',                          '#CopyFrom.2');
    CheckAssertion (CopyFrom ('1234543210', '9', True) = '1234543210',          '#CopyFrom.3');
    CheckAssertion (CopyFrom ('1234543210', '4', True, 5) = '43210',            '#CopyFrom.4');
    CheckAssertion (CopyFrom ('1234543210', '6', True, 5) = '543210',           '#CopyFrom.5');
    CheckAssertion (CopyFrom ('1234543210', ['4', '5']) = '4543210',            '#CopyFrom.6');
    CheckAssertion (CopyTo ('1234543210', '4', True) = '1234',                  '#CopyTo.1');
    CheckAssertion (CopyTo ('1234543210', '4', False) = '1234',                 '#CopyTo.2');
    CheckAssertion (CopyTo ('1234543210', '6', True) = '1234543210',            '#CopyTo.3');
    CheckAssertion (CopyTo ('1234543210', '6', False) = '',                     '#CopyTo.4');
    CheckAssertion (CopyBetween ('1234543210', '3', '3', False, False) = '454', '#CopyBetween.1');
    CheckAssertion (CopyBetween ('1234543210', '3', '4', False, False) = '',    '#CopyBetween.2');
    CheckAssertion (CopyBetween ('1234543210', '4', '3', False, False) = '54',  '#CopyBetween.3');
    CheckAssertion (CopyBetween ('1234543210', '4', '6', False, False) = '',    '#CopyBetween.4');
    CheckAssertion (CopyBetween ('1234543210', '4', '6', False, True) = '543210',       '#CopyBetween.5');
    CheckAssertion (CopyBetween ('1234543210', '3', ['2', '3'], False,  False) = '454', '#CopyBetween.6');
    CheckAssertion (CopyBetween ('1234543210', '3', ['4', '5'], False, False) = '',     '#CopyBetween.7');
    CheckAssertion (CopyBetween ('1234543210', '4', ['2', '3'], False, False) = '54',   '#CopyBetween.8');
    CheckAssertion (CopyBetween ('1234543210', '4', ['6', '7'], False, False) = '',     '#CopyBetween.9');
    CheckAssertion (CopyBetween ('1234543210', '4', ['6'], False, True) = '543210',     '#CopyBetween.10');

    S1 := '1234567890';
    Paste (S1, 5, '1234567890', 9, 3);
    CheckAssertion (S1 = '1234907890',                                  '#2.1');
    CheckAssertion (TrimLeft ('   123   ') = '123   ',                  '#2.2');
    CheckAssertion (TrimRight ('   123   ') = '   123',                 '#2.3');
    CheckAssertion (Trim ('   123   ') = '123',                         '#2.4');
    CheckAssertion (IsNumber ('1234567890'),                            '#2.5');
    CheckAssertion (IsInteger ('-1234567890'),                          '#2.6');
    CheckAssertion (IsReal ('-1234.567890'),                            '#2.7');
    CheckAssertion (IsQuotedString ('"ABC""D"'),                        '#2.8');
    CheckAssertion (TrimQuotes ('"1""23"') = '1""23',                   '#2.9');
    CheckAssertion (UnQuoteText ('"1""23"') = '1"23',                   '#2.10');
    CheckAssertion (Dup ('xy', 3) = 'xyxyxy',                           '#2.11');
    Ch := 'x';
    CheckAssertion (Dup (Ch, 6) = 'xxxxxx',                             '#2.11');
    CheckAssertion (Reverse ('12345') = '54321',                        '#2.12');
    CheckAssertion (Reverse ('1234') = '4321',                          '#2.13');
    CheckAssertion (IsPalindrome ('amanaplanacanalpanama'),             '#2.14');
    CheckAssertion (not IsPalindrome ('123'),                           '#2.15');

    CheckAssertion (Match (['a'..'z'], 'abcd', 2, 2),                   '#Match.1');
    CheckAssertion (not Match (['a'..'z'], 'ab', 2, 2),                 '#Match.2');
    CheckAssertion (not Match (['a'..'z'], 'abcd', 0, 2),               '#Match.3');
    CheckAssertion (Match (['a'..'z'], 'abcd', 1, 1),                   '#Match.4');
    CheckAssertion (not Match ('xx', 'abcdef', 1),                      '#Match.5');
    CheckAssertion (Match ('x', 'xbcdef', 1),                           '#Match.6');
    CheckAssertion (Match ('xxxxx', 'abcdxxxxx', 5),                    '#Match.7');
    CheckAssertion (Match ('abcdef', 'abcdef', 1),                      '#Match.8');

    CheckAssertion (PosNext (['a'], '1234567890abc', 0) = 11,           '#Pos.1');
    CheckAssertion (PosNext (['1'], '1234567890abc', 0) = 1,            '#Pos.2');
    CheckAssertion (PosNext (['8'], '1234567890abc', 8) = 0,            '#Pos.3');
    CheckAssertion (PosNext (['9', 'c'], '1234567890abc', 8) = 9,       '#Pos.4');
    CheckAssertion (PosNext ('a', '1234567890abc', 0) = 11,             '#Pos.5');
    CheckAssertion (PosNext ('123', '1234567890abc', 0) = 1,            '#Pos.6');
    CheckAssertion (PosNext ('8', '1234567890abc', 8) = 0,              '#Pos.7');
    CheckAssertion (PosNext ('90abc', '1234567890abc', 8) = 9,          '#Pos.8');
    CheckAssertion (PosNext ('aa', 'aabaabaaa', 8) = 0,                 '#Pos.9');
    CheckAssertion (PosNext ('aa', 'aaaa', 1) = 2,                      '#Pos.10');
    CheckAssertion (PosPrev ('a', '1234567890abc', 0) = 11,             '#Pos.11');
    CheckAssertion (PosPrev ('123', '1234567890abc', 0) = 1,            '#Pos.12');
    CheckAssertion (PosPrev ('8', '1234567890abc', 8) = 0,              '#Pos.13');
    CheckAssertion (PosPrev ('234', '1234567890abc', 8) = 2,            '#Pos.14');
    CheckAssertion (PosPrev ('aa', 'aabaabaaa', 8) = 7,                 '#Pos.15');
    CheckAssertion (PosPrev ('aa', 'aaaa', 3) = 2,                      '#Pos.16');
    CheckAssertion (PosPrev ('aa', 'aaaa', 1) = 0,                      '#Pos.17');

    CheckAssertion (Replace ('a', 'b', 'bababa') = 'bbbbbb',            '#Replace.1');
    CheckAssertion (Replace ('aba', 'x', 'bababa') = 'bxba',            '#Replace.2');
    CheckAssertion (Replace ('b', 'bb', 'bababa') = 'bbabbabba',        '#Replace.3');
    CheckAssertion (Replace ('c', 'aa', 'bababa') = 'bababa',           '#Replace.4');
    CheckAssertion (Replace ('ba', '', 'bababa') = '',                  '#Replace.5');
    CheckAssertion (Replace ('aa', '12', 'aaaaa') = '1212a',            '#Replace.6');
    CheckAssertion (Replace ('aa', 'a', 'aaaaa') = 'aaa',               '#Replace.7');
    CheckAssertion (Replace (['b'], 'z', 'bababa') = 'zazaza',          '#Replace.8');
    CheckAssertion (Replace (['b', 'a'], 'z', 'bababa') = 'zzzzzz',     '#Replace.9');
    CheckAssertion (QuoteText ('Abe''s', '''') = '''Abe''''s''',        '#Replace.10');
    CheckAssertion (Remove (['a', 'z'], 'bazabazza') = 'bb',            '#Replace.11');
    CheckAssertion (RemoveDup (['a', 'z'], 'azaazzel') = 'azazel',      '#Replace.12');

    CheckAssertion (Count ('xyz', 'abcxyzdexxyxyz') = 2,                '#Count.1');
    CheckAssertion (Count ('xx', 'axxbxxxx') = 3,                       '#Count.2');
    CheckAssertion (Count ('xx', 'axxbxxx') = 2,                        '#Count.3');
    CheckAssertion (Count ('x', 'abcxyzdexxyxyz') = 4,                  '#Count.4');
    CheckAssertion (Count ('q', 'abcxyzdexxyxyz') = 0,                  '#Count.5');
    CheckAssertion (Count (['a'..'z'], 'abcxyzdexxyxyz') = 14,          '#Count.6');
    CheckAssertion (Count (['x'], 'abcxyzdexxyxyz') = 4,                '#Count.7');
    CheckAssertion (Count (['q'], 'abcxyzdexxyxyz') = 0,                '#Count.8');
    CheckAssertion (CountWords ('This, is a test.') = 4,                '#Count.9');

    CheckAssertion (PosN ('X', '123X567X9XX2345', 2, False) = 8,        '#PosN.1');
    CheckAssertion (PosN ('X', '123X567X9XX2345', 2, True) = 10,        '#PosN.2');
    CheckAssertion (PosN ('X', '123X567X9XX2345', 9, False) = 0,        '#PosN.3');
    CheckAssertion (PosN ('X', '123X567X9XX2345', 9, True) = 0,         '#PosN.4');
    CheckAssertion (PosN ('XX', 'XX3XXX7XXXX', 1, False, True) = 1,     '#PosN.5');
    CheckAssertion (PosN ('XX', 'XX3XXX7XXXX', 2, False, True) = 4,     '#PosN.6');
    CheckAssertion (PosN ('XX', 'XX3XXX7XXXX', 3, False, True) = 8,     '#PosN.7');
    CheckAssertion (PosN ('XX', 'XX3XXX7XXXX', 4, False, True) = 10,    '#PosN.9');
    CheckAssertion (PosN ('XX', 'XX3XXX7XXXX', 5, False, True) = 0,     '#PosN.10');
    CheckAssertion (PosN ('XX', 'XX3XXX7XXXX', 2, False, False) = 4,    '#PosN.11');
    CheckAssertion (PosN ('XX', 'XX3XXX7XXXX', 3, False, False) = 5,    '#PosN.12');
    CheckAssertion (PosN ('XX', 'XX3XXX7XXXX', 1, True, True) = 10,     '#PosN.13');
    CheckAssertion (PosN ('XX', 'XX3XXX7XXXX', 4, True, True) = 1,      '#PosN.14');
    CheckAssertion (PosN ('XX', 'XX3XXX7XXXX', 1, True, False) = 10,    '#PosN.15');
    CheckAssertion (PosN ('XX', 'XX3XXX7XXXX', 6, True, False) = 1,     '#PosN.16');

    CheckAssertion (Join (Split ('x yy zzz')) = 'x yy zzz',             '#8.1');
    CheckAssertion (Join (Split (' x  yy  zzz ')) = ' x  yy  zzz ',     '#8.2');
    CheckAssertion (PadLeft ('xxx', 'y', 6) = 'yyyxxx',                 '#8.3');
    CheckAssertion (PadRight ('xxx', 'y', 6) = 'xxxyyy',                '#8.4');
    CheckAssertion (Pad ('xxx', 'y', 7) = 'yyxxxyy',                    '#8.5');
    CheckAssertion (Pad (123, 8) = '00000123',                          '#8.6');
  End;



{                                                                              }
{ cStream                                                                      }
{                                                                              }
type
  EStreamSelfTestFail = class (Exception);

Procedure TestStream (const X : TObject);
var I : Integer;
    N : TDateTime;
    C : Byte;
    S : String;
    M : TStringStream;
    Y : TExStream;
  Begin
{    Y := X as TExStream;
    Y.Append;
    I := Y.Position;
    Y.Write (123);
    CheckAssertion (Y.Position - I = 1,                               '#1.1');
    CheckAssertion (Y.Size <> 0,                                      '#1.2');
    Y.Write ('1234');
    CheckAssertion (Y.Position - I = 5,                               '#1.3');
    Y.WritePacked (True);
    Y.WritePacked ('1234');
    Y.WritePacked (123.4);
    N := Now;
    Y.WritePackedDateTime (N);
    Y.Position := I;
    CheckAssertion (Y.Read = 123,                                     '#1.4');
    CheckAssertion (Y.Position - I = 1,                               '#1.5');
    CheckAssertion (Y.Read (4) = '1234',                              '#1.6');
    CheckAssertion (Y.Position - I = 5,                               '#1.7');
    CheckAssertion (Y.ReadPackedBoolean = True,                       '#1.8');
    CheckAssertion (Y.ReadPackedString = '1234',                      '#1.9');
    CheckAssertion (Y.ReadPackedExtended = 123.4,                     '#1.10');
    CheckAssertion (Y.ReadPackedDateTime = N,                         '#1.11');
    CheckAssertion (Y.EOF,                                            '#1.12'); }

    Y.Reset;
    CheckAssertion (Y.Position = 0,                                   '#2.1');
    CheckAssertion (not Y.EOF,                                        '#2.2');
    Y.Skip (2);
    CheckAssertion (Y.Position = 2,                                   '#2.3');
    C := Y.Peek;
    CheckAssertion (C = Y.Read,                                       '#2.4');
    S := Y.Peek (4);
    CheckAssertion (S = Y.Read (4),                                   '#2.5');
    CheckAssertion (Y.Remaining <> 0,                                 '#2.6');
    Y.Reset;
    Y.Truncate;
    CheckAssertion (Y.EOF,                                            '#2.7');
    CheckAssertion (Y.Position = 0,                                   '#2.8');
    CheckAssertion (Y.Size = 0,                                       '#2.9');
    CheckAssertion (Y.Remaining <= 0,                                 '#2.10');

{    Y.Write ('A man a plan a canal, Panama');
    CheckAssertion (Y.Size > 0,                                       '#3.1');
    CheckAssertion (Y.Position = Y.Size,                              '#3.2');
    CheckAssertion (Y.EOF,                                            '#3.3');
    Y.Position := 0;
    Y.SkipChars (['A', ' ']);
    CheckAssertion (Y.Position = 2,                                   '#3.4');
    CheckAssertion (Y.Match ('man'),                                  '#3.5');
    CheckAssertion (Y.Match ('m'),                                    '#3.6');
    CheckAssertion (Y.Match (['M','m']),                              '#3.7');
    CheckAssertion (Y.Extract = 'm',                                  '#3.8');
    CheckAssertion (Y.Extract ('an') = 'an',                          '#3.9');
    CheckAssertion (Y.Extract ([' ']) = ' ',                          '#3.10');
    CheckAssertion (Y.ExtractRest <> '',                              '#3.11');
    CheckAssertion (Y.EOF,                                            '#3.12');

    Y.Reset;
    CheckAssertion (not Y.Match (['#', '!', 'X']),                    '#4.2');
    CheckAssertion (Y.Position = 0,                                   '#4.3');
    CheckAssertion (not Y.Match ('A banana'),                         '#4.4');
    CheckAssertion (Y.Position = 0,                                   '#4.5');
    CheckAssertion (not Y.Match ('x'),                                '#4.6');
    CheckAssertion (Y.Position = 0,                                   '#4.7');
    CheckAssertion (not Y.MatchText ('a banana'),                     '#4.8');
    CheckAssertion (Y.Position = 0,                                   '#4.9'); }

    M := TStringStream.CreateEx ('');
    M.Assign (Y);
    CheckAssertion (M.Size = Y.Size,                                  '#5.1');
    M.Reset;
    Y.Reset;
    CheckAssertion (M.Read (M.Size) = Y.Read (Y.Size),                '#5.2');
    CheckAssertion (Y.EOF and M.EOF,                                  '#5.3');
    M.Free;

    Y.Free;
  End;

Procedure cStreamSelfTest;
var F : TFileStream;
  Begin
    DoObjectTest (TStringStream.CreateEx (''), TestStream);
    DoObjectTest (TStringStream.CreateEx ('0123456789'), TestStream);
    F := TFileStream.Create ('d:\temp\cStream.$$$', [fsmCreate]);
    DoObjectTest (F, TestStream);
    F := TFileStream.Create ('d:\temp\cStream.$$$', [fsmCreate]);
    F.Position := 0;
    F.Truncate;
    F.Write ('0123456789');
    DoObjectTest (F, TestStream);
  End;



{                                                                              }
{ cMaths                                                                       }
{                                                                              }
Procedure cMathsSelfTest;
var I, J, K : Integer;
    TD : T3DPoint;
    F, G : THugeInteger;
  Begin
    CheckAssertion (SwapBits ($F0181111) = $8888180F,       '#SwapBits');
    CheckAssertion (LSBit ($FF008000) = 15,                 '#LSBit');
    CheckAssertion (MSBit ($0008F000) = 19,                 '#MSBit');
    CheckAssertion (BitCount ($1234FF07) = 16,              '#BitCount');

    CheckAssertion (SwapEndian ($12345678) = $78563412,     '#SwapEndian');
    CheckAssertion (DecodeBase ('123', 10) = 123,           '#DecodeBase.1');
    CheckAssertion (DecodeBase ('123', 16) = 291,           '#DecodeBase.2');
    CheckAssertion (DecodeBase ('123', 8) = 83,             '#DecodeBase.3');
    CheckAssertion (DecodeBase ('1011', 2) = 11,            '#DecodeBase.4');
    CheckAssertion (DecodeBase ('ZZ', 36) = 1295,           '#DecodeBase.5');
    CheckAssertion (BinToInt ('10111') = 23,                '#DecodeBase.6');
    CheckAssertion (OctToInt ('71') = 57,                   '#DecodeBase.8');
    CheckAssertion (HexToInt ('1234FFFF') = $1234FFFF,      '#DecodeBase.9');

    CheckAssertion (EncodeBase ($1234, 16) = '1234',         '#EncodeBase.1');
    CheckAssertion (EncodeBase ($1234, 10) = '4660',         '#EncodeBase.2');
    CheckAssertion (EncodeBase ($1234, 2) = '1001000110100', '#EncodeBase.3');
    CheckAssertion (EncodeBase ($1234, 8) = '11064',         '#EncodeBase.4');
    CheckAssertion (EncodeBase ($1234, 36) = '3LG',          '#EncodeBase.5');
    CheckAssertion (IntToBin (23) = '10111',                 '#DecodeBase.6');
    CheckAssertion (IntToOct (57) = '71',                    '#DecodeBase.7');
    CheckAssertion (IntToHex ($1234FFFF) = '1234FFFF',       '#DecodeBase.8');

    Randomize;
    For K := 0 to 1024 do
      begin
        if K <= 513 then
          I := K else
          I := Random (MaxInt);
        CheckAssertion (BinToInt (IntToBin (I)) = I,                '#Conv.2.' + IntToStr (I) + '.' + IntToStr (J));
        CheckAssertion (OctToInt (IntToOct (I)) = I,                '#Conv.1.' + IntToStr (I) + '.' + IntToStr (J));
        CheckAssertion (HexToInt (IntToHex (I)) = I,                '#Conv.3.' + IntToStr (I) + '.' + IntToStr (J));
        J := Random (35) + 2;
        CheckAssertion (DecodeBase (EncodeBase (I, J), J) = I,      '#Conv.4.' + IntToStr (I) + '.' + IntToStr (J));
      end;

    CheckAssertion (IsPrime (0) = False,                     '#Prime.1');
    CheckAssertion (IsPrime (1) = False,                     '#Prime.2');
    CheckAssertion (IsPrime (2) = True,                      '#Prime.3');
    CheckAssertion (IsPrime (3) = True,                      '#Prime.4');
    CheckAssertion (IsPrime (4) = False,                     '#Prime.5');
    CheckAssertion (IsPrime (71) = True,                     '#Prime.6');
    CheckAssertion (IsPrime (71*47) = False,                 '#Prime.7');
    CheckAssertion (IsPrimeFactor (71, 71*47) = True,        '#Prime.8');
    CheckAssertion (IsPrimeFactor (3, 71*47) = False,        '#Prime.9');
    CheckAssertion (GCD (71, 41) = 1,                        '#Prime.10');
    CheckAssertion (GCD (71*47, 71*41) = 71,                 '#Prime.11');
    CheckAssertion (GCD (71*3, 41*3) = 3,                    '#Prime.12');

    TD := T3DPoint.CreatePoint (1, 1, 1);
    TD.Scale (1,1,1);

    F := THugeInteger.Create;
    G := THugeInteger.Create;
    F.AsInteger := -1;
    CheckAssertion (F.Negative,                 '#HugeInt.Negative');
    G.AsInteger := 2;
    F.Add (G);
    CheckAssertion (F.AsInteger = 1,            '#HugeInt.1');
    F.Add (G);
    CheckAssertion (F.AsInteger = 3,            '#HugeInt.2');
    F.Multiply (MaxLongInt);
    FreeAndNil (G);
    FreeAndNil (F);
  End;


(*
{                                                                              }
{                                                                              }
{                                                                              }
Procedure cInternetStandardsSelfTest;
var V       : TDateTime;
    I, J    : Integer;
    S, T, U : String;
  Begin
    CheckAssertion (RFCTimeZoneToGMTBias ('+2', False) = 120,     '#GMTBias.1');
    CheckAssertion (RFCTimeZoneToGMTBias ('UT', False) = 0,       '#GMTBias.2');
    CheckAssertion (RFCTimeZoneToGMTBias ('GMT', False) = 0,      '#GMTBias.3');
    CheckAssertion (RFCTimeZoneToGMTBias ('EST', False) = -300,   '#GMTBias.4');
    CheckAssertion (RFCTimeZoneToGMTBias ('Z', False) = 0,        '#GMTBias.5');
    CheckAssertion (RFCTimeZoneToGMTBias ('A', False) = -60,      '#GMTBias.6');
    CheckAssertion (RFCTimeZoneToGMTBias ('Y', False) = 720,      '#GMTBias.7');
    CheckAssertion (RFCTimeZoneToGMTBias ('+0230', False) = 150,  '#GMTBias.8');
    CheckAssertion (RFCTimeZoneToGMTBias ('-2', False) = -120,    '#GMTBias.9');

    CheckAssertion (GMTDateTimeToRFC1123DateTime (EncodeDate (1999, 12, 31) + EncodeTime (23, 59, 59, 0), True) =
                    'Fri, 31 Dec 1999 23:59:59 GMT',                            '#RFC1123DateTime.1');
    CheckAssertion (GMTDateTimeToRFC1123DateTime (EncodeDate (2000, 1, 1), True) =
                    'Sat, 01 Jan 2000 00:00:00 GMT',                            '#RFC1123DateTime.2');

    CheckAssertion (IsEqual (RFCDateTimeToGMTDateTime ('Fri, 31 Dec 1999 23:59:59 EST'),
                    EncodeDate (1999, 12, 31) + EncodeTime (18, 59, 59, 0)),    '#RFCDateTime.1');
    CheckAssertion (IsEqual (RFCDateTimeToGMTDateTime ('Fri, 31 Dec 1999 23:59:59 +0200'),
                    EncodeDate (2000, 1, 1) + EncodeTime (1, 59, 59, 0)),       '#RFCDateTime.2');
    CheckAssertion (IsEqual (RFCDateTimeToGMTDateTime ('Fri, 31 Dec 1999 23:59:59 GMT'),
                    EncodeDate (1999, 12, 31) + EncodeTime (23, 59, 59, 0)),    '#RFCDateTime.3');
    CheckAssertion (IsEqual (RFCDateTimeToGMTDateTime ('31 Dec 1999 23:59:59 GMT'),
                    EncodeDate (1999, 12, 31) + EncodeTime (23, 59, 59, 0)),    '#RFCDateTime.4');
    CheckAssertion (IsEqual (RFCDateTimeToGMTDateTime ('31 Dec 1999'),
                    EncodeDate (1999, 12, 31)),                                 '#RFCDateTime.5');

    Randomize;
    For I := 1 to 1000 do
      begin
        try
          V := EncodeDate (1900 + Random (200), Random (12) + 1, Random (31) + 1) +
               EncodeTime (Random (24), Random (60), Random (60), 0);
        except
          continue;
        end;
        CheckAssertion (RFCDateTimeToGMTDateTime (GMTDateTimeToRFC1123DateTime (V, I mod 2 = 0)) = V, '#RFCDateTime.Conv.' + IntToStr (I));
      end;

//    CheckAssertion (httpDecodeURI ('http://%20%41.com') = 'http:// A.com',      '#httpDecodeURI');

{    httpDecodeRequestLine ('GET /money%20.txt HTTP/1.20', I, J, S, T);
    CheckAssertion (I = 1,             '#httpDecodeRequestLine.1');
    CheckAssertion (J = 20,            '#httpDecodeRequestLine.2');
    CheckAssertion (S = 'GET',         '#httpDecodeRequestLine.3');
    CheckAssertion (T = '/money .txt', '#httpDecodeRequestLine.4');

    httpDecodeURL ('http://www.e.co.za/~david/index.html', S, T, U);
    CheckAssertion (S = 'http',               '#httpDecodeURL.1');
    CheckAssertion (T = 'www.e.co.za',        '#httpDecodeURL.2');
    CheckAssertion (U = '/~david/index.html', '#httpDecodeURL.3');  }

{    CheckAssertion (htmlSafeText ('<">&') = '&lt;&quot;&gt;&amp;',   '#html.1');
    CheckAssertion (htmlBold ('abc') = '<B>abc</B>',                 '#html.2');
    CheckAssertion (htmlColour ('RED') = clRed,                      '#html.3'); }
  End; *)

Procedure DoSelfTest;
  Begin
    TestReports := TStringList.Create;
    try
      DoUnitTest ('cUtil', cUtilSelfTest);
      DoUnitTest ('cDateTime', cDateTimeSelfTest);
      DoUnitTest ('cStrings', cStringsSelfTest);
      DoUnitTest ('cMaths', cMathsSelfTest);
      SetUnitStats;
      if TestReports.Count > 0 then
        MessageDlg ('ERROR SUMMARY'#13#13 + TestReports.GetText, mtInformation, [mbOk], -1);
      MessageDlg ('Test statistics'#13#13 + TestStats, mtInformation, [mbOk], -1);
    finally
      FreeAndNil (TestReports);
    end;
  End;

end.

