{$INCLUDE cHeader.inc}
unit cMaths;

interface

uses
  Math,
  SysUtils,
  Classes,

  // Delphi Fundamentals (L0)
  cUtils,
  cStreams,
  cTypes,
  cDataStructs;



{                                                                              }
{ Maths unit v0.40 (L0)                                                        }
{                                                                              }
{   A collection of mathematical functions.                                    }
{                                                                              }
{                                                                              }
{ This unit is copyright  1995-2000 by David Butler (david@e.co.za)           }
{                                                                              }
{ This unit is part of Delphi Fundamentals, it's original name is cMaths.      }
{                                                                              }
{ I invite you to use this unit, free of charge.                               }
{ I invite you to distibute this unit, but it must be for free.                }
{ I also invite you to contribute to its development, but do not distribute    }
{ a modified copy of this file, send modifications, suggestions and bug        }
{ reports to david@e.co.za                                                     }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   1995-96     v0.01  Wrote statistical and actuarial functions.              }
{   1998/03     v0.02  Added Solver and SecantSolver.                          }
{   1998/10     v0.03  Removed functions now found in Delphi 3's Math unit     }
{                      Uses Delphi's math exceptions (eg EOverflow,            }
{                      EInvalidArgument, etc)                                  }
{   1999/08/29  v0.04  Added ASet, TRangeSet, TFlatSet, TSparseFlatSet.        }
{   1999/09/27  v0.05  Added TMatrix, TVector.                                 }
{                      Added BinomialCoeff.                                    }
{   1999/10/02  v0.06  Added TComplex.                                         }
{   1999/10/03  v0.07  Added DerivateSolvers.                                  }
{                      Completed TMatrix.                                      }
{   1999/10/04  v0.08  Added TLifeTable.                                       }
{   1999/10/05  v0.09  T3DPoint                                                }
{   1999/10/06  v0.10  Transform matrices.                                     }
{   1999/10/13  v0.11  TRay, TSphere                                           }
{   1999/10/14  v0.12  TPlane                                                  }
{   1999/10/26  v0.13  Upgraded to Delphi 4. Compared the assembly code of the }
{                      new dynamic arrays with that of pointers to arrays.     }
{                      Its basically the same. Converted all PRealList type    }
{                      references to RealArray type.                           }
{   1999/10/30  v0.14  Added TVector.StdDev                                    }
{                      Changed some functions to the same name (since Delphi   }
{                      now supports overloading).                              }
{                      Removed Min and Max functions (now in Math).            }
{   1999/11/04  v0.15  Added TVector.Pos, TVector.Append                       }
{                      855 lines interface, 3071 lines implementation.         }
{   1999/11/07  v0.16  Added RandomSeed function.                              }
{                      Added assembly bit functions.                           }
{   1999/11/10  v0.17  Added hashing functions. Coded XOR8 in assembly.        }
{                      Added MD5 hashing.                                      }
{   1999/11/11  v0.18  Added EncodeBase.                                       }
{   1999/11/21  v0.19  Added TComplex.Power                                    }
{   1999/11/25  v0.20  Moved TRay, TSphere and TPlane to cRayTrace.            }
{                      Added Primes.                                           }
{   1999/11/26  v0.21  Added Rational numbers (can convert to/from TReal).     }
{                      Added GCD.                                              }
{                      Added RealArray/IntegerArray functions.                 }
{   1999/11/27  v0.22  Replaced GCD algorithm with Euclid's algorithm.         }
{                      Added SI constants.                                     }
{                      Added TMatrix.Normalise, TMatrix.Multiply (Row, Value)  }
{   1999/12/01  v0.23  Added RandomUniform.                                    }
{   1999/12/03  v0.24  Added RandomNormal.                                     }
{   1999/12/16  v0.25  Bug fixes.                                              }
{   1999/12/23  v0.26  Fixed bug in TRational.CalcFrac.                        }
{   1999/12/26  v0.27  Added Reverse procedures for RealArray/IntegerArray.    }
{   2000/01/22  v0.28  Added TStatistic.                                       }
{   2000/01/23  v0.29  Added RandomPseudoWord.                                 }
{   2000/03/08  v0.30  Moved TInteger/TReal, IntegerArray/RealArray to cUtil.  }
{                      Moved AArray to cDataStructs.                           }
{   2000/04/01  v0.31  Moved ASet and set implmentations to cDataStructs.      }
{   2000/04/09  v0.32  Added SetFPUPrecision.                                  }
{   2000/05/03  v0.33  Added Bit functions (ToggleBit, SetBit, ClearBit,       }
{                      IsBitSet).                                              }
{   2000/05/08  V0.34  Moved SetFPUPrecision to cUtil.                         }
{   2000/05/25  v0.35  Started THugeInteger.                                   }
{   2000/06/06  v0.36  Moved bit functions to cUtil.                           }
{   2000/06/08  v0.37  TVector now inherits from TExtendedArray.               }
{                      Added TIntegerVector.                                   }
{                      Removed TInteger/TReal, TIntegerArray/TRealArray.       }
{   2000/06/16  v0.38  Updated documentation for financial functions.          }
{   2000/06/17  v0.39  Recalculated all constants to 34 digits.                }
{   2000/06/25  v0.40  Fixed bug in CalcCheckSum32 reported by Ondrej Hrabal   }
{                      (twilight.ia@worldonline.cz)                            }
{                                                                              }



{                                                                              }
{ Mathematical constants                                                       }
{                                                                              }
const
  Pi        = 3.14159265358979323846      +        { Pi (200 digits)           }
              0.26433832795028841971e-20  +        {                           }
              0.69399375105820974944e-40  +        {                           }
              0.59230781640628620899e-60  +        {                           }
              0.86280348253421170679e-80  +        {                           }
              0.82148086513282306647e-100 +        {                           }
              0.09384460955058223172e-120 +        {                           }
              0.53594081284811174502e-140 +        {                           }
              0.84102701938521105559e-160 +        {                           }
              0.64462294895493038196e-180;         {                           }
  Pi2       = 6.283185307179586476925286766559006; { Pi * 2                    }
  Pi3       = 9.424777960769379715387930149838509; { Pi * 3                    }
  Pi4       = 12.56637061435917295385057353311801; { Pi * 4                    }
  PiOn2     = 1.570796326794896619231321691639751; { Pi / 2                    }
  PiOn3     = 1.047197551196597746154214461093168; { Pi / 3                    }
  PiOn4     = 0.785398163397448309615660845819876; { Pi / 4                    }
  PiSq      = 9.869604401089358618834490999876151; { Pi^2                      }
  PiE       = 22.45915771836104547342715220454374; { Pi^e                      }
  LnPi      = 1.144729885849400174143427351353059; { Ln (Pi)                   }
  LogPi     = 0.497149872694133854351268288290899; { Log (Pi)                  }
  SqrtPi    = 1.772453850905516027298167483341145; { Sqrt (Pi)                 }
  Sqrt2Pi   = 2.506628274631000502415765284811045; { Sqrt (2 * Pi)             }
  LnSqrt2Pi = 0.918938533204672741780329736405618; { Ln (Sqrt (2 * Pi))        }
  RadPerDeg = 0.017453292519943295769236907684886; { Pi / 180                  }
  DegPerRad = 57.29577951308232087679815481410517; { 180 / Pi                  }
  E         = 2.718281828459045235360287471352663; { e                         }
  E2        = 7.389056098930650227230427460575008; { e^2                       }
  ExpM2     = 0.135335283236612691893999494972484; { e^-2                      }
  Ln2       = 0.693147180559945309417232121458177; { Ln (2)                    }
  Ln10      = 2.302585092994045684017991454684364; { Ln (10)                   }
  LogE      = 0.434294481903251827651128918916605; { Log (e)                   }
  Log2      = 0.301029995663981195213738894724493; { Log (2)                   }
  Log3      = 0.477121254719662437295027903255115; { Log (3)                   }
  Sqrt2     = 1.414213562373095048801688724209698; { Sqrt (2)                  }
  Sqrt3     = 1.732050807568877293527446341505872; { Sqrt (3)                  }
  Sqrt5     = 2.236067977499789696409173668731276; { Sqrt (5)                  }
  Sqrt7     = 2.645751311064590590501615753639260; { Sqrt (7)                  }



{                                                                              }
{ Dynamic array functions                                                      }
{                                                                              }
Function  Sum (const R : IntegerArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64; overload;
Function  Sum (const R : Int64Array; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64; overload;
Function  Sum (const R : SingleArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;
Function  Sum (const R : DoubleArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;
Function  Sum (const R : ExtendedArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;

Function  WeightedSum (const R : ExtendedArray; const W : ExtendedArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;
Function  WeightedSum (const R : IntegerArray; const W : ExtendedArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;
Function  WeightedSum (const R : IntegerArray; const W : IntegerArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64; overload;
Function  WeightedSum (const R : Int64Array; const W : ExtendedArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;
Function  WeightedSum (const R : Int64Array; const W : Int64Array; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64; overload;

Function  Product (const R : IntegerArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64; overload;
Function  Product (const R : Int64Array; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64; overload;
Function  Product (const R : SingleArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;
Function  Product (const R : DoubleArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;
Function  Product (const R : ExtendedArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;



{                                                                              }
{ Miscellaneous functions                                                      }
{   Sgn returns sign of argument, +1 or -1. Returns +1 for 0.                  }
{                                                                              }
Function Sgn (const R : Integer) : Integer; overload;
Function Sgn (const R : Int64) : Integer; overload;
Function Sgn (const R : Single) : Integer; overload;
Function Sgn (const R : Double) : Integer; overload;
Function Sgn (const R : Extended) : Integer; overload;



{                                                                              }
{ Unit Conversion                                                              }
{                                                                              }
const
  // SI prefixes
  Centi = 1e-2;
  Milli = 1e-3;
  Micro = 1e-6;
  Nano  = 1e-9;
  Pico  = 1e-12;

  Kilo  = 1e3;
  Mega  = 1e6;
  Giga  = 1e9;
  Terra = 1e12;

  // Distance
  Meter_per_Inch    = 0.0254;
  Meter_per_Foot    = 0.3048;
  Meter_per_Yard    = 0.9144;
  Meter_per_Rod     = 5.029;    // Also refered to as a Pole or a Perch
  Meter_per_Furlong = 201.168;
  Meter_per_Mile    = 1609.4;
  Meter_per_League  = 4830;

Function KelvinToFahrenheit (const T : Extended) : Extended;
Function FahrenheitToKelvin (const T : Extended) : Extended;
Function CelsiusToKelvin (const T : Extended) : Extended;
Function KelvinToCelsius (const T : Extended) : Extended;
Function CelsiusToFahrenheit (const T : Extended) : Extended;
Function FahrenheitToCelsius (const T : Extended) : Extended;



{                                                                              }
{ Trigonometric functions                                                      }
{   Delphi's Math unit includes most commonly used trigonometric functions.    }
{                                                                              }
Function  ATan360 (const X, Y : Extended) : Extended;
Function  InverseTangentDeg (const X, Y : Extended) : Extended;
Function  InverseTangentRad (const X, Y : Extended) : Extended;
Function  InverseSinDeg (const Y, R : Extended) : Extended;
Function  InverseSinRad (const Y, R : Extended) : Extended;
Function  InverseCosDeg (const X, R : Extended) : Extended;
Function  InverseCosRad (const X, R : Extended) : Extended;
Function  DMSToReal (const Degs, Mins, Secs : Extended) : Extended;
Procedure RealToDMS (const X : Extended; var Degs, Mins, Secs : Extended);
Function  Distance (const X1, Y1, X2, Y2 : Extended) : Extended;
Procedure PolarToRectangular (const R, Theta : Extended; var X, Y : Extended);
Procedure RectangularToPolar (const X, Y : Extended; var R, Theta : Extended);



{                                                                              }
{ Primes                                                                       }
{   IsPrime returns True if N is a prime number.                               }
{   IsPrimeFactor returns True if F is a prime factor of N.                    }
{   PrimeFactors returns an array of prime factors of N, sorted in acending    }
{   order.                                                                     }
{   GCD returns the Greatest Common Divisor of N1 and N2.                      }
{                                                                              }
Function  IsPrime (const N : Int64) : Boolean;
Function  IsPrimeFactor (const F, N : Int64) : Boolean;
Function  PrimeFactors (const N : Int64) : Int64Array;
Function  GCD (const N1, N2 : Int64) : Int64;
Function  Fibonacci (const N : Integer) : Int64;



{                                                                              }
{ Arbitrary size Integers                                                      }
{                                                                              }
type
  THugeInteger = class
    protected
    FValue    : CardinalArray;
    FNegative : Boolean;

    Procedure SetAsInteger (const X : Integer);
    Function  GetAsInteger : Integer;

    public
    Constructor Create (const X : Integer = 0);

    Procedure AssignZero;
    Property  AsInteger : Integer read GetAsInteger write SetAsInteger;
    Property  Negative : Boolean read FNegative write FNegative;
    Procedure Negate;
    Procedure Add (const X : THugeInteger); overload;
    Procedure Add (const X : Integer); overload;
    Procedure Subtract (const X : THugeInteger);
    Procedure Multiply (const X : Integer);
  end;



{                                                                              }
{ Rational numbers                                                             }
{   Class that represents a rational number (Numerator / Denominator pair)     }
{                                                                              }
type
  TRational = class
    private
    FT, FN : Int64;

    protected
    Procedure Simplify;

    public
    Constructor Create; overload;
    Constructor Create (const Numerator : Int64; const Denominator : Int64 = 1); overload;
    Constructor Create (const R : Extended); overload;

    Property Numerator : Int64 read FT;
    Property Denominator : Int64 read FN;

    Function  GetAsString : String;
    Procedure SetAsString (const S : String);
    Property  AsString : String read GetAsString write SetAsString;

    Function  GetAsReal : Extended;
    Procedure SetAsReal (const R : Extended);
    Property  AsReal : Extended read GetAsReal write SetAsReal;

    Procedure Assign (const R : TRational); overload;
    Procedure Assign (const R : Extended); overload;
    Procedure Assign (const Numerator : Int64; const Denominator : Int64 = 1); overload;
    Procedure AssignZero;
    Procedure AssignOne;

    Function  Duplicate : TRational;

    Function  IsEqual (const R : TRational) : Boolean; reintroduce; overload;
    Function  IsEqual (const Numerator : Int64; const Denominator : Int64 = 1) : Boolean; reintroduce; overload;
    Function  IsEqual (const R : Extended) : Boolean; reintroduce; overload;
    Function  IsZero : Boolean;
    Function  IsOne : Boolean;

    Procedure Add (const R : TRational); overload;
    Procedure Add (const V : Extended); overload;
    Procedure Add (const V : Int64); overload;

    Procedure Subtract (const R : TRational); overload;
    Procedure Subtract (const V : Extended); overload;
    Procedure Subtract (const V : Int64); overload;

    Procedure Negate;
    Procedure Abs;
    Function  Sgn : Integer;

    Procedure Multiply (const R : TRational); overload;
    Procedure Multiply (Const V : Extended); overload;
    Procedure Multiply (const V : Int64); overload;

    Procedure Reciprocal;
    Procedure Divide (const R : TRational); overload;
    Procedure Divide (const V : Extended); overload;
    Procedure Divide (const V : Int64); overload;

    Procedure Sqrt;
    Procedure Sqr;
    Procedure Power (const R : TRational); overload;
    Procedure Power (const V : Int64); overload;
    Procedure Power (const V : Extended); overload;
  end;



{                                                                              }
{ Complex numbers                                                              }
{   Class that represents a complex number (Real + i * Imag)                   }
{                                                                              }
type
  EComplex = class (Exception);
  TComplex = class
    private
    FReal,
    FImag  : Extended;

    Function  GetAsString : String;
    Procedure SetAsString (const S : String);

    public
    Constructor Create (const TheRealPart : Extended = 0.0; const TheImaginaryPart : Extended = 0.0);

    Property  RealPart : Extended read FReal write FReal;
    Property  ImaginaryPart : Extended read FImag write FImag;

    Property  AsString : String read GetAsString write SetAsString;

    Procedure Assign (const C : TComplex); overload;
    Procedure Assign (const V : Extended); overload;
    Procedure AssignZero;
    Procedure AssignI;
    Procedure AssignMinI;

    Function  Duplicate : TComplex;

    Function  IsEqual (const C : TComplex) : Boolean; overload;
    Function  IsEqual (const R, I : Extended) : Boolean; overload;
    Function  IsZero : Boolean;
    Function  IsI : Boolean;

    Procedure Add (const C : TComplex); overload;
    Procedure Add (const V : Extended); overload;
    Procedure Subtract (const C : TComplex); overload;
    Procedure Subtract (const V : Extended); overload;
    Procedure Multiply (const C : TComplex); overload;
    Procedure Multiply (Const V : Extended); overload;
    Procedure MultiplyI;
    Procedure MultiplyMinI;
    Procedure Divide (const C : TComplex); overload;
    Procedure Divide (const V : Extended); overload;
    Procedure Negate;

    Function  Modulo : Extended;
    Function  Denom : Extended;
    Procedure Conjugate;
    Procedure Inverse;

    Procedure Sqrt;
    Procedure Exp;
    Procedure Ln;
    Procedure Sin;
    Procedure Cos;
    Procedure Tan;
    Procedure Power (const C : TComplex);
  end;



{                                                                              }
{ Vector                                                                       }
{   General purpose vector class with statistical and mathematical functions.  }
{   Todo: Support complex numbers                                              }
{                                                                              }
type
  EVector = class (Exception);
  TFloatVector = class (TExtendedArray)
    { AType implementations                                                    }
    class Function CreateInstance : AType; override;

    { Mathematical functions                                                   }
    Function  IsZero : Boolean;
    Procedure Add (const V : TFloatVector); overload;
    Procedure Add (const V : TFloatVector; const Factor : Extended); overload;
    Procedure Add (const Value : Extended); overload;
    Procedure Multiply (const V : TFloatVector); overload;
    Procedure Multiply (const Value : Extended); overload;
    Function  DotProduct (const V : TFloatVector) : Extended;
    Function  Norm : Extended;
    Procedure Invert;
    Procedure SquareValues;
    Function  Angle (const V : TFloatVector) : Extended;                                // UV=|U||V|Cos

    { Statistical functions                                                    }
    Function  Sum : Extended; overload;
    Function  Sum (const LoIdx, HiIdx : Integer) : Extended; overload;
    Function  WeightedSum (const Weights : TFloatVector) : Extended;
    Function  Product : Extended; overload;
    Function  Product (const LoIdx, HiIdx : Integer) : Extended; overload;
    Function  MaxValue : Extended;
    Function  MinValue : Extended;
    Function  Mean : Extended;
    Function  HarmonicMean : Extended;
    Function  GeometricMean : Extended;
    Procedure Normalize;                                                        // Divide each element with Sum
    Function  Median : Extended;
    Function  Variance : Extended;
    Function  StdDev (var Mean : Extended) : Extended;                          // Sample StdDev
    Function  SumOfSquares : Extended;
    Procedure SumAndSquares (var Sum, SumOfSquares : Extended);
    Function  TotalVariance : Extended;
    Function  PopulationVariance : Extended;
    Function  PopulationStdDev : Extended;                                      // Population StdDev
  end;

  TIntegerVector = class (TInt64Array)
    { AType implementations                                                    }
    class Function CreateInstance : AType; override;

    { Mathematical functions                                                   }
    Function  IsZero : Boolean;
    Procedure Add (const V : TIntegerVector); overload;
    Procedure Add (const V : TIntegerVector; const Factor : Int64); overload;
    Procedure Add (const Value : Int64); overload;
    Procedure Multiply (const V : TIntegerVector); overload;
    Procedure Multiply (const Value : Int64); overload;
    Function  DotProduct (const V : TIntegerVector) : Int64;
    Function  Norm : Extended;
    Procedure SquareValues;
    Function  Angle (const V : TIntegerVector) : Extended;                      // UV=|U||V|Cos

    { Statistical functions                                                    }
    Function  Sum : Int64; overload;
    Function  Sum (const LoIdx, HiIdx : Integer) : Int64; overload;
    Function  WeightedSum (const Weights : TFloatVector) : Extended;
    Function  Product : Int64; overload;
    Function  Product (const LoIdx, HiIdx : Integer) : Int64; overload;
    Function  MaxValue : Int64;
    Function  MinValue : Int64;
    Function  Mean : Extended;
    Function  HarmonicMean : Extended;
    Function  GeometricMean : Extended;
    Function  Median : Extended;
    Function  Variance : Extended;
    Function  StdDev (var Mean : Extended) : Extended;                          // Sample StdDev
    Function  SumOfSquares : Int64;
    Procedure SumAndSquares (var Sum, SumOfSquares : Int64);
    Function  TotalVariance : Extended;
    Function  PopulationVariance : Extended;
    Function  PopulationStdDev : Extended;                                      // Population StdDev
  end;



{                                                                              }
{ Matrix                                                                       }
{                                                                              }
type
  EMatrix = class (Exception);
  TMatrix = class
    private
    FColCount : Integer;
    FRows     : Array of ExtendedArray;

    Function  GetRow (const Row : Integer) : TFloatVector;
    // Returns a *reference* to a row in the matrix as a TFloatVector. Caller must free.

    protected
    Function  GetAsString : String; virtual;

    public
    Constructor CreateSquare (const N : Integer);
    Constructor CreateIdentity (const N : Integer);
    Constructor CreateDiagonal (const D : TFloatVector);                             // D in diagonal

    Procedure SetSize (const Rows, Cols : Integer);

    Function  GetRowCount : Integer;
    Procedure SetRowCount (const NewRowCount : Integer);
    Procedure SetColCount (const NewColCount : Integer);
    Property  ColCount : Integer read FColCount write SetColCount;
    Property  RowCount : Integer read GetRowCount write SetRowCount;

    Procedure SetItem (const Row, Col : Integer; const Value : Extended);
    Function  GetItem (const Row, Col : Integer) : Extended;
    Property  Item [const Row, Col : Integer] : Extended read GetItem write SetItem; default;

    Property  AsString : String read GetAsString;

    Procedure Assign (const M : TMatrix); overload;
    Procedure Assign (const V : TFloatVector); overload;
    Procedure Assign (const Value : Extended); overload;
    Procedure AssignRow (const Row : Integer; const V : TFloatVector);
    Procedure AssignCol (const Col : Integer; const V : TFloatVector);
    Procedure AssignRowValues (const Row : Integer; const Values : Array of Extended);

    Function  Duplicate : TMatrix; overload;
    Function  Duplicate (const R1, C1, R2, C2 : Integer) : TMatrix; overload;
    Function  DuplicateRow (const Row : Integer) : TFloatVector;
    Function  DuplicateCol (const Col : Integer) : TFloatVector;
    Function  DuplicateDiagonal : TFloatVector;

    { Mathematical functions                                                   }
    Function  IsEqual (const M : TMatrix) : Boolean; overload;
    Function  IsEqual (const V : TFloatVector) : Boolean; overload;
    Function  IsIdentity : Boolean;
    Function  IsZero : Boolean;

    Function  Transposed : TMatrix;
    Procedure Transpose;                                                        // swap rows/cols
    Procedure Add (const M : TMatrix);
    Procedure AddRows (const I, J : Integer; const Factor : Extended);          // inc (row i, row j * Factor)
    Procedure SwapRows (const I, J : Integer);
    Procedure Multiply (const Value : Extended); overload;                      // multiply matrix with constant value
    Procedure Multiply (const Row : Integer; const Value : Extended); overload; // multiply row with constant value
    Procedure Multiply (const M : TMatrix); overload;
    Function  Multiplied (const M : TMatrix) : TMatrix;

    Function  Normalise (const M : TMatrix = nil) : Extended;                   // make diagonal 1's by multiplying each row with a factor. also applies operations to M (if specified)

    Function  Trace : Extended;                                                 // sum of diagonal
    Function  IsSquare : Boolean;
    Function  IsOrtogonal : Boolean;                                            // X'X = I
    Function  IsIdempotent : Boolean;                                           // XX = X
    Function  SolveMatrix (var M : TMatrix) : Extended;                         // Returns determinant
    Function  Determinant : Extended;
    Procedure Inverse;
    Function  SolveLinearSystem (const V : TFloatVector) : TFloatVector;
  end;



{                                                                              }
{ TStatistic                                                                   }
{   Computes statistics on a sample without storing the sample values.         }
{                                                                              }
type
  EStatistic = class (Exception);
  TStatistic = class
    protected
    FCount        : Integer;
    FLast,
    FMin,
    FMax,
    FSum,
    FSumOfSquares : Extended;

    Function GetMin : Extended;
    Function GetMax : Extended;

    public
    Procedure Add (const V : Extended); overload;
    Procedure Add (const V : ExtendedArray); overload;
    Procedure Add (const V : Array of Extended); overload;
    Procedure Add (const V : TFloatVector); overload;

    Procedure Clear;

    Function  GetAsString : String;

    Property  Count              : Integer read FCount;
    Property  Sum                : Extended read FSum;
    Property  SumOfSquares       : Extended read FSumOfSquares;
    Property  Min                : Extended read GetMin;
    Property  Max                : Extended read GetMax;
    Property  Last               : Extended read FLast;
    Function  Range              : Extended;
    Function  Mean               : Extended;
    Function  TotalVariance      : Extended;
    Function  Variance           : Extended;
    Function  StdDev             : Extended;
    Function  PopulationVariance : Extended;
    Function  PopulationStdDev   : Extended;
  end;



{                                                                              }
{ T3DPoint                                                                     }
{  Stores a (x,y,z)-value which can represent a point or a vector in 3D        }
{    space.                                                                    }
{  Internally it inherits from TFloatVector so all the vector operations are   }
{    available.                                                                }
{  A point is represented as [x,y,z,1] and a vector as [x,y,z,0]. The 4th      }
{    element is needed when multiplying with transformation matrices (see      }
{    "3D Transformation matrices") to preserve scale.                          }
{                                                                              }
type
  E3DPoint = class (Exception);
  T3DPoint = class (TFloatVector)
    private
    Function GetX : Extended;
    Function GetY : Extended;
    Function GetZ : Extended;

    Procedure SetX (const NewX : Extended);
    Procedure SetY (const NewY : Extended);
    Procedure SetZ (const NewZ : Extended);

    public
    Constructor CreatePoint (const X, Y, Z : Extended);
    Constructor CreateVector (const X, Y, Z : Extended);

    Function  Duplicate : T3DPoint;

    { Transformations                                                          }
    Procedure RotateX (const Angle : Extended);
    Procedure RotateY (const Angle : Extended);
    Procedure RotateZ (const Angle : Extended);
    Procedure RotateXYZ (const XAngle, YAngle, ZAngle : Extended);
    Procedure RotateVector (const NX, NY, NZ, Angle : Extended);

    Procedure Scale (const XScale, YScale, ZScale : Single);
    Procedure Origin (const XOrigin, YOrigin, ZOrigin : Extended);
    Procedure CrossProduct (const P : T3DPoint);
    Procedure Homogenize;

    { Parallel projections                                                     }
    { Angle typically 30 or 45                                                 }
    Procedure CavalierProject (const Angle : Extended; var X, Y : Extended);    // (x,y)=(x+z*cos(Angle),y+z*sin(Angle))
    Procedure CabinetProject (const Angle : Extended; var X, Y : Extended);     // (x,y)=(x+z/2*cos(Angle),y+z/2*sin(Angle))

    { Perspective projections                                                  }
    { Zv = distance from origin of z-axis vanishing point                      }
    { Xv = distance from origin of x-axis vanishing point                      }
    Procedure OnePointPerspectiveProject (const Angle, Zv : Extended; var X, Y : Extended);
    Procedure TwoPointPerspectiveProject (const Angle, Xv, Zv : Extended; var X, Y : Extended);

    Property X : Extended read GetX write SetX;
    Property Y : Extended read GetY write SetY;
    Property Z : Extended read GetZ write SetZ;
  end;



{                                                                              }
{ 3D Transformation matrices                                                   }
{   Multiply with a T3DPoint to transform. Transform matrices can also be      }
{   multiplied with each other before being applied to a T3DPoint.             }
{   All are 4x4 matrices.                                                      }
{                                                                              }
Function OriginAndScaleTransform (const TX, TY, TZ, SX, SY, SZ : Extended) : TMatrix;
{ Translates origin with (TX, TY, TZ) and scale by (SX, SY, SZ)                }
Function XRotateTransform (const Angle : Extended) : TMatrix;
Function YRotateTransform (const Angle : Extended) : TMatrix;
Function ZRotateTransform (const Angle : Extended) : TMatrix;
Function XYZRotateTransform (const XAngle, YAngle, ZAngle : Extended) : TMatrix;
{ Rotate around x, y and z-axis                                                }



{                                                                              }
{ Combinatoric functions                                                       }
{                                                                              }
Function  Factorial (const N : Integer) : Extended;
Function  Combinations (const N, C : Integer) : Extended;
Function  Permutations (const N, P : Integer) : Extended;




{                                                                              }
{ Statistical functions                                                        }
{                                                                              }
Function  GammaLn (X : Extended) : Extended;
{ Returns the natural logarithm of the gamma function with paramater X         }
Function  BinomialCoeff (N, R : Integer) : Extended;
{ Returns the binomial coeff for the bin (n) distribution                      }

{ =================================================== Random number generators }
Function  RandomSeed : LongWord;
{ Returns a random seed value, based on the Windows counter, the CPU counter   }
{ and the current date/time.                                                   }
Function  RandomUniform : LongWord;
{ Returns a random number that is uniformly distributed. See implementation    }
{ for specifics.                                                               }
Function  RandomUniformF : Extended;
{ Returns a uniformly distributed real number between 0 and 1.                 }
Function  RandomNormalF : Extended;
{ Return a random number that is normal (0, 1) distributed (gaussian           }
{ distribution)                                                                }
Function  RandomPseudoword (const Length : Integer) : String;
{ Returns a random pseudo-word that's human readable. Sometimes padded with    }
{ one or two numerical digits. Ideal for use as a password generator.          }

{ ========================================== Cumulative distribution functions }
Function  CummNormal (const u, s, X : Extended) : Extended;
{ CumNormal returns the area under the N(u,s) distribution.                    }
Function  CummNormal01 (const X : Extended) : Extended;
{ CumNormal01 returns the area under the N(0,1) distribution.                  }
Function  CummChiSquare (const Chi, Df : Extended) : Extended;
{ CumChiSquare returns the area under the X^2 (Chi-squared) (Chi, Df)          }
{ distribution.                                                                }
Function  CumF (const f, Df1, Df2 : Extended) : Extended;
{ CumF returns the area under the F (f, Df1, Df2) distribution.                }
Function  CummPoisson (const X : Integer; const u : Extended) : Extended;
{ CummPoison returns the area under the Poi(u)-distribution.                   }

{ =============================== Inverse of cumulative distribution functions }
Function  InvCummNormal01 (Y0 : Extended) : Extended;
{ InvCummNormal01 returns position on X-axis that gives cummulative area       }
{   of Y0 under the N(0,1) distribution.                                       }
Function  InvCummNormal (const u, s, Y0 : Extended) : Extended;
{ InvCummNormal returns position on X-axis that gives cummulative area         }
{   of Y0 under the N(u,s) distribution.                                       }



{                                                                              }
{ Computer maths                                                               }
{                                                                              }
Function DecodeBase (const S : String; const Base : Byte) : Int64;
{ Converts string S of Base to an integer.                                     }
{ Uses an alphabeth of up to 36 characters (0-9A-Z)                            }
Function BinToInt (const S : String) : Int64;
Function OctToInt (const S : String) : Int64;
Function HexToInt (const S : String) : Int64;

Function EncodeBase (const I : Int64; const Base : Byte) : String;
{ Converts number I to Base.                                                   }
{ Uses an alphabeth of up to 36 characters (0-9A-Z)                            }
Function IntToBin (const I : Int64) : String;
Function IntToOct (const I : Int64) : String;
Function IntToHex (const I : Int64) : String;

Function DecodeBase64 (const S, Alphabet : String; const Zeros : CharSet = []) : String;
{ Converts a base 64 string using Alphabet (64 characters for values 0-63) to  }
{ a binary string.                                                             }

const
  b64_MIMEBase64 = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';
  b64_UUEncode   = ' !"#$%&''()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_';
  b64_XXEncode   = '+-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';

Function UUDecode (const S : String) : String;
Function MIMEBase64Decode (const S : String) : String;
Function XXDecode (const S : String) : String;



{                                                                              }
{ Hashing functions                                                            }
{   The CRC16 function is the CCITT V.41 polynomial hashing function used for  }
{     calculating CRCs in communications. "This polynomial traps 100% of       }
{     1 bit, 2 bit, odd numbers of bit errors, 100% of <= 16 bit burst errors  }
{     and over 99% of all other errors." - See implementation for more detail. }
{   MD5 is an Internet standard secure hashing function, that was              }
{     produced by RSA Data Security, Inc. and placed in the public domain.     }
{     It returns a 128-bit (4 LongWords) digest. IMAC (another Internet        }
{     standard) needs to be implemented for variable keys.                     }
{   XOR and Checksum is general purpose hashing functions.                     }
{   See the implementation notes for speed comparisons.                        }
{                                                                              }
{   Hash is the general purpose hashing function. It returns a value in the    }
{   range 0..Slots-1. It's currently implemented using CRC32.                  }
{                                                                              }
type
  LongWordArray = Array of LongWord;

Function CalcCCITT_CRC16 (const Data : String) : Word; overload;
Function CalcCCITT_CRC16 (const Octet : Byte; const CRC16 : Word) : Word; overload;
Function CalcCRC32 (const Octet : Byte; const CRC32 : LongWord) : LongWord; overload;
Function CalcCRC32 (var Buf; const BufLength : Integer; const CRC32 : LongWord) : LongWord; overload;
Function CalcCRC32 (const Data : String) : LongWord; overload;
Function CalcChecksum32 (const Data : String) : LongWord;
Function CalcXOR8 (const Data : String) : Byte;                                 // Blazingly fast, +/- 1.2 clock ticks per character on P166MMX
Function CalcXOR16 (const Data : String) : Word;
Function CalcXOR32 (const Data : String) : LongWord;
Function CalcMD5 (const Data : TExStream) : LongWordArray; overload;
Function CalcMD5 (const Data : String) : LongWordArray; overload;

Function Hash (const S : String; const Slots : LongWord = MaxLongWord) : LongWord;



{                                                                              }
{ Interest conversions                                                         }
{                                                                              }
{   i         = nominal interest rate (per period)                             }
{   n         = number of periods                                              }
{   v(i)      = discounted interest rate                                       }
{               ie present value of a future payment of 1 made at the end of   }
{               the term that applies to the interest rate.                    }
{   v(i,n)    = interest rate, i, discounted for n periods                     }
{               ie present value of a payment of 1 made at end of period n.    }
{   d(i)      = interest rate, i, as a discount rate.                          }
{   ForceI(i) = interest rate, i, as a force of interest.                      }
{               ie, interest rate over a period that tends to zero, divided by }
{               the period.                                                    }
{   i(i,p)    = nominal interest rate over the term (divide it by p to get the }
{               nominal interest rate per period), if the effective interest   }
{               rate over the term is i.                                       }
{   d(i,p)    = discount rate over the term, given the effective interest rate }
{               over p periods is i.                                           }
{   ic(i,n)   = compound interest function                                     }
{               ie future value of an amount of 1, accumulating at i per       }
{               period for n periods.                                          }
Function ForceAsI (const d : Extended) : Extended;
{ Force of intrest d as an intrest rate, i                                     }
Function DiscountAsI (const d : Extended) : Extended;
{ Discount rate, d, as intrest rate, i                                         }
Function VAsI (const v : Extended) : Extended;
{ Discounted value of i                                                        }
Function IAsDiscount (const i : Extended) : Extended;
{ Interest rate, i, as a discount rate, d                                      }
Function IAsForce (const i : Extended) : Extended;
{ Interest rate, i, as a force of Interest                                     }
Function IAsV (const i : Extended) : Extended;
{ Interest rate, i, discounted                                                 }
{ ie Present value of future payment of 1 at end of period 1                   }
Function v (const i, n : Extended) : Extended;
{ Interest rate, i, discounted for n periods                                   }
{ ie Present value of future payment of 1 at end of period n                   }
Function ip (const i, p : Extended) : Extended;
{ i(p), interest rate per period for p periods                                 }
Function dp (const i, p : Extended) : Extended;
{ d(p), discount rate per period for p periods                                 }


{                                                                              }
{ Annuity functions                                                            }
{                                                                              }
{   "payable in arrear"  = payable at the end of the period.                   }
{                          also called an "ordinary annuity".                  }
{   "payable in advance" = payable at the beginning of the period.             }
{                          also called "annuity due".                          }
{                                                                              }
{   a____  = present value of an annuity of 1 payable in arrear for n periods  }
{    i,n|    discounted with a nominal interest rate of i.                     }
{            prenounced: "a-n"                                                 }
{                                                                              }
{   ____  = present value of an annuity of 1 payable in advance for n periods }
{    i,n|    discounted with a nominal interest rate of i.                     }
{            prenounced: "a-dot-n"                                             }
{   _                                                                          }
{   a____  = present value of an annuity of 1 payable continuously for n       }
{    i,n|    periods, discounted with a nominal interest rate of i.            }
{            prenounced: "a-continuous-n"                                      }
{                                                                              }
{   s____  = future value of an annuity of 1 payable in arrear for n periods   }
{    i,n|    accumulated with a nominal interest rate of i.                    }
{            prenounced: "s-n"                                                 }
{                                                                              }
{  s"____  = future value of an annuity of 1 payable in advance for n periods  }
{    i,n|    accumulated with a nominal interest rate of i.                    }
{            prenounced: "s-dot-n"                                             }
{                                                                              }
{   (p)      present value of an annuity of 1/p payable in arrear p times      }
{   a----  = per period for n periods, discounted with a nominal interest      }
{    i,n|    rate of i per period.                                             }
{            prenounced: "a-p-n"                                               }
{                                                                              }
{   (p)      present value of an annuity of 1/p payable in advance p times     }
{   ----  = per period for n periods, discounted with a nominal interest      }
{    i,n|    rate of i per period.                                             }
{            prenounced: "a-dot-p-n"                                           }
{                                                                              }
{   (p)      future value of an annuity of 1/p payable in arrear p times       }
{   s----  = per period for n periods, discounted with a nominal interest      }
{    i,n|    rate of i per period.                                             }
{            prenounced: "s-p-n"                                               }
{                                                                              }
{   (p)      future value of an annuity of 1/p payable in advance p times      }
{  s"----  = per period for n periods, discounted with a nominal interest      }
{    i,n|    rate of i per period.                                             }
{            prenounced: "s-dot-p-n"                                           }
{                                                                              }
{  Ia____  = present value of an increasing annuity (1, 2, 3, ...) payable in  }
{    i,n|    arrear, discounted with a nominal interest rate of i.             }
{            prenounced: "I-a-n"                                               }
{                                                                              }
{  I____  = present value of an increasing annuity (1, 2, 3, ...) payable in  }
{    i,n|    advance, discounted with a nominal interest rate of i.            }
{            prenounced: "I-a-dot-n"                                           }
{   _                                                                          }
{  Ia____  = present value of an increasing annuity (1, 2, 3, ...) payable     }
{    i,n|    continuously, discounted with a nominal interest rate of i.       }
{            prenounced: "I-a-continuous-n"                                    }
{  __                                                                          }
{  Ia____  = present value of a continously increasing annuity (1, 2, 3, ...)  }
{    i,n|    payable continuously, discounted with a nominal interest rate of i}
{            prenounced: "I-continious-a-continuous-n"                         }
Function an (const i, n : Extended) : Extended;
Function aDOTn (const i, n : Extended) : Extended;
Function sn (const i, n : Extended) : Extended;
Function sDOTn (const i, n : Extended) : Extended;
Function aCONTn (const i, n : Extended) : Extended;
Function Ian (const i, n : Extended) : Extended;
Function IaDOTn (const i, n : Extended) : Extended;
Function IaCONTn (const i, n : Extended) : Extended;
Function ICONTaCONTn (const i, n : Extended) : Extended;
Function apn (const i, p, n : Extended) : Extended;
Function aDOTpn (const i, p, n : Extended) : Extended;
Function spn (const i, p, n : Extended) : Extended;
Function sDOTpn (const i, p, n : Extended) : Extended;



{                                                                              }
{ TLifeTable                                                                   }
{                                                                              }
type
  TLifeTable = class (TFloatVector)
    Function l (const x : Integer) : Extended;
    Function d (const x, n : Integer) : Extended;
    { = lx - l(x+n)                                                            }

    Function p (const x, n : Integer) : Extended;
    { m|nPx  Probability of survival for n periods for individual aged x       }
    Function q (const x, n, m : Integer) : Extended;
    { m|nQx  Probibility of mortality for n periods for individual aged x with }
    {        death deferred for m periods.                                     }
    {        = (l(x+m) - l(x+m+n)) / lx                                        }
    {        = mPx nQ(x+m)                                                     }

    Function Dx (const i : Extended; const x : Integer) : Extended;
    { Dx = v^x * lx                                                            }
    Function Nx (const i : Extended; const x : Integer) : Extended;
    { Nx = Sum (Di)  i = x..infinity                                           }
    Function Sx (const i : Extended; const x : Integer) : Extended;
    { Sx = Sum (Ni)  i = x..infinity                                           }
    Function Cx (const i : Extended; const x : Integer) : Extended;
    { Cx = v^(x+1) * dx                                                        }
    Function Mx (const i : Extended; const x : Integer) : Extended;
    { Mx = Sum (Ci)  i = x..infinity                                           }
    Function Rx (const i : Extended; const x : Integer) : Extended;
    { Rx = Sum (Mi)  i = x..infinity                                           }
  end;

const
  { Example table: A1967-70 Select Life table }
  A1967T70S : Array [0..80] of Extended = (
    34481.408, 34456.927, 34433.841, 34412.836, 34393.221, 34375.681, 34359.181,
    34344.063, 34329.638, 34315.907, 34303.210, 34290.518, 34277.830, 34264.461,
    34250.070, 34232.259, 34209.439, 34179.680, 34143.368, 34109.166, 34076.957,
    34046.610, 34017.983, 33990.921, 33965.254, 33940.795, 33917.341, 33894.668,
    33872.531, 33650.662, 33827.764, 33806.514, 33783.557, 33759.503, 33733.924,
    33706.352, 33676.272, 33643.122, 33606.286, 33565.089, 33518.794, 33466.599,
    33407.624, 33340.915, 33265.431, 33180.042, 33083.523, 32974.549, 32851.686,
    32713.392, 32558.008, 32383.756, 32188.740, 31970.942, 31728.226, 31458.342,
    31158.931, 30827.543, 30461.645, 30058.648, 29615.936, 29130.898, 28600.975,
    28023.708, 27396.808, 26718.225, 25986.236, 25199.536, 24357.348, 23459.538,
    22506.732, 21500.445, 20443.198, 19338.635, 18191.617, 17008.294, 15796.140,
    14563.940, 13321.717, 12080.592, 10852.568);



{                                                                              }
{ Numerical routines                                                           }
{                                                                              }
type
  fx = Function (const x : Extended) : Extended;

Function SecantSolver (const f : fx; const y, Guess1, Guess2 : Extended) : Extended;
{ Uses Secant method to solve for x in f(x) = y                                }

Function NewtonSolver (const f, df : fx; const y, Guess : Extended) : Extended;
{ Uses Newton's method to solve for x in f(x) = y.                             }
{ df = f'(x).                                                                  }
{ Note: This implementation does not check if the solver goes on a tangent     }
{       (which can happen with certain Guess values)                           }

Function FirstDerivative (const f : fx; const x : Extended) : Extended;
{ Returns the value of f'(x)                                                   }
{ Uses (-f(x+2h) + 8f(x+h) - 8f(x-h) + f(x-2h)) / 12h                          }

Function SecondDerivative (const f : fx; const x : Extended) : Extended;
{ Returns the value of f''(x)                                                  }
{ Uses (-f(x+2h) + 16f(x+h) - 30f(x) + 16f(x-h) - f(x-2h)) / 12h^2             }

Function ThirdDerivative (const f : fx; const x : Extended) : Extended;
{ Returns the value of f'''(x)                                                 }
{ Uses (f(x+2h) - 2f(x+h) + 2f(x-h) - f(x-2*h)) / 2h^3                         }

Function FourthDerivative (const f : fx; const x : Extended) : Extended;
{ Returns the value of f''''(x)                                                }
{ Uses (f(x+2h) - 4f(x+h) + 6f(x) - 4f(x-h) + f(x-2h)) / h^4                   }

Function SimpsonIntegration (const f : fx; const a, b : Extended; N : Integer) : Extended;
{ Returns the area under f from a to b, by applying Simpson's 1/3 Rule with    }
{ N subdivisions.                                                              }




implementation

uses
  // System units
  Windows,              // GetTickCount

  // Delphi Fundamentals (L0)
  cStrings;



{                                                                              }
{ To-do's:                                                                     }
{   * Finish HugeInt functions. Add HugeFloat functions.                       }
{   * CRC16/32 in assembly.                                                    }
{   * TMAC (RFC2104) to be applied to MD5 to get variable key (currently       }
{     fixed).                                                                  }
{   * Add user/machine/other "random" states to RandomSeed.                    }
{   * Rectangular -> Polar (incl polar mode for TComplex)                      }
{   * Regression functions                                                     }
{   * EncodeBase to work from right instead of left (get rid of those          }
{     logs)                                                                    }
{                                                                              }



{                                                                              }
{ Dynamic array functions                                                      }
{                                                                              }
Function Sum (const R : IntegerArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64; overload;
var I, H : Integer;
  Begin
    H := Length (R) - 1;
    Result := 0;
    For I := Max (LoIdx, 0) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Inc (Result, R [I]);
  End;

Function Sum (const R : Int64Array; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64; overload;
var I, H : Integer;
  Begin
    H := Length (R) - 1;
    Result := 0;
    For I := Max (LoIdx, 0) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Inc (Result, R [I]);
  End;

Function Sum (const R : SingleArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;
var I, H : Integer;
  Begin
    H := Length (R) - 1;
    Result := 0.0;
    For I := Max (LoIdx, 0) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Result := Result + R [I];
  End;

Function Sum (const R : DoubleArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;
var I, H : Integer;
  Begin
    H := Length (R) - 1;
    Result := 0.0;
    For I := Max (LoIdx, 0) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Result := Result + R [I];
  End;

Function Sum (const R : ExtendedArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended; overload;
var I, H : Integer;
  Begin
    H := Length (R) - 1;
    Result := 0.0;
    For I := Max (LoIdx, 0) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Result := Result + R [I];
  End;

Function WeightedSum (const R : ExtendedArray; const W : ExtendedArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended;
var I, H : Integer;
  Begin
    Result := 0.0;
    H := Min (Length (R), Length (W)) - 1;
    For I := Max (LoIdx, 0) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Result := Result + R [I] * W [I];
  End;

Function WeightedSum (const R : IntegerArray; const W : ExtendedArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended;
var I, H : Integer;
  Begin
    Result := 0.0;
    H := Min (Length (R), Length (W)) - 1;
    For I := Max (LoIdx, 0) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Result := Result + R [I] * W [I];
  End;

Function WeightedSum (const R : IntegerArray; const W : IntegerArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64;
var I, H : Integer;
  Begin
    Result := 0;
    H := Min (Length (R), Length (W)) - 1;
    For I := Max (LoIdx, 0) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Inc (Result, R [I] * W [I]);
  End;

Function WeightedSum (const R : Int64Array; const W : ExtendedArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended;
var I, H : Integer;
  Begin
    Result := 0;
    H := Min (Length (R), Length (W)) - 1;
    For I := Max (LoIdx, 0) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Result := Result + R [I] * W [I];
  End;

Function WeightedSum (const R : Int64Array; const W : Int64Array; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64;
var I, H : Integer;
  Begin
    Result := 0;
    H := Min (Length (R), Length (W)) - 1;
    For I := Max (LoIdx, 0) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Inc (Result, R [I] * W [I]);
  End;

Function Product (const R : IntegerArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64;
var I, H : Integer;
  Begin
    Result := 1;
    H := Length (R) - 1;
    For I := Max (Integer (0), LoIdx) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Result := Result * R [I];
  End;

Function Product (const R : Int64Array; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Int64;
var I, H : Integer;
  Begin
    Result := 1;
    H := Length (R) - 1;
    For I := Max (Integer (0), LoIdx) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Result := Result * R [I];
  End;

Function Product (const R : SingleArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended;
var I, H : Integer;
  Begin
    Result := 1.0;
    H := Length (R) - 1;
    For I := Max (Integer (0), LoIdx) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Result := Result * R [I];
  End;

Function Product (const R : DoubleArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended;
var I, H : Integer;
  Begin
    Result := 1.0;
    H := Length (R) - 1;
    For I := Max (Integer (0), LoIdx) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Result := Result * R [I];
  End;

Function Product (const R : ExtendedArray; const LoIdx : Integer = 0; const HiIdx : Integer = -1) : Extended;
var I, H : Integer;
  Begin
    Result := 1.0;
    H := Length (R) - 1;
    For I := Max (Integer (0), LoIdx) to Cond (HiIdx < 0, H, Min (H, HiIdx)) do
      Result := Result * R [I];
  End;



{                                                                              }
{ Miscellaneous functions                                                      }
{                                                                              }
Function Sgn (const R : Integer) : Integer;
  Begin
    if R < 0 then
      Result := -1 else
      Result := 1;
  End;

Function Sgn (const R : Int64) : Integer;
  Begin
    if R < 0 then
      Result := -1 else
      Result := 1;
  End;

Function Sgn (const R : Single) : Integer;
  Begin
    if R < 0.0 then
      Result := -1 else
      Result := 1;
  End;

Function Sgn (const R : Double) : Integer;
  Begin
    if R < 0.0 then
      Result := -1 else
      Result := 1;
  End;

Function Sgn (const R : Extended) : Integer;
  Begin
    if R < 0.0 then
      Result := -1 else
      Result := 1;
  End;

Function Distance (const X1, Y1, X2, Y2 : Extended) : Extended;
var DX, DY : Extended;
  Begin
    DX := X1 - X2;
    DY := Y1 - Y2;
    Result := Sqrt (DX * DX + DY * DY);
  End;

Procedure RealToDMS (const X : Extended; var Degs, Mins, Secs : Extended);
var Y : Extended;
  Begin
    Degs := Int (X);
    Y := Frac (X) * 60.0;
    Mins := Int (Y);
    Secs := Frac (Y) * 60.0;
  End;

Function DMSToReal (const Degs, Mins, Secs : Extended) : Extended;
  Begin
    Result := Degs + Mins / 60.0 + Secs / 3600.0;
  End;

Function CanonicalForm (const Theta : Extended) : Extended;                     {-PI < theta <= PI}
  Begin
    if Abs (Theta) > Pi then
       Result := Round (Theta / (Pi * 2)) * 2 * Pi else
       Result := Theta;
  End;

Procedure PolarToRectangular (const R, Theta : Extended; var X, Y : Extended);
var S, C : Extended;
  Begin
    SinCos (Theta, S, C);
    X := R * C;
    Y := R * S;
  End;

Procedure RectangularToPolar (const X, Y : Extended; var R, Theta : Extended);
  Begin
    if FloatZero (X) then
      if FloatZero (Y) then
        R := 0.0 else
        if Y > 0.0 then
          R := Y else
          R := -Y else
      R := Sqrt (Sqr (X) + Sqr (Y));
    Theta := ArcTan2 (Y, X);
  End;

{                                                                              }
{ TRational                                                                    }
{                                                                              }
Constructor TRational.Create (const Numerator, Denominator : Int64);
  Begin
    inherited Create;
    Assign (Numerator, Denominator);
  End;

Constructor TRational.Create;
  Begin
    inherited Create;
    AssignZero;
  End;

Constructor TRational.Create (const R : Extended);
  Begin
    inherited Create;
    Assign (R);
  End;

Procedure TRational.Simplify;
var I : Int64;
  Begin
    if FN < 0 then
      begin
        FT := -FT;
        FN := -FN;
      end;
    if (FT = 1) or (FN = 1) or (FT = 0) then
      exit;

    I := GCD (FT, FN);
    FT := FT div I;
    FN := FN div I;
  End;

Procedure TRational.Assign (const Numerator, Denominator : Int64);
  Begin
    if Denominator = 0 then
      raise EDivByZero.Create ('Invalid rational number');

    FT := Numerator;
    FN := Denominator;
    if FN <> 1 then
      Simplify;
  End;

Procedure TRational.Assign (const R : TRational);
  Begin
    FT := R.FT;
    FN := R.FN;
  End;

{ See http://forum.swarthmore.edu/dr.math/faq/faq.fractions.html for an        }
{ explanation on how to convert decimal numbers to fractions.                  }
const
  CalcFracAccuracy : Int64 = 1000000000000000000;
  // Max 1e18 for Int64/Extended, 1e9 for LongInt/Double

Procedure TRational.Assign (const R : Extended);

  // Pre: Abs (R) < 1.0
  Function CalcFrac (const R : Extended; const Level : Integer = 1) : TRational;
  var I : Extended;
      Z : Int64;
    Begin
      if FloatZero (R) or (Level = 12) then  // 0 (if Level = 12 we get an approximation)
        Result := TRational.Create else
      if FloatEqual (R, 1.0) then            // 1
        begin
          Result := TRational.Create;
          Result.AssignOne;
        end else
      if System.Abs (Frac (R * CalcFracAccuracy)) < 1.0 then // terminating decimal (<8)
        Result := TRational.Create (Round (R * CalcFracAccuracy), CalcFracAccuracy) else
        begin                               // recursive process
          I := 1.0 / R;
          Result := CalcFrac (Frac (I), Level + 1);
          Z := Trunc (I);
          Result.Add (Z);
          Result.Reciprocal;
        end;
    End;

var T : TRational;
    Z : Int64;

  Begin
    T := CalcFrac (Frac (R));
    try
      Z := Trunc (R);
      T.Add (Z);
      Assign (T);
    finally
      T.Free;
    end;
  End;

Procedure TRational.AssignOne;
  Begin
    FT := 1;
    FN := 1;
  End;

Procedure TRational.AssignZero;
  Begin
    FT := 0;
    FN := 1;
  End;

Function TRational.IsEqual (const Numerator, Denominator : Int64) : Boolean;
var R : TRational;
  Begin
    R := TRational.Create (Numerator, Denominator);
    Result := IsEqual (R);
    R.Free;
  End;

Function TRational.IsEqual (const R : TRational) : Boolean;
  Begin
    Result := (FT = R.FT) and (FN = R.FN);
  End;

Function TRational.IsEqual (const R : Extended) : Boolean;
  Begin
    Result := FloatEqual (R, GetAsReal);
  End;

Function TRational.IsOne : Boolean;
  Begin
    Result := (FT = 1) and (FN = 1);
  End;

Function TRational.IsZero : Boolean;
  Begin
    Result := FT = 0;
  End;

Function TRational.Duplicate : TRational;
  Begin
    Result := TRational.Create (FT, FN);
  End;

Procedure TRational.SetAsReal (const R : Extended);
  Begin
    Assign (R);
  End;

Procedure TRational.SetAsString (const S : String);
var F : Integer;
  Begin
    F := Pos ('/', S);
    if F = 0 then
      Assign (StrToFloat (S)) else
      Assign (StrToInt (CopyLeft (S, F - 1)), StrToInt (CopyFrom (S, F + 1)));
  End;

Function TRational.GetAsReal : Extended;
  Begin
    Result := FT / FN;
  End;

Function TRational.GetAsString : String;
  Begin
    Result := IntToStr (FT) + '/' + IntToStr (FN);
  End;

Procedure TRational.Add (const R : TRational);
  Begin
    FT := FT * R.FN + R.FT * FN;
    FN := FN * R.FN;
    Simplify;
  End;

Procedure TRational.Add (const V : Int64);
  Begin
    Inc (FT, FN * V);
  End;

Procedure TRational.Add (const V : Extended);
  Begin
    Assign (GetAsReal + V);
  End;

Procedure TRational.Subtract (const V : Extended);
  Begin
    Assign (GetAsReal - V);
  End;

Procedure TRational.Subtract (const R : TRational);
  Begin
    FT := FT * R.FN - R.FT;
    FN := FN * R.FN;
    Simplify;
  End;

Procedure TRational.Subtract (const V : Int64);
  Begin
    Dec (FT, FN * V);
  End;

Procedure TRational.Negate;
  Begin
    FT := -FT;
  End;

Procedure TRational.Abs;
  Begin
    FT := System.Abs (FT);
    FN := System.Abs (FN);
  End;

Function TRational.Sgn : Integer;
  Begin
    if cMaths.Sgn (FT) = cMaths.Sgn (FN) then
      Sgn := 1 else
      Sgn := -1;
  End;

Procedure TRational.Divide (const V : Int64);
  Begin
    if V = 0 then
      raise EDivByZero.Create ('Division by zero');

    FN := FN * V;
    Simplify;
  End;

Procedure TRational.Divide (const R : TRational);
  Begin
    if R.FT = 0 then
      raise EDivByZero.Create ('Rational division by zero');

    FT := FT * R.FN;
    FN := FN * R.FT;
    Simplify;
  End;

Procedure TRational.Divide (const V : Extended);
  Begin
    Assign (GetAsReal / V);
  End;

Procedure TRational.Reciprocal;
  Begin
    if FT = 0 then
      raise EDivByZero.Create ('Rational division by zero');

    Swap (FT, FN);
  End;

Procedure TRational.Multiply (const R : TRational);
  Begin
    FT := FT * R.FT;
    FN := FN * R.FN;
    Simplify;
  End;

Procedure TRational.Multiply (const V : Int64);
  Begin
    FT := FT * V;
    Simplify;
  End;

Procedure TRational.Multiply (const V : Extended);
  Begin
    Assign (GetAsReal * V);
  End;

Procedure TRational.Power (const R : TRational);
  Begin
    Assign (Math.Power (GetAsReal, R.GetAsReal));
  End;

Procedure TRational.Power (const V : Int64);
var T, N : Extended;
  Begin
    T := FT;
    N := FN;
    FT := Round (IntPower (T, V));
    FN := Round (IntPower (N, V));
  End;

Procedure TRational.Power (const V : Extended);
  Begin
    Assign (Math.Power (FT, V) / Math.Power (FN, V));
  End;

Procedure TRational.Sqrt;
  Begin
    Assign (System.Sqrt (FT / FN));
  End;

Procedure TRational.Sqr;
  Begin
    FT := System.Sqr (FT);
    FN := System.Sqr (FN);
  End;



{                                                                              }
{ TComplex                                                                     }
{                                                                              }
Constructor TComplex.Create (const TheRealPart, TheImaginaryPart : Extended);
  Begin
    inherited Create;
    FReal := TheRealPart;
    FImag := TheImaginaryPart;
  End;

Function TComplex.IsI : Boolean;
  Begin
    Result := FloatZero (FReal) and FloatEqual (FImag, 1.0);
  End;

Function TComplex.IsZero : Boolean;
  Begin
    Result := FloatZero (FReal) and FloatZero (FImag);
  End;

Function TComplex.IsEqual (const C : TComplex) : Boolean;
  Begin
    Result := FloatEqual (FReal, C.FReal) and FloatEqual (FImag, C.FImag);
  End;

Function TComplex.IsEqual (const R, I : Extended) : Boolean;
  Begin
    Result := FloatEqual (FReal, R) and FloatEqual (FImag, I);
  End;

Procedure TComplex.AssignZero;
  Begin
    FReal := 0.0;
    FImag := 0.0;
  End;

Procedure TComplex.AssignI;
  Begin
    FReal := 0.0;
    FImag := 1.0;
  End;

Procedure TComplex.AssignMinI;
  Begin
    FReal := 0.0;
    FImag := -1.0;
  End;

Procedure TComplex.Assign (const C : TComplex);
  Begin
    FReal := C.FReal;
    FImag := C.FImag;
  End;

Procedure TComplex.Assign (const V : Extended);
  Begin
    FReal := V;
    FImag := 0.0;
  End;

Function TComplex.Duplicate : TComplex;
  Begin
    Result := TComplex.Create (FReal, FImag);
  End;

Function TComplex.GetAsString : String;
var RZ, IZ : Boolean;
  Begin
    RZ := FloatZero (FReal);
    IZ := FloatZero (FImag);
    if IZ then
      Result := FloatToStr (FReal) else
      begin
        Result := Result + FloatToStr (FImag) + 'i';
        if not RZ then
          Result := Result + Cond (Sgn (FReal) > 0, '+', '-') + FloatToStr (Abs (FReal));
      end;
  End;

Procedure TComplex.SetAsString (const S : String);
var F, G, H : Integer;
  Begin
    F := Pos ('(', S);
    G := Pos (',', S);
    H := Pos (')', S);
    if (F <> 1) or (H <> Length (S)) or (G < F) or (G > H) then
      raise EConvertError.Create ('Can not convert string to complex number');
    FReal := StrToFloat (CopyRange (S, F + 1, G - 1));
    FImag := StrToFloat (CopyRange (S, G + 1, H - 1));
  End;

Procedure TComplex.Add (const C : TComplex);
  Begin
    FReal := FReal + C.FReal;
    FImag := FImag + C.FImag;
  End;

Procedure TComplex.Add (const V : Extended);
  Begin
    FReal := FReal + V;
  End;

Procedure TComplex.Subtract (const C : TComplex);
  Begin
    FReal := FReal - C.FReal;
    FImag := FImag - C.FImag;
  End;

Procedure TComplex.Subtract (const V : Extended);
  Begin
    FReal := FReal - V;
  End;

Procedure TComplex.Multiply (const C : TComplex);
var R : Extended;
  Begin
    R := FReal;
    FReal := R * C.FReal - FImag * C.FImag;
    FImag := R * C.FImag + FImag * C.FReal;
  End;

Procedure TComplex.Multiply (const V : Extended);
  Begin
    FReal := FReal * V;
    FImag := FImag * V;
  End;

Procedure TComplex.MultiplyI;
var R : Extended;
  Begin
    R := FReal;
    FReal := -FImag;
    FImag := R;
  End;

Procedure TComplex.MultiplyMinI;
var R : Extended;
  Begin
    R := FReal;
    FReal := FImag;
    FImag := -R;
  End;

Function TComplex.Denom : Extended;
  Begin
    Result := Sqr (FReal) + Sqr (FImag);
  End;

Procedure TComplex.Divide (const C : TComplex);
var R, D : Extended;
  Begin
    D := Denom;
    if FloatZero (D) then
      raise EDivByZero.Create ('Complex division by zero') else
      begin
        R := FReal;
        FReal := (R * C.FReal + FImag * C.FImag) / D;
        FImag := (FImag * C.FReal - FReal * C.FImag) / D;
      end;
  End;

Procedure TComplex.Divide (const V : Extended);
var D : Extended;
  Begin
    D := Denom;
    if FloatZero (D) then
      raise EDivByZero.Create ('Complex division by zero') else
      begin
        FReal := (FReal * V) / D;
        FImag := (FImag * V) / D;
      end;
  End;

Procedure TComplex.Negate;
  Begin
    FReal := -FReal;
    FImag := -FImag;
  End;

Procedure TComplex.Conjugate;
  Begin
    FImag := -FImag;
  End;

Procedure TComplex.Inverse;
var D : Extended;
  Begin
    D := Denom;
    if FloatZero (D) then
      raise EDivByZero.Create ('Complex division by zero');
    FReal := FReal / D;
    FImag := - FImag / D;
  End;

Procedure TComplex.Exp;
var ExpZ : Extended;
    S, C : Extended;
  Begin
    ExpZ := System.Exp (FReal);
    SinCos (FImag, S, C);
    FReal := ExpZ * C;
    FImag := ExpZ * S;
  End;

Procedure TComplex.Ln;
var ModZ : Extended;
  Begin
    ModZ := Denom;
    if FloatZero (ModZ) then
      raise EDivByZero.Create ('Complex log zero');
    FReal := System.Ln (ModZ);
    FImag := ArcTan2 (FReal, FImag);
  End;

Procedure TComplex.Power (const C : TComplex);
  Begin
    if not IsZero then
      begin
        Ln;
        Multiply (C);
        Exp;
      end else
      if C.IsZero then
        Assign (1.0) else      { lim a^a = 1 as a-> 0 }
        AssignZero;            { 0^a = 0 for a <> 0   }
  End;

Function TComplex.Modulo : Extended;
  Begin
    Result := System.Sqrt (Denom);
  End;

Procedure TComplex.Sqrt;
var Root, Q : Extended;
  Begin
    if not FloatZero (FReal) or not FloatZero (FImag) then
      begin
        Root := System.Sqrt (0.5 * (Abs (FReal) + Modulo));
        Q := FImag / (2.0 * Root);
        if FReal >= 0.0 then
          begin
            FReal := Root;
            FImag := Q;
          end else
          if FImag < 0.0 then
            begin
              FReal := - Q;
              FImag := - Root;
            end else
            begin
              FReal := Q;
              FImag := Root;
            end;
      end else
      AssignZero;
  End;

Procedure TComplex.Cos;
  Begin
    FReal := System.Cos (FReal) * Cosh (FImag);
    FImag := -System.Sin (FReal) * Sinh (FImag);
  End;

Procedure TComplex.Sin;
  Begin
    FReal := System.Sin (FReal) * Cosh (FImag);
    FImag := -System.Cos (FReal) * Sinh (FImag);
  End;

Procedure TComplex.Tan;
var CCos : TComplex;
  Begin
    CCos := TComplex.Create (FReal, FImag);
    try
      CCos.Cos;
      if CCos.IsZero then
        raise EDivByZero.Create ('Complex division by zero');
      self.Sin;
      self.Divide (CCos);
    finally
      CCos.Free;
    end;
  End;



{                                                                              }
{ Primes                                                                       }
{   Note: It's only necessary to check up to Sqrt (N) for factors of N to      }
{   determinte primality. PrimeCacheLimit = 65537 < Sqrt (MaxLongInt), in      }
{   other words, for N <= MaxLongInt it will always do a lookup.               }
{                                                                              }
const
  PrimeCacheLimit = 65537; // 8K lookup table. Note: Sqrt(MaxLongInt) < 65537

var
  PrimeSet : TFlatBitArray = nil;

{ Initializes the bit-array lookup of prime numbers using Sieve's algorithm.   }
Procedure InitPrimeSet;
var I, J : Integer;
  Begin
    PrimeSet := TFlatBitArray.Create;
    PrimeSet.Count := PrimeCacheLimit + 1;
    PrimeSet.Fill (2, PrimeCacheLimit, True);
    PrimeSet.Fill (0, 1, False);
    For I := 2 to Trunc (Sqrt (PrimeCacheLimit)) do
      if PrimeSet.Bit [I] then
        For J := 2 to PrimeCacheLimit div I do
          PrimeSet.Bit [I * J] := False;
  End;

{ For small values of N (<=PrimeCacheLimit), uses a bit-array lookup. For      }
{ larger values, tries to find a prime factor.                                 }
Function IsPrime (const N : Int64) : Boolean;
var I : Integer;
    R : Extended;
  Begin
    if N < 0 then
      Result := IsPrime (-N) else
    if N < 2 then
      Result := False else
      begin
        if not Assigned (PrimeSet) then // initialize look-up table
          InitPrimeSet;
        if N <= PrimeCacheLimit then // do look-up
          Result := PrimeSet [N] else
          begin // calculate
            R := N;
            For I := 2 to Round (Sqrt (R)) do
              if ((I > PrimeCacheLimit) or PrimeSet [I]) and (N mod I = 0) then
                begin
                  Result := False;
                  exit;
                end;
            Result := True;
          end;
      end;
  End;

Function PrimeFactors (const N : Int64) : Int64Array;
var I, L : Integer;
    J    : Int64;
    R    : Extended;
  Begin
    SetLength (Result, 0);
    if N < 0 then
      Result := PrimeFactors (-N) else
    if N = 1 then
      exit else
      begin
        if not Assigned (PrimeSet) then // initialize look-up table
          InitPrimeSet;

        L := 0;
        J := N;
        R := N;
        For I := 2 to Round (Sqrt (R)) do
          if ((I > PrimeCacheLimit) or PrimeSet [I]) and (N mod I = 0) then
            begin // I is a prime factor
              Inc (L);
              SetLength (Result, L);
              Result [L - 1] := I;

              Repeat
                J := J div I;
                if J = 1 then // no more factors
                  exit;
              Until J mod I <> 0;
            end;
      end;
  End;

Function IsPrimeFactor (const F, N : Int64) : Boolean;
  Begin
    Result := (N mod F = 0) and IsPrime (F);
  End;

{ Find the GCD using Euclid's algorithm                                        }
Function GCD (const N1, N2 : Int64) : Int64;
var X, Y, J : Int64;
  Begin
    X := N1;
    Y := N2;
    if X < Y then
      Swap (X, Y);

    While (X <> 1) and (X <> 0) and (Y <> 1) and (Y <> 0) do
      begin
        J := (X - Y) mod Y;
        if J = 0 then
          begin
            Result := Y;
            exit;
          end;
        X := Y;
        Y := J;
      end;
    Result := 1;
  End;



{                                                                              }
{ TFloatVector                                                                 }
{                                                                              }
class Function TFloatVector.CreateInstance : AType;
  Begin
    Result := TFloatVector.Create;
  End;

Function TFloatVector.IsZero : Boolean;
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      if not FloatZero (FData [I]) then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

{ Mathematical functions                                                       }
Procedure TFloatVector.Add (const V : TFloatVector; const Factor : Extended);
var I : Integer;
  Begin
    if Length (V.FData) <> Length (FData) then
      raise EVector.Create ('Vector size mismatch');

    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] + V.FData [I] * Factor;
  End;

Procedure TFloatVector.Add (const V : TFloatVector);
  Begin
    Add (V, 1.0);
  End;

Procedure TFloatVector.Add (const Value : Extended);
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] + Value;
  End;

Procedure TFloatVector.Multiply (const V : TFloatVector);
var I : Integer;
  Begin
    if V.Count <> Length (FData) then
      raise EVector.Create ('Vector size mismatch');

    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] * V.FData [I];
  End;

Procedure TFloatVector.Multiply (const Value : Extended);
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] * Value;
  End;

Procedure TFloatVector.SquareValues;
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := Sqr (FData [I]);
  End;

Function TFloatVector.Angle (const V : TFloatVector) : Extended;
  Begin
    Result := ArcCos (DotProduct (V) / (Norm * V.Norm));
  End;

Function TFloatVector.DotProduct (const V : TFloatVector) : Extended;
var I : Integer;
  Begin
    if V.Count <> Length (FData) then
      raise EVector.Create ('Vector size mismatch');

    Result := 0.0;
    For I := 0 to Length (FData) - 1 do
      Result := Result + FData [I] * V.FData [I];
  End;

Function TFloatVector.Norm : Extended;
  Begin
    Result := Sqrt (DotProduct (self));
  End;

Procedure TFloatVector.Invert;
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := 1.0 / FData [I];
  End;

{ Statistical functions                                                        }
Function TFloatVector.Sum (const LoIdx, HiIdx : Integer) : Extended;
  Begin
    Result := cMaths.Sum (FData, LoIdx, HiIdx);
  End;

Function TFloatVector.Sum : Extended;
  Begin
    Result := cMaths.Sum (FData);
  End;

Function TFloatVector.WeightedSum (const Weights : TFloatVector) : Extended;
  Begin
    if Weights.Count <> Length (FData) then
      raise EVector.Create ('Vector size mismatch');

    Result := cMaths.WeightedSum (FData, Weights.FData);
  End;

Function TFloatVector.Product (const LoIdx, HiIdx : Integer) : Extended;
  Begin
    Result := cMaths.Product (FData, LoIdx, HiIdx);
  End;

Function TFloatVector.Product : Extended;
  Begin
    Result := cMaths.Product (FData);
  End;

Function TFloatVector.MaxValue : Extended;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise EVector.Create ('No maximum');

    Result := FData [0];
    For I := 1 to Length (FData) - 1 do
      if FData [I] > Result then
        Result := FData [I];
  End;

Function TFloatVector.MinValue : Extended;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise EVector.Create ('No minimum');

    Result := FData [0];
    For I := 1 to Length (FData) - 1 do
      if FData [I] < Result then
        Result := FData [I];
  End;

Function TFloatVector.Mean : Extended;
  Begin
    if Length (FData) = 0 then
      raise Exception.Create ('No mean');

    Result := Sum / Length (FData);
  End;

Function TFloatVector.StdDev (var Mean : Extended) : Extended;
var S    : Extended;
    I, N : Integer;
  Begin
    N := Length (FData);
    if N = 0 then
      raise EDivByZero.Create ('No standard deviation');

    if N = 1 then
      begin
        Mean := FData [0];
        Result := FData [0];
      end else
      begin
        Mean := self.Mean;
        S := 0.0;
        For I := 0 to N - 1 do
          S := S + Sqr (Mean - FData [I]);
        Result := Sqrt (S / (N - 1));
      end;
  End;

Procedure TFloatVector.Normalize;
var S : Extended;
    I : Integer;
  Begin
    S := Norm;
    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] / S;
  End;

Function TFloatVector.HarmonicMean : Extended;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise Exception.Create ('No harmonic mean');

    Result := 0.0;
    For I := 0 to Length (FData) - 1 do
      if FData [I] < 0.0 then
        raise Exception.Create ('Vector contains negative values') else
        Result := Result + 1.0 / FData [I];
    Result := Length (FData) / Result;
  End;

Function TFloatVector.GeometricMean : Extended;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise Exception.Create ('No geometric mean');

    Result := 1.0;
    For I := 0 to Length (FData) - 1 do
      if FData [I] < 0.0 then
        raise Exception.Create ('Vector contains negative values') else
        Result := Result * FData [I];
    Result := Power (Result, 1.0 / Length (FData));
  End;

Function TFloatVector.Median : Extended;
var V : TFloatVector;
  Begin
    if Length (FData) = 0 then
      raise EVector.Create ('No median');

    V := TFloatVector (Duplicate);
    try
      V.Sort;
      Result := V.FData [(Length (V.FData) - 1) div 2];
    finally
      V.Free;
    end;
  End;

Function TFloatVector.SumOfSquares : Extended;
var I : Integer;
  Begin
    Result := 0.0;
    For I := 0 to Length (FData) - 1 do
      Result := Result + Sqr (FData [I]);
  End;

Procedure TFloatVector.SumAndSquares (var Sum, SumOfSquares : Extended);
var I : Integer;
    D : Extended;
  Begin
    Sum := 0.0;
    SumOfSquares := 0.0;
    For I := 0 to Length (FData) - 1 do
      begin
        D := FData [I];
        Sum := Sum + D;
        SumOfSquares := SumOfSquares + Sqr (D);
      end;
  End;

Function TFloatVector.TotalVariance : Extended;
var Sum, SumSquares : Extended;
  Begin
    SumAndSquares (Sum, SumSquares);
    Result := SumSquares - Sqr (Sum) / Length (FData);
  End;

Function TFloatVector.Variance : Extended;
  Begin
    Result := TotalVariance / (Length (FData) - 1);
  End;

Function TFloatVector.PopulationVariance : Extended;
  Begin
    Result := TotalVariance / Length (FData);
  End;

Function TFloatVector.PopulationStdDev : Extended;
  Begin
    Result := Sqrt (PopulationVariance);
  End;



{                                                                              }
{ TIntegerVector                                                               }
{                                                                              }
class Function TIntegerVector.CreateInstance : AType;
  Begin
    Result := TIntegerVector.Create;
  End;

Function TIntegerVector.IsZero : Boolean;
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      if not FloatZero (FData [I]) then
        begin
          Result := False;
          exit;
        end;
    Result := True;
  End;

{ Mathematical functions                                                       }
Procedure TIntegerVector.Add (const V : TIntegerVector; const Factor : Int64);
var I : Integer;
  Begin
    if Length (V.FData) <> Length (FData) then
      raise EVector.Create ('Vector size mismatch');

    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] + V.FData [I] * Factor;
  End;

Procedure TIntegerVector.Add (const V : TIntegerVector);
  Begin
    Add (V, 1);
  End;

Procedure TIntegerVector.Add (const Value : Int64);
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] + Value;
  End;

Procedure TIntegerVector.Multiply (const V : TIntegerVector);
var I : Integer;
  Begin
    if V.Count <> Length (FData) then
      raise EVector.Create ('Vector size mismatch');

    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] * V.FData [I];
  End;

Procedure TIntegerVector.Multiply (const Value : Int64);
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := FData [I] * Value;
  End;

Procedure TIntegerVector.SquareValues;
var I : Integer;
  Begin
    For I := 0 to Length (FData) - 1 do
      FData [I] := Sqr (FData [I]);
  End;

Function TIntegerVector.Angle (const V : TIntegerVector) : Extended;
  Begin
    Result := ArcCos (DotProduct (V) / (Norm * V.Norm));
  End;

Function TIntegerVector.DotProduct (const V : TIntegerVector) : Int64;
var I : Integer;
  Begin
    if V.Count <> Length (FData) then
      raise EVector.Create ('Vector size mismatch');

    Result := 0;
    For I := 0 to Length (FData) - 1 do
      Result := Result + FData [I] * V.FData [I];
  End;

Function TIntegerVector.Norm : Extended;
var X : Extended;
  Begin
    X := DotProduct (self);
    Result := Sqrt (X);
  End;

{ Statistical functions                                                        }
Function TIntegerVector.Sum (const LoIdx, HiIdx : Integer) : Int64;
  Begin
    Result := cMaths.Sum (FData, LoIdx, HiIdx);
  End;

Function TIntegerVector.Sum : Int64;
  Begin
    Result := cMaths.Sum (FData);
  End;

Function TIntegerVector.WeightedSum (const Weights : TFloatVector) : Extended;
  Begin
    if Weights.Count <> Length (FData) then
      raise EVector.Create ('Vector size mismatch');

    Result := cMaths.WeightedSum (FData, Weights.FData);
  End;

Function TIntegerVector.Product (const LoIdx, HiIdx : Integer) : Int64;
  Begin
    Result := cMaths.Product (FData, LoIdx, HiIdx);
  End;

Function TIntegerVector.Product : Int64;
  Begin
    Result := cMaths.Product (FData);
  End;

Function TIntegerVector.MaxValue : Int64;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise EVector.Create ('No maximum');

    Result := FData [0];
    For I := 1 to Length (FData) - 1 do
      if FData [I] > Result then
        Result := FData [I];
  End;

Function TIntegerVector.MinValue : Int64;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise EVector.Create ('No minimum');

    Result := FData [0];
    For I := 1 to Length (FData) - 1 do
      if FData [I] < Result then
        Result := FData [I];
  End;

Function TIntegerVector.Mean : Extended;
  Begin
    if Length (FData) = 0 then
      raise Exception.Create ('No mean');

    Result := Sum / Length (FData);
  End;

Function TIntegerVector.StdDev (var Mean : Extended) : Extended;
var S    : Extended;
    I, N : Integer;
  Begin
    N := Length (FData);
    if N = 0 then
      raise EDivByZero.Create ('No standard deviation');

    if N = 1 then
      begin
        Mean := FData [0];
        Result := FData [0];
      end else
      begin
        Mean := self.Mean;
        S := 0.0;
        For I := 0 to N - 1 do
          S := S + Sqr (Mean - FData [I]);
        Result := Sqrt (S / (N - 1));
      end;
  End;

Function TIntegerVector.HarmonicMean : Extended;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise Exception.Create ('No harmonic mean');

    Result := 0.0;
    For I := 0 to Length (FData) - 1 do
      if FData [I] < 0.0 then
        raise Exception.Create ('Vector contains negative values') else
        Result := Result + 1.0 / FData [I];
    Result := Length (FData) / Result;
  End;

Function TIntegerVector.GeometricMean : Extended;
var I : Integer;
  Begin
    if Length (FData) = 0 then
      raise Exception.Create ('No geometric mean');

    Result := 1.0;
    For I := 0 to Length (FData) - 1 do
      if FData [I] < 0.0 then
        raise Exception.Create ('Vector contains negative values') else
        Result := Result * FData [I];
    Result := Power (Result, 1.0 / Length (FData));
  End;

Function TIntegerVector.Median : Extended;
var V : TIntegerVector;
  Begin
    if Length (FData) = 0 then
      raise EVector.Create ('No median');

    V := TIntegerVector (Duplicate);
    try
      V.Sort;
      Result := V.FData [(Length (V.FData) - 1) div 2];
    finally
      V.Free;
    end;
  End;

Function TIntegerVector.SumOfSquares : Int64;
var I : Integer;
  Begin
    Result := 0;
    For I := 0 to Length (FData) - 1 do
      Result := Result + Sqr (FData [I]);
  End;

Procedure TIntegerVector.SumAndSquares (var Sum, SumOfSquares : Int64);
var I : Integer;
    D : Int64;
  Begin
    Sum := 0;
    SumOfSquares := 0;
    For I := 0 to Length (FData) - 1 do
      begin
        D := FData [I];
        Sum := Sum + D;
        SumOfSquares := SumOfSquares + Sqr (D);
      end;
  End;

Function TIntegerVector.TotalVariance : Extended;
var Sum, SumSquares : Int64;
  Begin
    SumAndSquares (Sum, SumSquares);
    Result := SumSquares - Sqr (Sum) / Length (FData);
  End;

Function TIntegerVector.Variance : Extended;
  Begin
    Result := TotalVariance / (Length (FData) - 1);
  End;

Function TIntegerVector.PopulationVariance : Extended;
  Begin
    Result := TotalVariance / Length (FData);
  End;

Function TIntegerVector.PopulationStdDev : Extended;
  Begin
    Result := Sqrt (PopulationVariance);
  End;



{                                                                              }
{ TStatistic                                                                   }
{                                                                              }
Procedure TStatistic.Clear;
  Begin
    FCount := 0;
    FSum := 0.0;
    FSumOfSquares := 0.0;
  End;

Function TStatistic.GetMin : Extended;
  Begin
    if FCount = 0 then
      raise EStatistic.Create ('No minimum');

    Result := FMin;
  End;

Function TStatistic.GetMax : Extended;
  Begin
    if FCount = 0 then
      raise EStatistic.Create ('No maximum');

    Result := FMax;
  End;

Function TStatistic.Range : Extended;
  Begin
    if FCount = 0 then
      raise EStatistic.Create ('No range');

    Result := FMax - FMin;
  End;

Function TStatistic.Mean : Extended;
  Begin
    if FCount = 0 then
      raise EDivByZero.Create ('No mean');

    Result := FSum / FCount;
  End;

Function TStatistic.TotalVariance : Extended;
  Begin
    if FCount = 0 then
      raise EDivByZero.Create ('No variance');

    Result := FSumOfSquares - Sqr (FSum) / FCount;
  End;

Function TStatistic.Variance : Extended;
  Begin
    Result := TotalVariance / (FCount - 1);
  End;

Function TStatistic.StdDev : Extended;
  Begin
    Result := Sqrt (Variance);
  End;

Function TStatistic.PopulationVariance : Extended;
  Begin
    Result := TotalVariance / FCount;
  End;

Function TStatistic.PopulationStdDev : Extended;
  Begin
    Result := Sqrt (PopulationVariance);
  End;

Procedure TStatistic.Add (const V : Extended);
  Begin
    Inc (FCount);
    FSum := FSum + V;
    FSumOfSquares := FSumOfSquares + Sqr (V);
    if FCount = 1 then
      begin
        FMin := V;
        FMax := V;
      end else
      begin
        FMin := Math.Min (FMin, V);
        FMax := Math.Max (FMax, V);
      end;
    FLast := V;
  End;

Procedure TStatistic.Add (const V : ExtendedArray);
var I : Integer;
  Begin
    For I := 0 to Length (V) - 1 do
      Add (V [I]);
  End;

Procedure TStatistic.Add (const V : Array of Extended);
var I : Integer;
  Begin
    For I := 0 to High (V) - 1 do
      Add (V [I]);
  End;

Procedure TStatistic.Add (const V : TFloatVector);
var I : Integer;
  Begin
    For I := 0 to V.Count - 1 do
      Add (V.FData [I]);
  End;

Function TStatistic.GetAsString : String;
  Begin
    if Count = 0 then
      Result := 'No statistics' else
      Result := 'Count: ' + IntToStr (Count) + '  Sum: ' + FloatToStr (Sum) + '  Sum of Squares: ' + FloatToStr (SumOfSquares) + #13 +
                'Min: ' + FloatToStr (Min) + '  Max: ' + FloatToStr (Max) + '  Range: ' + FloatToStr (Range) + #13 +
                'Mean: ' + FloatToStr (Mean) + '  Std Dev: ' + FloatToStr (StdDev) + '  Variance: ' + FloatToStr (Variance);
  End;



{                                                                              }
{ TMatrix                                                                      }
{                                                                              }
Constructor TMatrix.CreateSquare (const N : Integer);
  Begin
    inherited Create;
    SetSize (N, N);
  End;

Constructor TMatrix.CreateIdentity (const N : Integer);
var I : Integer;
  Begin
    inherited Create;
    SetSize (N, N);
    For I := 0 to N - 1 do
      FRows [I, I] := 1.0;
  End;

Constructor TMatrix.CreateDiagonal (const D : TFloatVector);
var I, N : Integer;
  Begin
    inherited Create;
    N := Length (D.FData);
    SetSize (N, N);
    For I := 0 to N - 1 do
      FRows [I, I] := D.FData [I];
  End;

Procedure TMatrix.SetRowCount (const NewRowCount : Integer);
var I, OldRowCount : Integer;
  Begin
    OldRowCount := Length (FRows);
    if OldRowCount = NewRowCount then
      exit;

    SetLength (FRows, NewRowCount);
    if OldRowCount < NewRowCount then
      For I := OldRowCount to NewRowCount - 1 do
        SetLength (FRows [I], FColCount);
  End;

Function TMatrix.GetRowCount : Integer;
  Begin
    Result := Length (FRows);
  End;

Procedure TMatrix.SetColCount (const NewColCount : Integer);
var I : Integer;
  Begin
    if FColCount = NewColCount then
      exit;

    For I := 0 to Length (FRows) - 1 do
      SetLength (FRows [I], NewColCount);
    FColCount := NewColCount;
  End;

Procedure TMatrix.SetSize (const Rows, Cols : Integer);
  Begin
    SetLength (FRows, 0);
    SetColCount (Cols);
    SetRowCount (Rows);
  End;

Procedure TMatrix.SetItem (const Row, Col : Integer; const Value : Extended);
  Begin
    FRows [Row, Col] := Value;
  End;

Function TMatrix.GetItem (const Row, Col : Integer) : Extended;
  Begin
    Result := FRows [Row, Col];
  End;

Function TMatrix.GetRow (const Row : Integer) : TFloatVector;
  Begin
    Result := TFloatVector.Create;
    Result.FData := FRows [Row]; // reference
  End;

Function TMatrix.GetAsString : String;
var I, J : Integer;
  Begin
    Result := '';
    For I := 0 to Length (FRows) - 1 do
      begin
        Result := Result + '(';
        For J := 0 to FColCount - 1 do
          Result := Result + FloatToStr (GetItem (I, J)) + Cond (J = FColCount - 1, '', ',');
        Result := Result + ')' + Cond (I = Length (FRows) - 1, '', ',');
      end;
  End;

Procedure TMatrix.Assign (const M : TMatrix);
var I : Integer;
  Begin
    SetSize (M.RowCount, M.ColCount);
    For I := 0 to Length (FRows) - 1 do
      FRows [I] := Copy (M.FRows [I]);
  End;

Procedure TMatrix.Assign (const Value : Extended);
var I, J : Integer;
  Begin
    For I := 0 to Length (FRows) - 1 do
      begin
        if Length (FRows [I]) <> FColCount then
          SetLength (FRows [I], FColCount);
        For J := 0 to FColCount - 1 do
          FRows [I, J] := Value;
      end;
  End;

Procedure TMatrix.AssignRowValues (const Row : Integer; const Values : Array of Extended);
var I : Integer;
  Begin
    For I := 0 to High (Values) do
      FRows [Row, I] := Values [I];
  End;

Procedure TMatrix.AssignRow (const Row : Integer; const V : TFloatVector);
  Begin
    SetColCount (Length (V.FData));
    FRows [Row] := Copy (V.FData);
  End;

Procedure TMatrix.AssignCol (const Col : Integer; const V : TFloatVector);
var I : Integer;
  Begin
    SetRowCount (Length (V.FData));
    For I := 0 to Length (FRows) - 1 do
      SetItem (I, Col, V.FData [I]);
  End;

Procedure TMatrix.Assign (const V : TFloatVector);
  Begin
    SetSize (1, Length (V.FData));
    FRows [0] := Copy (V.FData);
  End;

Function TMatrix.Duplicate : TMatrix;
  Begin
    Result := TMatrix.Create;
    Result.Assign (self);
  End;

Function TMatrix.Duplicate (const R1, C1, R2, C2 : Integer) : TMatrix;
var I : Integer;
    _R1, _C1, _R2, _C2 : Integer;
  Begin
    Result := TMatrix.Create;

    _R1 := Max (R1, 0);
    _R2 := Min (R2, Length (FRows));
    _C1 := Max (C1, 0);
    _C2 := Min (C2, FColCount);

    if (_R1 > _R2) or (_C1 > _C2) then
      exit;

    Result.SetSize (R2 - R1 + 1, C2 - C1 + 1);
    For I := R1 to R2 do
      Result.FRows [I - R1] := Copy (FRows [I], C1, C2 - C1 + 1);
  End;

Function TMatrix.DuplicateRow (const Row : Integer) : TFloatVector;
  Begin
    Result := TFloatVector.Create;
    try
      Result.FData := Copy (FRows [Row]);
      SetLength (Result.FData, FColCount);
    except
      Result.Free;
      raise;
    end;
  End;

Function TMatrix.DuplicateCol (const Col : Integer) : TFloatVector;
var I : Integer;
  Begin
    if (Col > FColCount - 1) or (Col < 0) then
      raise EMatrix.Create ('Column index out of range');

    Result := TFloatVector.Create;
    SetLength (Result.FData, Length (FRows));
    For I := 0 to Length (FRows) - 1 do
      if Length (FRows [I]) >= Col then
        Result.FData [I] := FRows [I, Col];
  End;

Function TMatrix.DuplicateDiagonal : TFloatVector;
var I : Integer;
  Begin
    if not IsSquare then
      raise EMatrix.Create ('Not square');

    Result := TFloatVector.Create;
    SetLength (Result.FData, Length (FRows));
    For I := 0 to Length (FRows) - 1 do
      if Length (FRows [I]) >= I then
        Result.FData [I] := FRows [I, I];
  End;

Function TMatrix.IsSquare : Boolean;
  Begin
    Result := Length (FRows) = FColCount;
  End;

Function TMatrix.IsZero : Boolean;
var I, J : Integer;
  Begin
    For I := 0 to Length (FRows) - 1 do
      For J := 0 to Length (FRows [I]) do
        if not FloatZero (FRows [I, J]) then
          begin
            Result := False;
            exit;
          end;
    Result := True;
  End;

Function TMatrix.IsIdentity : Boolean;
var I, J : Integer;
    R    : Extended;
  Begin
    if not IsSquare then
      begin
        Result := False;
        exit;
      end;

    For I := 0 to Length (FRows) - 1 do
      For J := 0 to Length (FRows [I]) - 1 do
        begin
          R := FRows [I, J];
          if ((J = I) and not FloatEqual (R, 1.0)) or
             ((J <> I) and not FloatZero (R)) then
            begin
              Result := False;
              exit;
            end;
        end;
    Result := True;
  End;

Function TMatrix.IsEqual (const M : TMatrix) : Boolean;
var I, J   : Integer;
  Begin
    if (Length (FRows) <> Length (M.FRows)) or (FColCount <> M.FColCount) then
      begin
        Result := False;
        exit;
      end;

    For I := 0 to Length (FRows) - 1 do
      For J := 0 to FColCount - 1 do
        if not FloatEqual (FRows [I, J], M.FRows [I, J]) then
          begin
            Result := False;
            exit;
          end;
    Result := True;
  End;

Function TMatrix.IsEqual (const V : TFloatVector) : Boolean;
var I : Integer;
  Begin
    if (Length (FRows) = 1) and (Length (V.FData) = FColCount) then
      begin
        For I := 0 to FColCount - 1 do
          if not FloatEqual (V.FData [I], FRows [0, I]) then
            begin
              Result := False;
              exit;
            end;
        Result := True;
      end else
      Result := False;
  End;

Procedure TMatrix.SwapRows (const I, J : Integer);
var P : ExtendedArray;
  Begin
    if Max (I, J) > Length (FRows) - 1 then
      raise EMatrix.Create ('Row index out of range');

    P := FRows [I];
    FRows [I] := FRows [J];
    FRows [J] := P;
  End;

Procedure TMatrix.AddRows (const I, J : Integer; const Factor : Extended);
var F : Integer;
  Begin
    if Max (I, J) > Length (FRows) - 1 then
      raise EMatrix.Create ('Row index out of range');

    For F := 0 to Min (Length (FRows [J]), FColCount) - 1 do
      FRows [I, F] := FRows [I, F] + FRows [J, F] * Factor;
  End;

Function TMatrix.Transposed : TMatrix;
var I, J : Integer;
  Begin
    Result := TMatrix.Create;
    Result.SetSize (FColCount, Length (FRows) - 1);
    For I := 0 to Length (Result.FRows) - 1 do
      For J := 0 to Result.FColCount - 1 do
        if I <= Length (FRows [J]) - 1 then
          Result.FRows [I, J] := FRows [J, I];
  End;

Procedure TMatrix.Transpose;
var M : TMatrix;
  Begin
    M := Transposed;
    try
      Assign (M);
    finally
      M.Free;
    end;
  End;

Procedure TMatrix.Add (const M : TMatrix);
var R, I, J : Integer;
  Begin
    R := RowCount;
    if (M.RowCount <> R) or (M.ColCount <> ColCount) then
      raise EMatrix.Create ('Matrix size mismatch');

    For I := 0 to R - 1 do
      For J := 0 to FColCount - 1 do
        FRows [I, J] := FRows [I, J] + M.FRows [I, J];
  End;

Function TMatrix.Multiplied (const M : TMatrix) : TMatrix;
var I, J, K : Integer;
    R       : Extended;
  Begin
    if ColCount <> M.RowCount then
      raise EMatrix.Create ('Matrix size mismatch');

    Result := TMatrix.Create;
    Result.SetSize (RowCount, M.ColCount);
    For I := 0 to Result.RowCount - 1 do
      For J := 0 to Result.ColCount - 1 do
        begin
          R := 0.0;
          For K := 0 to ColCount - 1 do
            R := R + FRows [I, K] * M.FRows [K, J];
          Result.FRows [I, J] := R;
        end;
  End;

Procedure TMatrix.Multiply (const M : TMatrix);
var NM : TMatrix;
  Begin
    NM := Multiplied (M);
    try
      Assign (NM);
    finally
      NM.Free;
    end;
  End;

Procedure TMatrix.Multiply (const Row : Integer; const Value : Extended);
var I : Integer;
  Begin
    For I := 0 to FColCount - 1 do
      FRows [Row, I] := FRows [Row, I] * Value
  End;

Procedure TMatrix.Multiply (const Value : Extended);
var I : Integer;
  Begin
    For I := 0 to RowCount - 1 do
      Multiply (I, Value);
  End;

Function TMatrix.Trace : Extended;
var I : Integer;
  Begin
    if not IsSquare then
      raise EMatrix.Create ('Matrix not square');
    Result := 0.0;
    For I := 0 to RowCount - 1 do
      Result := Result + FRows [I, I];
  End;

Function TMatrix.IsOrtogonal : Boolean;
var M : TMatrix;
  Begin
    M := Duplicate;
    try
      M.Transpose;
      M.Multiply (self);
      Result := M.IsIdentity;
    finally
      M.Free;
    end;
  End;

Function TMatrix.IsIdempotent : Boolean;
var M : TMatrix;
  Begin
    M := Duplicate;
    try
      M.Multiply (M);
      Result := M.IsEqual (M);
    finally
      M.Free;
    end;
  End;

Function TMatrix.Normalise (const M : TMatrix = nil) : Extended;
var I : Integer;
    R : Extended;
  Begin
    Result := 1.0;
    For I := 0 to RowCount - 1 do
      begin
        R := GetItem (I, I);
        Result := Result * R;
        if not FloatZero (R) then
          begin
            R := 1.0 / R;
            Multiply (I, R);
            if Assigned (M) then
              M.Multiply (I, R);
          end;
      end;
  End;

Function TMatrix.SolveMatrix (var M : TMatrix) : Extended;
var I, J   : Integer;
    R      : Extended;
  Begin
    Result := 1.0;
    For I := 0 to RowCount - 1 do
      begin
        J := 0;
        While J < RowCount do
          if not FloatZero (GetItem (I, J))  then
            break else
            Inc (J);
        if J = RowCount then
          begin
            Result := 0.0;
            exit;
          end;
        SwapRows (I, J);
        M.SwapRows (I, J);
        if Odd (M.ColCount) then
          Result := -Result;

        For J := I + 1 to RowCount - 1 do
          begin
            R := -(GetItem (J, I) / GetItem (I, I));
            AddRows (J, I, R);
            M.AddRows (J, I, R);
          end;
      end;

    For I := RowCount - 1 downto 0 do
      For J := I - 1 downto 0 do
        begin
          R := -(GetItem (J, I) / GetItem (I, I));
          AddRows (J, I, R);
          M.AddRows (J, I, R);
        end;

    Result := Normalise (M);
  End;

Function TMatrix.Determinant : Extended;
var A, B : TMatrix;
  Begin
    if not IsSquare then
      raise EMatrix.Create ('No determinant');

    A := Duplicate; try
    B := TMatrix.CreateIdentity (RowCount);
    try
      Result := A.SolveMatrix (B);
    finally B.Free; end;
    finally A.Free; end;
  End;

Procedure TMatrix.Inverse;
var A : TMatrix;
  Begin
    if not IsSquare then
      raise EMatrix.Create ('No inverse');

    A := TMatrix.CreateIdentity (RowCount);
    try
      if FloatZero (SolveMatrix (A)) then
        raise EMatrix.Create ('Can not invert');
      Assign (A);
    finally
      A.Free;
    end;
  End;

Function TMatrix.SolveLinearSystem (const V : TFloatVector) : TFloatVector;
var C, M : TMatrix;
  Begin
    if not IsSquare or (V.Count <> RowCount) then
      raise EMatrix.Create ('Not a linear system');
    C := Duplicate; try
    M := TMatrix.Create;
    try
      M.Assign (V);
      if FloatZero (C.SolveMatrix (M)) then
        raise EMatrix.Create ('Can not solve this system');
      Result := TFloatVector.Create;
      Result.Assign (M.GetRow (0));
    finally M.Free; end;
    finally C.Free; end;
  End;



{                                                                              }
{ 3D Transformation matrices                                                   }
{                                                                              }
Function OriginAndScaleTransform (const TX, TY, TZ, SX, SY, SZ : Extended) : TMatrix;
  Begin
    Result := TMatrix.CreateSquare (4);
    Result.AssignRowValues (0, [ SX, 0.0, 0.0, -TX]);
    Result.AssignRowValues (1, [0.0,  SY, 0.0, -TY]);
    Result.AssignRowValues (2, [0.0, 0.0,  SZ, -TZ]);
    Result.AssignRowValues (3, [0.0, 0.0, 0.0, 1.0]);
  End;

Function XRotateTransform (const Angle : Extended) : TMatrix;
var S, C : Extended;
  Begin
    SinCos (Angle, S, C);
    Result := TMatrix.CreateSquare (4);
    Result.AssignRowValues (0, [1.0, 0.0, 0.0, 0.0]);
    Result.AssignRowValues (1, [0.0, C  , -S , 0.0]);
    Result.AssignRowValues (2, [0.0, S  , C  , 0.0]);
    Result.AssignRowValues (3, [0.0, 0.0, 0.0, 1.0]);
  End;

Function YRotateTransform (const Angle : Extended) : TMatrix;
var S, C : Extended;
  Begin
    SinCos (Angle, S, C);
    Result := TMatrix.CreateSquare (4);
    Result.AssignRowValues (0, [C  , 0.0, -S , 0.0]);
    Result.AssignRowValues (1, [0.0, 1.0, 0.0, 0.0]);
    Result.AssignRowValues (2, [S  , 0.0, C  , 0.0]);
    Result.AssignRowValues (3, [0.0, 0.0, 0.0, 1.0]);
  End;

Function ZRotateTransform (const Angle : Extended) : TMatrix;
var S, C : Extended;
  Begin
    SinCos (Angle, S, C);
    Result := TMatrix.CreateSquare (4);
    Result.AssignRowValues (0, [C  , -S , 0.0, 0.0]);
    Result.AssignRowValues (1, [S  , C  , 0.0, 0.0]);
    Result.AssignRowValues (2, [0.0, 0.0, 1.0, 0.0]);
    Result.AssignRowValues (3, [0.0, 0.0, 0.0, 1.0]);
  End;

Function XYZRotateTransform (const XAngle, YAngle, ZAngle : Extended) : TMatrix;
var SX, CX, SY, CY, SZ, CZ : Extended;
    SXSY, CXSY : Extended;
  Begin
    Result := TMatrix.CreateSquare (4);
    SinCos (XAngle, SX, CX);
    SinCos (YAngle, SY, CY);
    SinCos (ZAngle, SZ, CZ);
    SXSY := SX * SY;
    CXSY := CX * SY;
    Result.AssignRowValues (0, [        CY*CZ,         CY*SZ,   -SY, 0.0]);
    Result.AssignRowValues (1, [SXSY*CZ-CX*SZ, SXSY*SZ+CX*CZ, SX*CY, 0.0]);
    Result.AssignRowValues (2, [CXSY*CZ+SX*SZ, CXSY*SZ-SX*CZ, CX*CY, 0.0]);
    Result.AssignRowValues (3, [          0.0,           0.0,   0.0, 1.0]);
  End;



{                                                                              }
{ T3DPoint                                                                     }
{                                                                              }
Constructor T3DPoint.CreateVector (const X, Y, Z : Extended);
  Begin
    inherited Create;
    Assign ([X, Y, Z, 0.0]);
  End;

Constructor T3DPoint.CreatePoint (const X, Y, Z : Extended);
  Begin
    inherited Create;
    Assign ([X, Y, Z, 1.0]);
  End;

Function T3DPoint.Duplicate : T3DPoint;
  Begin
    Result := T3DPoint.Create;
    Assign ([FData [0], FData [1], FData [2], FData [3]]);
  End;

Procedure T3DPoint.SetX (const NewX : Extended);
  Begin
    FData [0] := NewX;
  End;

Procedure T3DPoint.SetY (const NewY : Extended);
  Begin
    FData [1] := NewY;
  End;

Procedure T3DPoint.SetZ (const NewZ : Extended);
  Begin
    FData [2] := NewZ;
  End;

Function T3DPoint.GetX : Extended;
  Begin
    Result := FData [0];
  End;

Function T3DPoint.GetY : Extended;
  Begin
    Result := FData [1];
  End;

Function T3DPoint.GetZ : Extended;
  Begin
    Result := FData [2];
  End;

Procedure T3DPoint.RotateX (const Angle : Extended);
var S, C : Extended;
    Y, Z : Extended;
  Begin
    SinCos (Angle, S, C);
    Y := FData [1];
    Z := FData [2];
    FData [1] := C * Y - S * Z;
    FData [2] := S * Y + C * Z;
  End;

Procedure T3DPoint.RotateY (const Angle : Extended);
var S, C : Extended;
    X, Z : Extended;
  Begin
    SinCos (Angle, S, C);
    X := FData [0];
    Z := FData [2];
    FData [0] := C * X - S * Z;
    FData [2] := S * X + C * Z;
  End;

Procedure T3DPoint.RotateZ (const Angle : Extended);
var S, C : Extended;
    X, Y : Extended;
  Begin
    SinCos (Angle, S, C);
    X := FData [0];
    Y := FData [1];
    FData [0] := C * X - S * Y;
    FData [1] := S * X + C * Y;
  End;

Procedure T3DPoint.RotateXYZ (const XAngle, YAngle, ZAngle : Extended);
var SX, CX, SY, CY, SZ, CZ : Extended;
    F1, F2, YCX, ZSX,
    X, Y, Z                : Extended;
  Begin
    X := FData [0];
    Y := FData [1];
    Z := FData [2];
    SinCos (XAngle, SX, CX);
    SinCos (YAngle, SY, CY);
    SinCos (ZAngle, SZ, CZ);
    F2 := Y * SX + Z * CX;
    F1 := X * CY + SY * F2;
    YCX := Y * CX;
    ZSX := Z * SX;
    FData [0] := CZ * F1 + SZ * (ZSX - YCX);
    FData [1] := SZ * F1 + CZ * (YCX - ZSX);
    FData [2] := CY * F2 - X * SY;
  End;

Procedure T3DPoint.RotateVector (const NX, NY, NZ, Angle : Extended);
var S, C : Extended;
    X, Y, Z,
    F1 : Extended;
  Begin
    X := FData [0];
    Y := FData [1];
    Z := FData [2];
    SinCos (Angle, S, C);
    F1 := (1.0 - C) * (X * NX + Y * Y * NY + Z * Z * NZ);

    FData [0] := NX * F1 + C * X + S * (Y * NZ - Z * NY);
    FData [1] := NY * F1 + C * Y + S * (Z * NX - X * NZ);
    FData [2] := NZ * F1 + C * Z + S * (X * NY - Y * NX);
  End;

Procedure T3DPoint.Homogenize;
var W : Extended;
  Begin
    W := FData [3];
    if W = 0.0 then
      raise E3DPoint.Create ('Not a point');
    FData [0] := FData [0] / W;
    FData [1] := FData [1] / W;
    FData [2] := FData [2] / W;
    FData [3] := 1.0;
  End;

Procedure T3DPoint.CrossProduct (const P : T3DPoint);
var X, Y, Z,
    BX, BY, BZ : Extended;
  Begin
    X := FData [0];
    Y := FData [1];
    Z := FData [2];
    BX := P.FData [0];
    BY := P.FData [1];
    BZ := P.FData [2];
    FData [0] := Y * BZ - Z * BY;
    FData [1] := Z * BX - X * BZ;
    FData [2] := X * BY - Y * BX;
  End;

Procedure T3DPoint.Scale (const XScale, YScale, ZScale : Single);
  Begin
    FData [0] := FData [0] * XScale;
    FData [1] := FData [1] * YScale;
    FData [2] := FData [2] * ZScale;
  End;

Procedure T3DPoint.Origin (const XOrigin, YOrigin, ZOrigin : Extended);
  Begin
    FData [0] := FData [0] + XOrigin;
    FData [1] := FData [1] + YOrigin;
    FData [2] := FData [2] + ZOrigin;
  End;

Procedure CavalierProjection (const Angle, X1, Y1, Z1 : Extended; var X, Y : Extended);
var S, C : Extended;
  Begin
    SinCos (Angle * RadPerDeg, S, C);
    X := X1 + Z1 * C;
    Y := Y1 + Z1 * S;
  End;

Procedure T3DPoint.CavalierProject (const Angle : Extended; var X, Y : Extended);
  Begin
    CavalierProjection (Angle, FData [0], FData [1], FData [2], X, Y);
  End;

Procedure T3DPoint.CabinetProject (const Angle : Extended; var X, Y : Extended);
  Begin
    CavalierProjection (Angle, FData [0], FData [1], 0.5 * FData [2], X, Y);
  End;

Function ClipPerspectiveProjection (const P, V : Extended) : Extended;
  Begin
    if V > 0 then
      Result := Min (P, V) else
      Result := Max (P, V);
  End;

Procedure T3DPoint.OnePointPerspectiveProject (const Angle, Zv : Extended; var X, Y : Extended);
var Z, ZF : Extended;
  Begin
    Z := ClipPerspectiveProjection (FData [2], Zv);
    ZF := (Zv - Z) / Zv;
    CavalierProjection (Angle, FData [0] * ZF, FData [1] * ZF, Z, X, Y);
  End;

Procedure T3DPoint.TwoPointPerspectiveProject (const Angle, Xv, Zv : Extended; var X, Y : Extended);
var XP, ZP, ZF, XF : Extended;
  Begin
    XP := ClipPerspectiveProjection (FData [0], Xv);
    ZP := ClipPerspectiveProjection (FData [2], Zv);
    XF := (Xv - XP) / Xv;
    ZF := (Zv - ZP) / Zv;
    CavalierProjection (Angle, XP * ZF, FData [1] * XF * ZF, ZP, X, Y);
  End;

{                                                                              }
{ Numerical solvers                                                            }
{                                                                              }
Function SecantSolver (const f : fx; const y, Guess1, Guess2 : Extended) : Extended;
var xn, xnm1, xnp1, fxn, fxnm1 : Extended;
  Begin
    xnm1 := Guess1;
    xn := Guess2;
    fxnm1 := f (xnm1) - y;
    Repeat
      fxn := f (xn) - y;
      xnp1 := xn - fxn * (xn - xnm1) / (fxn - fxnm1);
      fxnm1 := fxn;
      xnm1 := xn;
      xn := xnp1;
    Until (f (xn - 0.00000001) - y) * (f (xn + 0.00000001) - y) <= 0.0;
    Result := xn;
  End;

Function NewtonSolver (const f, df : fx; const y, Guess : Extended) : Extended;
var xn, xnp1 : Extended;
  Begin
    xnp1 := Guess;
    Repeat
      xn := xnp1;
      xnp1 := xn - f (xn) / df (xn);
    Until Abs (xnp1 - xn) < 0.000000000000001;
    Result := xn;
  End;

const h = 1e-15;

Function FirstDerivative (const f : fx; const x : Extended) : Extended;
  Begin
    Result := (-f (x + 2 * h) + 8 * f (x + h) - 8 * f (x - h) + f (x - 2 * h)) / (12 * h);
  End;

Function SecondDerivative (const f : fx; const x : Extended) : Extended;
  Begin
    Result := (-f (x + 2 * h) + 16 * f (x + h) - 30 * f (x) + 16 * f (x - h) - f (x - 2 * h)) / (12 * h * h);
  End;

Function ThirdDerivative (const f : fx; const x : Extended) : Extended;
  Begin
    Result := (f (x + 2 * h) - 2 * f (x + h) + 2 * f (x - h) - f (x - 2 * h)) / (2 * h * h * h);
  End;

Function FourthDerivative (const f : fx; const x : Extended) : Extended;
  Begin
    Result := (f (x + 2 * h) - 4 * f (x + h) + 6 * f (x) - 4 * f (x - h) + f (x - 2 * h)) / (h * h * h * h);
  End;

Function SimpsonIntegration (const f : fx; const a, b : Extended; N : Integer) : Extended;
var h : Extended;
    I : Integer;
  Begin
    if N mod 2 = 1 then
     Inc (N); // N must be multiple of 2

    h := (b - a) / N;

    Result := 0.0;
    For I := 1 to N - 1 do
      Result := Result + ((I mod 2) * 2 + 2) * f (a + (I - 0.5) * h);
    Result := (Result + f (a) + f (b)) * h / 3.0;
  End;





{                                                                              }
{ Annuity functions                                                            }
{                                                                              }
Function an (const i, n : Extended) : Extended;
  Begin
    if n = 0.0 then
      Result := 0.0 else
    if i = 0.0 then
      Result := n else
    if Abs (i) < 1e-9 then
      Result := n / (1.0 + n * i) else
      Result := (1.0 - Power (1.0 + i, -n)) / i;
  End;

Function aDOTn (const i, n : Extended) : Extended;
  Begin
    if n = 0.0 then
      Result := 0.0 else
    if i = 0.0 then
      Result := n else
    if Abs (i) < 1e-9 then
      Result := n * (1.0 + i) / (1.0 + n * i) else
      Result := (1.0 + i - Power (1.0 + i, -n + 1.0)) / i;
  End;

Function sn (const i, n : Extended) : Extended;
  Begin
    if n = 0.0 then
      Result := 0.0 else
    if i = 0.0 then
      Result := n else
    if Abs (i) < 1e-9 then
      Result := n * (1.0 + (n - 1.0) * i / 2.0) else
      Result := (Power (1.0 + i, n) - 1.0) / i;
  End;

Function sDOTn (const i, n : Extended) : Extended;
  Begin
    if n = 0.0 then
      Result := 0.0 else
    if i = 0.0 then
      Result := n else
    if Abs (i) < 1e-9 then
      Result := sn (i, n) * (1.0 + i) else
      Result := (Power (1.0 + i, n + 1.0) - 1.0 - i) / i;
  End;

Function aCONTn (const i, n : Extended) : Extended;
  Begin
    aCONTn := (1.0 - Power (1.0 + i, -n)) / Ln (1.0 + i);
  End;

Function Ian (const i, n : Extended) : Extended;
var P : Extended;
  Begin
    P := Power (1.0 + i, n);
    Ian := (P * (1.0 + i) - 1.0 - i * (1.0 + n)) / (P * i * i);
  End;

Function IaDOTn (const i, n : Extended) : Extended;
  Begin
    IaDOTn := Ian (i, n) * (1.0 + i);
  End;

Function IaCONTn (const i, n : Extended) : Extended;
  Begin
    IaCONTn := (aDOTn (i, n) - n * Power (1.0 + i, -n)) / Ln (1.0 + i);
  End;

Function ICONTaCONTn (const i, n : Extended) : Extended;
  Begin
    ICONTaCONTn := (aCONTn (i, n) - n * Power (1.0 + i, -n)) / Ln (1.0 + i);
  End;

Function aDOTpn (const i, p, n : Extended) : Extended;
  Begin
    aDOTpn := (1 - Power (1.0 + i, -n)) / dp (i, p);
  End;

Function spn (const i, p, n : Extended) : Extended;
  Begin
    spn := (Power (1.0 + i, n) - 1.0) / ip (i, p);
  End;

Function sDOTpn (const i, p, n : Extended) : Extended;
  Begin
    sDOTpn := (Power (1.0 + i, n) - 1.0) / dp (i, p);
  End;

Function ForceAsI (const d : Extended) : Extended;
  Begin
    ForceAsI := Exp (d) - 1.0;
  End;

Function DiscountAsI (const d : Extended) : Extended;
  Begin
    DiscountAsI := 1.0 / (1.0 - d) - 1.0;
  End;

Function VAsI (const v : Extended) : Extended;
  Begin
    VAsI := 1.0 / v - 1.0;
  End;

Function IAsDiscount (const i : Extended) : Extended;
  Begin
    IAsDiscount := i / (1.0 + i);
  End;

Function IAsForce (const i : Extended) : Extended;
  Begin
    IAsForce := Ln (1.0 + i);
  End;

Function IAsV (const i : Extended) : Extended;
  Begin
    IAsV := 1.0 / (1.0 + i);
  End;

Function V (const i, n : Extended) : Extended;
  Begin
    V := Power (IAsV (i), n);
  End;

Function ip (const i, p : Extended) : Extended;
  Begin
    ip := p * (Power (1.0 + i, 1.0 / p) - 1.0);
  End;

Function dp (const i, p : Extended) : Extended;
  Begin
    dp := p * (1.0 - Power (1.0 / (1.0 + i), 1.0 / p));
  End;

Function apn (const i, p, n : Extended) : Extended;
  Begin
    apn := (1 - Power (1.0 + i, -n)) / ip (i, p);
  End;





{                                                                              }
{ TLifeTable                                                                   }
{                                                                              }
Function TLifeTable.l (const x : Integer) : Extended;
  Begin
    Result := GetItem (x);
  End;

Function TLifeTable.d (const x, n : Integer) : Extended;
  Begin
    Result := l(x) - l(x+n);
  End;

Function TLifeTable.p (const x, n : Integer) : Extended;
  Begin
    Result := l(x+n) / l(x);
  End;

Function TLifeTable.q (const x, n, m : Integer) : Extended;
  Begin
    Result := (l(x+m) - l(x+m+n)) / l(x);
  End;

Function TLifeTable.Dx (const i : Extended; const x : Integer) : Extended;
  Begin
    Result := V (i, x) * l (x);
  End;

Function TLifeTable.Nx (const i : Extended; const x : Integer) : Extended;
var F : Integer;
  Begin
    Result := 0.0;
    For F := x to Count - 1 do
      Result := Result + Dx (i, F);
  End;

Function TLifeTable.Sx (const i : Extended; const x : Integer) : Extended;
var F : Integer;
  Begin
    Result := 0.0;
    For F := x to Count - 1 do
      Result := Result + Nx (i, F);
  End;

Function TLifeTable.Cx (const i : Extended; const x : Integer) : Extended;
  Begin
    Result := V (i, x + 1) * d (x, 1);
  End;

Function TLifeTable.Mx (const i : Extended; const x : Integer) : Extended;
var F : Integer;
  Begin
    Result := 0.0;
    For F := x to Count - 1 do
      Result := Result + Cx (i, F);
  End;

Function TLifeTable.Rx (const i : Extended; const x : Integer) : Extended;
var F : Integer;
  Begin
    Result := 0.0;
    For F := x to Count - 1 do
      Result := Result + Mx (i, F);
  End;



{                                                                              }
{ Computer maths                                                               }
{                                                                              }

{ Base conversion functions -------------------------------------------------- }
const ConversionAlphabeth : String [36] = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';

Function DecodeBase (const S : String; const Base : Byte) : Int64;
var Tot, L : Int64;
    P, F : Byte;
  Begin
    if Base > 36 then
      raise EInvalidArgument.Create ('Invalid base');
    L := 1;
    P := Length (S);
    Tot := 0;
    Repeat
      F := Pos (UpCase (S [P]), ConversionAlphabeth);
      if (F = 0) or (F > Base) then
        raise EInvalidArgument.Create ('Invalid character ''' + S [P] + '''');
      Tot := Tot + L * (F - 1);
      L := L * Base;
      Dec (P);
    Until P = 0;
    DecodeBase := Tot;
  End;

Function BinToInt (const S : String) : Int64;
  Begin
    Result := DecodeBase (S, 2);
  End;

Function OctToInt (const S : String) : Int64;
  Begin
    Result := DecodeBase (S, 8);
  End;

Function HexToInt (const S : String) : Int64;
  Begin
    Result := DecodeBase (S, 16);
  End;

Function EncodeBase (const I : Int64; const Base : Byte) : String;
var D, J : Int64;
    N    : Byte;
  Begin
    if I = 0 then
      begin
        Result := '0';
        exit;
      end;
    D := Round (Power (Base, Trunc (Log10 (I) / Log10 (Base)) + 1));            // +1 to fix occasional real "fuzz"
    J := I;
    Result := '';
    While D > 0 do
      begin
        N := J div D;
        if (N <> 0) or (Result <> '') then                                      // "fuzz" bug
          Result := Result + ConversionAlphabeth [N + 1];
        J := J mod D;
        D := D div Base;
      end;
  End;

Function IntToBin (const I : Int64) : String;
  Begin
    Result := EncodeBase (I, 2);
  End;

Function IntToOct (const I : Int64) : String;
  Begin
    Result := EncodeBase (I, 8);
  End;

Function IntToHex (const I : Int64) : String;
  Begin
    Result := EncodeBase (I, 16);
  End;

Function DecodeBase64 (const S, Alphabet : String; const Zeros : CharSet) : String;
var F : Integer;
    B : Byte;
    OutPos : Byte;
    OutB : Array [1..3] of Byte;
  Begin
    if Length (Alphabet) <> 64 then
      raise EInvalidArgument.Create ('Invalid base 64 alphabet');

    Result := '';
    OutPos := 0;
    For F := 1 to Length (S) do
      begin
        if S [F] in Zeros then
          B := 0 else
          begin
            B := Pos (S [F], Alphabet);
            if B = 0 then
              raise EInvalidArgument.Create ('Invalid character (#' + IntToStr (Ord (S [F])) + ')');
            Dec (B);
          end;
        Case OutPos of
            0 : OutB [1] := B shl 2;
            1 : begin
                  OutB [1] := OutB [1] or (B shr 4);
                  Result := Result + Char (OutB [1]);
                  OutB [2] := (B shl 4) and $FF;
                end;
            2 : begin
                  OutB [2] := OutB [2] or (B shr 2);
                  Result := Result + Char (OutB [2]);
                  OutB [3] := (B shl 6) and $FF;
                end;
            3 : begin
                  OutB [3] := OutB [3] or B;
                  Result := Result + Char (OutB [3]);
                end;
          end;
        OutPos := (OutPos + 1) mod 4;
      end;

    if OutPos > 0 then
      Result := Result + Char (OutB [OutPos]);
  End;

Function UUDecode (const S : String) : String;
  Begin
    Result := DecodeBase64 (S, b64_UUEncode, ['`']);
  End;

Function MIMEBase64Decode (const S : String) : String;
  Begin
    Result := DecodeBase64 (S, b64_MIMEBase64, ['=']);
  End;

Function XXDecode (const S : String) : String;
  Begin
    Result := DecodeBase64 (S, b64_XXEncode, []);
  End;

{                                                                              }
{ Hashing functions                                                            }
{                                                                              }
{   Some speed comparisons on a P166MMX, as a reference:                       }
{     CRC16         7.5m cps                                                   }
{     CRC32         8.5m cps                                                   }
{     Checksum32   19.7m cps                                                   }
{     XOR8        126.9m cps                                                   }
{     XOR16        11.5m cps                                                   }
{     XOR32        12.3m cps                                                   }
{     MD5           1.4m cps                                                   }
{   Note the 16 bit functions are slower than the 32 bit ones.                 }
{   XOR8 has been hand optimized. It averages about 1 character every 1.3      }
{   clock ticks on a P166MMX.                                                  }
{   The loops in MD5 have not been unrolled for brevity.                       }
{                                                                              }

{ XOR-8 ---------------------------------------------------------------------- }
Function CalcXOR8 (const Data : String) : Byte; register;
  Asm
    push ebx
    mov ebx, Data                 // ebx = Data [1]
    xor eax, eax                  // al = Result
    mov ecx, [ebx - 4]            // ecx = Length (Data)
    or ecx, ecx
    jz @Fin                       // Length (Data) = 0
    cmp ecx, 4
    jb @DoRest                    // Length (Data) < 4

    // The following is an optimization of:                                   //
    //  @l1:   xor al, [ebx + ecx - 1]                                        //
    //         loop @l1                                                       //
    push ecx                                                                  //
    shr ecx, 2                    { ecx = Length (Data) div 4 }               //
  @l1:                                                                        //
    xor eax, [ebx + ecx * 4 - 4]  { Data [ecx * 4 - 3] }                      //
    dec ecx                                                                   //
    jnz @l1                                                                   //
                                                                              //
    mov ecx, eax                  { XOR bytes in eax }                        //
    xor al, ch                                                                //
    shr ecx, 16                                                               //
    xor al, cl                                                                //
    xor al, ch                                                                //
                                                                              //
    pop ecx                                                                   //
    add ebx, ecx                                                              //
    and ecx, 3                    { ecx = bytes remaining (0-3) }             //
    sub ebx, ecx                                                              //
    jz @Fin                                                                   //
  @DoRest:                                                                    //
    xor al, [ebx + ecx - 1]       { Faster than increasing ebx }              //
    dec ecx                                                                   //
    jnz @DoRest                   { Faster than loop @DoReset }               //
  @Fin:
    pop ebx
  End;

{ XOR-16 --------------------------------------------------------------------- }
Function CalcXOR16 (const Data : String) : Word;
var I : Integer;
  Begin
    Result := 0;
    For I := 1 to Length (Data) do
      Result := Result xor (Byte (Data [I]) shl (((I - 1) mod 2) * 8));
  End;

{ XOR-32 --------------------------------------------------------------------- }
Function CalcXOR32 (const Data : String) : LongWord;
var I : Integer;
  Begin
    Result := 0;
    For I := 1 to Length (Data) do
      Result := Result xor (Byte (Data [I]) shl (((I - 1) mod 4) * 8));
  End;

{ Checksum-32 ---------------------------------------------------------------- }
Function CalcChecksum32 (const Data : String) : LongWord;
  Asm                          // eax = Data [1]
      mov ecx, [eax - 4]       // ecx = length (Data)
      or ecx, ecx
      jz @fin
      push esi
      mov esi, eax             // esi = Data [1]
      xor eax, eax             // eax = CheckSum
      xor edx, edx
    @l1:
      mov dl, [esi + ecx - 1]  // edx = Data [ecx]
      add eax, edx
      loop @l1
      pop esi
    @fin:
  End;

{ CCITT CRC-16                                                                 }
{ The theory behind CCITT V.41 CRCs: (from CRCS.DOC by Guy Hirson)             }
{                                                                              }
{      1. Select the magnitude of the CRC to be used (typically 16 or 32       }
{         bits) and choose the polynomial to use. In the case of 16 bit        }
{         CRCs, the CCITT polynomial is recommended and is                     }
{                                                                              }
{                       16    12    5                                          }
{               G(x) = x   + x   + x  + 1                                      }
{                                                                              }
{         This polynomial traps 100% of 1 bit, 2 bit, odd numbers of bit       }
{         errors, 100% of <= 16 bit burst errors and over 99% of all           }
{         other errors.                                                        }
{                                                                              }
{                                                                              }
{      2. The CRC is calculated as                                             }
{                               r                                              }
{               D(x) = (M(x) * 2 )  mod G(x)                                   }
{                                                                              }
{         This may be better described as : Add r bits (0 content) to          }
{         the end of M(x). Divide this by G(x) and the remainder is the        }
{         CRC.                                                                 }
{                                                                              }
{      3. Tag the CRC onto the end of M(x).                                    }
{                                                                              }
{      4. To check it, calculate the CRC of the new message D(x), using        }
{         the same process as in 2. above. The newly calculated CRC            }
{         should be zero.                                                      }
{                                                                              }
{   This effectively means that using CRCs, it is possible to calculate a      }
{   series of bits to tag onto the data which makes the data an exact          }
{   multiple of the polynomial.                                                }
const
  CRC16Table : Array [0..255] of Word = (
    $0000, $1021, $2042, $3063, $4084, $50a5, $60c6, $70e7,
    $8108, $9129, $a14a, $b16b, $c18c, $d1ad, $e1ce, $f1ef,
    $1231, $0210, $3273, $2252, $52b5, $4294, $72f7, $62d6,
    $9339, $8318, $b37b, $a35a, $d3bd, $c39c, $f3ff, $e3de,
    $2462, $3443, $0420, $1401, $64e6, $74c7, $44a4, $5485,
    $a56a, $b54b, $8528, $9509, $e5ee, $f5cf, $c5ac, $d58d,
    $3653, $2672, $1611, $0630, $76d7, $66f6, $5695, $46b4,
    $b75b, $a77a, $9719, $8738, $f7df, $e7fe, $d79d, $c7bc,
    $48c4, $58e5, $6886, $78a7, $0840, $1861, $2802, $3823,
    $c9cc, $d9ed, $e98e, $f9af, $8948, $9969, $a90a, $b92b,
    $5af5, $4ad4, $7ab7, $6a96, $1a71, $0a50, $3a33, $2a12,
    $dbfd, $cbdc, $fbbf, $eb9e, $9b79, $8b58, $bb3b, $ab1a,
    $6ca6, $7c87, $4ce4, $5cc5, $2c22, $3c03, $0c60, $1c41,
    $edae, $fd8f, $cdec, $ddcd, $ad2a, $bd0b, $8d68, $9d49,
    $7e97, $6eb6, $5ed5, $4ef4, $3e13, $2e32, $1e51, $0e70,
    $ff9f, $efbe, $dfdd, $cffc, $bf1b, $af3a, $9f59, $8f78,
    $9188, $81a9, $b1ca, $a1eb, $d10c, $c12d, $f14e, $e16f,
    $1080, $00a1, $30c2, $20e3, $5004, $4025, $7046, $6067,
    $83b9, $9398, $a3fb, $b3da, $c33d, $d31c, $e37f, $f35e,
    $02b1, $1290, $22f3, $32d2, $4235, $5214, $6277, $7256,
    $b5ea, $a5cb, $95a8, $8589, $f56e, $e54f, $d52c, $c50d,
    $34e2, $24c3, $14a0, $0481, $7466, $6447, $5424, $4405,
    $a7db, $b7fa, $8799, $97b8, $e75f, $f77e, $c71d, $d73c,
    $26d3, $36f2, $0691, $16b0, $6657, $7676, $4615, $5634,
    $d94c, $c96d, $f90e, $e92f, $99c8, $89e9, $b98a, $a9ab,
    $5844, $4865, $7806, $6827, $18c0, $08e1, $3882, $28a3,
    $cb7d, $db5c, $eb3f, $fb1e, $8bf9, $9bd8, $abbb, $bb9a,
    $4a75, $5a54, $6a37, $7a16, $0af1, $1ad0, $2ab3, $3a92,
    $fd2e, $ed0f, $dd6c, $cd4d, $bdaa, $ad8b, $9de8, $8dc9,
    $7c26, $6c07, $5c64, $4c45, $3ca2, $2c83, $1ce0, $0cc1,
    $ef1f, $ff3e, $cf5d, $df7c, $af9b, $bfba, $8fd9, $9ff8,
    $6e17, $7e36, $4e55, $5e74, $2e93, $3eb2, $0ed1, $1ef0);

Function CalcCCITT_CRC16 (const Octet : Byte; const CRC16 : Word) : Word;
  Begin
    Result := CRC16Table [Hi (CRC16) xor Octet] xor (CRC16 * 256);
  End;

Function CalcCCITT_CRC16 (const Data : String) : Word;
var I : Integer;
  Begin
    Result := $FFFF;
    For I := 1 to Length (Data) do
      Result := CalcCCITT_CRC16 (Byte (Data [I]), Result);
  End;

{ CRC-32 --------------------------------------------------------------------- }
var
  CRC32TableInit : Boolean = False;
  CRC32Table     : Array [$00..$FF] of LongWord;

Procedure InitCRC32Table;
var I, J : Byte;
    R    : LongWord;
const CRCPoly = $EDB88320;
  Begin
    For I := $00 to $FF do
      begin
        R := I;
        For J := 8 downto 1 do
          if (R and 1) <> 0 then
            R := (R shr 1) xor CRCPoly else
            R := R shr 1;
        CRC32Table [I] := R;
      end;
  End;

Function CalcCRC32 (const Octet : Byte; const CRC32 : LongWord) : LongWord;
  Begin
    if not CRC32TableInit then // First call initializes the lookup table
      begin
        InitCRC32Table;
        CRC32TableInit := True;
      end;
    Result := CRC32Table [Byte (CRC32 xor LongWord (Octet))] xor ((CRC32 shr 8) and $00FFFFFF);
  End;

Function CalcCRC32 (var Buf; const BufLength : Integer; const CRC32 : LongWord) : LongWord;
var BufBytes : Array [0..1073741824] of Byte absolute Buf;
    I : Integer;
  Begin
    Result := CRC32;
    For I := 0 to BufLength - 1 do
      Result := CalcCRC32 (BufBytes [I], Result);
  End;

Function CalcCRC32 (const Data : String) : LongWord;
var I : Integer;
  Begin
    Result := $FFFFFFFF;
    For I := 1 to Length (Data) do
      Result := CalcCRC32 (Byte (Data [I]), Result);
    Result := not Result;
  End;


{ MD5 ------------------------------------------------------------------------ }
const
  MD5Table_1 : Array [0..15] of LongWord = (
      $D76AA478, $E8C7B756, $242070DB, $C1BDCEEE,
      $F57C0FAF, $4787C62A, $A8304613, $FD469501,
      $698098D8, $8B44F7AF, $FFFF5BB1, $895CD7BE,
      $6B901122, $FD987193, $A679438E, $49B40821);
  MD5Table_2 : Array [0..15] of LongWord = (
      $F61E2562, $C040B340, $265E5A51, $E9B6C7AA,
      $D62F105D, $02441453, $D8A1E681, $E7D3FBC8,
      $21E1CDE6, $C33707D6, $F4D50D87, $455A14ED,
      $A9E3E905, $FCEFA3F8, $676F02D9, $8D2A4C8A);
  MD5Table_3 : Array [0..15] of LongWord = (
      $FFFA3942, $8771F681, $6D9D6122, $FDE5380C,
      $A4BEEA44, $4BDECFA9, $F6BB4B60, $BEBFBC70,
      $289B7EC6, $EAA127FA, $D4EF3085, $04881D05,
      $D9D4D039, $E6DB99E5, $1FA27CF8, $C4AC5665);
  MD5Table_4 : Array [0..15] of LongWord = (
      $F4292244, $432AFF97, $AB9423A7, $FC93A039,
      $655B59C3, $8F0CCC92, $FFEFF47D, $85845DD1,
      $6FA87E4F, $FE2CE6E0, $A3014314, $4E0811A1,
      $F7537E82, $BD3AF235, $2AD7D2BB, $EB86D391);

{ Calculates new MD5 Digest (Array [0..3] of LongWord) given a Buffer          }
{   (Array [0..15] of LongWord).                                               }
Procedure TransformMD5Buffer (var Digest : Array of LongWord; const Buffer : Array of LongWord);
var A, B, C, D : LongWord;
    I          : Integer;
    J          : Byte;
  Begin
    A := Digest [0];
    B := Digest [1];
    C := Digest [2];
    D := Digest [3];

    For I := 0 to 3 do
      begin
        J := I * 4;
        Inc (A, Buffer [J]     + MD5Table_1 [J]     + (D xor (B and (C xor D)))); A := A shl  7 or A shr 25 + B;
        Inc (D, Buffer [J + 1] + MD5Table_1 [J + 1] + (C xor (A and (B xor C)))); D := D shl 12 or D shr 20 + A;
        Inc (C, Buffer [J + 2] + MD5Table_1 [J + 2] + (B xor (D and (A xor B)))); C := C shl 17 or C shr 15 + D;
        Inc (B, Buffer [J + 3] + MD5Table_1 [J + 3] + (A xor (C and (D xor A)))); B := B shl 22 or B shr 10 + C;
      end;

    For I := 0 to 3 do
      begin
        J := I * 4;
        Inc (A, Buffer [J + 1]           + MD5Table_2 [J]     + (C xor (D and (B xor C)))); A := A shl  5 or A shr 27 + B;
        Inc (D, Buffer [(J + 6) mod 16]  + MD5Table_2 [J + 1] + (B xor (C and (A xor B)))); D := D shl  9 or D shr 23 + A;
        Inc (C, Buffer [(J + 11) mod 16] + MD5Table_2 [J + 2] + (A xor (B and (D xor A)))); C := C shl 14 or C shr 18 + D;
        Inc (B, Buffer [J]               + MD5Table_2 [J + 3] + (D xor (A and (C xor D)))); B := B shl 20 or B shr 12 + C;
      end;

    For I := 0 to 3 do
      begin
        J := 16 - (I * 4);
        Inc (A, Buffer [(J + 5) mod 16]  + MD5Table_3 [16 - J]     + (B xor C xor D)); A := A shl  4 or A shr 28 + B;
        Inc (D, Buffer [(J + 8) mod 16]  + MD5Table_3 [16 - J + 1] + (A xor B xor C)); D := D shl 11 or D shr 21 + A;
        Inc (C, Buffer [(J + 11) mod 16] + MD5Table_3 [16 - J + 2] + (D xor A xor B)); C := C shl 16 or C shr 16 + D;
        Inc (B, Buffer [(J + 14) mod 16] + MD5Table_3 [16 - J + 3] + (C xor D xor A)); B := B shl 23 or B shr  9 + C;
      end;

    For I := 0 to 3 do
      begin
        J := 16 - (I * 4);
        Inc (A, Buffer [J mod 16]        + MD5Table_4 [16 - J]     + (C xor (B or not D))); A := A shl  6 or A shr 26 + B;
        Inc (D, Buffer [(J + 7) mod 16]  + MD5Table_4 [16 - J + 1] + (B xor (A or not C))); D := D shl 10 or D shr 22 + A;
        Inc (C, Buffer [(J + 14) mod 16] + MD5Table_4 [16 - J + 2] + (A xor (D or not B))); C := C shl 15 or C shr 17 + D;
        Inc (B, Buffer [(J + 5) mod 16]  + MD5Table_4 [16 - J + 3] + (D xor (C or not A))); B := B shl 21 or B shr 11 + C;
      end;

    Inc (Digest [0], A);
    Inc (Digest [1], B);
    Inc (Digest [2], C);
    Inc (Digest [3], D);
  End;

Function MD5InitKey : LongWordArray;
  Begin
    SetLength (Result, 4);
    Result [0] := $67452301;        // fixed initialization key
    Result [1] := $EFCDAB89;
    Result [2] := $98BADCFE;
    Result [3] := $10325476;
  End;

Function CalcMD5 (const Data : TExStream) : LongWordArray;
var S : String;
  Begin
    Result := MD5InitKey;

    Data.Position := 0;
    if Data.EOF then
      exit;

    While not Data.EOF do
      begin
        S := Data.Read (64);
        if Length (S) < 64 then
          break;
        TransformMD5Buffer (Result, LongWordArray (S));
      end;
    S := S + #$80;
    if Length (S) > 64 - Sizeof (Int64) then
      begin
        S := PadRight (S, #0, 64);
        TransformMD5Buffer (Result, LongWordArray (S));
        S := '';
      end;
    S := PadRight (S, #0, 64 - Sizeof (Int64));
    S := S + Pack (Data.Position * 8);
    TransformMD5Buffer (Result, LongWordArray (S));
  End;

Function CalcMD5 (const Data : String) : LongWordArray;
var S : TStringStream;
  Begin
    S := TStringStream.CreateEx (Data);
    try
      Result := CalcMD5 (S);
    finally
      S.Free;
    end;
  End;

{ Hash ----------------------------------------------------------------------- }
Function Hash (const S : String; const Slots : LongWord) : LongWord;
  Begin
    Result := CalcCRC32 (S) mod Slots;
  End;


{                                                                              }
{ Fast factorial                                                               }
{                                                                              }
{   For small values of N, calculate normally using 2*3*..*N                   }
{   For larger values of N, calculate using Gamma (N+1)                        }
{   Smaller values of N is cached to avoid recalculation.                      }
{                                                                              }
const
  FactorialCacheLimit = 409;

var
  FactorialCache : ExtendedArray;

Function Factorial (const N : Integer) : Extended;
const
  MaxLimit = 1754.0;
  SwitchLimit = 34;
var
  L : Extended;
  I : Integer;

  Begin
    if N > MaxLimit then
      raise EOverflow.Create ('') else
      if N < 0 then
        raise EInvalidArgument.Create ('') else
        if (N <= FactorialCacheLimit) and Assigned (FactorialCache) and (FactorialCache [N] >= 1.0) then
          Result := FactorialCache [N] else
          begin
            if N < SwitchLimit then
              begin
                L := 1.0;
                I := 2;
                While I <= N do
                  begin
                    L := L * I;
                    Inc (I);
                  end;
                Result := L;
              end else
              Result := Exp (GammaLn (N + 1));

            if N <= FactorialCacheLimit then
              begin
                if not Assigned (FactorialCache) then
                  SetLength (FactorialCache, FactorialCacheLimit);
                FactorialCache [N] := Result;
              end;
          end;
  End;



{                                                                              }
{ Combinatorics                                                                }
{                                                                              }
Function Combinations (const N, C : Integer) : Extended;
  Begin
    Result := Factorial (N) / (Factorial (C) * Factorial (N - C));
  End;

Function Permutations (const N, P : Integer) : Extended;
  Begin
    Result := Factorial (N) / Factorial (N - P);
  End;

Function Fibonacci (const N : Integer) : Int64;
  Begin
    if N < 0 then
      raise ERangeError.Create ('Fibonacci: Invalid parameter');
    if N <= 1 then
      Result := N else
      Result := Fibonacci (N - 1) + Fibonacci (N - 2);
  End;

{                                                                              }
{ Unit Conversion                                                              }
{                                                                              }
Function KelvinToFahrenheit (const T : Extended) : Extended;
  Begin
    Result := ((9 / 5) * (T - 273.15)) + 32;
  End;

Function FahrenheitToKelvin (const T : Extended) : Extended;
  Begin
    Result := ((5 / 9) * (T - 32)) + 273.15;
  End;

Function CelsiusToKelvin (const T : Extended) : Extended;
  Begin
    Result := T + 273.15;
  End;

Function KelvinToCelsius (const T : Extended) : Extended;
  Begin
    Result := T - 273.15;
  End;

Function CelsiusToFahrenheit (const T : Extended) : Extended;
  Begin
    Result := ((9 / 5) * T) + 32;
  End;

Function FahrenheitToCelsius (const T : Extended) : Extended;
  Begin
    Result := (5 / 9) * (T - 32);
  End;

{                                                                              }
{ Trig                                                                         }
{                                                                              }
Function InverseTangentDeg (const X, Y : Extended) : Extended;
{ 0 <= Result <= 360 }
var Angle : Extended;
  Begin
    if FloatZero (X) then
      Angle := Pi / 2.0 else
      Angle := ArcTan (Y / X);
    Angle := Angle * 180.0 / Pi;

    if (X <= 0.0) and (Y < 0.0) then
      Angle := Angle - 180.0 else
    if (X < 0.0) and (Y > 0.0) then
      Angle := Angle + 180.0;

    If Angle < 0.0 then
      Angle := Angle + 360.0;

    InverseTangentDeg := Angle;
  End;

Function InverseTangentRad (const X, Y : Extended) : Extended;
{ 0 <= result <= 2pi }
var Angle : Extended;
  Begin
    if FloatZero (X) then
      Angle := Pi / 2.0 else
      Angle := ArcTan (Y / X);
    if (X <= 0.0) and (Y < 0) then
      Angle := Angle - Pi;
    if (X < 0.0) and (Y > 0) then
      Angle := Angle + Pi;
    If Angle < 0 then
      Angle := Angle + 2 * Pi;
    InverseTangentRad := Angle;
  End;

Function InverseSinDeg (const Y, R : Extended) : Extended;
{ -90 <= result <= 90 }
var X : Extended;
  Begin
    X := Sqrt (Sqr (R) - Sqr (Y));
    Result := InverseTangentDeg (X, Y);
    If Result > 90.0 then
      Result := Result - 360.0;
  End;

Function InverseSinRad (const Y, R : Extended) : Extended;
{ -90 <= result <= 90 }
var X : Extended;
  Begin
    X := Sqrt (Sqr (R)-Sqr (Y));
    Result := InverseTangentRad (X, Y);
    if Result > 90.0 then
      Result := Result - 360.0;
  End;

Function InverseCosDeg (const X, R : Extended) : Extended;
{ -90 <= result <= 90 }
var Y : Extended;
  Begin
    Y := Sqrt (Sqr (R)-Sqr (X));
    Result := InverseTangentDeg(X, Y);
    if Result > 90.0 then
      Result := Result - 360.0;
  End;

Function InverseCosRad (const X, R : Extended) : Extended;
{ -90 <= result <= 90 }
var Y : Extended;
  Begin
    Y := Sqrt (Sqr (R)-Sqr (X));
    Result := InverseTangentRad (X, Y);
    if Result > 90.0 then
      Result := Result - 360.0;
  End;

Function ATan360 (const X, Y : Extended) : Extended;
var Angle: Extended;
  Begin
    if FloatZero (X) then
      Angle := Pi / 2.0 else
      Angle := ArcTan (Y / X);
    Angle := Angle * DegPerRad;
    if (X <= 0.0) and (Y < 0.0) then
      Angle := Angle - 180.0;
    if (X < 0.0) and (Y > 0.0) then
      Angle := Angle + 180.0;
    If Angle < 0.0 then
      Angle := Angle + 360.0;
    ATan360 := Angle;
  End;






{                                                                              }
{ Statistical functions                                                        }
{                                                                              }
Function RandomSeed : LongWord;
var I            : Int64;
    Ye, Mo, Da   : Word;
    H, Mi, S, S1 : Word;
  Begin
    DecodeDate (Date, Ye, Mo, Da);
    Result := Ye xor (Mo shl 16) xor (Da shl 24);
    Result := Result xor GetTickCount;
    if QueryPerformanceFrequency (I) then
      Result := Result xor LongWord (I) xor LongWord (I shr 32);
    if QueryPerformanceCounter (I) then
      Result := Result xor LongWord (I) xor LongWord (I shr 32);
    DecodeTime (Time, H, Mi, S, S1);
    Result := Result xor H xor (Mi shl 8) xor (S1 shl 16) xor (S shl 24);
  End;

Function BinomialCoeff (N, R : Integer) : Extended;
var I, K : Integer;
  Begin
    if (N = 0) or (R > N) then
      raise EInvalidArgument.Create ('Invalid parameters to BinomialCoeff');
    if N > 1547 then
      raise EOverflow.Create ('BinomialCoeff overflow');

    Result := 1.0;
    if (R = 0) or (R = N) then
     exit;

    if R > N div 2 then
     R := N - R;

    K := 2;
    For I := N - R + 1 to N do
      begin
	Result := Result * I;
	if K <= R then
	  begin
	    Result := Result / K;
	    Inc (K);
          end;
      end;
    Result := Int (Result + 0.5);
  End;



{ Random number generator from ACM Transactions on Modeling and Computer       }
{ Simulation 8(1) 3-30, 1990.  Supposedly it has a period of -1 + 2^19937.     }
{ The original was in C; this translation returns the same values as the       }
{ original.  It is called the Mersenne Twister.                                }
{ The following code was written by Toby Ewing <ewing@iastate.edu>, slightly   }
{ modified by Frank Heckenbach <frank@pascal.gnu.de> and again slightly        }
{ modified by David Butler <david@e.co.za> for use in Delphi. It was inspired  }
{ by C code, released under the GNU Library General Public License, written by }
{ Makoto Matsumoto <matumoto@math.keio.ac.jp> and Takuji Nishimura,            }
{ considering the suggestions by Topher Cooper and Marc Rieffel in July-       }
{ Aug 97.                                                                      }
const
  N = 624; // Period parameters
  M = 397;

var
  mti : Integer;
  mt  : Array [0 .. N] of LongWord; // the array for the state vector
  RandomUniformInitialized : Boolean = False;

{ Set initial seeds to mt [N] using the generator Line 25 of Table 1 in        }
{ [KNUTH 1981, The Art of Computer Programming Vol. 2 (2nd Ed.), pp 102].      }
Procedure RandomUniformInit (const Seed : LongWord);
  Begin
    mt [0] := Seed;
    For mti := 1 to N do
      mt [mti] := LongWord (Int64 (69069) * mt [mti - 1]);
    mti := N;
    RandomUniformInitialized := True
  End;

Function RandomUniform : LongWord;
const
  Matrix_A = $9908B0DF; // constant vector a
  T_Mask_B = $9D2C5680; // Tempering parameters
  T_Mask_C = $EFC60000;
  Up_Mask  = $80000000; // most significant w-r bits
  Low_Mask = $7FFFFFFF; // least significant r bits
  mag01    : Array [0..1] of LongWord = (0, Matrix_A);

var
  y  : LongWord;
  kk : Integer;

  Begin
    if not RandomUniformInitialized then
      RandomUniformInit (RandomSeed);

    if mti >= N then { generate N words at one time }
      begin
        For kk := 0 to N - M do
          begin
            y := (mt [kk] and Up_Mask) or (mt [kk + 1] and Low_Mask);
            mt [kk] := mt [kk + M] xor (y shr 1) xor mag01 [y and 1]
          end;
        For kk := N - M to N - 1 do
          begin
            y := (mt [kk] and Up_Mask) or (mt [kk + 1] and Low_Mask);
            mt [kk] := mt [kk + M - N] xor (y shr 1) xor mag01 [y and 1]
          end;
        y := (mt [N - 1] and Up_Mask) or (mt [0] and Low_Mask);
        mt [N - 1] := mt [M - 1] xor (y shr 1) xor mag01 [y and 1];
        mti := 0
      end;
    y := mt [mti];
    Inc (mti);
    y := y xor (y shr 11);
    y := y xor ((y shl 7) and T_Mask_B);
    y := y xor ((y shl 15) and T_Mask_C);
    y := y xor (y shr 18);
    RandomUniform := y;
  End;

Function RandomUniformF : Extended;
  Begin
    Result := RandomUniform / High (LongWord);
  End;


{ Generates a real number from the normal (0,1) (gaussian) distribution.       }
var
  HasRandomNormal : Boolean = False;
  ARandomNormal : Extended;

Function RandomNormalF : Extended;
var fac, r, v1, v2 : Extended;
  Begin
    if not HasRandomNormal then
      begin
        Repeat
          v1 := 2.0 * RandomUniformF - 1.0;
          v2 := 2.0 * RandomUniformF - 1.0;
          r := Sqr (v1) + Sqr (v2);
        Until r < 1.0;
        fac := Sqrt (-2.0 * ln (r) / r);
        ARandomNormal := v1 * fac;
        Result := v2 * fac;
        HasRandomNormal := True;
      end else
      begin
        Result := ARandomNormal;
        HasRandomNormal := False;
      end;
  End;

{ These password "nibbles" are from RFC1760 by Neil Haller.                    }
const
  WordNibbles = 2048;
  WordNibble : Array [1..WordNibbles] of String [4] = (
              'A',     'ABE',   'ACE',   'ACT',   'AD',    'ADA',   'ADD',
     'AGO',   'AID',   'AIM',   'AIR',   'ALL',   'ALP',   'AM',    'AMY',
     'AN',    'ANA',   'AND',   'ANN',   'ANT',   'ANY',   'APE',   'APS',
     'APT',   'ARC',   'ARE',   'ARK',   'ARM',   'ART',   'AS',    'ASH',
     'ASK',   'AT',    'ATE',   'AUG',   'AUK',   'AVE',   'AWE',   'AWK',
     'AWL',   'AWN',   'AX',    'AYE',   'BAD',   'BAG',   'BAH',   'BAM',
     'BAN',   'BAR',   'BAT',   'BAY',   'BE',    'BED',   'BEE',   'BEG',
     'BEN',   'BET',   'BEY',   'BIB',   'BID',   'BIG',   'BIN',   'BIT',
     'BOB',   'BOG',   'BON',   'BOO',   'BOP',   'BOW',   'BOY',   'BUB',
     'BUD',   'BUG',   'BUM',   'BUN',   'BUS',   'BUT',   'BUY',   'BY',
     'BYE',   'CAB',   'CAL',   'CAM',   'CAN',   'CAP',   'CAR',   'CAT',
     'CAW',   'COD',   'COG',   'COL',   'CON',   'COO',   'COP',   'COT',
     'COW',   'COY',   'CRY',   'CUB',   'CUE',   'CUP',   'CUR',   'CUT',
     'DAB',   'DAD',   'DAM',   'DAN',   'DAR',   'DAY',   'DEE',   'DEL',
     'DEN',   'DES',   'DEW',   'DID',   'DIE',   'DIG',   'DIN',   'DIP',
     'DO',    'DOE',   'DOG',   'DON',   'DOT',   'DOW',   'DRY',   'DUB',
     'DUD',   'DUE',   'DUG',   'DUN',   'EAR',   'EAT',   'ED',    'EEL',
     'EGG',   'EGO',   'ELI',   'ELK',   'ELM',   'ELY',   'EM',    'END',
     'EST',   'ETC',   'EVA',   'EVE',   'EWE',   'EYE',   'FAD',   'FAN',
     'FAR',   'FAT',   'FAY',   'FED',   'FEE',   'FEW',   'FIB',   'FIG',
     'FIN',   'FIR',   'FIT',   'FLO',   'FLY',   'FOE',   'FOG',   'FOR',
     'FRY',   'FUM',   'FUN',   'FUR',   'GAB',   'GAD',   'GAG',   'GAL',
     'GAM',   'GAP',   'GAS',   'GAY',   'GEE',   'GEL',   'GEM',   'GET',
     'GIG',   'GIL',   'GIN',   'GO',    'GOT',   'GUM',   'GUN',   'GUS',
     'GUT',   'GUY',   'GYM',   'GYP',   'HA',    'HAD',   'HAL',   'HAM',
     'HAN',   'HAP',   'HAS',   'HAT',   'HAW',   'HAY',   'HE',    'HEM',
     'HEN',   'HER',   'HEW',   'HEY',   'HI',    'HID',   'HIM',   'HIP',
     'HIS',   'HIT',   'HO',    'HOB',   'HOC',   'HOE',   'HOG',   'HOP',
     'HOT',   'HOW',   'HUB',   'HUE',   'HUG',   'HUH',   'HUM',   'HUT',
     'I',     'ICY',   'IDA',   'IF',    'IKE',   'ILL',   'INK',   'INN',
     'IO',    'ION',   'IQ',    'IRA',   'IRE',   'IRK',   'IS',    'IT',
     'ITS',   'IVY',   'JAB',   'JAG',   'JAM',   'JAN',   'JAR',   'JAW',
     'JAY',   'JET',   'JIG',   'JIM',   'JO',    'JOB',   'JOE',   'JOG',
     'JOT',   'JOY',   'JUG',   'JUT',   'KAY',   'KEG',   'KEN',   'KEY',
     'KID',   'KIM',   'KIN',   'KIT',   'LA',    'LAB',   'LAC',   'LAD',
     'LAG',   'LAM',   'LAP',   'LAW',   'LAY',   'LEA',   'LED',   'LEE',
     'LEG',   'LEN',   'LEO',   'LET',   'LEW',   'LID',   'LIE',   'LIN',
     'LIP',   'LIT',   'LO',    'LOB',   'LOG',   'LOP',   'LOS',   'LOT',
     'LOU',   'LOW',   'LOY',   'LUG',   'LYE',   'MA',    'MAC',   'MAD',
     'MAE',   'MAN',   'MAO',   'MAP',   'MAT',   'MAW',   'MAY',   'ME',
     'MEG',   'MEL',   'MEN',   'MET',   'MEW',   'MID',   'MIN',   'MIT',
     'MOB',   'MOD',   'MOE',   'MOO',   'MOP',   'MOS',   'MOT',   'MOW',
     'MUD',   'MUG',   'MUM',   'MY',    'NAB',   'NAG',   'NAN',   'NAP',
     'NAT',   'NAY',   'NE',    'NED',   'NEE',   'NET',   'NEW',   'NIB',
     'NIL',   'NIP',   'NIT',   'NO',    'NOB',   'NOD',   'NON',   'NOR',
     'NOT',   'NOV',   'NOW',   'NU',    'NUN',   'NUT',   'O',     'OAF',
     'OAK',   'OAR',   'OAT',   'ODD',   'ODE',   'OF',    'OFF',   'OFT',
     'OH',    'OIL',   'OK',    'OLD',   'ON',    'ONE',   'OR',    'ORB',
     'ORE',   'ORR',   'OS',    'OTT',   'OUR',   'OUT',   'OVA',   'OW',
     'OWE',   'OWL',   'OWN',   'OX',    'PA',    'PAD',   'PAL',   'PAM',
     'PAN',   'PAP',   'PAR',   'PAT',   'PAW',   'PAY',   'PEA',   'PEG',
     'PEN',   'PEP',   'PER',   'PET',   'PEW',   'PHI',   'PI',    'PIE',
     'PIN',   'PIT',   'PLY',   'PO',    'POD',   'POE',   'POP',   'POT',
     'POW',   'PRO',   'PRY',   'PUB',   'PUG',   'PUN',   'PUP',   'PUT',
     'QUO',   'RAG',   'RAM',   'RAN',   'RAP',   'RAT',   'RAW',   'RAY',
     'REB',   'RED',   'REP',   'RET',   'RIB',   'RID',   'RIG',   'RIM',
     'RIO',   'RIP',   'ROB',   'ROD',   'ROE',   'RON',   'ROT',   'ROW',
     'ROY',   'RUB',   'RUE',   'RUG',   'RUM',   'RUN',   'RYE',   'SAC',
     'SAD',   'SAG',   'SAL',   'SAM',   'SAN',   'SAP',   'SAT',   'SAW',
     'SAY',   'SEA',   'SEC',   'SEE',   'SEN',   'SET',   'SEW',   'SHE',
     'SHY',   'SIN',   'SIP',   'SIR',   'SIS',   'SIT',   'SKI',   'SKY',
     'SLY',   'SO',    'SOB',   'SOD',   'SON',   'SOP',   'SOW',   'SOY',
     'SPA',   'SPY',   'SUB',   'SUD',   'SUE',   'SUM',   'SUN',   'SUP',
     'TAB',   'TAD',   'TAG',   'TAN',   'TAP',   'TAR',   'TEA',   'TED',
     'TEE',   'TEN',   'THE',   'THY',   'TIC',   'TIE',   'TIM',   'TIN',
     'TIP',   'TO',    'TOE',   'TOG',   'TOM',   'TON',   'TOO',   'TOP',
     'TOW',   'TOY',   'TRY',   'TUB',   'TUG',   'TUM',   'TUN',   'TWO',
     'UN',    'UP',    'US',    'USE',   'VAN',   'VAT',   'VET',   'VIE',
     'WAD',   'WAG',   'WAR',   'WAS',   'WAY',   'WE',    'WEB',   'WED',
     'WEE',   'WET',   'WHO',   'WHY',   'WIN',   'WIT',   'WOK',   'WON',
     'WOO',   'WOW',   'WRY',   'WU',    'YAM',   'YAP',   'YAW',   'YE',
     'YEA',   'YES',   'YET',   'YOU',   'ABED',  'ABEL',  'ABET',  'ABLE',
     'ABUT',  'ACHE',  'ACID',  'ACME',  'ACRE',  'ACTA',  'ACTS',  'ADAM',
     'ADDS',  'ADEN',  'AFAR',  'AFRO',  'AGEE',  'AHEM',  'AHOY',  'AIDA',
     'AIDE',  'AIDS',  'AIRY',  'AJAR',  'AKIN',  'ALAN',  'ALEC',  'ALGA',
     'ALIA',  'ALLY',  'ALMA',  'ALOE',  'ALSO',  'ALTO',  'ALUM',  'ALVA',
     'AMEN',  'AMES',  'AMID',  'AMMO',  'AMOK',  'AMOS',  'AMRA',  'ANDY',
     'ANEW',  'ANNA',  'ANNE',  'ANTE',  'ANTI',  'AQUA',  'ARAB',  'ARCH',
     'AREA',  'ARGO',  'ARID',  'ARMY',  'ARTS',  'ARTY',  'ASIA',  'ASKS',
     'ATOM',  'AUNT',  'AURA',  'AUTO',  'AVER',  'AVID',  'AVIS',  'AVON',
     'AVOW',  'AWAY',  'AWRY',  'BABE',  'BABY',  'BACH',  'BACK',  'BADE',
     'BAIL',  'BAIT',  'BAKE',  'BALD',  'BALE',  'BALI',  'BALK',  'BALL',
     'BALM',  'BAND',  'BANE',  'BANG',  'BANK',  'BARB',  'BARD',  'BARE',
     'BARK',  'BARN',  'BARR',  'BASE',  'BASH',  'BASK',  'BASS',  'BATE',
     'BATH',  'BAWD',  'BAWL',  'BEAD',  'BEAK',  'BEAM',  'BEAN',  'BEAR',
     'BEAT',  'BEAU',  'BECK',  'BEEF',  'BEEN',  'BEER',  'BEET',  'BELA',
     'BELL',  'BELT',  'BEND',  'BENT',  'BERG',  'BERN',  'BERT',  'BESS',
     'BEST',  'BETA',  'BETH',  'BHOY',  'BIAS',  'BIDE',  'BIEN',  'BILE',
     'BILK',  'BILL',  'BIND',  'BING',  'BIRD',  'BITE',  'BITS',  'BLAB',
     'BLAT',  'BLED',  'BLEW',  'BLOB',  'BLOC',  'BLOT',  'BLOW',  'BLUE',
     'BLUM',  'BLUR',  'BOAR',  'BOAT',  'BOCA',  'BOCK',  'BODE',  'BODY',
     'BOGY',  'BOHR',  'BOIL',  'BOLD',  'BOLO',  'BOLT',  'BOMB',  'BONA',
     'BOND',  'BONE',  'BONG',  'BONN',  'BONY',  'BOOK',  'BOOM',  'BOON',
     'BOOT',  'BORE',  'BORG',  'BORN',  'BOSE',  'BOSS',  'BOTH',  'BOUT',
     'BOWL',  'BOYD',  'BRAD',  'BRAE',  'BRAG',  'BRAN',  'BRAY',  'BRED',
     'BREW',  'BRIG',  'BRIM',  'BROW',  'BUCK',  'BUDD',  'BUFF',  'BULB',
     'BULK',  'BULL',  'BUNK',  'BUNT',  'BUOY',  'BURG',  'BURL',  'BURN',
     'BURR',  'BURT',  'BURY',  'BUSH',  'BUSS',  'BUST',  'BUSY',  'BYTE',
     'CADY',  'CAFE',  'CAGE',  'CAIN',  'CAKE',  'CALF',  'CALL',  'CALM',
     'CAME',  'CANE',  'CANT',  'CARD',  'CARE',  'CARL',  'CARR',  'CART',
     'CASE',  'CASH',  'CASK',  'CAST',  'CAVE',  'CEIL',  'CELL',  'CENT',
     'CERN',  'CHAD',  'CHAR',  'CHAT',  'CHAW',  'CHEF',  'CHEN',  'CHEW',
     'CHIC',  'CHIN',  'CHOU',  'CHOW',  'CHUB',  'CHUG',  'CHUM',  'CITE',
     'CITY',  'CLAD',  'CLAM',  'CLAN',  'CLAW',  'CLAY',  'CLOD',  'CLOG',
     'CLOT',  'CLUB',  'CLUE',  'COAL',  'COAT',  'COCA',  'COCK',  'COCO',
     'CODA',  'CODE',  'CODY',  'COED',  'COIL',  'COIN',  'COKE',  'COLA',
     'COLD',  'COLT',  'COMA',  'COMB',  'COME',  'COOK',  'COOL',  'COON',
     'COOT',  'CORD',  'CORE',  'CORK',  'CORN',  'COST',  'COVE',  'COWL',
     'CRAB',  'CRAG',  'CRAM',  'CRAY',  'CREW',  'CRIB',  'CROW',  'CRUD',
     'CUBA',  'CUBE',  'CUFF',  'CULL',  'CULT',  'CUNY',  'CURB',  'CURD',
     'CURE',  'CURL',  'CURT',  'CUTS',  'DADE',  'DALE',  'DAME',  'DANA',
     'DANE',  'DANG',  'DANK',  'DARE',  'DARK',  'DARN',  'DART',  'DASH',
     'DATA',  'DATE',  'DAVE',  'DAVY',  'DAWN',  'DAYS',  'DEAD',  'DEAF',
     'DEAL',  'DEAN',  'DEAR',  'DEBT',  'DECK',  'DEED',  'DEEM',  'DEER',
     'DEFT',  'DEFY',  'DELL',  'DENT',  'DENY',  'DESK',  'DIAL',  'DICE',
     'DIED',  'DIET',  'DIME',  'DINE',  'DING',  'DINT',  'DIRE',  'DIRT',
     'DISC',  'DISH',  'DISK',  'DIVE',  'DOCK',  'DOES',  'DOLE',  'DOLL',
     'DOLT',  'DOME',  'DONE',  'DOOM',  'DOOR',  'DORA',  'DOSE',  'DOTE',
     'DOUG',  'DOUR',  'DOVE',  'DOWN',  'DRAB',  'DRAG',  'DRAM',  'DRAW',
     'DREW',  'DRUB',  'DRUG',  'DRUM',  'DUAL',  'DUCK',  'DUCT',  'DUEL',
     'DUET',  'DUKE',  'DULL',  'DUMB',  'DUNE',  'DUNK',  'DUSK',  'DUST',
     'DUTY',  'EACH',  'EARL',  'EARN',  'EASE',  'EAST',  'EASY',  'EBEN',
     'ECHO',  'EDDY',  'EDEN',  'EDGE',  'EDGY',  'EDIT',  'EDNA',  'EGAN',
     'ELAN',  'ELBA',  'ELLA',  'ELSE',  'EMIL',  'EMIT',  'EMMA',  'ENDS',
     'ERIC',  'EROS',  'EVEN',  'EVER',  'EVIL',  'EYED',  'FACE',  'FACT',
     'FADE',  'FAIL',  'FAIN',  'FAIR',  'FAKE',  'FALL',  'FAME',  'FANG',
     'FARM',  'FAST',  'FATE',  'FAWN',  'FEAR',  'FEAT',  'FEED',  'FEEL',
     'FEET',  'FELL',  'FELT',  'FEND',  'FERN',  'FEST',  'FEUD',  'FIEF',
     'FIGS',  'FILE',  'FILL',  'FILM',  'FIND',  'FINE',  'FINK',  'FIRE',
     'FIRM',  'FISH',  'FISK',  'FIST',  'FITS',  'FIVE',  'FLAG',  'FLAK',
     'FLAM',  'FLAT',  'FLAW',  'FLEA',  'FLED',  'FLEW',  'FLIT',  'FLOC',
     'FLOG',  'FLOW',  'FLUB',  'FLUE',  'FOAL',  'FOAM',  'FOGY',  'FOIL',
     'FOLD',  'FOLK',  'FOND',  'FONT',  'FOOD',  'FOOL',  'FOOT',  'FORD',
     'FORE',  'FORK',  'FORM',  'FORT',  'FOSS',  'FOUL',  'FOUR',  'FOWL',
     'FRAU',  'FRAY',  'FRED',  'FREE',  'FRET',  'FREY',  'FROG',  'FROM',
     'FUEL',  'FULL',  'FUME',  'FUND',  'FUNK',  'FURY',  'FUSE',  'FUSS',
     'GAFF',  'GAGE',  'GAIL',  'GAIN',  'GAIT',  'GALA',  'GALE',  'GALL',
     'GALT',  'GAME',  'GANG',  'GARB',  'GARY',  'GASH',  'GATE',  'GAUL',
     'GAUR',  'GAVE',  'GAWK',  'GEAR',  'GELD',  'GENE',  'GENT',  'GERM',
     'GETS',  'GIBE',  'GIFT',  'GILD',  'GILL',  'GILT',  'GINA',  'GIRD',
     'GIRL',  'GIST',  'GIVE',  'GLAD',  'GLEE',  'GLEN',  'GLIB',  'GLOB',
     'GLOM',  'GLOW',  'GLUE',  'GLUM',  'GLUT',  'GOAD',  'GOAL',  'GOAT',
     'GOER',  'GOES',  'GOLD',  'GOLF',  'GONE',  'GONG',  'GOOD',  'GOOF',
     'GORE',  'GORY',  'GOSH',  'GOUT',  'GOWN',  'GRAB',  'GRAD',  'GRAY',
     'GREG',  'GREW',  'GREY',  'GRID',  'GRIM',  'GRIN',  'GRIT',  'GROW',
     'GRUB',  'GULF',  'GULL',  'GUNK',  'GURU',  'GUSH',  'GUST',  'GWEN',
     'GWYN',  'HAAG',  'HAAS',  'HACK',  'HAIL',  'HAIR',  'HALE',  'HALF',
     'HALL',  'HALO',  'HALT',  'HAND',  'HANG',  'HANK',  'HANS',  'HARD',
     'HARK',  'HARM',  'HART',  'HASH',  'HAST',  'HATE',  'HATH',  'HAUL',
     'HAVE',  'HAWK',  'HAYS',  'HEAD',  'HEAL',  'HEAR',  'HEAT',  'HEBE',
     'HECK',  'HEED',  'HEEL',  'HEFT',  'HELD',  'HELL',  'HELM',  'HERB',
     'HERD',  'HERE',  'HERO',  'HERS',  'HESS',  'HEWN',  'HICK',  'HIDE',
     'HIGH',  'HIKE',  'HILL',  'HILT',  'HIND',  'HINT',  'HIRE',  'HISS',
     'HIVE',  'HOBO',  'HOCK',  'HOFF',  'HOLD',  'HOLE',  'HOLM',  'HOLT',
     'HOME',  'HONE',  'HONK',  'HOOD',  'HOOF',  'HOOK',  'HOOT',  'HORN',
     'HOSE',  'HOST',  'HOUR',  'HOVE',  'HOWE',  'HOWL',  'HOYT',  'HUCK',
     'HUED',  'HUFF',  'HUGE',  'HUGH',  'HUGO',  'HULK',  'HULL',  'HUNK',
     'HUNT',  'HURD',  'HURL',  'HURT',  'HUSH',  'HYDE',  'HYMN',  'IBIS',
     'ICON',  'IDEA',  'IDLE',  'IFFY',  'INCA',  'INCH',  'INTO',  'IONS',
     'IOTA',  'IOWA',  'IRIS',  'IRMA',  'IRON',  'ISLE',  'ITCH',  'ITEM',
     'IVAN',  'JACK',  'JADE',  'JAIL',  'JAKE',  'JANE',  'JAVA',  'JEAN',
     'JEFF',  'JERK',  'JESS',  'JEST',  'JIBE',  'JILL',  'JILT',  'JIVE',
     'JOAN',  'JOBS',  'JOCK',  'JOEL',  'JOEY',  'JOHN',  'JOIN',  'JOKE',
     'JOLT',  'JOVE',  'JUDD',  'JUDE',  'JUDO',  'JUDY',  'JUJU',  'JUKE',
     'JULY',  'JUNE',  'JUNK',  'JUNO',  'JURY',  'JUST',  'JUTE',  'KAHN',
     'KALE',  'KANE',  'KANT',  'KARL',  'KATE',  'KEEL',  'KEEN',  'KENO',
     'KENT',  'KERN',  'KERR',  'KEYS',  'KICK',  'KILL',  'KIND',  'KING',
     'KIRK',  'KISS',  'KITE',  'KLAN',  'KNEE',  'KNEW',  'KNIT',  'KNOB',
     'KNOT',  'KNOW',  'KOCH',  'KONG',  'KUDO',  'KURD',  'KURT',  'KYLE',
     'LACE',  'LACK',  'LACY',  'LADY',  'LAID',  'LAIN',  'LAIR',  'LAKE',
     'LAMB',  'LAME',  'LAND',  'LANE',  'LANG',  'LARD',  'LARK',  'LASS',
     'LAST',  'LATE',  'LAUD',  'LAVA',  'LAWN',  'LAWS',  'LAYS',  'LEAD',
     'LEAF',  'LEAK',  'LEAN',  'LEAR',  'LEEK',  'LEER',  'LEFT',  'LEND',
     'LENS',  'LENT',  'LEON',  'LESK',  'LESS',  'LEST',  'LETS',  'LIAR',
     'LICE',  'LICK',  'LIED',  'LIEN',  'LIES',  'LIEU',  'LIFE',  'LIFT',
     'LIKE',  'LILA',  'LILT',  'LILY',  'LIMA',  'LIMB',  'LIME',  'LIND',
     'LINE',  'LINK',  'LINT',  'LION',  'LISA',  'LIST',  'LIVE',  'LOAD',
     'LOAF',  'LOAM',  'LOAN',  'LOCK',  'LOFT',  'LOGE',  'LOIS',  'LOLA',
     'LONE',  'LONG',  'LOOK',  'LOON',  'LOOT',  'LORD',  'LORE',  'LOSE',
     'LOSS',  'LOST',  'LOUD',  'LOVE',  'LOWE',  'LUCK',  'LUCY',  'LUGE',
     'LUKE',  'LULU',  'LUND',  'LUNG',  'LURA',  'LURE',  'LURK',  'LUSH',
     'LUST',  'LYLE',  'LYNN',  'LYON',  'LYRA',  'MACE',  'MADE',  'MAGI',
     'MAID',  'MAIL',  'MAIN',  'MAKE',  'MALE',  'MALI',  'MALL',  'MALT',
     'MANA',  'MANN',  'MANY',  'MARC',  'MARE',  'MARK',  'MARS',  'MART',
     'MARY',  'MASH',  'MASK',  'MASS',  'MAST',  'MATE',  'MATH',  'MAUL',
     'MAYO',  'MEAD',  'MEAL',  'MEAN',  'MEAT',  'MEEK',  'MEET',  'MELD',
     'MELT',  'MEMO',  'MEND',  'MENU',  'MERT',  'MESH',  'MESS',  'MICE',
     'MIKE',  'MILD',  'MILE',  'MILK',  'MILL',  'MILT',  'MIMI',  'MIND',
     'MINE',  'MINI',  'MINK',  'MINT',  'MIRE',  'MISS',  'MIST',  'MITE',
     'MITT',  'MOAN',  'MOAT',  'MOCK',  'MODE',  'MOLD',  'MOLE',  'MOLL',
     'MOLT',  'MONA',  'MONK',  'MONT',  'MOOD',  'MOON',  'MOOR',  'MOOT',
     'MORE',  'MORN',  'MORT',  'MOSS',  'MOST',  'MOTH',  'MOVE',  'MUCH',
     'MUCK',  'MUDD',  'MUFF',  'MULE',  'MULL',  'MURK',  'MUSH',  'MUST',
     'MUTE',  'MUTT',  'MYRA',  'MYTH',  'NAGY',  'NAIL',  'NAIR',  'NAME',
     'NARY',  'NASH',  'NAVE',  'NAVY',  'NEAL',  'NEAR',  'NEAT',  'NECK',
     'NEED',  'NEIL',  'NELL',  'NEON',  'NERO',  'NESS',  'NEST',  'NEWS',
     'NEWT',  'NIBS',  'NICE',  'NICK',  'NILE',  'NINA',  'NINE',  'NOAH',
     'NODE',  'NOEL',  'NOLL',  'NONE',  'NOOK',  'NOON',  'NORM',  'NOSE',
     'NOTE',  'NOUN',  'NOVA',  'NUDE',  'NULL',  'NUMB',  'OATH',  'OBEY',
     'OBOE',  'ODIN',  'OHIO',  'OILY',  'OINT',  'OKAY',  'OLAF',  'OLDY',
     'OLGA',  'OLIN',  'OMAN',  'OMEN',  'OMIT',  'ONCE',  'ONES',  'ONLY',
     'ONTO',  'ONUS',  'ORAL',  'ORGY',  'OSLO',  'OTIS',  'OTTO',  'OUCH',
     'OUST',  'OUTS',  'OVAL',  'OVEN',  'OVER',  'OWLY',  'OWNS',  'QUAD',
     'QUIT',  'QUOD',  'RACE',  'RACK',  'RACY',  'RAFT',  'RAGE',  'RAID',
     'RAIL',  'RAIN',  'RAKE',  'RANK',  'RANT',  'RARE',  'RASH',  'RATE',
     'RAVE',  'RAYS',  'READ',  'REAL',  'REAM',  'REAR',  'RECK',  'REED',
     'REEF',  'REEK',  'REEL',  'REID',  'REIN',  'RENA',  'REND',  'RENT',
     'REST',  'RICE',  'RICH',  'RICK',  'RIDE',  'RIFT',  'RILL',  'RIME',
     'RING',  'RINK',  'RISE',  'RISK',  'RITE',  'ROAD',  'ROAM',  'ROAR',
     'ROBE',  'ROCK',  'RODE',  'ROIL',  'ROLL',  'ROME',  'ROOD',  'ROOF',
     'ROOK',  'ROOM',  'ROOT',  'ROSA',  'ROSE',  'ROSS',  'ROSY',  'ROTH',
     'ROUT',  'ROVE',  'ROWE',  'ROWS',  'RUBE',  'RUBY',  'RUDE',  'RUDY',
     'RUIN',  'RULE',  'RUNG',  'RUNS',  'RUNT',  'RUSE',  'RUSH',  'RUSK',
     'RUSS',  'RUST',  'RUTH',  'SACK',  'SAFE',  'SAGE',  'SAID',  'SAIL',
     'SALE',  'SALK',  'SALT',  'SAME',  'SAND',  'SANE',  'SANG',  'SANK',
     'SARA',  'SAUL',  'SAVE',  'SAYS',  'SCAN',  'SCAR',  'SCAT',  'SCOT',
     'SEAL',  'SEAM',  'SEAR',  'SEAT',  'SEED',  'SEEK',  'SEEM',  'SEEN',
     'SEES',  'SELF',  'SELL',  'SEND',  'SENT',  'SETS',  'SEWN',  'SHAG',
     'SHAM',  'SHAW',  'SHAY',  'SHED',  'SHIM',  'SHIN',  'SHOD',  'SHOE',
     'SHOT',  'SHOW',  'SHUN',  'SHUT',  'SICK',  'SIDE',  'SIFT',  'SIGH',
     'SIGN',  'SILK',  'SILL',  'SILO',  'SILT',  'SINE',  'SING',  'SINK',
     'SIRE',  'SITE',  'SITS',  'SITU',  'SKAT',  'SKEW',  'SKID',  'SKIM',
     'SKIN',  'SKIT',  'SLAB',  'SLAM',  'SLAT',  'SLAY',  'SLED',  'SLEW',
     'SLID',  'SLIM',  'SLIT',  'SLOB',  'SLOG',  'SLOT',  'SLOW',  'SLUG',
     'SLUM',  'SLUR',  'SMOG',  'SMUG',  'SNAG',  'SNOB',  'SNOW',  'SNUB',
     'SNUG',  'SOAK',  'SOAR',  'SOCK',  'SODA',  'SOFA',  'SOFT',  'SOIL',
     'SOLD',  'SOME',  'SONG',  'SOON',  'SOOT',  'SORE',  'SORT',  'SOUL',
     'SOUR',  'SOWN',  'STAB',  'STAG',  'STAN',  'STAR',  'STAY',  'STEM',
     'STEW',  'STIR',  'STOW',  'STUB',  'STUN',  'SUCH',  'SUDS',  'SUIT',
     'SULK',  'SUMS',  'SUNG',  'SUNK',  'SURE',  'SURF',  'SWAB',  'SWAG',
     'SWAM',  'SWAN',  'SWAT',  'SWAY',  'SWIM',  'SWUM',  'TACK',  'TACT',
     'TAIL',  'TAKE',  'TALE',  'TALK',  'TALL',  'TANK',  'TASK',  'TATE',
     'TAUT',  'TEAL',  'TEAM',  'TEAR',  'TECH',  'TEEM',  'TEEN',  'TEET',
     'TELL',  'TEND',  'TENT',  'TERM',  'TERN',  'TESS',  'TEST',  'THAN',
     'THAT',  'THEE',  'THEM',  'THEN',  'THEY',  'THIN',  'THIS',  'THUD',
     'THUG',  'TICK',  'TIDE',  'TIDY',  'TIED',  'TIER',  'TILE',  'TILL',
     'TILT',  'TIME',  'TINA',  'TINE',  'TINT',  'TINY',  'TIRE',  'TOAD',
     'TOGO',  'TOIL',  'TOLD',  'TOLL',  'TONE',  'TONG',  'TONY',  'TOOK',
     'TOOL',  'TOOT',  'TORE',  'TORN',  'TOTE',  'TOUR',  'TOUT',  'TOWN',
     'TRAG',  'TRAM',  'TRAY',  'TREE',  'TREK',  'TRIG',  'TRIM',  'TRIO',
     'TROD',  'TROT',  'TROY',  'TRUE',  'TUBA',  'TUBE',  'TUCK',  'TUFT',
     'TUNA',  'TUNE',  'TUNG',  'TURF',  'TURN',  'TUSK',  'TWIG',  'TWIN',
     'TWIT',  'ULAN',  'UNIT',  'URGE',  'USED',  'USER',  'USES',  'UTAH',
     'VAIL',  'VAIN',  'VALE',  'VARY',  'VASE',  'VAST',  'VEAL',  'VEDA',
     'VEIL',  'VEIN',  'VEND',  'VENT',  'VERB',  'VERY',  'VETO',  'VICE',
     'VIEW',  'VINE',  'VISE',  'VOID',  'VOLT',  'VOTE',  'WACK',  'WADE',
     'WAGE',  'WAIL',  'WAIT',  'WAKE',  'WALE',  'WALK',  'WALL',  'WALT',
     'WAND',  'WANE',  'WANG',  'WANT',  'WARD',  'WARM',  'WARN',  'WART',
     'WASH',  'WAST',  'WATS',  'WATT',  'WAVE',  'WAVY',  'WAYS',  'WEAK',
     'WEAL',  'WEAN',  'WEAR',  'WEED',  'WEEK',  'WEIR',  'WELD',  'WELL',
     'WELT',  'WENT',  'WERE',  'WERT',  'WEST',  'WHAM',  'WHAT',  'WHEE',
     'WHEN',  'WHET',  'WHOA',  'WHOM',  'WICK',  'WIFE',  'WILD',  'WILL',
     'WIND',  'WINE',  'WING',  'WINK',  'WINO',  'WIRE',  'WISE',  'WISH',
     'WITH',  'WOLF',  'WONT',  'WOOD',  'WOOL',  'WORD',  'WORE',  'WORK',
     'WORM',  'WORN',  'WOVE',  'WRIT',  'WYNN',  'YALE',  'YANG',  'YANK',
     'YARD',  'YARN',  'YAWL',  'YAWN',  'YEAH',  'YEAR',  'YELL',  'YOGA',
     'YOKE');

Function RandomPseudoword (const Length : Integer) : String;
var I, J, L : Integer;
  Begin
    I := Length;
    Result := '';
    While I > 0 do
      begin
        if I <= 2 then
          begin
            if I = 1 then
              Result := Result + IntToStr (Round (RandomUniformF * 9)) else
              Result := Result + IntToStr (Round (RandomUniformF * 99));
            I := 0;
          end else
          begin
            Repeat
              J := Round (RandomUniformF * (WordNibbles - 1)) + 1;
              L := System.Length (WordNibble [J]);
            Until L <= I;
            Result := Result + WordNibble [J];
            Dec (I, L);
          end;
      end;
  End;


{ Calculates polynomial of degree N:        }
{                                           }
{                      2          N         }
{  y  =  C  + C x + C x  +...+ C x          }
{         0    1     2          N           }
{  Coefficients are stored in reverse order }
Function PolEvl (X : Extended; var Coef : array of Extended; N : Integer) : Extended;
var Ans, P : Extended;
  Begin
    P := 1.0;
    Ans := 0.0;
    While N >= 0 do
      begin
        Ans := Ans + Coef [N] * P;
        P := P * X;
        Dec (N);
      end;
    PolEvl := Ans;
  End;



{ "For arguments greater than 13, the logarithm of the gamma      }
{ function is approximated by the logarithmic version of          }
{ Stirling's formula using a polynomial approximation of          }
{ degree 4. Arguments between -33 and +33 are reduced by          }
{ recurrence to the interval [2,3] of a rational approximation.   }
{ The cosecant reflection formula is employed for arguments       }
{ less than -33.                                                  }
{                                                                 }
{ Arguments greater than MAXLGM return MAXNUM and an error        }
{ message."                                                       }
{                                                                 }
{ Algorithm translated into Delphi by David Butler                }
{ <david@e.co.za> from Cephes C library by Stephen L. Moshier     }
{ <moshier@na-net.ornl.gov>.                                      }
Function GammaLn (X : Extended) : Extended;
const MaxLGM = 2.556348e305;
var P, Q, W, Z : Extended;
{ Stirling's formula expansion of log gamma }
const Stir : Array [0..4] of Extended = (
              8.11614167470508450300E-4,
             -5.95061904284301438324E-4,
              7.93650340457716943945E-4,
             -2.77777777730099687205E-3,
              8.33333333333331927722E-2);
{ B[], C[]: log gamma function between 2 and 3 }
      B    : Array [0..5] of Extended = (
             -1.37825152569120859100E3,
             -3.88016315134637840924E4,
             -3.31612992738871184744E5,
             -1.16237097492762307383E6,
             -1.72173700820839662146E6,
             -8.53555664245765465627E5);
      C    : Array [0..7] of Extended = (
              1.00000000000000000000E0,
             -3.51815701436523470549E2,
             -1.70642106651881159223E4,
             -2.20528590553854454839E5,
             -1.13933444367982507207E6,
             -2.53252307177582951285E6,
             -2.01889141433532773231E6,
              1.00000000000000000000E0);

  Begin
    if X < -34.0 then
      begin
	Q := -X;
	W := GammaLn (Q);
	P := Trunc (Q);
        if P = Q then
          raise EOverflow.Create ('GammaLn') else
          begin
            Z := Q - P;
            if Z > 0.5 then
              begin
                P := P + 1.0;
                Z := P - Q;
              end;
            Z := Q * Sin (Pi * Z);
            if Z = 0.0 then
              raise EOverflow.Create ('GammaLn') else
              GammaLn := LnPi - Ln (Z) - W;
          end;
      end else
    if X <= 13 then
      begin
	Z := 1.0;
        While X >= 3.0 do
          begin
	    X := X - 1.0;
            Z := Z * X;
          end;
        While (X < 2.0) and (X <> 0.0) do
          begin
            Z := Z / X;
            X := X + 1.0;
          end;
        if X = 0.0 then
          raise EOverflow.Create ('GammaLn') else
	if Z < 0.0 then
          Z := -Z;
        if X = 2.0 then
          GammaLn := Ln (Z) else
          begin
            X := X - 2.0;
            P := X * PolEvl (X, B, 5) / PolEvl (X, C, 7);
            GammaLn := Ln (Z) + P;
          end;
      end else
    if X > MAXLGM then
      raise EOverflow.Create ('GammaLn') else
      begin
        Q := (X - 0.5) * Ln (X) - X + LnSqrt2Pi;
        if X > 1.0e8 then
          GammaLn := Q else
          begin
            P := 1.0 / (X * X);
            if X >= 1000.0 then
              GammaLn := Q + ((7.9365079365079365079365e-4 * P
                            - 2.7777777777777777777778e-3)
                            * P + 0.0833333333333333333333) / X else
              GammaLn := Q + PolEvl (P, Stir, 4) / X;
          end;
      end;
  End;

{ gamma function, incomplete, series evaluation                                }
{ The gamma functions were translated from "NUMERICAL RECIPES".                }
Procedure gser (const a, x : Extended; var gamser, gln : Extended);
const itmax = 100;
      eps   = 3.0e-7;
var n : Integer;
    sum, del, ap : Extended;
  Begin
    gln := GammaLn (a);
    if FloatZero (x) then
      GamSer := 0.0 else
      if X < 0.0 then
        raise EUnderflow.Create ('gser') else
        begin
          ap := a;
          sum := 1.0 / a;
          del := sum;
          for n := 1 to itmax do
            begin
              ap := ap + 1.0;
              del := del * x / ap;
              sum := sum + del;
              if abs (del) < abs (sum) * eps then
                begin
                  GamSer := sum * exp (-x + a * ln(x) - gln);
                  exit;
                end;
            end;
          raise EOverflow.Create ('gser: A too large, itmax too small');
        end;
  End;

{ gamma function, incomplete, continued fraction evaluation                    }
Procedure gcf (const a, x : Extended; var gammcf, gln : Extended);
const itmax = 100;
      eps   = 3.0e-7;
var n : integer;
    gold, g, fac, b1, b0, anf, ana, an, a1, a0 : Extended;
  Begin
    gln := GammaLn (a);
    gold := 0.0;
    g := 0.0;
    a0 := 1.0;
    a1 := x;
    b0 := 0.0;
    b1 := 1.0;
    fac := 1.0;
    For n := 1 to itmax do
      begin
        an := 1.0 * n;
        ana := an - a;
        a0 := (a1 + a0 * ana) * fac;
        b0 := (b1 + b0 * ana) * fac;
        anf := an * fac;
        a1 := x * a0 + anf * a1;
        b1 := x * b0 + anf * b1;
        if not FloatZero (a1) then
          begin
            fac := 1.0 / a1;
            g := b1 * fac;
            if abs ((g - gold) / g) < eps then
              break;
            gold := g;
          end;
      end;
    Gammcf := exp (-x + a * ln (x) - gln) * g;
  End;

{ GAMMP  gamma function, incomplete                                            }
Function GammP (const a,x : Extended) : Extended;
var gammcf, gln : Extended;
  Begin
    if (x < 0.0) or (a <= 0.0) then
      raise EInvalidArgument.Create ('GammP') else
      if x < a + 1.0 then
        begin
          gser (a, x, gammcf, gln);
          gammp := gammcf
        end else
        begin
          gcf (a, x, gammcf, gln);
          gammp := 1.0 - gammcf
        end;
  End;

{ gamma function, incomplete, complementary                                    }
Function gammq (const a, x : Extended) : Extended;
var gamser, gln : Extended;
  Begin
    if (x < 0.0) or (a <= 0.0) then
      raise EInvalidArgument.Create ('gammq');
    if x < a + 1.0 then
      begin
        gser (a, x, gamser, gln);
        Result := 1.0 - gamser;
      end else
      begin
        gcf (a, x, gamser, gln);
        Result := gamser
      end;
  End;

{ error function, complementary                                                }
Function erfc (const x : Extended) : Extended;
  Begin
    if x < 0.0 then
      erfc := 1.0 + gammp (0.5, sqr (x)) else
      erfc := gammq (0.5, sqr(x));
  End;

Function CummNormal (const u, s, X : Extended) : Extended;
  Begin
    CummNormal := ERFC (((X - u) / s) / Sqrt2) / 2.0;
  End;

Function CummNormal01 (const X : Extended) : Extended;
  Begin
    CummNormal01 := ERFC (X / Sqrt2) / 2.0;
  End;

Function CummChiSquare (const Chi, Df : Extended) : Extended;
  Begin
    CummChiSquare := 1.0 - gammq (0.5 * Df, 0.5 * Chi);
  End;

{ BETACF  beta function, incomplete, continued fraction evaluation             }
{ The beta functions were translated from "NUMERICAL RECIPES".                 }
Function betacf (const a, b, x : Extended) : Extended;
const itmax = 100;
      eps   = 3.0e-7;
var tem, qap, qam, qab, em, d : Extended;
    bz, bpp, bp, bm, az, app  : Extended;
    am, aold, ap              : Extended;
    m                         : Integer;
  Begin
    am := 1.0;
    bm := 1.0;
    az := 1.0;
    qab := a + b;
    qap := a + 1.0;
    qam := a - 1.0;
    bz := 1.0 - qab * x / qap;
    For m := 1 to itmax do
      begin
        em := m;
        tem := em + em;
        d := em * (b - m) * x / ((qam + tem) * (a + tem));
        ap := az + d * am;
        bp := bz + d * bm;
        d := -(a + em) * (qab + em) * x / ((a + tem) * (qap + tem));
        app := ap + d * az;
        bpp := bp + d * bz;
        aold := az;
        am := ap / bpp;
        bm := bp / bpp;
        az := app / bpp;
        bz := 1.0;
        if abs (az - aold) < eps * abs (az) then
          begin
            Result := az;
            exit;
          end;
      end;
    raise EOverflow.Create ('betacf: A or B too big or itmax too small');
  End;

{ BETAI  beta function, incomplete                                             }
Function betai (const a, b, x : Extended) : Extended;
var bt : Extended;
  Begin
    if (x < 0.0) or (x > 1.0) then
      raise EInvalidArgument.Create ('betai');

    if FloatZero (x) or FloatEqual (x, 1.0) then
      bt := 0.0 else
      bt := exp (GammaLn (a + b) - GammaLn (a) - GammaLn (b) + a * ln (x) + b * ln (1.0 - x));
    if x < (a + 1.0) / (a + b + 2.0) then
      Result := bt * betacf (a, b, x) / a else
      Result := 1.0 - bt * betacf (b, a, 1.0 - x) / b;
  End;

Function CumF (const f, Df1, Df2 : Extended) : Extended;
  Begin
    if F <= 0.0 then
      raise EInvalidArgument.Create ('') else
      CumF := 1.0 - (betai (0.5 * df2, 0.5 * df1, df2 / (df2 + df1 * f))
           + (1.0 - betai (0.5 * df1, 0.5 * df2, df1 / (df1 + df2 / f)))) / 2.0;
  End;



{                                                                   }
{ Returns the argument, x, for which the area under the             }
{ Gaussian probability density function (integrated from            }
{ minus infinity to x) is equal to y.                               }
{                                                                   }
{ For small arguments 0 < y < exp(-2), the program computes         }
{ z = sqrt( -2.0 * log(y) );  then the approximation is             }
{ x = z - log(z)/z  - (1/z) P(1/z) / Q(1/z).                        }
{ There are two rational functions P/Q, one for 0 < y < exp(-32)    }
{ and the other for y up to exp(-2).  For larger arguments,         }
{ w = y - 0.5, and  x/sqrt(2pi) = w + w**3 R(w**2)/S(w**2)).        }
{                                                                   }
{ Algorithm translated into Delphi by David Butler                  }
{ <david@e.co.za> from Cephes C library by Stephen L. Moshier       }
{ <moshier@na-net.ornl.gov>.                                        }
Function InvCummNormal01 (Y0 : Extended) : Extended;
const P0 : Array [0..4] of Extended = (
           -5.99633501014107895267e1,
            9.80010754185999661536e1,
           -5.66762857469070293439e1,
            1.39312609387279679503e1,
           -1.23916583867381258016e0);
      Q0 : Array [0..8] of Extended = (
            1.00000000000000000000e0,
            1.95448858338141759834e0,
            4.67627912898881538453e0,
            8.63602421390890590575e1,
           -2.25462687854119370527e2,
            2.00260212380060660359e2,
           -8.20372256168333339912e1,
            1.59056225126211695515e1,
           -1.18331621121330003142e0);
      P1 : Array [0..8] of Extended = (
            4.05544892305962419923e0,
            3.15251094599893866154e1,
            5.71628192246421288162e1,
            4.40805073893200834700e1,
            1.46849561928858024014e1,
            2.18663306850790267539e0,
           -1.40256079171354495875e-1,
           -3.50424626827848203418e-2,
           -8.57456785154685413611e-4);
      Q1 : Array [0..8] of Extended = (
            1.00000000000000000000e0,
            1.57799883256466749731e1,
            4.53907635128879210584e1,
            4.13172038254672030440e1,
            1.50425385692907503408e1,
            2.50464946208309415979e0,
           -1.42182922854787788574e-1,
           -3.80806407691578277194e-2,
           -9.33259480895457427372e-4);
      P2 : Array [0..8] of Extended = (
            3.23774891776946035970e0,
            6.91522889068984211695e0,
            3.93881025292474443415e0,
            1.33303460815807542389e0,
            2.01485389549179081538e-1,
            1.23716634817820021358e-2,
            3.01581553508235416007e-4,
            2.65806974686737550832e-6,
            6.23974539184983293730e-9);
      Q2 : Array [0..8] of Extended = (
            1.00000000000000000000e0,
            6.02427039364742014255e0,
            3.67983563856160859403e0,
            1.37702099489081330271e0,
            2.16236993594496635890e-1,
            1.34204006088543189037e-2,
            3.28014464682127739104e-4,
            2.89247864745380683936e-6,
            6.79019408009981274425e-9);
var X, Z, Y2, X0, X1 : Extended;
    Code             : Boolean;
  Begin
    if Y0 <= 0.0 then
      raise EUnderflow.Create ('InvCummNormal01') else
      if Y0 >= 1.0 then
        raise EOverflow.Create ('InvCummNormal01') else
        begin
          Code := True;
          if Y0 > 1.0 - ExpM2 then
            begin
              Y0 := 1.0 - Y0;
              Code := False;
            end;
          if Y0 > ExpM2 then
            begin
              Y0 := Y0 - 0.5;
              Y2 := Y0 * Y0;
              X := Y0 + Y0 * (Y2 * PolEvl (Y2, P0, 4) / PolEvl (Y2, Q0, 8));
              InvCummNormal01 := X * Sqrt2Pi;
            end else
            begin
              X := Sqrt (-2.0 * Ln (Y0));
              X0 := X - Ln (X) / X;
              Z := 1.0 / X;
              if X < 8.0 then
                X1 := Z * PolEvl (Z, P1, 8) / PolEvl (Z, Q1, 8) else
                X1 := Z * PolEvl (Z, P2, 8) / PolEvl (Z, Q2, 8);
              X := X0 - X1;
              if Code then
                X := -X;
              InvCummNormal01 := X;
            end;
        end;
  End;

Function InvCummNormal (const u, s, Y0 : Extended) : Extended;
  Begin
    InvCummNormal := InvCummNormal01 (Y0) * s + u;
  End;

Function CummPoisson (const X : Integer; const u : Extended) : Extended;
  Begin
    CummPoisson := GammQ (X + 1, u);
  End;

{                                                                              }
{ Huge Cardinals                                                               }
{                                                                              }
type
  THugeCardinal = CardinalArray;

Procedure HugeCardinalPack (var R : THugeCardinal);
var I, L : Integer;
  Begin
    L := Length (R);
    I := L;
    While (I > 1) and (R [I - 1] = 0) do
      Dec (I);
    if I < L then
      SetLength (R, I);
  End;

{ Pre: X and Y is packed.                                                      }
Function HugeCardinalCompare (const X, Y : THugeCardinal) : TCompareResult;
var I, L1, L2 : Integer;
    A, B      : Cardinal;
  Begin
    L1 := Length (X);
    L2 := Length (Y);
    if L1 < L2 then
      Result := crLess else
    if L1 > L2 then
      Result := crGreater else
      begin
        For I := L1 - 1 downto 0 do
          begin
            A := X [I];
            B := Y [I];
            if A < B then
              begin
                Result := crLess;
                exit;
              end else
            if A > B then
              begin
                Result := crGreater;
                exit;
              end;
          end;
        Result := crEqual;
      end;
  End;

{ Post: R packed.                                                              }
Procedure HugeCardinalAdd (var R : THugeCardinal; const O : THugeCardinal);
  Begin
    SetLength (R, Max (Length (O), Length (R)) + 1);
    asm
      push esi
      push edi
      push ebx
      push eax

      mov esi, O
      mov ecx, [esi - 4]                        // ecx = Length (O)

      mov edi, R
      mov edi, [edi]

      xor ebx, ebx                              // Add loop
      clc                                       //
    @NextCard:                                  //
      mov eax, [esi + ebx * 4]                  //
      adc [edi + ebx * 4], eax                  //
      inc ebx                                   //
      loop @NextCard                            //

      mov eax, 0                                // Carry loop
    @NextCarry:                                 //
      adc [edi + ebx * 4], eax                  //
      inc ebx                                   //
      jc @NextCarry                             //

      pop eax
      pop ebx
      pop edi
      pop esi
    end;
    HugeCardinalPack (R);
  End;

{ Pre:  R >= O.                                                                }
{ Post: R packed.                                                              }
Procedure HugeCardinalSubtract (var R : THugeCardinal; const O : THugeCardinal);
  Begin
    asm
      push esi
      push edi
      push ebx
      push eax

      mov esi, O
      mov ecx, [esi - 4]                        // ecx = Length (O)

      mov edi, R
      mov edi, [edi]

      xor ebx, ebx                              // Subtract loop
      clc                                       //
    @NextCard:                                  //
      mov eax, [esi + ebx * 4]                  //
      sbb [edi + ebx * 4], eax                  //
      inc ebx                                   //
      loop @NextCard                            //

      jnc @Fin
      mov eax, 0                                // Carry loop
    @NextCarry:                                 //
      sbb [edi + ebx * 4], eax                  //
      inc ebx                                   //
      jc @NextCarry                             //

    @Fin:
      pop eax
      pop ebx
      pop edi
      pop esi
    end;
    HugeCardinalPack (R);
 End;

{ Post: R packed.                                                              }
Procedure HugeCardinalMultiply (var R : THugeCardinal; const O : Cardinal);
  Begin
    SetLength (R, Length (R) + 1);
    asm
      push esi
      push edi
      push eax
      push ebx
      push edx

      mov edi, R
      mov edi, [edi]                            // edi = R
      mov ecx, [edi - 4]                        // ecx = Length (R)

      xor ebx, ebx                              // Multiply loop
      xor esi, esi                              //
      clc                                       //
    @NextCard:                                  //
      mov eax, [edi + ebx * 4]                  //
      pushf                                     //
      mul O                                     //
      popf                                      //
      adc eax, esi                              //
      mov [edi + ebx * 4], eax                  //
      mov esi, edx                              //
      inc ebx                                   //
      loop @NextCard                            //

      pop edx
      pop ebx
      pop eax
      pop edi
      pop esi
    end;
    HugeCardinalPack (R);
  End;

{                                                                              }
{ THugeInteger                                                                 }
{                                                                              }
Constructor THugeInteger.Create (const X : Integer);
  Begin
    inherited Create;
    AsInteger := X;
  End;

Procedure THugeInteger.AssignZero;
  Begin
    SetLength (FValue, 1);
    FValue [0] := 0;
    FNegative := False;
  End;

Procedure THugeInteger.SetAsInteger (const X : Integer);
  Begin
    SetLength (FValue, 1);
    FValue [0] := Abs (X);
    FNegative := X < 0;
  End;

Procedure THugeInteger.Negate;
  Begin
    FNegative := not FNegative;
  End;

Procedure THugeInteger.Add (const X : THugeInteger);
var Y : THugeCardinal;
  Begin
    if FNegative = X.FNegative then
      HugeCardinalAdd (FValue, X.FValue) else
      Case HugeCardinalCompare (FValue, X.FValue) of
        crEqual   : AssignZero;
        crGreater : HugeCardinalSubtract (FValue, X.FValue);
        crLess    : begin
                      Y := Copy (X.FValue);
                      HugeCardinalSubtract (Y, FValue);
                      FValue := Y;
                      FNegative := not FNegative;
                    end;
      end;
  End;

Procedure THugeInteger.Add (const X : Integer);
var H : THugeInteger;
  Begin
    H := THugeInteger.Create (X);
    Add (H);
    FreeAndNil (H);
  End;

Procedure THugeInteger.Subtract (const X : THugeInteger);
var Y : THugeInteger;
  Begin
    Y := THugeInteger.Create;
    Y.FValue := X.FValue; // assign reference
    Y.FNegative := not X.FNegative;
    Add (Y);
    FreeAndNil (Y);
  End;

Procedure THugeInteger.Multiply (const X : Integer);
  Begin
    HugeCardinalMultiply (FValue, Abs (X));
    if X < 0 then
      FNegative := not FNegative;
  End;

Function THugeInteger.GetAsInteger : Integer;
  Begin
    if (Length (FValue) > 1) or (FValue [0] > High (Integer)) then
      raise EOverflow.Create ('Value too large to assign to Integer');

    if FNegative then
      Result := -FValue [0] else
      Result := FValue [0];
  End;

initialization
finalization
  FreeAndNil (PrimeSet);
End. { Math }

