{$INCLUDE cHeader.inc}
unit cODSOrg;

{                                                                              }
{                  ods.org Dynamic Domain updater v0.02 (L1)                   }
{                                                                              }
{        This unit is copyright  2000 by David Butler (david@e.co.za)         }
{                                                                              }
{                  This unit is part of Delphi Fundamentals.                   }
{                   It's original file name is cODSOrg.pas                     }
{                                                                              }
{                I invite you to use this unit, free of charge.                }
{        I invite you to distibute this unit, but it must be for free.         }
{             I also invite you to contribute to its development,              }
{             but do not distribute a modified copy of this file.              }
{       Send modifications, suggestions and bug reports to david@e.co.za       }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   2000/06/18  v0.01  Initial version                                         }
{   2000/06/22  v0.02  Cleanup code.                                           }
{                                                                              }
interface

uses
  SysUtils,

  // Delphi Fundamentals
  cStreams;



{                                                                              }
{ TODSOrgUpdater                                                               }
{   Before connecting, set UserName and Password.                              }
{   Call AddDynamicDomain for each domain to use. Leave IP blank to tell the   }
{   ODS server to use the IP of the incomming connection (CONNIP).             }
{   Domains can also be added by setting the DomainIP or DomainActive          }
{   properties.                                                                }
{   Call ActivateDomains/DeactivateDomains to activate/deactivate all the      }
{   dynamic domains in one session with the server.                            }
{   Set the DomainActive property to activate/deactivate a single domain in a  }
{   session with the server.                                                   }
{   Return from a method assumes the method was successful, otherwise an       }
{   exception will be raised.                                                  }
{                                                                              }
{   Examples:                                                                  }
{     With TODSOrgUpdater.Create do                                            }
{       try                                                                    }
{         DomainActive ['dj.nailed.org'] := True;                              }
{       finally                                                                }
{         Free;                                                                }
{       end;                                                                   }
{                                                                              }
{     With TODSOrgUpdater.Create do                                            }
{       try                                                                    }
{         AddDomain ('dj.nailed.org', '');                                     }
{         AddDomain ('dummy.ods.org', '127.0.0.1');                            }
{         ActivateDomains;                                                     }
{       finally                                                                }
{         Free;                                                                }
{       end;                                                                   }
{                                                                              }
{     With TODSOrgUpdater.Create do                                            }
{       try                                                                    }
{         DomainIP ['dj.nailed.org'] := '';                                    }
{         DomainIP ['dummy.ods.org'] := '127.0.0.1';                           }
{         DeactivateDomains;                                                   }
{       finally                                                                }
{         Free;                                                                }
{       end;                                                                   }
{                                                                              }
const
  // Status codes
  ODS_Connecting      = -1;
  ODS_ConnectFailed   = -2;
  ODS_Connected       = -3;
  ODS_LoggingIn       = -4;
  ODS_LoggedIn        = -5;
  ODS_Deactivating    = -6;
  ODS_Deactivated     = -7;
  ODS_Activating      = -8;
  ODS_Activated       = -9;

  // Status codes: Server replies
  ODS_RPL_ServerReady          = 100;
  ODS_RPL_ServerBusy           = 999;
  ODS_RPL_LoginSuccessful      = 225;
  ODS_RPL_AdminLoginSuccessful = 226;
  ODS_RPL_AddedOK              = 795;
  ODS_RPL_DeletedOK            = 901;
  ODS_RPL_UnableToDelete       = 903;

type
  TDomainEntry = record
    Domain : String;
    IP     : String;
    Active : Boolean;
  end;
  EODSOrgUpdater = class (Exception);
  TODSOrgUpdater = class; // forward
  TODSOrgOnDomainActivated = Procedure (const Sender : TODSOrgUpdater; const Domain : String);
  TODSOrgOnDomainDeactivated = Procedure (const Sender : TODSOrgUpdater; const Domain : String);
  TODSOrgOnStatus = Procedure (const Sender : TODSOrgUpdater; const StatusCode : Integer; const StatusMessage : String) of object;
  TODSOrgUpdater = class
    protected
    FODSServer           : String;
    FODSPort             : Integer;
    FUserName            : String;
    FPassword            : String;
    FDomains             : Array of TDomainEntry;
    FConn                : TTextParser;
    FStatusCode          : Integer;
    FStatusMessage       : String;
    FOnStatus            : TODSOrgOnStatus;
    FOnDomainActivated   : TODSOrgOnDomainActivated;
    FOnDomainDeactivated : TODSOrgOnDomainDeactivated;

    Function  GetReply : Integer;
    Function  Connect : Boolean;
    Procedure Disconnect;
    Function  DeactivateDomain (const DomainIdx : Integer) : Boolean;
    Function  ActivateDomain (const DomainIdx : Integer) : Boolean;
    Procedure SetStatus (const StatusCode : Integer; const StatusMessage : String);
    Function  GetDomainActive (const Domain : String) : Boolean;
    Procedure SetDomainActive (const Domain : String; const Active : Boolean);
    Function  GetDomainIP (const Domain : String) : String;
    Procedure SetDomainIP (const Domain : String; const IP : String);
    Function  IndexOf (const Domain : String) : Integer;

    public
    Constructor Create;
    Destructor Destroy; override;

    Property  ODSServer : String read FODSServer write FODSServer;
    Property  ODSPort : Integer read FODSPort write FODSPort;
    Property  UserName : String read FUserName write FUserName;
    Property  Password : String read FPassword write FPassword;

    Procedure AddDynamicDomain (const Domain : String; const IP : String = '');
    Procedure RemoveDynamicDomain (const Domain : String);
    Procedure ActivateDomains;
    Procedure DeactivateDomains;
    Property  DomainActive [const Domain : String] : Boolean read GetDomainActive write SetDomainActive;
    Property  DomainIP [const Domain : String] : String read GetDomainIP write SetDomainIP;

    Property  OnStatus : TODSOrgOnStatus read FOnStatus write FOnStatus;
    Property  OnDomainActivated : TODSOrgOnDomainActivated read FOnDomainActivated write FOnDomainActivated;
    Property  OnDomainDeactivated : TODSOrgOnDomainDeactivated read FOnDomainDeactivated write FOnDomainDeactivated;
    Property  StatusCode : Integer read FStatusCode;
    Property  StatusMessage : String read FStatusMessage;
  end;

implementation

uses
  // Delphi Fundamentals
  cUtils,
  cStrings;


Constructor TODSOrgUpdater.Create;
  Begin
    inherited Create;
    FODSServer := 'update.ods.org';
    FODSPort := 7070;
  End;

Destructor TODSOrgUpdater.Destroy;
  Begin
    FreeAndNil (FConn);
    inherited Destroy;
  End;

Procedure TODSOrgUpdater.AddDynamicDomain (const Domain : String; const IP : String);
var L : Integer;
  Begin
    L := Length (FDomains);
    SetLength (FDomains, L + 1);
    FDomains [L].Domain := Domain;
    FDomains [L].IP := IP;
  End;

Procedure TODSOrgUpdater.RemoveDynamicDomain (const Domain : String);
var I, J, L : Integer;
  Begin
    I := IndexOf (Domain);
    if I >= 0 then
      begin
        L := Length (FDomains);
        For J := I to L - 2 do
          FDomains [J] := FDomains [J + 1];
        SetLength (FDomains, L - 1);
      end;
  End;

Function TODSOrgUpdater.IndexOf (const Domain : String) : Integer;
var I : Integer;
  Begin
    For I := 0 to Length (FDomains) - 1 do
      if FDomains [I].Domain = Domain then
        begin
          Result := I;
          exit;
        end;
    Result := -1;
  End;

Function TODSOrgUpdater.GetDomainIP (const Domain : String) : String;
var I : Integer;
  Begin
    I := IndexOf (Domain);
    if I = -1 then
      raise EODSOrgUpdater.Create ('Domain not found.');
    Result := FDomains [I].IP;
  End;

Function TODSOrgUpdater.GetDomainActive (const Domain : String) : Boolean;
var I : Integer;
  Begin
    I := IndexOf (Domain);
    if I = -1 then
      raise EODSOrgUpdater.Create ('Domain not found.');
    Result := FDomains [I].Active;
  End;

Procedure TODSOrgUpdater.SetDomainIP (const Domain : String; const IP : String);
var I : Integer;
  Begin
    I := IndexOf (Domain);
    if I = -1 then
      AddDynamicDomain (Domain, IP) else
      FDomains [I].IP := IP;
  End;

Procedure TODSOrgUpdater.SetStatus (const StatusCode : Integer; const StatusMessage : String);
  Begin
    FStatusCode := StatusCode;
    FStatusMessage := StatusMessage;
    if Assigned (FOnStatus) then
      FOnStatus (self, StatusCode, StatusMessage);
  End;

Function TODSOrgUpdater.GetReply : Integer;
var L, Msg : String;
  Begin
    try
      L := FConn.ExtractLine;
    except
      on E : Exception do raise EODSOrgUpdater.Create ('No reply from the server (' + E.Message + ').');
    end;
    try
      Msg := CopyAfter (L, ' ');
      Result := StrToInt (CopyBefore (L, ' '));
    except
      Result := -1;
    end;
    SetStatus (Result, Msg);
  End;

Function TODSOrgUpdater.Connect : Boolean;
var ReplyCode : Integer;
    S         : String;
  Begin
    if (FUserName = '') or (FPassword = '') then
      raise EODSOrgUpdater.Create ('Username and password not set.');
    if Length (FDomains) = 0 then
      raise EODSOrgUpdater.Create ('No domains set.');

    S := FODSServer;
    SetStatus (ODS_Connecting, 'Connecting to ' + S + '.');
    try
      FConn := TTextParser.Create (TSocketStream.Create (FODSServer, FODSPort, 5 * 60 * 1000, True));
    except
      on E : Exception do
        begin
          S := 'Connect failed (' + E.Message + ').';
          SetStatus (ODS_ConnectFailed, S);
          raise EODSOrgUpdater.Create (S);
        end;
    end;
    try
      FConn.EOLType := [eol_CRLF];
      SetStatus (ODS_Connected, 'Connected to ' + S + '.');

      if GetReply <> ODS_RPL_ServerReady then
        raise EODSOrgUpdater.Create ('Server not ready.');

      SetStatus (ODS_LoggingIn, 'Logging in.');
      FConn.Stream.Write ('LOGIN ' + FUserName + ' ' + FPassword + #13);
      ReplyCode := GetReply;
      if (ReplyCode <> ODS_RPL_LoginSuccessful) and (ReplyCode <> ODS_RPL_AdminLoginSuccessful) then
        raise EODSOrgUpdater.Create ('Login failed.');
      SetStatus (ODS_LoggedIn, 'Logged in.');
    except
      Disconnect;
      raise;
    end;
  End;

Procedure TODSOrgUpdater.Disconnect;
  Begin
    if Assigned (FConn) then
      try
        FConn.Stream.Write ('QUIT'#13);
      except end;
    FreeAndNil (FConn);
  End;

Function TODSOrgUpdater.DeactivateDomain (const DomainIdx : Integer) : Boolean;
var ReplyCode : Integer;
  Begin
    With FDomains [DomainIdx] do
      begin
        SetStatus (ODS_Deactivating, 'Deactivating ' + Domain + '.');
        FConn.Stream.Write ('DELRR ' + Domain + ' A' + #13);
        ReplyCode := GetReply;
        Active := False;
        if Assigned (FOnDomainDeactivated) then
          FOnDomainDeactivated (self, Domain);
        SetStatus (ODS_Deactivated, 'Deactivated ' + Domain + '.');
      end;
  End;

Function TODSOrgUpdater.ActivateDomain (const DomainIdx : Integer) : Boolean;
var ReplyCode : Integer;
    State     : String;
  Begin
    With FDomains [DomainIdx] do
      begin
        State := 'Activating ' + Domain + '.';
        SetStatus (ODS_Activating, State);
        try
          FConn.Stream.Write ('DELRR ' + Domain + ' A' + #13);
          ReplyCode := GetReply;
        except end;
        SetStatus (ODS_Activating, State + '.');
        FConn.Stream.Write ('ADDRR ' + Domain + ' A ' + Cond (IP = '', 'CONNIP', IP) + #13);
        ReplyCode := GetReply;
        if ReplyCode <> ODS_RPL_AddedOK then
          raise EODSOrgUpdater.Create ('Domain not activated.');
        Active := True;
        if Assigned (FOnDomainActivated) then
          FOnDomainActivated (self, Domain);
        SetStatus (ODS_Activated, 'Activated ' + Domain + '.');
      end;
  End;

Procedure TODSOrgUpdater.SetDomainActive (const Domain : String; const Active : Boolean);
var I : Integer;
  Begin
    I := IndexOf (Domain);
    if I = -1 then
      begin
        AddDynamicDomain (Domain, '');
        I := Length (FDomains) - 1;
      end else
      if Active = FDomains [I].Active then
        exit;

    Connect;
    try
      if Active then
        ActivateDomain (I) else
        DeactivateDomain (I);
    finally
      Disconnect;
    end;
  End;

Procedure TODSOrgUpdater.ActivateDomains;
var I : Integer;
    Msg : String;
  Begin
    Connect;
    try
      Msg := '';
      For I := 0 to Length (FDomains) - 1 do
        With FDomains [I] do
          if not Active then
            try
              ActivateDomain (I);
            except
              on E : Exception do
                Msg := Msg + Cond (Msg <> '', #13#10, '') + Domain + ' (' + E.Message + ')';
            end;
      if Msg <> '' then
        raise EODSOrgUpdater.Create ('Some or all domains not activated:' + #13#10 + Msg);
    finally
      Disconnect;
    end;
  End;

Procedure TODSOrgUpdater.DeactivateDomains;
var I : Integer;
    Msg : String;
  Begin
    Connect;
    try
      Msg := '';
      For I := 0 to Length (FDomains) - 1 do
        With FDomains [I] do
          if not Active then
            try
              DeactivateDomain (I);
            except
              on E : Exception do
                Msg := Msg + Cond (Msg <> '', #13#10, '') + Domain + ' (' + E.Message + ')';
            end;
      if Msg <> '' then
        raise EODSOrgUpdater.Create ('Some or all domains not deactivated:' + #13#10 + Msg);
    finally
      Disconnect;
    end;
  End;


end.

