{$I cHeader.inc}
unit cInternetStandards;

interface

uses
  // System units
  SysUtils,
  Controls,      // TControl
  Graphics,      // TColor

  // L0 units
  cStrings,
  cStreams;



{                                                                              }
{ Internet Standards v0.12 (L1)                                                }
{                                                                              }
{   Implementation of various RFC standards                                    }
{                                                                              }
{ This unit is copyrighted  2000 by David Butler (david@e.co.za)              }
{                                                                              }
{ This unit is part of Delphi Fundamentals, it's original name is              }
{ cInternetStandards.                                                          }
{                                                                              }
{ I invite you to use this unit, free of charge.                               }
{ I invite you to distibute this unit, but it must be for free.                }
{ I also invite you to contribute to its development, but do not distribute    }
{ a modified copy of this file, send modifications, suggestions and bug        }
{ reports to david@e.co.za                                                     }
{                                                                              }
{                                                                              }
{ 01/02/2000  v0.01  Initial html Functions                                    }
{ 20/02/2000  v0.02  DecomposeFromField, DecodeMIME functions,                 }
{                    DecodeQuotedPrintableText, htmlSafeText                   }
{                    Document code                                             }
{ 04/03/2000  v0.03  RFC Date-Time functions                                   }
{ 05/03/2000  v0.04  HTTP response functions                                   }
{ 06/03/2000  v0.05  Message header decoding functions                         }
{ 08/03/2000  v0.06  Nested MIME messages.                                     }
{ 09/04/2000  v0.07  ThtmlParser.                                              }
{ 11/04/2000  v0.08  ThtmlRenderer.                                            }
{ 12/04/2000  v0.09  Added AhtmlRenderer and moved ThtmlRenderer to            }
{                    cHTMLRenderer.                                            }
{                    710 lines interface. 2090 lines implementation.           }
{ 15/04/2000  v0.10  Rewrote ThtmlParser from HTML 4.0 specifications.         }
{                    769 lines interface. 2400 lines implementation.           }
{ 28/04/2000  v0.11  Moved HTTP code to cHTTPServer.                           }
{ 11/05/2000  v0.12  Moved HTML code to cHTML, moved XML code to cXML.         }
{                                                                              }



{                                                                              }
{ Miscellaneous                                                                }
{                                                                              }
const
  CRLF  = ASCII_CR + ASCII_LF;
  HTAB  = ASCII_HT;
  CTL   = [#0..#31];
  SPACE = [ASCII_HT, ASCII_SP];

Function IsIPAddress (const S : String) : Boolean;
Function TransferRate (const Length, Speed : Integer) : String;
Function HeaderField (const FieldName, FieldValue : String; const Parameters : String = '') : String;
Function RFCTime (const D : TDateTime) : String;
Function DateField : String;
Function ContentLengthField (const Length : Integer) : String;
Function ContentTypeField (const ContentType : String) : String;


{                                                                              }
{ Date/Time Encoding/Decoding                                                  }
{   From RFC822, RFC850, RFC1123, RFC1036, RFC1945 (See implementation for     }
{   details)                                                                   }
{                                                                              }
Function RFCTimeZoneToGMTBias (const Zone : String; const Strict : Boolean = False) : Integer;
{ Returns the bias of the specified Zone relative to GMT (UT) in minutes       }
Function GMTDateTimeToRFC1123DateTime (const D : TDateTime; const IncludeDayOfWeek : Boolean = True) : String;
{ Converts a GMT Date-Time to a RFC1123 Date-Time                              }
{ This is the preferred representation on the Internet for all Date-Times.     }
{ See cDateTime.LocalTimeToGMTTime for converting to GMT time.                 }
Function RFCDateTimeToGMTDateTime (const S : String) : TDateTime;
{ Converts a Date-Time that complies with any of the RFC standards to          }
{   a GMT Date-Time. This is a tolerant implementation.                        }
{ See cDateTime.GMTTimeToLocalTime for converting to local time.               }



{                                                                              }
{ Header Decoding/Encoding                                                     }
{                                                                              }
{ From RFC822 (Standard for the format of ARPA Internet Text Messages) and     }
{   RFC2616 (Hypertext Transfer Protocol -- HTTP/1.1). See implementation for  }
{   details.                                                                   }
type
  { AHeaderField                                                               }
  AHeaderField = class
    protected
    Function  GetFieldName : String; virtual; abstract;
    Procedure SetFieldName (const FieldName : String); virtual; abstract;
    Function  GetFieldBody : String; virtual; abstract;
    Procedure SetFieldBody (const FieldBody : String); virtual; abstract;
    Function  GetAsString : String; virtual;
    Procedure SetAsString (const F : String); virtual;
    Function  GetFieldBodyAsInteger : Integer; virtual;
    Procedure SetFieldBodyAsInteger (const F : Integer); virtual;

    public
    Constructor Create (const FieldBody : String); overload;
    Constructor Create (const FieldName, FieldBody : String); overload;

    Property AsString : String read GetAsString write SetAsString;
    Property FieldName : String read GetFieldName write SetFieldName;
    Property FieldBody : String read GetFieldBody write SetFieldBody;
    Property FieldBodyAsInteger : Integer read GetFieldBodyAsInteger write SetFieldBodyAsInteger;
  end;

  { THeaderField                                                               }
  THeaderField = class (AHeaderField)
    protected
    FFieldName : String;
    FFieldBody : String;

    Function  GetFieldName : String; override;
    Procedure SetFieldName (const FieldName : String); override;
    Function  GetFieldBody : String; override;
    Procedure SetFieldBody (const FieldBody : String); override;
  end;

  { THeader                                                                    }
  THeader = class
    protected
    HeaderFields : Array of AHeaderField;

    Procedure Decode (const Header : String); virtual;
    Function  Encode : String; virtual;
    Function  GetFieldByIndex (const Idx : Integer) : AHeaderField;
    Function  GetField (const FieldName : String) : AHeaderField;
    Procedure SetField (const FieldName : String; const Field : AHeaderField);

    public
    Destructor Destroy; override;

    Procedure AddField (const Field : AHeaderField); overload;
    Procedure AddField (const FieldName, FieldBody : String); overload; virtual;
    Procedure AddField (const FieldLine : String); overload;
    Function  HasField (const FieldName : String) : Boolean;
    Function  GetFieldBody (const FieldName : String) : String;
    Function  Count : Integer;

    Property  Field [const FieldName : String] : AHeaderField read GetField write SetField; default;
    Property  FieldByIndex [const Idx : Integer] : AHeaderField read GetFieldByIndex;
    Property  AsString : String read Encode write Decode;
  end;



{                                                                              }
{ Miscellaneous header fields                                                  }
{                                                                              }
type
  { TDateField                                                                 }
  TDateField = class (AHeaderField)
    protected
    Function  GetFieldBody : String; override;
    Procedure SetFieldBody (const FieldBody : String); override;
    Function  GetFieldName : String; override;

    public
    GMTDateTime : TDateTime;

    Constructor Create (const D : TDateTime);
  end;

  { TEMailField                                                                }
  TEMailField = class (THeaderField)
    protected
    Function  GetFieldBody : String; override;
    Procedure SetFieldBody (const FieldBody : String); override;

    public
    Address : String;
    Name    : String;
  end;



{                                                                              }
{ MIME Decoding                                                                }
{   From RFC2045                                                               }
{                                                                              }
Function MimeParameterValue (const Parameters, ParameterName : String) : String;

type
  { TMimeVersionField                                                          }
  TMimeVersionField = class (AHeaderField)
    protected
    Function  GetFieldName : String; override;
    Procedure SetFieldBody (const FieldBody : String); override;
    Function  GetFieldBody : String; override;

    public
    MajorVersion,
    MinorVersion  : Integer;
    Comment       : String;
  end;

  { TMimeContentTypeField                                                      }
  TMimeContentTypeField = class (THeaderField)
    protected
    Procedure SetFieldBody (const FieldBody : String); override;

    public
    MajorType,
    SubType     : String;
    Parameters  : String;
  end;

  { TMimeContentTransferEncodingField                                          }
  TTransferEncoding = (te_none, te_7bit, te_8bit, te_binary,
                       te_quoted_printable, te_base64, te_x_uuencode);
  TMimeContentTransferEncodingField = class (THeaderField)
    protected
    Procedure SetFieldBody (const FieldBody : String); override;

    public
    TransferEncoding : TTransferEncoding;
  end;

  { TMimeMessageHeader                                                         }
  TMimeMessageHeader = class (THeader)
    protected
    FIsMimeMessage      : Boolean;
    FBoundary           : String;
    FContentDisposition : String;
    FTransferEncoding   : TTransferEncoding;

    Procedure AddField (const FieldName, FieldBody : String); override;

    public
    Procedure Decode (const Header : String); override;
    Property  IsMimeMessage : Boolean read FIsMimeMessage;
  end;



{                                                                              }
{ Mail Message Decoding                                                        }
{   If InlineUUDecode is set TMailMessageDecoder will try to decode uuencoded  }
{     attachments contained in the message body.                               }
{                                                                              }
type
  TMessageBodyPart = class
    Header : THeader;
    Body   : String;
  end;
  TMultiPartBodyPart = class (TMessageBodyPart)
    Body : Array of TMessageBodyPart;
    Procedure AddPart (const BodyPart : TMessageBodyPart);
  end;
  TTextBodyPart = class (TMessageBodyPart)
  end;
  TAttachmentBodyPart = class (TMessageBodyPart)
    Name : String;
  end;
  TOnTextMessageLine = Function (var Line : String) : Boolean of object;
  TMailMessageDecoder = class
    protected
    FInlineUUDecode    : Boolean;
    FHeader            : THeader;
    FBody              : TMessageBodyPart;
    FOnTextMessageLine : TOnTextMessageLine;

    public
    Constructor Create;

    Property  InlineUUDecode : Boolean read FInlineUUDecode write FInlineUUDecode;
    Procedure Decode (const Header : String; const Body : TExStream);
    Property  OnTextMessageLine : TOnTextMessageLine read FOnTextMessageLine write FOnTextMessageLine;

    Property  Header : THeader read FHeader write FHeader;
    Property  Body : TMessageBodyPart read FBody write FBody;
  end;


{                                                                              }
{ Message Encoding                                                             }
{                                                                              }
Function GenerateMIMEBoundary : String;


{                                                                              }
{ HTML Text Encoding                                                           }
{   From RFC 1866 (Hypertext Markup Language - 2.0)                            }
{                                                                              }
const
  htmlNBSP           = '&nbsp;';       // non-breaking space

  { HTML 3.2 "Western Latin-1 (ISO-8559-1)" character set }
  ISO_8559_1 =
        '                                 !"#$%&''()*+,-./0123456789:;<=>?' +
        '@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_`abcdefghijklmnopqrstuvwxyz{|}~'  +
        ' '  +
        '';

  { US-ASCII character set }
  US_ASCII =
        '                                 !"#$%&''()*+,-./0123456789:;<=>?' +
        '@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_`abcdefghijklmnopqrstuvwxyz{|}~'  +
        'ܢPѪ                  '  +
        '                                aGpSstFTOd8fen     vn ';


  { HTML 3.2 "Added Western Latin-1" entity set }
  htmlCopyright  = '&copy;';
  ISO_8559_1_AddedWestern : Array [1..100, 1..2] of String = (
      ('&quot;',  '&#34;'),     ('&amp;',   '&#38;'),    ('&lt;',    '&#60;'),
      ('&gt;',    '&#62;'),     ('&nbsp;',  '&#160;'),   ('&iexcl;', '&#161;'),
      ('&cent;',  '&#162;'),    ('&pound;', '&#163;'),   ('&curren;','&#164;'),
      ('&yen;',   '&#165;'),    ('&brvbar;','&#166;'),   ('&sect;',  '&#167;'),
      ('&uml;',   '&#168;'),    ('&copy;',  '&#169;'),   ('&ordf;',  '&#170;'),
      ('&laquo;', '&#171;'),    ('&not;',   '&#172;'),   ('&shy;',   '&#173;'),
      ('&reg;',   '&#174;'),    ('&macr;',  '&#175;'),   ('&deg;',   '&#176;'),
      ('&plusmn;','&#177;'),    ('&sup2;',  '&#178;'),   ('&sup3;',  '&#179;'),
      ('&acute;', '&#180;'),    ('&micro;', '&#181;'),   ('&para;',  '&#182;'),
      ('&middot;','&#183;'),    ('&cedil;', '&#184;'),   ('&sup1;',  '&#185;'),
      ('&ordm;',  '&#186;'),    ('&raquo;', '&#187;'),   ('&frac14;','&#188;'),
      ('&frac12;','&#189;'),    ('&frac34;','&#190;'),   ('&iquest;','&#191;'),
      ('&Agrave;','&#192;'),    ('&Aacute;','&#193;'),   ('&Acirc;', '&#194;'),
      ('&Atilde;','&#195;'),    ('&Auml;',  '&#196;'),   ('&Aring;', '&#197;'),
      ('&AElig;', '&#198;'),    ('&Ccedil;','&#199;'),   ('&Egrave;','&#200;'),
      ('&Eacute;','&#201;'),    ('&Ecirc;', '&#202;'),   ('&Euml;',  '&#203;'),
      ('&Igrave;','&#204;'),    ('&Iacute;','&#205;'),   ('&Icirc;', '&#206;'),
      ('&Iuml;',  '&#207;'),    ('&ETH;',   '&#208;'),   ('&Ntilde;','&#209;'),
      ('&Ograve;','&#210;'),    ('&Oacute;','&#211;'),   ('&Ocirc;', '&#212;'),
      ('&Otilde;','&#213;'),    ('&Ouml;',  '&#214;'),   ('&times;', '&#215;'),
      ('&Oslash;','&#216;'),    ('&Ugrave;','&#217;'),   ('&Uacute;','&#218;'),
      ('&Ucirc;', '&#219;'),    ('&Uuml;',  '&#220;'),   ('&Yacute;','&#221;'),
      ('&THORN;', '&#222;'),    ('&szlig;', '&#223;'),   ('&agrave;','&#224;'),
      ('&aacute;','&#225;'),    ('&acirc;', '&#226;'),   ('&atilde;','&#227;'),
      ('&auml;',  '&#228;'),    ('&aring;', '&#229;'),   ('&aelig;', '&#230;'),
      ('&ccedil;','&#231;'),    ('&egrave;','&#232;'),   ('&eacute;','&#233;'),
      ('&ecirc;', '&#234;'),    ('&euml;',  '&#235;'),   ('&igrave;','&#236;'),
      ('&iacute;','&#237;'),    ('&icirc;', '&#238;'),   ('&iuml;',  '&#239;'),
      ('&eth;',   '&#240;'),    ('&ntilde;','&#241;'),   ('&ograve;','&#242;'),
      ('&oacute;','&#243;'),    ('&ocirc;', '&#244;'),   ('&otilde;','&#245;'),
      ('&ouml;',  '&#246;'),    ('&divide;','&#247;'),   ('&oslash;','&#248;'),
      ('&ugrave;','&#249;'),    ('&uacute;','&#250;'),   ('&ucirc;', '&#251;'),
      ('&uuml;',  '&#252;'),    ('&yacute;','&#253;'),   ('&thorn;', '&#254;'),
      ('&yuml;',  '&#255;'));

  { HTML elements }
  htmlBR             = '<BR>' + CRLF;
  htmlHorisontalRule = '<HR>' + CRLF;



{                                                                              }
{ HTML Graphics Rendering                                                      }
{                                                                              }
{ AhtmlRenderer                                                                }
{   Base class for a visual HTML renderer.                                     }
{   HorSpace returns the numbers of pixels left horisontally before the        }
{     Renderer will attempt a Break.                                           }
{   BOL returns True if renderer is at the beginning of a line.                }
{   AddControl gives a visual control to a renderer to be added to the         }
{     display.                                                                 }
{   BreakLeft moves to beginning of the next rendering "line" (newline). An    }
{     additional SkipHeight pixel lines are skipped.                           }
{                                                                              }
type
  ThtmlRenderAttributes = Set of (raBold, raItalic, raUnderline, raSubscript,
                                  raSuperScript, raStrikethrough,
                                  raStrong, raEmphasis, raCode, raKeyboard,
                                  raVariable, raTeletype, raCitation,
                                  raDefinition, raSample, raPreformatted,
                                  raCentered, raSmall, raBig, raIsLink);
type
  ThtmlFontSize = Integer;
  ThtmlFontColour = String;
  ThtmlRendererBreakType = (btMargin,
                            btNextMargin,
                            btLeftMargin);
  ThtmlHorAlign = (haLeft, haRight, haCenter);
  ThtmlVerAlign = (vaBottom, vaTop, vaCenter);
  AhtmlRenderer = class
    protected
    FFontSize       : ThtmlFontSize;
    FFontColour     : ThtmlFontColour;
    FFontBackColour : ThtmlFontColour;
    FAttributes     : ThtmlRenderAttributes;

    FIndent         : Integer;
    FDesiredWidth   : Integer;
    FVAlign         : ThtmlVerAlign;
    FHAlign         : ThtmlHorAlign;

    public
    Constructor Create (const Parent : AhtmlRenderer);

    { HTML attributes }
    Property  FontSize : ThtmlFontSize read FFontSize write FFontSize;
    Property  FontColour : ThtmlFontColour read FFontColour write FFontColour;
    Property  FontBackColour : ThtmlFontColour read FFontBackColour write FFontBackColour;
    Property  Attributes : ThtmlRenderAttributes read FAttributes write FAttributes;

    { Rendering attributes }
    Property  DesiredWidth : Integer read FDesiredWidth write FDesiredWidth;
    Property  Indent : Integer read FIndent write FIndent;
    Function  HorSpace : Integer; virtual; abstract;
    Function  BOL : Boolean; virtual; abstract;
    Function  LeftMargin : Boolean; virtual; abstract;
    Property  VerAlign : ThtmlVerAlign read FVAlign write FVAlign;
    Property  HorAlign : ThtmlHorAlign read FHAlign write FHAlign;

    { Rendering methods }
    Procedure BreakLine (const BreakType : ThtmlRendererBreakType = btMargin; const SkipHeight : Integer = 0); virtual; abstract;
    Procedure AddControl (const Control : TControl; const SetMargin : Boolean = False); virtual; abstract;
  end;




Function  htmlColour (const Colour : String) : TColor;
{ Returns the Delphi TColor given an HTML Colour string.                       }

type
  ThtmlSizeType = (stNone, stPixels, stPercent, stRelative);

Procedure htmlDecodeSize (const S : String; var Size : Integer; var SizeType : ThtmlSizeType);
{ Decodes a HTML size (pixels, percentage or relative).                        }

{ Translate                                                                    }
{   Default Translate function translates from the ISO-8559-1 character set.   }
Function Translate (const Ch : Char; const TranslationTable : String = ISO_8559_1) : Char; overload;
Function Translate (const S : String; const TranslationTable : String = ISO_8559_1) : String; overload;



implementation



uses
  // L0 units
  cUtils,
  cDateTime,  // LocalTimeToGMTTime
  cMaths;     // HexToInt


{                                                                              }
{ Miscellaneous                                                                }
{                                                                              }
Function IsIPAddress (const S : String) : Boolean;
  Begin
    try
      Result := Length (StrToByteArray (Remove (cs_WhiteSpace, S), '.')) = 4;
    except
      Result := False;
    end;
  End;

Function TransferRate (const Length, Speed : Integer) : String;
var cps : Double;
  Begin
    if Speed <= 0 then
      Result := '' else
      begin
        cps := Length / (Speed / 1000.0);
        if cps >= 1024 * 1024 then
          Result := Format ('%0.1f', [cps / (1024 * 1024)]) + 'Mb/s' else
        if cps >= 1024 then
          Result := Format ('%0.1f', [cps / 1024]) + 'Kb/s' else
          Result := IntToStr (Round (cps)) + 'cps';
      end;
  End;

Function HeaderField (const FieldName, FieldValue : String; const Parameters : String = '') : String;
  Begin
    Result := FieldName + ': ' + FieldValue +
              Cond (Parameters <> '', '; ' + Parameters, '') +
              CRLF;
  End;

Function RFCTime (const D : TDateTime) : String;
  Begin
    Result := GMTDateTimeToRFC1123DateTime (LocalTimeToGMTTime (D));
  End;

Function DateField : String;
  Begin
    Result := HeaderField ('Date', GMTDateTimeToRFC1123DateTime (LocalTimeToGMTTime (Now)));
  End;

Function ContentLengthField (const Length : Integer) : String;
  Begin
    Result := HeaderField ('Content-Length', IntToStr (Length));
  End;

Function ContentTypeField (const ContentType : String) : String;
  Begin
    Result := HeaderField ('Content-Type', ContentType);
  End;



{                                                                              }
{ Date/Time Encoding/Decoding                                                  }
{                                                                              }
{ From RFC 822 (Standard for the format of ARPA INTERNET Text Messages):       }
{    "time        =  hour zone                      ; ANSI and Military        }
{     hour        =  2DIGIT ":" 2DIGIT [":" 2DIGIT] ; 00:00:00 - 23:59:59      }
{     zone        =  "UT"  / "GMT"                  ; Universal Time           }
{                                                   ; North American : UT      }
{                 /  "EST" / "EDT"                  ;  Eastern:  - 5/ - 4      }
{                 /  "CST" / "CDT"                  ;  Central:  - 6/ - 5      }
{                 /  "MST" / "MDT"                  ;  Mountain: - 7/ - 6      }
{                 /  "PST" / "PDT"                  ;  Pacific:  - 8/ - 7      }
{                 /  1ALPHA                         ; Military: Z = UT;        }
{                                                   ;  A:-1; (J not used)      }
{                                                   ;  M:-12; N:+1; Y:+12      }
{                 / ( ("+" / "-") 4DIGIT )          ; Local differential       }
{                                                   ;  hours+min. (HHMM)       }
{     date-time   =  [ day "," ] date time          ; dd mm yy                 }
{                                                   ;  hh:mm:ss zzz            }
{     day         =  "Mon"  / "Tue" /  "Wed"  / "Thu"                          }
{                 /  "Fri"  / "Sat" /  "Sun"                                   }
{     date        =  1*2DIGIT month 2DIGIT        ; day month year             }
{                                                 ;  e.g. 20 Jun 82            }
{     month       =  "Jan"  /  "Feb" /  "Mar"  /  "Apr"                        }
{                 /  "May"  /  "Jun" /  "Jul"  /  "Aug"                        }
{                 /  "Sep"  /  "Oct" /  "Nov"  /  "Dec"                    "   }
{                                                                              }
{ Note that even though RFC 822 states hour=2DIGIT":"2DIGIT, none of the       }
{   examples given in the appendix include the ":",                            }
{   for example: "26 Aug 76 1429 EDT"                                          }
{                                                                              }
{                                                                              }
{ From RFC 1036 (Standard for Interchange of USENET Messages):                 }
{                                                                              }
{   "Its format must be acceptable both in RFC-822 and to the getdate(3)       }
{    routine that is provided with the Usenet software.   ...                  }
{    One format that is acceptable to both is:                                 }
{                                                                              }
{                      Wdy, DD Mon YY HH:MM:SS TIMEZONE                        }
{                                                                              }
{    Note in particular that ctime(3) format:                                  }
{                                                                              }
{                          Wdy Mon DD HH:MM:SS YYYY                            }
{                                                                              }
{    is not acceptable because it is not a valid RFC-822 date.  However,       }
{    since older software still generates this format, news                    }
{    implementations are encouraged to accept this format and translate        }
{    it into an acceptable format.                                         "   }
{                                                                              }
{   "Here is an example of a message in the old format (before the             }
{    existence of this standard). It is recommended that                       }
{    implementations also accept messages in this format to ease upward        }
{    conversion.                                                               }
{                                                                              }
{               Posted: Fri Nov 19 16:14:55 1982                           "   }
{                                                                              }
{                                                                              }
{ From RFC 1945 (Hypertext Transfer Protocol -- HTTP/1.0)                      }
{                                                                              }
{  "HTTP/1.0 applications have historically allowed three different            }
{   formats for the representation of date/time stamps:                        }
{                                                                              }
{       Sun, 06 Nov 1994 08:49:37 GMT    ; RFC 822, updated by RFC 1123        }
{       Sunday, 06-Nov-94 08:49:37 GMT   ; RFC 850, obsoleted by RFC 1036      }
{       Sun Nov  6 08:49:37 1994         ; ANSI C's asctime() format           }
{                                                                              }
{   The first format is preferred as an Internet standard and represents       }
{   a fixed-length subset of that defined by RFC 1123 [6] (an update to        }
{   RFC 822 [7]). The second format is in common use, but is based on the      }
{   obsolete RFC 850 [10] date format and lacks a four-digit year.             }
{   HTTP/1.0 clients and servers that parse the date value should accept       }
{   all three formats, though they must never generate the third               }
{   (asctime) format.                                                          }
{                                                                              }
{      Note: Recipients of date values are encouraged to be robust in          }
{      accepting date values that may have been generated by non-HTTP          }
{      applications, as is sometimes the case when retrieving or posting       }
{      messages via proxies/gateways to SMTP or NNTP.                       "  }
{                                                                              }
{  "All HTTP/1.0 date/time stamps must be represented in Universal Time        }
{   (UT), also known as Greenwich Mean Time (GMT), without exception.          }
{                                                                              }
{       HTTP-date      = rfc1123-date | rfc850-date | asctime-date             }
{                                                                              }
{       rfc1123-date   = wkday "," SP date1 SP time SP "GMT"                   }
{       rfc850-date    = weekday "," SP date2 SP time SP "GMT"                 }
{       asctime-date   = wkday SP date3 SP time SP 4DIGIT                      }
{                                                                              }
{       date1          = 2DIGIT SP month SP 4DIGIT                             }
{                        ; day month year (e.g., 02 Jun 1982)                  }
{       date2          = 2DIGIT "-" month "-" 2DIGIT                           }
{                        ; day-month-year (e.g., 02-Jun-82)                    }
{       date3          = month SP ( 2DIGIT | ( SP 1DIGIT ))                    }
{                        ; month day (e.g., Jun  2)                            }
{                                                                              }
{       time           = 2DIGIT ":" 2DIGIT ":" 2DIGIT                          }
{                        ; 00:00:00 - 23:59:59                                 }
{                                                                              }
{       wkday          = "Mon" | "Tue" | "Wed"                                 }
{                      | "Thu" | "Fri" | "Sat" | "Sun"                         }
{                                                                              }
{       weekday        = "Monday" | "Tuesday" | "Wednesday"                    }
{                      | "Thursday" | "Friday" | "Saturday" | "Sunday"         }
{                                                                              }
{       month          = "Jan" | "Feb" | "Mar" | "Apr"                         }
{                      | "May" | "Jun" | "Jul" | "Aug"                         }
{                      | "Sep" | "Oct" | "Nov" | "Dec"                      "  }
{                                                                              }
Function RFCTimeZoneToGMTBias (const Zone : String; const Strict : Boolean = False) : Integer;
type TZoneBias = record
       Zone : String;
       Bias : Integer;
     end;
const TimeZones = 35;
      ZoneBias : Array [1..TimeZones] of TZoneBias =
          ((Zone:'GMT'; Bias:0),                 (Zone:'UT';  Bias:0),
           (Zone:'EST'; Bias:-5*60),             (Zone:'EDT'; Bias:-4*60),
           (Zone:'CST'; Bias:-6*60),             (Zone:'CDT'; Bias:-5*60),
           (Zone:'MST'; Bias:-7*60),             (Zone:'MDT'; Bias:-6*60),
           (Zone:'PST'; Bias:-8*60),             (Zone:'PDT'; Bias:-7*60),
           (Zone:'Z';   Bias:0),                 (Zone:'A';   Bias:-1*60),
           (Zone:'B';   Bias:-2*60),             (Zone:'C';   Bias:-3*60),
           (Zone:'D';   Bias:-4*60),             (Zone:'E';   Bias:-5*60),
           (Zone:'F';   Bias:-6*60),             (Zone:'G';   Bias:-7*60),
           (Zone:'H';   Bias:-8*60),             (Zone:'I';   Bias:-9*60),
           (Zone:'K';   Bias:-10*60),            (Zone:'L';   Bias:-11*60),
           (Zone:'M';   Bias:-12*60),            (Zone:'N';   Bias:1*60),
           (Zone:'O';   Bias:2*60),              (Zone:'P';   Bias:3*60),
           (Zone:'Q';   Bias:4*60),              (Zone:'R';   Bias:3*60),
           (Zone:'S';   Bias:6*60),              (Zone:'T';   Bias:3*60),
           (Zone:'U';   Bias:8*60),              (Zone:'V';   Bias:3*60),
           (Zone:'W';   Bias:10*60),             (Zone:'X';   Bias:3*60),
           (Zone:'Y';   Bias:12*60));
var S : String;
    I : Integer;
  Begin
    if Zone [1] in ['+', '-'] then // +hhmm format
      begin
        if Strict and (Length (Zone) <> 5) then
          raise EConvertError.Create ('Not a RFC1123 time zone: Invalid differential');
        S := Trim (Zone, SPACE);
        try
          I := StrToInt (CopyLeft (S, 3));
        except
          raise EConvertError.Create ('Not a RFC1123 time zone: Invalid hour in differential');
        end;
        if Abs (I) > 12 then
          raise EConvertError.Create ('Not a RFC1123 time zone: Invalid hour in differential');
        Result := I * 60;
        S := CopyFrom (S, 4);
        if S <> '' then
          if Length (S) <> 2 then
            raise EConvertError.Create ('Not a RFC1123 time zone: Invalid minute in differential') else
            try
              I := StrToInt (S);
              Result := Result + I;
            except
              raise EConvertError.Create ('Not a RFC1123 time zone: Invalid minute in differential');
            end;
      end else
      begin // named format
        S := UpperCase (Zone);
        For I := 1 to TimeZones do
          if ZoneBias [I].Zone = S then
            begin
              Result := ZoneBias [I].Bias;
              exit;
            end;
        raise EConvertError.Create ('Not a RFC1123 time zone: Unknown zone name');
      end;
  End;

{ Converts a RFC Time to GMT Time                                              }
Function RFCTimeToGMTTime (const S : String; const StrictlyRFC1123 : Boolean = False) : TDateTime;
var I : Integer;
    T : String;
    HH, MM, SS : Integer;
    U : StringArray;
  Begin
    T := Trim (S, SPACE);

    // Get Zone bias
    I := PosPrev (SPACE, T);
    if I > 0 then
      begin
        Result := Int (RFCTimeZoneToGMTBias (CopyFrom (T, I + 1), StrictlyRFC1123)) / 1440.0;
        T := Trim (CopyLeft (T, I - 1), SPACE);
      end else
      if StrictlyRFC1123 then
        raise EConvertError.Create ('Not a RFC1123 time: No Zone') else
        Result := 0;

    // Get time
    U := Split (T, ':');
    if not StrictlyRFC1123 and (Length (U) = 1) and (Length (U [0]) = 4) then
      try // old hhmm format
        HH := StrToInt (Copy (U [0], 1, 2));
        MM := StrToInt (Copy (U [0], 3, 2));
        SS := 0;
      except
        raise EConvertError.Create ('Not a RFC time: Invalid time');
      end else
      begin // hh:mm[:ss] format (RFC1123)
        if (Length (U) < 2) or (Length (U) > 3) then
          raise EConvertError.Create ('Not a RFC1123 time: Invalid time');
        try
          HH := StrToInt (Trim (U [0], SPACE));
          MM := StrToInt (Trim (U [1], SPACE));
          if Length (U) = 3 then
            SS := StrToInt (Trim (U [2], SPACE)) else
            SS := 0;
        except
          raise EConvertError.Create ('Not a RFC1123 time: Invalid time');
        end;
      end;

    if (HH < 0) or (HH > 23) then
      raise EConvertError.Create ('Not a RFC1123 time: Invalid hour');
    if (MM < 0) or (MM > 59) then
      raise EConvertError.Create ('Not a RFC1123 time: Invalid minute');
    if (SS < 0) or (SS > 59) then
      raise EConvertError.Create ('Not a RFC1123 time: Invalid second');
    Result := Result + EncodeTime (HH, MM, SS, 0);
  End;

{ Converts GMT Time to a RFC1123 Time                                          }
Function GMTTimeToRFC1123Time (const D : TDateTime; const IncludeSeconds : Boolean = True) : String;
var Ho, Mi, Se, Ms : Word;
  Begin
    DecodeTime (D, Ho, Mi, Se, Ms);
    Result := PadLeft (IntToStr (Ho), '0', 2) + ':' +
              PadLeft (IntToStr (Mi), '0', 2) +
              Cond (IncludeSeconds, ':' + PadLeft (IntToStr (Se), '0', 2), '') +
              ' GMT';
  End;


{ Date constants                                                               }
const
  RFC850DayNames : Array [1..7] of String = (
      'Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday');
  RFC1123DayNames : Array [1..7] of String = (
      'Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat');
  RFCMonthNames : Array [1..12] of String = (
      'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec');

Function RFC850DayOfWeek (const S : String) : Integer;
var I : Integer;
    T : String;
  Begin
    T := UpperCase (S);
    For I := 1 to 7 do
      if UpperCase (RFC850DayNames [I]) = T then
        begin
          Result := I;
          exit;
        end;
    Result := -1;
  End;

Function RFC1123DayOfWeek (const S : String) : Integer;
var I : Integer;
    T : String;
  Begin
    T := UpperCase (S);
    For I := 1 to 7 do
      if UpperCase (RFC1123DayNames [I]) = T then
        begin
          Result := I;
          exit;
        end;
    Result := -1;
  End;

Function RFCMonth (const S : String) : Integer;
var I : Integer;
    T : String;
  Begin
    T := UpperCase (S);
    For I := 1 to 12 do
      if UpperCase (RFCMonthNames [I]) = T then
        begin
          Result := I;
          exit;
        end;
    Result := -1;
  End;

Function GMTDateTimeToRFC1123DateTime (const D : TDateTime; const IncludeDayOfWeek : Boolean = True) : String;
var Ye, Mo, Da : Word;
  Begin
    DecodeDate (D, Ye, Mo, Da);
    Result := Cond (IncludeDayOfWeek, RFC1123DayNames [DayOfWeek (D)] + ', ', '') +
              PadLeft (IntToStr (Da), '0', 2) + ' ' +
              RFCMonthNames [Mo] + ' ' +
              IntToStr (Ye) + ' ' +
              GMTTimeToRFC1123Time (D);
  End;

Function RFCDateTimeToGMTDateTime (const S : String) : TDateTime;
var T, U : String;
    I : Integer;
    D, M, Y, DOW : Integer;
    V, W : StringArray;
  Begin
    T := Trim (S, SPACE);

    // Extract Day of week
    I := PosNext (SPACE + [','], T);
    if I = 0 then
      DOW := -1 else
      begin
        U := CopyLeft (T, I - 1);
        DOW := RFC850DayOfWeek (U);
        if DOW = -1 then
          DOW := RFC1123DayOfWeek (U);
        if DOW <> -1 then
          T := Trim (CopyFrom (S, I + 1), SPACE);
      end;

    V := Split (T, SPACE);
    if Length (V) < 3 then
      raise EConvertError.Create ('Not a recognised date: Too few elements');

    if PosNext ('-', V [0]) > 0 then // RFC850 date, eg "Sunday, 06-Nov-94 08:49:37 GMT"
      begin
        W := Split (V [0], '-');
        if Length (W) <> 3 then
          raise EConvertError.Create ('Not a recognised date: Not a RFC850 date');
        M := RFCMonth (W [1]);
        if M = -1 then
          raise EConvertError.Create ('Not a recognised date: Not a RFC850 date');
        try
          D := StrToInt (W [0]);
          Y := StrToInt (W [2]);
          if Y < 100 then
            Y := TwoDigitYearToYear (Y);
        except
          raise EConvertError.Create ('Not a recognised date: Not a RFC850 date');
        end;
        Result := EncodeDate (Y, M, D) + RFCTimeToGMTTime (V [1]  + V [2]);
        exit;
      end;

    M := RFCMonth (V [1]);
    if M >= 1 then // RFC822 date, eg Sun, 06 Nov 1994 08:49:37 GMT
      begin
        try
          D := StrToInt (V [0]);
          Y := StrToInt (V [2]);
        except
          raise EConvertError.Create ('Not a recognised date: Not a RFC822 date');
        end;
        Result := EncodeDate (Y, M, D);
        if Length (V) = 4 then
          Result := Result + RFCTimeToGMTTime (V [3]) else
          if Length (V) >= 5 then
            Result := Result + RFCTimeToGMTTime (V [3] + ' ' + V [4]);
        exit;
      end;

    M := RFCMonth (V [0]);
    if M >= 1 then // ANSI C asctime() format, eg "Sun Nov  6 08:49:37 1994"
      begin
        try
          D := StrToInt (V [1]);
          Y := StrToInt (V [3]);
        except
          raise EConvertError.Create ('Not a recognised date: Not a asctime() date');
        end;
        Result := EncodeDate (Y, M, D) + RFCTimeToGMTTime (V [2]);
      end else
      raise EConvertError.Create ('Not a recognised date.');
  End;



{                                                                              }
{ Header Decoding/Encoding                                                     }
{                                                                              }
{ From RFC822 (Standard for the format of ARPA Internet Text Messages):        }
{     field       =  field-name ":" [ field-body ] CRLF                        }
{     field-name  =  1*<any CHAR, excluding CTLs, SPACE, and ":">              }
{     field-body  =  field-body-contents                                       }
{                    [CRLF LWSP-char field-body]                               }
{     LWSP-char   =  SPACE / HTAB                 ; semantics = SPACE          }
{                                                                              }
{ From RFC2616 (Hypertext Transfer Protocol -- HTTP/1.1):                      }
{   HTTP/1.1 header field values can be folded onto multiple lines if the      }
{   continuation line begins with a space or horizontal tab. All linear        }
{   white space, including folding, has the same semantics as SP. A            }
{   recipient MAY replace any linear white space with a single SP before       }
{   interpreting the field value or forwarding the message downstream.         }
{       LWS            = [CRLF] 1*( SP | HT )                                  }

{ AHeaderField                                                                 }
Constructor AHeaderField.Create (const FieldBody : String);
  Begin
    inherited Create;
    SetFieldBody (FieldBody);
  End;

Constructor AHeaderField.Create (const FieldName, FieldBody : String);
  Begin
    inherited Create;
    SetFieldName (FieldName);
    SetFieldBody (FieldBody);
  End;

Function AHeaderField.GetAsString : String;
  Begin
    Result := FieldName + ': ' + FieldBody + CRLF;
  End;

Procedure AHeaderField.SetAsString (const F : String);
var I : Integer;
    S : String;
  Begin
    S := Trim (F, SPACE);
    I := PosNext (':', F);
    if I <= 1 then
      raise EConvertError.Create ('Not a valid header field');
    FieldName := CopyLeft (S, I - 1);
    FieldBody := Trim (CopyFrom (S, I + 1), SPACE);
  End;

Function AHeaderField.GetFieldBodyAsInteger : Integer;
  Begin
    Result := StrToIntDef (GetFieldBody, 0);
  End;

Procedure AHeaderField.SetFieldBodyAsInteger (const F : Integer);
  Begin
    SetFieldBody (IntToStr (F));
  End;

{ THeaderField                                                                 }
Function THeaderField.GetFieldName : String;
  Begin
    Result := FFieldName;
  End;

Procedure THeaderField.SetFieldName (const FieldName : String);
  Begin
    FFieldName := FieldName;
  End;

Function THeaderField.GetFieldBody : String;
  Begin
    Result := FFieldBody;
  End;

Procedure THeaderField.SetFieldBody (const FieldBody : String);
  Begin
    FFieldBody := FieldBody;
  End;

{ THeader                                                                      }
Destructor THeader.Destroy;
var I : Integer;
  Begin
    For I := Length (HeaderFields) - 1 downto 0 do
      HeaderFields [I].Free;
    SetLength (HeaderFields, 0);
    inherited Destroy;
  End;

Procedure THeader.AddField (const Field : AHeaderField);
var L : Integer;
  Begin
    L := Length (HeaderFields);
    SetLength (HeaderFields, L + 1);
    HeaderFields [L] := Field;
  End;

Procedure THeader.AddField (const FieldName, FieldBody : String);
var F : THeaderField;
  Begin
    F := THeaderField.Create;
    F.FieldName := FieldName;
    F.FieldBody := FieldBody;
    AddField (F);
  End;

Procedure THeader.AddField (const FieldLine : String);
var I : Integer;
  Begin
    I := PosNext (':', FieldLine);
    if I = 0 then
      raise EConvertError.Create ('Not a valid field line');
    AddField (Trim (CopyLeft (FieldLine, I - 1)), Trim (CopyFrom (FieldLine, I + 1)));
  End;

Function THeader.Count : Integer;
  Begin
    Result := Length (HeaderFields);
  End;

Function THeader.GetFieldByIndex (const Idx : Integer) : AHeaderField;
  Begin
    Result := HeaderFields [Idx];
  End;

Function THeader.GetField (const FieldName : String) : AHeaderField;
var I : Integer;
    S : String;
  Begin
    S := UpperCase (Trim (FieldName, SPACE));
    For I := 0 to Length (HeaderFields) - 1 do
      if S = UpperCase (HeaderFields [I].FieldName) then
        begin
          Result := HeaderFields [I];
          exit;
        end;
    Result := nil;
  End;

Procedure THeader.SetField (const FieldName : String; const Field : AHeaderField);

  Procedure DeleteField (const FieldIndex : Integer);
  var L : Integer;
    Begin
      L := Length (HeaderFields);
      if FieldIndex + 1 < L then
        Move (HeaderFields [FieldIndex + 1], HeaderFields [FieldIndex], Sizeof (AHeaderField) * (L - FieldIndex - 1));
      SetLength (HeaderFields, L - 1);
    End;

var I : Integer;
    S : String;
  Begin
    S := UpperCase (Trim (FieldName, SPACE));
    For I := 0 to Length (HeaderFields) - 1 do
      if S = UpperCase (HeaderFields [I].FieldName) then
        begin
          FreeAndNil (HeaderFields [I]);
          if Assigned (Field) then
            HeaderFields [I] := Field else
            DeleteField (I);
          exit;
        end;
    if Assigned (Field) then
      AddField (Field);
  End;

Function THeader.HasField (const FieldName : String) : Boolean;
  Begin
    Result := Assigned (GetField (FieldName));
  End;

Function THeader.GetFieldBody (const FieldName : String) : String;
var F : AHeaderField;
  Begin
    F := GetField (FieldName);
    if Assigned (F) then
      Result := F.FieldBody else
      Result := '';
  End;

Function THeader.Encode : String;
var I : Integer;
  Begin
    Result := '';
    For I := 0 to Length (HeaderFields) - 1 do
      Result := Result + HeaderFields [I].AsString;
    Result := Result + CRLF;
  End;

Procedure THeader.Decode (const Header : String);
var S     : StringArray;
    I, J  : Integer;
    Name,
    Body  : String;
  Begin
    S := Split (Header, CRLF);
    Name := '';
    For I := 0 to Length (S) - 1 do
      begin
        J := PosNext (':', S [I]);
        if J > 0 then // header-field
          begin
            Name := Trim (CopyLeft (S [I], J - 1), SPACE);
            Body := Trim (CopyFrom (S [I], J + 1), SPACE);
          end else
          if (Name <> '') and (S [I] <> '') and (S [I][1] in SPACE) then // header-line continuation
            Body := Body + S [I] else
            begin
              if Name <> '' then
                AddField (Name, Body);
              Name := '';
            end;
      end;
    if Name <> '' then
      AddField (Name, Body);
  End;


{                                                                              }
{ Miscellaneous header fields                                                  }
{                                                                              }

{ TDateField                                                                   }
Constructor TDateField.Create (const D : TDateTime);
  Begin
    inherited Create;
    GMTDateTime := LocalTimeToGMTTime (D);
  End;

Function TDateField.GetFieldBody : String;
  Begin
    Result := GMTDateTimeToRFC1123DateTime (GMTDateTime);
  End;

Procedure TDateField.SetFieldBody (const FieldBody : String);
  Begin
    GMTDateTime := RFCDateTimeToGMTDateTime (FieldBody);
  End;

Function TDateField.GetFieldName : String;
  Begin
    Result := 'Date';
  End;

{ TEMailField                                                                  }
Function TEMailField.GetFieldBody : String;
  Begin
    Result := Address + Cond (Name <> '', ' (' + Name + ')', '');
  End;

{ Allowed Usenet News formats (RFC850):                                        }
{     mark@cbosgd.UUCP                                                         }
{     mark@cbosgd.UUCP (Mark Horton)                                           }
{     Mark Horton <mark@cbosgd.UUCP>                                           }
{ Additional formats supported:                                                }
{     <david@e.co.za> David Butler                                             }
{     "David Butler" david@e.co.za                                             }
{     david@e.co.za "David Butler"                                             }
Procedure DecomposeEMailField (const F : String; var EMailAddress, Name : String);
  Begin
    Name := CopyBetween (F, '"', '"');
    if Name = '' then
      begin
        Name := CopyBetween (F, '(', ')');
        if Name = '' then
          begin
            Name := Trim (CopyBefore (F, '<', False));
            if Name = '' then
              Name := Trim (CopyAfter (F, '>'));
          end;
      end;

    EmailAddress := CopyBetween (F, '<', '>');
    if EmailAddress = '' then
      begin
        EmailAddress := Trim (CopyBefore (F, [',', ':', ';', ' ', '"', '('], False));
        if EmailAddress = '' then
          begin
            EmailAddress := Trim (CopyAfter (F, '"'));
            if EmailAddress = '' then
              EmailAddress := F;
          end;
      end;
  End;

Procedure TEMailField.SetFieldBody (const FieldBody : String);
  Begin
    DecomposeEMailField (FieldBody, Address, Name);
  End;

Function UUHeaderLine (const S : String; var FileName : String) : Boolean;
var T : String;
    R : Boolean;
  Begin
    T := LowerCase (S);
    Result := (Copy (T, 1, 6) = 'begin ') or (PosNext ('--- cut here ---', T) > 0);
    if Result then
      begin
        if Copy (T, 1, 6) = 'begin ' then
          T := TrimLeft (TrimLeft (CopyFrom (T, 7), [' ']), ['0'..'9']);
        Repeat
          T := TrimLeft (T, ['-', ' ']);
          R := Copy (T, 1, 8) = 'cut here';
          if R then
            T := CopyFrom (T, 9);
        Until not R;
        FileName := T;
      end else
      FileName := '';
  End;

{                                                                              }
{ MIME Decoding                                                                }
{                                                                              }

{ Examples from RFC2045:                                                       }
{   "MIME-Version: 1.0                                                         }
{    MIME-Version: 1.0 (produced by MetaSend Vx.x)                             }
{    MIME-Version: (produced by MetaSend Vx.x) 1.0                             }
{    MIME-Version: 1.(produced by MetaSend Vx.x)0                           "  }
Procedure DecomposeMimeVersionField (const F : String; var MajorVersion, MinorVersion : Integer; var Comment : String);
var S : String;
  Begin
    Comment := CopyBetween (F, '(', ')', False);
    S := Remove ('(', ')', F);
    MajorVersion := StrToIntDef (Trim (CopyBefore (S, '.')), 0);
    MinorVersion := StrToIntDef (Trim (CopyAfter (S, '.')), 0);
  End;

Procedure TMimeVersionField.SetFieldBody (const FieldBody : String);
  Begin
    DecomposeMimeVersionField (FieldBody, MajorVersion, MinorVersion, Comment);
    inherited SetFieldBody (FieldBody);
  End;

Function TMimeVersionField.GetFieldBody : String;
  Begin
    Result := IntToStr (MajorVersion) + '.' + IntToStr (MinorVersion) +
              Cond (Comment <> '', ' (' + Comment + ')', '');
  End;

Function TMimeVersionField.GetFieldName : String;
  Begin
    Result := 'MIME-Version';
  End;

{ RFC2045:                                                                     }
{   " content := "Content-Type" ":" type "/" subtype                           }
{                *(";" parameter)                                           "  }
Function MimeParameterValue (const Parameters, ParameterName : String) : String;
  Begin
    Result := CopyBetween (LowerCase (Parameters), ParameterName + '=', ';', True);
    if IsQuotedString (Result, ['"']) then
      Result := Copy (Result, 2, Length (Result) - 2);
  End;

Procedure DecomposeMimeContentTypeField (const F : String; var MajorType, SubType, Parameters : String);
  Begin
    MajorType := LowerCase (CopyBefore (F, '/', True));
    SubType := LowerCase (CopyBetween (F, '/', ';', True));
    Parameters := CopyAfter (F, ';');
  End;

Procedure TMimeContentTypeField.SetFieldBody (const FieldBody : String);
  Begin
    DecomposeMimeContentTypeField (FieldBody, MajorType, SubType, Parameters);
  End;

{ From RFC2045 (Multipurpose Internet Mail Extensions (MIME) Part One: Format  }
{   of Internet Message Bodies)                                                }
{                                                                              }
{         encoding := "Content-Transfer-Encoding" ":" mechanism                }
{         mechanism := "7bit" / "8bit" / "binary" /                            }
{                      "quoted-printable" / "base64" /                         }
{                      ietf-token / x-token                                    }
{         x-token := "x-uuencode"                                              }
Procedure TMimeContentTransferEncodingField.SetFieldBody (const FieldBody : String);
var B : String;
  Begin
    B := LowerCase (FieldBody);
    if B = '7bit' then
      TransferEncoding := te_7bit else
    if B = '8bit' then
      TransferEncoding := te_8bit else
    if B = 'binary' then
      TransferEncoding := te_binary else
    if B = 'quoted-printable' then
      TransferEncoding := te_quoted_printable else
    if B = 'base64' then
      TransferEncoding := te_base64 else
    if B = 'x-uuencode' then
      TransferEncoding := te_x_uuencode else
      TransferEncoding := te_none;
  End;

Procedure TMimeMessageHeader.AddField (const FieldName, FieldBody : String);
var S : String;
  Begin
    S := LowerCase (FieldName);
    if FieldName = 'mime-version' then
      AddField (TMimeVersionField.Create (FieldBody)) else
    if FieldName = 'content-type' then
      AddField (TMimeContentTypeField.Create (FieldBody)) else
    if FieldName = 'content-transfer-encoding' then
      AddField (TMimeContentTransferEncodingField.Create (FieldBody)) else
      inherited AddField (FieldName, FieldBody);
  End;

Procedure TMimeMessageHeader.Decode (const Header : String);
var Ver : TMimeVersionField;
    CT : TMimeContentTypeField;
  Begin
    inherited Decode (Header);

    FContentDisposition := GetFieldBody ('Content-Disposition');
    Ver := TMimeVersionField (GetField ('MIME-Version'));
    CT := TMimeContentTypeField (GetField ('Content-Type'));
    FIsMimeMessage := Assigned (Ver) and (Ver.MajorVersion = 1) or Assigned (CT) or (FTransferEncoding <> te_none);

    if FIsMimeMessage then
      begin
        FBoundary := MimeParameterValue (CT.Parameters, 'boundary');
      end;
  End;



{                                                                              }
{ TMailMessageDecoder                                                          }
{                                                                              }
Constructor TMailMessageDecoder.Create;
  Begin
    inherited Create;
    FInlineUUDecode := True;
  End;

{ Mail Encoding                                                                }
Function GenerateMIMEBoundary : String;
  Begin
    Result := 'DF' + IntToHex (RandomUniform) + IntToHex (RandomUniform);
  End;

{ See RFC2045                                                                  }
Function DecodeQuotedPrintableText (const T : String) : String;
var I : Integer;
  Begin
    Result := Remove ([#0..#8, #11..#12, #14..#31], T);

    if Result <> '' then
      begin
        // Trim trailing soft line break (=)
        if Result [Length (Result)] = '=' then
          SetLength (Result, Length (Result) - 1);

        // =HH hex representation of char
        I := 0;
        Repeat
          I := PosNext ('=', Result, I);
          if (I > 0) and (I <= Length (Result) - 2)
             and (Result [I + 1] in cs_HexDigit)
             and (Result [I + 2] in cs_HexDigit) then
            begin
              Result [I] := Char (HexToInt (Result [I + 1] + Result [I + 2]));
              System.Delete (Result, I + 1, 2);
            end;
        Until I = 0;

        // Trim trailing spaces
        Result := TrimRight (Result, cs_WhiteSpace);
      end;
  End;

Function DecodeTransferEncoding (const T : String; const TransferEncoding : TTransferEncoding) : String;
  Begin
    Case TransferEncoding of
      te_quoted_printable : Result := DecodeQuotedPrintableText (T);
      te_base64           : Result := MIMEBase64Decode (T);
      te_x_uuencode       : Result := UUDecode (T);
      else Result := T;
    end;
  End;

Procedure TMultiPartBodyPart.AddPart (const BodyPart : TMessageBodyPart);
var L : Integer;
  Begin
    L := Length (Body);
    SetLength (Body, L + 1);
    Body [L] := BodyPart;
  end;

Procedure TMailMessageDecoder.Decode (const Header : String; const Body : TExStream);
var Line : String;
    TransferEncoding : TTransferEncoding;
    BodyP : TTextParser;

  Function ExtractMimeHeader : TMimeMessageHeader;
  var H, S : String;
    Begin
      H := '';
      Repeat
        S := BodyP.ExtractLine;
        if S <> '' then
          H := H + S + CRLF;
      Until S = '';
      Result := TMimeMessageHeader.Create;
      Result.Decode (H);
    End;

  Function ExtractRest : String;
    Begin
      Result := '';
      While not Body.EOF do
        begin
          Line := BodyP.ExtractLine;
          Result := Result + DecodeTransferEncoding (Line, TransferEncoding);
        end;
    End;

  Function ExtractToBoundary (const Boundary : String; var OpenBoundary : Boolean) : String;
    Begin
      Result := '';
      OpenBoundary := False;
      While not BodyP.EOF do
        begin
          Line := BodyP.ExtractLine;
          OpenBoundary := Line = '--' + Boundary;
          if OpenBoundary or (Line = '--' + Boundary + '--') then 
            exit;
          Result := Result + DecodeTransferEncoding (Line, TransferEncoding);
        end;
    End;

  Function ExtractMimeBodyPart (const ParentHdr : TMimeMessageHeader) : TMessageBodyPart;
  var Hdr  : TMimeMessageHeader;
      CT   : TMimeContentTypeField;
      Open : Boolean;
    Begin
      if ParentHdr.FBoundary <> '' then
        begin
          CT := TMimeContentTypeField (ParentHdr.GetField ('Content-Type'));
          if CT.MajorType = 'multipart' then
            Result := TMultipartBodyPart.Create else
            Result := TMessageBodyPart.Create;
          ExtractToBoundary (ParentHdr.FBoundary, Open);
          While Open do
            begin
              Hdr := ExtractMimeHeader;
              if CT.MajorType = 'multipart' then
                begin
                  TMultipartBodyPart (Result).AddPart (ExtractMimeBodyPart (Hdr));
                  ExtractToBoundary (ParentHdr.FBoundary, Open);
                end else
                Result.Body := Result.Body + ExtractToBoundary (ParentHdr.FBoundary, Open);
            end;
        end else
        begin
          Result := TMessageBodyPart.Create;
          Result.Body := ExtractRest;
        end;
    End;

  Function NextLine : String;
    Begin
      Result := BodyP.ExtractLine;
    End;

  Function ExtractInlineUUEncoding (var Name : String; var DataBad : Boolean) : String;
  var S, T : String;
      R    : Boolean;
      F    : Integer;
    Begin
      // UU header
      Repeat
        S := NextLine;
        R := UUHeaderLine (S, T);
        if R and (Name = '') then
          Name := T;
        R := R or (Length (T) <= 1);
      Until not R or BodyP.EOF;

      // UU data
      Result := '';
      DataBad := False;
      if not BodyP.EOF then
        Repeat
          if Length (T) > 1 then
            if LowerCase (Copy (T, 1, 3)) = 'end' then
              break else
              begin
                System.Delete (T, 1, 1);
                try
                  Result := Result + UUDecode (T);
                except
                  DataBad := True;
                end;
              end;
          R := BodyP.EOF;
          if not R then
            T := NextLine;
        Until R;

      // UU footer
      F := BodyP.Stream.Position;
      T := LowerCase (NextLine);
      if PosNext ('cut here', T) = 0 then
        BodyP.Stream.Position := F;
    End;

  Function ExtractRFC822BodyPart : TMessageBodyPart;
  var S, N    : String;
      Data    : String;
      DataBad : Boolean;
      MB      : TMultiPartBodyPart;
      B       : TMessageBodyPart;
      R       : Boolean;
    Begin
      MB := TMultiPartBodyPart.Create;
      While not BodyP.EOF do
        begin
          Data := '';
          R := False;
          While not BodyP.EOF and not R do
            begin
              S := NextLine;
              R := FInlineUUDecode and UUHeaderLine (S, N);
              if not R then
                if not Assigned (FOnTextMessageLine) then
                  Data := Data + S + CRLF else
                  if FOnTextMessageLine (S) then
                    Data := Data + S + CRLF;
            end;

          if Data <> '' then
            begin
              B := TTextBodyPart.Create;
              B.Body := Data;
              MB.AddPart (B);
            end;

          if R then
            begin
              B := TAttachmentBodyPart.Create;
              B.Body := ExtractInlineUUEncoding (N, DataBad);
              TAttachmentBodyPart (B).Name := N;
            end;
        end;
      Result := MB;
    End;

var MHdr : TMimeMessageHeader;
    TE   : TMimeContentTransferEncodingField;
  Begin
    //                                                             Decode header
    MHdr := TMimeMessageHeader.Create;
    MHdr.Decode (Header);
    FHeader := MHdr;

    //                                                               Decode body
    BodyP := TTextParser.Create (Body);
    if MHdr.IsMimeMessage then
      begin
        TE := TMimeContentTransferEncodingField (MHdr.GetField ('Content-Transfer-Encoding'));
        if Assigned (TE) then
          TransferEncoding := TE.TransferEncoding else
          TransferEncoding := te_None;
        FBody := ExtractMimeBodyPart (MHdr);
      end else
      FBody := ExtractRFC822BodyPart;
    FreeAndNil (BodyP);
  End;






{                                                                              }
{ HTML Representations                                                         }
{                                                                              }

Constructor AhtmlRenderer.Create (const Parent : AhtmlRenderer);
  Begin
    inherited Create;
    if Assigned (Parent) then
      begin
        FontSize := Parent.FontSize;
        FontColour := Parent.FontColour;
        FontBackColour := Parent.FontBackColour;
        Attributes := Parent.Attributes;
      end else
      begin
        FontSize := 3;
      end;
  End;

{                                                                              }
{ HTML Decoding                                                                }
{                                                                              }
type
  TColEntry = record
    htmlColour : String;
    Colour     : TColor;
  end;

const
  htmlPredefinedCols = 18;
  htmlColTable : Array [1..htmlPredefinedCols] of TColEntry = (
      { HTML4.0 Colours }
      (htmlColour: 'BLACK';   Colour:clBlack),  (htmlColour: 'GREEN';  Colour:clGreen),
      (htmlColour: 'SILVER';  Colour:clSilver), (htmlColour: 'LIME';   Colour:clLime),
      (htmlColour: 'GRAY';    Colour:clGray),   (htmlColour: 'OLVE';   Colour:clOlive),
      (htmlColour: 'WHITE';   Colour:clWhite),  (htmlColour: 'YELLOW'; Colour:clYellow),
      (htmlColour: 'MAROON';  Colour:clMaroon), (htmlColour: 'NAVY';   Colour:clNavy),
      (htmlColour: 'RED';     Colour:clRed),    (htmlColour: 'BLUE';   Colour:clBlue),
      (htmlColour: 'PURPLE';  Colour:clPurple), (htmlColour: 'TEAL';   Colour:clTeal),
      (htmlColour: 'FUCHSIA'; Colour:clFuchsia),(htmlColour: 'AQUA';   Colour:clAqua),
      { Custom extentions }
      (htmlColour: 'LIGHTGRAY'; Colour:clLtGray),  (htmlColour: 'DARKGRAY'; Colour:clDkGray));

Function htmlColour (const Colour : String) : TColor;
var I : Integer;
    C : Integer;
  Begin
    if Colour [1] = '#' then
      begin
        C := HexToInt (CopyFrom (Colour, 2));
        Result := ((C and $FF) shl 16) or
                  (C and $FF00) or
                  ((C and $FF0000) shr 16);
      end else
      begin
        For I := 1 to htmlPredefinedCols do
          begin
            if htmlColTable [I].htmlColour = Colour then
              begin
                Result := htmlColTable [I].Colour;
                exit;
              end;
          end;
        raise EConvertError.Create ('Invalid HTML colour.');
      end;
  End;

Procedure htmlDecodeSize (const S : String; var Size : Integer; var SizeType : ThtmlSizeType);
  Begin
    if S <> '' then
      try
        Case S [Length (S)] of
          '%' :
            begin
              SizeType := stPercent;
              Size := StrToInt (CopyLeft (S, Length (S) - 1));
            end;
          '*' :
            begin
              SizeType := stRelative;
              Size := StrToInt (CopyLeft (S, Length (S) - 1));
            end;
          else
            begin
              SizeType := stPixels;
              Size := StrToInt (S);
            end;
        end;
      except
        SizeType := stNone;
        Size := 0;
      end else
      begin
        SizeType := stNone;
        Size := 0;
      end;
  End;

{ Translate                                                                    }
Function Translate (const Ch : Char; const TranslationTable : String) : Char;
  Begin
    Assert (Length (TranslationTable) = 256, 'Translation table must be 256 characters long.');
    Result := TranslationTable [Ord (Ch) + 1];
  End;

Function Translate (const S : String; const TranslationTable : String) : String;
var I : Integer;
  Begin
    Assert (Length (TranslationTable) = 256, 'Translation table must be 256 characters long.');
    Result := S;
    For I := 1 to Length (Result) do
      Result [I] := TranslationTable [Ord (Result [I]) + 1];
  End;




end.



    TSocketOption = (soBroadcast, soDebug, soDontLinger,
                     soDontRoute, soKeepAlive, soOOBInLine,
                     soReuseAddr, soNoDelay, soBlocking, soAcceptConn);

    TSocketOptions = Set of TSocketOption;

    TSocketClass = class of TSocket;

    TSocket = class
    public
      Dead: Integer;
      FPort: DWORD;
      FAddr: DWORD;
      Handle: DWORD;
      Status: Integer;
      Registered: Boolean;
      procedure RegisterSelf;
      procedure DeregisterSelf;

      function Startup: Boolean; virtual;
      function Handshake: Boolean; virtual;
      destructor Destroy; override;

      function Read(var B; Size: DWORD): DWORD;
      function Write(const B; Size: DWORD): DWORD;
      function WriteStr(const s: string): DWORD;

      function _Write(const B; Size: DWORD): DWORD; virtual;
      function _Read(var B; Size: DWORD): DWORD; virtual;
    end;

