{$INCLUDE cHeader.inc}
unit cIdent;

{                                                                              }
{ Ident server v0.03 (L1)                                                      }
{                                                                              }
{   See RFC 1413 for details.                                                  }
{                                                                              }
{        This unit is copyright  2000 by David Butler (david@e.co.za)         }
{                                                                              }
{                  This unit is part of Delphi Fundamentals.                   }
{                    It's original file name is cIdent.pas                     }
{                                                                              }
{                I invite you to use this unit, free of charge.                }
{        I invite you to distibute this unit, but it must be for free.         }
{             I also invite you to contribute to its development,              }
{             but do not distribute a modified copy of this file.              }
{       Send modifications, suggestions and bug reports to david@e.co.za       }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   2000/06/21  v0.01  Initial version.                                        }
{   2000/06/27  v0.02  Added UserNameByIP.                                     }
{   2000/06/28  v0.03  Added UserNameByPort.                                   }
{                                                                              }
interface

uses
  scktcomp,
  SysUtils,

  cDataStructs;


{                                                                              }
{ TIdentServer                                                                 }
{   Set UserName to the default username (Set blank to return NO-USER message  }
{   by default).                                                               }
{   Set UserNameByIP to specify a username depending on the source of the      }
{   request.                                                                   }
{   Set UserNameByPort to specify a username depending on the local port       }
{   requested (takes precedence over a UserNameByIP setting).                  }
{                                                                              }
type
  TIdentServer = class;
  TOnIdentServerRequest = Procedure (const Server : TIdentServer; const ClientSocket : TServerClientWinSocket; const LocalPort, RemotePort : Integer; var UserName : String) of object;
  EIdentServer = class (Exception);
  TIdentServer = class
    protected
    FActive         : Boolean;
    FPort           : Integer;
    FUserName       : String;
    FUserNameByIP   : TStringDictionary;
    FUserNameByPort : TStringDictionary;
    FServer         : TServerSocket;
    FOnRequest      : TOnIdentServerRequest;
    FRequests       : Integer;
    FLastIP         : String;
    FLastHost       : String;

    Procedure SetActive (const Active : Boolean);
    Procedure OnGetThread (Sender : TObject; ClientSocket: TServerClientWinSocket; var SocketThread: TServerClientThread);
    Function  GetUserNameByIP (const IP : String) : String;
    Procedure SetUserNameByIP (const IP : String; const UserName : String);
    Function  GetUserNameByPort (const Port : Integer) : String;
    Procedure SetUserNameByPort (const Port : Integer; const UserName : String);

    public
    Constructor Create;
    Destructor Destroy; override;

    Property Active : Boolean read FActive write SetActive;
    Property Port : Integer read FPort write FPort default 113;
    Property UserName : String read FUserName write FUserName;
    Property UserNameByIP [const IP : String] : String read GetUserNameByIP write SetUserNameByIP;
    Property UserNameByPort [const Port : Integer] : String read GetUserNameByPort write SetUserNameByPort;
    Property OnRequest : TOnIdentServerRequest read FOnRequest write FOnRequest;
    Property RequestCount : Integer read FRequests;
    Property LastIP : String read FLastIP;
    Property LastHost : String read FLastHost;
  end;



implementation

uses
  cUtils,
  cStrings,
  cStreams;



const
  Ident_Socket_TimeOut = 60 * 1000;

{ TIdentServerClientThread                                                    }
type
  TIdentServerClientThread = class (TServerClientThread)
    protected
    FServer     : TIdentServer;
    FUserName   : String;
    FLocalPort  : Integer;
    FRemotePort : Integer;

    Procedure NotifyRequest;

    public
    Constructor Create (const Server : TIdentServer; const ClientSocket : TServerClientWinSocket);
    Procedure ClientExecute; override;
  end;

Constructor TIdentServerClientThread.Create (const Server : TIdentServer; const ClientSocket : TServerClientWinSocket);
  Begin
    inherited Create (False, ClientSocket);
    FServer := Server;
  End;

Procedure TIdentServerClientThread.NotifyRequest;
  Begin
    FServer.FLastIP := ClientSocket.RemoteAddress;
    FServer.FLastHost := ClientSocket.RemoteHost;
    Inc (FServer.FRequests);

    FUserName := FServer.UserNameByPort [FLocalPort];
    if FUserName = '' then
      begin
        FUserName := FServer.UserNameByIP [ClientSocket.RemoteHost];
        if FUserName = '' then
          FUserName := FServer.UserName;
      end;

    if Assigned (FServer.FOnRequest) then
      FServer.FOnRequest (FServer, ClientSocket, FLocalPort, FRemotePort, FUserName);
  End;

Procedure TIdentServerClientThread.ClientExecute;
var S : String;
    Client : TSocketStream;
  Begin
    try
      Client := TSocketStream.CreateEx (ClientSocket, Ident_Socket_TimeOut, True);
      try
        Repeat
          S := '';
          Repeat
            if Terminated or not ClientSocket.Connected then
              exit;
            S := S + Char (Client.Read);
          Until (Length (S) > 32) or (S [Length (S)] = #10);

          S := Remove ([#0..#32], Trim (S, [#0..#32]));
          FLocalPort := StrToIntDef (CopyBefore (S, ',', True), -1);
          FRemotePort := StrToIntDef (CopyAfter (S, ',', False), -1);
          Synchronize (NotifyRequest);

          if FUserName = '' then
            Client.Write (S + ':ERROR:NO-USER' + c_CRLF) else
            Client.Write (S + ':USERID:WIN32:' + FUserName + c_CRLF);
        Until Terminated or not ClientSocket.Connected;
      finally
        FreeAndNil (Client);
      end;
    except
    end;
  End;



{ TIdentServer                                                                }
Constructor TIdentServer.Create;
  Begin
    inherited Create;
    FPort := 113;
    FUsername := '';
    FUserNameByIP := TStringDictionary.Create;
    FUserNameByIP.CreateOnSet := True;
    FUserNameByIP.CreateOnGet := False;
    FUserNameByPort := TStringDictionary.Create;
    FUserNameByPort.CreateOnSet := True;
    FUserNameByPort.CreateOnGet := False;
  End;

Destructor TIdentServer.Destroy;
  Begin
    FreeAndNil (FServer);
    FreeAndNil (FUserNameByPort);
    FreeAndNil (FUserNameByIP);
    inherited Destroy;
  End;

Procedure TIdentServer.OnGetThread (Sender : TObject; ClientSocket: TServerClientWinSocket; var SocketThread: TServerClientThread);
  Begin
    SocketThread := TIdentServerClientThread.Create (self, ClientSocket);
  End;

Function TIdentServer.GetUserNameByIP (const IP : String) : String;
  Begin
    if FUserNameByIP.HasKey (IP) then
      Result := FUserNameByIP [IP] else
      Result := '';
  End;

Procedure TIdentServer.SetUserNameByIP (const IP : String; const UserName : String);
  Begin
    FUserNameByIP [IP] := UserName;
  End;

Function TIdentServer.GetUserNameByPort (const Port : Integer) : String;
var P : String;
  Begin
    P := IntToStr (Port);
    if FUserNameByPort.HasKey (P) then
      Result := FUserNameByPort [P] else
      Result := '';
  End;

Procedure TIdentServer.SetUserNameByPort (const Port : Integer; const UserName : String);
  Begin
    FUserNameByPort [IntToStr (Port)] := UserName;
  End;

Procedure TIdentServer.SetActive (const Active : Boolean);
  Begin
    if FActive = Active then
      exit;

    if Active then
      try
        FServer := TServerSocket.Create (nil);
        FServer.Port := FPort;
        FServer.ThreadCacheSize := 1;
        FServer.ServerType := stThreadBlocking;
        FServer.OnGetThread := OnGetThread;
        FServer.Active := True;
      except
        FreeAndNil (FServer);
        raise;
      end else
      begin
        try
          FServer.Active := False;
        except end;
        FreeAndNil (FServer);
      end;

    FActive := Active;
  End;

end.







{ Possible extentions to original RFC:                                         }
{   If remoteport = 0, the request is to identify the service listening on     }
{   the local port, for example:                                               }
{   80,0                         ' Request service listening on port 80        }
{   80,0:SERVICE::HTTP/1.1       ' Reply: HTTP/1.1 service running on port 80  }
{                                                                              }
{   81,0                                                                       }
{   81,0:ERROR:NO-SERVICE        ' Reply: No service running on port 81        }
{                                                                              }
{   To list all (public) services, specify localport = *,                      }
{   *,0                          ' Request all services listening for public   }
{   113,0:SERVICE::IDENT/2.0     ' Service IDENT/2.0 running on port 113       }
{   8080,0:SERVICE::HTTP/1.1     ' Service HTTP/1.1 running on port 8080       }
{   <empty line>                 ' List terminated with an empty line          }
{                                                                              }
{   To list all services belonging to a specific user:                         }
{   *,0:david                    ' Request all services listening for david    }
{   6121,0:SERVICE:david:ONE2ONE/1.0:vcard,chat,files,search                   }
{   8081,0:SERVICE:david:HTTP/1.1                                              }
{   <empty line>                 ' List terminated with an empty line          }
{                                                                              }
{   To request user information:                                               }
{   0,0:david                    ' Request user information for david          }
{   0,0:USER:david:DOMAIN/1.0:dj.nailed.org                                    }
{   0,0:USER:david:EMAIL/1.0:david@e.co.za                                     }
{   0,0:USER:david:NAME/1.0:David Butler                                       }
{   <empty line>                 ' List terminated with an empty line          }
{                                                                              }

