{$INCLUDE cHeader.inc}
unit cIRC;

{                                                                              }
{ IRC Client v0.02 (L1)                                                        }
{                                                                              }
{   Implements an IRC client that can connect to multiple servers              }
{   simultaneously. Everything runs event driven in the main thread.           }
{                                                                              }
{                                                                              }
{        This unit is copyright  2000 by David Butler (david@e.co.za)         }
{                                                                              }
{                  This unit is part of Delphi Fundamentals.                   }
{                    It's original file name is cIRC.pas                       }
{                                                                              }
{                I invite you to use this unit, free of charge.                }
{        I invite you to distibute this unit, but it must be for free.         }
{             I also invite you to contribute to its development,              }
{             but do not distribute a modified copy of this file.              }
{       Send modifications, suggestions and bug reports to david@e.co.za       }
{                                                                              }
{                                                                              }
{ Revision history:                                                            }
{   21/06/2000  v0.01  Initial version.                                        }
{   25/06/2000  v0.02  Further development.                                    }
{                                                                              }
interface

uses
  extctrls,
  scktcomp,
  Graphics, // TColor
  Classes,

  cUtils,
  cDataStructs;



{                                                                              }
{ TIRCClient                                                                   }
{   AutoReLogin: Tries to log in again if logged in connection is broken.      }
{   Tries to log in 'LogInRetries' times, waiting 'LogInRetryWait' ms between  }
{   attempts.                                                                  }
{   LogInTimeout: Time to wait for welcome line after logging in.              }
{                                                                              }
type
  TIRCClient = class;
  TOnIRCClientNotify = Procedure (const Sender : TIRCClient) of object;
  TOnIRCClientPrivMsg = Procedure (const Sender : TIRCClient;
                        const Destination, User, Msg : String) of object;
  TOnIRCClientMessage = Procedure (const Sender : TIRCClient;
                        const Name, User, Host : String;
                        const NumericCmd : Integer; const Cmd : String;
                        const Parameters : StringArray) of object;
  TOnIRCClientChannelJoin = Procedure (const Sender : TIRCClient; const Channel : String;
                            const NickList : StringArray) of object;
  TOnIRCClientNickInUse = Procedure (const Sender : TIRCClient; var NewNick : String) of object;
  TIRCClient = class
    protected
    FHost      : String;
    FPort      : Integer;
    FUserName  : String;
    FPassword  : String;
    FNick      : String;
    FRealName  : String;
    FBuddyList : StringArray;

    FActive         : Boolean;
    FLoggedIn       : Boolean;
    FLoggingIn      : Boolean;
    FOnLoggingIn    : TOnIRCClientNotify;
    FOnLoggedIn     : TOnIRCClientNotify;
    FOnLoggedOut    : TOnIRCClientNotify;
    FLoginTimer     : TTimer;
    FLoginTimeOut   : Integer;
    FLoginRetries   : Integer;
    FLoginRetryWait : Integer;
    FLoginAttemptNr : Integer;
    FAutoReLogin    : Boolean;

    FClient : TClientSocket;
    FInBuf  : String;

    FOnMessage   : TOnIRCClientMessage;
    FOnNickInUse : TOnIRCClientNickInUse;
    FOnPrivMsg   : TOnIRCClientPrivMsg;

    FBuddiesOnline         : StringArray;
    FBuddiesTmp            : StringArray;
    FBuddyTimer            : TTimer;
    FIsOnlineCount         : Integer;
    FOnBuddiesOnlineChange : TOnIRCClientNotify;

    FOnChannelJoin : TOnIRCClientChannelJoin;
    FChanNames     : TStringDictionary;

    FMOTD    : String;
    FGotMOTD : Boolean;

    Procedure SetActive (const Active : Boolean);
    Procedure OnClientConnect (Sender: TObject; Socket: TCustomWinSocket);
    Procedure OnClientDisconnect (Sender: TObject; Socket: TCustomWinSocket);
    Procedure OnClientError (Sender: TObject; Socket: TCustomWinSocket; ErrorEvent: TErrorEvent; var ErrorCode: Integer);
    Function  IsConnected : Boolean;
    Procedure SetLoggedIn;
    Procedure SetLoggedOut;
    Procedure AttemptLogin;
    Procedure LogIn;
    Procedure LogOut;
    Procedure OnLoginTimeOut (Sender : TObject);
    Procedure OnLoginRetry (Sender : TObject);
    Procedure OnReLogin (Sender : TObject);

    Procedure OnClientRead (Sender: TObject; Socket: TCustomWinSocket);

    Procedure SetNick (const Nick : String);

    Procedure CheckBuddyList;
    Procedure OnBuddyTimer (Sender : TObject);
    Procedure SetBuddyList (const BuddyList : StringArray);

    public
    Constructor Create;
    Destructor Destroy; override;

    Property  Host : String read FHost write FHost;
    Property  Port : Integer read FPort write FPort;
    Property  UserName : String read FUserName write FUserName;
    Property  Password : String read FPassword write FPassword;
    Property  Nick : String read FNick write SetNick;
    Property  RealName : String read FRealName write FRealName;
    Property  OnNickInUse : TOnIRCClientNickInUse read FOnNickInUse write FOnNickInUse;

    Property  Active : Boolean read FActive write SetActive;
    Property  LoginTimeout : Integer read FLoginTimeOut write FLoginTimeOut;
    Property  LoginRetries : Integer read FLoginRetries write FLoginRetries;
    Property  LoginRetryWait : Integer read FLoginRetryWait write FLoginRetryWait;
    Property  AutoReLogin : Boolean read FAutoReLogin write FAutoReLogin;
    Property  LoggingIn : Boolean read FLoggingIn;
    Property  LoginAttemptNr : Integer read FLoginAttemptNr;
    Property  LoggedIn : Boolean read FLoggedIn;
    Property  OnLoggingIn : TOnIRCClientNotify read FOnLoggingIn write FOnLoggingIn;
    Property  OnLoggedIn : TOnIRCClientNotify read FOnLoggedIn write FOnLoggedIn;
    Property  OnLoggedOut : TOnIRCClientNotify read FOnLoggedOut write FOnLoggedOut;

    Property  OnMessage : TOnIRCClientMessage read FOnMessage write FOnMessage;

    Property  MOTD : String read FMOTD;

    Procedure AddBuddy (const Server, Nick : String);
    Procedure RemoveBuddy (const Server, Nick : String);
    Property  BuddiesOnline : StringArray read FBuddiesOnline;
    Property  BuddyList : StringArray read FBuddyList write SetBuddyList;
    Property  OnBuddiesOnlineChange : TOnIRCClientNotify read FOnBuddiesOnlineChange write FOnBuddiesOnlineChange;

    Procedure Join (const Channel : String);
    Procedure Part (const Channel : String);
    Procedure PrivMsg (const Destination, Msg : String);
    Property  OnChannelJoin : TOnIRCClientChannelJoin read FOnChannelJoin write FOnChannelJoin;
    Property  OnPrivMsg : TOnIRCClientPrivMsg read FOnPrivMsg write FOnPrivMsg;
  end;



{                                                                              }
{ TMultiIRCClient                                                              }
{                                                                              }
type
  TJoinChannel = class
    Server  : String;
    Channel : String;
  end;
  TMultiIRCClient = class;
  TOnNewIRCClient = Procedure (const Sender : TMultiIRCClient; var Client : TIRCClient; const Host : String; const Port : Integer) of object;
  TMultiIRCClient = class
    protected
    FActive     : Boolean;
    FPersistent : StringArray;
    FJoins      : Array of TJoinChannel;

    FClients     : Array of TIRCClient;
    FOnNewClient : TOnNewIRCClient;

    Procedure CheckClientConnections;
    Procedure SetActive (const Active : Boolean);
    Function  GetServers : StringArray;
    Function  GetPersistentClient (const Server : String) : Boolean;
    Procedure SetPersistentClient (const Server : String; const PersistentClient : Boolean);
    Function  GetClient (const Server : String) : TIRCClient;
    Procedure RemoveAllJoins;
    Function  IndexOfChannel (const Server : String; const Channel : String) : Integer;
    Procedure RemoveAllClients;
    Procedure OnLoggedIn (const Sender : TIRCClient);

    public
    Destructor Destroy; override;

    Procedure JoinChannel (const Server : String; const Channel : String);
    Procedure PartChannel (const Server : String; const Channel : String);
    Property  PersistentClient [const Server : String] : Boolean read GetPersistentClient write SetPersistentClient;
    Procedure RemoveClient (const Server : String);

    Property  Active : Boolean read FActive write SetActive;
    Property  Client [const Server : String] : TIRCClient read GetClient;
    Property  Servers : StringArray read GetServers;

    Property  OnNewClient : TOnNewIRCClient read FOnNewClient write FOnNewClient;
  end;



{                                                                              }
{ IRC "Numerics" are from various sources, those that are not part of the      }
{ RFC1459 specification (ie not used, reserved or used in extentions) have a   }
{ RPL_X_ or ERR_X_ prefix.                                                     }
{                                                                              }
const
  { 000-999 Server -> Local client                                             }
  { 001-004 Welcome messages                                                   }
  RPL_X_WELCOME           = 001;
  RPL_X_YOURHOST          = 002;
  RPL_X_CREATED           = 003;
  RPL_X_MYINFO            = 004;

  RPL_X_MAP		  = 005; { Undernet extension                          }
  RPL_X_MAPMORE	          = 006; { Undernet extension                          }
  RPL_X_MAPEND	          = 007; { Undernet extension                          }
  RPL_X_SNOMASK	          = 008; { Undernet extension                          }
  RPL_X_STATMEMTOT        = 009; { Undernet extension                          }
  RPL_X_STATMEM	          = 010; { Undernet extension                          }
  RPL_X_YOURCOOKIE        = 014; { IRCnet extension                            }

  { 200-399 Replies to client commands to server                               }
  RPL_TRACELINK	          = 200; { Link <version & debug level> <destination> <next server> }
  RPL_TRACECONNECTING	  = 201; { Try. <class> <server>                       }
  RPL_TRACEHANDSHAKE	  = 202; { H.S. <class> <server>                       }
  RPL_TRACEUNKNOWN	  = 203; { ???? <class> [<client IP address in dot form>] }
  RPL_TRACEOPERATOR	  = 204; { Oper <class> <nick>                         }
  RPL_TRACEUSER	          = 205; { User <class> <nick>                         }
  RPL_TRACESERVER	  = 206; { Serv <class> <int>S <int>C <server> <nick!user|*!*>@<host|server> }
  RPL_X_TRACESERVICE      = 207;
  RPL_TRACENEWTYPE	  = 208; { <newtype> 0 <client name>                   }
  RPL_X_TRACECLASS        = 209;

  RPL_STATSLINKINFO	  = 211; { <linkname> <sendq> <sent messages> <sent bytes> <received messages> <received bytes> <time open> }
  RPL_STATSCOMMANDS	  = 212; { <command> <count>                           }
  RPL_STATSCLINE	  = 213; { C <host> * <name> <port> <class>            }
  RPL_STATSNLINE	  = 214; { N <host> * <name> <port> <class>            }
  RPL_STATSILINE	  = 215; { I <host> * <host> <port> <class>            }
  RPL_STATSKLINE	  = 216; { K <host> * <username> <port> <class>        }
  RPL_X_STATSQLINE        = 217;
  RPL_X_STATSPLINE        = 217; { Undernet extenstion                         }
  RPL_STATSYLINE	  = 218; { Y <class> <ping frequency> <connect frequency> <max sendq> }
  RPL_ENDOFSTATS	  = 219; { <stats letter> :End of /STATS report        }

  RPL_UMODEIS	          = 221; { <user mode string>                          }

  RPL_X_SERVICEINFO       = 231;
  RPL_X_ENDOFSERVICES     = 232;
  RPL_X_SERVICE           = 233;
  RPL_X_SERVLIST          = 234;
  RPL_X_SERVLISTEND       = 235;

  RPL_STATSLLINE	  = 241; { L <hostmask> * <servername> <maxdepth>      }
  RPL_STATSUPTIME	  = 242; { :Server Up %d days %d:%02d:%02d             }
  RPL_STATSOLINE	  = 243; { O <hostmask> * <name>                       }
  RPL_STATSHLINE	  = 244; { H <hostmask> * <servername>                 }
  RPL_X_STATSSLINE        = 245; { Reserved                                    }
  RPL_X_STATSTLINE	  = 246; { Undernet extension                          }
  RPL_X_STATSGLINE	  = 247; { Undernet extension                          }
  RPL_X_STATSULINE	  = 248; { Undernet extension                          }
  RPL_X_STATSDEBUG	  = 249; { Extension to RFC1459                        }
  RPL_X_STATSCONN	  = 250; { Undernet extension                          }

  RPL_LUSERCLIENT	  = 251; { :There are <integer> users and <integer> invisible on <integer> servers }
  RPL_LUSEROP	          = 252; { <integer> :operator(s) online               }
  RPL_LUSERUNKNOWN	  = 253; { <integer> :unknown connection(s)            }
  RPL_LUSERCHANNELS	  = 254; { <integer> :channels formed                  }
  RPL_LUSERME	          = 255; { :I have <integer> clients and <integer> servers }
  RPL_ADMINME	          = 256; { <server> :Administrative info               }
  RPL_ADMINLOC1	          = 257; { :<admin info>                               }
  RPL_ADMINLOC2	          = 258; { :<admin info>                               }
  RPL_ADMINEMAIL	  = 259; { :<admin info>                               }

  RPL_TRACELOG	          = 261; { File <logfile> <debug level>                }
  RPL_X_TRACEPING         = 262; { Extension to RFC1459                        }
  RPL_X_LOCUSERS          = 265; { Current local users: <integer>  Max: <integer> }
  RPL_X_GLOBALUSERS       = 266; { Current global users: <integer>  Max: <integer> }

  RPL_X_SILELIST	  = 271; { Undernet extension                          }
  RPL_X_ENDOFSILELIST     = 272; { Undernet extension                          }
  RPL_X_STATSDELTA        = 274; { IRCnet extension                            }
  RPL_X_STATSDLINE        = 275; { Undernet extension                          }

  RPL_X_GLIST             = 280; { Undernet extension                          }
  RPL_X_ENDOFGLIST	  = 281; { Undernet extension                          }

  RPL_NONE	          = 300; { Dummy reply number. Not used.               }
  RPL_AWAY                = 301; { <nick> :<away message>                      }
  RPL_USERHOST	          = 302; { :[<reply><space><reply>]                    }
  RPL_ISON	          = 303; { :[<nick> <space><nick>]                     }
  RPL_X_TEXT              = 304;
  RPL_UNAWAY	          = 305; { :You are no longer marked as being away     }
  RPL_NOWAWAY	          = 306; { :You have been marked as being away         }
  RPL_X_USERIP            = 307; { Undernet extension                          }

  { 311-313, 316-319 replies to WHOIS                                          }
  RPL_WHOISUSER	          = 311; { <nick> <user> <host> * :<real name>         }
  RPL_WHOISSERVER	  = 312; { <nick> <server> :<server info>              }
  RPL_WHOISOPERATOR	  = 313; { <nick> :is an IRC operator                  }
  // RPL_WHOWASUSER       = 314; { defined below                               }
  // RPL_ENDOFWHO         = 315; { defined below                               }
  RPL_X_WHOISCHANOP       = 316; { redundant, reserved                         }
  RPL_WHOISIDLE	          = 317; { <nick> <integer> :seconds idle              }
  RPL_ENDOFWHOIS	  = 318; { <nick> :End of /WHOIS list                  }
  RPL_WHOISCHANNELS	  = 319; { <nick> :[@|+]<channel><space>               }

  RPL_WHOWASUSER	  = 314; { <nick> <user> <host> * :<real name>         }
  RPL_ENDOFWHOWAS	  = 369; { <nick> :End of WHOWAS                       }

  RPL_WHOREPLY	          = 352; { <channel> <user> <host> <server> <nick> <H|G>[*][@|+] :<hopcount> <real name> }
  RPL_ENDOFWHO	          = 315; { <name> :End of /WHO list                    }

  RPL_LISTSTART	          = 321; { Channel :Users  Name                        }
  RPL_LIST	          = 322; { <channel> <# visible> :<topic>              }
  RPL_LISTEND	          = 323; { :End of /LIST                               }

  RPL_CHANNELMODEIS	  = 324; { <channel> <mode> <mode params>              }
  RPL_CHANNELPASSIS       = 325; { IRCnet extension                            }
  RPL_NOCHANPASS          = 326; { IRCnet extension                            }
  RPL_CHPASSUNKNOWN       = 327; { IRCnet extension                            }

  RPL_NOTOPIC	          = 331; { <channel> :No topic is set                  }
  RPL_TOPIC	          = 332; { <channel> :<topic>                          }
  RPL_X_TOPICWHOTIME      = 333; { Undernet extension                          }
  RPL_X_LISTUSAGE	  = 334; { Undernet extension                          }
  RPL_CHANPASSOK          = 338; { IRCnet extension                            }
  RPL_BADCHANPASS         = 339; { IRCnet extension                            }

  RPL_INVITING	          = 341; { <channel> <nick>                            }
  RPL_X_SUMMONING         = 342; { <user> :Summoning user to IRC (removed from RFC1459) }
  RPL_X_EXCEPTLIST        = 348; { IRCnet extension                            }
  RPL_X_ENDOFEXCEPTLIST   = 349; { IRCnet extension                            }

  RPL_VERSION	          = 351; { <version>.<debuglevel> <server> :<comments> }
  // RPL_WHOREPLY         = 352; { defined above                               }
  RPL_X_WHOSPCRPL         = 354; { Undernet extension                          }

  RPL_NAMREPLY	          = 353; { <channel> :[[@|+]<nick> [[@|+]<nick> [...]]]}
                                 { Note: Most servers send "= " before above.  }
  RPL_ENDOFNAMES	  = 366; { <channel> :End of /NAMES list               }

  RPL_X_KILLDONE          = 361;
  RPL_X_CLOSING           = 362;
  RPL_X_CLOSEEND          = 363;
  RPL_LINKS	          = 364; { <mask> <server> :<hopcount> <server info>   }
  RPL_ENDOFLINKS	  = 365; { <mask> :End of /LINKS list                  }
  // RPL_ENDOFNAMES       = 366; { defined above                               }
  RPL_BANLIST	          = 367; { <channel> <banid>                           }
  RPL_ENDOFBANLIST	  = 368; { <channel> :End of channel ban list          }
  // RPL_ENDOFWHOWAS      = 369; { defined above                               }

  RPL_INFO	          = 371; { :<string>                                   }
  RPL_ENDOFINFO	          = 374; { :End of /INFO list                          }

  RPL_MOTDSTART	          = 375; { ":- <server> Message of the day -,"         }
  RPL_MOTD	          = 372; { :- <text>                                   }
  RPL_ENDOFMOTD	          = 376; { :End of /MOTD command                       }

  RPL_X_INFOSTART         = 373;

  RPL_YOUREOPER	          = 381; { :You are now an IRC operator                }
  RPL_REHASHING	          = 382; { <config file> :Rehashing                    }
  RPL_X_YOURESERVICE      = 383;
  RPL_X_MYPORTIS          = 384;
  RPL_X_NOTOPERANYMORE    = 385; { Extension to RFC1459                        }

  RPL_TIME                = 391;
  RPL_USERSSTART	  = 392; { :UserID   Terminal  Host                    }
  RPL_USERS	          = 393; { :%-8s %-9s %-8s                             }
  RPL_ENDOFUSERS	  = 394; { :End of users                               }
  RPL_NOUSERS	          = 395; { :Nobody logged in                           }

  { 400-599 Errors                                                             }
  ERR_NOSUCHNICK	  = 401; { <nickname> :No such nick/channel            }
  ERR_NOSUCHSERVER	  = 402; { <server name> :No such server               }
  ERR_NOSUCHCHANNEL	  = 403; { <channel name> :No such channel             }
  ERR_CANNOTSENDTOCHAN	  = 404; { <channel name> :Cannot send to channel      }
  ERR_TOOMANYCHANNELS	  = 405; { <channel name> :You have joined too many channels }
  ERR_WASNOSUCHNICK	  = 406; { <nickname> :There was no such nickname      }
  ERR_TOOMANYTARGETS	  = 407; { <target> :Duplicate recipients. No message delivered }
  ERR_X_NOSUCHSERVICE     = 408;
  ERR_NOORIGIN	          = 409; { :No origin specified                        }

  ERR_NORECIPIENT	  = 411; { :No recipient given (<command>)             }
  { 412-414 indicate PRIVMSG wasn't delivered                                  }
  ERR_NOTEXTTOSEND	  = 412; { :No text to send                            }
  ERR_NOTOPLEVEL	  = 413; { <mask> :No toplevel domain specified        }
  ERR_WILDTOPLEVEL	  = 414; { <mask> :Wildcard in toplevel domain         }
  ERR_X_QUERYTOOLONG      = 416; { Undernet extension                          }

  ERR_UNKNOWNCOMMAND	  = 421; { <command> :Unknown command                  }
  ERR_NOMOTD	          = 422; { :MOTD File is missing                       }
  ERR_NOADMININFO	  = 423; { <server> :No administrative info available  }
  ERR_X_FILEERROR         = 424; { :File error doing <file op> on <file> (removed from RFC1459) }

  ERR_NONICKNAMEGIVEN	  = 431; { :No nickname given                          }
  ERR_ERRONEUSNICKNAME	  = 432; { <nick> :Erroneus nickname                   }
  ERR_NICKNAMEINUSE	  = 433; { <nick> :Nickname is already in use          }
  ERR_X_SERVICENAMEINUSE  = 434;
  ERR_X_SERVICECONFUSED   = 435;
  ERR_NICKCOLLISION	  = 436; { <nick> :Nickname collision KILL             }
  ERR_X_BANNICKCHANGE     = 437; { Undernet extension                          }
  ERR_X_NICKTOOFAST	  = 438; { Undernet extension                          }
  ERR_X_TARGETTOOFAST     = 439; { Undernet extension                          }

  ERR_USERNOTINCHANNEL	  = 441; { <nick> <channel> :They aren't on that channel }
  ERR_NOTONCHANNEL	  = 442; { <channel> :You're not on that channel       }
  ERR_USERONCHANNEL	  = 443; { <user> <channel> :is already on channel     }
  ERR_X_NOLOGIN	          = 444; { <user> :User not logged in                  }
  ERR_X_SUMMONDISABLED	  = 445; { :SUMMON has been disabled                   }
  ERR_X_USERSDISABLED	  = 446; { :USERS has been disabled                    }

  ERR_NOTREGISTERED	  = 451; { :You have not registered                    }
  ERR_IDCOLLISION         = 452; { IRCnet extension                            }
  ERR_NICKLOST            = 453; { IRCnet extension                            }

  ERR_NEEDMOREPARAMS   	  = 461; { <command> :Not enough parameters            }
  ERR_ALREADYREGISTRED    = 462; { :You may not reregister                     }
  ERR_NOPERMFORHOST	  = 463; { :Your host isn't among the privileged       }
  ERR_PASSWDMISMATCH	  = 464; { :Password incorrect                         }
  ERR_YOUREBANNEDCREEP	  = 465; { :You are banned from this server            }
  ERR_X_YOUWILLBEBANNED   = 466;
  ERR_KEYSET	          = 467; { <channel> :Channel key already set          }
  ERR_X_KEYSET	          = 467; { Undernet extension                          }
  ERR_X_INVALIDUSERNAME   = 468; { Undernet extension                          }

  ERR_CHANNELISFULL	  = 471; { <channel> :Cannot join channel (+l)         }
  ERR_UNKNOWNMODE	  = 472; { <char> :is unknown mode char to me          }
  ERR_INVITEONLYCHAN	  = 473; { <channel> :Cannot join channel (+i)         }
  ERR_BANNEDFROMCHAN	  = 474; { <channel> :Cannot join channel (+b)         }
  ERR_BADCHANNELKEY	  = 475; { <channel> :Cannot join channel (+k)         }
  ERR_X_BADCHANMASK	  = 476; { Undernet extension                          }
  ERR_X_MODELESS          = 477; { Extension to RFC1459                        }
  ERR_X_BANLISTFULL	  = 478; { Undernet extension                          }

  ERR_NOPRIVILEGES	  = 481; { :Permission Denied- You're not an IRC operator }
  ERR_CHANOPRIVSNEEDED	  = 482; { <channel> :You're not channel operator      }
  ERR_CANTKILLSERVER	  = 483; { :You cant kill a server!                    }
  ERR_X_ISCHANSERVICE     = 484; { Undernet extension                          }
  ERR_X_CHANTOORECENT     = 487; { IRCnet extension                            }
  ERR_X_TSLESSCHAN        = 488; { IRCnet extension                            }

  ERR_NOOPERHOST	  = 491; { :No O-lines for your host                   }
  ERR_X_NOSERVICEHOST     = 492;

  ERR_UMODEUNKNOWNFLAG	  = 501; { :Unknown MODE flag                          }
  ERR_USERSDONTMATCH	  = 502; { :Cant change mode for other users           }
  ERR_X_SILELISTFULL      = 511; { Undernet extension                          }
  ERR_X_NOSUCHGLINE	  = 512; { Undernet extension                          }
  ERR_X_BADPING           = 513; { Undernet extension                          }


Function IRCColourToTColor (const IRCCol : Integer) : TColor;
Function RandomNick : String;



implementation

uses
  SysUtils,
  cStrings,
  cMaths;



{ IRC Colours                                                                  }
{                                                                              }
{ 0  white                                                                     }
{ 1  black                                                                     }
{ 2  blue     (navy)                                                           }
{ 3  green                                                                     }
{ 4  red                                                                       }
{ 5  brown    (maroon)                                                         }
{ 6  purple                                                                    }
{ 7  orange   (olive)                                                          }
{ 8  yellow                                                                    }
{ 9  lt.green (lime)                                                           }
{ 10 teal     (a kinda green/blue cyan)                                        }
{ 11 lt.cyan  (cyan ?) (aqua)                                                  }
{ 12 lt.blue  (royal)                                                          }
{ 13 pink     (light purple) (fuchsia)                                         }
{ 14 grey                                                                      }
{ 15 lt.grey  (silver)                                                         }
Function IRCColourToTColor (const IRCCol : Integer) : TColor;
const ColorTable : Array [0..15] of TColor = (clWhite, clBlack, clBlue, clGreen,
      clRed, clMaroon, clPurple, clOlive, clYellow, clLime, clTeal,
      $00A0A000, $00A00000, clFuchsia, clGray, clSilver);
  Begin
    if IRCCol > 15 then
      Result := clBlack else
      Result := ColorTable [IRCCol];
  End;


{                                                                              }
{ RandomNick                                                                   }
{                                                                              }
Function RandomNick : String;
  Begin
    Result := LowerCase (RandomPseudoword (5 + Random (5)));
  End;



{                                                                              }
{   <message>  ::= [':' <prefix> <SPACE> ] <command> <params> <crlf>           }
{   <prefix>   ::= <servername> | <nick> [ '!' <user> ] [ '@' <host> ]         }
{   <command>  ::= <letter>+  | <number> <number> <number>                     }
{   <SPACE>    ::= ' '+                                                        }
{   <params>   ::= <SPACE> [ ':' <trailing> | <middle> <params> ]              }
{                                                                              }
{   <middle>   ::= <Any *non-empty* sequence of octets not including SPACE     }
{                  or NUL or CR or LF, the first of which may not be ':'>      }
{   <trailing> ::= <Any, possibly *empty*, sequence of octets not including    }
{                  NUL or CR or LF>                                            }
{                                                                              }
{   <crlf>     ::= CR LF                                                       }
{                                                                              }
{ DecomposeIRCMessage                                                          }
{   Returns Prefix, Command and Parameters.                                    }
{   Command is always returned in upper case.                                  }
{ DecomposeIRCPrefix                                                           }
{   Returns Nick (or server name), User and Host.                              }
{   Host is always returned in lower case.                                     }
{                                                                              }

{ DecomposeIRCMessage                                                          }
Procedure DecomposeIRCMessage (const Msg : String; var Prefix, Command : String;
          var Params : StringArray);
var F, G, L : Integer;
  Begin
    Prefix := '';
    Command := '';
    Params := nil;
    if Msg = '' then
      exit;

    if Msg [1] = ':' then
      begin
        F := PosNext (' ', Msg);
        if F = 0 then
          begin
            Prefix := CopyFrom (Msg, 2);
            exit;
          end;
        Prefix := CopyRange (Msg, 2, F - 1);
      end else
      F := 0;

    F := PosNext (cs_Everything - [' '], Msg, F);
    G := PosNext (' ', Msg, F);
    if G = 0 then
      begin
        Command := UpperCase (CopyRange (Msg, F, Length (Msg)));
        exit;
      end;
    Command := UpperCase (CopyRange (Msg, F, G - 1));

    Repeat
      G := PosNext (cs_Everything - [' '], Msg, G);
      if G = 0 then
        exit;

      if (G < Length (Msg)) and (Msg [G] = ':') then
        begin
          Append (Params, CopyFrom (Msg, G + 1));
          exit;
        end;

      F := PosNext (' ', Msg, G);
      if F = 0 then
        begin
          Append (Params, CopyFrom (Msg, G));
          exit;
        end;

      Append (Params, CopyRange (Msg, G, F - 1));
      G := F;
    Until False;
  End;

{ DecomposeIRCPrefix                                                           }
Procedure DecomposeIRCPrefix (const Prefix : String; var Name, User, Host : String);
var I, J : Integer;
  Begin
    I := PosNext (['!', '@'], Prefix);
    if I = 0 then
      begin
        Name := Prefix;
        User := '';
        Host := '';
      end else
      begin
        Name := CopyLeft (Prefix, I - 1);
        if Prefix [I] = '!' then
          begin
            J := PosNext ('@', Prefix, I);
            if J = 0 then
              begin
                User := CopyFrom (Prefix, I + 1);
                Host := '';
              end else
              begin
                User := CopyRange (Prefix, I + 1, J - 1);
                Host := LowerCase (CopyFrom (Prefix, J + 1));
              end;
          end else
          begin
            J := PosNext ('!', Prefix, I);
            if J = 0 then
              begin
                User := '';
                Host := LowerCase (CopyFrom (Prefix, I + 1));
              end else
              begin
                User := CopyFrom (Prefix, J + 1);
                Host := LowerCase (CopyRange (Prefix, I + 1, J - 1));
              end;
          end;
      end;
  End;

{                                                                              }
{ Decomposes and unquotes an CTCP message                                      }
{                                                                              }
Procedure DecomposeCTCPMessage (const Msg : String; var Command : String; var Params : StringArray);
var F : Integer;
    S : String;
  Begin
    S := UnescapeText (TrimQuotes (Msg), '\', ['\', 'a'], ['\', #1], False);
    S := UnescapeText (S, #16, [#16, '0', 'n', 'r'], [#16, #0, #10, #13], True);
    F := PosNext (' ', S);
    if F = 0 then
      begin
        Command := Copy (S, 2, Length (S) - 2);
        Params := nil;
      end else
      begin
        Command := Copy (S, 2, F - 2);
        Params := Split (CopyRange (S, F + 1, Length (S) - 1), [' ']);
      end;
  End;



{                                                                              }
{ TIRCClient                                                                   }
{   Uses single thread, non blocking.                                          }
{                                                                              }
Constructor TIRCClient.Create;
  Begin
    inherited Create;
    FChanNames := TStringDictionary.Create;
    FChanNames.CreateOnGet := True;
    FChanNames.CreateOnSet := True;
    FLoginTimeOut := 2 * 60 * 1000;
    FLoginRetries := 1;
    FLoginRetryWait := 15 * 1000;
    FAutoReLogin := True;
  End;

Destructor TIRCClient.Destroy;
  Begin
    FreeAndNil (FLoginTimer);
    FreeAndNil (FBuddyTimer);
    FreeAndNil (FClient);
    FreeAndNil (FChanNames);
    inherited Destroy;
  End;

Function TIRCClient.IsConnected : Boolean;
  Begin
    Result := Assigned (FClient) and FClient.Socket.Connected;
  End;

Procedure TIRCClient.OnLoginRetry (Sender : TObject);
  Begin
    FreeAndNil (FLoginTimer);
    if FActive and (FLoginAttemptNr < FLoginRetries) then
      AttemptLogin;
  End;

Procedure TIRCClient.OnReLogin (Sender : TObject);
  Begin
    FreeAndNil (FLoginTimer);
    if FActive then
      Login;
  End;

Procedure TIRCClient.SetLoggedOut;
var ReLogon : Boolean;
    Retry   : Boolean;
  Begin
    ReLogon := FLoggedIn and FAutoReLogin;
    Retry := FLoggingIn and (FLoginAttemptNr < FLoginRetries);

    FLoggedIn := False;
    FLoggingIn := False;
    FreeAndNil (FLoginTimer);
    FreeAndNil (FBuddyTimer);

    if Assigned (FOnLoggedOut) then
      FOnLoggedOut (self);

    FreeAndNil (FClient);

    if FActive and (ReLogon or Retry) then
      begin
        if FLoginRetryWait = 0 then
          begin
            if ReLogon then
              Login else
              AttemptLogin;
          end else
          begin
            FLoginTimer := TTimer.Create (nil);
            FLoginTimer.Interval := FLoginRetryWait;
            if ReLogon then
              FLoginTimer.OnTimer := OnReLogin else
              FLoginTimer.OnTimer := OnLoginRetry;
            FLoginTimer.Enabled := True;
          end;
      end;
  End;

Procedure TIRCClient.OnClientError (Sender: TObject; Socket: TCustomWinSocket; ErrorEvent: TErrorEvent; var ErrorCode: Integer);
  Begin
    if ErrorEvent in [eeGeneral, eeConnect] then
      begin
        SetLoggedOut;
        ErrorCode := 0;
      end;
  End;

Procedure TIRCClient.OnClientDisconnect (Sender: TObject; Socket: TCustomWinSocket);
  Begin
    SetLoggedOut;
  End;

Procedure TIRCClient.OnLoginTimeOut (Sender : TObject);
  Begin
    LogOut;
  End;

Procedure TIRCClient.OnClientConnect (Sender: TObject; Socket: TCustomWinSocket);
  Begin
    FClient.OnRead := OnClientRead;

    // Log in
    if FPassword = '' then
      FPassword := RandomNick;
    FClient.Socket.SendText ('PASS :' + FPassword + c_CRLF);

    if FNick = '' then
      FNick := RandomNick;
    SetNick (FNick);

    FClient.Socket.SendText ('USER ' + FUserName + ' host ' + FHost + ' :' + FRealName + c_CRLF);

    FLoginTimer := TTimer.Create (nil);
    FLoginTimer.Interval := FLoginTimeOut;
    FLoginTimer.OnTimer := OnLoginTimeOut;
    FLoginTimer.Enabled := True;
  End;

Procedure TIRCClient.SetLoggedIn;
  Begin
    FLoggingIn := False;
    FLoggedIn := True;

    if Assigned (FOnLoggedIn) then
      FOnLoggedIn (self);

    FBuddyTimer := TTimer.Create (nil);
    FBuddyTimer.Interval := 2 * 60 * 1000;
    FBuddyTimer.OnTimer := OnBuddyTimer;
    OnBuddyTimer (FBuddyTimer);
    FBuddyTimer.Enabled := True;
  End;

Procedure TIRCClient.AttemptLogin;
  Begin
    Inc (FLoginAttemptNr);
    FLoggingIn := True;

    if Assigned (FOnLoggingIn) then
      FOnLoggingIn (self);

    FClient := TClientSocket.Create (nil);
    FClient.Host := FHost;
    FClient.Port := FPort;
    FClient.ClientType := ctNonBlocking;
    FClient.OnConnect := OnClientConnect;
    FClient.OnDisconnect := OnClientDisconnect;
    FClient.OnError := OnClientError;
    FClient.Open;
  End;

Procedure TIRCClient.LogIn;
  Begin
    if FLoggingIn or FLoggedIn then
      exit;
    FLoginAttemptNr := 0;
    AttemptLogin;
  End;

Procedure TIRCClient.LogOut;
  Begin
    if IsConnected then
      try
        FClient.Socket.SendText ('QUIT' + c_CRLF);
        FClient.Socket.Close;
      except end;

    SetLoggedOut;
  End;

Procedure TIRCClient.SetActive (const Active : Boolean);
  Begin
    if Active = FActive then
      exit;

    FActive := Active;
    if Active then
      LogIn else
      LogOut;
  End;

Function IsOnlineIndicator (const MsgNumeric : Integer) : Boolean;
  Begin
    Case MsgNumeric of
      RPL_X_WELCOME, RPL_X_YOURHOST, RPL_X_CREATED, RPL_X_MYINFO,
      RPL_LUSERCLIENT, RPL_LUSEROP, RPL_LUSERUNKNOWN,
      RPL_LUSERCHANNELS, RPL_LUSERME,
      RPL_MOTDSTART, RPL_MOTD, RPL_ENDOFMOTD : Result := True;
      else Result := False;
    end;
  End;

Procedure TIRCClient.OnClientRead (Sender: TObject; Socket: TCustomWinSocket);
var Msg, Pre, Cmd : String;
    Name, User, Host : String;
    CmdN : Integer;
    Par : StringArray;
    LastPar : String;
    F, G : Integer;
    I : Integer;
    S : String;
    L : StringArray;
    R : Boolean;
  Begin
    FInBuf := FInBuf + Socket.ReceiveText;

    Repeat
      F := PosNext (ASCII_LF, FInBuf);
      if F > 0 then
        begin
          Msg := CopyLeft (FInBuf, F - 1);
          FInBuf := CopyFrom (FInBuf, F + 1);

          if Msg = '' then
            continue;
          if Msg [Length (Msg)] = #13 then
            SetLength (Msg, Length (Msg) - 1);
          Msg := TrimRight (Msg);
          if Msg = '' then
            continue;

          DecomposeIRCMessage (Msg, Pre, Cmd, Par);
          CmdN := StrToIntDef (Cmd, -1);
          DecomposeIRCPrefix (Pre, Name, User, Host);

          if FLoggingIn and IsOnlineIndicator (CmdN) then
            SetLoggedIn;

          if Assigned (FOnMessage) then
            FOnMessage (self, Name, User, Host, CmdN, Cmd, Par);

          if Cmd = 'PING' then
            FClient.Socket.SendText ('PONG' + Cond (Length (Par) > 0, ' :' + Par [0], '') + c_CRLF) else
          if Cmd = 'PRIVMSG' then
            begin
              if Assigned (FOnPrivMsg) and (Length (Par) >= 2) then
                FOnPrivMsg (self, Par [0], Name, Par [1]);
            end else
          Case CmdN of
            RPL_ISON :
              begin
                For G := 1 to Length (Par) - 1 do
                  Append (FBuddiesTmp, Par [G]);
                Dec (FIsOnlineCount);
                if FIsOnlineCount = 0 then
                  begin
                    R := not IsEqual (FBuddiesOnline, FBuddiesTmp);
                    FBuddiesOnline := FBuddiesTmp;
                    if R then
                      if Assigned (FOnBuddiesOnlineChange) then
                        FOnBuddiesOnlineChange (self);
                  end;
              end;
            RPL_NAMREPLY :
              begin
                I := Cond (Par [1] = '=', 1, 0);
                S := FChanNames [Par [1 + I]];
                FChanNames [Par [1 + I]] := S + Cond (S <> '', ' ', '') + Trim (Par [2 + I]);
              end;
            RPL_ENDOFNAMES :
              if Assigned (FOnChannelJoin) then
                begin
                  L := Split (FChanNames [Par [1]], [' ']);
                  Sort (L);
                  FOnChannelJoin (self, Par [1], L);
                end;
            ERR_NICKNAMEINUSE :
              begin
                S := RandomNick;
                if Assigned (FOnNickInUse) then
                  FOnNickInUse (self, S);
                SetNick (S);
              end;
            RPL_MOTDSTART :
              begin
                FMOTD := '';
                FGotMOTD := False;
              end;
            RPL_MOTD :
              if Length (Par) > 0 then
                FMOTD := FMOTD + Par [Length (Par) - 1] + #13#10;
            RPL_ENDOFMOTD :
              FGotMOTD := True;
          end;
        end;
    Until F = 0;
  End;

Procedure TIRCClient.Join (const Channel : String);
  Begin
    if Assigned (FClient) and FLoggedIn then
      begin
        FChanNames.Delete (Channel);
        FClient.Socket.SendText ('JOIN :' + Channel + c_CRLF);
      end;
  End;

Procedure TIRCClient.Part (const Channel : String);
  Begin
    if Assigned (FClient) and FLoggedIn then
      FClient.Socket.SendText ('PART :' + Channel + c_CRLF);
  End;

Procedure TIRCClient.PrivMsg (const Destination, Msg : String);
  Begin
    if Assigned (FClient) then
      FClient.Socket.SendText ('PRIVMSG ' + Destination + ' :' + Msg + c_CRLF);
  End;

Procedure TIRCClient.SetNick (const Nick : String);
  Begin
    if Assigned (FClient) then
      FClient.Socket.SendText ('NICK :' + Nick + c_CRLF);
    FNick := Nick;
  End;

Procedure TIRCClient.CheckBuddyList;

  Procedure SendISON (var S : String);
    Begin
      if S <> '' then
        begin
          Inc (FIsOnlineCount);
          FClient.Socket.SendText ('ISON ' + S + c_CRLF);
          S := '';
        end;
    End;

var F, L : Integer;
    S : String;

  Begin
    SetLength (FBuddiesTmp, 0);
    FIsOnlineCount := 0;
    S := '';
    L := Length (FBuddyList);
    For F := 0 to L - 1 do
      begin
        S := S + Cond (S <> '', ' ', '') + FBuddyList [F];
        if (F < L - 1) and (Length (S) + Length (FBuddyList [F + 1]) + 8 > 128) then
          SendISON (S);
      end;
    SendISON (S);
  End;

Procedure TIRCClient.OnBuddyTimer (Sender : TObject);
  Begin
    CheckBuddyList;
  End;

Procedure TIRCClient.SetBuddyList (const BuddyList : StringArray);
  Begin
    FBuddyList := BuddyList;
  End;

Procedure TIRCClient.AddBuddy (const Server, Nick : String);
  Begin
    Append (FBuddyList, Nick + '@' + Server);
  End;

Procedure TIRCClient.RemoveBuddy (const Server, Nick : String);
var I : Integer;
  Begin
    I := PosNext (Nick + '@' + Server, FBuddyList);
    if I >= 0 then
      Remove (FBuddyList, I, 1);
  End;



{                                                                              }
{ TMultiIRCClient                                                              }
{                                                                              }
Destructor TMultiIRCClient.Destroy;
  Begin
    RemoveAllClients;
    RemoveAllJoins;
    inherited Destroy;
  End;

Procedure TMultiIRCClient.RemoveAllClients;
var I : Integer;
  Begin
    For I := Length (FClients) - 1 downto 0 do
      Remove (ObjectArray (FClients), I, 1, True);
  End;

Procedure TMultiIRCClient.RemoveAllJoins;
var I : Integer;
  Begin
    For I := Length (FJoins) - 1 downto 0 do
      Remove (ObjectArray (FJoins), I, 1, True);
  End;

Function TMultiIRCClient.GetPersistentClient (const Server : String) : Boolean;
  Begin
    Result := PosNext (Server, FPersistent) >= 0;
  End;

Procedure TMultiIRCClient.SetPersistentClient (const Server : String; const PersistentClient : Boolean);
var I : Integer;
  Begin
    I := PosNext (Server, FPersistent);
    if PersistentClient then
      begin
        if I = -1 then
          begin
            Append (FPersistent, Server);
            CheckClientConnections;
          end;
      end else
      begin
        if I >= 0 then
          begin
            Remove (FPersistent, I);
            CheckClientConnections;
          end;
      end;
  End;

Function TMultiIRCClient.GetServers : StringArray;
var I : Integer;
  Begin
    // Persistent servers
    Result := Copy (FPersistent);

    // Servers for joined channels
    For I := 0 to Length (FJoins) - 1 do
      if PosNext (FJoins [I].Server, Result) = -1 then
        Append (Result, FJoins [I].Server);
  End;

Function TMultiIRCClient.GetClient (const Server : String) : TIRCClient;
var I : Integer;
  Begin
    For I := 0 to Length (FClients) - 1 do
      With FClients [I] do
        if FHost + ':' + IntToStr (FPort) = Server then
          begin
            Result := FClients [I];
            exit;
          end;
    Result := nil;
  End;

Procedure TMultiIRCClient.CheckClientConnections;
var SL : StringArray;
    TL : StringArray;
    I  : Integer;
    Cl : TIRCClient;
  Begin
    if not FActive then
      begin
        For I := Length (FClients) - 1 downto 0 do
          FClients [I].LogOut;
        RemoveAllClients;
        exit;
      end;

    SL := GetServers;

    // Remove clients
    I := Length (FClients) - 1;
    While I >= 0 do
      With FClients [I] do
        begin
          if PosNext (FHost + ':' + IntToStr (FPort), SL) = -1 then
            begin
              LogOut;
              Remove (ObjectArray (FClient), I, 1, True);
            end;
          Dec (I);
        end;

    // Add clients
    For I := 0 to Length (SL) - 1 do
      begin
        Cl := GetClient (SL [I]);
        if not Assigned (Cl) then
          if Assigned (FOnNewClient) then
            begin
              TL := Split (SL [I], ':');
              FOnNewClient (self, Cl, TL [0], StrToInt (TL [1]));
              if Assigned (Cl) then
                begin
                  Cl.OnLoggedIn := OnLoggedIn;
                  Cl.Active := True;
                  Append (ObjectArray (FClients), Cl);
                end;
            end;
      end;
  End;

Procedure TMultiIRCClient.OnLoggedIn (const Sender : TIRCClient);
var I : Integer;
    S : String;
  Begin
    // Join channels
    S := Sender.Host + ':' + IntToStr (Sender.Port);
    For I := 0 to Length (FJoins) - 1 do
      if FJoins [I].Server = S then
        Sender.Join (FJoins [I].Channel);
  End;

Function TMultiIRCClient.IndexOfChannel (const Server : String; const Channel : String) : Integer;
var I : Integer;
  Begin
    For I := 0 to Length (FJoins) - 1 do
      if (FJoins [I].Server = Server) and (FJoins [I].Channel = Channel) then
        begin
          Result := I;
          exit;
        end;
    Result := -1;
  End;

Procedure TMultiIRCClient.PartChannel (const Server : String; const Channel : String);
var I  : Integer;
    Cl : TIRCClient;
  Begin
    I := IndexOfChannel (Server, Channel);
    if I >= 0 then
      begin
        Cl := GetClient (Server);
        if Assigned (Cl) then
          Cl.Part (Channel);
        Remove (ObjectArray (FJoins), I, 1, True);
        CheckClientConnections;
      end;
  End;

Procedure TMultiIRCClient.JoinChannel (const Server : String; const Channel : String);
var I : Integer;
    J : TJoinChannel;
  Begin
    I := IndexOfChannel (Server, Channel);
    if I = -1 then
      begin
        J := TJoinChannel.Create;
        J.Server := Server;
        J.Channel := Channel;
        Append (ObjectArray (FJoins), J);
        CheckClientConnections;
      end;
  End;

Procedure TMultiIRCClient.RemoveClient (const Server : String);
var I : Integer;
  Begin
    SetPersistentClient (Server, False);
    I := Length (FJoins) - 1;
    While I >= 0 do
      begin
        if FJoins [I].Server = Server then
          PartChannel (Server, FJoins [I].Channel);
        Dec (I);
      end;
  End;

Procedure TMultiIRCClient.SetActive (const Active : Boolean);
  Begin
    if Active = FActive then
      exit;

    FActive := Active;
    CheckClientConnections;
  End;


end.




{                                                                              }
{ Index  Name  RGB                                                             }
{    0  Black    000,000,000                                                   }
{    1  Blue     000,000,128                                                   }
{    2  Green    000,128,000                                                   }
{    3  Cyan     000,128,128                                                   }
{    4  Red      128,000,000                                                   }
{    5  Purple   128,000,128                                                   }
{    6  Brown    128,128,000                                                   }
{    7  Lt Gray  204,204,204                                                   }
{    8  Gray     128,128,128                                                   }
{    9  Lt Blue  000,000,255                                                   }
{    A  Lt Green 000,255,000                                                   }
{    B  Lt Cyan  000,255,255                                                   }
{    C  Lt Red   255,000,000                                                   }
{    D  Pink     255,000,255                                                   }
{    E  Yellow   255,255,000                                                   }
{    F  White    255,255,255                                                   }
{                                                                              }

