/* Grammar for TxQuery dataset (Delphi 3,4,5), (c) 2000 Alfonso Moreno
   NOTES :
   DON'T FORGET TO MOVE THE GENERATED CONSTANTS TO THE PLACE INDICATED
*/

%{
{**********************************************}
{   Parser for TxQuery component               }
{   Copyright (c) 2000 by Alfonso Moreno       }
{**********************************************}
unit xqYacc;

{$I XQ_FLAG.INC}
{$R xqyacc.res}
interface

uses
   SysUtils, Classes, Dialogs, Windows, LexLibQ, YaccLibQ, xqBase, xquery,
   DB, xqmiscel
{$IFDEF LEVEL3}
   , DBTables
{$ENDIF}
   ;

//
// The generated constants must be placed here
// HERE !!!!
//

type
  TxqParser = class( TCustomParser )
  private
      fAnalizer        : TSqlAnalizer;
      fCurrAnalizer    : TSqlAnalizer;    { only a pointer. never created }
      fInPredicateList : TStringList;
      fIsNotInList     : Boolean;
      fForLength       : String;
      fEscapeChar      : String;
      fJoinKind        : TJoinKind;
      fTrimPosition    : Integer;
      fExtractField    : Integer;
      fCastType        : Integer;
      fCastLen         : Integer;
      fCurrentJoin     : Integer;
      fIsDistinctAggr  : Boolean;
      fAsAlias         : String;
      fJoinRelOperator : TRelationalOperator;
      fNumInserts      : Integer;   { used in INSERTO INTO...}
      fAggregateList   : TAggregateList;
      fColSubqueryList : TList;     { the list of subqueries for the current select }

      { for CREATE TABLE }
      FNumTables, FFieldType, FScale, FPrecision, FSize, FBlobType: Integer;
      procedure SetFieldParams(AFieldType, AScale, APrecision, ASize, ABlobType: Integer);
      procedure SetTableName(const TableName: String);
      procedure AddCreateField(const FieldName: String);
      procedure AddPrimaryKeyField(const FieldName: String);
      function CurrentCreateTable: TCreateTableItem;
      function GetCurrentAnalizer: TSqlAnalizer;
      function CurrentInsertItem : TInsertItem;
      function CreateInListExpression( const Expr : String ): String;
      function AddSubqueryInSelect : String;
      procedure CreateNewSubquery;
      function GetRootAnalizer: TSQLAnalizer;
  public

      constructor Create(Analizer: TSqlAnalizer);
      destructor Destroy; override;

      function yyparse : integer; override;
      procedure yyerror(const msg : string); override;

      { specials }
      procedure AddColumn(const AColText : String; IsTransformColumn: Boolean);
      function AddAggregate( pAggregate  : TAggregateKind;
		    const pAggregateStr : String ) : String;
      procedure AddGroupBy(const ColName: String);
      procedure AddOrderBy(const ColName: String; Descending: Boolean);
      procedure AddTable(const TableName, Alias: String);
      procedure AddJoin(const LeftExpr, RightExpr, AndExpr: String);
      procedure AddJoinCandidate(const LeftExpr, RightExpr: String);
      procedure AddHavingColumn( const ColText : String );
      procedure AddUpdateColumn(const ColumnName, ColumnExpr: String);
      procedure AddWhereOptimize(const AFieldName, ARangeStart,
				 ARangeEnd : String; ARelOperator : TRelationalOperator);
      procedure AddParam(const ParamName: String);

      property Analizer: TSqlAnalizer read fAnalizer write fAnalizer;
      property AsAlias: String read fAsAlias write fAsAlias;
      property CurrentAnalizer: TSqlAnalizer read GetCurrentAnalizer;
  end;

implementation

uses
   xqLex, xqConsts;

(*----------------------------------------------------------------------------*)
procedure TxqParser.yyerror(const msg : string);
begin
   yyerrorMsg := msg;
   { yyerrorMsg := IntToStr(yyLexer.yylineno) +  ': ' + msg + ' at or before '+ yyLexer.yytext + '. ' +
	 Format('Line %d, Column %d',[yyLexer.yylineno,yyLexer.yycolno]);
   if Analizer.xQuery.ShowErrorMessage then
      ShowMessage( yyerrorMsg ); }
end;

constructor TxqParser.Create(Analizer: TSqlAnalizer);
begin
   inherited Create;
   fAnalizer:= Analizer;
   fInPredicateList := TStringList.Create;
end;

destructor TxqParser.Destroy;
begin
   fInPredicateList.Free;
   if Assigned(fAggregateList) then
      fAggregateList.Free;
   inherited Destroy;
end;

procedure TxqParser.CreateNewSubquery;
var
  TmpAnalizer : TSqlAnalizer;
begin
  TmpAnalizer := GetCurrentAnalizer;
  fCurrAnalizer := TSqlAnalizer.Create( TmpAnalizer, fAnalizer.xQuery );
  TmpAnalizer.SubqueryList.Add( fCurrAnalizer );
end;

function TxqParser.GetRootAnalizer: TSQLAnalizer;
begin
  Result := GetCurrentAnalizer;
  while Result.ParentAnalizer <> nil do Result := Result.ParentAnalizer;
end;

function TxqParser.GetCurrentAnalizer: TSqlAnalizer;
begin
   if fCurrAnalizer= nil then fCurrAnalizer := Self.fAnalizer;
   Result := fCurrAnalizer;
end;

procedure TxqParser.AddColumn(const AColText : String; IsTransformColumn: Boolean);
var
   Column : TColumnItem;
   TmpAnalizer: TSqlAnalizer;

   function StripFields(const s: String): String;
   var
      p, i, j: Integer;
      Found: Boolean;
   begin
      Result:= s;
      p:= Pos('\f"', Result);
      while p > 0 do
      begin
         j:= p + 3;
         i:= j;
         Found:= True;
         while j <= Length(Result) do
         begin
            if Result[j] = '"' then
            begin
               Found:= True;
               Break;
            end;
            Inc(j);
         end;
         if Not Found then Exit;    { fatal error }
         if j <= Length(Result) then
            Result:= Copy(Result, 1, p - 1) + Copy(Result, i, j - i) +
               Copy(Result, j + 1, Length(Result));

         p:= Pos('\f"', Result);
      end;
   end;

begin
   if IsTransformColumn then
      Column := CurrentAnalizer.TransformColumnList.Add
   else
      Column := CurrentAnalizer.ColumnList.Add;
   with Column do
   begin
      ColumnExpr:= AColText;
      { this mean that aggregate columns are embedded in ColumnExpr}
      if Assigned(Self.fAggregateList) then
      begin
        Column.AggregateList.Free;    { free previous aggregate list}
        Column.AggregateList := Self.fAggregateList; { assign the current list}
        Self.fAggregateList:= nil;    { define as nil the current }
      end;
      if Assigned(Self.fColSubqueryList) then
      begin
        TmpAnalizer := GetRootAnalizer;
        if TmpAnalizer = CurrentAnalizer then begin
           Column.SubqueryList.Free;
           Column.SubqueryList := Self.fColSubqueryList;
           Self.fColSubqueryList:= nil;
           if Length(Self.fAsAlias)= 0 then
              Self.fAsAlias := 'Subquery';
        end;
      end;
      CastType  := Self.fCastType;
      CastLen   := IMax(1, Self.fCastLen);      // only used for strings
      if Length(self.fAsAlias) > 0 then
      begin
        AsAlias:= self.fAsAlias;
        IsAsExplicit:= True;
      end else
      begin
	      IsAsExplicit := False;
	      AsAlias := StripFields(AColText);
        if AggregateList.Count > 0  then
          case AggregateList[0].Aggregate of
	           akSUM: AsAlias := SAggrSUM + StripFields(AggregateList[0].AggregateStr);
	           akAVG: AsAlias := SAggrAVG + StripFields(AggregateList[0].AggregateStr);
	           akSTDEV: AsAlias := SAggrSTDEV + StripFields(AggregateList[0].AggregateStr);  { LAS: 05-30-00 }
	           akMIN: AsAlias := SAggrMIN + StripFields(AggregateList[0].AggregateStr);
	           akMAX: AsAlias := SAggrMAX + StripFields(AggregateList[0].AggregateStr);
	           akCOUNT: AsAlias := SAggrCOUNT;
	        end;
      end;
   end;

   fAsAlias  := '';
   fCastType := 0;
   fCastLen  := 0;
end;

{ This function will return an aggregate encoded with something like :
  (Aggregate 1) }
function TxqParser.AddAggregate( pAggregate  : TAggregateKind;
   const pAggregateStr : String ) : String;
begin
  if fAggregateList = nil then
     fAggregateList := TAggregateList.Create;
  with fAggregateList.Add do
  begin
     AggregateStr := pAggregateStr;
     Aggregate    := pAggregate;
     IsDistinctAg := Self.fIsDistinctAggr;
  end;
  Result := Format('{Aggregate %d}', [fAggregateList.Count - 1]);

  Self.fIsDistinctAggr := False;
end;

{ This function will return a subquery encoded with something like :
  (Subquery 1) }
function TxqParser.AddSubqueryInSelect : String;
var
  MainAnalizer, Analizer: TSqlAnalizer;
begin
  if fColSubqueryList = nil then
     fColSubqueryList := TList.Create;
  { Now add the subquery.
    Important: No nested subqueries are allowed in this release. Example
    SELECT custno, (SELECT custno FROM customer WHERE
      custno >= ALL(SELECT custno FROM customer WHERE custno BETWEEN 1000 AND 2000)) FROM customer; )
    But this is allowed:
     SELECT custno, (SELECT SUM(amountpaid) FROM customer WHERE custno BETWEEN 1000 AND 2000) /
        (SELECT Count(*) FROM customer WHERE custno BETWEEN 2000 AND 3000) FROM customer;
  }
  { do to the following trick, nested subqueries in the SELECT columns are not allowed }
  MainAnalizer := GetRootAnalizer;
  Analizer := TSqlAnalizer( MainAnalizer.SubqueryList[0] );
  MainAnalizer.SubQueryList.Clear;
  fColSubqueryList.Add( Analizer );
  Result := Format( '{Subquery %d}',[fColSubqueryList.Count - 1] );
end;

procedure TxqParser.AddGroupBy(const ColName: String);
var
   GroupBy: TOrderByItem;
   Index, Code: integer;
begin
   Val(ColName, Index, Code);
   GroupBy := CurrentAnalizer.GroupByList.Add;
   if Code = 0 then
   begin
      GroupBy.Alias := '';
      GroupBy.ColIndex := Index - 1;
   end else
   begin
      GroupBy.Alias := ColName;
      GroupBy.ColIndex:= -1;   { means: not defined by number }
   end;
end;

procedure TxqParser.AddOrderBy(const ColName: String; Descending: Boolean);
var
  OrderBy     : TOrderByItem;
  Index, Code : integer;
begin
  Val(ColName, Index, Code);
  OrderBy:= CurrentAnalizer.OrderByList.Add;
  if Code = 0 then
  begin
     OrderBy.Alias := '';
     OrderBy.ColIndex := Index - 1;
  end else
  begin
     OrderBy.Alias := ColName;
     { means: not defined by number and must be solved in checkintegrity }
     OrderBy.ColIndex:= -1;
  end;
  OrderBy.Desc:= Descending;
end;

procedure TxqParser.AddTable(const TableName, Alias: String);
var
   Table: TTableItem;
begin
   Table := CurrentAnalizer.TableList.Add;
   Table.TableName := TableName;
   if Length(Alias) > 0 then
      Table.Alias := Alias
   else
      Table.Alias := TableName;
end;

procedure TxqParser.AddJoin(const LeftExpr, RightExpr, AndExpr: String);
var
   JoinOnItem          : TJoinOnItem;
   LeftJoinOnExprItem  : TJoinOnExprItem;
   RightJoinOnExprItem : TJoinOnExprItem;
   AndJoinOnExprItem   : TJoinOnExprItem;
begin

   if CurrentAnalizer.JoinList.Count <= fCurrentJoin then
   begin
      JoinOnItem          := CurrentAnalizer.JoinList.Add;
      LeftJoinOnExprItem  := JoinOnItem.LeftJoinOn.Add;
      RightJoinOnExprItem := JoinOnItem.RightJoinOn.Add;
      AndJoinOnExprItem   := JoinOnItem.AndJoinOn.Add;
   end else
   begin
      JoinOnItem          := CurrentAnalizer.JoinList[fCurrentJoin];
      LeftJoinOnExprItem  := JoinOnItem.LeftJoinOn.Add;
      RightJoinOnExprItem := JoinOnItem.RightJoinOn.Add;
      AndJoinOnExprItem   := JoinOnItem.AndJoinOn.Add;
   end;

   with JoinOnItem do
   begin
      JoinKind:= Self.fJoinKind;
      RelOperator:= fJoinRelOperator;
   end;
   LeftJoinOnExprItem.Expression  := LeftExpr;
   RightJoinOnExprItem.Expression := RightExpr;
   AndJoinOnExprItem.Expression   := AndExpr;

   Self.fJoinKind := jkInnerJoin;
   Self.fJoinRelOperator := ropBETWEEN;
end;

procedure TxqParser.AddJoinCandidate(const LeftExpr, RightExpr: String);
begin
  CurrentAnalizer.LJoinCandidateList.Add( LeftExpr );
  CurrentAnalizer.RJoinCandidateList.Add( RightExpr );
end;

procedure TxqParser.AddHavingColumn( const ColText : String );
var
   Column: TColumnItem;
begin
   Column := CurrentAnalizer.ColumnList.Add;
   with Column do
   begin
      ColumnExpr := ColText;
      if Assigned(Self.fAggregateList) then
      begin
        AggregateList.Free;    { free previous aggregate list}
        AggregateList := Self.fAggregateList; { assign the current list}
        Self.fAggregateList:= nil;    { define as nil the current }
      end;
      { mark also as a temporary column that will be deleted after result set is
        generated }
      IsTemporaryCol := True;
   end;
end;

procedure TxqParser.AddUpdateColumn(const ColumnName, ColumnExpr: String);
var
   UpdateItem: TUpdateItem;
begin
   UpdateItem := CurrentAnalizer.UpdateColumnList.Add;
   with UpdateItem do
   begin
      ColName := ColumnName;
      ColExpr := ColumnExpr;
   end;

end;

procedure TxqParser.AddWhereOptimize( const AFieldName, ARangeStart,
   ARangeEnd: String; ARelOperator: TRelationalOperator );
var
   WhereOptimize : TWhereOptimizeItem;
   N             : Integer;

   function DeleteStringDelim(const s: String): String;
   begin
      Result := s;
      if Length(Result) > 1 then
      begin
	       if (Result[1] in xqbase.SQuote) and (Result[Length(Result)] in xqbase.SQuote) then
	       Result:= Copy(Result, 2, Length(Result) - 2);
      end;
   end;

begin
   if Not (yyLexer as TxqLexer).IsWhereActive then Exit;
   N := CurrentAnalizer.WhereOptimizeList.Count - 1;
   if ((ARelOperator in [ropLE, ropLT]) or (ARelOperator in [ropGE, ropGT]) ) and (N > 0) then
   begin

      { I will check if the previous command was something like (CustNo >= 1000)
     	 and if so, and this is something like (CustNo <= 3000) then
    	 I will add to the previous optimize and will generate a ropBETWEEN range   }

      WhereOptimize:= CurrentAnalizer.WhereOptimizeList[N - 1];
      if ( ((ARelOperator in [ropLE, ropLT]) and (WhereOptimize.RelOperator in [ropGE, ropGT])) or
      	 ((ARelOperator in [ropGE, ropGT]) and (WhereOptimize.RelOperator in [ropLE, ropLT])) ) and
      	 (AnsiCompareText(WhereOptimize.FieldNames, AFieldName) = 0) then
      begin
      	 WhereOptimize.RangeEnd    := DeleteStringDelim( ARangeEnd );
      	 WhereOptimize.RelOperator := ropBETWEEN;
      	 Exit;
      end;
   end;

   WhereOptimize := CurrentAnalizer.WhereOptimizeList.Add;
   with WhereOptimize do
   begin
      FieldNames  := AFieldName;
      RangeStart  := DeleteStringDelim(ARangeStart);
      RangeEnd    := DeleteStringDelim(ARangeEnd);
      RelOperator := ARelOperator;
      CanOptimize := False;
   end;

end;

procedure TxqParser.AddParam(const ParamName: String);
begin
  CurrentAnalizer.Params.CreateParam(ftString, ParamName, ptUnknown);
end;

{ CREATE TABLE }
function TxqParser.CurrentCreateTable: TCreateTableItem;
begin
  if FNumTables > CurrentAnalizer.CreateTableList.Count - 1 then
     Result := CurrentAnalizer.CreateTableList.Add
  else
     Result:= CurrentAnalizer.CreateTableList[ FNumTables ];
end;

procedure TxqParser.SetTableName(const TableName: String);
begin
   CurrentCreateTable.TableName := TableName;
end;

procedure TxqParser.SetFieldParams(AFieldType, AScale, APrecision, ASize, ABlobType: Integer);
begin
  FFieldType := AFieldType;
  FScale     := AScale;
  FPrecision := APrecision;
  FSize      := ASize;
  FBlobType  := ABlobType;
end;

procedure TxqParser.AddCreateField(const FieldName: String);
var
  I: Integer;
begin
  { check if field exists }
  with CurrentCreateTable do
     for I := 0 to FieldCount - 1 do
	if AnsiCompareText(Fields[I].FieldName, FieldName) = 0 then
	begin
	   yyerror(SDuplicateFieldName);
	   yyabort;
	   Exit;
	end;
  if (FFieldType = RW_BLOB) and not (FBlobType in [1..5]) then
  begin
    yyerror(SBlobFieldWrongType);
    yyabort;
  end;
  CurrentCreateTable.Fields.AddField(FieldName,
				     FFieldType,
				     FScale,
				     FPrecision,
				     FSize,
				     FBlobType);
end;

procedure TxqParser.AddPrimaryKeyField(const FieldName: String);
var
  I : Integer;
  Found : Boolean;
begin
  { check that the field exists in the list of field names }
  Found := False;
  with CurrentCreateTable do
     for I := 0 to FieldCount - 1 do
	if AnsiCompareText(Fields[I].FieldName, FieldName) = 0 then
	begin
	   Found:= True;
	   Break;
	end;
  if Not Found then
  begin
     yyerror(SFieldNameNotFound);
     yyabort;
  end;
  CurrentCreateTable.PrimaryKey.Add( FieldName );
end;

function TxqParser.CurrentInsertItem: TInsertItem;
begin
   if fNumInserts >= CurrentAnalizer.InsertList.Count then
      Result := CurrentAnalizer.InsertList.Add
   else
      Result := CurrentAnalizer.InsertList[fNumInserts];
end;

function TxqParser.CreateInListExpression( const Expr : String ) : String;
var
   i : Integer;
begin
   { This subroutine helps to change the syntax:
     custno IN (1000, 2000, 3000)
     to this :
     (custno = 1000) OR (custno = 2000) OR (custno = 3000) }
   Result := '';
   for i := 0 to fInPredicateList.Count - 1 do begin
      Result := Result + Format('(%s = %s)', [Expr, fInPredicateList[i]]);
      if i < fInPredicateList.Count - 1 then
	 Result := Result + ' OR ';
   end;
   if fIsNotInList then
      Result := 'NOT ( ' + Result + ' )'
   else
      Result := '( ' + Result + ' )';
   fInPredicateList.Clear;
end;

%}

%start sql

%token _IDENTIFIER
%token _UINTEGER
%token _SINTEGER
%token _NUMERIC
%token _STRING

%token _COMA
%token _LPAREN
%token _RPAREN
%token _PERIOD
%token _SEMICOLON
%token _COLON

%left  _EQ _NEQ _GT _LT _GE _LE RW_BETWEEN RW_IN RW_LIKE
%left  _PLUS _SUB RW_OR
%left  _DIV _MULT RW_MOD RW_IDIV RW_SHL RW_SHR RW_AND
%right _NEG       /* Negation--unary minus */
%right _EXP       /* exponentiation */
%left  RW_NOT
%token _ILLEGAL


/* other reserved words and tokens */
%token  RW_TRUE
	RW_FALSE
	RW_SELECT
	RW_DISTINCT
	RW_FROM
	RW_WHERE
	RW_ORDER
	RW_BY
	RW_ASC
	RW_DESC
	RW_AS
	RW_INNER
	RW_OUTER
	RW_FULL
	RW_JOIN
	RW_ON
	RW_GROUP
	RW_HAVING
	RW_ANY
	RW_ALL
	RW_SUM
	RW_AVG
	RW_COUNT
	RW_MIN
	RW_MAX
	RW_STDEV
	RW_LEFT
	RW_RIGHT
	RW_LEADING
	RW_TRAILING
	RW_BOTH
	RW_TRIM
	RW_EXTRACT
	RW_YEAR
	RW_MONTH
	RW_DAY
	RW_HOUR
	RW_MINUTE
	RW_SECOND
	RW_FOR
	RW_SUBSTRING
	RW_DELETE
	RW_UPDATE
	RW_INSERT
	RW_INTO
	RW_VALUES
	RW_SET
	RW_CAST
	RW_CHAR
	RW_INTEGER
	RW_BOOLEAN
	RW_DATE
	RW_TIME
	RW_DATETIME
	RW_FLOAT
	RW_ESCAPE
	RW_NOW
	_COMMENT
	_BLANK
	_TAB
	_NEWLINE
	RW_CREATE
	RW_TABLE
	RW_SMALLINT
	RW_MONEY
	RW_AUTOINC
	RW_PRIMARY
	RW_KEY
	RW_BLOB
	RW_INDEX
	RW_UNIQUE
	RW_DROP
	RW_TRANSFORM
	RW_PIVOT
	RW_MERGE
	RW_WITH
	RW_IS
	RW_NULL


%type <string>

%%

sql : select_statement        { fAnalizer.Statement := ssSelect;      }
    | transform_statement     { fAnalizer.Statement := ssSelect;      }
    | merge_statement         { fAnalizer.Statement := ssMerge;       }
    | update_statement        { fAnalizer.Statement := ssUpdate;      }
    | delete_statement        { fAnalizer.Statement := ssDelete;      }
    | insert_statement        { fAnalizer.Statement := ssInsert;      }
    | createtable_statement   { fAnalizer.Statement := ssCreateTable; }
    | createindex_statement   { fAnalizer.Statement := ssCreateIndex; }
    | droptable_statement     { fAnalizer.Statement := ssDropTable;   }
    | dropindex_statement     { fAnalizer.Statement := ssDropIndex;   }
    ;

/*  SELECT statement */
select_statement : select_clause from_clause where_clause groupby_clause orderby_clause end_statement
		 ;

/* SELECT clause */
select_clause : select_case _MULT              {CurrentAnalizer.DoSelectAll := True;}
              | select_case list_expr_field
              ;

select_case : RW_SELECT
      	    | RW_SELECT RW_DISTINCT {CurrentAnalizer.IsDistinct:= True;}
      	    ;

list_expr_field : expr_field
                | list_expr_field _COMA expr_field
                ;

expr_field : select_expr as_identifier             { AddColumn( $<string>1, False ); }
           | _IDENTIFIER _PERIOD _MULT             { CurrentAnalizer.TableAllFields.Add( $<string>1 ); }
           | RW_CAST _LPAREN select_expr RW_AS data_type _RPAREN as_identifier { AddColumn( $<string>3, False ); }
           ;

as_identifier : /* empty */
              | as_optional _IDENTIFIER  {fAsAlias:= $<string>2;}
              ;

as_optional : /* empty */
            | RW_AS
            ;

select_expr  : define_field
             | _STRING
             | _SINTEGER
             | _UINTEGER
             | _NUMERIC
             | RW_TRUE
             | RW_FALSE
             | RW_NOW
             | select_func
             | trim_function
             | extract_function
             | substring_function
             | subquery                     { $<string>$ := AddSubQueryInSelect; }

             | select_expr _PLUS select_expr {$<string>$ := $<string>1 + ' + '  + $<string>3; }
             | select_expr _SUB select_expr  {$<string>$ := $<string>1 + ' - '  + $<string>3; }
             | select_expr _MULT select_expr {$<string>$ := $<string>1 + ' * '  + $<string>3; }
             | select_expr _DIV select_expr  {$<string>$ := $<string>1 + ' / '  + $<string>3; }
             | select_expr _EXP select_expr  {$<string>$ := $<string>1 + ' ^ '  + $<string>3; }
             | select_expr RW_MOD select_expr  {$<string>$ := $<string>1 + ' MOD '  + $<string>3; }
             | select_expr RW_IDIV select_expr  {$<string>$ := $<string>1 + ' DIV '  + $<string>3; }
             | select_expr RW_SHL select_expr  {$<string>$ := $<string>1 + ' SHL '  + $<string>3; }
             | select_expr RW_SHR select_expr  {$<string>$ := $<string>1 + ' SHR '  + $<string>3; }

             | select_expr _GT select_expr   {$<string>$ := $<string>1 + ' > '  + $<string>3; }
             | select_expr _LT select_expr   {$<string>$ := $<string>1 + ' < '  + $<string>3; }
             | select_expr _NEQ select_expr  {$<string>$ := $<string>1 + ' <> ' + $<string>3; }
             | select_expr _GE select_expr   {$<string>$ := $<string>1 + ' >= ' + $<string>3; }
             | select_expr _LE select_expr   {$<string>$ := $<string>1 + ' <= ' + $<string>3; }
             | select_expr _EQ select_expr   {$<string>$ := $<string>1 + ' = '  + $<string>3; }

             | _SUB select_expr %prec _NEG   {$<string>$ := $<string>1 + $<string>2;}
             | _PLUS select_expr %prec _NEG  {$<string>$ := $<string>1 + $<string>2;}

             | _LPAREN select_expr _RPAREN    {$<string>$ := $<string>1 + $<string>2 + $<string>3;}

             /* Aggregate functions follows */
             | RW_SUM _LPAREN distinct_aggr select_expr _RPAREN
               { $<string>$ := AddAggregate(akSUM, $<string>4); }
             | RW_MIN _LPAREN distinct_aggr select_expr _RPAREN
               { $<string>$ := AddAggregate(akMIN, $<string>4); }
             | RW_MAX _LPAREN distinct_aggr select_expr _RPAREN
               { $<string>$ := AddAggregate(akMAX, $<string>4); }
             | RW_AVG _LPAREN distinct_aggr select_expr _RPAREN
               { $<string>$ := AddAggregate(akAVG, $<string>4); }
             | RW_STDEV _LPAREN distinct_aggr select_expr _RPAREN
               { $<string>$ := AddAggregate(akSTDEV, $<string>4); }
             | RW_COUNT _LPAREN _MULT _RPAREN
               { $<string>$ := AddAggregate(akCOUNT, '0'); }
             | RW_COUNT _LPAREN distinct_aggr select_expr _RPAREN
               { $<string>$ := AddAggregate(akCOUNT, $<string>4); }
             ;

distinct_aggr : /* empty */
              | RW_DISTINCT  { Self.fIsDistinctAggr:= True; }
              ;

select_func : _IDENTIFIER _LPAREN select_list_param _RPAREN {$<string>$ := $<string>1 + $<string>2 + $<string>3 + $<string>4;}
            ;

trim_function : RW_TRIM _LPAREN trim_position _STRING RW_FROM where_expr _RPAREN
		{$<string>$ := Format('SQLTRIM(%s, %s, %d)',[$<string>4, $<string>6, fTrimPosition]);}
	      ;

trim_position : RW_LEADING   {fTrimPosition := 0;}
              | RW_TRAILING  {fTrimPosition := 1;}
              | RW_BOTH      {fTrimPosition := 2;}
              ;

extract_function : RW_EXTRACT _LPAREN extract_field RW_FROM where_expr _RPAREN
		   {$<string>$ := Format('SQLEXTRACT(%s, %d)',[$<string>5, fExtractField]);}
		 ;

extract_field : RW_YEAR   { fExtractField:= 0; }
              | RW_MONTH  { fExtractField:= 1; }
              | RW_DAY    { fExtractField:= 2; }
              | RW_HOUR   { fExtractField:= 3; }
              | RW_MINUTE { fExtractField:= 4; }
              | RW_SECOND { fExtractField:= 5; }
              ;

substring_function : RW_SUBSTRING _LPAREN where_expr RW_FROM _UINTEGER for_length _RPAREN
                     { if Length(fForLength) > 0 then
                          $<string>$ := Format('COPY(%s,%s,%s)',[$<string>3,$<string>5,fForLength])
                       else
                          $<string>$ := Format('COPY(%s,%s,LENGTH(%s))',[$<string>3,$<string>5,$<string>3]);
                       fForLength := '';}
		   ;

for_length : /* empty */
           | RW_FOR _UINTEGER {fForLength:= $<string>2;}
           ;

data_type : RW_CHAR _LPAREN _UINTEGER _RPAREN
            { fCastType := RW_CHAR;
						  fCastLen  := StrToInt( $<string>3 ); }
          | RW_INTEGER                          { fCastType := RW_INTEGER; }
          | RW_BOOLEAN                          { fCastType := RW_BOOLEAN; }
          | RW_DATE                             { fCastType := RW_DATE; }
          | RW_TIME                             { fCastType := RW_TIME; }
          | RW_DATETIME                         { fCastType := RW_DATETIME; }
          | RW_FLOAT                            { fCastType := RW_FLOAT; }
          | RW_MONEY                            { fCastType := RW_MONEY; }
          ;

define_field : _IDENTIFIER      {$<string>$ := Format('\f"%s"',[$<string>1]); }
             | qualified_field
             ;

define_param : _COLON _IDENTIFIER  { $<string>$ := '{' + $<string>1 + $<string>2 + '}';
				     AddParam( $<string>2 ); }
	     ;

qualified_field : _IDENTIFIER _PERIOD _IDENTIFIER
		    { $<string>$ := Format('\f"%s.%s"',[$<string>1, $<string>3]); }
		;

select_list_param : /* empty */
                  | _MULT
                  | list_param_expr
                  ;

list_param_expr : select_expr
		| list_param_expr _COMA select_expr {$<string>$:= $<string>1 + ', ' + $<string>3;}
		;

/* FROM clause */
from_clause : RW_FROM list_tables
	    | RW_FROM table_identifier list_join
	    ;

list_tables : table_identifier
	    | list_tables _COMA table_identifier
	    ;

table_identifier : _IDENTIFIER               {AddTable($<string>1, '');}
		 | _IDENTIFIER _IDENTIFIER   {AddTable($<string>1, $<string>2);}
		 ;

/* JOIN predicate */
list_join : list_join_table
	  | list_join    list_join_table
	  ;

list_join_table : join_action table_identifier RW_ON list_joined_fields  {Inc(fCurrentJoin);}
		;

join_action : RW_JOIN                     { fJoinKind := jkInnerJoin;      }
            | RW_INNER RW_JOIN            { fJoinKind := jkInnerJoin;      }
            | RW_LEFT RW_JOIN             { fJoinKind := jkLeftOuterJoin;  }
            | RW_LEFT RW_OUTER RW_JOIN    { fJoinKind := jkLeftOuterJoin;  }
            | RW_RIGHT RW_JOIN            { fJoinKind := jkRightOuterJoin; }
            | RW_RIGHT RW_OUTER RW_JOIN   { fJoinKind := jkRightOuterJoin; }
            | RW_FULL RW_JOIN             { fJoinKind := jkFullOuterJoin;  }
            | RW_FULL RW_OUTER RW_JOIN    { fJoinKind := jkFullOuterJoin;  }
            ;

list_joined_fields: field_joined
                  | list_joined_fields RW_AND field_joined
                  | list_joined_fields RW_OR field_joined     /* not yet supported !!! */
                  ;

field_joined : _LPAREN define_field join_rel_operator define_field _RPAREN
               { AddJoin($<string>2, $<string>4, ''); }
             | _LPAREN define_field join_rel_operator define_field RW_AND where_expr _RPAREN
               { AddJoin($<string>2, $<string>4, $<string>6); }
             | define_field join_rel_operator define_field       /* without parenthesis */
               { AddJoin($<string>1, $<string>3, ''); }
             ;

join_rel_operator : _EQ    { fJoinRelOperator := ropBETWEEN; }
                  | _GT    { fJoinRelOperator := ropGT;      }
                  | _LT    { fJoinRelOperator := ropLT;      }
                  | _GE    { fJoinRelOperator := ropGE;      }
                  | _LE    { fJoinRelOperator := ropLE;      }
                  | _NEQ   { fJoinRelOperator := ropNEQ;     }
                  ;

/* WHERE clause */
where_expr : define_field
           | where_constant
           | RW_NOW           /* NOW is a reserved word */
           | trim_function
           | extract_function
           | substring_function
           | is_null

           | _IDENTIFIER _LPAREN where_list_param _RPAREN
             {$<string>$ := $<string>1 + $<string>2 + $<string>3 + $<string>4; }
           | RW_LEFT _LPAREN where_list_param _RPAREN
             {$<string>$ := $<string>1 + $<string>2 + $<string>3 + $<string>4; }
           | RW_RIGHT _LPAREN where_list_param _RPAREN
             {$<string>$ := $<string>1 + $<string>2 + $<string>3 + $<string>4; }
           | where_expr RW_BETWEEN define_field RW_AND define_field
             {$<string>$ := Format('(%s >= %s) AND (%s <= %s)',
            [$<string>1, $<string>3, $<string>1, $<string>5]); }
           | where_expr RW_BETWEEN where_constant RW_AND where_constant
             {$<string>$ := Format('(%s >= %s) AND (%s <= %s)',
            [$<string>1, $<string>3, $<string>1, $<string>5]);
             AddWhereOptimize( $<string>1, $<string>3, $<string>5, ropBETWEEN ); }
           | where_expr in_operator _LPAREN in_predicate _RPAREN
               { $<string>$ := CreateInListExpression( $<string>1 ); }
           | where_expr in_operator anyall_subquery
             { if Pos('NOT',UpperCase($<string>2)) = 0 then
                 $<string>$ := $<string>1 + Format(' = (subquery %d)', [CurrentAnalizer.SubqueryList.Count-1])
              else
                 $<string>$ := $<string>1 + Format(' <> (subquery %d)', [CurrentAnalizer.SubqueryList.Count-1]); }

           | where_expr RW_LIKE where_constant escape_character
               { if fEscapeChar = '' then fEscapeChar := #39#39;
            $<string>$ := Format('SQLLIKE(%s, %s, %s)',[$<string>1, $<string>3, fEscapeChar]); fEscapeChar:= '';}
           | where_expr RW_NOT RW_LIKE where_constant escape_character
               { if fEscapeChar = '' then fEscapeChar := #39#39;
           $<string>$ := Format('SQLNOTLIKE(%s, %s, %s)',[$<string>1, $<string>4, fEscapeChar]); fEscapeChar:= '';}

           | _SUB where_expr %prec _NEG     {$<string>$ := $<string>1 + $<string>2;}
           | _PLUS where_expr %prec _NEG    {$<string>$ := $<string>1 + $<string>2;}

           | where_expr _PLUS where_expr    {$<string>$ := $<string>1 + ' + ' + $<string>3;}
           | where_expr _SUB where_expr     {$<string>$ := $<string>1 + ' - ' + $<string>3;}
           | where_expr _MULT where_expr    {$<string>$ := $<string>1 + ' * ' + $<string>3;}
           | where_expr _DIV where_expr     {$<string>$ := $<string>1 + ' / ' + $<string>3;}
           | where_expr _EXP where_expr     {$<string>$ := $<string>1 + ' ^ ' + $<string>3;}
           | where_expr RW_MOD where_expr   {$<string>$ := $<string>1 + ' MOD ' + $<string>3;}
           | where_expr RW_IDIV where_expr  {$<string>$ := $<string>1 + ' DIV ' + $<string>3;}
           | where_expr RW_SHL where_expr   {$<string>$ := $<string>1 + ' SHL ' + $<string>3;}
           | where_expr RW_SHR where_expr   {$<string>$ := $<string>1 + ' SHR ' + $<string>3;}

           | where_expr _GE where_expr  {$<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
                 AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropGE); }
           | where_expr _LE where_expr  {$<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
                 AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropLE); }
           | where_expr _GT where_expr  {$<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
                 AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropGT); }
           | where_expr _LT where_expr  {$<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
                 AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropLT); }
           | where_expr _EQ where_expr
             { $<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
               AddJoinCandidate( $<string>1, $<string>3 );
               AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropBETWEEN);  }

           | where_expr _NEQ where_expr
             { $<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
               AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropNEQ);   }

           | where_expr RW_AND where_expr
             { $<string>$ := $<string>1 + ' AND ' + $<string>3;
               // this is for filtering in a JOIN clause only
               with CurrentAnalizer.JoinWhereParseList do
               begin
                  if Count = 0 then
                     with Add do
                     begin
                        Expres := $<string>1;
                        RelOperator := roAND;  // not used for now
                     end;
                  with Add do
                  begin
                     Expres := $<string>3;
                     RelOperator := roAND;   // not used for now
                  end;
               end;
             }
           | where_expr RW_OR where_expr
             { $<string>$ := $<string>1 + ' OR ' + $<string>3;
               // not supported for now the OR relational operator for filtering in a JOIN
               CurrentAnalizer.JoinWhereParseList.Clear;
             }
           | RW_NOT where_expr            {$<string>$ := ' NOT ' + $<string>2;}
           | _LPAREN where_expr _RPAREN   {$<string>$ := $<string>1 + $<string>2 + $<string>3;}
           | anyall_subquery              {$<string>$ := Format('(subquery %d)', [CurrentAnalizer.SubqueryList.Count-1]);}
           ;

is_null : define_field RW_IS RW_NULL         { $<string>$ := Format('ISNULL(%s,TRUE)', [$<string>1]); }
        | define_field RW_IS RW_NOT RW_NULL  { $<string>$ := Format('ISNULL(%s,FALSE)', [$<string>1]); }
        ;


in_operator : RW_IN         { fIsNotInList := False; }
            | RW_NOT RW_IN  { $<string>$ := $<string>1 + #32 + $<string>2; fIsNotInList := True; }
            ;

in_predicate : where_constant                    { fInPredicateList.Add( $<string>1 ); }
             | in_predicate _COMA where_constant { fInPredicateList.Add( $<string>3 ); }
             ;

where_constant : _UINTEGER
               | _SINTEGER
               | _NUMERIC
               | _STRING
               | RW_TRUE         { $<string>$ := 'DummyBoolean(True)'; }
               | RW_FALSE        { $<string>$ := 'DummyBoolean(False)'; }
               | define_param
               ;

where_list_param : where_expr
                 | where_list_param _COMA where_expr {$<string>$:= $<string>1 + ', ' + $<string>3;}
                 ;

where_clause : /* empty */
             | RW_WHERE where_expr                  { CurrentAnalizer.WhereStr := $<string>2; }
             ;

anyall_subquery : subquery         {CurrentAnalizer.SubQueryKindList.Add( Pointer(skAny) );}
                | RW_ANY subquery  {CurrentAnalizer.SubQueryKindList.Add( Pointer(skAny) );}
                | RW_ALL subquery  {CurrentAnalizer.SubQueryKindList.Add( Pointer(skAll) );}
                ;

subquery : _LPAREN subq_select_clause from_clause where_clause groupby_clause orderby_clause end_subquery
         ;

/* start a new subquery */
subq_select_clause : subq_select_case _MULT              {CurrentAnalizer.DoSelectAll := True;}
                   | subq_select_case list_expr_field
                   ;

subq_select_case : start_subquery
                 | start_subquery RW_DISTINCT { CurrentAnalizer.IsDistinct := True; }
                 ;

start_subquery : RW_SELECT   { CreateNewSubquery; }
               ;

end_subquery : _RPAREN   { Self.fCurrAnalizer := Self.fCurrAnalizer.ParentAnalizer; }
             ;

escape_character : /* empty */        {fEscapeChar := '';}
		 | RW_ESCAPE _STRING  {fEscapeChar := $<string>2;}
		 ;

/* GROUP BY clause */

groupby_clause : /* empty */
	       | RW_GROUP RW_BY list_fields_group having_predicate
	       ;


list_fields_group : define_field_index                         {AddGroupBy( $<string>1 );}
		  | list_fields_group _COMA define_field_index {AddGroupBy( $<string>3 );}
		  ;

		   /*  \f"field" is a special code for signaling fields without table qualifier and solved later */
define_field_index : _IDENTIFIER         { $<string>$ := Format('\f"%s"', [$<string>1]); }
		   | qualified_field
		   | _UINTEGER           { $<string>$ := $<string>1; }
		   ;

having_predicate : /* empty */
		 | RW_HAVING select_expr
		   { AddHavingColumn( $<string>2 );
		     CurrentAnalizer.HavingCol := CurrentAnalizer.ColumnList.Count-1; }
		 ;

/* ORDER BY clause */

orderby_clause : /* empty */
	       | RW_ORDER RW_BY list_fields_order
	       ;

list_fields_order : define_fields_order
		  | list_fields_order _COMA define_fields_order
		  ;

define_fields_order : define_field_index           {AddOrderBy( $<string>1, False );}
		    | define_field_index RW_ASC    {AddOrderBy( $<string>1, False );}
		    | define_field_index RW_DESC   {AddOrderBy( $<string>1, True );}
		    ;

/* end of select */
end_statement : /* empty */
	      | _SEMICOLON
	      ;

/* UPDATE statement */
update_statement : RW_UPDATE table_identifier RW_SET list_update_columns where_clause end_statement
                 ;

list_update_columns : update_column
                    | list_update_columns _COMA update_column
                    ;

update_column : _IDENTIFIER _EQ where_expr {AddUpdateColumn($<string>1,$<string>3);}
	            ;

/* DELETE statement */
delete_statement : RW_DELETE RW_FROM table_identifier where_clause end_statement
                 ;

/* INSERT statement */
insert_statement : insert_statement_list end_statement
                 ;

insert_statement_list : insert_one_record                        { Inc(fNumInserts); }
                      | insert_satement_list insert_one_record   { Inc(fNumInserts); }
                      ;

insert_one_record : RW_INSERT RW_INTO _IDENTIFIER insert_columns_list insert_values_list
		      { CurrentInsertItem.TableName := $<string>3 ; }
		  ;

insert_columns_list : _LPAREN columns_list _RPAREN
		    | _LPAREN _MULT _RPAREN
		    ;

/* define the fields to update */
columns_list : _IDENTIFIER                       {CurrentInsertItem.FieldNames.Add($<string>1);}
	     | columns_list _COMA _IDENTIFIER    {CurrentInsertItem.FieldNames.Add($<string>3);}
	     ;

insert_values_list : RW_VALUES _LPAREN values_list _RPAREN
		   | subquery
		   ;

/* define the values to store in fields */
values_list : where_expr                    {CurrentInsertItem.ExprList.Add( $<string>1 );}
	    | values_list _COMA where_expr  {CurrentInsertItem.ExprList.Add( $<string>3 );}
	    ;

/* CREATE TABLE STATEMENT */
createtable_statement : createtable_list end_statement
		      ;

createtable_list : createone_table                   {Inc(FNumTables);}
		 | createtable_list createone_table  {Inc(FNumTables);}
		 ;

createone_table : RW_CREATE RW_TABLE _STRING _LPAREN create_list primary_key _RPAREN
		  {SetTableName( Copy($<string>3, 2, Length($<string>3) - 2) );}
		;

/* the type of fields that can be created */
field_type : RW_CHAR _LPAREN _UINTEGER _RPAREN  {SetFieldParams(RW_CHAR,0,0,StrToInt($<string>3),0);}
            | RW_INTEGER                         {SetFieldParams(RW_INTEGER,0,0,0,0);}
            | RW_SMALLINT                        {SetFieldParams(RW_SMALLINT,0,0,0,0);}
            | RW_BOOLEAN                         {SetFieldParams(RW_BOOLEAN,0,0,0,0);}
            | RW_DATE                            {SetFieldParams(RW_DATE,0,0,0,0);}
            | RW_TIME                            {SetFieldParams(RW_TIME,0,0,0,0);}
            | RW_DATETIME                        {SetFieldParams(RW_DATETIME,0,0,0,0);}
            | RW_MONEY                           {SetFieldParams(RW_MONEY,0,0,0,0);}
            | RW_FLOAT                           {SetFieldParams(RW_FLOAT,0,0,0,0);}
            | RW_FLOAT _LPAREN _UINTEGER _RPAREN {SetFieldParams(RW_FLOAT,StrToInt($<string>3),0,0,0);}
            | RW_FLOAT _LPAREN _UINTEGER _COMA _UINTEGER _RPAREN {SetFieldParams(RW_FLOAT,StrToInt($<string>3),StrToInt($<string>5),0,0);}
            | RW_AUTOINC                         {SetFieldParams(RW_AUTOINC,0,0,0,0);}
            | RW_BLOB _LPAREN _UINTEGER _RPAREN  {SetFieldParams(RW_BLOB,0,0,0,StrToInt($<string>3));}
            ;

create_list : _IDENTIFIER field_type                     {AddCreateField($<string>1);}
            | create_list _COMA _IDENTIFIER field_type   {AddCreateField($<string>3);}
            ;

primary_key : /* empty */
            | RW_PRIMARY RW_KEY _LPAREN create_field_list _RPAREN
            ;

create_field_list : _IDENTIFIER                           { AddPrimaryKeyField( $<string>1 ); }
                  | create_field_list _COMA _IDENTIFIER   { AddPrimaryKeyField( $<string>3 ); }
                  ;

/* CREATE INDEX statement */
createindex_statement : RW_CREATE index_unique index_order RW_INDEX _IDENTIFIER RW_ON _STRING _LPAREN index_field_list _RPAREN end_statement
                        { CurrentAnalizer.IndexName  := $<string>5;
                          CurrentAnalizer.IndexTable := Copy( $<string>7, 2, Length($<string>7) - 2 ); }
                      ;

index_unique : /* empty */
             | RW_UNIQUE  { CurrentAnalizer.IndexUnique := True; }
             ;

index_order : /* empty */
	    | RW_ASC
	    | RW_DESC      { CurrentAnalizer.IndexDescending := True; }
	    ;

index_field_list : _IDENTIFIER   {CurrentAnalizer.IndexColumnList.Add($<string>1);}
		 | index_field_list _COMA _IDENTIFIER  {CurrentAnalizer.IndexColumnList.Add($<string>3);}
		 ;

/* DROP TABLE statement */
droptable_statement : RW_DROP RW_TABLE _STRING end_statement {CurrentAnalizer.IndexTable:= Copy($<string>3, 2, Length($<string>3) - 2);}
		    ;

/* DROP INDEX statement */
dropindex_statement : RW_DROP RW_INDEX _STRING _IDENTIFIER end_statement
		     { CurrentAnalizer.IndexTable:= Copy($<string>3, 2, Length($<string>3) - 2);
		       CurrentAnalizer.IndexName := $<string>4; }
		    ;

/* TRANSFORM...PIVOT statement */
transform_statement : RW_TRANSFORM transf_aggregate_list select_statement RW_PIVOT select_expr pivot_in_predicate end_statement
		                { CurrentAnalizer.PivotStr := $<string>5; }
		              ;

/* this is needed in order to reduce this column first */
transf_aggregate : select_expr  { AddColumn( $<string>1, True ); }
                 | RW_CAST _LPAREN select_expr RW_AS data_type _RPAREN { AddColumn( $<string>3, True ); }
                 ;

transf_aggregate_list : transf_aggregate
                      | transf_aggregate_list _COMA transf_aggregate
                      ;

pivot_in_predicate : /* empty */
		             | RW_IN _LPAREN pivot_in_list _RPAREN
		             ;

pivot_in_list : where_constant
         		{CurrentAnalizer.PivotInList.Add( RemoveStrDelim( $<string>1));}
              | pivot_in_list _COMA where_constant
         		{CurrentAnalizer.PivotInList.Add( RemoveStrDelim( $<string>3));}
     	        ;

/* MERGE WITH statement */
merge_statement : RW_MERGE select_statement RW_WITH second_select_statement end_statement

second_select_statement : second_select_clause from_clause where_clause groupby_clause orderby_clause
                        ;

/* start a new select statement */
second_select_clause : second_select_case _MULT              {CurrentAnalizer.DoSelectAll := True;}
                     | second_select_case list_expr_field
                     ;

second_select_case : second_select
                   | second_select RW_DISTINCT { CurrentAnalizer.IsDistinct := True; }
                   ;

/* create the second select */
second_select : RW_SELECT
                { Self.fAnalizer.MergeAnalizer := TSqlAnalizer.Create(nil, fAnalizer.xQuery);
                  Self.fCurrAnalizer := Self.fAnalizer.MergeAnalizer; }
              ;

%%




