{-------------------------------------------------------------------------------
 Module:    TGlobe Testbed

 Comment:   Test program for TGlobe 4

 Author:    Graham Knight
 Email:     tglobe@iname.com
-------------------------------------------------------------------------------}
unit Unit1;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Spin, Printers, Globe4, TGSysUtils, TGClasses,
  TGDBMapper, TGLYRMapper, TGObjects, Db, DBTables, GlobeProjections,
  ComCtrls, Buttons;

type
  TForm1 = class(TForm)
    Panel1: TPanel;
    bOpen: TButton;
    bExtents: TButton;
    Globe: TGlobe4;
    OpenDialog1: TOpenDialog;
    lbLayers: TListBox;
    bPrint: TButton;
    bPlus: TButton;
    bMinus: TButton;
    bBitmap: TButton;
    GroupBox1: TGroupBox;
    cbDoubleBuffer: TCheckBox;
    cbTransparent: TCheckBox;
    cbInterruptable: TCheckBox;
    cbCache3D: TCheckBox;
    GroupBox2: TGroupBox;
    Label2: TLabel;
    lblMemUsed: TLabel;
    bImage: TButton;
    SpinEdit1: TSpinEdit;
    bSaveLYR: TButton;
    SaveDialog1: TSaveDialog;
    bSaveEnv: TButton;
    bLoadEnv: TButton;
    bClear: TButton;
    GlobeB: TGlobe4;
    OpenDialog2: TOpenDialog;
    bTextureMap: TButton;
    DataSource1: TDataSource;
    Table1: TTable;
    bDBLayer: TButton;
    SpinEdit2: TSpinEdit;
    cbProjectionType: TComboBox;
    bUp: TSpeedButton;
    bDown: TSpeedButton;
    bTrash: TSpeedButton;
    procedure bOpenClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure SpinEdit1Change(Sender: TObject);
    procedure GlobeRender(Sender: TObject);
    procedure bExtentsClick(Sender: TObject);
    procedure lbLayersClick(Sender: TObject);
    procedure GlobeObjectClick(Sender: TGlobe4; GlobeObject: TGlobeObject);
    procedure bPrintClick(Sender: TObject);
    procedure bPlusClick(Sender: TObject);
    procedure bMinusClick(Sender: TObject);
    procedure GlobePaint(Sender: TObject);
    procedure GlobeProgress(Sender: TGlobe4; MsgType: TProgressMessage;
      const MsgText: String; var Abort: Boolean);
    procedure bBitmapClick(Sender: TObject);
    procedure cbDoubleBufferClick(Sender: TObject);
    procedure cbTransparentClick(Sender: TObject);
    procedure cbInterruptableClick(Sender: TObject);
    procedure cbCache3DClick(Sender: TObject);
    procedure bImageClick(Sender: TObject);
    procedure GlobeMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure bSaveLYRClick(Sender: TObject);
    procedure bSaveEnvClick(Sender: TObject);
    procedure bLoadEnvClick(Sender: TObject);
    procedure bClearClick(Sender: TObject);
    procedure GlobeSourceCreate(Sender: TGlobe4;
      ObjectSource: TGlobeObjectSource);
    procedure bTextureMapClick(Sender: TObject);

    procedure bDBLayerClick(Sender: TObject);
    procedure SpinEdit2Change(Sender: TObject);
    procedure cbProjectionTypeChange(Sender: TObject);
    procedure bDownClick(Sender: TObject);
    procedure bUpClick(Sender: TObject);
    procedure bTrashClick(Sender: TObject);
  private
    { Private declarations }
    procedure UpdateLayerList;
  public
    { Public declarations }
  end;

var
  Form1: TForm1;

implementation

{$R *.DFM}

Uses fmImage, GlobeUtils, TGPresenters;

{------------------------------------------------------------------------------
  TForm1.bOpenClick
------------------------------------------------------------------------------}
procedure TForm1.bOpenClick(Sender: TObject);
var
  Start : DWord;
  idx : integer;
  ExePath : string;
begin
  ExePath := ExtractFilePath( Application.ExeName );

  if OpenDialog1.Execute then
    for idx := 0 to OpenDialog1.Files.Count - 1 do
    begin
      Start := GetTickCount;

      CreateFileLayer( Globe, OpenDialog1.Files[idx] );

      Caption := 'Load Time ' + IntToStr( GetTickCount - Start);
    end;
  UpdateLayerList;
end;

{------------------------------------------------------------------------------
  TForm1.FormCreate
------------------------------------------------------------------------------}
procedure TForm1.FormCreate(Sender: TObject);
begin
  SpinEdit1.Value := Globe.CacheCapacity;
  cbProjectionType.ItemIndex := 0;
end;

{------------------------------------------------------------------------------
  TForm1.SpinEdit1Change
------------------------------------------------------------------------------}
procedure TForm1.SpinEdit1Change(Sender: TObject);
begin
  Globe.CacheCapacity := SpinEdit1.Value;
end;

{------------------------------------------------------------------------------
  TForm1.GlobeRender
------------------------------------------------------------------------------}
procedure TForm1.GlobeRender(Sender: TObject);
// display lat and long at the all cross points of the globe grid
var
  i, j: integer;
  Long, Lat : integer;
  Units : TGlobeUnitTypes;
  iLeft, iTop, iTA, iBkMode : integer;
begin
  // Draw a scale onto the globe
  iLeft := Globe.GlobeCanvas.CanvasWidth - 150;
  iTop := Globe.GlobeCanvas.CanvasHeight - 50;

  with Globe.GlobeCanvas do
  begin
    Font.Color := clRed;
    iBkMode := SetBkMode(Handle, TRANSPARENT);
    iTA := SetTextAlign(Handle, TA_CENTER);

    Units := KiloMeter;
    TextOut(iLeft, iTop, '0');
    TextOut(iLeft + CanvasPixelsPerInch, iTop,
        Format('%d %s', [Round(GlobeUnitsTo(Projection.UnitsPerInch, Units)),
          UnitsToStr(Units)]));

    SetBkMode(Handle, iBkMode);
    SetTextAlign(Handle, iTA);

    MoveTo(iLeft, iTop + 16);
    LineTo(iLeft, iTop + 20);
    LineTo(iLeft + CanvasPixelsPerInch, iTop + 20);
    LineTo(iLeft + CanvasPixelsPerInch, iTop + 15);
  end;

Exit;
  // display the grid coordinates on the graticule
  with Globe.GlobeCanvas do
  begin
    Font.Name := 'Arial'; // need to select a TT font
    Font.Size := 6;  //change font size
    Font.Color := clBlack;

    with globe do
    for i := -11 to 12 do
    for j := -5 to 5 do // show all lat and long
    begin
      Long := i * GU_DEGREE * GraticuleLongitudeStep; // 15 degree steps
      Lat := j * GU_DEGREE * GraticuleLatitudeStep; // 15 degree steps

      RenderTextOut( Long, Lat,
        GlobeUnitsToStr( Long, '%d E') + '  ' + GlobeUnitsToStr( Lat, '%d N' ),
        Pixel ); // Try NauticalMile instead of Pixel and zoom in.
    end;
  end;

  // show the limits of the display area
{  with Globe do
  begin
    GlobeCanvas.Pen.Color := clBlack;
    RenderLine(
      DecimalToPointLL( MinLongitude, MinLatitude ),
      DecimalToPointLL( MinLongitude, MaxLatitude ), 32);
    RenderLine(
      DecimalToPointLL( MinLongitude, MaxLatitude ),
      DecimalToPointLL( MaxLongitude, MaxLatitude ), 32);
    RenderLine(
      DecimalToPointLL( MaxLongitude, MaxLatitude ),
      DecimalToPointLL( MaxLongitude, MinLatitude ), 32);
    RenderLine(
      DecimalToPointLL( MaxLongitude, MinLatitude ),
      DecimalToPointLL( MinLongitude, MinLatitude ), 32);
  end;
}
end;

{------------------------------------------------------------------------------
  TForm1.bExtentsClick
------------------------------------------------------------------------------}
procedure TForm1.bExtentsClick(Sender: TObject);
begin
  Globe.Projection.Altitude := 0;
end;

{------------------------------------------------------------------------------
  TForm1.bPlusClick
------------------------------------------------------------------------------}
procedure TForm1.bPlusClick(Sender: TObject);
begin
  Globe.Projection.Altitude := Round( Globe.Projection.Altitude * 0.8 );
end;

{------------------------------------------------------------------------------
  TForm1.bMinusClick
------------------------------------------------------------------------------}
procedure TForm1.bMinusClick(Sender: TObject);
begin
  if Round( Globe.Projection.Altitude * 1.2 ) < MaxInt then
    Globe.Projection.Altitude := Round( Globe.Projection.Altitude * 1.2 );
end;


{------------------------------------------------------------------------------
  TForm1.lbLayersClick
------------------------------------------------------------------------------}
procedure TForm1.lbLayersClick(Sender: TObject);
var
  iLayer : integer;
begin
  iLayer := lbLayers.ItemIndex;

  // Check to see that the layer has a data read
  bSaveLYR.Enabled := Globe.Layers[iLayer].Objects <> nil;
  bTrash.Enabled := bSaveLYR.Enabled;
end;

{------------------------------------------------------------------------------
  TForm1.GlobeObjectClick
------------------------------------------------------------------------------}
procedure TForm1.GlobeObjectClick(Sender: TGlobe4;
  GlobeObject: TGlobeObject);
begin
  Caption := LayerFromObject( Globe, GlobeObject ).Name + ': ' + GlobeObject.Title;
end;

{------------------------------------------------------------------------------
  TForm1.bPrintClick
------------------------------------------------------------------------------}
procedure TForm1.bPrintClick(Sender: TObject);
begin
  // Print the current image.
  Printer.BeginDoc;
  Globe.RenderToCanvas( Printer.Canvas, Globe.ClientRect, True );
  Printer.EndDoc;
end;

{------------------------------------------------------------------------------
  TForm1.GlobePaint
------------------------------------------------------------------------------}
procedure TForm1.GlobePaint(Sender: TObject);
begin
  lblMemUsed.Caption := IntToStr( Globe.CacheMemoryUsed div 1024 );
end;

{------------------------------------------------------------------------------
  TForm1.GlobeProgress
------------------------------------------------------------------------------}
procedure TForm1.GlobeProgress(Sender: TGlobe4; MsgType: TProgressMessage;
  const MsgText: String; var Abort: Boolean);
begin
  case MsgType of
  pmPercent : Caption := 'Mapped ' + MsgText + ' Percent';
  else
    Caption := MsgText;
  end;
end;

{------------------------------------------------------------------------------
  TForm1.bBitmapClick
------------------------------------------------------------------------------}
procedure TForm1.bBitmapClick(Sender: TObject);
var
  aLayer : TGlobeLayer;
begin
  bBitmap.Enabled := false;

  aLayer := TGlobeLayer.Create( Globe );
  aLayer.Name := 'Bitmap Test';

  with TBitmapObject.Create( aLayer.Objects ) do
  begin
    BMPFilename := 'World.bmp';
    GlobeUnitsPerPixel := GU_MINUTE * 4;
    Centroid := PointLL( -GU_DEGREE * 20, GU_DEGREE * 10 );
  end;
end;

{------------------------------------------------------------------------------
  TForm1.bSaveLYRClick
------------------------------------------------------------------------------}
procedure TForm1.bSaveLYRClick(Sender: TObject);
begin
  with SaveDialog1 do
  begin
    InitialDir := ExtractFilePath( Application.ExeName );
    Filename := ChangeFileExt( Globe.Layers[LbLayers.ItemIndex].Name, '.LYR' );

    if Execute then
      WriteLayerToLYRfile( Globe.Layers[LbLayers.ItemIndex], Filename );
  end;
end;

{------------------------------------------------------------------------------
  TForm1.cbDoubleBufferClick
------------------------------------------------------------------------------}
procedure TForm1.cbDoubleBufferClick(Sender: TObject);
begin
  if TCheckBox( Sender ).Checked then
    Globe.GlobeOptions := Globe.GlobeOptions + [goDoubleBuffered]
  else
    Globe.GlobeOptions := Globe.GlobeOptions - [goDoubleBuffered];
end;

{------------------------------------------------------------------------------
  TForm1.cbTransparentClick
------------------------------------------------------------------------------}
procedure TForm1.cbTransparentClick(Sender: TObject);
begin
  if TCheckBox( Sender ).Checked then
    Globe.GlobeOptions := Globe.GlobeOptions + [goTransparentGlobe]
  else
    Globe.GlobeOptions := Globe.GlobeOptions - [goTransparentGlobe];
end;

{------------------------------------------------------------------------------
  TForm1.cbInterruptableClick
------------------------------------------------------------------------------}
procedure TForm1.cbInterruptableClick(Sender: TObject);
begin
  if TCheckBox( Sender ).Checked then
    Globe.GlobeOptions := Globe.GlobeOptions + [goInterruptable]
  else
    Globe.GlobeOptions := Globe.GlobeOptions - [goInterruptable];
end;

{------------------------------------------------------------------------------
  TForm1.cbCache3DClick
------------------------------------------------------------------------------}
procedure TForm1.cbCache3DClick(Sender: TObject);
begin
  if TCheckBox( Sender ).Checked then
    Globe.GlobeOptions := Globe.GlobeOptions + [goCache3Dpoints]
  else
    Globe.GlobeOptions := Globe.GlobeOptions - [goCache3Dpoints];
end;

{------------------------------------------------------------------------------
  TForm1.bImageClick
------------------------------------------------------------------------------}
procedure TForm1.bImageClick(Sender: TObject);
begin
  FormImage.Globe := Globe;
  FormImage.ShowModal;
end;

{------------------------------------------------------------------------------
  TForm1.GlobeMouseMove
------------------------------------------------------------------------------}
procedure TForm1.GlobeMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
var
  ptLL : TPointLL;
begin
  if Globe.Projection.DeviceXYToLL( X, Y, ptLL ) then
    Caption := Format( '(%d,%d)', [ptLL.iLatY, ptLL.iLongX] )
//    Caption := Format( '(%0.2f,%0.2f)', [GlobeUnitsToDecimal( ptLL.iLatY ),
//      GlobeUnitsToDecimal( ptLL.iLongX )] )
  else
    Caption := '';

end;

{------------------------------------------------------------------------------
  TForm1.bSaveEnvClick
------------------------------------------------------------------------------}
procedure TForm1.bSaveEnvClick(Sender: TObject);
begin
  Globe.SaveEnvironmentToFile( ChangeFileExt( Application.ExeName, '.ENV' ));
end;

{------------------------------------------------------------------------------
  TForm1.bLoadEnvClick
------------------------------------------------------------------------------}
procedure TForm1.bLoadEnvClick(Sender: TObject);
begin
  Globe.Clear;
  GlobeB.Clear;
  Globe.LoadEnvironmentFromFile( ChangeFileExt( Application.ExeName, '.ENV' ));
  cbProjectionType.ItemIndex := cbProjectionType.Items.IndexOf( Globe.Projection.ProjectionClass );
  UpdateLayerList;
end;

{------------------------------------------------------------------------------
  TForm1.bClearClick
------------------------------------------------------------------------------}
procedure TForm1.bClearClick(Sender: TObject);
begin
  Globe.Clear;
  GlobeB.Clear;
  UpdateLayerList;
end;


{------------------------------------------------------------------------------
  TForm1.GlobeSourceCreate
------------------------------------------------------------------------------}
procedure TForm1.GlobeSourceCreate(Sender: TGlobe4;
  ObjectSource: TGlobeObjectSource);
var
  aLayer : TGlobeLayer;
begin
  // Attach the Layer to GlobeB
  aLayer := TGlobeLayer.Create( GlobeB );
  aLayer.Objects := ObjectSource;
end;

{------------------------------------------------------------------------------
  TForm1.bTextureMapClick
------------------------------------------------------------------------------}
procedure TForm1.bTextureMapClick(Sender: TObject);
begin
  with OpenDialog2 do
    if Execute then
      Globe.SurfaceTextureName := Filename
    else
      Globe.SurfaceTextureName := '';
end;

{------------------------------------------------------------------------------
  TForm1.UpdateLayerList
------------------------------------------------------------------------------}
procedure TForm1.UpdateLayerList;
var
  idx, iIndex : integer;
begin
  iIndex := lbLayers.ItemIndex;

  // update the list of loaded layers
  lbLayers.Items.Clear;
  for idx := 0 to Globe.Layers.Count - 1 do
    lbLayers.Items.Add( Globe.Layers[idx].Name );

  if iIndex < lbLayers.Items.Count then
    lbLayers.ItemIndex := iIndex;
end;

{------------------------------------------------------------------------------
  TForm1.bDBLayerClick
------------------------------------------------------------------------------}
procedure TForm1.bDBLayerClick(Sender: TObject);
var
  aLayer : TGlobeLayer;
begin
  Table1.DatabaseName := ExtractFilePath( Application.ExeName );

  aLayer := TGlobeLayer.Create( Globe );
  aLayer.Objects := TGlobeDBPointReader.Create( Globe );
  with TGlobeDBPointReader( aLayer.Objects ) do
  begin
    DataSource := DataSource1;
    IDColName := 'ID';
    LongitudeColName := 'LONG';
    LatitudeColName := 'LAT';
    Active := True;
    SpinEdit1.Enabled := Active;
  end;
end;

{------------------------------------------------------------------------------
  TForm1.SpinEdit2Change
------------------------------------------------------------------------------}
procedure TForm1.SpinEdit2Change(Sender: TObject);
begin
  Table1.Active := False;
  Table1.Filter := 'elev > ' + IntToStr( SpinEdit2.Value );
  Table1.Filtered := True;
  Table1.Active := True;
end;

{------------------------------------------------------------------------------
  TForm1.cbProjectionTypeChange
------------------------------------------------------------------------------}
procedure TForm1.cbProjectionTypeChange(Sender: TObject);
begin
  with cbProjectionType do
    Globe.Projection.ProjectionClass := Items[ItemIndex];
end;

{------------------------------------------------------------------------------
  TForm1.bDownClick
------------------------------------------------------------------------------}
procedure TForm1.bDownClick(Sender: TObject);
begin
  if lbLayers.ItemIndex < lbLayers.Items.Count - 1 then
  begin
    Globe.Layers.Move( lbLayers.ItemIndex, lbLayers.ItemIndex + 1 );
    lbLayers.ItemIndex := lbLayers.ItemIndex + 1;
    UpdateLayerList;
  end;
end;

{------------------------------------------------------------------------------
  TForm1.bUpClick
------------------------------------------------------------------------------}
procedure TForm1.bUpClick(Sender: TObject);
begin
  if lbLayers.ItemIndex > 0 then
  begin
    Globe.Layers.Move( lbLayers.ItemIndex, lbLayers.ItemIndex - 1 );
    lbLayers.ItemIndex := lbLayers.ItemIndex - 1;
    UpdateLayerList;
  end;
end;

{------------------------------------------------------------------------------
  TForm1.bTrashClick
------------------------------------------------------------------------------}
procedure TForm1.bTrashClick(Sender: TObject);
begin
  if lbLayers.ItemIndex >= 0 then
  begin
    Globe.Layers.Delete( Globe.Layers[lbLayers.ItemIndex] );
    UpdateLayerList;
  end;
end;

end.
