{-------------------------------------------------------------------------
 Module:    TGlobe XML Classes

 Comment:   XML parsing and management classes

 Classes:   TGXML_Document
            TGXML_Element

 Author:    Graham Knight
 Email:     tglobe@iname.com

 1.01 22-Aug-2000 Code Isolate from the windows DecimalSeparator setting.
 1.02 24-Sep-2000 AddBinaryElement code
-------------------------------------------------------------------------}
unit TGXML;

interface

uses
  SysUtils, Classes;

type
  TGXML_Element = class(TStringList)
  private
    function GetAttributeNames( Index : integer ) : string;
  public
    destructor Destroy; override;

    procedure AddTextElement(const Ident, Text: string);
    procedure AddBinaryElement(const Ident : string; var Buffer; BufSize: Integer);
    procedure AddElement( const Ident : string; Element : TGXML_Element );
    function ElementByName( const Ident : string; iOccourance : integer ) : TGXML_Element;

    procedure AddAttribute( const Ident, Value : string );
    procedure AddBoolAttribute( const Ident : string; Value : Boolean );
    procedure AddFloatAttribute( const Ident : string; Value : Extended );
    procedure AddIntAttribute( const Ident : string; Value : integer );

    function AttributeByName( const Ident, DefaultValue : string ) : string;
    function BoolAttributeByName( const Ident : String; DefaultValue : Boolean ) : Boolean;
    function FloatAttributeByName( const Ident : String; DefaultValue : Extended ) : Extended;
    function IntAttributeByName( const Ident : String; DefaultValue : integer ) : integer;
    function AsBinary( var Buffer; BufSize: Integer) : integer;

    property AttributeNames[Index : integer] : string read GetAttributeNames;
  end;

  TGXML_Document = class
  private
    FDocument : TGXML_Element;
    function GetXMLString : string;
    procedure SetXMLString( const XML : string );
    procedure SetDocument( doc : TGXML_Element );
  public
    constructor Create;
    destructor Destroy; override;

    procedure SaveToFile( const sFilename : string );
    function LoadFromFile( const sFilename : string ) : Boolean;

    property AsString : string read GetXMLstring write SetXMLstring;
    property Document : TGXML_Element read FDocument write SetDocument;
  end;

implementation

{------------------------------------------------------------------------------
  TGXML_Element.Destroy
------------------------------------------------------------------------------}
destructor TGXML_Element.Destroy;
var
  idx : integer;
begin
  for idx := 0 to Count - 1 do
    if Objects[idx] <> Self then
      Objects[idx].Free;

  inherited;
end;

{------------------------------------------------------------------------------
  TGXML_Element.AddAttribute
------------------------------------------------------------------------------}
{**
  @Param Ident Name of Attribute.
  @Param Value Value for named Attribute.
}
procedure TGXML_Element.AddAttribute(const Ident, Value: string);
begin
  AddObject( Trim( Ident ) + '="' + Trim( Value ) + '"', Self );
end;

{------------------------------------------------------------------------------
  TGXML_Element.AddBoolAttribute
------------------------------------------------------------------------------}
{**
  @Param Ident Name of Attribute.
  @Param Value Value for named Attribute.
}
procedure TGXML_Element.AddBoolAttribute( const Ident : string; Value : Boolean );
begin
  AddAttribute( Ident, 'FT'[Ord(Value)+1]);
end;

{------------------------------------------------------------------------------
  TGXML_Element.AddFloatAttribute
------------------------------------------------------------------------------}
{**
  @Param Ident Name of Attribute.
  @Param Value Value for named Attribute.
}
procedure TGXML_Element.AddFloatAttribute( const Ident : string; Value : Extended );
var
  SaveDecimalSeparator : Char;
begin
  SaveDecimalSeparator := DecimalSeparator;
  DecimalSeparator := '.';
  AddAttribute( Ident, FloatToStr( Value ));
  DecimalSeparator := SaveDecimalSeparator;
end;

{------------------------------------------------------------------------------
  TGXML_Element.AddIntAttribute
------------------------------------------------------------------------------}
{**
  @Param Ident Name of Attribute.
  @Param Value Value for named Attribute.
}
procedure TGXML_Element.AddIntAttribute( const Ident : string; Value : integer );
begin
  AddAttribute( Ident, IntToStr( Value ));
end;

{------------------------------------------------------------------------------
  TGXML_Element.AddTextElement
------------------------------------------------------------------------------}
{**
  @Param Ident Name of Element.
  @Param Text Content for Element.
}
procedure TGXML_Element.AddTextElement( const Ident, Text: string );
var
  el : TGXML_Element;
begin
  el := TGXML_Element.Create;
  el.Add( Trim( Text ));
  AddElement( Trim( Ident ), el );
end;

{------------------------------------------------------------------------------
  TGXML_Element.AddBinaryElement
------------------------------------------------------------------------------}
{**
  @Param Ident Name of Element.
  @Param Buffer Data to write.
  @Param BufSize number of bytes to write.
}
procedure TGXML_Element.AddBinaryElement(const Ident: string; var Buffer;
  BufSize: Integer);
var
  el : TGXML_Element;
  sText : string;
  idx : integer;
begin
  SetLength( sText, BufSize * 2 );
  for idx := 0 to BufSize - 1 do
  begin
    sText[idx*2 + 1] := '0123456789ABCDEF'[(Ord(PChar(Buffer)[idx]) shr 4 ) and $0F + 1];
    sText[idx*2 + 2] := '0123456789ABCDEF'[Ord(PChar(Buffer)[idx]) and $0F + 1];
  end;
  el := TGXML_Element.Create;
  el.Add( sText );
  AddElement( Trim( Ident ), el );
end;

{------------------------------------------------------------------------------
  TGXML_Element.AsBinary
------------------------------------------------------------------------------}
{**
  @Param Buffer Where to write the data.
  @Param BufSize Size of the Buffer.
  @Result number of bytes written into the buffer.
}
function TGXML_Element.AsBinary( var Buffer; BufSize: Integer) : integer;
begin
  Result := 0;
  while Result < BufSize do
  begin

    Inc( Result );
  end
end;

{------------------------------------------------------------------------------
  TGXML_Element.AddElement
------------------------------------------------------------------------------}
{**
  @Param Ident Name of Sub Element to add.
  @Param Element Sub Element to add.
}
procedure TGXML_Element.AddElement(const Ident: string; Element: TGXML_Element);
begin
  AddObject( Trim( Ident ), Element );
end;

{------------------------------------------------------------------------------
  TGXML_Element.ElementByName
------------------------------------------------------------------------------}
{**
  @Param Ident name of Sub Element to find.
  @Result Element if found.
}
function TGXML_Element.ElementByName( const Ident : string; iOccourance : integer ) : TGXML_Element;
var
  idx : integer;
begin
  for idx := 0 to Count - 1 do
  begin
    Result := TGXML_Element(Objects[idx]);
    if ( Result <> nil ) and ( Result <> Self ) then
      if strings[idx] = Ident then
        if iOccourance = 0 then
          Exit
        else
          Dec( iOccourance );
  end;
  Result := nil;
end;

{------------------------------------------------------------------------------
  TGXML_Element.GetAttributeNames
------------------------------------------------------------------------------}
{**
  @Param Index Index of Attribute to return.
  @Result Name of Attribute at Index position or '' if no Attribute exists.
}
function TGXML_Element.GetAttributeNames( Index : integer ) : string;
var
  idx : integer;
begin
  for idx := 0 to Count -1 do
    if objects[idx] = self then
      if Index = 0 then
      begin
        Result := Names[idx];
        Exit;
      end;
  Result := '';
end;

{------------------------------------------------------------------------------
  TGXML_Element.AttributeByName
------------------------------------------------------------------------------}
{**
  @Param Ident name of Attribute to find.
  @Param DefaultValue Value to use if Ident does not exist.
  @Result Value of Attribute or DefaultValue if the Ident is not found.
}
function TGXML_Element.AttributeByName( const Ident, DefaultValue : string ) : string;
begin
  Result := Trim( Values[Ident] );
  if Result <> '' then
    Result := Copy( Result, 2, Length(Result) - 2 )
  else
    Result := DefaultValue;
end;

{------------------------------------------------------------------------------
  TGXML_Element.BoolAttributeByName
------------------------------------------------------------------------------}
{**
  @Param Ident name of Attribute to find.
  @Param DefaultValue Value to use if Ident does not exist.
  @Result Value of Attribute or DefaultValue if the Ident is not found.
}
function TGXML_Element.BoolAttributeByName( const Ident : String; DefaultValue : Boolean ) : Boolean;
var
  tmp : string;
begin
  tmp := AttributeByName( Ident, '' );
  if tmp = 'F' then
    Result := False
  else
    if tmp = 'T' then
      Result := True
    else
      Result := DefaultValue;
end;

{------------------------------------------------------------------------------
  TGXML_Element.FloatAttributeByName
------------------------------------------------------------------------------}
{**
  @Param Ident name of Attribute to find.
  @Param DefaultValue Value to use if Ident does not exist.
  @Result Value of Attribute or DefaultValue if the Ident is not found.
}
function TGXML_Element.FloatAttributeByName( const Ident : string; DefaultValue : Extended ) : Extended;
var
  sValue : string;
  SaveDecimalSeparator : Char;
begin
  sValue := AttributeByName( Ident, '' );
  if sValue = '' then
    Result := DefaultValue
  else
  begin
    SaveDecimalSeparator := DecimalSeparator;
    DecimalSeparator := '.';
    Result := StrToFloat( sValue );
    DecimalSeparator := SaveDecimalSeparator;
  end;
end;

{------------------------------------------------------------------------------
  TGXML_Element.IntAttributeByName
------------------------------------------------------------------------------}
{**
  @Param Ident name of Attribute to find.
  @Param DefaultValue Value to use if Ident does not exist.
  @Result Value of Attribute or DefaultValue if the Ident is not found.
}
function TGXML_Element.IntAttributeByName( const Ident : String; DefaultValue : integer ) : integer;
begin
  Result := StrToInt( AttributeByName( Ident, IntToStr( DefaultValue )));
end;

{------------------------------------------------------------------------------
  TGXML_Document.Create
------------------------------------------------------------------------------}
constructor TGXML_Document.Create;
begin
  inherited;
  FDocument := TGXML_Element.Create;
end;

{------------------------------------------------------------------------------
  TGXML_Document.Destroy
------------------------------------------------------------------------------}
destructor TGXML_Document.Destroy;
begin
  FDocument.Free;

  inherited;
end;

{------------------------------------------------------------------------------
  TGXML_Document.GetXMLString
------------------------------------------------------------------------------}
{**
  @Result Returns the document as a string.
}
function TGXML_Document.GetXMLString: string;

  function Attributes( el : TGXML_Element ) : string;
  var
    idx, iCount : integer;
  begin
    Result := ' ';
    iCount := 0;
    for idx := 0 to el.Count - 1 do
      if el.objects[idx] = el then
      begin
        Result := Result + el[idx] + ' ';
        Inc( iCount );
      end;

    if iCount = el.Count then
      Result := Result + '/';
  end;

  function WriteElement( el : TGXML_Element; iLevel : integer ) : string;
  var
    idx : integer;
    sAttr : string;
  begin
    Result := '';
    for idx := 0 to el.count - 1 do
      if el.objects[idx] <> el then
        if el.objects[idx] = nil then
          Result := Result + el[idx] + #13#10
        else
        begin
          sAttr := Attributes( TGXML_Element( el.objects[idx] ));
          Result := Result + StringOfChar( ' ', iLevel ) + '<' + Trim( el[idx] + sAttr ) + '>'#13#10;
          if sAttr[Length(sAttr)] <> '/' then
          begin
            Result := Result + WriteElement( TGXML_Element( el.objects[idx] ), iLevel + 1 );
            Result := Result + StringOfChar( ' ', iLevel ) + '</' + Trim( el[idx] ) + '>'#13#10;
          end;
        end;
  end;
begin
  Result := WriteElement( Document, 0 );
end;

{------------------------------------------------------------------------------
  TGXML_Document.LoadFromFile
------------------------------------------------------------------------------}
{**
  @Param sFilename Name of the file to read the XML from.
}
function TGXML_Document.LoadFromFile( const sFilename : string ) : Boolean;
var
  InF : TextFile;
  Line, XML : string;
begin
  XML := '';
  Result := FileExists(sFileName);
  
  if Result then
  begin
    AssignFile(InF, sFilename);
    Reset(InF);
    while not EOF( InF ) do
    begin
      ReadLn( InF, Line );
      XML := XML + Line;
    end;
    CloseFile( InF );
    Result := Document <> nil;
  end;
  AsString := XML;
end;

{------------------------------------------------------------------------------
  TGXML_Document.SaveToFile
------------------------------------------------------------------------------}
{**
  @Param sFilename Name of the file to store the XML into.
}
procedure TGXML_Document.SaveToFile( const sFilename : string );
var
  OutF : TextFile;
begin
  AssignFile(OutF, sFilename);
  Rewrite(OutF);
  Write( OutF, AsString );
  CloseFile( OutF );
end;

{------------------------------------------------------------------------------
  TGXML_Document.SetDocument
------------------------------------------------------------------------------}
{**
  @Param doc Document element to set.
}
procedure TGXML_Document.SetDocument(doc: TGXML_Element);
begin
  if doc <> FDocument then
  begin
    FDocument.Free;
    FDocument := doc
  end;
end;

{------------------------------------------------------------------------------
  TGXML_Document.SetXMLString
------------------------------------------------------------------------------}
{**
  @Param XML string to parse into the Document property.
}
procedure TGXML_Document.SetXMLString(const XML: string);
var
  iOffset, iEOS : integer;

  //---------------------------------------------------------------------------
  procedure ParseBody( el : TGXML_Element );
  var
    sBody : string;
  begin
    sBody := '';
    while ( XML[iOffset] <> '<' ) and ( iOffset <= iEOS ) do
    begin
      sBody := sBody + XML[iOffset];
      Inc( iOffset );
    end;
    Inc( iOffset );

    sBody := Trim( sBody );
    if sBody <> '' then
      el.Add( sBody );
  end;

  //---------------------------------------------------------------------------
  procedure ParseAttributes( el : TGXML_Element; sAttr : string );
  var
    iPos : integer;
    sIdent, sValue : string;
  begin
    iPos := Pos( '="', sAttr );
    while iPos > 0 do
    begin
      sIdent := Copy( sAttr, 1, iPos - 1 );
      sAttr := Copy( sAttr, iPos + 2, Length( sAttr ));
      iPos := Pos( '"', sAttr );
      sValue := Copy( sAttr, 1, iPos - 1 );
      sAttr := Copy( sAttr, iPos + 1, Length( sAttr ));

      el.AddAttribute( sIdent, sValue );
      iPos := Pos( '="', sAttr );
    end;
  end;

  //---------------------------------------------------------------------------
  function ParseTagIdent( var sIdent : string ) : Boolean;
  begin
    sIdent := '';
    while ( not ( XML[iOffset] in ['>', ' '])) and ( iOffset <= iEOS ) do
    begin
      sIdent := sIdent + XML[iOffset];
      Inc( iOffset );
    end;
    Result := ( sIdent = '' ) or ( sIdent[1] = '/' );
    if Result then
      Inc( iOffset );
  end;

  //---------------------------------------------------------------------------
  function ParseTagAttributes( el : TGXML_Element ) : Boolean;
  var
    sAttr : string;
  begin
    sAttr := '';
    while ( not ( XML[iOffset] in ['>'] )) and ( iOffset <= iEOS) do
    begin
      sAttr := sAttr + XML[iOffset];
      Inc( iOffset );
    end;

    Result := False;
    sAttr := Trim( sAttr );
    Inc( iOffset );
    if sAttr <> '' then
    begin
      ParseAttributes( el, sAttr );
      Result := sAttr[Length(sAttr)] = '/';
    end;
  end;

  //---------------------------------------------------------------------------
  procedure ParseElement( el : TGXML_Element );
  var
    sIdent : string;
    sub : TGXML_Element;
  begin
    while iOffset <= iEOS do
    begin
      ParseBody( el );
      if ParseTagIdent( sIdent ) then
        Break;

      sub := TGXML_Element.Create;
      el.AddElement( sIdent, sub );

      if not ParseTagAttributes( sub ) then
        ParseElement( sub );
    end;
  end;

begin
  iOffset := 1;
  iEOS := Length( XML );
  FDocument.Free;
  FDocument := TGXML_Element.Create;
  ParseElement(FDocument);
end;

end.
