{-------------------------------------------------------------------------
 Module:    TGlobe Text file reader

 Comment:   Buffers and reads a text file line by line

 Classes:   TTextReader

 Author:    Graham Knight
 Email:     tglobe@iname.com
-------------------------------------------------------------------------}
unit TGTextReader;

interface

uses
  SysUtils, Classes, TGSysUtils, TGMMStream;

const
  DEFAULT_BUF_SIZE = 4096;

type
  TTextReader = class(TGlobeRoot)
  private
    FbEOT : Boolean;
    FTextBufferSize : integer;
    DataStream: TStream;
    BufEnd, BufPos: integer;
    FiPosition : integer;

    function GetSize : integer;
    procedure SetPosition(iPosition: integer);
    procedure SetTextBufferSize( Value : integer );
  public
    TextBuffer: PChar;
    constructor Create(const FileName: string);
    destructor Destroy; override;

    procedure ReadLn(var Line: string);
    property EOT: Boolean read FbEOT;
    property Size : integer read GetSize;
    property Position: integer read FiPosition write SetPosition;
    property TextBufferSize : integer read FTextBufferSize write SetTextBufferSize;
  end;

implementation

{------------------------------------------------------------------------------
  TTextReader.Create
------------------------------------------------------------------------------}
{**
  @Param Filename Name of the file to read from.
} 
constructor TTextReader.Create(const FileName: string);
begin
  inherited Create;

  DataStream := TMemoryMapStream.Create(FileName);
//  DataStream := TFileStream.Create(FileName, fmOpenRead or fmShareDenyNone	);

  SetTextBufferSize( DEFAULT_BUF_SIZE );
end;

{------------------------------------------------------------------------------
  TTextReader.SetTextBufferSize
------------------------------------------------------------------------------}
{**
  @Param Value Size of the buffer to use when reading from the Stream.
} 
procedure TTextReader.SetTextBufferSize(Value: integer);
begin
  if FTextBufferSize <> Value then
  begin
    FTextBufferSize := Value;

    FreeMem( TextBuffer );
    TextBuffer := AllocMem( FTextBufferSize + 1 );
    FiPosition := MaxInt;
    SetPosition( 0 );
  end;
end;

{------------------------------------------------------------------------------
  TTextReader.Destroy
------------------------------------------------------------------------------}
destructor TTextReader.Destroy;
begin
  DataStream.Free;
  DataStream := nil;

  FreeMem( TextBuffer );

  inherited Destroy;
end;

{------------------------------------------------------------------------------
  TTextReader.GetSize
------------------------------------------------------------------------------}
{**
  @Result The size of the input stream.
} 
function TTextReader.GetSize : integer;
begin
  Result := DataStream.Size;
end;

{------------------------------------------------------------------------------
  TTextReader.SetPosition
------------------------------------------------------------------------------}
{**
  @Param iPosition Position in the stream to start reading from.
} 
procedure TTextReader.SetPosition(iPosition: integer);
begin
  if ( FiPosition div FTextBufferSize ) <> ( iPosition div FTextBufferSize ) then
  begin
    DataStream.Position := ( iPosition div FTextBufferSize ) * FTextBufferSize;

    BufEnd := DataStream.Read(TextBuffer^, FTextBufferSize);
  end;

  BufPos := iPosition mod FTextBufferSize;
  FbEOT := BufEnd < BufPos;
  FiPosition := iPosition;
end;

{------------------------------------------------------------------------------
  TTextReader.ReadLn
------------------------------------------------------------------------------}
{**
  @Param Line Var to store a line from the input stream.
} 
procedure TTextReader.ReadLn(var Line: string);
var
  P, Start: PChar;
  S: string;
begin
  Line := '';

  repeat
    P := TextBuffer;
    Inc(P, BufPos);
    Start := P;

    while not (P^ in [#0, #10, #13]) do
      Inc(P);

    SetString(S, Start, P - Start);

    if P^ = #13 then
      Inc(P);
    Inc( FiPosition, P - Start );

    if P^ <> #0 then // if we have not hit the end of the Buffer
      Break;

    Line := Line + S;
    BufEnd := DataStream.Read(TextBuffer^, FTextBufferSize);
    TextBuffer[BufEnd] := #0;
    FbEOT := BufEnd = 0;
    BufPos := 0;
  until FbEOT;

  if P^ = #10 then
  begin
//    Inc(P);
    Position := Position + 1;
  end;
//  Inc( BufPos, P - Start );

  Line := Line + S;
end;

end.

