{------------------------------------------------------------------------------
 Module:		TGSHPMapper.pas

 Comment:   Reads ESRI .SHP file

 Classes:   TGlobeSHPReader

 Author:	  Graham Knight
 Email:		  tglobe@iname.com
------------------------------------------------------------------------------}
unit TGSHPMapper;

interface

uses
  Windows, SysUtils, Classes, TGDBFReader, Globe4, TGSysUtils, TGObjects,
  TGClasses, TGXML, TGMMStream;

const
  TG_SHP_READERVERSION = $0102;

type
  TSHPItem = record
    iOffset : integer;
    iDBFRecordID : integer;
  end;
  TSHPItemPtr = ^TSHPItem;

  {---------------------------- TGlobeSHPReader -------------------------------}
  TGlobeSHPReader = class(TGlobeFileReader)
  private
    FLongXDivisor : Double;
    FLatYDivisor : Double;
    FLongXShift : Double;
    FLatYShift : Double;
    FTitleColumn : integer;
    FDataItems : DynArray;
    procedure SetTitleColumn( Value : integer );
  protected
    DBF : TGlobeDBFReader;
    SHPStream : TStream;

    procedure InternalOpen; override;
    procedure InternalClose; override;
    procedure MapSHPFile;

    function CorrectX( X: Extended ): Extended;
    function CorrectY( Y: Extended ): Extended;

    function ReadBigEndianInt : integer;
    function ReadLittleEndianInt : integer;
    procedure ReadSHPHeader;
    function ReadPolyObject( bPolygon : Boolean ) : TGeoDataObject;
    function ReadMultiPointObject : TGeoDataObject;
  public
    procedure LoadEnvironment( Element : TGXML_Element ); override;
    function SaveEnvironment : TGXML_Element; override;
    procedure SaveMetaData; override;
    procedure LoadMetaData; override;

    function LoadObject(iIndex : integer; bNewObject : Boolean) : TGlobeObject; override;
  published
    property LongXDivisor : Double read FLongXDivisor write FLongXDivisor;
    property LatYDivisor : Double read FLatYDivisor write FLatYDivisor;
    property LongXShift : Double read FLongXShift write FLongXShift;
    property LatYShift : Double read FLatYShift write FLatYShift;
    property TitleColumn : integer read FTitleColumn write SetTitleColumn;
  end;

implementation

Uses TGResource;

{-------------------------------------------------------------------------}
procedure SwapWord( iLength : integer; wordP : PChar );
var
	idx : integer;
	cTmp : char;
begin
	for idx := 0 to iLength div 2 - 1 do
	begin
		cTmp := wordP[idx];
		wordP[idx] := wordP[iLength - idx - 1];
		wordP[iLength - idx - 1] := cTmp;
	end;
end;

{-------------------------------------------------------------------------}
procedure TGlobeSHPReader.SetTitleColumn( Value : integer );
begin
  if Value <> FTitleColumn then
  begin
    FTitleColumn := Value;
    UnloadObjects;
  end;
end;

{-------------------------------------------------------------------------}
function TGlobeSHPReader.CorrectX( X: Extended ): Extended;
begin
  Result := LongXShift + X / LongXDivisor;
end; // CorrectX

{-------------------------------------------------------------------------}
function TGlobeSHPReader.CorrectY( Y: Extended ): Extended;
begin
  Result := LatYShift + Y / LatYDivisor;
end; // CorrectY

{-------------------------------------------------------------------------}
function TGlobeSHPReader.ReadBigEndianInt : integer;
begin
	SHPStream.Read( Result, SizeOf( integer ));
	SwapWord( 4, PChar( @Result ));
end;

{-------------------------------------------------------------------------}
function TGlobeSHPReader.ReadLittleEndianInt : integer;
begin
	SHPStream.Read( Result, SizeOf( integer ));
end;

{-------------------------------------------------------------------------}
procedure TGlobeSHPReader.ReadSHPHeader;
var
	idx : integer;
  eTmp : Double;
begin
  SHPStream.Position := 0;
	if ReadBigEndianInt <> 9994 then
    raise EGlobeException.CreateFmt( 'Invalid SHP file ', [Filename] );

  for idx := 0 to 4 do
    ReadLittleEndianInt;

  ReadBigEndianInt; // Shape file size

  if ReadLittleEndianInt <> 1000 then	{ Version }
    raise EGlobeException.CreateFmt( 'Invalid SHP file version ', [Filename] );

  ReadLittleEndianInt;	{ Shape File Type }

  for idx := 0 to 3 do  // read the bounding box
    SHPStream.Read( eTmp, SizeOf( Double ));

  for idx := 0 to 7 do
    ReadLittleEndianInt;
end;


{-------------------------------------------------------------------------}
function TGlobeSHPReader.ReadPolyObject( bPolygon : Boolean ) : TGeoDataObject;
var
	idx, iParts, iChain : integer;
	eTmp, eX, eY : double;
	iPoints : integer;
  offsets : DynArray;
begin
  Result := TGeoDataObject.Create( Self );

	for idx := 0 to 3 do
		SHPStream.Read( eTmp, SizeOf( double ));	{ read the box for the object }

	iParts := ReadLittleEndianInt;	{ Number of parts }
	iPoints := ReadLittleEndianInt;	{ Number of points }

  offsets := DynArrayCreate( SizeOf( integer ), iParts + 1 );
  DynArraySetAsInteger( Offsets, iParts, iPoints );

  // Read all the offsets
  for idx := 0 to iParts - 1 do
    DynArraySetAsInteger( Offsets, idx, ReadLittleEndianInt );

  Result.Chains.Count := iParts;
	for idx := 0 to iParts - 1  do
    Result.Chains[idx].Count := DynArrayAsInteger( Offsets, idx+1 ) - DynArrayAsInteger( Offsets, idx );

  iChain := 0;
	for idx := 0 to iPoints - 1 do
	begin
    if idx = DynArrayAsInteger( Offsets, iChain + 1) then
      Inc( iChain );
		SHPStream.Read( eX, SizeOf( double ));	{ X }
		SHPStream.Read( eY, SizeOf( double ));	{ Y }
    Result.Chains[iChain].AsLL[idx - DynArrayAsInteger( Offsets, iChain )] :=
      DecimalToPointLL( CorrectX( eX ), CorrectY( eY ));
  end;

  Result.Closed := bPolygon;

  DynArrayFree( offsets );
end;

{-------------------------------------------------------------------------}
function TGlobeSHPReader.ReadMultiPointObject : TGeoDataObject;
var
	idx, iPoints : integer;
	eTmp, eX, eY : double;
begin
	for idx := 0 to 3 do
		SHPStream.Read( eTmp, SizeOf( double ));	{ read the box for the object }
	iPoints := ReadLittleEndianInt;	{ Number of points }

  Result := TGeoDataObject.Create( Self );

	for idx := 1 to iPoints do
	begin
		SHPStream.Read( eX, SizeOf( Double ));	{ X }
		SHPStream.Read( eY, SizeOf( Double ));	{ Y }
    Result.Chains[0].Add( DecimalToPointLL( CorrectX( eX ), CorrectY( eY )));
	end;
end;

{------------------------------------------------------------------------------
 TGlobeSHPReader.LoadObject
------------------------------------------------------------------------------}
function TGlobeSHPReader.LoadObject( iIndex : integer; bNewObject : Boolean ) : TGlobeObject;
var
  eX,eY : Double;
  NewObject : TGeoDataObject;
begin
  Result := nil;
  with TSHPItemPtr( DynArrayPtr( FDataItems, iIndex))^ do
  begin
    SHPStream.Position := iOffset;
    if DBF <> nil then
      DBF.RecordID := iDBFRecordID;

		ReadBigEndianInt;	{ record number }
    ReadBigEndianInt; { content length }

		case ReadLittleEndianInt of	{ ShapeType }
		1 :	{ Point }
			begin
				SHPStream.Read( eX, SizeOf( double ));	{ X }
				SHPStream.Read( eY, SizeOf( double ));	{ Y }
				NewObject := TGeoDataObject.Create( Self );
				NewObject.Centroid := DecimaltoPointLL( CorrectX( eX ), CorrectY( eY ));
			end;
		8 : { MultiPoint }
			NewObject := ReadMultiPointObject;
		3 :	{ Arc }
			NewObject := ReadPolyObject( False );
		5 :	{ Polygon }
			NewObject := ReadPolyObject( True );
		else
			Exit;
		end;

    if DBF <> nil then
    begin
      NewObject.Title := DBF.AsString[TitleColumn];
      NewObject.ID := iDBFRecordID;
    end;
	end;

  Result := NewObject;
end;

{------------------------------------------------------------------------------
  TGlobeSHPReader.SaveMetaData
------------------------------------------------------------------------------}
{**
  @Param sFilename Name of file to save meta data to.
}
procedure TGlobeSHPReader.SaveMetaData;
var
  idx : integer;
  Writer : TGlobeStreamWriter;
  sMetaDataName : TFilename;
begin
  sMetaDataName := MetaDataFilename( Filename, True );
  if sMetaDataName = '' then
    Exit;

  if not FileExists( sMetaDataName ) then
  begin
    Writer := TGlobeStreamWriter.Create( TMemoryStream.Create );
    try
      Writer.WriteWord( TG_SHP_READERVERSION );

      WriteMetaData( Writer );

      Writer.WriteDouble( LongXDivisor );
      Writer.WriteDouble( LatYDivisor );
      Writer.WriteDouble( LongXShift );
      Writer.WriteDouble( LatYShift );
      Writer.WriteInteger( TitleColumn );

      Writer.WriteInteger( FDataItems.Count ); // save the count of items

      // save the class data.
      for idx := 0 to FDataItems.Count - 1 do
      begin
        Writer.WriteInteger( TSHPItemPtr( DynArrayPtr( FDataItems, idx))^.iOffset );
        Writer.WriteInteger( TSHPItemPtr( DynArrayPtr( FDataItems, idx))^.iDBFRecordID );
      end;

      Writer.WriteShortString( ExtractFilename( Filename ));
      TMemoryStream( Writer.DataStream ).SaveToFile( sMetaDataName );
    finally
      Writer.Free;
    end;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeSHPReader.LoadMetaData
------------------------------------------------------------------------------}
{**
  @Param sFilename Name of file to save meta data to.
}
procedure TGlobeSHPReader.LoadMetaData;
var
  idx, iCount : Integer;
  Reader : TGlobeStreamReader;
  sMetaDataName : TFilename;
begin
  sMetaDataName := MetaDataFilename( Filename, False );
  if not FileExists( sMetaDataName ) then
    Exit;

  // Check the metadata file is older than the data file
  if FileAge( sMetaDataName ) < FileAge( Filename ) then
    Exit;

  Reader := TGlobeStreamReader.Create( TMemoryStream.Create );
  try
    TMemoryStream( Reader.DataStream ).LoadFromFile( sMetaDataName );

    if Reader.ReadWord <> TG_SHP_READERVERSION then
    begin
      DeleteFile( sMetaDataName );
      Exit;
    end;

    ReadMetaData( Reader );

    LongXDivisor := Reader.ReadDouble;
    LatYDivisor := Reader.ReadDouble;
    LongXShift := Reader.ReadDouble;
    LatYShift := Reader.ReadDouble;
    FTitleColumn := Reader.ReadInteger;

    DynArrayFree( FDataItems );  // clear the data item list

    // get the number of items for this class
    iCount := Reader.ReadInteger;

    // set the length of the Reader Item array
    FDataItems := DynArrayCreate( SizeOf( TSHPItem ), iCount );

    // load in the Reader data
    for idx := 0 to iCount - 1 do
    begin
      TSHPItemPtr( DynArrayPtr( FDataItems, idx))^.iOffset := Reader.ReadInteger;
      TSHPItemPtr( DynArrayPtr( FDataItems, idx))^.iDBFRecordID := Reader.ReadInteger;
    end;

    Filename := Globe.ResolveFilename( Reader.ReadShortString ); // get the data file name
  finally
    Reader.Free;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeSHPReader.InternalOpen
------------------------------------------------------------------------------}
procedure TGlobeSHPReader.InternalOpen;
begin
  Name := ExtractFilename( Filename );

  LongXDivisor := 1;
  LatYDivisor := 1;
  LongXShift := 0;
  LatYShift := 0;

  // Create the data stream
  if FileExists( Filename ) then
  begin
    SHPStream := TMemoryMapStream.Create( Filename );

//    SHPStream := TFileStream.Create( Filename, fmOpenRead or fmShareDenyWrite );

    if FileExists( ChangeFileExt( Filename, '.DBF' )) then
      DBF := TGlobeDBFReader.Create(ChangeFileExt( Filename, '.DBF' ));

    if FDataItems = nil then
      MapSHPFile;

    inherited;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeSHPReader.InternalClose
------------------------------------------------------------------------------}
procedure TGlobeSHPReader.InternalClose;
begin
  Count := 0;
  DynArrayFree( FDataItems ); // clear the data item list

  DBF.Free;

  SHPStream.Free;
  SHPStream := nil;

  inherited;
end;

{------------------------------------------------------------------------------
 TGlobeSHPReader.MapSHPFile
------------------------------------------------------------------------------}
procedure TGlobeSHPReader.MapSHPFile;
var
  iFileSize, iItems : integer;
begin
  Globe.ProgressMessage( pmStart, rsMapping);

  FDataItems := DynArrayCreate( Sizeof( TSHPItem ), 0 );
  Count := 0;

  iFileSize := SHPStream.Size;

  ReadSHPHeader;

  if DBF <> nil then
    DBF.First;
  iItems := 0;
	while ( SHPStream.Position < iFileSize ) do
	begin
    Inc( iItems );
    if iItems >= FDataItems.Count then
      DynArraySetLength(FDataItems, iItems + 64);

    with TSHPItemPtr( DynArrayPtr( FDataItems, iItems - 1))^ do
    begin
      iOffset := SHPStream.Position;
      if DBF <> nil then
        iDBFRecordID := DBF.RecordID;
    end;

		ReadBigEndianInt;	{ record number }

		SHPStream.Position := SHPStream.Position + ( 2 + ReadBigEndianInt ) * 2;	{ record length }
    if DBF <> nil then
      DBF.Next;

    if ( iItems mod 256 ) = 0 then
      Globe.ProgressMessage( pmPercent, IntToStr( MulDiv( SHPStream.Position, 100, iFileSize )));
  end;

  DynArraySetLength( FDataItems, iItems );

  Count := FDataItems.Count;
  Globe.ProgressMessage( pmEnd, rsFinished);
end;

{------------------------------------------------------------------------------
  TGlobeSHPReader.SaveEnvironment
------------------------------------------------------------------------------}
{**
  @Result Element containing the SHPReader environment.
}
function TGlobeSHPReader.SaveEnvironment : TGXML_Element;
begin
  Result := inherited SaveEnvironment;

  Result.AddAttribute( 'Filename', Filename);
  Result.AddFloatAttribute( 'LongXDivisor', LongXDivisor);
  Result.AddFloatAttribute( 'LatYDivisor', LatYDivisor);
  Result.AddFloatAttribute( 'LongXShift', LongXShift);
  Result.AddFloatAttribute( 'LatYShift', LatYShift);
  Result.AddIntAttribute( 'TitleColumn', TitleColumn );
end;

{------------------------------------------------------------------------------
  TGlobeSHPReader.LoadEnvironment
------------------------------------------------------------------------------}
{**
  @Param Element containing the SHPReader environment to load.
}
procedure TGlobeSHPReader.LoadEnvironment( Element : TGXML_Element );
begin
  if Element <> nil then
    with Element do
    begin
      Filename := AttributeByName( 'Filename', Filename );
      LongXDivisor := FloatAttributeByName( 'LongXDivisor', LongXDivisor);
      LatYDivisor := FloatAttributeByName( 'LatYDivisor', LatYDivisor);
      LongXShift := FloatAttributeByName( 'LongXShift', LongXShift);
      LatYShift := FloatAttributeByName( 'LatYShift', LatYShift);
      TitleColumn := IntAttributeByName( 'TitleColumn', TitleColumn );
    end;

  inherited LoadEnvironment( Element );
end;

initialization
  RegisterClasses( [TGlobeSHPReader] );
end.
