{------------------------------------------------------------------------------
 Module:		MIFMapper.pas

 Comment:   Reads MapInfo .MIF file

 Classes:   TGlobeMIFReader

 Author:	  Graham Knight
 Email:		  tglobe@iname.com
------------------------------------------------------------------------------}
unit TGMIFMapper;

interface

uses
  Windows, Forms, Graphics, Classes, SysUtils, Globe4, TGSysUtils,
  TGObjects, TGPresenters, TGTextReader, TGClasses, TGXML;

const
  TG_MIF_READERVERSION = $0102;

type
  TMIFMIDItem = record
    iMIDOffset : integer;
    iMIFOffset : integer;
  end;
  TMIFItemPtr = ^TMIFMIDItem;

  {---------------------------- TGlobeMIFReader -------------------------------}
  TGlobeMIFReader = class(TGlobeFileReader)
  private
    FMIFReader : TTextReader;
    FMIDReader : TTextReader;
    FDataItems : DynArray;
    FTitleColumn : integer;
    FMIFCoordSys : string;
    FMIDDelimiter : string;
    FMIDColumnNames : TStringList;
  protected
    iMIFVersion : integer;
    CurrentMIFLine : string;
    CurrentMIDLine : string;

    aPen : TGlobePen;
    aBrush : TGlobeBrush;
    aFont : TGlobeFont;
    aTitleFont : TGlobeFont;

    procedure InternalOpen; override;
    procedure InternalClose; override;
    procedure MapMIFFile;

    function MIF2DelphiBrush(iStyle : Integer) : TBrushStyle;
    function MIF2DelphiPen(iStyle : Integer) : TPenStyle;
    function MIF2DelphiColor(iColor : Integer) : Integer;
    function ReadLLPointFromLine : TPointLL;
    function ReadLLPoint(iLongPos, iLatPos : integer) : TPointLL;
    function ReadPenObject : Boolean;
    function ReadBrushObject : Boolean;
    function ReadFontObject : Boolean;
    function ReadLineObject(bNewObject : Boolean) : TGeoDataObject;
    function ReadPLineObject(bNewObject : Boolean) : TGeoDataObject;
    function ReadPointObject(bNewObject : Boolean) : TGeoDataObject;
    function ReadRegionObject(bNewObject : Boolean) : TGeoDataObject;
    function ReadTextObject(bNewObject : Boolean) : TGeoDataObject;

    function ReadMIFLine : Boolean;
    function ReadMIDLine : Boolean;
    procedure ReadMIFHeader;

    function GetMIFWord(iIndex : integer) : string;
    function GetMIDWord(iIndex : integer) : string;
    function ReplaceStr(const sOrigin, sToReplace, sReplacer : string) : string;
    function StrSplitToList(const sToSplit, sSeparator : string;
      var tsStrList : TStrings) : Integer;
  public
    procedure LoadEnvironment( Element : TGXML_Element ); override;
    function SaveEnvironment : TGXML_Element; override;
    procedure SaveMetaData; override;
    procedure LoadMetaData; override;

    function LoadObject(iIndex : integer; bNewObject : Boolean) : TGlobeObject; override;
  published
    property MIFCoordSys : string read FMIFCoordSys write FMIFCoordSys;
    property MIDDelimiter : string read FMIDDelimiter write FMIDDelimiter;
    property MIDColumnNames : TStringList read FMIDColumnNames write FMIDColumnNames;
    property TitleColumn : integer read FTitleColumn write FTitleColumn;
  end;

implementation

Uses TGResource;

{------------------------------------------------------------------------------
  TGlobeMIFReader.GetMIFWord
------------------------------------------------------------------------------}
function TGlobeMIFReader.GetMIFWord(iIndex : integer) : string;
var
  iLen : integer;
  P, S : PChar;
  bSpace : Boolean;
begin
  iLen := Length(CurrentMIFLine);

  if iLen = 0 then
  begin
    Result := '';
    Exit;
  end;
  P := Pointer(@CurrentMIFLine[1]);

  bSpace := True;
  repeat
    while ( iLen > 0 ) and ( bSpace = (P^ in [#9, ' '])) do
    begin
      Inc(P);
      Dec(iLen);
    end;
    bSpace := not bSpace;
    if not bSpace then
      Dec( iIndex );
  until (iLen = 0) or (iIndex < 0);

  S := P;
  while ( iLen > 0 ) and not (S^ in [#9, ' ']) do
  begin
    Inc(S);
    Dec(iLen);
  end;

  SetString(Result, P, S - P);
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.GetMIDWord
------------------------------------------------------------------------------}
function TGlobeMIFReader.GetMIDWord(iIndex : integer) : string;
begin
//ToDo: extract the right bit of the line using MIDDelimiter
  Result := CurrentMIDLine;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.StrSplitToList
------------------------------------------------------------------------------}
function TGlobeMIFReader.StrSplitToList(const sToSplit, sSeparator : string;
  var tsStrList : TStrings) : Integer;
var
  iCurPos, iNoStrings : Integer;
  sTmpRet, sTmpStr : string;
begin
  if tsStrList = nil then
    tsStrList := TStringList.Create;

  sTmpRet := Copy(sToSplit, 1, Length(sToSplit));
  iCurPos := Pos(sSeparator, sToSplit);
  tsStrList.Clear;
  iNoStrings := 0;
  if iCurPos > 0 then
  begin
    while iCurPos > 0 do
    begin
      sTmpStr := Copy(sTmpRet, 0, iCurPos - 1);
      tsStrList.Add(sTmpStr);
      Inc(iNoStrings, 1);
      sTmpRet := Copy(sTmpRet, iCurPos + Length(sSeparator), Length(sTmpRet));
      iCurPos := Pos(sSeparator, sTmpRet);
    end;
    if Length(sTmpRet) > 0 then
    begin
      tsStrList.Add(sTmpRet);
      Inc(iNoStrings, 1);
    end;
  end
  else
  begin
    tsStrList.Add(sTmpRet);
    Inc(iNoStrings, 1);
  end;
  Result := iNoStrings;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReplaceStr
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReplaceStr(const sOrigin, sToReplace, sReplacer : string) : string;
{ ex: sNewString := ReplaceStr('Hej hopp', ' ', '_');
  results in sNewString := 'Hej_hopp');}
var
  sToReturn : string;
  iCurPos : Integer;
  sTmpRet, sTmpStr : string;
begin
  iCurPos := Pos(sToReplace, sOrigin);
  if iCurPos > 0 then
  begin
    sTmpRet := Copy(sOrigin, 1, Length(sOrigin));
    sToReturn := '';
    while iCurPos > 0 do
    begin
      sTmpStr := Copy(sTmpRet, 1, iCurPos - 1);
      sToReturn := sToReturn + sTmpStr + sReplacer;
      sTmpRet := Copy(sTmpRet, iCurPos + Length(sToReplace), Length(sTmpRet));
      iCurPos := Pos(sToReplace, sTmpRet);
    end;
    sToReturn := sToReturn + sTmpRet;
    Result := sToReturn;
  end
  else
    Result := sOrigin;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.MIF2DelphiBrush
------------------------------------------------------------------------------}
function TGlobeMIFReader.MIF2DelphiBrush(iStyle : Integer) : TBrushStyle;
begin
  case iStyle of
    1 : Result := bsClear;
    3, 19..23 : Result := bsHorizontal;
    4, 24..28 : Result := bsVertical;
    5, 29..33 : Result := bsBDiagonal;
    6, 34..38 : Result := bsFDiagonal;
    7, 39..43, 56..60 : Result := bsCross;
    8, 44..52, 55, 61..70 : Result := bsDiagCross;
  else
    Result := bsSolid;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.MIF2DelphiPen
------------------------------------------------------------------------------}
function TGlobeMIFReader.MIF2DelphiPen(iStyle : Integer) : TPenStyle;
begin
  case iStyle of
    1 : Result := psClear;
    3..9 : Result := psDash;
    10..13 : Result := psDot;
    14..16 : Result := psDashDot;
    20..25 : Result := psDashDotDot;
    71..77 : Result := psInsideFrame;
  else
    Result := psSolid;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.MIF2DelphiColor
------------------------------------------------------------------------------}
function TGlobeMIFReader.MIF2DelphiColor(iColor : Integer) : Integer;
var
  sTmp : string;
begin
  sTmp := IntToHex(iColor, 6);
  Result := StrToInt('$' + Copy(sTmp, 5, 2) + Copy(sTmp, 3, 2) + Copy(sTmp, 1, 2));
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadMIFLine
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadMIFLine : Boolean;
begin
  repeat
    FMIFReader.ReadLn(CurrentMIFLine);
    Result := Length(CurrentMIFLine) > 0;
  until FMIFReader.EOT or Result;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadMIDLine
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadMIDLine : Boolean;
begin
  Result := ( FMIDReader <> nil ) and not FMIDReader.EOT;
  if Result then
    FMIDReader.ReadLn(CurrentMIDLine);
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadLLPoint
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadLLPoint(iLongPos, iLatPos : integer) : TPointLL;
var
  Long, Lat : Double;
begin
  Long := StrToExtended(GetMIFWord(iLongPos));
  Lat := StrToExtended(GetMIFWord(iLatPos));

{ToDo:  if Assigned( OnProjectCoords ) then
    OnProjectCoords( Self, Long, Lat );}

  Result := DecimalToPointLL(Long, Lat);
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadLLPointFromLine
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadLLPointFromLine : TPointLL;
begin
  ReadMIFLine;
  Result := ReadLLPoint( 0, 1 );
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadPointObject
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadPointObject(bNewObject : Boolean) : TGeoDataObject;
var
  ptLL : TPointLL;
  aPresenter : TPointPresenter;
begin
  ptLL := ReadLLPoint( 1, 2 );

  Result := TGeoDataObject.Create( Self );
  Result.Centroid := ptLL;
  Result.Title := GetMIDWord( TitleColumn );

  if bNewObject then
  begin
    Result.PresenterID := TPointPresenter.FindMatch( Presenters,
      aPen, aBrush, aFont, nil, ppCircle, Pixel, 6, '');

    if Result.PresenterID = 0 then
    begin
      aPresenter := TPointPresenter.Create(Globe, 0);
      aPresenter.PointPen.Assign( aPen );
      aPresenter.PointBrush.Assign( aBrush );
      aPresenter.PointFont.Assign( aFont );
      aPresenter.TitleFont.Assign( aTitleFont );
      Result.PresenterID := Presenters.Add( aPresenter );
    end;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadPenObject
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadPenObject : Boolean;
var
  sTmp : string;
  slPen : TStrings;
begin
  Result := CompareText(GetMIFWord(0), 'Pen') = 0;

  if not Result then
    Exit;

  sTmp := GetMIFWord(1);
  slPen := TStringList.Create;
  StrSplitToList(Copy(sTmp, 2, Length(sTmp) - 2), ',', slPen);

  try
    aPen.Define(
      MIF2DelphiColor(StrToInt(slPen[2])),
      MIF2DelphiPen(StrToInt(slPen[1])),
      StrToIntDef(slPen[0], 1), Pixel);
  finally
    slPen.Free;
  end;

  ReadMIFLine;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadBrushObject
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadBrushObject : Boolean;
var
  sTmp : string;
  slBrush : TStrings;
begin
  Result := CompareText(GetMIFWord(0), 'Brush') = 0;

  if not Result then
    Exit;

  sTmp := GetMIFWord(1);
  slBrush := TStringList.Create;
  StrSplitToList(Copy(sTmp, 2, Length(sTmp) - 2), ',', slBrush);
  with aBrush do
  try
    if (StrToInt(slBrush[0]) > 3) and (slBrush.Count > 2) then
      BrushColor := MIF2DelphiColor(StrToInt(slBrush[2]))
    else
      BrushColor := MIF2DelphiColor(StrToInt(slBrush[1]));
    BrushStyle := MIF2DelphiBrush(StrToInt(slBrush[0]));
  finally
    slBrush.Free;
  end;

  ReadMIFLine;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadFontObject
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadFontObject : Boolean;
var
  sTmp : string;
  slFont : TStrings;
begin
  Result := CompareText(GetMIFWord(0), 'Font') = 0;

  if not Result then
    Exit;

  slFont := TStringList.Create;
	sTmp:= Trim(CurrentMIFLine);
	StrSplitToList(Copy(sTmp, 7, Length(sTmp) - 7), ',', slFont);

  try
    with Globe.GlobeCanvas do
    begin
      aTitleFont.FontName := ReplaceStr(slFont[0], '"', '');
      aTitleFont.FontStyle := [];
      aTitleFont.FontSize := StrToInt(slFont[2]);
      aTitleFont.FontColor := MIF2DelphiColor(StrToInt(slFont[3]));
      aTitleFont.FontAngle := 0;
    end;

    while ReadMIFLine do
    begin
      if CompareText(slFont[0], 'Angle') = 0 then
      begin
        aTitleFont.FontAngle := Round(StrToExtended(slFont[1])) * 10;
        continue;
      end;
      if CompareText(slFont[0], 'Spacing') = 0 then
        continue;
      if CompareText(slFont[0], 'justify') = 0 then
        continue;
      if CompareText(slFont[0], 'Label') = 0 then
        continue;
      break;
    end;
  finally
    slFont.free;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadRegionObject
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadRegionObject(bNewObject : Boolean) : TGeoDataObject;
var
  idx, iParts, iPoints : integer;
  aPresenter : TPolyPresenter;
begin
  Result := TGeoDataObject.Create( Self );
  Result.Closed := True;

  Result.Chains.Count := StrToInt(GetMIFWord(1)); { get number of parts to region }
  for iParts := 0 to Result.Chains.Count - 1 do
  begin
    ReadMIFLine;
    iPoints := StrToInt(GetMIFWord(0)); { get number of points in this region }
    for idx := 1 to iPoints do
      Result.Chains[iParts].Add(ReadLLPointFromLine);
  end;

  if ReadMIDLine then
    Result.Title := GetMIDWord(0)
  else
    Result.Title := 'New Polygon';

  if bNewObject then
  begin
    { read attributes }
    ReadMIFLine;
    while True do
      if not ReadPenObject then
        if not ReadBrushObject then
          if CompareText(GetMIFWord(0), 'Center') <> 0 then
            break
          else
            if not ReadMIFLine then
              break;

    Result.PresenterID := TPolyPresenter.FindMatch( Presenters, aPen, aBrush, nil);

    if Result.PresenterID = 0 then
    begin
      aPresenter := TPolyPresenter.Create( Globe, 0 );
      aPresenter.PolyPen.Assign( aPen );
      aPresenter.PolyBrush.Assign( aBrush );
      Result.PresenterID := Presenters.Add( aPresenter );
    end;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadPLineObject
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadPLineObject(bNewObject : Boolean) : TGeoDataObject;
var
  idx, iParts, iPoints : integer;
  sTmp : string;
  aPresenter : TPolyPresenter;
  MultipleParts: Boolean;
begin
  Result := TGeoDataObject.Create( Self );

  MultipleParts:= False;
  iPoints := 0;
  Result.Chains.Count := 1;
  case iMIFVersion of
    300 :
      begin
        sTmp := GetMIFWord(1);
        if sTmp <> '' then
        begin
          if CompareText(sTmp, 'Multiple') = 0 then
          begin
            Result.Chains.Count := StrToInt(GetMIFWord(2)); { get number of parts to region }
//            ReadMIFLine;
//            iPoints := StrToInt(GetMIFWord(0)); { get number of points in this region }
						MultipleParts:= True; { region has multiple parts }
          end
          else
            iPoints := StrToInt(GetMIFWord(1)); { get number of points in this region }
        end
        else
        begin
          ReadMIFLine;
          iPoints := StrToInt(GetMIFWord(0)); { get number of points in this region }
        end;
      end;
    1 :
      iPoints := StrToInt(GetMIFWord(1)); { get number of points in this region }
  else
    Result.Chains.Count := StrToInt(GetMIFWord(1)); { get number of parts to region }
    MultipleParts:= True; { region has multiple parts }
//    ReadMIFLine;
//    iPoints := StrToInt(GetMIFWord(0)); { get number of points in this region }
  end;

  for iParts := 0 to Result.Chains.Count - 1 do
  begin
  	if MultipleParts then
    begin
	    ReadMIFLine;
  	  iPoints := StrToInt(GetMIFWord(0)); { get number of points in this region }
    end;
    
    for idx := 1 to iPoints do
      Result.Chains[iParts].Add(ReadLLPointFromLine);
  end;

  Result.Title := 'New Polyline';

  if bNewObject then
  begin
    { read attributes }
    ReadMIFLine;
    while True do
      if not ReadPenObject then
        if CompareText(GetMIFWord(0), 'Smooth') <> 0 then
          break
        else
          if not ReadMIFLine then
            break;

    Result.PresenterID := TPolylinePresenter.FindMatch( Presenters, aPen, aBrush, nil);

    if Result.PresenterID = 0 then
    begin
      aPresenter := TPolyPresenter.Create( Globe, 0 );
      aPresenter.PolyPen.Assign( aPen );
      aPresenter.PolyBrush.Assign( aBrush );
      Result.PresenterID := Presenters.Add( aPresenter );
    end;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadLineObject
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadLineObject(bNewObject : Boolean) : TGeoDataObject;
var
  aPresenter : TPolyPresenter;
begin
  Result := TGeoDataObject.Create( Self );

  Result.Chains[0].Add(ReadLLPoint(1, 2));
  Result.Chains[0].Add(ReadLLPoint(3, 4));

  Result.Title := 'New Polyline';

  if bNewObject then
  begin
    ReadMIFLine;
    ReadPenObject;

    Result.PresenterID := TPolylinePresenter.FindMatch( Presenters, aPen, aBrush, nil);

    if Result.PresenterID = 0 then
    begin
      aPresenter := TPolyPresenter.Create( Globe, 0 );
      aPresenter.PolyPen.Assign( aPen );
      aPresenter.PolyBrush.Assign( aBrush );
      Result.PresenterID := Presenters.Add( aPresenter );
    end;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadTextObject
------------------------------------------------------------------------------}
function TGlobeMIFReader.ReadTextObject(bNewObject : Boolean) : TGeoDataObject;
var
  sText : string;
  ptLL1, ptLL2 : TPointLL;
  aPresenter : TPointPresenter;
begin
  ReadMIFLine;
  sText := CurrentMIFLine;
  sText := ReplaceStr(ReplaceStr(Trim(sText), '"', ''), '\n', ' ');

  ReadMIFLine;
  ptLL1 := ReadLLPoint( 0, 1 );
  ptLL2 := ReadLLPoint( 0, 1 );

  Result := TGeoDataObject.Create( Self );

  with Result do
  begin
    Centroid := ptLL1;

    if bNewObject then
    begin
      ReadMIFLine;
      ReadFontObject;

      with Globe.GlobeCanvas do
      begin
        aTitleFont.FontUnit := Meter;
        with ptLL2 do
          aTitleFont.FontSize := -Abs(ptLL1.iLatY - ptLL2.iLatY);

      Result.PresenterID := TPointPresenter.FindMatch( Presenters,
        aPen, aBrush, aFont, nil, ppCircle, Pixel, 6, '');

      if Result.PresenterID = 0 then
      begin
        aPresenter := TPointPresenter.Create(Globe, 0);
        aPresenter.PointPen.Assign( aPen );
        aPresenter.PointBrush.Assign( aBrush );
        aPresenter.PointFont.Assign( aFont );
        aPresenter.TitleFont.Assign( aTitleFont );

        Result.PresenterID := Presenters.Add( aPresenter );
      end;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.ReadMIFHeader
------------------------------------------------------------------------------}
procedure TGlobeMIFReader.ReadMIFHeader;
var
  sTmp : string;
  bColumns : Boolean;
begin
  MIDDelimiter := #9;

  bColumns := False;
  while ReadMIFLine do
  begin
    sTmp := GetMIFWord(0);

    if CompareText(sTmp, 'Data') = 0 then
      Break;

    if CompareText(sTmp, 'Version') = 0 then
      iMIFVersion := StrToInt(GetMIFWord(1));

    if CompareText(sTmp, 'Delimiter') = 0 then
      MIDDelimiter := GetMIFWord(1)[2];

    if CompareText(sTmp, 'CoordSys') = 0 then
      MIFCoordSys := CurrentMIFLine;

    if bColumns then
      MIDColumnNames.Add( sTmp + '=' + GetMIFWord(1));

    if CompareText(sTmp, 'Columns') = 0 then
      bColumns := True;
  end;

  // Check the Projection Model used.
  CurrentMIFLine := MIFCoordSys;
  if not ((CompareText(GetMIFWord(1), 'Earth') = 0)
      and (CompareText(GetMIFWord(2), 'Projection') = 0)
      and (CompareText(GetMIFWord(3), '1,') = 0)) then
    raise EGlobeException.Create( 'MIF Import only supports Lat/Long tables.');
end;

{------------------------------------------------------------------------------
 TGlobeMIFReader.LoadObject
------------------------------------------------------------------------------}
function TGlobeMIFReader.LoadObject(iIndex : integer; bNewObject : Boolean) : TGlobeObject;
var
  sTmp : string;
begin
  Result := nil;

  with TMIFItemPtr( DynArrayPtr( FDataItems, iIndex))^ do
  begin
    FMIFReader.Position := iMIFOffset;
    
    if FMIDReader <> nil then
      FMIDReader.Position := iMIDOffset;

    ReadMIFLine;

    sTmp := GetMIFWord(0);

    if CompareText(sTmp, 'Point') = 0 then
      Result := ReadPointObject(bNewObject)
    else
      if CompareText(sTmp, 'Region') = 0 then
        Result := ReadRegionObject(bNewObject)
      else
        if CompareText(sTmp, 'Line') = 0 then
          Result := ReadLineObject(bNewObject)
        else
          if CompareText(sTmp, 'PLine') = 0 then
            Result := ReadPlineObject(bNewObject)
          else
            if CompareText(sTmp, 'Text') = 0 then
              Result := ReadTextObject(bNewObject);
  end;
end;

{------------------------------------------------------------------------------
 TGlobeMIFReader.SaveMetaData
------------------------------------------------------------------------------}
{**
  @Param sFilename Name of file to save meta data to.
}
procedure TGlobeMIFReader.SaveMetaData;
var
  idx : integer;
  Writer : TGlobeStreamWriter;
  sMetaDataName : TFilename;
begin
  sMetaDataName := MetaDataFilename( Filename, True );
  if sMetaDataName = '' then
    Exit;

  if not FileExists( sMetaDataName ) then
  begin
    Writer := TGlobeStreamWriter.Create( TMemoryStream.Create );
    try
      Writer.WriteWord(TG_MIF_READERVERSION);

      WriteMetaData( Writer );
      
      Writer.WriteInteger(iMIFVersion); // save Version of MIF file
      Writer.WriteInteger( TitleColumn );
      Writer.WriteShortString( MIDDelimiter );
      Writer.WriteShortString( FMIFCoordSys );

      // Save the MID Column names
      Writer.WriteInteger(MIDColumnNames.Count); // save the count of Column Names
      for idx := 0 to MIDColumnNames.Count - 1 do
        Writer.WriteShortString( MIDColumnNames[idx] );

      Writer.WriteInteger(FDataItems.Count); // save the count of items

      // save the class data.
      for idx := 0 to FDataItems.Count - 1 do
      begin
        Writer.WriteInteger(TMIFItemPtr( DynArrayPtr( FDataItems, idx))^.iMIFOffset);
        Writer.WriteInteger(TMIFItemPtr( DynArrayPtr( FDataItems, idx))^.iMIDOffset);
      end;

      Writer.WriteShortString( ExtractFilename( Filename ));
      TMemoryStream(Writer.DataStream).SaveToFile( sMetaDataName );
    finally
      Writer.Free;
    end;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.LoadMetaData
------------------------------------------------------------------------------}
{**
  @Param sFilename Name of file to save meta data to.
}
procedure TGlobeMIFReader.LoadMetaData;
var
  idx, iCount : Integer;
  Reader : TGlobeStreamReader;
  sMetaDataName : TFilename;
begin
  sMetaDataName := MetaDataFilename( Filename, False );
  if not FileExists( sMetaDataName ) then
    Exit;

  // Check the metadata file is older than the data file
  if FileAge( sMetaDataName ) < FileAge( Filename ) then
    Exit;

  Reader := TGlobeStreamReader.Create( TMemoryStream.Create );
  try
    TMemoryStream(Reader.DataStream).LoadFromFile( sMetaDataName );

    if Reader.ReadWord <> TG_MIF_READERVERSION then
    begin
      DeleteFile( sMetaDataName );
      Exit;
    end;

    ReadMetaData( Reader );

    DynArrayFree( FDataItems ); // clear the data item list

    iMIFVersion := Reader.ReadInteger; // get Version of MIF file
    TitleColumn := Reader.ReadInteger;
    MIDDelimiter := Reader.ReadShortString;
    MIFCoordSys := Reader.ReadShortString;

    if MIDColumnNames = nil then
      MIDColumnNames := TStringList.Create;
    MIDColumnNames.Clear;

    // get the number of ColumnNames
    iCount := Reader.ReadInteger;
    for idx := 0 to iCount - 1 do
      MIDColumnNames.Add( Reader.ReadShortString );

    // get the number of items for this class
    iCount := Reader.ReadInteger;

    // set the length of the Reader Item array
    FDataItems := DynArrayCreate( SizeOf( TMIFMIDItem ), iCount );

    // load in the Reader data
    for idx := 0 to iCount - 1 do
    begin
      TMIFItemPtr( DynArrayPtr( FDataItems, idx))^.iMIFOffset := Reader.ReadInteger;
      TMIFItemPtr( DynArrayPtr( FDataItems, idx))^.iMIDOffset := Reader.ReadInteger;
    end;

    Filename := Globe.ResolveFilename( Reader.ReadShortString ); // get the data file name
  finally
    Reader.Free;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.InternalOpen
------------------------------------------------------------------------------}
procedure TGlobeMIFReader.InternalOpen;
begin
  Name := ExtractFilename( Filename );

  aPen := TGlobePen.Create;
  aBrush := TGlobeBrush.Create;
  aFont := TGlobeFont.Create;
  aTitleFont := TGlobeFont.Create;

  if FMIDColumnNames = nil then
    FMIDColumnNames := TStringList.Create;

  if FileExists( Filename ) then
  begin
    // Create the MIF data stream
    FMIFReader := TTextReader.Create(Filename);

    if FileExists( ChangeFileExt( Filename, '.MID' )) then
      FMIDReader := TTextReader.Create(ChangeFileExt( Filename, '.MID' ));

    if FDataItems = nil then
      MapMIFFile;

    inherited;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.InternalClose
------------------------------------------------------------------------------}
procedure TGlobeMIFReader.InternalClose;
begin
  SaveMetaData;

  Count := 0;
  DynArrayFree( FDataItems ); // clear the data item list

  FMIDColumnNames.Free;

  FMIFReader.Free;
  FMIFReader := nil;
  FMIDReader.Free;
  FMIDReader := nil;

  aPen.Free;
  aBrush.Free;
  aFont.Free;
  aTitleFont.Free;

  inherited;
end;

{------------------------------------------------------------------------------
 TGlobeMIFReader.MapMIFFile
------------------------------------------------------------------------------}
procedure TGlobeMIFReader.MapMIFFile;
var
  iFileSize, iItems, iMIFPosition, iMIDPosition : integer;
  sTmp : string;
begin
  Globe.ProgressMessage( pmStart, rsMapping);

  FDataItems := DynArrayCreate( Sizeof( TMIFMIDItem ), 0 );
  Count := 0;

  iFileSize := FMIFReader.Size;
  FMIFReader.TextBufferSize := 1024 * 32;

  ReadMIFHeader;

  iItems := 0;
  iMIDPosition := 0;
  while not FMIFReader.EOT do
  begin
    iMIFPosition := FMIFReader.Position;

    ReadMIFLine;
    sTmp := GetMIFWord(0);

    if (CompareText(sTmp, 'Point') = 0)
      or (CompareText(sTmp, 'Region') = 0)
      or (CompareText(sTmp, 'Line') = 0)
      or (CompareText(sTmp, 'PLine') = 0)
      or (CompareText(sTmp, 'Text') = 0) then
    begin
      Inc( iItems );
      if iItems >= FDataItems.Count then
        DynArraySetLength(FDataItems, iItems + 64);

      with TMIFItemPtr( DynArrayPtr( FDataItems, iItems - 1))^ do
      begin
        iMIFOffset := iMIFPosition;
        iMIDOffset := iMIDPosition;
        if ReadMIDLine then
          iMIDPosition := FMIDReader.Position;
      end;
    end;
    if ( iItems mod 256 ) = 0 then
      Globe.ProgressMessage( pmPercent, IntToStr( MulDiv( iMIFPosition, 100, iFileSize )));
  end;
  DynArraySetLength(FDataItems, iItems );

  FMIFReader.TextBufferSize := 4096 * 2;

  Count := FDataItems.Count;

  Globe.ProgressMessage( pmEnd, rsFinished);
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.SaveEnvironment
------------------------------------------------------------------------------}
{**
  @Result Element containing the MIFReader environment.
}
function TGlobeMIFReader.SaveEnvironment : TGXML_Element;
begin
  Result := inherited SaveEnvironment;

  Result.AddAttribute( 'Filename', Filename);
  Result.AddAttribute( 'MIFCoordSys', MIFCoordSys );
  Result.AddAttribute( 'MIDDelimiter', MIDDelimiter );
  Result.AddIntAttribute( 'TitleColumn', TitleColumn );
end;

{------------------------------------------------------------------------------
  TGlobeMIFReader.LoadEnvironment
------------------------------------------------------------------------------}
{**
  @Param Element containing the MIFReader environment to load.
}
procedure TGlobeMIFReader.LoadEnvironment( Element : TGXML_Element );
begin
  inherited LoadEnvironment( Element );

  if Element <> nil then
    with Element do
    begin
      Filename := AttributeByName( 'Filename', Filename );
      MIFCoordSys := AttributeByName( 'MIFCoordSys', MIFCoordSys );
      MIDDelimiter := AttributeByName( 'MIDDelimiter', MIDDelimiter );
      TitleColumn := IntAttributeByName( 'TitleColumn', 0 );
      Active := True;
    end;
end;

initialization
  RegisterClasses( [TGlobeMIFReader] );
end.

