{------------------------------------------------------------------------------
 Module:		TGLayerDS.pas

 Comment:   TGlobe Layer DataSet implementation

 Classes:   TGlobeLayerDS

 Author:		Graham Knight
 Email:     tglobe@iname.com

 3.00  01-Dec-98  Converted to TGlobe Version 3
------------------------------------------------------------------------------}
{$IFDEF VER130}
{$DEFINE DELPHI_5}
{$ENDIF}
unit TGLayerDS;

interface

uses Windows, SysUtils, Forms, DB, Classes, Globe4, TGClasses,
  {$IFNDEF DELPHI_5}Dsgnintf,{$ENDIF} dialogs;

type
{$IFNDEF DELPHI_5}
  TLayerNameProperty = class( TPropertyEditor )
  public
    function GetAttributes : TPropertyAttributes; override;
    function GetValue : string; override;
    procedure GetValues( Proc : TGetStrProc ); override;
    procedure SetValue( const Value : string ); override;
  end;
{$ENDIF}

  TBookMarkInfo = record
    BookmarkRecNo : integer;
    BookmarkFlag : TBookmarkFlag;
  end;
  PTBookMarkInfo = ^TBookMarkInfo;

  TGlobeRecord = record
    Title : shortstring;
    PresenterID : Integer;
    ID : Integer;
    Index : Integer;
    Selected : Boolean;
    Hidden : Boolean;
  end;
  PTGlobeRecord = ^TGlobeRecord;

  TGlobeLayerSort = ( lsIndex, lsTitle );

  TGlobeLayerDS = class( TDataSet )
  private
    FiCurRec : Integer;
    FGlobe : TGlobe4;
    FLayer : TGlobeLayer;
    FLayerName : string;
    FLayerSort : TGlobeLayerSort;
    FSortedList : TList;

    procedure SetLayerSort( Value : TGlobeLayerSort );
    procedure SetLayerName( sValue : string );
    procedure SetLayer( ALayer : TGlobeLayer );
    procedure SetGlobe( AGlobe : TGlobe4 );
    procedure SortLayer;
  protected
 { Overriden abstract methods (required) }
    function AllocRecordBuffer : PChar; override;
    procedure FreeRecordBuffer( var Buffer : PChar ); override;
    function IsCursorOpen : Boolean; override;

    procedure GetBookmarkData( Buffer : PChar; Data : Pointer ); override;
    procedure SetBookmarkData( Buffer : PChar; Data : Pointer ); override;
    function GetBookmarkFlag( Buffer : PChar ) : TBookmarkFlag; override;
    procedure SetBookmarkFlag( Buffer : PChar; Value : TBookmarkFlag ); override;

    procedure InternalAddRecord( Buffer : Pointer; Append : Boolean ); override;
    procedure InternalClose; override;
    procedure InternalDelete; override;
    procedure InternalFirst; override;
    procedure InternalGotoBookmark( Bookmark : Pointer ); override;
    procedure InternalHandleException; override;
    procedure InternalInitFieldDefs; override;
    procedure InternalInitRecord( Buffer : PChar ); override;
    procedure InternalLast; override;
    procedure InternalOpen; override;
    procedure InternalPost; override;
    procedure InternalSetToRecord( Buffer : PChar ); override;
    procedure xNotification( AComponent : TComponent; Operation : TOperation );

  public
    function GetFieldData( Field : TField; Buffer : Pointer ) : Boolean; override;
    procedure SetFieldData( Field : TField; Buffer : Pointer ); override;
    function GetRecord( Buffer : PChar; GetMode : TGetMode; DoCheck : Boolean ) : TGetResult; override;
    function GetRecordSize : Word; override;

    function GetRecordCount : Integer; override;
    function GetRecNo : Integer; override;
    procedure SetRecNo( Value : Integer ); override;

    constructor Create( AOwner : TComponent ); override;
    property Layer : TGlobeLayer read FLayer write SetLayer;
  published
    property Globe : TGlobe4 read FGlobe write SetGlobe;
    property LayerName : string read FLayerName write SetLayerName;
    property SortColumn : TGlobeLayerSort read FLayerSort write SetLayerSort;
    property Active;
  end;

procedure Register;

{------------------------------------------------------------------------------}
implementation

//{$R *.DCR}

{------------------------------------------------------------------------------
  TGlobeLayerDS.Create
------------------------------------------------------------------------------}
constructor TGlobeLayerDS.Create( AOwner : TComponent );
begin
  inherited Create( AOwner );

  FGlobe := nil;
  FLayer := nil;
  FLayerName := '';
  FSortedList := nil;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.SetGlobe
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.SetGlobe( AGlobe : TGlobe4 );
begin
  if FGlobe <> AGlobe then
  begin
    Active := False;
    if FGlobe <> nil then
      FGlobe.FreeNotification( Self );
    FGlobe := AGlobe;
    Layer := nil;
    FLayerName := '';
  end;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.SetLayer
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.SetLayer( ALayer : TGlobeLayer );
begin
  if FLayer <> ALayer then
  begin
    FiCurRec := -1;
    FLayer := ALayer;
    if FLayer = nil then
      Active := False
    else
    begin
      FLayerName := FLayer.Name;
      FGlobe := TGlobe4( FLayer.Globe );
      SortLayer;
    end;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.xNotification
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.xNotification( AComponent : TComponent; Operation : TOperation );
begin
  inherited Notification( AComponent, Operation );

  if ( Operation = opRemove ) and ( AComponent = FGlobe ) then
    FGlobe := nil;
end;

{------------------------------------------------------------------------------
  TitleCompare
------------------------------------------------------------------------------}
function TitleCompare( Item1, Item2 : Pointer ) : Integer;
begin
  Result := CompareText( TGlobeObject( Item1 ).Title, TGlobeObject( Item2 ).Title );
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.SortLayer
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.SortLayer;
var
  idx : integer;
begin
  if FLayer = nil then
    Exit;
  if FLayerSort = lsIndex then
    FSortedList := nil
  else
    FSortedList := TList.Create;

  if FSortedList <> nil then
  begin
    FSortedList.Capacity := FLayer.Objects.Count;
    for idx := 0 to FLayer.Objects.Count - 1 do
      FSortedList.Add( FLayer.Objects[idx] );

    FSortedList.Sort( TitleCompare );
  end;
  if not IsEmpty then
    Resync( [] );
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.SetLayerName
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.SetLayerName( sValue : string );
begin
  FLayerName := sValue;
  if FGlobe <> nil then
    with FGlobe do
      SetLayer( Layers[Layers.IndexByName( sValue )]);
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.SetLayerSort
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.SetLayerSort( Value : TGlobeLayerSort );
begin
  if FLayerSort <> Value then
  begin
    FSortedList.Free;
    FLayerSort := Value;
    SortLayer;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.InternalOpen
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalOpen;
begin
  FiCurRec := -1;

  if ( FGlobe <> nil ) and ( FLayer = nil ) then
    with FGlobe do
      SetLayer( Layers[Layers.IndexByName( FLayerName )]);

  if FLayer = nil then
    raise Exception.Create( 'Missing LayerName property' );

  BookmarkSize := SizeOf( TBookmarkInfo );
  InternalInitFieldDefs;
  if DefaultFields then
    CreateFields;
  BindFields( True );
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.InternalClose
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalClose;
begin
  BindFields( False );
  if DefaultFields then
    DestroyFields;
  FiCurRec := -1;
  Flayer := nil;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.IsCursorOpen
------------------------------------------------------------------------------}
function TGlobeLayerDS.IsCursorOpen : Boolean;
begin
  Result := FLayer <> nil;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.InternalHandleException
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalHandleException;
begin
  Application.HandleException( Self );
end;

{------------------------------------------------------------------------------
 TGlobeLayerDS.InternalGotoBookmark
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalGotoBookmark( Bookmark : Pointer );
begin
  FiCurRec := PTBookMarkInfo( Bookmark )^.BookmarkRecNo;
end;

{------------------------------------------------------------------------------
 TGlobeLayerDS.InternalSetToRecord
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalSetToRecord( Buffer : PChar );
begin
  InternalGotoBookmark( PTBookmarkInfo( Buffer + SizeOf( TGlobeRecord ) ) );
end;

{------------------------------------------------------------------------------
 TGlobeLayerDS.GetBookmarkFlag
------------------------------------------------------------------------------}
function TGlobeLayerDS.GetBookmarkFlag( Buffer : PChar ) : TBookmarkFlag;
begin
  Result := PTBookMarkInfo( Buffer + SizeOf( TGlobeRecord ) )^.BookmarkFlag;
end;

{------------------------------------------------------------------------------
 TGlobeLayerDS.GetBookmarkData
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.GetBookmarkData( Buffer : PChar; Data : Pointer );
begin
  PInteger( Data )^ := Integer( PTBookmarkInfo( Buffer + SizeOf( TGlobeRecord ) )^.BookmarkRecNo );
end;

{------------------------------------------------------------------------------
 TGlobeLayerDS.SetBookmarkData
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.SetBookmarkData( Buffer : PChar; Data : Pointer );
begin
  PTBookmarkInfo( Buffer + SizeOf( TGlobeRecord ) )^.BookmarkRecNo := PInteger( Data )^;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.SetBookmarkFlag
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.SetBookmarkFlag( Buffer : PChar; Value : TBookmarkFlag );
begin
  PTBookMarkInfo( Buffer + SizeOf( TGlobeRecord ) )^.BookmarkFlag := Value;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.GetRecordSize
------------------------------------------------------------------------------}
function TGlobeLayerDS.GetRecordSize : Word;
begin
  Result := SizeOf( TGlobeRecord );
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.AllocRecordBuffer
------------------------------------------------------------------------------}
function TGlobeLayerDS.AllocRecordBuffer : PChar;
begin
  GetMem( Result, SizeOf( TGlobeRecord ) + SizeOf( TBookMarkInfo ) );
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.FreeRecordBuffer
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.FreeRecordBuffer( var Buffer : PChar );
begin
  FreeMem( Buffer, SizeOf( TGlobeRecord ) + SizeOf( TBookMarkInfo ) );
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.GetRecord
------------------------------------------------------------------------------}
function TGlobeLayerDS.GetRecord( Buffer : PChar; GetMode : TGetMode;
  DoCheck : Boolean ) : TGetResult;
var
  GlobeObject : TGlobeObject;
begin
  if FLayer.Objects.Count < 1 then
    Result := grEOF
  else
  begin
    Result := grOK;
    case GetMode of
      gmNext :
        if FiCurRec >= FLayer.Objects.Count - 1 then
          Result := grEOF
        else
          Inc( FiCurRec );
      gmPrior :
        if FiCurRec <= 0 then
          Result := grBOF
        else
          Dec( FiCurRec );
      gmCurrent :
        if ( FiCurRec < 0 ) or ( FiCurRec >= FLayer.Objects.Count ) then
          Result := grError;
    end;

    if Result = grOK then
    begin
      if FSortedList <> nil then
        GlobeObject := TGLobeObject( FSortedList[FiCurRec] )
      else
        GlobeObject := FLayer.objects[FiCurRec];
      with PTGlobeRecord( Buffer )^ do
      begin
        Index := GlobeObject.Index;
        Title := GlobeObject.Title;
        PresenterID := GlobeObject.PresenterID;
        ID := GlobeObject.ID;
        Selected := GlobeObject.Selected;
        Hidden := GlobeObject.Hidden;
      end;
      with PTBookMarkInfo( Buffer + SizeOf( TGlobeRecord ) )^ do
      begin
        BookMarkRecNo := FiCurRec;
        BookmarkFlag := bfCurrent;
      end;
    end
    else
      if ( Result = grError ) and DoCheck then
        DatabaseError( 'No Records' );
  end;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.InternalInitRecord
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalInitRecord( Buffer : PChar );
begin
  if Buffer = nil then Exit;
  FillChar( Buffer^, SizeOf( TGlobeRecord ), 0 );
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.InternalInitFieldDefs
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalInitFieldDefs;
begin
  FieldDefs.Clear;
  TFieldDef.Create( FieldDefs, 'Index', ftInteger, 0, True, 1 );
  TFieldDef.Create( FieldDefs, 'Title', ftString, 32, False, 2 );
  TFieldDef.Create( FieldDefs, 'PresenterID', ftInteger, 0, False, 3 );
  TFieldDef.Create( FieldDefs, 'ID', ftInteger, 0, False, 4 );
  TFieldDef.Create( FieldDefs, 'Hidden', ftBoolean, 0, False, 5 );
  TFieldDef.Create( FieldDefs, 'Selected', ftBoolean, 0, False, 6 );
end;

{------------------------------------------------------------------------------
 TGlobeLayerDS.GetFieldData
------------------------------------------------------------------------------}
function TGlobeLayerDS.GetFieldData( Field : TField; Buffer : Pointer ) : Boolean;
begin
  Result := not IsEmpty;
  if ( not isCursorOpen ) or ( Buffer = nil ) then Exit;

  with PTGlobeRecord( ActiveBuffer )^ do
    case Field.FieldNo of
      1 : Move( Index, Buffer^, Field.DataSize );
      2 : StrPCopy( PChar( Buffer ), Title );
      3 : Move( PresenterID, Buffer^, Field.DataSize );
      4 : Move( ID, Buffer^, Field.DataSize );
      5 : Move( Hidden, Buffer^, Field.DataSize );
      6 : Move( Selected, Buffer^, Field.DataSize );
    end;
  Result := True;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.SetFieldData
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.SetFieldData( Field : TField; Buffer : Pointer );
begin
  if Buffer = nil then Exit;

  with PTGlobeRecord( ActiveBuffer )^ do
    case Field.FieldNo of
      1 : Index := PInteger( Buffer )^;
      2 : Title := string( PChar( Buffer ) );
      3 : PresenterID := PInteger( Buffer )^;
      4 : ID := PInteger( Buffer )^;
      5 : Hidden := Boolean( PInteger( Buffer )^ );
      6 : Selected := Boolean( PInteger( Buffer )^ );
    end;
  DataEvent( deFieldChange, Longint( Field ) );
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.InternalFirst
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalFirst;
begin
  FiCurRec := -1;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.InternalLast
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalLast;
begin
  if FLayer <> nil then
    FiCurRec := FLayer.Objects.Count
  else
    FiCurRec := -1;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.InternalPost
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalPost;
var
  GlobeObject : TGlobeObject;
  iTmpIndex : integer;
begin
  GlobeObject := nil;

  if State = dsEdit then
    if FSortedList <> nil then
      GlobeObject := TGLobeObject( FSortedList[FiCurRec] )
    else
      GlobeObject := FLayer.Objects[FiCurRec];

  if GlobeObject = nil then
    raise Exception.Create( 'Insert or Append not supported in TGlobeLayerDS' );

  with PTGlobeRecord( ActiveBuffer )^ do
  begin
    GlobeObject.Title := Title;
    GlobeObject.PresenterID := PresenterID;
    GlobeObject.ID := ID;
    GlobeObject.Hidden := Hidden;
    GlobeObject.Selected := Selected;

    iTmpIndex := GlobeObject.Index;
    GlobeObject.Index := Index;
    if iTmpIndex <> Index then
      Resync( [] );
  end;
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.InternalAddRecord
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalAddRecord( Buffer : Pointer; Append : Boolean );
begin
  raise Exception.Create( 'Insert or Append not supported in TGlobeLayerDS' );
end;

{------------------------------------------------------------------------------
  TGlobeLayerDS.InternalDelete
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.InternalDelete;
begin
//  FLayer.Objects[FiCurRec].delete;
  Resync( [] );
end;

{------------------------------------------------------------------------------
 TGlobeLayerDS.GetRecordCount
------------------------------------------------------------------------------}
function TGlobeLayerDS.GetRecordCount : Longint;
begin
  if FLayer <> nil then
    Result := FLayer.Objects.Count
  else
    Result := 0;
end;

{------------------------------------------------------------------------------
 TGlobeLayerDS.GetRecNo
------------------------------------------------------------------------------}
function TGlobeLayerDS.GetRecNo : Longint;
begin
  UpdateCursorPos;
  if ( FiCurRec = -1 ) and ( FLayer <> nil ) and ( FLayer.Objects.Count > 0 ) then
    Result := 1
  else
    Result := FiCurRec + 1;
end;

{------------------------------------------------------------------------------
 TGlobeLayerDS.SetRecNo
------------------------------------------------------------------------------}
procedure TGlobeLayerDS.SetRecNo( Value : Integer );
begin
  if ( Value >= 0 ) and ( Value < FLayer.Objects.Count ) then
  begin
    FiCurRec := Value - 1;
    Resync( [] );
  end;
end;

{$IFNDEF DELPHI_5}
{------------------------------------------------------------------------------
 TLayerNameProperty.GetAttributes
------------------------------------------------------------------------------}
function TLayerNameProperty.GetAttributes : TPropertyAttributes;
begin
  Result := [paValueList, paRevertable];
end;

{------------------------------------------------------------------------------
 TLayerNameProperty.GetValue
------------------------------------------------------------------------------}
function TLayerNameProperty.GetValue : string;
begin
  Result := GetStrValue;
end;

{------------------------------------------------------------------------------
 TLayerNameProperty.GetValues
------------------------------------------------------------------------------}
procedure TLayerNameProperty.GetValues( Proc : TGetStrProc );
var
  idx : integer;
  AGlobe : TGlobe4;
begin
  AGlobe := TGlobeLayerDS( GetComponent( 0 ) ).Globe;
  if AGlobe <> nil then
    for idx := 0 to AGlobe.Layers.Count - 1 do
      Proc( AGlobe.Layers[idx].Name );
end;

{------------------------------------------------------------------------------
 TLayerNameProperty.SetValue
------------------------------------------------------------------------------}
procedure TLayerNameProperty.SetValue( const Value : string );
begin
  SetStrValue( Value );
end;
{$ENDIF}

{------------------------------------------------------------------------------
 Register
------------------------------------------------------------------------------}
procedure Register;
begin
  RegisterComponents( 'TGlobe', [TGlobeLayerDS] );
{$IFNDEF DELPHI_5}
  RegisterPropertyEditor( TypeInfo( string ), TGlobeLayerDS, 'LayerName', TLayernameProperty );
{$ENDIF}
end;

end.



