{------------------------------------------------------------------------------
 Module:		TGLYRMapper.pas

 Comment:   Reads and writes a LYR file

 Classes:   TGlobeLYRReader
            TGlobeLYRWriter

 Author:	  Graham Knight
 Email:		  tglobe@iname.com
------------------------------------------------------------------------------}
unit TGLYRMapper;

interface

uses
  Windows, SysUtils, Classes, Globe4, TGPresenters, TGSysUtils,
  TGObjects, TGTextReader, TGClasses, TGXML, TGMMStream;

const
  TG_LYR_READERVERSION = $0102;

type
  TLYRItem = record
    iObjClass : integer;
    iOffset : integer;
  end;
  TLYRItemPtr = ^TLYRItem;

  {---------------------------- TGlobeLYRReader -------------------------------}
  TGlobeLYRReader = class(TGlobeFileReader)
  private
    FDataItems : DynArray;
    FObjectClasses : TStringList;
  protected
    LYRStream : TStream;
    Reader : TGlobeStreamReader;

    procedure InternalOpen; override;
    procedure InternalClose; override;
    procedure MapLYRFile;

    procedure ReadLYRHeader;
  public
    constructor Create( ParentGlobe : TCustomGlobe4 ); override;
    destructor Destroy; override;

    function SaveEnvironment : TGXML_Element; override;
    procedure LoadEnvironment( Element : TGXML_Element ); override;
    procedure SaveMetaData; override;
    procedure LoadMetaData; override;

    function LoadObject(iIndex : integer; bNewObject : Boolean) : TGlobeObject; override;
  end;

  {---------------------------- TGlobeLYRWriter -------------------------------}
  TGlobeLYRWriter = class( TGlobeFileWriter )
  private
    FObjectClasses : TStringList;
    FPresenterClasses : TStringList;
    procedure WriteLYRBody( Writer : TGLobeStreamWriter );
    procedure WriteLYRHeader( Writer : TGLobeStreamWriter );
  public
    constructor Create( aLayer : TGlobeLayer ); override;
    destructor Destroy; override;
    function SaveToFile( const sFilename : TFilename ) : Boolean; override;
  end;

function WriteLayerToLYRfile( aLayer : TGlobeLayer; const sFilename : string ) : Boolean;

implementation

Uses TGResource;

{------------------------------------------------------------------------------
  WriteLayerToLYRfile
------------------------------------------------------------------------------}
{**
  @Param ALayer The Layer to write.
  @Param sFilename Name of file to write.
  @Result True if write OK.
}
function WriteLayerToLYRfile( aLayer : TGlobeLayer; const sFilename : string ) : Boolean;
begin
  with TGlobeLYRWriter.Create( aLayer ) do
  try
    Result := SaveToFile( sFilename );
  finally
    Free;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeLYRWriter.WriteLYRBody
------------------------------------------------------------------------------}
constructor TGlobeLYRWriter.Create(aLayer : TGlobeLayer);
begin
  inherited Create( aLayer );

  FObjectClasses := TStringList.Create;
  FPresenterClasses := TStringList.Create;
end;

{------------------------------------------------------------------------------
  TGlobeLYRWriter.Destroy
------------------------------------------------------------------------------}
destructor TGlobeLYRWriter.Destroy;
begin
  FObjectClasses.Free;
  FObjectClasses := nil;
  FPresenterClasses.Free;
  FPresenterClasses := nil;

  inherited Destroy;
end;

{------------------------------------------------------------------------------
  TGlobeLYRWriter.SaveToFile
------------------------------------------------------------------------------}
{**
  @Param sFilename Filename to save Layer data to.
  @Result True if the file was saved OK.
}
function TGlobeLYRWriter.SaveToFile(const sFilename: TFilename): Boolean;
var
  Writer : TGlobeStreamWriter;
  idx, iIndex : integer;
begin
  Writer := TGlobeStreamWriter.Create( TFileStream.Create( sFilename, fmCreate ));
  try
    // Build a list of all object classes on the layer
    with FObjectClasses do
      for idx := 0 to FLayer.Objects.Count - 1 do
      begin
        iIndex := IndexOf( FLayer.Objects[idx].Classname );
        if iIndex = -1 then
          iIndex := Add( FLayer.Objects[idx].Classname );

        Objects[iIndex] := TObject( integer( Objects[iIndex] ) + 1 );
      end;

    WriteLYRHeader( Writer );
    WriteLYRBody( Writer );

    Result := True;
  finally
    Writer.Free;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeLYRWriter.WriteLYRBody
------------------------------------------------------------------------------}
{**
  @Param Writer Wtier object to output GlobeObjects to.
}
procedure TGlobeLYRWriter.WriteLYRBody( Writer : TGLobeStreamWriter );
var
  idx, jdx, iCount : integer;
  sClassName : string;
begin
  for idx := 0 to FObjectClasses.Count - 1 do
  begin
    iCount := integer( FObjectClasses.Objects[idx] );
    sClassName := FObjectClasses[idx];

    if iCount > 0 then
    begin
      Writer.WriteInteger( iCount );

      Writer.WriteShortString( sClassName );
      // Save the class data
      for jdx := 0 to FLayer.Objects.Count - 1 do
        if sClassName = FLayer.Objects[jdx].Classname then
          FLayer.Objects[jdx].WriteProperties( Writer );
    end;
  end;
  Writer.WriteInteger( 0 );  // sentinal at the end of the data
end;

{------------------------------------------------------------------------------
  TGlobeLYRWriter.WriteLYRHeader
------------------------------------------------------------------------------}
{**
  @Param Writer Writer object to output Presenter data to.
}
procedure TGlobeLYRWriter.WriteLYRHeader( Writer : TGLobeStreamWriter );
var
  idx, jdx, iIndex, iCount : integer;
  sClassName : string;
  aPresenterStore : TGlobePresenterStore;
  aPresenter : TGlobePresenter;
begin
  Writer.WriteSmallInt( TG_FILEVERSION400 ); { write the version of globe data file }

  Writer.WriteMER( FLayer.LayerMER );  { write the Layer MER }

  aPresenterStore := TGlobePresenterStore.Create( FLayer.Globe );
  try
    // Build the list of used presenters for the objects in this ObjectStore
    for idx := 0 to FLayer.Objects.Count -1 do
      with FLayer.Objects[idx] do
        if aPresenterStore.ByID( PresenterID, False ) = nil then
        begin
          aPresenter := FLayer.FindPresenter( PresenterID );
          if aPresenter <> nil then
            aPresenterStore.Add( aPresenter.Clone(FLayer.Globe));
        end;


    // Build a list of all presenter classes on the layer
    with FPresenterClasses do
      for idx := 0 to aPresenterStore.Count - 1 do
      begin
        iIndex := IndexOf( aPresenterStore[idx].Classname );
        if iIndex = -1 then
          iIndex := Add( aPresenterStore[idx].Classname );

        Objects[iIndex] := TObject( integer( Objects[iIndex] ) + 1 );
      end;

    for idx := 0 to FPresenterClasses.Count - 1 do
    begin
      iCount := integer( FPresenterClasses.Objects[idx] );
      sClassName := FPresenterClasses[idx];

      if iCount > 0 then
      begin
        Writer.WriteInteger( iCount );
        Writer.WriteShortString( sClassName );
        // Save the class data
        for jdx := 0 to aPresenterStore.Count -1 do
          if sClassName = aPresenterStore[jdx].Classname then
            aPresenterStore[jdx].WriteProperties( Writer );
      end;
    end;
  finally
    aPresenterStore.Free;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeLYRReader.ReadLYRHeader
------------------------------------------------------------------------------}
procedure TGlobeLYRReader.ReadLYRHeader;
var
  idx, iCount : Integer;
  sClassName : ShortString;
  oNew : TGlobePresenter;
begin
  LYRStream.Position := 0;

  giFileVersion := Reader.ReadSmallInt; { read the version of globe data file }
  if giFileVersion < TG_FILEVERSION300 then
    raise EGlobeException.CreateFmt( rsEDataFileVersionMsg, [giFileVersion, FileName] );

  { read in the bounding rectangle }
  Reader.ReadRect;

  iCount := Reader.ReadInteger;
  while ( iCount > 0 ) and ( LYRStream.Position < LYRStream.Size ) do
  begin
    sClassName := Reader.ReadShortString; { Get class name }

      // Stop if we hit the TGlobeObject data
    with Globe do
      if FindClass( sClassName ).InheritsFrom( TGlobeObject ) then
      begin
        LYRStream.Position := LYRStream.Position - Length( sClassName ) - 1 - SizeOf( integer );
        Exit;
      end;

    for idx := 0 to iCount - 1 do
      with Globe do
      begin
        { Create new instance of class the Presenter }
        oNew := TGlobePresenterClass( FindClass( sClassName )).Create(Globe, 0);
        oNew.ReadProperties( Reader );
        Presenters.Add( oNew );
      end;
    iCount := Reader.ReadInteger;
  end;
end;

{------------------------------------------------------------------------------
 TGlobeLYRReader.LoadObject
------------------------------------------------------------------------------}
function TGlobeLYRReader.LoadObject( iIndex : integer; bNewObject : Boolean ) : TGlobeObject;
begin
  with TLYRItemPtr( DynArrayPtr( FDataItems, iIndex))^ do
  begin
    LYRStream.Position := iOffset;

    { Create new instance of class the Presenter }
    Result := TGlobeObjectClass( FObjectClasses.Objects[iObjClass] ).Create( Self );
    Result.ReadProperties( Reader );
	end;
end;

{------------------------------------------------------------------------------
  TGlobeLYRReader.SaveMetaData
------------------------------------------------------------------------------}
procedure TGlobeLYRReader.SaveMetaData;
var
  idx : integer;
  Writer : TGlobeStreamWriter;
  sMetaDataName : TFilename;
begin
  sMetaDataName := MetaDataFilename( Filename, True );
  if sMetaDataName = '' then
    Exit;

  if not FileExists( sMetaDataName ) then
  begin
    Writer := TGlobeStreamWriter.Create( TMemoryStream.Create );
    try
      Writer.WriteWord( TG_LYR_READERVERSION );

      WriteMetaData( Writer );

      Writer.WriteInteger( FObjectClasses.Count ); // save the count of Classes
      // Save the class names
      for idx := 0 to FObjectClasses.Count -1 do
        Writer.WriteShortString( FObjectClasses[idx] );

      Writer.WriteInteger( FDataItems.Count ); // save the count of items
      // save the class data.
      for idx := 0 to FDataItems.Count - 1 do
        with TLYRItemPtr( DynArrayPtr( FDataItems, idx))^ do
        begin
          Writer.WriteShortInt( iObjClass );
          Writer.WriteInteger( iOffset );
        end;

      Writer.WriteShortString( ExtractFilename( Filename ));
      TMemoryStream(Writer.DataStream).SaveToFile( sMetaDataName );
    finally
      Writer.Free;
    end;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeLYRReader.LoadMetaData
------------------------------------------------------------------------------}
procedure TGlobeLYRReader.LoadMetaData;
var
  idx, iCount : Integer;
  sClassName : string;
  Reader : TGlobeStreamReader;
  sMetaDataName : TFilename;
begin
  sMetaDataName := MetaDataFilename( Filename, False );
  if not FileExists( sMetaDataName ) then
    Exit;

  // Check the metadata file is older than the data file
  if FileAge( sMetaDataName ) < FileAge( Filename ) then
    Exit;

  Reader := TGlobeStreamReader.Create( TMemoryStream.Create );
  try
    TMemoryStream(Reader.DataStream).LoadFromFile( sMetaDataName );

    if Reader.ReadWord <> TG_LYR_READERVERSION then
    begin
      DeleteFile( sMetaDataName );
      Exit;
    end;

    ReadMetaData( Reader );

    DynArrayFree( FDataItems ); // clear the data item list

    FObjectClasses.Clear;

    // get the number of classes
    iCount := Reader.ReadInteger;
    for idx := 0 to iCount -1 do
    begin
      sClassName := Reader.ReadShortString;
      FObjectClasses.AddObject( sClassName, TObject( FindClass( sClassName )));
    end;

    // get the number of items for this class
    iCount := Reader.ReadInteger;

    // set the length of the Reader Item array
    FDataItems := DynArrayCreate( SizeOf( TLYRItem ), iCount );

    // load in the Reader data
    for idx := 0 to iCount - 1 do
      with TLYRItemPtr( DynArrayPtr( FDataItems, idx))^ do
      begin
        iObjClass := Reader.ReadShortInt;
        iOffset := Reader.ReadInteger;
      end;

    Filename := Globe.ResolveFilename( Reader.ReadShortString ); // get the data file name
  finally
    Reader.Free;
  end;
end;


{------------------------------------------------------------------------------
  TGlobeLYRReader.Create
------------------------------------------------------------------------------}
constructor TGlobeLYRReader.Create( ParentGlobe : TCustomGlobe4 );
begin
  inherited;
  FObjectClasses := TStringList.Create;
end;

{------------------------------------------------------------------------------
  TGlobeLYRReader.Destroy
------------------------------------------------------------------------------}
destructor TGlobeLYRReader.Destroy;
begin
  FObjectClasses.Free;
  FObjectClasses := nil;

  inherited Destroy;
end;

{------------------------------------------------------------------------------
  TGlobeLYRReader.InternalOpen
------------------------------------------------------------------------------}
procedure TGlobeLYRReader.InternalOpen;
begin
  Name := ExtractFilename( Filename );

  if FDataItems = nil then
    MapLYRFile;

  if FileExists( Filename ) then
  begin
    // Create the data stream
    LYRStream := TMemoryMapStream.Create( Filename );
//    LYRStream := TFileStream.Create( Filename, fmOpenRead or fmShareDenyWrite );
    Reader := TGlobeStreamReader.Create( LYRStream );

    inherited;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeLYRReader.InternalClose
------------------------------------------------------------------------------}
procedure TGlobeLYRReader.InternalClose;
begin
  Count := 0;
  DynArrayFree( FDataItems ); // clear the data item list

  Presenters.Clear;
  Reader.Free;
  LYRStream := nil;

  inherited;
end;

{------------------------------------------------------------------------------
 TGlobeLYRReader.MapLYRFile
------------------------------------------------------------------------------}
procedure TGlobeLYRReader.MapLYRFile;
var
  idx, iFileSize, iClass, iCount, iSlot : Integer;
  sClassName : ShortString;
begin
  // Use a memory mapped stream to map the file for speed
  LYRStream := TMemoryMapStream.Create( Filename );
  Reader := TGlobeStreamReader.Create( LYRStream );
  try
    Globe.ProgressMessage( pmStart, rsMapping);

    iFileSize := LYRStream.Size;

    FDataItems := DynArrayCreate( Sizeof( TLYRItem ), 0 );

    Count := 0;
    iSlot := 0;

    ReadLYRHeader;  // Load in Presenters

    FObjectClasses.Clear;
    iCount := Reader.ReadInteger;
    while ( iCount > 0 ) and ( LYRStream.Position < LYRStream.Size ) do
    begin
      sClassName := Reader.ReadShortString; { Get class name }

      iClass := FObjectClasses.IndexOf( sClassName );
      if iClass = -1 then
        iClass := FObjectClasses.AddObject( sClassName, TObject( FindClass( sClassName )));

      // Allocate the Mapper Space and Slot space for the objects
      DynArraySetLength( FDataItems, iSlot + iCount );
      Count := iSlot + iCount;

      for idx := 0 to iCount - 1 do
      begin
        with TLYRItemPtr( DynArrayPtr( FDataItems, iSlot))^ do
        begin
          iOffset := LYRStream.Position;
          iObjClass := iClass;
        end;

        GlobeObject[iSlot];  // Load in the object
        Inc( iSlot );

        if ( iSlot mod 256 ) = 0 then
          Globe.ProgressMessage( pmPercent, IntToStr( MulDiv( LYRStream.Position, 100, iFileSize )));
      end;
      iCount := Reader.ReadInteger;
    end;
    Globe.ProgressMessage( pmEnd, rsFinished);
  finally
    // Reset the LYRStream back to the file
    Reader.Free;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeLYRReader.SaveEnvironment
------------------------------------------------------------------------------}
{**
  @Result Element containing the LYRReader environment.
}
function TGlobeLYRReader.SaveEnvironment : TGXML_Element;
begin
  Result := inherited SaveEnvironment;

  Result.AddAttribute( 'Filename', Filename);
end;

{------------------------------------------------------------------------------
  TGlobeLYRReader.LoadEnvironment
------------------------------------------------------------------------------}
{**
  @Param Element containing the LYRReader environment to load.
}
procedure TGlobeLYRReader.LoadEnvironment( Element : TGXML_Element );
begin
  if Element <> nil then
    Filename := Element.AttributeByName( 'Filename', Filename );

  inherited LoadEnvironment( Element );
end;


initialization
  RegisterClasses( [TGlobeLYRReader] );
end.
