unit TGDBMapper;

interface

uses
  Windows, Classes, SysUtils, DB, Globe4, TGSysUtils, 
  TGObjects, TGPresenters, TGClasses, TGXML;

const
  TG_DBPOINT_READERVERSION = $0100;

type
  TReaderDataLink = class;

  {---------------------------- TGlobeDBReader --------------------------------}
  TGlobeDBReader = class(TGlobeObjectSource)
  private
    FDataLink : TReaderDataLink;
    function GetDataSource: TDataSource;
    procedure SetDataSource(const Value: TDataSource);
  protected
    IDArray : DynArray;
    procedure MapDataSource; virtual; abstract;
  public
    constructor Create( ParentGlobe : TCustomGlobe4 ); override;
    destructor Destroy; override;

    property DataSource : TDataSource read GetDataSource write SetDataSource;
  end;

  TReaderDataLink = class(TDataLink)
  private
    FDBReader: TGlobeDBReader;
  protected
    procedure ActiveChanged; override;
  public
    constructor Create(aDBReader: TGlobeDBReader);
    destructor Destroy; override;
  end;

  {---------------------------- TGlobeDBPointReader ---------------------------}
  TGlobeDBPointReader = class(TGlobeDBReader)
  private
    FLatitudeColName: string;
    FLongitudeColName: string;
    FIDColName: string;
    FTitleColName: string;

    iLongitudeColumn : integer;
    iLatitudeColumn : integer;
    iIDColumn : integer;
    iTitleColumn : integer;
  protected
    procedure MapDataSource; override;
    procedure InternalOpen; override;
    procedure InternalClose; override;
  public
    function LoadObject(iIndex : integer; bNewObject : Boolean) : TGlobeObject; override;

    function SaveEnvironment : TGXML_Element; override;
    procedure LoadEnvironment( Element : TGXML_Element ); override;

    property IDColName : string read FIDColName write FIDColName;
    property LongitudeColName : string read FLongitudeColName write FLongitudeColName;
    property LatitudeColName : string read FLatitudeColName write FLatitudeColName;
    property TitleColName : string read FTitleColName write FTitleColName;
  end;

implementation

{------------------------------------------------------------------------------
  TReaderDataLink.ActiveChanged
------------------------------------------------------------------------------}
procedure TReaderDataLink.ActiveChanged;
begin
  if ( FDBReader <> nil ) and ( DataSource <> nil )
      and ( DataSource.DataSet <> nil ) then
    FDBReader.Active := DataSource.DataSet.Active;
end;

{------------------------------------------------------------------------------
  TReaderDataLink.Create
------------------------------------------------------------------------------}
constructor TReaderDataLink.Create(aDBReader: TGlobeDBReader);
begin
  inherited Create;

  FDBReader := aDBReader;
//  VisualControl := True;
end;

{------------------------------------------------------------------------------
  TReaderDataLink.Destroy
------------------------------------------------------------------------------}
destructor TReaderDataLink.Destroy;
begin
  FDBReader := nil;
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  TGlobeDBReader.Create
------------------------------------------------------------------------------}
constructor TGlobeDBReader.Create(ParentGlobe: TCustomGlobe4);
begin
  inherited;
  IDArray := DynArrayCreate( SizeOf( integer ), 0 );
  FDataLink := TReaderDataLink.Create( Self );
end;

{------------------------------------------------------------------------------
  TGlobeDBReader.Destroy
------------------------------------------------------------------------------}
destructor TGlobeDBReader.Destroy;
begin
  FDataLink.DataSource := nil;
  FDataLink.Free;
  FDataLink := nil;
  DynArrayFree( IDArray );
  inherited;
end;

{------------------------------------------------------------------------------
  TGlobeDBReader.GetDataSource
------------------------------------------------------------------------------}
function TGlobeDBReader.GetDataSource: TDataSource;
begin
  if FDataLink <> nil then
    Result := FDataLink.DataSource
  else
    Result := nil;
end;

{------------------------------------------------------------------------------
  TGlobeDBReader.SetDataSource
------------------------------------------------------------------------------}
procedure TGlobeDBReader.SetDataSource(const Value: TDataSource);
begin
  FDataLink.DataSource := Value;
end;

{------------------------------------------------------------------------------
  TGlobeDBPointReader.LoadEnvironment
------------------------------------------------------------------------------}
procedure TGlobeDBPointReader.LoadEnvironment(Element: TGXML_Element);
begin
  inherited;

end;


{------------------------------------------------------------------------------
  TGlobeDBPointReader.LoadObject
------------------------------------------------------------------------------}
function TGlobeDBPointReader.LoadObject(iIndex: integer;
  bNewObject: Boolean): TGlobeObject;
var
  iID : integer;
begin
  Result := nil;
  iID := DynArrayAsInteger( IDArray, iIndex );
  with DataSource.DataSet do
    if ( Fields[iIDColumn].AsInteger = iID ) or Locate( IDColName, iID, [] ) then
    begin
      Result := TGeoDataObject.Create( Self );
      Result.Centroid := DecimalToPointLL(
        Fields[iLongitudeColumn].AsFloat,
        Fields[iLatitudeColumn].AsFloat );
    end;
end;

{------------------------------------------------------------------------------
  TGlobeDBPointReader.MapDataSource
------------------------------------------------------------------------------}
procedure TGlobeDBPointReader.MapDataSource;
begin
  Count := 0;
  with DataSource.DataSet do
  begin
    DisableControls;
    try
      Globe.ProgressMessage( pmStart, 'Mapping');
      First;
      while not EOF do
      begin
        Count := Count + 1;
        if Count > IDArray.Count then
          DynArraySetLength( IDArray, Count + 32 );

        DynArraySetAsInteger( IDArray, Count - 1, Fields[iIDColumn].AsInteger );
        GlobeObject[Count - 1];  // Load in the object

        Next;
      end;
    finally
      EnableControls;
      Globe.ProgressMessage( pmEnd, 'Finshed');
    end;
  end;
end;

{------------------------------------------------------------------------------
  TGlobeDBPointReader.SaveEnvironment
------------------------------------------------------------------------------}
function TGlobeDBPointReader.SaveEnvironment: TGXML_Element;
begin
  Result := nil;
end;

{------------------------------------------------------------------------------
  TGlobeDBPointReader.InternalOpen
------------------------------------------------------------------------------}
procedure TGlobeDBPointReader.InternalOpen;
begin
  if ( DataSource = nil ) or ( DataSource.DataSet = nil ) then
    raise EGlobeException.Create( 'TGlobeDBReader: Invalid DataSource');
  if ( IDColName = '' ) or ( LongitudeColName = '' ) or ( LatitudeColName = '' ) then
    raise EGlobeException.Create( 'TGlobeDBReader: Missing Property');

  Name := DataSource.DataSet.Name;
  with DataSource.DataSet do
  begin
    Active := True;
    iIDColumn := FieldByName( IDColName ).Index;
    iLongitudeColumn := FieldByName( LongitudeColName ).Index;
    iLatitudeColumn := FieldByName( LatitudeColName ).Index;
    if TitleColName <> '' then
      iTitleColumn := FieldByName( TitleColName ).Index;
  end;
  MapDataSource;

  inherited;
end;

{------------------------------------------------------------------------------
  TGlobeDBPointReader.InternalClose
------------------------------------------------------------------------------}
procedure TGlobeDBPointReader.InternalClose;
begin
  if ( DataSource <> nil ) and ( DataSource.DataSet <> nil ) then
    DataSource.DataSet.Active := False;
  
  inherited;
end;

end.
