{------------------------------------------------------------------------------
 Module:    GLobe Utility functions

 Comment:   General utility routines

 Author:    Graham Knight
 Email:     tglobe@iname.com
------------------------------------------------------------------------------}
unit GlobeUtils;

interface

uses WinTypes, WinProcs, Classes, SysUtils, Graphics, Globe4, TGSysUtils,
    TGObjects, TGClasses;

{------------------------------------------------------------------------------}

function LayerFromObject( Globe : TCustomGlobe4; Obj : TGlobeObject ) : TGlobeLayer;
function CreateFileLayer( Globe : TGlobe4; const sFilename : string ) : integer;
procedure LayerFromResourceName( Layer : TGlobeLayer; const ResourceName : string );
procedure LayerFromResourceID( Layer : TGlobeLayer; ResourceID : Integer );

function EllipsoidDistanceLLToLL(const FromLL, ToLL : TPointLL; Spheroid : TSpheroid) : Integer;
function DistanceLLToLL( const FromLL, ToLL : TPointLL ) : Integer;
function AngleLLToLL( const FromLL, ToLL : TPointLL ) : Integer;
function GreatCirclePoint( const FromLL, ToLL : TPointLL; alpha : Extended ) : TPointLL;
function AngleDistToGCPoint( const ptLL : TPointLL; iAngle, iDistance : Integer ) : TPointLL;

procedure DrawProjectedMER(Globe : TGlobe4; const MER : TMER);

function SplitGeoDataObject( APoly : TGeoDataObject; ALayer : TGlobeLayer ) : integer;
function CombineGeoDataObjects( const Polys : array of TGeoDataObject; ALayer : TGlobeLayer ) : TGeoDataObject;
procedure ConcatPoints( ptStore1, ptStore2 : TPointStore );

procedure ZoomToLayer(Globe: TCustomGlobe4; GlobeLayer: TGlobeLayer);
procedure ZoomToMER( Globe : TCustomGlobe4; MER : TMER );
procedure ZoomToObject( Globe : TCustomGlobe4; Obj : TGlobeObject );
procedure ZoomToSelectedObjects(Globe : TCustomGlobe4);

{------------------------------------------------------------------------------}
implementation

Uses TGLYRMapper, TGSHPMapper, TGMIFMapper;

{------------------------------------------------------------------------------
  LayerFromObject
------------------------------------------------------------------------------}
{**
  @Param Globe TGlobe object to use as parent for the new layer.
  @Param Obj Object to find layer for.
  @Result The Layer that the object is on or nil if not found.
}
function LayerFromObject( Globe : TCustomGlobe4; Obj : TGlobeObject ) : TGlobeLayer;
var
  idx : integer;
begin
  for idx := 0 to Globe.Layers.count - 1 do
  begin
    Result := Globe.Layers[idx];
    if Result.Objects = Obj.Parent then
      Exit;
  end;
  Result := nil;
end;

{------------------------------------------------------------------------------
  CreateFileLayer
------------------------------------------------------------------------------}
{**
  @Param Globe TGlobe object to use as parent for the new layer.
  @Param sFilename Filename of file to create a layer from.
  @Result The Index of the newly created Layer or -1 if the file does not exist.
}
function CreateFileLayer( Globe : TGlobe4; const sFilename : string ) : integer;
var
  aLayer : TGlobeLayer;
begin
  Result := -1;

  // Find the file and Load
  if FileExists( Globe.ResolveFilename( sFilename )) then
  begin
    aLayer := TGlobeLayer.Create( Globe );

    if CompareText( ExtractFileExt( sFilename ), '.LYR' ) = 0 then
    begin
      aLayer.Objects := TGlobeLYRReader.Create( Globe );
      TGlobeLYRReader( aLayer.Objects ).Filename := sFilename;
    end
    else
      if CompareText( ExtractFileExt( sFilename ), '.SHP' ) = 0 then
      begin
        aLayer.Objects := TGlobeSHPReader.Create( Globe );
        TGlobeSHPReader( aLayer.Objects ).Filename := sFilename;
      end
      else
        if CompareText( ExtractFileExt( sFilename ), '.MIF' ) = 0 then
        begin
          aLayer.Objects := TGlobeMIFReader.Create( Globe );
          TGlobeMIFReader( aLayer.Objects ).Filename := sFilename;
        end;

    aLayer.Objects.Active := True;
    Result := aLayer.Index;
  end;
end;

{------------------------------------------------------------------------------
 LayerFromResourceName

 Loads a data layer by name that is store as a resource of the app.

 To add a data layer as a resource you need to create a resource script
 file (world.rc) which is used to generate the WORLD.RES file. This has
 just one line :-

  WORLD RCDATA "TGWORLD.LYR"

 You run the command line resource compiler ( in the Borland\Delphix\Bin
 folder) BRCC32 with the command line -r world.rc.  This will generate
 WORLD.RES which you then include with your program using {$R WORLD.RES}
{------------------------------------------------------------------------------}
{**
  @Param Layer Layer to load the data into.
  @Param ResourceName The name of the resource to load the layer from.
}
procedure LayerFromResourceName( Layer : TGlobeLayer; const ResourceName : string );
var
  Stream : TCustomMemoryStream;
begin
  Stream := TResourceStream.Create( hInstance, ResourceName, RT_RCDATA );
  try
//    Layer.LoadFromStream( Stream );
  finally
    Stream.Free;
  end;
end;

{------------------------------------------------------------------------------
 LayerFromResourceID

 Loads a data layer by ID that is store as a resource of the app.

 See above for a description of how to add a resource to an application.
------------------------------------------------------------------------------}
{**
  @Param Layer Layer to load the data into.
  @Param ResourceID The ID of the resource to load the layer from.
}
procedure LayerFromResourceID( Layer : TGlobeLayer; ResourceID : Integer );
var
  Stream : TCustomMemoryStream;
begin
  Stream := TResourceStream.CreateFromID( hInstance, ResourceID, RT_RCDATA );
  try
//    Layer.LoadFromStream( Stream );
  finally
    Stream.Free;
  end;
end;

{-------------------------------------------------------------------------
 AngleDistToGCPoint

 Returns a point on the great circle defined by a given point,
 a distance and an angle from that point. An angle of 0 is due North.
-------------------------------------------------------------------------}
{**
  @Param ptLL The point to start from.
  @Param iAngle The angle in GlobeUnits to travel in.
  @Param iDistance The distance along the Great Circle line to travel.
  @Result The destination Point.
}
function AngleDistToGCPoint( const ptLL : TPointLL; iAngle, iDistance : integer ) : TPointLL;
var
  sinlat1, coslat1,
    sind, cosd, sintc, costc,
    dlon, lat, lon : extended;
begin
  with PtLL do
  begin
    SinCos( ilatY * GU_TORADIANS, sinlat1, coslat1 );
    SinCos( iDistance / EARTHRADIUS, sind, cosd );
    SinCos( iAngle * GU_TORADIANS, sintc, costc );
    lat := arcsin( sinlat1 * cosd + coslat1 * sind * costc );
    dlon := arctan2( sintc * sind * coslat1, cosd - sinlat1 * sin( lat ) );
    lon := SphericalMod( ilongX * GU_TORADIANS - dlon + LocalPI ) - LocalPI;
  end;
  Result := PointLL( Round( lon * GU_FROMRADIANS ), Round( lat * GU_FROMRADIANS ) );
end;

{-------------------------------------------------------------------------
 GreatCirclePoint

 Returns a point on the Great Circle line between the supplied Points.
 If Alpha is 0.0 then FromLL is returned and if Alpha is 1.0 then ToLL
 is returned.
-------------------------------------------------------------------------}
{**
  @Param FromLL The starting point for the Great Circle line.
  @Param ToLL The ending point for the Great Circle line.
  @Param alpha The interval between the points to locate.
  @Result The located point on the Great Circle line.
}
function GreatCirclePoint( const FromLL, ToLL : TPointLL;
  alpha : Extended ) : TPointLL;
var
  a, b : TV3D;
  mat : TMatrix;
begin
  a := PointLLToV3D( FromLL );
  b := PointLLToV3D( ToLL );
  QuatToMatrix( AxisAngleToQuat( V3DCross( a, b ),
    -ArcCos( V3DDot( a, b ) ) * alpha ), mat );

  with V3DToPointLL( V3DMatrixMul( mat, a ) ) do
    Result := PointLL( iLongX, iLatY );
end;

{------------------------------------------------------------------------------
 AngleLLToLL

 Calculates the Angle between the supplied points. The angle is returned
 in Globeunits with 0 being due north. To convert the result to Radians
 multiply by the GU_TORADIANS constant.
------------------------------------------------------------------------------}
{**
  @Param FromLL The starting point.
  @Param ToLL The ending point.
  @Result The angle between the two points.
}
function AngleLLToLL( const FromLL, ToLL : TPointLL ) : integer;
var
  SLat1, SLat2, CLat1, CLat2 : Extended;
  SLonDiff, CLonDiff : Extended;
begin
  SinCos( FromLL.iLatY * GU_TORADIANS, SLat1, CLat1 );
  SinCos( ToLL.iLatY * GU_TORADIANS, SLat2, CLat2 );
  SinCos( ( FromLL.iLongX - ToLL.iLongX ) * GU_TORADIANS, SLonDiff, CLonDiff );

  Result := Round( SphericalMod( ArcTan2( SLonDiff * CLat2,
    CLat1 * SLat2 - SLat1 * CLat2 * CLonDiff ) ) * GU_FROMRADIANS );
end;

{-------------------------------------------------------------------------
 EllipsoidDistanceLLToLL
 Adapted from CEarth.cpp
 Author: Samuel R. Blackburn, Internet: sblackbu@erols.com
-------------------------------------------------------------------------}
{**
  @Param FromLL Starting point for measurement.
  @Param ToLL Ending point for measurement.
  @Result Distance between the two points in GlobeUnits.
}
function EllipsoidDistanceLLToLL(const FromLL, ToLL : TPointLL; Spheroid : TSpheroid) : Integer;
var
  C, c_value_1, c_value_2, c2a, cy, cz : Extended;
  D, E, r_value : Extended;
  S, s_value_1, sA, sy : Extended;
  tangent_1, tangent_2, X, Y : Extended;
  Heading_FromTo, Heading_ToFrom : Extended;
  term1, term2 : Extended;
  Flattening : Extended;
begin
  Result := 0;

  if (FromLL.iLongX - ToLL.iLongX = 0) and (FromLL.iLatY - ToLL.iLatY = 0) then
    Exit;

  Flattening := SpheroidData[Ord(Spheroid)].f;

  r_value := 1.0 - Flattening;
  tangent_1 := (r_value * Sin(FromLL.iLatY * GU_TORADIANS)) / Cos(FromLL.iLatY * GU_TORADIANS);
  tangent_2 := (r_value * Sin(ToLL.iLatY * GU_TORADIANS)) / Cos(ToLL.iLatY * GU_TORADIANS);
  c_value_1 := 1.0 / Sqrt((tangent_1 * tangent_1) + 1.0);
  s_value_1 := c_value_1 * tangent_1;
  c_value_2 := 1.0 / Sqrt((tangent_2 * tangent_2) + 1.0);
  S := c_value_1 * c_value_2;

  Heading_ToFrom := S * tangent_2; { backward_azimuth }
  Heading_FromTo := Heading_ToFrom * tangent_1;

  X := ToLL.iLongX * GU_TORADIANS - FromLL.iLongX * GU_TORADIANS;

  repeat
    tangent_1 := c_value_2 * Sin(X);
    tangent_2 := Heading_ToFrom - (s_value_1 * c_value_2 * Cos(X));
    sy := Sqrt((tangent_1 * tangent_1) + (tangent_2 * tangent_2));
    cy := (S * Cos(X)) + Heading_FromTo;
    Y := ArcTan2(sy, cy);
    sA := (S * Sin(X)) / sy;
    c2a := (-sA * sA) + 1.0;
    cz := Heading_FromTo + Heading_FromTo;

    if c2a > 0.0 then
      cz := (-cz / c2a) + cy;

    E := (cz * cz * 2.0) - 1.0;
    C := (((((-3.0 * c2a) + 4.0) * Flattening) + 4.0) * c2a * Flattening) / 16.0;
    D := X;
    X := ((((E * cy * C) + cz) * sy * C) + Y) * sA;
    X := ((1.0 - C) * X * Flattening) + ToLL.iLongX * GU_TORADIANS - FromLL.iLongX * GU_TORADIANS;
  until Abs(D - X) < 5.0E20;

  X := Sqrt((((1.0 / r_value / r_value) - 1) * c2a) + 1.0) + 1.0;
  X := (X - 2.0) / X;
  C := 1.0 - X;
  C := (((X * X) * 0.25) + 1.0) / C;
  D := ((0.375 * (X * X)) - 1.0) * X;
  X := X * cy;

  S := (1.0 - E) - E;

  term1 := ((sy * sy * 4.0) - 3.0) * (((S * cz * D) / 6.0) - X);
  term2 := ((((term1 * D) * 0.25) + cz) * sy * D) + Y;

  Result := Round(term2 * C * SpheroidData[Ord(Spheroid)].a * r_value);
end;

{-------------------------------------------------------------------------
 DistanceLLToLL

 Calculates the distance between the supplied points using the Haversine
 formula.  This is quick but not very accurate, if better accuracy is
 required use the Globe.EllipsoidDistanceLLtoLL() method.
-------------------------------------------------------------------------}
{
  @Param FromLL Starting Point.
  @Param ToLL Ending Point
  @Result Distance between the two points in GlobeUnits.
}
function DistanceLLToLL( const FromLL, ToLL : TPointLL ) : Integer;
var
  dist, latFrom, LatTo, lonDist : Extended;
begin
  lonDist := FromLL.iLongX * GU_TORADIANS - ToLL.iLongX * GU_TORADIANS;
  latFrom := FromLL.iLatY * GU_TORADIANS;
  LatTo := ToLL.iLatY * GU_TORADIANS;

  dist := 2 * ArcSin( Sqrt( ( Sqr( Sin( ( latFrom - LatTo ) * 0.5 ) ) )
    + Cos( latFrom ) * Cos( LatTo ) * Sqr( Sin( lonDist * 0.5 ) ) ) );

  Result := Round( dist * GU_FROMRADIANS );
end;

{------------------------------------------------------------------------------
 SplitGeoDataObject
------------------------------------------------------------------------------}
{
  @Param APoly An object to split.
  @Param ALayer The Layer on which to create the individual objects from APoly.
  @Result The count of the number of new objects created.
}
function SplitGeoDataObject( APoly : TGeoDataObject; ALayer : TGlobeLayer ) : integer;
var
  idx : integer;
begin
  ALayer.Objects.Capacity := ALayer.Objects.Count + APoly.Chains.Count;

  Result := APoly.Chains.Count;
  for idx := 0 to Result - 1 do
    with TGeoDataObject.Create( ALayer.Objects ) do
    begin
      Title := APoly.Title;
      Chains.Count := 1;
      Chains[0] := APoly.Chains[idx].Clone;
      APoly.Chains[idx] := nil;
    end;
  APoly.Chains.Count := 0;
end;

{------------------------------------------------------------------------------
 CombineGeoDataObjects
------------------------------------------------------------------------------}
{
  @Param Polys An array of objects to combine.
  @Param ALayer The layer to store the new combined object.
  @Result The new object.
}
function CombineGeoDataObjects( const Polys : array of TGeoDataObject; ALayer : TGlobeLayer ) : TGeoDataObject;
var
  idx, jdx, iChains : integer;
begin
  Result := TGeoDataObject.Create( ALayer.Objects );
  Result.Title := 'New Object';

  iChains := 0;
  for idx := 0 to High( Polys ) do
    Inc( iChains, Polys[idx].Chains.Count );

  Result.Chains.Count := iChains;

  iChains := 0;
  for idx := 0 to High( Polys ) do
  begin
    for jdx := 0 to Polys[idx].Chains.Count - 1 do
    begin
      Result.Chains[iChains] := Polys[idx].Chains[jdx];
//      Polys[idx].Chains[jdx] := nil;
      Inc( iChains );
    end;
  end;
end;

{------------------------------------------------------------------------------
  ConcatPoints
------------------------------------------------------------------------------}
{**
  @Param ptStore1 Target point store to add to.
  @Param ptStore2 Source of points to add points from.
}
procedure ConcatPoints( ptStore1, ptStore2 : TPointStore );
var
  idx : integer;
begin
  for idx := 0 to ptStore2.Count - 1 do
    ptStore1.Add( ptStore2[idx] );
end;

{------------------------------------------------------------------------------
 DrawProjectedMER
------------------------------------------------------------------------------}
{**
  @Param Globe TGlobe component on which to render the MER.
  @Param MER The MER to draw.
}
procedure DrawProjectedMER(Globe : TGlobe4; const MER : TMER);
begin
  // display the bounding rectangle
  with MER do
  begin
    globe.RenderLine(PointLL(iLongX, iLatY), PointLL(iLongX + iWidthX, iLatY), 32);
    globe.RenderLine(PointLL(iLongX + iWidthX, iLatY), PointLL(iLongX + iWidthX, iLatY + iHeightY), 32);
    globe.RenderLine(PointLL(iLongX + iWidthX, iLatY + iHeightY), PointLL(iLongX, iLatY + iHeightY), 32);
    globe.RenderLine(PointLL(iLongX, iLatY + iHeightY), PointLL(iLongX, iLatY), 32);
  end;
end;

{------------------------------------------------------------------------------
  ZoomToMER
------------------------------------------------------------------------------}
{**
  @Param Globe TGlobe object to Zoom.
  @Param MER MER to Zoom to.
}
procedure ZoomToMER( Globe : TCustomGlobe4; MER : TMER );
begin
  Globe.Projection.CenterXY := Point( Globe.Width div 2, Globe.Height div 2 );
  with MER do
  begin
    Globe.ViewRect := Rect( 0,0, iWidthX, iHeightY );
    Globe.LocateToLL( iLongX + iWidthX div 2, iLatY + iHeightY div 2 );
  end;
end;

{------------------------------------------------------------------------------
  ZoomToSelectedObjects
------------------------------------------------------------------------------}
{**
  @Param Globe TGlobe object to Zoom.
}
procedure ZoomToSelectedObjects(Globe : TCustomGlobe4);
var
  idx: Integer;
  MER : TMER;
begin
  MER := Globe.Layers.SelectedObjects[0].ObjectMER;

  if Globe.Layers.SelectedCount > 1 then
    for idx := 1 to Globe.Layers.SelectedCount - 1 do
      MER := UnionMER(MER, Globe.Layers.SelectedObjects[idx].ObjectMER);

  ZoomToMER(Globe,MER);
end;

{------------------------------------------------------------------------------
  ZoomToLayer
------------------------------------------------------------------------------}
{**
  @Param Globe TGlobe object to Zoom.
  @Param GlobeLayer Layer to Zoom to.
}
procedure ZoomToLayer(Globe: TCustomGlobe4; GlobeLayer: TGlobeLayer);
begin
 ZoomToMER(Globe, GlobeLayer.LayerMER);
end;

{------------------------------------------------------------------------------
  ZoomToObject
------------------------------------------------------------------------------}
{**
  @Param Globe TGlobe object to Zoom.
  @Param Obj Object to Zoom to.
}
procedure ZoomToObject( Globe : TCustomGlobe4; Obj : TGlobeObject );
begin
  ZoomToMER( Globe, Obj.ObjectMER );
end;

end.

