{------------------------------------------------------------------------------
  Module:   TGProp.pas

  Classes:  TGlobePropertyHandler
            TGlobeProperties
            TGlobeColorHandler
            TProjectionClassHandler

  Author:   Graham Knight
  Email:    tglobe@iname.com

  Note:     For Delphi 5 include the Delphi5\Source\ToolsAPI directory in the
            project search path.

  3.00  01-Dec-98  Converted to TGlobe Version 3

------------------------------------------------------------------------------}
unit TGProp;

interface

uses WinTypes, WinProcs, Classes, SysUtils, Graphics, Globe4, dsgnintf, TypInfo,
  Forms, Dialogs, StdCtrls, Buttons, Controls, ExtCtrls, checklst, TGSysUtils,
  TGPresenters, TGObjects, TGClasses, GlobeUtils;

type
  TGlobePropertyAttribute = (paClass, paList, paSimple);
  TGlobePropertyAttributes = set of TGlobePropertyAttribute;

  TPropertyForm = class(TForm)
    Notebook1 : TNotebook;
    lbSetMembers : TCheckListBox;
    Panel1 : TPanel;
    BitBtn1 : TBitBtn;
    BitBtn2 : TBitBtn;
    edtGU : TEdit;
    edtDD : TEdit;
    edtNM : TEdit;
    Label1 : TLabel;
    Label2 : TLabel;
    Label3 : TLabel;
    Label4 : TLabel;
    edtKM : TEdit;
    lblException : TLabel;
    lstFonts : TListBox;
    procedure edtGUChange(Sender : TObject);
    procedure edtValueChange(Sender : TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

  {----------------------------------------------------------------------------}
  TGlobePropertyHandler = class
  public
    procedure GetPicklist(Instance : TObject; PropInfo : PPropInfo; PickList : TStrings); virtual;
    function GetValue(Instance : TObject; PropInfo : PPropInfo) : string; virtual;
    procedure SetValue(Instance : TObject; PropInfo : PPropInfo; const sValue : string); virtual;
    function PropertyEditor(Instance : TObject; PropInfo : PPropInfo) : Boolean; virtual;
    function GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes; virtual;
  end;
  TGlobePropertyHandlerClass = class of TGlobePropertyHandler;

  TGlobeHiddenHandler = class( TGlobePropertyHandler )
  end;

  {----------------------------------------------------------------------------}
  TGlobeProperties = class
  private
    FiCount : integer;
    FInfoList : TList;
    FHandlerList : TList;
    FInstance : TPersistent;
  protected
    function GetPropInfo(iIndex : integer) : PPropInfo;
    function GetName(iIndex : integer) : string;
    function GetKind(iIndex : integer) : TTypeKind; virtual;
    function GetAttributes(iIndex : integer) : TGlobePropertyAttributes; virtual;
    function GetOrdinalProp(iIndex : integer) : LongInt;
    function GetValue(iIndex : integer) : string;
    procedure SetValue(iIndex : integer; const sValue : string);
  public
    constructor Create(Instance : TPersistent);
    destructor Destroy; override;

    function GetPropertyName(PropInfo : PPropInfo) : string;
    procedure GetPicklist(iIndex : integer; PickList : TStrings);
    function PropertyEditor(iIndex : integer) : Boolean;

    property Count : integer read FiCount;
    property PropertyInfo[iIndex : integer] : PPropInfo read GetPropInfo; default;
    property Name[iIndex : integer] : string read GetName;
    property Kind[iIndex : integer] : TTypeKind read GetKind;
    property Value[iIndex : integer] : string read GetValue write SetValue;
    property Attributes[iIndex : integer] : TGlobePropertyAttributes read GetAttributes;
    property OrdinalProp[iIndex : integer] : LongInt read GetOrdinalProp;
  end;

  {----------------------------------------------------------------------------}
  TGlobeColorHandler = class(TGlobePropertyHandler)
  public
    function GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes; override;
    procedure GetPicklist(Instance : TObject; PropInfo : PPropInfo; PickList : TStrings); override;
    function GetValue(Instance : TObject; PropInfo : PPropInfo) : string; override;
    procedure SetValue(Instance : TObject; PropInfo : PPropInfo; const sValue : string); override;
    function PropertyEditor(Instance : TObject; PropInfo : PPropInfo) : Boolean; override;
  end;

  {----------------------------------------------------------------------------}
  TProjectionClassHandler = class(TGlobePropertyHandler)
  public
    function GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes; override;
    procedure GetPicklist(Instance : TObject; PropInfo : PPropInfo; PickList : TStrings); override;
  end;

  {----------------------------------------------------------------------------}
  TGlobeSetHandler = class(TGlobePropertyHandler)
  public
    function GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes; override;
    function PropertyEditor(Instance : TObject; PropInfo : PPropInfo) : Boolean; override;
  end;

  {----------------------------------------------------------------------------}
  TGlobeUnitHandler = class(TGlobePropertyHandler)
  public
    function GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes; override;
    function PropertyEditor(Instance : TObject; PropInfo : PPropInfo) : Boolean; override;
  end;

  {----------------------------------------------------------------------------}
  TFontNameHandler = class(TGlobePropertyHandler)
  public
    function GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes; override;
    function PropertyEditor(Instance : TObject; PropInfo : PPropInfo) : Boolean; override;
  end;


procedure RegisterPropertyHandler(const sName : string; AClass : TGlobePropertyHandlerClass);

{------------------------------------------------------------------------------}
implementation

{$R *.DFM}

var
  gPropertyHandlers : TStringList;

{------------------------------------------------------------------------------
  RegisterPropertyHandler
------------------------------------------------------------------------------}
procedure RegisterPropertyHandler(const sName : string; AClass : TGlobePropertyHandlerClass);
begin
  gPropertyHandlers.AddObject(Uppercase( sName ), TObject(AClass));
end;

{------------------------------------------------------------------------------
  TGlobePropertyHandler.GetValue
------------------------------------------------------------------------------}
function TGlobePropertyHandler.GetValue(Instance : TObject; PropInfo : PPropInfo) : string;
{----------------------------------------------------------------------------}
  function GetSet(TypeInfo : PTypeInfo; iVal : integer) : string;
  var
    idx : integer;
    TypeData : PTypeData;
  begin
    Result := '';
    TypeInfo := GetTypeData(TypeInfo)^.CompType^;
    TypeData := GetTypeData(TypeInfo);
    for idx := TypeData^.MinValue to TypeData^.MaxValue do
    begin
      if (iVal and 1) <> 0 then
        Result := sIFE(Length(Result) > 0, Result + ',', '') + GetEnumName(TypeInfo, idx);
      iVal := iVal shr 1;
    end;
  end;

begin
  case PropInfo^.PropType^.Kind of
    tkClass : Result := '(' + PropInfo^.PropType^.Name + ')';
    tkLString,
      tkString : Result := GetStrProp(Instance, PropInfo);
    tkEnumeration : Result := GetEnumName(PropInfo^.PropType^, GetOrdProp(Instance, PropInfo));
    tkInteger : Result := IntToStr(GetOrdProp(Instance, PropInfo));
    tkFloat : Result := FloatToStr(GetFloatProp(Instance, PropInfo));
    tkSet : Result := '[' + GetSet(PropInfo^.PropType^, GetOrdProp(Instance, PropInfo)) + ']';
  end;
end;

{------------------------------------------------------------------------------
  TGlobePropertyHandler.SetValue
------------------------------------------------------------------------------}
procedure TGlobePropertyHandler.SetValue(Instance : TObject; PropInfo : PPropInfo; const sValue : string);
begin
  if PropInfo^.PropType^.Kind in [tkLString, tkString] then
    SetStrProp(Instance, PropInfo, sValue)
  else
    if Length(sValue) > 0 then
    try
      case PropInfo^.PropType^.Kind of
        tkInteger :
          SetOrdProp(Instance, PropInfo, StrToIntDef(sValue, 0));
        tkEnumeration :
          SetOrdProp(Instance, PropInfo, GetEnumValue(PropInfo^.PropType^, sValue));
        tkFloat :
          SetFloatProp(Instance, PropInfo, StrToFloat(sValue));
      end;
    except
    end;
end;

{------------------------------------------------------------------------------
  TGlobePropertyHandler.PropertyEditor
------------------------------------------------------------------------------}
function TGlobePropertyHandler.PropertyEditor(Instance : TObject; PropInfo : PPropInfo) : Boolean;
begin
  { Only return false if a class object property }
  Result := PropInfo^.PropType^.Kind <> tkClass;
end;

{------------------------------------------------------------------------------
  TGlobePropertyHandler.GetAttributes
------------------------------------------------------------------------------}
function TGlobePropertyHandler.GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes;
begin
  Result := [];

  case PropInfo^.PropType^.Kind of
    tkClass :
      Include(Result, paClass);
    tkEnumeration :
      Include(Result, paList);
  else
    Include(Result, paSimple);
  end;
end;

{------------------------------------------------------------------------------
  TGlobePropertyHandler.GetPicklist
------------------------------------------------------------------------------}
procedure TGlobePropertyHandler.GetPicklist(Instance : TObject; PropInfo : PPropInfo; PickList : TStrings);
var
  TypeData : PTypeData;
  idx : integer;
begin
  TypeData := GetTypeData(PropInfo^.PropType^);
  for idx := TypeData.MinValue to TypeData.MaxValue do
    PickList.Add(GetEnumName(PropInfo^.PropType^, idx));
end;

{-------------------------------------------------------------------------
 TGlobeProperties.Create
-------------------------------------------------------------------------}
constructor TGlobeProperties.Create(Instance : TPersistent);
var
  idx, jdx, iPropCount : integer;
  sPropName, sPropType : string;
  TypeData : PTypeData;
  PropertyList : PPropList;
  TypeInfo : PTypeInfo;
  PropHandler : TObject;
  PropHandlerClass : TGlobePropertyHandlerClass;
begin
  FInfoList := TList.Create;
  FHandlerList := TList.Create;
  FInstance := Instance;

  TypeInfo := Instance.ClassInfo;
  TypeData := GetTypeData(TypeInfo);
  iPropCount := TypeData^.PropCount;
  FiCount := 0;

  if iPropCount > 0 then
  begin
    GetMem(PropertyList, iPropCount * Sizeof(Pointer));
    try
      GetPropInfos(TypeInfo, PropertyList);

      for idx := 0 to iPropCount - 1 do
        if PropertyList^[idx]^.PropType^.Kind in [tkClass, tkSet, tkLString, tkString, tkEnumeration, tkInteger, tkFloat] then
        begin
          sPropName := PropertyList^[idx]^.Name;
          sPropType := PropertyList^[idx]^.PropType^.Name;

          PropHandlerClass := nil;
          { check for a specific Class.Property handler }
          jdx := gPropertyHandlers.IndexOf( Uppercase(Instance.ClassName + '.' + sPropName));
          if jdx < 0 then
            jdx := gPropertyHandlers.IndexOf( Uppercase( '.' + sPropName));
          if jdx < 0 then
            jdx := gPropertyHandlers.IndexOf( Uppercase( '.' + sPropType));
          if jdx >= 0 then
            PropHandlerClass := TGlobePropertyHandlerClass(gPropertyHandlers.Objects[jdx]);

//          for jdx := 0 to gPropertyHandlers.count - 1 do
//          begin
//            if (CompareText(gPropertyHandlers[jdx], Instance.ClassName + '.' + sPropName) = 0)
//              or (CompareText(gPropertyHandlers[jdx], '.' + sPropName) = 0) { check for a named property handler }
//            or (CompareText(gPropertyHandlers[jdx], sPropType) = 0) then { check for a specific type match }
//              PropHandler := TGlobePropertyHandlerClass(gPropertyHandlers.Objects[jdx]).Create;
//          end;

          PropHandler := nil;
          if PropHandlerClass = nil then
            PropHandler := TGlobePropertyHandler.Create
          else
            if PropHandlerClass <> TGlobeHiddenHandler then
              PropHandler := PropHandlerClass.Create;
              
          if PropHandler <> nil then
          begin
            FInfoList.Add(PropertyList^[idx]);
            FHandlerList.Add(PropHandler); { add the handler if present }
            Inc(FiCount);
          end;
        end;
    finally
      Freemem(PropertyList);
    end;
  end;
end;

{-------------------------------------------------------------------------
 TGlobeProperties.Destroy
-------------------------------------------------------------------------}
destructor TGlobeProperties.Destroy;
var
  idx : integer;
begin
  FInfoList.Free;

  for idx := 0 to FHandlerList.Count - 1 do
    TGlobePropertyHandler(FHandlerList[idx]).Free;
  FHandlerList.Free;

  inherited Destroy;
end;

{-------------------------------------------------------------------------
 TGlobeProperties.GetPropInfo
-------------------------------------------------------------------------}
function TGlobeProperties.GetPropInfo(iIndex : integer) : PPropInfo;
begin
  if (iIndex < 0) or (iIndex >= FiCount) then
    Result := nil
  else
    Result := PPropInfo(FInfoList[iIndex]);
end;

{-------------------------------------------------------------------------
 TGlobeProperties.GetName
-------------------------------------------------------------------------}
function TGlobeProperties.GetName(iIndex : integer) : string;
begin
  Result := GetPropInfo(iIndex)^.Name;
end;

{------------------------------------------------------------------------------
  TGlobeProperties.GetKind
------------------------------------------------------------------------------}
function TGlobeProperties.GetKind(iIndex : integer) : TTypeKind;
begin
  Result := GetPropInfo(iIndex)^.PropType^.Kind;
end;

{------------------------------------------------------------------------------
  TGlobeProperties.GetAttributes
------------------------------------------------------------------------------}
function TGlobeProperties.GetAttributes(iIndex : integer) : TGlobePropertyAttributes;
begin
  Result := TGlobePropertyHandler(FHandlerList[iIndex]).GetAttributes(FInstance, GetPropInfo(iIndex));
end;

{------------------------------------------------------------------------------
  TGlobeProperties.PropertyEditor
------------------------------------------------------------------------------}
function TGlobeProperties.PropertyEditor(iIndex : integer) : Boolean;
begin
  Result := TGlobePropertyHandler(FHandlerList[iIndex]).PropertyEditor(FInstance, GetPropInfo(iIndex));
end;

{------------------------------------------------------------------------------
  TGlobeProperties.GetOrdinalProp
------------------------------------------------------------------------------}
function TGlobeProperties.GetOrdinalProp(iIndex : integer) : LongInt;
begin
  Result := GetOrdProp(FInstance, GetPropInfo(iIndex));
end;

{-------------------------------------------------------------------------
 TGlobeProperties.GetValue
-------------------------------------------------------------------------}
function TGlobeProperties.GetValue(iIndex : integer) : string;
begin
  Result := TGlobePropertyHandler(FHandlerList[iIndex]).GetValue(FInstance, GetPropInfo(iIndex));
end;

{-------------------------------------------------------------------------
 TGlobeProperties.SetValue
-------------------------------------------------------------------------}
procedure TGlobeProperties.SetValue(iIndex : integer; const sValue : string);
var
  PropInfo : PPropInfo;
begin
  PropInfo := GetPropInfo(iIndex);

  { ignore read only properties }
  if PropInfo^.SetProc = nil then
    Exit;

  { update the property }
  TGlobePropertyHandler(FHandlerList[iIndex]).SetValue(FInstance, PropInfo, sValue);
end;

{-------------------------------------------------------------------------
 TGlobeProperties.GetPicklist
-------------------------------------------------------------------------}
procedure TGlobeProperties.GetPicklist(iIndex : integer; PickList : TStrings);
begin
  TGlobePropertyHandler(FHandlerList[iIndex]).GetPickList(FInstance, GetPropInfo(iIndex), PickList)
end;

{-------------------------------------------------------------------------
 TGlobeProperties.GetPropertyName
-------------------------------------------------------------------------}
function TGlobeProperties.GetPropertyName(PropInfo : PPropInfo) : string;
begin
  Result := PropInfo^.Name;
end;

{------------------------------------------------------------------------------
  TGlobeColorHandler.PropertyEditor
------------------------------------------------------------------------------}
function TGlobeColorHandler.PropertyEditor(Instance : TObject; PropInfo : PPropInfo) : Boolean;
begin
  with TColorDialog.Create(Application) do
  begin
    Color := GetOrdProp(Instance, PropInfo);
    if Execute then
      SetOrdProp(Instance, PropInfo, Color);
    Free;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  TGlobeColorHandler.GetAttributes
------------------------------------------------------------------------------}
function TGlobeColorHandler.GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes;
begin
  Result := [paList];
end;

{------------------------------------------------------------------------------
  TGlobeColorHandler.GetPicklist
------------------------------------------------------------------------------}
procedure TGlobeColorHandler.GetPicklist(Instance : TObject; PropInfo : PPropInfo; PickList : TStrings);
begin
  with PickList do
  begin
    Add('clBlack');
    Add('clMaroon');
    Add('clGreen');
    Add('clOlive');
    Add('clNavy');
    Add('clPurple');
    Add('clTeal');
    Add('clGray');
    Add('clSilver');
    Add('clRed');
    Add('clLime');
    Add('clYellow');
    Add('clBlue');
    Add('clFuchsia');
    Add('clAqua');
    Add('clWhite');
  end;
end;

{------------------------------------------------------------------------------
  TGlobeColorHandler.GetValue
------------------------------------------------------------------------------}
function TGlobeColorHandler.GetValue(Instance : TObject; PropInfo : PPropInfo) : string;
begin
  Result := ColorToString(TColor(GetOrdProp(Instance, PropInfo)));
end;

{------------------------------------------------------------------------------
  TGlobeColorHandler.SetValue
------------------------------------------------------------------------------}
procedure TGlobeColorHandler.SetValue(Instance : TObject; PropInfo : PPropInfo; const sValue : string);
var
  Color : integer;
begin
  try
    if IdentToColor(sValue, Color) then
      SetOrdProp(Instance, PropInfo, Color)
    else
      SetOrdProp(Instance, PropInfo, StrToInt(sValue));
  except
  end;
end;

{------------------------------------------------------------------------------
  TProjectionClassHandler.GetAttributes
------------------------------------------------------------------------------}
function TProjectionClassHandler.GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes;
begin
  Result := [paList];
end;

{------------------------------------------------------------------------------
  TProjectionClassHandler.GetPicklist
------------------------------------------------------------------------------}
procedure TProjectionClassHandler.GetPicklist(Instance : TObject; PropInfo : PPropInfo; PickList : TStrings);
begin
  PickList.Add('TSphericalPrj');
  PickList.Add('TMercatorPrj');
  PickList.Add('TCartesianPrj');
end;

{------------------------------------------------------------------------------
  TGlobeSetHandler.GetAttributes
------------------------------------------------------------------------------}
function TGlobeSetHandler.GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes;
begin
  Result := [paClass];
end;

{------------------------------------------------------------------------------
  TGlobeSetHandler.PropertyEditor
------------------------------------------------------------------------------}
function TGlobeSetHandler.PropertyEditor(Instance : TObject; PropInfo : PPropInfo) : Boolean;
var
  idx, iVal : integer;
  TypeData : PTypeData;
  TypeInfo : PTypeInfo;
begin
  with TPropertyForm.Create(Application) do
  begin
    Caption := 'Select set members';
    Notebook1.ActivePage := 'Sets';

    iVal := GetOrdProp(Instance, PropInfo);

    TypeInfo := GetTypeData(PropInfo^.PropType^)^.CompType^;
    TypeData := GetTypeData(TypeInfo);
    for idx := TypeData^.MinValue to TypeData^.MaxValue do
    begin
      lbSetMembers.Items.Add(GetEnumName(TypeInfo, idx));
      lbSetMembers.Checked[idx] := (iVal and 1) <> 0;
      iVal := iVal shr 1;
    end;

    if ShowModal = mrOK then
    begin
      iVal := 0;
      for idx := 0 to lbSetMembers.Items.Count - 1 do
        iVal := iVal or (Ord(lbSetMembers.Checked[idx]) shl idx);
      SetOrdProp(Instance, PropInfo, iVal);
    end;
    Free;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  TGlobeUnitHandler.GetAttributes
------------------------------------------------------------------------------}
function TGlobeUnitHandler.GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes;
begin
  Result := [paClass];
end;

{------------------------------------------------------------------------------
  TGlobeUnitHandler.PropertyEditor
------------------------------------------------------------------------------}
function TGlobeUnitHandler.PropertyEditor(Instance : TObject; PropInfo : PPropInfo) : Boolean;
var
  iGU : integer;
begin
  with TPropertyForm.Create(Application) do
  begin
    Caption := 'Set GlobeUnits';
    Notebook1.ActivePage := 'GlobeUnits';
    iGU := GetOrdProp(Instance, PropInfo);
    edtGU.Text := IntTostr(iGU);

    if ShowModal = mrOK then
      SetOrdProp(Instance, PropInfo, StrToIntDef(edtGU.Text, iGU));
    Free;
  end;
  Result := True;
end;

{------------------------------------------------------------------------------
  TFontNameHandler.GetAttributes
------------------------------------------------------------------------------}
function TFontNameHandler.GetAttributes(Instance : TObject; PropInfo : PPropInfo) : TGlobePropertyAttributes;
begin
  Result := [paClass];
end;

{------------------------------------------------------------------------------
  TFontNameHandler.PropertyEditor
------------------------------------------------------------------------------}
function TFontNameHandler.PropertyEditor(Instance : TObject; PropInfo : PPropInfo) : Boolean;
var
  sFontName : string;
begin
  with TPropertyForm.Create(Application) do
    with lstFonts do
    begin
      Caption := 'Set Font Name';
      Notebook1.ActivePage := 'FontName';
      Items.assign(Screen.Fonts);

      sFontName := GetStrProp(Instance, PropInfo);

      ItemIndex := Items.IndexOf(sFontName);
      if ItemIndex = -1 then
        ItemIndex := Items.IndexOf('Arial');

      if ShowModal = mrOK then
        if ItemIndex >= 0 then
          SetStrProp(Instance, PropInfo, Items[ItemIndex]);
      Free;
    end;
  Result := True;
end;

{------------------------------------------------------------------------------
  TPropertyForm.edtGUChange
------------------------------------------------------------------------------}
procedure TPropertyForm.edtGUChange(Sender : TObject);
var
  iGU : integer;
begin
  iGU := StrToIntDef(edtGU.Text, 0);
  if not edtDD.Modified then
    edtDD.Text := FloatToStr(Round(GlobeUnitsToDecimal(iGU) * 100) / 100);
  if not edtNM.Modified then
    edtNM.Text := FloatToStr(Round(GlobeUnitsTo(iGU, NauticalMile) * 100) / 100);
  if not edtKM.Modified then
    edtKM.Text := FloatToStr(Round(GlobeUnitsTo(iGU, Kilometer) * 100) / 100);
end;

{------------------------------------------------------------------------------
  TPropertyForm.edtDDChange
------------------------------------------------------------------------------}
procedure TPropertyForm.edtValueChange(Sender : TObject);
var
  iGU : integer;
begin
  with Sender as TEdit do
  begin
    if Length(Text) > 0 then
    try
      case Tag of
        0 : { edtDD }
          iGU := DecimalToGlobeUnits(StrToFloat(Text));
        1 : { edtNM }
          iGU := GlobeUnitsFrom(StrToFloat(Text), NauticalMile);
        2 : { edtKM }
          iGU := GlobeUnitsFrom(StrToFloat(Text), Kilometer);
      else
        iGU := 0; { Should never happen }
      end;
      edtGU.Text := IntToStr(iGU);
      lblException.Caption := '';
    except
      on E : Exception do
        lblException.Caption := E.Message;
    end;
    Modified := False;
  end;
end;

initialization
  gPropertyHandlers := TStringList.Create;
  gPropertyHandlers.Sorted := True;

  RegisterPropertyHandler('TColor', TGlobeColorHandler);
  RegisterPropertyHandler('TFontStyles', TGlobeSetHandler);
  RegisterPropertyHandler('.XRotation', TGlobeUnitHandler);
  RegisterPropertyHandler('.YRotation', TGlobeUnitHandler);
  RegisterPropertyHandler('.ZRotation', TGlobeUnitHandler);
  RegisterPropertyHandler('.CentralMeridian', TGlobeUnitHandler);
  RegisterPropertyHandler('.FirstParallel', TGlobeUnitHandler);
  RegisterPropertyHandler('.SecondParallel', TGlobeUnitHandler);
  RegisterPropertyHandler('.FontName', TFontNameHandler);
  RegisterPropertyHandler('.ProjectionClass', TProjectionClassHandler);

  RegisterPropertyHandler('.Tag', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.Name', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.CacheCapacity', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.GlobeOptions', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.PeriodicUpdateTime', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.Left', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.Top', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.Height', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.Width', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.Cursor', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.Hint', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.HelpContext', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.Align', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.Anchors', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.DragCursor', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.DragMode', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.Enabled', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.ParentShowHint', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.PopupMenu', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.ScrollBars', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.ShowHint', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.Visible', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.MouseZoomButton', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.MouseSelectButton', TGlobeHiddenHandler);
  RegisterPropertyHandler('TGlobe4.MouseRotateButton', TGlobeHiddenHandler);
finalization
  gPropertyHandlers.Free;
end.

