{------------------------------------------------------------------------------
	Module:   conversions.pas

	Comment:  Some donated conversion routines to convert coordinates.
                I do not know if they work or if they are accurate since
                they have come from the public domain.
                Let me have any comments / corrections / fixes that you
                make so that I can update this file. - Graham Knight.
                     Email: gknight@helmstone.co.uk


      { Converts (latitude, longitude) WGS-84 coordinates to (X,Y) RD-coordinates. }
      procedure LLToRD(y_rd,x_rd:Real;Var East,North:Real);

      { Converts (latitude, longitude) WGS-84 coordinates to (X,Y) United Kingdom 
         National Grid Reference - coordinates. }
      procedure LLToNGR(Long,Lat:Real;Var East,North:Real);

      { Converts (latitude, longitude) WGS-84 coordinates to (X,Y) UTM-coordinates. }
      procedure LLToUTM(Long,Lat:Real;Var East,North:Real);
------------------------------------------------------------------------------}


{------------------------------------------------------------------------------
  procedure LLToRD
------------------------------------------------------------------------------}
procedure LLToRD(y_rd,x_rd:Real;Var East,North:Real);

   // This function converts the (latitude, longitude) WGS-84 coordinates
   // to (X,Y) RD-coordinates
Var
   phi_wgs, lambda_wgs:Double;
   a_wgs, b_wgs, e_2_wgs, N_wgs:Double;
   x_wgs, y_wgs, z_wgs:Double;
   h_wgs:Double;

   phi0_bessel, lambda0_bessel, dphi_bessel, dlambda_bessel:Double;
   a_bessel, b_bessel, e_2_bessel:Double;
   x_bessel, y_bessel, z_bessel:Double;
   phi_bessel, lambda_bessel:Double;

   x0_rd, y0_rd, dx_rd, dy_rd:Double;
   tx, ty, tz, alpha, beta, gamma, delta:Double;

   w2, p, q, t, r2, last_t, error:Double;

   c01, c03, c05, c11, c13, c21, c23, c31, c41:Double;
   d02, d04, d10, d12, d14, d20, d22, d30, d32, d40:Double;

Const
   eps = 1e-15;
Begin
   // Origin of RD-system is in Amersfoort */
   // x0_rd := 155000.00;                                   // in meters */
   // y0_rd := 463000.00;                                   // in meters */

   // Conversion parameters from latitude- and longitude-coordinates
   // (Bessel-ellipsoid) to X and Y-coordinates (RD) */

   c01:=190066.98903;
   c03:=-32.38360;
   c05:=-0.00661;
   c11:=-11830.85831;
   c13:=-0.60639;
   c21:=-114.19754;
   c23:=0.15774;
   c31:=-2.34078;
   c41:=-0.04158;

   d02:=3638.36193;
   d04:=0.09351;
   d10:=309020.31810;
   d12:=-157.95222;
   d14:=-0.05419;
   d20:=72.97141;
   d22:=-6.43481;
   d30:=59.79734;
   d32:=-0.07379;
   d40:=-0.03444;

   // Parameters WGS-84 ellipsoid */
   h_wgs := 0.0;
   a_wgs := 6378137.00;         // equatorial radius of the earth (WGS-84) */
   b_wgs := 6356752.314;                            // polar axis (WGS-84) */
   e_2_wgs := ( a_wgs*a_wgs - b_wgs*b_wgs )/(a_wgs*a_wgs);

   // Parameters Bessel ellipsoid */
   a_bessel := 6377397.155;    // equatorial radius of the earth (Bessel) */
   b_bessel := 6356078.963;                        // polar axis (Bessel) */
   e_2_bessel := (a_bessel*a_bessel - b_bessel*b_bessel)/(a_bessel*a_bessel);

   // Transformation parameters from WGS-84 (X,Y,Z) to Bessel (X,Y,Z) */
   // reference system                                                */
   tx := -577.57;                                       // translation */
   ty := -14.45;
   tz := -458.65;

   alpha := 6.3238e-6;                                     // rotation */
   beta := 5.0083e-7;
   gamma := 5.5487e-6;
   delta := 1;                                         // scale factor */

   // Origin of RD-system is in Amersfoort */
   x0_rd := 155000.00;                       // in meters */
   y0_rd := 463000.00;                       // in meters */

   phi0_bessel := 187762.178;                // in seconds */
   lambda0_bessel := 19395.500;              // in seconds */

   // Conversion from latitude and longitude (WGS-84) to X,Y,Z (WGS-84) */
   phi_wgs := x_rd;
   lambda_wgs := y_rd;

   N_wgs := a_wgs/(sqrt (1-e_2_wgs*sin(phi_wgs*pi/180)*sin(phi_wgs*pi/180)) );
   x_wgs := (N_wgs+h_wgs)*cos(phi_wgs*pi/180)*cos(lambda_wgs*pi/180);
   y_wgs := (N_wgs+h_wgs)*cos(phi_wgs*pi/180)*sin(lambda_wgs*pi/180);
   z_wgs := (N_wgs*(1-e_2_wgs)+h_wgs)*sin(phi_wgs*pi/180);

   // Transformation from WGS-84 to Bessel reference system */
   x_bessel := tx + delta*x_wgs - gamma*y_wgs + beta*z_wgs;
   y_bessel := ty + gamma*x_wgs + delta*y_wgs - alpha*z_wgs;
   z_bessel := tz - beta*x_wgs + alpha*y_wgs + delta*z_wgs;

   // Conversion from X,Y,Z (Bessel) to latitude, longitude (Bessel) */
   lambda_bessel := arctan(y_bessel/x_bessel);
   p:=z_bessel/( (1-e_2_bessel)*sqrt(x_bessel*x_bessel+y_bessel*y_bessel));
   q:=e_2_bessel/((1-e_2_bessel)*sqrt(x_bessel*x_bessel+y_bessel*y_bessel));
   t:=0.0;
   Repeat
       phi_bessel:=arctan(p-q*t);
       w2:=sqrt(1-e_2_bessel*sin(phi_bessel)*sin(phi_bessel));
       r2:=a_bessel/w2;
       last_t := t;
       t:=z_bessel-r2*(1-e_2_bessel)*sin(phi_bessel);
       error := abs(t-last_t)-eps;
   Until (error <= 0);

   phi_bessel:=phi_bessel*180/pi;
   lambda_bessel:=lambda_bessel*180/pi;

   // Conversion from latitude, longitude (Bessel) reference system to
   // X,Y (RD) reference system */

   dphi_bessel := ( (phi_bessel)*3600-phi0_bessel)*1e-4;
   dlambda_bessel := ( (lambda_bessel)*3600-lambda0_bessel)*1e-4;

   dx_rd := c01*dlambda_bessel+c11*dphi_bessel*dlambda_bessel
    + c21*dphi_bessel*dphi_bessel*dlambda_bessel
    + c03*dlambda_bessel*dlambda_bessel*dlambda_bessel
    + c31*dphi_bessel*dphi_bessel*dphi_bessel*dlambda_bessel
    + c13*dphi_bessel*dlambda_bessel*dlambda_bessel*dlambda_bessel
    + c23*dphi_bessel*dphi_bessel*dlambda_bessel*dlambda_bessel*dlambda_bessel
    + c41*dphi_bessel*dphi_bessel*dphi_bessel*dphi_bessel*dlambda_bessel
    + c05*dlambda_bessel*dlambda_bessel*dlambda_bessel*dlambda_bessel*dlambda_bessel;

   dy_rd := d10*dphi_bessel+d20*dphi_bessel*dphi_bessel
    + d02*dlambda_bessel*dlambda_bessel
    + d12*dphi_bessel*dlambda_bessel*dlambda_bessel
    + d30*dphi_bessel*dphi_bessel*dphi_bessel
    + d22*dphi_bessel*dphi_bessel*dlambda_bessel*dlambda_bessel
    + d40*dphi_bessel*dphi_bessel*dphi_bessel*dphi_bessel
    + d04*dlambda_bessel*dlambda_bessel*dlambda_bessel*dlambda_bessel
    + d32*dphi_bessel*dphi_bessel*dphi_bessel*dlambda_bessel*dlambda_bessel
    + d14*dphi_bessel*dlambda_bessel*dlambda_bessel*dlambda_bessel*dlambda_bessel;

   East := (x0_rd + dx_rd) / 1000.0;
   North := (y0_rd + dy_rd) / 1000.0;
End;





{------------------------------------------------------------------------------
  procedure LLToNGR
------------------------------------------------------------------------------}
procedure LLToNGR(Long,Lat:Real;Var East,North:Real);
Const
    A1=6375020.481;
    B1=6353722.490;
    K0=49.0*pi/180.0;
    L0=-2.0*pi/180.0;
    N1=0.00167322025;
    E2=0.006670539762;
    E0=400000.0;
    N0=-100000.0;
Var
    K,L,SK,CK,TK,J3,J4,J5,J6,J7,J8,J9:Real;
    K3,K4,M,P,R,V,H2:Real;
Begin
    K:=Lat*pi/180.0;
    L:=Long*pi/180.0;
    K3:=K-K0;K4:=K+K0;
    //The next bit is in-line (arcmed)
    J3:=(1+N1+5/4*power(N1,2)+5/4*power(N1,3))*K3;
    J4:=(3*N1+3*power(N1,2)*21/8*power(N1,3))*sin(K3)*cos(K4);
    J5:=(15.0/8.0*power(N1,2)+15.0/8.0*power(N1,3))*sin(2*K3)*cos(2*K4);
    J6:=35.0/24.0*power(N1,3)*sin(3*K3)*cos(3*K4);
    M:=B1*(J3-J4+J5-J6);
    //The next bit is in line (nu)
    V:=A1/sqrt(1-E2*power(sin(K),2));
    R:=V*(1.0-E2)/(1.0-E2*power(sin(K),2));
    H2:=V/R-1;
    //Now back to (LLToNGR)
    P:=L-L0;
    J3:=M+N0;
    SK:=sin(K);
    CK:=cos(K);
    TK:=SK/CK;
    J4:=V/2.0*SK*CK;
    J5:=V/24.0*SK*power(CK,3)*(5-TK*TK+9.0*H2);
    J6:=V/720.0*SK*power(CK,5)*(61.0-58.0*power(TK,2)+power(TK,4));
    North:=(J3+power(P,2)*J4+power(P,4)*J5+power(P,6)*J6)/1000.0;
    J7:=V*CK;
    J8:=V/6.0*power(CK,3)*(V/R-power(TK,2));
    J9:=V/120.0*power(CK,5);
    J9:=J9*(5-18*power(TK,2)+power(TK,4)+14*H2-58*power(TK,2)*H2);
    East:=(E0+P*J7+power(P,3)*J8+power(P,5)*J9)/1000.0;
End;






{------------------------------------------------------------------------------
  procedure LLToUTM
------------------------------------------------------------------------------}
procedure LLToUTM(Long,Lat:Real;Var East,North:Real);
const
    F0=0.9996;
    A1=6378388.000*F0;
    B1=6356911.946*F0;
    K0=0;
//    Merid=15.0; (malta !!!)
    Merid=3.0;    {france !}
    N0=0;
    E0=500000;
var
    K,L,SINK,COSK,TANK,TANK2,COSK2,COSK3,K3,K4:Real;
    N1,N2,N3,E2,L0:Real;
    J3,J4,J5,J6,J7,J8,J9,M,Temp,V,R,H2:Real;
    P,P2,P4:Real;
begin
    L0:=DegtoRad(Merid); // Long of True Origin (3,9,15 etc)

    K:=DegtoRad(Lat);
    L:=DegtoRad(Long);
    SINK:=sin(K);
    COSK:=cos(K);
    TANK:=SINK/COSK;
    TANK2:=TANK*TANK;
    COSK2:=COSK*COSK;
    COSK3:=COSK2*COSK;
    K3:=K-K0;
    K4:=K+K0;

    N1:=(A1-B1)/(A1+B1); // n
    N2:=N1*N1;
    N3:=N2*N1;
    E2:=((A1*A1)-(B1*B1))/(A1*A1); // e^2

    // ArcofMeridian
    J3:=K3*(1+N1+1.25*(N2+N3));
    J4:=sin(K3)*cos(K4)*(3*(N1+N2+0.875*N3));
    J5:=sin(2*K3)*cos(2*K4)*(1.875*(N2+N3));
    J6:=sin(3*K3)*cos(3*K4)*35/24*N3;
    M:=(J3-J4+J5-J6)*B1;

    // VRH2
    Temp:=1-E2*SINK*SINK;
    V:=A1/sqrt(Temp);
    R:=V*(1-E2)/Temp;
    H2:=V/R-1.0;

    P:=L-L0;
    P2:=P*P;
    P4:=P2*P2;
    J3:=M+N0;
    J4:=V/2*SINK*COSK;
    J5:=V/24*SINK*(COSK3)*(5-(TANK2)+9*H2);
    J6:=V/720*SINK*COSK3*COSK2*(61-58*(TANK2)+TANK2*TANK2);
    North:=J3+P2*J4+P4*J5+P4*P2*J6;
    //if (((Area=='UTM1') || (Area=='UTM2')) && South) North=North+10000000.0; // UTM S hemisphere
    J7:=V*COSK;
    J8:=V/6*COSK3*(V/R-TANK2);
    J9:=V/120*COSK3*COSK2;
    J9:=J9*(5-18*TANK2+TANK2*TANK2+14*H2-58*TANK2*H2);
    East:=E0+P*J7+P2*P*J8+P4*P*J9;
end;
