unit UTM;

interface

{-----------------------------------------------------------------------------
  Geographic angles are given in degrees (floating point).
  Zones are given in zone number with sign:
      positive sign to indicate northern hemisphere
      negative sign to indicate southern hemisphere.
  Zonewidth may be changed by calling ChangeZoneWidth with parameters:
      ZoneWidth in degrees
      ScaleFactor related with the new width
          (could/should be calculated automaticly - next version :-) ).
  GetNearestZone is used to find the nearest UTM zone.
  ------
  Some of the routines support northern and southern hemisphere, but some
      still need to be checked and changed if needed.
------------------------------------------------------------------------------}


function DegreesToStr(AnAngle: extended;
                      ShowSeconds: boolean; FractionDigits: integer): string;
function StrToDegrees(AnAngle: string): extended;

procedure ChangeZoneWidth(AZoneWidth, AScaleFactor: extended);
function  GetNearestZone(Latitude, Longitude: extended): integer;

function UTMToLatitude(Easting, Northing: extended; Zone: integer): extended;
function UTMToLongitude(Easting, Northing: extended; Zone: integer): extended;
function GeoToEasting(Latitude, Longitude: extended; Zone: integer): extended;
function GeoToNorthing(Latitude, Longitude: extended; Zone: integer): extended;

procedure UTMToGeo(Easting, Northing: extended; Zone: integer;
                   var Latitude, Longitude: extended);
procedure GeoToUTM(Latitude, Longitude: extended; Zone: integer;
                   var Easting, Northing: extended);


implementation

uses
  SysUtils, Math;


procedure InternalUTMToGeo(Easting, Northing: extended; Zone: integer;
                           var Latitude, Longitude: extended;
                           CalcLatitude, CalcLongitude: boolean); forward;

procedure InternalGeoToUTM(Latitude, Longitude: extended; Zone: integer;
                           var Easting, Northing: extended;
                           CalcEasting, CalcNorthing: boolean); forward;

{-----------------------------------------------------------------------------}

const
  FalseNorthing = 10000000.0;
  FalseEasting = 500000.0;
  ToRad = Pi / 180.0;
  ToDeg = 180.0 / Pi;
  F = 1 / 297.0;
  A = 6378388.0;

  ZoneWidth: extended = 6.0; {- degrees zone-width -}
  K0: extended = 0.9996;     {- scalefactor at central meridian -}


var
  K0_3, K0_5, K0_7,
  B, E, E_2, E_4, E_6,
  Edash, C: extended;



{-----------------------------------------------------------------------------}

function DegreesToStr(AnAngle: extended;
                      ShowSeconds: boolean; FractionDigits: integer): string;
var
  Deg, Min, Sec: integer;
  Fract, Width: string;
begin
  Deg := Trunc(AnAngle);
  AnAngle := Frac(AnAngle) * 60;
  Min := Trunc(AnAngle);
  if ShowSeconds then begin
    AnAngle := Frac(AnAngle) * 60;
    Sec := Trunc(AnAngle);
  end else
    Sec := 0;

  if FractionDigits > 0 then begin
    Width := IntToStr(FractionDigits);
    AnAngle := Frac(AnAngle) * IntPower(10, FractionDigits);
    Fract := Format(DecimalSeparator + '%' + Width + '.' + Width + 'd', [Trunc(AnAngle)]);
  end else
    Fract := '';

  if ShowSeconds then
    Result := Format('%d%2.2d''%2.2d' + Fract + '"', [Deg, Min, Sec])
  else
    Result := Format('%d%2.2d' + Fract + '''', [Deg, Min]);
end;


function StrToDegrees(AnAngle: string): extended;
var
  i: integer;
  Deg, Min, Sec: extended;
  Neg: boolean;

  function GetNumber(var ANum: string): extended;
  var
    i: integer;
  begin
    i := 0;
    while (i < Length(ANum)) and (ANum[i + 1] in ['0'..'9', DecimalSeparator]) do
      Inc(i);
    if i > 0 then begin
      Result := StrToFloat(Copy(ANum, 1, i));
      Delete(ANum, 1, i);
    end else
      Result := 0;
  end;

begin
  i := 1;
  while (i <= Length(AnAngle)) do begin
    if AnAngle[i] = ' ' then
      Delete(AnAngle, i, 1)
    else
      Inc(i);
  end;

  Neg := (Copy(AnAngle, 1, 1) = '-');
  if Neg then
    Delete(AnAngle, 1, 1);

  Deg := GetNumber(AnAngle);
  if (Length(AnAngle) = 0) or (not (AnAngle[1] in ['', ThousandSeparator])) then
    raise Exception.Create('Error converting angle');
  Delete(AnAngle, 1, 1);

  Min := GetNumber(AnAngle) / 60;
  if (Length(AnAngle) > 0) then begin
    if not (AnAngle[1] in ['''', ThousandSeparator]) then
      raise Exception.Create('Error converting angle');
    Delete(AnAngle, 1, 1);
  end;

  if Length(AnAngle) > 0 then begin
    Sec := GetNumber(AnAngle) / 3600;
    if (Length(AnAngle) > 0) or (AnAngle[1] <> '"') then
      Delete(AnAngle, 1, 1);
  end else
    Sec := 0;

  if Length(AnAngle) <> 0 then
    raise Exception.Create('Error converting angle');

  Result := Deg + Min + Sec;
  if Neg then
    Result := -Result;
end;


procedure ChangeZoneWidth(AZoneWidth, AScaleFactor: extended);
begin
  ZoneWidth := AZoneWidth;
  K0 := AScaleFactor;
  K0_3 := IntPower(K0, 3);
  K0_5 := IntPower(K0, 5);
  K0_7 := IntPower(K0, 7);
  B := A * (1 - F);
  E := Sqrt((2 * F) - IntPower(F, 2));
  E_2 := IntPower(E, 2);
  E_4 := IntPower(E_2, 2);
  E_6 := E_2 * E_4;
  Edash := Sqrt((IntPower(A, 2) - IntPower(B, 2)) / IntPower(B, 2));
  C := A / Sqrt(1 - IntPower(E, 2));
end;


{-----------------------------------------------------------------------------}


function AddFalseNorthing(Northing: extended; AZone: integer): extended;
begin
  if AZone < 0 then
    Result := FalseNorthing - Northing
  else
    Result := Northing;
end;


function AddFalseEasting(Easting: extended): extended;
begin
  Result := Easting + FalseEasting;
end;


function RemoveFalseNorthing(Northing: extended; AZone: integer): extended;
{--- Same as add false northing ---}
begin
  if AZone < 0 then
    Result := FalseNorthing - Northing
  else
    Result := Northing;
end;


function RemoveFalseEasting(Easting: extended): extended;
begin
  Result := Easting - FalseEasting;
end;


function CalcCentralMeridian(Zone: integer): extended;
begin
  Result := -183 + ZoneWidth * Abs(Zone);
end;


function GetNearestZone(Latitude, Longitude: extended): integer;
begin
  Result := Round( (Longitude + 183.0) / ZoneWidth );
  if Latitude < 0 then
    Result := -Result;
end;


procedure CalcPsiRhoNu(Phi: extended; var Psi, Rho, Nu: extended);
var
  SqrPsi: extended;
begin
  Psi := 1 + IntPower(Edash, 2) * IntPower(Cos(Phi * ToRad), 2);
  SqrPsi := Sqrt(Psi);
  Rho := C / IntPower(SqrPsi, 3);
  Nu := C / SqrPsi;
end;


function CalcMeridian(Latitude: extended): extended;
var
  A0, A2, A4, A6: extended;
begin
  A0 := 1 - (E_2 / 4) - (3 * E_4 / 64) - (5 * E_6 / 256);
  A2 := 3 / 8 * (E_2 + (E_4 / 4) + (15 * E_6 / 128));
  A4 := 15 / 256 * (E_4 + (3 * E_6 / 4));
  A6 := 35 * E_6 / 3072;
  Result := A * (A0 * Latitude * ToRad - A2 * Sin(ToRad * 2 * Latitude) +
                 A4 * Sin(ToRad * 4 * Latitude) - A6 * Sin(ToRad * 6 * Latitude));
end;


function CalcPhi(Northing: extended): extended;
var
  Lat,
  Psi, Rho, Nu,
  R, M, M1: extended;
begin
  Lat := Northing * 9.002576 * 1E-6;
  CalcPsiRhoNu(Lat, Psi, Rho, Nu);

  R := Sqrt(Rho * Nu);
  M1 := Northing / K0;
  repeat
    M := CalcMeridian(Lat);
    if Abs(M - M1) >= 0.0005 then
      Lat := Lat + ((M1 - M) / R) * ToDeg;
  until Abs(M1 - M) < 0.0005;
  Result := Lat;
end;


{-----------------------------------------------------------------------------}
procedure InternalUTMToGeo(Easting, Northing: extended; Zone: integer;
                           var Latitude, Longitude: extended;
                           CalcLatitude, CalcLongitude: boolean);
var
  Easting_2, Easting_3, Easting_4, Easting_5, Easting_6, Easting_7, Easting_8,
  Psi, Phi, Rho, Nu,
  InvCosPhi, T, T_2, T_4, T_6,
  Nu_3, Nu_5, Nu_7,
  Psi_2, Psi_3, Psi_4,
  TK0Rho,
  L1, L2, L3, L4: extended;
begin
  Northing := RemoveFalseNorthing(Northing, Zone);
  Easting := RemoveFalseEasting(Easting);

  Easting_2 := IntPower(Easting, 2);
  Easting_3 := Easting * Easting_2;
  Easting_4 := IntPower(Easting_2, 2);
  Easting_5 := Easting * Easting_4;
  Easting_6 := Easting_2 * Easting_4;
  Easting_7 := Easting * Easting_6;
  Easting_8 := IntPower(Easting_4, 2);

  Phi := CalcPhi(Northing);
  InvCosPhi := 1 / Cos(ToRad * Phi);
  T := Tan(ToRad * Phi);
  T_2 := IntPower(T, 2);
  T_4 := IntPower(T_2, 2);
  T_6 := T_2 * T_4;

  CalcPsiRhoNu(Phi, Psi, Rho, Nu);
  Nu_3 := IntPower(Nu, 3);
  Nu_5 := IntPower(Nu, 5);
  Nu_7 := IntPower(Nu, 7);
  Psi_2 := IntPower(Psi, 2);
  Psi_3 := IntPower(Psi, 3);
  Psi_4 := IntPower(Psi, 4);

  TK0Rho := T / (K0 * Rho);

  if CalcLatitude then begin
    L1 := TK0Rho * (Easting_2 / (2 * K0 * Nu));
    L2 := TK0Rho
          * (Easting_4 / (24 * K0_3 * Nu_3))
          * (-4 * Psi_2 + 9 * Psi * (1 - T_2) + 12 * T_2);
    L3 := TK0Rho * (Easting_6 / (720 * K0_5 * Nu_5)) *
          (8 * Psi_4 * (11 - 24 * T_2)
           - 12 * Psi_3 * (21 - 71 * T_2)
           + 15 * Psi_2 * (15 - 98 * T_2 + 15 * T_4)
           + 180 * Psi * (5 * T_2 - 3 * T_4) + 360 * T_4);
    L4 := TK0Rho
          * (Easting_8 / (40320 * K0_7 * Nu_7))
          * (1385 + 3633 * T_2 + 4095 * T_4 + 1575 * T_6);
    Latitude := ToDeg * ((Phi * ToRad) - L1 + L2 - L3 + L4);
  end;

  if CalcLongitude then begin
    L1 := InvCosPhi * (Easting / (K0 * Nu));
    L2 := InvCosPhi * (Easting_3 / (6 * K0_3 * Nu_3)) * (Psi + 2 * T_2);
    L3 := InvCosPhi * (Easting_5 / (120 * K0_5 * Nu_5))
          * (-4 * Psi_3 * (1 - 6 * T_2)
             + (Psi_2 * (9 - 68 * T_2))
             + 72 * Psi * T_2
             + 24 * T_4);
    L4 := InvCosPhi * (Easting_7 / (5040 * K0_7 * Nu_7))
          * (61 + 662 * T_2 + 1320 * T_4 + 720 * T_6);

    Longitude := CalcCentralMeridian(Zone) + ((L1 - L2 + L3 - L4) * ToDeg);
  end;
end;



function UTMToLatitude(Easting, Northing: extended; Zone: integer): extended;
var
  Lat, Long: extended;
begin
  InternalUTMToGeo(Easting, Northing, Zone, Lat, Long, True, False);
  Result := Lat;
end;


function UTMToLongitude(Easting, Northing: extended; Zone: integer): extended;
var
  Lat, Long: extended;
begin
  InternalUTMToGeo(Easting, Northing, Zone, Lat, Long, False, True);
  Result := Long;
end;


procedure UTMToGeo(Easting, Northing: extended; Zone: integer;
                   var Latitude, Longitude: extended);
begin
  InternalUTMToGeo(Easting, Northing, Zone, Latitude, Longitude, True, True);
end;



{-----------------------------------------------------------------------------}
procedure InternalGeoToUTM(Latitude, Longitude: extended; Zone: integer;
                           var Easting, Northing: extended;
                           CalcEasting, CalcNorthing: boolean);
var
  Psi, Rho, Nu,
  NuSinLat,
  CosLat, CosLat_3, CosLat_5, CosLat_7,
  Psi_2, Psi_3, Psi_4,
  E1, E2, E3, E4, N1, N2, N3, N4,
  W, W_2, W_3, W_4, W_5, W_6, W_7, W_8,
  T, T_2, T_4, T_6: extended;
begin
  CalcPsiRhoNu(Latitude, Psi, Rho, Nu);
  Psi_2 := IntPower(Psi, 2);
  Psi_3 := Psi * Psi_2;
  Psi_4 := IntPower(Psi_2, 2);

  W := (Longitude - CalcCentralMeridian(Zone)) * ToRad;
  W_2 := IntPower(W, 2);
  W_3 := W * W_2;
  W_4 := IntPower(W_2, 2);
  W_5 := W * W_4;
  W_6 := W_2 * W_4;
  W_7 := W * W_6;
  W_8 := IntPower(W_4, 2);

  T := Tan(ToRad * Latitude);
  T_2 := IntPower(T, 2);
  T_4 := IntPower(T_2, 2);
  T_6 := T_2 * T_4;

  CosLat := Cos(ToRad * Latitude);
  CosLat_3 := IntPower(CosLat, 3);
  CosLat_5 := IntPower(CosLat, 5);
  CosLat_7 := IntPower(CosLat, 7);

  NuSinLat := Nu * Sin(ToRad * Latitude);

  if CalcEasting then begin
    E1 := Nu * W * CosLat;
    E2 := Nu * W_3 / 6 * CosLat_3 * (Psi - T_2);
    E3 := Nu * W_5 / 120 * CosLat_5
          * (4 * Psi_3 * (1 - 6 * T_2)
             + Psi_2 * (1 + 8 * T_2) - Psi * (2 * T_2) + T_4);
    E4 := Nu * W_7 / 5040 * CosLat_7 * (61 - 479 * T_2 + 179 * T_4 - T_6);

    Easting := AddFalseEasting(K0 * (E1 + E2 + E3 + E4));
  end;
  if CalcNorthing then begin
    N1 := NuSinLat * W_2 / 2 * CosLat;
    N2 := NuSinLat * W_4 / 24 * CosLat_3 * (4 * Psi_2 + Psi - T_2);
    N3 := NuSinLat * W_6 / 720 * CosLat_5
          * (8 * Psi_4 * (11 - 24 * T_2)
             - 28 * Psi_3 * (1 - 6 * T_2) + Psi_2 * (1 - 32 * T_2)
             - Psi * (2 * T_2) + T_4);
    N4 := NuSinLat * W_8 / 40320
          * CosLat_7 * (1385 - 3111 * T_2 + 543 * T_4 - T_6);

    Northing := AddFalseNorthing(K0 * (CalcMeridian(Latitude) +
                                       N1 + N2 + N3 + N4),
                                 Zone);
  end;
end;



function GeoToEasting(Latitude, Longitude: extended; Zone: integer): extended;
var
  Easting, Northing: extended;
begin
  InternalGeoToUTM(Latitude, Longitude, Zone, Easting, Northing, True, False);
  Result := Easting;
end;


function GeoToNorthing(Latitude, Longitude: extended; Zone: integer): extended;
var
  Easting, Northing: extended;
begin
  InternalGeoToUTM(Latitude, Longitude, Zone, Easting, Northing, False, True);
  Result := Northing;
end;


procedure GeoToUTM(Latitude, Longitude: extended; Zone: integer;
                   var Easting, Northing: extended);
begin
  InternalGeoToUTM(Latitude, Longitude, Zone, Easting, Northing, True, True);
end;


{-----------------------------------------------------------------------------}
initialization
  ChangeZoneWidth(ZoneWidth, K0);
end.
