{-------------------------------------------------------------------------
 Module:    TGWindBarb.pas

 Comment:  TGlobe WindBarb Object and Presenter

 Classes:  TWindBarbObject
           TWindBarbPresenter

 Author:   Graham Knight
 Email:     tglobe@iname.com
------------------------------------------------------------------------------}

unit TGWindBarb;

interface

Uses WinTypes, WinProcs, SysUtils, Classes, Graphics, Globe4, TGSysUtils,
  TGClasses;

type
	TWindBarbPresenter = class( TGlobePresenter )
	private
		FBarbColor : TColor;
		FBarbLength : integer;
		FBarbUnit : TGlobeUnitTypes;
	public
    constructor Create( ParentGlobe : TCustomGlobe4; iID : integer ); override;
    procedure RenderObject( Globe : TCustomGlobe4; Geod : TGlobeObject; State : TGlobeObjectStateSet); override;
	published
		property BarbColor : TColor read FBarbColor write FBarbColor;
		property BarbLength : integer read FBarbLength write FBarbLength;
		property BarbUnit : TGlobeUnitTypes read FBarbUnit write FBarbUnit;
	end;

	TWindBarbObject = class( TGlobeObject )
	private
		FDirection : integer;
		FStrength : integer;
		FTemp : integer;
	published
		property Direction : integer read FDirection write FDirection;
		property Strength : integer read FStrength write FStrength;
		property Temp : integer read FTemp write FTemp;
	end;

implementation

{------------------------------------------------------------------------------
  TWindBarbPresenter.Create
------------------------------------------------------------------------------}
constructor TWindBarbPresenter.Create( ParentGlobe : TCustomGlobe4; iID : integer );
begin
	inherited Create( ParentGlobe, iID );

	FBarbColor := clBlack;
	FBarbUnit := Pixel;
	FBarbLength := 20;
end;

{------------------------------------------------------------------------------
	TWindBarbPresenter.RenderObject
------------------------------------------------------------------------------}
procedure TWindBarbPresenter.RenderObject( Globe : TCustomGlobe4;
  Geod : TGlobeObject; State : TGlobeObjectStateSet );
var
	CalmRadius : integer;
	FlagLen    : single;
	LineLen    : single;
	LineSpace  : single;
	FlagWidth  : single;
	LineLean   : single;
	offset     : single;		// amount of stem left
	anyFlags   : Boolean;
	Hemisphere : integer;

	eSin, eCos   : Extended;
	startRadius : integer;
	iTmp, iX, iY : integer;
	Speed : integer;
begin
	if not( Geod is TWindBarbObject ) then
		Exit;

	Globe.GlobeCanvas.Pen.Color := BarbColor;
	Globe.GlobeCanvas.Brush.Color := BarbColor;

	iTmp := TGlobe4( Globe ).GlobeUnitsFrom( BarbLength, BarbUnit );

	CalmRadius := iTmp div 7;
	startRadius := 0;
	offset := iTmp;
	flagwidth := iTmp div 7;
	linelean := iTmp div 10;
	linespace := iTmp div 10;

	with Geod as TWindBarbObject do
		if Globe.Projection.ProjectionModel.PointLLToXY( Centroid, 0 ) then
			with Globe.GlobeCanvas do
			begin
				if Centroid.iLatY < 0 then	{ if in southern hemisphere }
					Hemisphere := -1
				else
					Hemisphere := +1;

				flaglen := iTmp div 2 * Hemisphere;
				Linelen := iTmp div 2 * Hemisphere;
				Speed := Strength;

				iX := gaPoints^[0].X;
				iY := gaPoints^[0].Y;
				SinCos( Direction * LocalPi / 1800, eSin, eCos );

				if Speed = 0 then
					Globe.GlobeCanvas.Ellipse( iX - CalmRadius, iY - CalmRadius, iX + CalmRadius + 1, iY + CalmRadius + 1)
				else
				begin
					MoveTo( Round(iX + startRadius * eSin), Round(iY - startRadius * eCos));
					LineTo( Round(iX + offset * eSin), Round(iY - offset * eCos));

					if Speed > 2 then 	//needs barbs
					begin
						anyFlags := False;

						while Speed >= 48 do
						begin
							anyFlags := True;
							Polygon( [
								Point( Round(iX + offset * eSin), Round(iY - offset * eCos)),
								Point( Round(iX + (offset - FlagWidth) * eSin), Round(iY - (offset - FlagWidth) * eCos)),
								Point( Round(iX + ((offset - FlagWidth/2) * eSin) + (FlagLen * eCos)),
											Round(iY - ((offset - FlagWidth/2) * eCos) + (FlagLen * eSin)))
								] );

							Speed  := Speed - 50;
							offset := offset - FlagWidth;
						end; //while

						if AnyFlags then
							offset := offset - LineSpace;

						while Speed >= 8 do
						begin
							MoveTo( Round(iX + offset * eSin), Round(iY - offset * eCos));
							LineTo( Round(iX + ((offset + LineLean) * eSin) + (LineLen * eCos)),
											Round(iY - ((offset + LineLean) * eCos) + (LineLen * eSin)));
							anyFlags := true;
							offset := offset - LineSpace;
							Speed  := Speed - 10;
						end;//while

						while Speed >= 3 do
						begin
							if not AnyFlags then
								offset := offset - LineSpace;

							Speed  := Speed - 5;
							MoveTo( Round(iX + offset * eSin), Round(iY - offset * eCos));
							LineTo( Round(iX + ((offset + LineLean/2) * eSin) + ((LineLen / 2) * eCos)),
											Round(iY - ((offset + LineLean/2) * eCos) + ((LineLen / 2) * eSin)));
						end;//while
					end;
				end;
			end;//with
end;

end.
