{------------------------------------------------------------------------------
	Module:		TGUtils.pas

	Author:		Graham Knight
	Email:		gknight@helmstone.co.uk

	3.00  01-Dec-98  Converted to TGlobe Version 3
------------------------------------------------------------------------------}
unit TGUtils;

interface

Uses WinTypes, WinProcs, Classes, SysUtils, Graphics, Globe3;

{------------------------------------------------------------------------------}

procedure LayerFromResourceName( Layer : TGlobeLayer; const ResourceName: string);
procedure LayerFromResourceID( Layer : TGlobeLayer; ResourceID: Integer);

function DistanceLLToLL( const FromLL, ToLL : TPointLL ) : Integer;
function AngleLLToLL( const FromLL, ToLL : TPointLL ) : Integer;
function GreatCirclePoint( const FromLL, ToLL : TPointLL; alpha : Extended ) : TPointLL;
function AngleDistToGCPoint( const ptLL : TPointLL; iAngle, iDistance : Integer ) : TPointLL;

{------------------------------------------------------------------------------}
implementation

{------------------------------------------------------------------------------
 LayerFromResourceName

	Loads a data layer by name that is store as a resource of the app.

	To add a data layer as a resource you need to create a resource script
	file (world.rc) which is used to generate the WORLD.RES file. This has 
	just one line :-

		WORLD RCDATA "TGWORLD.LYR"

	You run the command line resource compiler ( in the Borland\Delphix\Bin
	folder) BRCC32 with the command line -r world.rc.  This will generate
	WORLD.RES which you then include with your program using {$R WORLD.RES}
{------------------------------------------------------------------------------}
procedure LayerFromResourceName( Layer : TGlobeLayer; const ResourceName: string);
var
	Stream: TCustomMemoryStream;
begin
	Stream := TResourceStream.Create( hInstance, ResourceName, RT_RCDATA);
	try
		Layer.LoadFromStream( Stream );
	finally
		Stream.Free;
	end;
end;

{------------------------------------------------------------------------------
 LayerFromResourceID

	Loads a data layer by ID that is store as a resource of the app.

	See above for a description of how to add a resource to an application.
------------------------------------------------------------------------------}
procedure LayerFromResourceID( Layer : TGlobeLayer; ResourceID: Integer);
var
	Stream: TCustomMemoryStream;
begin
	Stream := TResourceStream.CreateFromID( hInstance, ResourceID, RT_RCDATA);
	try
		Layer.LoadFromStream( Stream );
	finally
		Stream.Free;
	end;
end;

{-------------------------------------------------------------------------
 AngleDistToGCPoint

	Returns a point on the great circle defined by a given point, 
	a distance and an angle from that point. An angle of 0 is due North.
-------------------------------------------------------------------------}
function AngleDistToGCPoint( const ptLL : TPointLL; iAngle, iDistance : integer ) : TPointLL;
var
	sinlat1, coslat1,
	sind, cosd,	sintc, costc,
	dlon,	lat, lon : extended;
begin
	with PtLL do
	begin
		SinCos( ilatY * TORADIANS, sinlat1, coslat1 );
		SinCos( iDistance / EARTHRADIUS, sind, cosd );
		SinCos( iAngle * TORADIANS, sintc, costc );
		lat := arcsin( sinlat1 * cosd + coslat1 * sind * costc );
		dlon := arctan2( sintc * sind * coslat1, cosd - sinlat1 * sin( lat ) );
		lon := SphericalMod( ilongX * TORADIANS - dlon + LocalPI ) - LocalPI;
	end;
	Result := PointLL( Round( lon * FROMRADIANS ), Round( lat * FROMRADIANS ));
end;

{-------------------------------------------------------------------------
 GreatCirclePoint

	Returns a point on the Great Circle line between the supplied Points.
	If Alpha is 0.0 then FromLL is returned and if Alpha is 1.0 then ToLL
	is returned.
-------------------------------------------------------------------------}
function GreatCirclePoint( const FromLL, ToLL : TPointLL;
	alpha : Extended ) : TPointLL;
var
	a, b : TV3D;
	mat : TMatrix;
begin
	a := PointLLToV3D( FromLL );
	b := PointLLToV3D( ToLL );
	QuatToMatrix( AxisAngleToQuat( V3DCross( a, b ),
		-ArcCos( V3DDot( a, b ) ) * alpha ), mat );

	with V3DToPointLL( V3DMatrixMul( mat, a ) ) do
		Result := PointLL( iLongX, iLatY );
end;

{------------------------------------------------------------------------------
 AngleLLToLL

	Calculates the Angle between the supplied points. The angle is returned
	in Globeunits with 0 being due north. To convert the result to Radians
	multiply by the TORADIANS constant.
------------------------------------------------------------------------------}
function AngleLLToLL( const FromLL, ToLL : TPointLL ) : integer;
var
	SLat1, SLat2, CLat1, CLat2 : Extended;
	SLonDiff, CLonDiff : Extended;
begin
	SinCos( FromLL.iLatY * TORADIANS, SLat1, CLat1 );
	SinCos( ToLL.iLatY * TORADIANS, SLat2, CLat2 );
	SinCos( ( FromLL.iLongX - ToLL.iLongX ) * TORADIANS, SLonDiff, CLonDiff );

	Result := Round( SphericalMod( ArcTan2( SLonDiff * CLat2,
		CLat1 * SLat2 - SLat1 * CLat2 * CLonDiff ) ) * FROMRADIANS );
end;

{-------------------------------------------------------------------------
 DistanceLLToLL
 
	Calculates the distance between the supplied points using the Haversine 
	formula.  This is quick but not very accurate, if better accuracy is
	required use the Globe.EllipsoidDistanceLLtoLL() method.
-------------------------------------------------------------------------}
function DistanceLLToLL( const FromLL, ToLL : TPointLL ) : Integer;
var
	dist, latFrom, LatTo, lonDist : Extended;
begin
	lonDist := FromLL.iLongX * TORADIANS - ToLL.iLongX * TORADIANS;
	latFrom := FromLL.iLatY * TORADIANS;
	LatTo := ToLL.iLatY * TORADIANS;

	dist := 2 * ArcSin( Sqrt( ( Sqr( Sin( ( latFrom - LatTo ) * 0.5 ) ) )
		+ Cos( latFrom ) * Cos( LatTo ) * Sqr( Sin( lonDist * 0.5 ) ) ) );

	Result := Round( dist * FROMRADIANS );
end;

end.
