{------------------------------------------------------------------------------
	Module:		TGProject.pas

	Comment:  TGlobe Suplemental projections

	Classes:  TEquidistantPrj
						TCylEqualAreaPrj
						TBehrmannCylEqualAreaPrj
						TPetersPrj
						TAlbersEqualAreaConicPrj
						TBonnePrj
						TSinusoidalPrj

	Author:		Graham Knight
	Email:		gknight@helmstone.co.uk

 These projections can be used by creating and assigning to the 
 Globe.ProjectionModel property i.e.

	 Globe.ProjectionModel := TAlbersEqualAreaConicPrj.Create( Globe );

 You do not need to free the existing projection as this is done
 automatically when you assign a new projection.

------------------------------------------------------------------------------}

unit TGProject;

interface

Uses WinTypes, WinProcs, SysUtils, Classes, Globe3;

type
	TAzimuthalEquidistantPrj = class( TCartesianPrj )
	public
		constructor Create( Parent : TGlobe3 ); override;
		function PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean; override;
		function XYToLL( iX, iY, iIndex : Integer ) : Boolean; override;
		procedure PaintBackground; override;
	end;
	{---------------------------- TEquidistantPrj -------------------------------}
	TEquidistantPrj = class( TCartesianPrj )
	public
		constructor Create( Parent : TGlobe3 ); override;
		function PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean; override;
		function XYToLL( iX, iY, iIndex : Integer ) : Boolean; override;
		procedure PaintBackground; override;
	end;

	{---------------------------- TCylEqualAreaPrj ------------------------------}
	TCylEqualAreaPrj = class( TCartesianPrj )
	public
		constructor Create( Parent : TGlobe3 ); override;
		function PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean; override;
		function XYToLL( iX, iY, iIndex : Integer ) : Boolean; override;
	end;

	{---------------------------- TBehrmannCylEqualAreaPrj ----------------------}
	TBehrmannCylEqualAreaPrj = class( TCylEqualAreaPrj )
	public
		constructor Create( Parent : TGlobe3 ); override;
	end;

	{---------------------------- TPetersPrj ------------------------------------}
	TPetersPrj = class( TCylEqualAreaPrj )
	public
		constructor Create( Parent : TGlobe3 ); override;
	end;

	{---------------------------- TAlbersEqualAreaConicPrj ----------------------}
	TAlbersEqualAreaConicPrj = class( TCartesianPrj )
	private
		n, Rho0, C : Extended;
	protected
		procedure SetProperty( iIndex : integer; iValue : Integer ); override;
	public
		constructor Create( Parent : TGlobe3 ); override;
		function PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean; override;
		function XYToLL( iX, iY, iIndex : Integer ) : Boolean; override;
	end;

	{---------------------------- TBonnePrj -------------------------------------}
	TBonnePrj = class( TCartesianPrj )
	private
		cotSP : Extended;
	protected
		procedure SetProperty( iIndex : integer; iValue : Integer ); override;
	public
		constructor Create( Parent : TGlobe3 ); override;
		function PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean; override;
		function XYToLL( iX, iY, iIndex : Integer ) : Boolean; override;
	end;

	{---------------------------- TSinusoidalPrj -------------------------------------}
	TSinusoidalPrj = class( TCartesianPrj )
	public
		function PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean; override;
		function XYToLL( iX, iY, iIndex : Integer ) : Boolean; override;
	end;

implementation

{-------------------------------------------------------------------------
 TEquidistantPrj.Create
-------------------------------------------------------------------------}
constructor TEquidistantPrj.Create( Parent : TGlobe3 );
begin
	ExtentsLL := Rect( -GU_180_DEGREE, -GU_180_DEGREE,
		GU_180_DEGREE, GU_180_DEGREE );
	inherited Create( Parent );

	Include( FFlags, pfContinuous );
end;

{-------------------------------------------------------------------------
 TEquidistantPrj.XYtoLL()
-------------------------------------------------------------------------}
function TEquidistantPrj.XYToLL( iX, iY, iIndex : Integer ) : Boolean;
var
	x, y, D, sinD, cosD : Extended;
	lat, lon : Extended;
begin
	x := ( iX - XOrigin ) / ScaleFactor * TORADIANS;
	y := -( iY - YOrigin ) / ScaleFactor * TORADIANS;
	D := Hypot( x, y );
	if D <= EPSILON then
	begin
		lat := 0;
		lon := 0;
		Result := True;
	end
	else
	begin
		SinCos( D, sinD, cosD );
		lat := ArcSin(( y * sinD ) / D );
		lon := ArcCos( cosD / Cos( lat ));

		{ check to see if the coords are outside the globe }
		Result := ( abs( lon ) < LocalPi ) and ( abs( lat ) < HalfPi );
	end;

	if Result then
		gaPoints^[iIndex] := Point(
			Mod180( Round( lon * FROMRADIANS ) + CentralMeridian),
			Round( lat * FROMRADIANS ));
end;

{-------------------------------------------------------------------------
 TEquidistantPrj.LLToXY
-------------------------------------------------------------------------}
function TEquidistantPrj.PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean;
var
	sinlat2, coslat2,
	d, tc, lat2, lon2,
	xcoord, ycoord,
	sinx, cosx : extended;
begin
	with ptLL do
	begin
		{ check to see if the coords are outside the globe }
		with ExtentsLL do
			Result := ( iLongX >= Left ) and ( iLongX <= Right ) and ( iLatY >= Top ) and ( iLatY <= Bottom );

		if Result then
		begin
			lat2 := iLatY * TORADIANS;
			lon2 := Mod180( iLongX - CentralMeridian ) * TORADIANS;

			SinCos( lat2, sinlat2, coslat2 );
			d := arccos( coslat2 * cos( lon2 ) );
			SinCos( d, sinx, cosx );
			if Abs( sinx ) < 10E-6 then
				tc := 0.0
			else
			begin
				tc := arccos( sinlat2 / sinx );
				if sin( lon2 ) < 0.0 then
					tc := 2.0 * pi - tc;
			end;
			SinCos( tc, sinx, cosx );
			xcoord := Sinx * d * FROMRADIANS; //-pi .. pi
			ycoord := Cosx * d * FROMRADIANS; //-pi .. pi

			gaPoints^[iIndex] := Point(
				XOrigin + Round( xcoord * ScaleFactor ),
				YOrigin - Round( ycoord * ScaleFactor ) );
		end;
	end;
end;

{-------------------------------------------------------------------------
 TEquidistantPrj.PaintBackground
-------------------------------------------------------------------------}
procedure TEquidistantPrj.PaintBackground;
const
	SEGMENTS = 90;
var
	idx : Integer;
	eLastX, eLastY, eX, eY : Extended;
	eCosAngle, eSinAngle : Extended;
begin
	eY := 0;
	eX := EARTHRADIUS * ScaleFactor * LocalPi;

	gaPoints^[0] := Point( Round( XOrigin + eX ), Round( YOrigin + eY ) );
	SinCos( ( GU_360_DEGREE / SEGMENTS ) * TORADIANS, eSinAngle, eCosAngle );

	idx := 1;
	while idx < SEGMENTS do
	begin
		eLastX := eX;
		eLastY := eY;
		eX := eLastX * eCosAngle - eLastY * eSinAngle;
		eY := eLastX * eSinAngle + eLastY * eCosAngle;
		gaPoints^[idx] := Point( Round( XOrigin + eX ), Round( YOrigin + eY ) );
		Inc( idx );
	end;
	Globe.GlobeCanvas.ClippedPolygon( SEGMENTS, $FF );
end;

{------------------------------------------------------------------------------
	TCylEqualAreaPrj.Create
------------------------------------------------------------------------------}
constructor TCylEqualAreaPrj.Create( Parent : TGlobe3 );
begin
	ExtentsLL := Rect( -GU_180_DEGREE, -GU_90_DEGREE, GU_180_DEGREE, GU_90_DEGREE );
	inherited Create( Parent );
end;

{-------------------------------------------------------------------------
 TCylEqualAreaPrj.XYtoLL()
-------------------------------------------------------------------------}
function TCylEqualAreaPrj.XYToLL( iX, iY, iIndex : Integer ) : Boolean;
var
	x, y, lat, long, eCos : Extended;
begin
	x := ( iX - XOrigin ) / ScaleFactor * TORADIANS;
	y := -( iY - YOrigin ) / ScaleFactor * TORADIANS;

	begin
		eCos := Cos( FirstParallel * TORADIANS );
		lat := ArcSin( y * eCos);
		Long := ( x / eCos );

		{ check to see if the coords are outside the globe }
		Result := ( abs( long ) < LocalPi ) and ( abs( lat ) < HalfPi );
	end;

	if Result then
		gaPoints^[iIndex] := Point(
			Mod180( Round( long * FROMRADIANS ) + CentralMeridian),
			Round( lat * FROMRADIANS ));
end;

{-------------------------------------------------------------------------
 TCylEqualAreaPrj.PointLLToXY
-------------------------------------------------------------------------}
function TCylEqualAreaPrj.PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean;
var
	lat, long, eCos,
	xcoord, ycoord : extended;
begin
	with ptLL do
	begin
		{ check to see if the coords are outside the globe }
		with ExtentsLL do
			Result := ( iLongX >= Left ) and ( iLongX <= Right ) and ( iLatY >= Top ) and ( iLatY <= Bottom );

		if Result then
		begin
			lat := iLatY * TORADIANS;
			long := Mod180( iLongX - CentralMeridian ) * TORADIANS;

			eCos := Cos( FirstParallel * TORADIANS );
			xcoord := long * eCos * FROMRADIANS;
			ycoord := Sin( lat ) / eCos * FROMRADIANS;

			gaPoints^[iIndex] := Point(
				XOrigin + Round( xcoord * ScaleFactor ),
				YOrigin - Round( ycoord * ScaleFactor ) );
		end;
	end;
end;

{------------------------------------------------------------------------------
	TBehrmannCylEqualAreaPrj.Create
------------------------------------------------------------------------------}
constructor TBehrmannCylEqualAreaPrj.Create( Parent : TGlobe3 );
begin
	inherited Create( Parent );
	FirstParallel := GU_DEGREE * 30;	{ For Behrmann Projection }
end;

{------------------------------------------------------------------------------
	TPetersPrj.Create
------------------------------------------------------------------------------}
constructor TPetersPrj.Create( Parent : TGlobe3 );
begin
	inherited Create( Parent );
	FirstParallel := GU_DEGREE * 45;	{ For Peters Projection }
end;

{------------------------------------------------------------------------------
	TAlbersEqualAreaConicPrj.Create
------------------------------------------------------------------------------}
constructor TAlbersEqualAreaConicPrj.Create( Parent : TGlobe3 );
begin
	inherited Create( Parent );

	FirstParallel := GU_DEGREE * 45;
	SecondParallel := GU_DEGREE * 0;
end;

{------------------------------------------------------------------------------
	TAlbersEqualAreaConicPrj.SetProperty
------------------------------------------------------------------------------}
procedure TAlbersEqualAreaConicPrj.SetProperty( iIndex, iValue : integer );
var
	eCos, eSin : Extended;
begin
	inherited SetProperty( iIndex, iValue );

	{ if either the first or second parallels changed }
	if ( iIndex = 4 ) or ( iIndex = 5 ) then
	begin
		SinCos( FirstParallel * TORADIANS, eSin, eCos );
		n := 0.5 * ( eSin + Sin( SecondParallel * TORADIANS ));

		if n = 0 then n := 1;

		C := Sqr( eCos ) + 2 * n * eSin;
		Rho0 := Sqrt( C ) / n;
	end;
end;

{------------------------------------------------------------------------------
	TAlbersEqualAreaConicPrj.PointLLToXY
------------------------------------------------------------------------------}
function TAlbersEqualAreaConicPrj.PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean;
var
	lat, long, eCos, eSin : Extended;
	xcoord, ycoord : Extended;
	Rho, Theta : Extended;
begin
	with ptLL do
	begin
		{ check to see if the coords are outside the globe }
		with ExtentsLL do
			Result := ( iLongX >= Left ) and ( iLongX <= Right ) and ( iLatY >= Top ) and ( iLatY <= Bottom );

		if Result then
		begin
			lat := iLatY * TORADIANS;
			long := Mod180( iLongX - CentralMeridian ) * TORADIANS;

			theta := n * ( long );
			Rho := Sqrt( C - 2 * n * Sin( lat )) / n;

			SinCos( Theta, eSin, eCos );

			xcoord := Rho * eSin * FROMRADIANS;
			ycoord := ( Rho0 - Rho * eCos ) * FROMRADIANS;

			gaPoints^[iIndex] := Point(
				XOrigin + Round( xcoord * ScaleFactor ),
				YOrigin - Round( ycoord * ScaleFactor ) );
		end;
	end;
end;

{------------------------------------------------------------------------------
	TAlbersEqualAreaConicPrj.XYToLL
------------------------------------------------------------------------------}
function TAlbersEqualAreaConicPrj.XYToLL( iX, iY, iIndex : Integer ) : Boolean;
var
	x, y, lat, long, Rho, Theta : Extended;
begin
	x := ( iX - XOrigin ) / ScaleFactor * TORADIANS;
	y := -( iY - YOrigin ) / ScaleFactor * TORADIANS;

	Rho := Sqrt( x * x + Sqr( Rho0 - y ));
	Theta := ArcTan( x / ( Rho0 - y ));

	begin
		lat := ArcSin(( C - Rho * Rho * n * n ) / ( n + n ));
		Long := Theta / n;

		{ check to see if the coords are outside the globe }
		Result := ( abs( long ) < LocalPi ) and ( abs( lat ) < HalfPi );
	end;

	if Result then
		gaPoints^[iIndex] := Point(
			Mod180( Round( long * FROMRADIANS ) + CentralMeridian), Round( lat * FROMRADIANS ));
end;

{-------------------------------------------------------------------------
 TAzimuthalEquidistantPrj.Create
-------------------------------------------------------------------------}
constructor TAzimuthalEquidistantPrj.Create( Parent : TGlobe3 );
begin
	ExtentsLL := Rect( -GU_180_DEGREE, -GU_180_DEGREE,
		GU_180_DEGREE, GU_180_DEGREE );
	inherited Create( Parent );

	Include( FFlags, pfContinuous );
end;

{-------------------------------------------------------------------------
 TAzimuthalEquidistantPrj.XYtoLL()
-------------------------------------------------------------------------}
function TAzimuthalEquidistantPrj.XYToLL( iX, iY, iIndex : Integer ) : Boolean;
var
	x, y,
	d, sind, cosd,
	b, sinb,
	lat2, lon2, v : extended;
begin
	x := ( iX - XOrigin ) / ScaleFactor * TORADIANS;
	y := -( iY - YOrigin ) / ScaleFactor * TORADIANS;
	d := Hypot( x, y );

	if d <= EPSILON then
	begin
		lat2 := 0;
		lon2 := 0;
		Result := True;
	end
	else
	begin
		SinCos( d, sind, cosd );
		sinb := ( y * sind ) / d;
		b := ArcSin( sinb );
		lat2 := b;
		v := cosd / Cos( b );
		lon2 := ArcCos( v );

		{ check to see if the coords are outside the globe }
		Result := ( abs( lon2 ) < LocalPi ) and ( abs( lat2 ) < HalfPi );
	end;

	if Result then
		gaPoints^[iIndex] := Point(
			Mod180( Round( lon2 * FROMRADIANS ) + CentralMeridian),
			Round( lat2 * FROMRADIANS ));
end;

{-------------------------------------------------------------------------
 TAzimuthalEquidistantPrj.LLToXY
-------------------------------------------------------------------------}
function TAzimuthalEquidistantPrj.PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean;
var
	sinlat2, coslat2,
	d, tc, lat2, lon2,
	xcoord, ycoord : extended;
begin
	with ptLL do
	begin
		{ check to see if the coords are outside the globe }
		with ExtentsLL do
			Result := ( iLongX >= Left ) and ( iLongX <= Right ) and ( iLatY >= Top ) and ( iLatY <= Bottom );

		if Result then
		begin
			lat2 := iLatY * TORADIANS;
			lon2 := Mod180( iLongX - CentralMeridian ) * TORADIANS;

			SinCos( lat2, sinlat2, coslat2 );
			d := arccos( coslat2 * cos( lon2 ));

			if Abs( d ) < EPSILON then
				tc := 0
			else
				tc := d / sin( d );

			xcoord := tc * Coslat2 * Sin( lon2 ) * FROMRADIANS; //-pi .. pi
			ycoord := tc * sinlat2 * FROMRADIANS; //-pi .. pi

			gaPoints^[iIndex] := Point(
				XOrigin + Round( xcoord * ScaleFactor ),
				YOrigin - Round( ycoord * ScaleFactor ) );
		end;
	end;
end;

{-------------------------------------------------------------------------
 TAzimuthalEquidistantPrj.PaintBackground
-------------------------------------------------------------------------}
procedure TAzimuthalEquidistantPrj.PaintBackground;
const
	SEGMENTS = 90;
var
	idx : Integer;
	eLastX, eLastY, eX, eY : Extended;
	eCosAngle, eSinAngle : Extended;
begin
	eY := 0;
	eX := EARTHRADIUS * ScaleFactor * LocalPi;

	gaPoints^[0] := Point( Round( XOrigin + eX ), Round( YOrigin + eY ) );
	SinCos( ( GU_360_DEGREE / SEGMENTS ) * TORADIANS, eSinAngle, eCosAngle );

	idx := 1;
	while idx < SEGMENTS do
	begin
		eLastX := eX;
		eLastY := eY;
		eX := eLastX * eCosAngle - eLastY * eSinAngle;
		eY := eLastX * eSinAngle + eLastY * eCosAngle;
		gaPoints^[idx] := Point( Round( XOrigin + eX ), Round( YOrigin + eY ) );
		Inc( idx );
	end;
	Globe.GlobeCanvas.ClippedPolygon( SEGMENTS, $FF );
end;

{------------------------------------------------------------------------------
	TBonnePrj.Create
------------------------------------------------------------------------------}
constructor TBonnePrj.Create( Parent : TGlobe3 );
begin
	inherited Create( Parent );

	FirstParallel := GU_90_DEGREE;
end;

{------------------------------------------------------------------------------
	TBonnePrj.SetProperty
------------------------------------------------------------------------------}
procedure TBonnePrj.SetProperty( iIndex, iValue : integer );
begin
	inherited SetProperty( iIndex, iValue );

	{ if the first parallel changes }
	if iIndex = 4 then
		if FirstParallel = 0 then
			cotSP := LocalPI
		else
			cotSP := 1 / Tan( FirstParallel * TORADIANS );
end;

{------------------------------------------------------------------------------
	TBonnePrj.PointLLToXY
------------------------------------------------------------------------------}
function TBonnePrj.PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean;
var
	lat, long : Extended;
	xcoord, ycoord : Extended;
	Rho, E : Extended;
begin
	with ptLL do
	begin
		{ check to see if the coords are outside the globe }
		with ExtentsLL do
			Result := ( iLongX >= Left ) and ( iLongX <= Right ) and ( iLatY >= Top ) and ( iLatY <= Bottom );

		if Result then
		try
			lat := iLatY * TORADIANS;
			long := Mod180( iLongX - CentralMeridian ) * TORADIANS;

			Rho := ( cotSP + FirstParallel * TORADIANS - lat );
			E := long * ( Cos( lat ) / Rho );

			xcoord := Rho * Sin( E );
			ycoord := ( cotSP - Rho * Cos( E ));

			gaPoints^[iIndex] := Point(
				XOrigin + Round( xcoord * ScaleFactor * FROMRADIANS ),
				YOrigin - Round( ycoord * ScaleFactor * FROMRADIANS ));
		except
		end;
	end;
end;

{------------------------------------------------------------------------------
	TBonnePrj.XYToLL
------------------------------------------------------------------------------}
function TBonnePrj.XYToLL( iX, iY, iIndex : Integer ) : Boolean;
var
	x, y, lat, long, Rho : Extended;
begin
	x := ( iX - XOrigin ) / ScaleFactor * TORADIANS;
	y := -( iY - YOrigin ) / ScaleFactor * TORADIANS;

	Rho := Sign( FirstParallel ) * Sqrt( x * x + Sqr( cotSP - y ));

	lat := cotSP + FirstParallel * TORADIANS - Rho;
	Long := ( Rho / Cos( lat )) * ArcTan2( x, cotSP - y );

	{ check to see if the coords are outside the globe }
	Result := ( abs( long ) < LocalPi ) and ( abs( lat ) < HalfPi );

	if Result then
		gaPoints^[iIndex] := Point(
			Mod180( Round( CentralMeridian + long * FROMRADIANS )),
			Round( lat * FROMRADIANS ));
end;

{------------------------------------------------------------------------------
	TSinusoidalPrj.PointLLToXY
------------------------------------------------------------------------------}
function TSinusoidalPrj.PointLLToXY( const ptLL : TPointLL; iIndex : Integer ) : Boolean;
var
	xcoord, ycoord : Extended;
begin
	with ptLL do
	begin
		{ check to see if the coords are outside the globe }
		with ExtentsLL do
			Result := ( iLongX >= Left ) and ( iLongX <= Right ) and ( iLatY >= Top ) and ( iLatY <= Bottom );

		if Result then
		begin
			ycoord := iLatY * TORADIANS;
			xcoord := Mod180( iLongX - CentralMeridian ) * TORADIANS * Cos( ycoord );

			gaPoints^[iIndex] := Point(
				XOrigin + Round( xcoord * FROMRADIANS * ScaleFactor ),
				YOrigin - Round( ycoord * FROMRADIANS * ScaleFactor ) );
		end;
	end;
end;

{------------------------------------------------------------------------------
	TSinusoidalPrj.XYToLL
------------------------------------------------------------------------------}
function TSinusoidalPrj.XYToLL( iX, iY, iIndex : Integer ) : Boolean;
var
	x, y, lat, long : Extended;
begin
	x := ( iX - XOrigin ) / ScaleFactor * TORADIANS;
	y := -( iY - YOrigin ) / ScaleFactor * TORADIANS;

	lat := y;
	Long := x / Cos( lat );

	{ check to see if the coords are outside the globe }
	Result := ( abs( long ) < LocalPi ) and ( abs( lat ) < HalfPi );

	if Result then
		gaPoints^[iIndex] := Point(
			Mod180( Round( CentralMeridian + long * FROMRADIANS )),
			Round( lat * FROMRADIANS ));
end;

end.