{-------------------------------------------------------------------------
	Comment:   PolygonPresenter Source

	Author:	   Graham Knight
  Copyright: Graham Knight 1998
	Email:     gknight@helmstone.co.uk

 This file is extracted from the TGlobe3 source code. It is the source for the
 TPolygonPresenter.

 This is given here to show how a complete presenter can be constructed.
-------------------------------------------------------------------------}

type
	{---------------------------- TPolygonPresenter -----------------------------}
	TPolygonPresenter = class( TGlobePresenter )
	private
		FPolygonPen : TGlobePen;
		FPolygonBrush : TGlobeBrush;
	public
		constructor CreateObject( ParentLayer : TGlobeLayer ); override;
		destructor Destroy; override;

		class function FindMatch( ParentLayer : TGlobeLayer;
			APen : TGlobePen; ABrush : TGlobeBrush; ATitleFont : TGlobeFont ) : TPolygonPresenter;

		procedure Assign(Source: TPersistent); override;
		procedure RenderObject( Geod : TGlobeObject ); override;
		procedure SaveToStream( AStream : TStream ); override;
		procedure LoadFromStream( AStream : TStream ); override;

		function LLInObject( Geod : TGlobeObject; ptLL : TPointLL ) : Boolean; override;
	published
		property PolygonPen : TGlobePen read FPolygonPen;
		property PolygonBrush : TGlobeBrush read FPolygonBrush;
	end;


implementation

{------------------------------------------------------------------------------
	TPolygonPresenter.CreateObject
------------------------------------------------------------------------------}
constructor TPolygonPresenter.CreateObject( ParentLayer : TGlobeLayer );
begin
	inherited CreateObject( ParentLayer );

	FPolygonPen := TGlobePen.CreateObject( ParentLayer );
	FPolygonBrush := TGlobeBrush.CreateObject( ParentLayer );
end;

{------------------------------------------------------------------------------
	TPolygonPresenter.Destroy
------------------------------------------------------------------------------}
destructor TPolygonPresenter.Destroy;
begin
	FPolygonPen.Free;
	FPolygonBrush.Free;
	inherited Destroy;
end;

{-------------------------------------------------------------------------
 TPolygonPresenter.Assign
-------------------------------------------------------------------------}
procedure TPolygonPresenter.Assign( Source: TPersistent );
begin
	inherited Assign( Source );

	if Source is TPolygonPresenter then
	begin
		PolygonPen.Assign( TPolygonPresenter( Source ).PolygonPen );
		PolygonBrush.Assign( TPolygonPresenter( Source ).PolygonBrush );
	end;
end;

{-------------------------------------------------------------------------
 TPolygonPresenter.FindMatch
-------------------------------------------------------------------------}
class function TPolygonPresenter.FindMatch( ParentLayer : TGlobeLayer;
	APen : TGlobePen; ABrush : TGlobeBrush; ATitleFont : TGlobeFont ) : TPolygonPresenter;
var
	idx : integer;
begin
	with ParentLayer do
		for idx := 0 to FPresenterList.Count - 1 do
			if TGlobePresenter( FPresenterList[idx] ) is TPolygonPresenter then
			begin
				Result := TPolygonPresenter( FPresenterList[idx] );
				if ( APen <> nil ) and not Result.PolygonPen.Equals( APen ) then
					continue;
				if ( ABrush <> nil ) and not Result.PolygonBrush.Equals( ABrush ) then
					continue;
				if ( ATitleFont <> nil ) and not Result.TitleFont.Equals( ATitleFont ) then
					continue;
				Exit;
			end;
		Result := nil;
end;

{-------------------------------------------------------------------------
 TPolygonPresenter.LLInObject
-------------------------------------------------------------------------}
function TPolygonPresenter.LLInObject( Geod : TGlobeObject; ptLL : TPointLL ) : Boolean;
var
	iZone, idx, iX1, iX2, iY1, iY2 : Integer;
	iLong, iLat : integer;
begin
	Result := False;
	iZone := Geod.LLInObjectBounds( ptLL );
	if iZone <> 0 then
	begin
		iX2 := 0;
		iY2 := 0;
		for idx := Geod.CountLL - 1 downto 0 do
		begin
			with Geod.PointsLL[idx] do
			begin
				iX2 := iLongX;
				iY2 := iLatY;
			end;
			if ( iZone = 1 ) and ( iX2 >= 0 ) then
				continue;
			if ( iZone = 2 ) and ( iX2 <= 0 ) then
				continue;
			Break;
		end;

		iLong := ptLL.iLongX;
		iLat := ptLL.iLatY;
		idx := 0;
		while idx < Geod.CountLL do
		begin
			with Geod.PointsLL[idx] do
			begin
				iX1 := iLongX;
				iY1 := iLatY;
			end;
			Inc( idx );

			if iZone <> 3 then
			begin
				if ( iZone = 1 ) and ( iX1 > 0 ) then
					continue;
				if ( iZone = 2 ) and ( iX1 < 0 ) then
					continue;
			end;

			if ( ( ( ( iY1 <= iLat ) and ( iLat < iY2 ) )
				or ( ( iY2 <= iLat ) and ( iLat < iY1 ) ) )
				and ( iLong < MulDiv( ( iX2 - iX1 ), ( iLat - iY1 ),
				( iY2 - iY1 ) ) + iX1 ) ) then
				Result := not Result;

			iX2 := iX1;
			iY2 := iY1;
		end;
	end;
end;

{-------------------------------------------------------------------------
 TPolygonPresenter.RenderObject
-------------------------------------------------------------------------}
procedure TPolygonPresenter.RenderObject( Geod : TGlobeObject );
begin
	PolygonPen.RenderAttribute( osSelected in Geod.State );
	PolygonBrush.RenderAttribute( osSelected in Geod.State );
	Globe.DrawLLPolygon( Geod );
end;

{-------------------------------------------------------------------------
 TPolygonPresenter.SaveToStream
-------------------------------------------------------------------------}
procedure TPolygonPresenter.SaveToStream( AStream : TStream );
begin
	inherited SaveToStream( AStream );

	PolygonPen.SaveToStream( AStream );
	PolygonBrush.SaveToStream( AStream );
end;

{-------------------------------------------------------------------------
 TPolygonPresenter.LoadFromStream
-------------------------------------------------------------------------}
procedure TPolygonPresenter.LoadFromStream( AStream : TStream );
begin
	inherited LoadFromStream( AStream );

	FPolygonPen.LoadFromStream( AStream );
	FPolygonBrush.LoadFromStream( AStream );
end;
