{-------------------------------------------------------------------------
	Module:		TGlobe Demo program

	Comment:	Simple demo program for TGlobe.

	Author:		Graham Knight
	Email:		gknight@helmstone.co.uk
	Version:	3.0
	Date:			January 1999

	2.1a:     Fix to object selection in lbxLocationsclick()
	2.2:      Find Location code
	3.00			Converted to TGlobe version 3.00
-------------------------------------------------------------------------}
unit uDemo;

interface

uses
	WinProcs, WinTypes, Messages, SysUtils, Classes, Graphics, Controls, Forms,
	Dialogs, StdCtrls, Buttons, ExtCtrls, Menus, Tabs, inifiles, Spin, clipbrd,
	TGClasses, Globe4, TGObjects, GLobeUtils, TGPresenters, TGsysUtils, Printers;

type
	TfrmMain = class(TForm)
		MainMenu1: TMainMenu;
		File1: TMenuItem;
		Open1: TMenuItem;
		N1: TMenuItem;
		Exit1: TMenuItem;
		Panel3: TPanel;
		OpenDialog1: TOpenDialog;
		pnlHint: TPanel;
		View1: TMenuItem;
		Spherical1: TMenuItem;
		Mercator1: TMenuItem;
		Cartesian1: TMenuItem;
		Panel1: TPanel;
		Notebook1: TNotebook;
		lbxLocations: TListBox;
		Panel4: TPanel;
		btnType: TButton;
		btnTitle: TButton;
		OpenProfile1: TMenuItem;
		OpenDialog3: TOpenDialog;
		NewGlobe: TMenuItem;
		N2: TMenuItem;
		Timer1: TTimer;
		N3: TMenuItem;
		Print1: TMenuItem;
		Panel2: TPanel;
		Panel5: TPanel;
		edtFind: TEdit;
		lbxFind: TListBox;
		Find1: TMenuItem;
		N4: TMenuItem;
		About1: TMenuItem;
    AutomateGlobe1: TMenuItem;
    DayNightShadow1: TMenuItem;
    Panel6: TPanel;
		tabLayers: TTabSet;
    pnlZoom: TPanel;
    SpinButton1: TSpinButton;
    btnZoomIn: TBitBtn;
    btnZoomOut: TBitBtn;
		btnZoomExtents: TBitBtn;
    Transparent1: TMenuItem;
    Globe: TGlobe4;
		procedure btnZoomInClick(Sender: TObject);
		procedure btnZoomOutClick(Sender: TObject);
		procedure btnZoomExtentsClick(Sender: TObject);
		procedure lbxLocationsClick(Sender: TObject);
		procedure FormCreate(Sender: TObject);
		procedure Open1Click(Sender: TObject);
		procedure Exit1Click(Sender: TObject);
		procedure lbxLocationsDrawItem(Control: TWinControl; Index: Integer;
			Rect: TRect; State: TOwnerDrawState);
		procedure GlobeRender(Sender: TObject);
		procedure ProjectionClick(Sender: TObject);
		procedure btnHeaderClick(Sender: TObject);
		procedure SpinButton1DownClick(Sender: TObject);
		procedure SpinButton1UpClick(Sender: TObject);
		procedure tabLayersClick(Sender: TObject);
		procedure OpenProfile1Click(Sender: TObject);
		procedure NewGlobeClick(Sender: TObject);
		procedure GlobeMouseMove(Sender: TObject; Shift: TShiftState; X,
			Y: Integer);
		procedure Timer1Timer(Sender: TObject);
    procedure Print1Click(Sender: TObject);
    procedure GlobePaint(Sender: TObject);
		procedure edtFindChange(Sender: TObject);
    procedure Find1Click(Sender: TObject);
		procedure About1Click(Sender: TObject);
		procedure GlobeDblClick(Sender: TObject);
		procedure AutomateGlobe1Click(Sender: TObject);
		procedure View1Click(Sender: TObject);
		procedure DayNightShadow1Click(Sender: TObject);
    procedure Transparent1Click(Sender: TObject);
    procedure GlobeSelected(Sender: TGlobe4; GlobeObject: TGlobeObject);
	private
		{ Private declarations }
	public
		{ Public declarations }
		procedure RebuildTabs;
		procedure LoadFile( const sFname : string );
		procedure LoadEnvironment( const sPname : string );
	end;

{-------------------------------------------------------------------------}
var
	frmMain: TfrmMain;
	gCurrentLayer : TGlobeLayer;
	MyObject : TGeoDataObject;
	gbAutomate, gbShadow : Boolean;
	giTimerCounter : integer;
{-------------------------------------------------------------------------}
implementation

{$R *.DFM}

{-------------------------------------------------------------------------}
procedure TfrmMain.RebuildTabs;
var
	idx : integer;
begin
	with tabLayers do
	begin
		Tabs.Clear;
		for idx := 0 to Globe.Layers.Count - 1 do
			Tabs.AddObject( Globe.Layers[idx].Name, Globe.Layers[idx] );
		if Tabs.Count > 0 then
			TabIndex := 0;
	end;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.LoadFile( const sFname : string );
var
	LayerIndex : integer;
begin
	with tablayers do
	begin
    LayerIndex := CreateFileLayer( Globe, sFname );

		Tabs.AddObject( Globe.Layers[LayerIndex].Name, Globe.Layers[LayerIndex] );
		TabIndex := Tabs.Count - 1;
	end;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.NewGlobeClick(Sender: TObject);
begin
	lbxLocations.Clear;
	tabLayers.Tabs.Clear;
	Globe.Clear;
	MyObject := nil;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.LoadEnvironment( const sPname : string );
begin
	NewGlobeClick( nil );
	Globe.EnvironmentFile := sPname;
	RebuildTabs;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.SpinButton1DownClick(Sender: TObject);
begin
	with Globe.Projection do
		ScaleFactor := ScaleFactor / 1.1;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.SpinButton1UpClick(Sender: TObject);
begin
	with Globe.Projection do
		ScaleFactor := ScaleFactor * 1.1;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.btnZoomInClick(Sender: TObject);
begin
	with Globe.Projection do
		ScaleFactor := ScaleFactor * 2;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.btnZoomOutClick(Sender: TObject);
begin
	with Globe.Projection do
		ScaleFactor := ScaleFactor / 2;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.btnZoomExtentsClick(Sender: TObject);
begin
	with Globe do
		if Layers.SelectedObject <> nil then
		begin
      ZoomToMER( Globe, Layers.SelectedObject.ObjectMER );
		end
		else
      Projection.ScaleFactor := 0.0;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.lbxLocationsClick(Sender: TObject);
begin
	with TListBox( Sender ) do
		if ItemIndex <> -1 then
		begin
			Globe.Layers.SelectedObject := TGlobeObject( Items.Objects[ItemIndex] );
			Globe.LocateToObject( Globe.Layers.SelectedObject );
		end;
end;


{-------------------------------------------------------------------------}
procedure TfrmMain.FormCreate(Sender: TObject);
var
	ALayer : TGlobeLayer;
//	PointPresenter : TPointPresenter;
	PointPresenter : TGlobePresenter;
begin
	LoadEnvironment( 'tglobe.env' );	{ load the default Environment }

	ALayer := TGlobeLayer.Create( Globe );
	ALayer.Animated := True;

	PointPresenter := TPointPresenter.Create( Globe, 99 );
  Globe.Layers.GlobalPresenters.Add( PointPresenter );

	MyObject :=	TGeoDataObject.Create( ALayer.Objects );
  MyObject.PresenterID := PointPresenter.PresenterID;

	with TPointPresenter( PointPresenter ) do
	begin
		PointType := ppFontSymbol;	{ display as a TT font character }
		with PointFont do		{ set up the font }
		begin
			FontName := 'WingDings';
			FontColor := clRed;
			FontUnit := Pixel;
			FontSize := 14;
		end;
		SymbolIndex := 81;		{ select the aircraft character from the font }
	end;

	Globe.Projection.Altitude := 0;	{ Zoom out to globe extents }
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.Exit1Click(Sender: TObject);
begin
	Globe.Clear;
	Close;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.Open1Click(Sender: TObject);
begin
	with OpenDialog1 do
	begin
		InitialDir := ExtractFilePath(Application.ExeName);
		FileName := '';

		if Execute then
			LoadFile( FileName );
	end;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.OpenProfile1Click(Sender: TObject);
begin
	with OpenDialog3 do
	begin
		InitialDir := ExtractFilePath(Application.ExeName);
		FileName := Globe.EnvironmentFile;
		if Execute then
			LoadEnvironment( FileName );
	end;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.lbxLocationsDrawItem(Control: TWinControl;
	Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
	oObj : TGlobeObject;
begin
	with TListBox( Control ) do
	begin
		oObj := TGlobeObject( Items.Objects[Index] );
		Canvas.TextRect( Rect, Rect.Left, Rect.Top, Copy( oObj.ClassName, 7, 255 ));

		Rect.Left := btnType.Width;
		Canvas.TextRect( Rect, Rect.Left, Rect.Top, oObj.Title);
	end;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.btnHeaderClick(Sender: TObject);
var
	idx, jdx : integer;
	iGap : integer;
	lExchanges : Longint;
	bSwap : Boolean;
begin
	Screen.Cursor := crHourGlass;
	with lbxLocations do
	begin
		Items.BeginUpdate;

		bSwap := False;

		iGap := Items.Count - 1;
		repeat
			iGap := Trunc( iGap / 1.3 );
			Case iGap of
			0 :			iGap := 1;
			9,10 :	iGap := 11;
			end;

			lExchanges := 0;
			for idx := 0 to Items.Count - 1 - iGap do
			begin
				jdx := idx + iGap;

				case TBitBtn( Sender ).Tag of
				0 :	{ By Type }
					bSwap := TGlobeObject( Items.objects[idx]).ClassName > TGlobeObject( Items.objects[jdx]).ClassName;
				1 :	{ By Title }
					bSwap := TGlobeObject( Items.objects[idx]).Title > TGlobeObject( Items.objects[jdx]).Title;
				end;

				if bSwap then
				begin
					Items.Exchange( idx, jdx );
					Inc( lExchanges );
				end;
			end;
		until ( lExchanges = 0 ) and ( iGap = 1 );

		Items.EndUpdate;
	end;
	Screen.Cursor := crDefault;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.GlobeRender(Sender: TObject);
var
	iLeft, iTop, iBkMode, iTA : integer;
	Units : TGlobeUnitTypes;
	iBias : integer;
{$IFDEF WIN32}
	TimeZone : TTimeZoneInformation;
{$ENDIF}
begin
	if gbShadow then
	begin
		iBias := 0;
{$IFDEF WIN32}
		case GetTimeZoneInformation( TimeZone ) of
		1 :		iBias := TimeZone.Bias + TimeZone.StandardBias;
		2 :		iBias := TimeZone.Bias + TimeZone.DayLightBias;
		end;
{$ENDIF}
		Globe.RenderShadow( Now + iBias/1440, clBlack, clSilver, 2 );
	end;

	iLeft := Globe.Width - ( 50 + Screen.PixelsPerInch );
	iTop := Globe.Height - 50;

	with Globe.GlobeCanvas do
	begin
		Font.Assign( Self.Font );

		iBkMode := SetBkMode( Handle, TRANSPARENT );
		iTA := SetTextAlign( Handle, TA_CENTER );

		Font.Color := clRed;
		TextOut( iLeft, iTop, '0' );
		Units := KiloMeter;
		if GlobeUnitsTo( Globe.Projection.UnitsPerInch, KiloMeter ) < 10 then
			if GlobeUnitsTo( Globe.Projection.UnitsPerInch, Meter ) < 10 then
				Units := Centimeter
			else
				Units := Meter;

		Font.Color := clRed;
		TextOut( iLeft + Screen.PixelsPerInch, iTop,
			Format( '%d %s',[Round( GlobeUnitsTo( Globe.Projection.UnitsPerInch, Units )), UnitsToStr( Units )] ));

		SetBkMode( Handle, iBkMode );
		SetTextAlign( Handle, iTA );

		Pen.color := clRed;
		MoveTo( iLeft, iTop + 16 );
		LineTo( iLeft, iTop + 20 );
		LineTo( iLeft + Screen.PixelsPerInch, iTop + 20 );
		LineTo( iLeft + Screen.PixelsPerInch, iTop + 15 );
	end;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.ProjectionClick(Sender: TObject);
begin
	gbAutomate := false;
	Spherical1.Checked := False;
	Mercator1.Checked := False;
	Cartesian1.Checked := False;

	with Sender as TmenuItem do
		Checked := True;

	if Spherical1.Checked then
		Globe.Projection.ProjectionClass := 'TSphericalPrj';
	if Mercator1.Checked then
		Globe.Projection.ProjectionClass := 'TMercatorPrj';
	if Cartesian1.Checked then
		Globe.Projection.ProjectionClass := 'TCartesianPrj';
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.tabLayersClick(Sender: TObject);
var
	idx : integer;
begin
	Screen.Cursor := crHourGlass;

	with lbxLocations do
	try
		Items.BeginUpdate;
		Clear;

		with tabLayers do
			if TabIndex <> -1 then
			begin
				gCurrentLayer := TGlobeLayer( Tabs.Objects[TabIndex] );
				with gCurrentLayer do
				begin
					for idx := 0 to Objects.Count - 1 do
					begin
						if idx < 32700 then
							Items.AddObject( '', Objects[idx] );
					end;
				end;
			end;
	finally
		Items.EndUpdate;
		Screen.Cursor := crDefault;
	end;
end;


{-------------------------------------------------------------------------}
procedure TfrmMain.GlobeSelected(Sender: TGlobe4;
	GlobeObject: TGlobeObject);
var
	idx : integer;
  ObjLayer : TGlobeLayer;
begin
  caption := 'Select';

	if Globe.Layers.SelectedObject <> nil then
	begin
		btnZoomExtents.Caption := 'Object Extents';
    ObjLayer := LayerFromObject( Globe, Globe.Layers.SelectedObject );
		if gCurrentLayer <> ObjLayer then
			with TabLayers do
				for idx := 0 to Tabs.Count - 1 do
					if TGlobeLayer( Tabs.objects[idx] ) = ObjLayer then
					begin
						tabIndex := idx;
						Break;
					end;

		with lbxLocations do
			ItemIndex := Items.IndexOfObject( Globe.Layers.SelectedObject );
	end
	else
		btnZoomExtents.Caption := 'World Extents';
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.GlobeMouseMove(Sender: TObject; Shift: TShiftState; X,
	Y: Integer);
var
	sTmp : string;
	pt : TPointLL;
	MouseObj : TGlobeObject;
begin
	with Globe do
	begin
		Projection.DeviceXYToLL( X, Y, pt );

		with pt do
			sTmp := GlobeUnitsToStr( iLongX, '%d.%m.%s.%t%E ' )
        + GlobeUnitsToStr( iLatY, '%d.%m.%s.%t%N' );

		MouseObj := Layers.ObjectAtXY( X, Y );
		if MouseObj <> nil then
			sTmp := sTmp + ' ' + MouseObj.Title;

		pnlHint.Caption := Format( ' Mouse Position: %s', [sTmp] );
	end;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.Timer1Timer(Sender: TObject);
var
	iX, iY : integer;
begin
	if MyObject <> nil then
		with MyObject.Centroid do
		begin
			iX := ( iLongX + GU_DEGREE ) mod GU_360_DEGREE;
			iY := Round( 60 * GU_DEGREE * Cos( iLongX * GU_TORADIANS ));
			MyObject.Centroid := PointLLH( iX, iY, GU_DEGREE * 5 );
		end;

	if gbAutomate then
		Globe.Projection.YRotation := Globe.Projection.YRotation - 20 * GU_MINUTE;

	if gbShadow then
	begin
		Dec( giTimerCounter );
		if giTimerCounter <= 0 then
		begin
			giTimerCounter := 40;
			Globe.RedrawLayers;	{ update every 10 seconds }
		end;
	end;
end;


{-------------------------------------------------------------------------}
procedure TfrmMain.Print1Click(Sender: TObject);
begin
  Printer.BeginDoc;
  Globe.RenderToCanvas( Printer.Canvas, Globe.ClientRect );
  Printer.EndDoc;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.GlobePaint(Sender: TObject);
var
	ptLondon : TPointLL;
	pt, pt1 : TPoint;
begin
	if MyObject <> nil then
	begin
		ptLondon := PointLL( DecimalToGlobeUnits( -0.007 ), DecimalToGlobeUnits( 51.52 ));

		Globe.RenderCircle( ptLondon, GlobeUnitsFrom( 200, kilometer ), 3 );
		Globe.RenderCircle( MyObject.Centroid, GU_DEGREE * 10, 36 );
		Globe.RenderGreatCircleLine( ptLondon, MyObject.Centroid, gcShortest );

//		Caption := Format( 'GreatCircle Distance from London %f km', [GlobeUnitsTo(
//			Globe.EllipsoidDistanceLLtoLL( ptLondon, MyObject[0] ), KiloMeter ) ] );

		Globe.Projection.LLToDeviceXY( MyObject.Centroid.iLongX, MyObject.Centroid.iLatY, pt );

		with GreatCirclePoint( MyObject.Centroid, ptLondon, 5 * GU_DEGREE / EllipsoidDistanceLLToLL( MyObject.Centroid, ptLondon, WGS84 )) do
			Globe.Projection.LLToDeviceXY( iLongX, iLatY, pt1 );

  	{ rotate the font so that the aircraft points away from London }
    with TPointPresenter( Globe.Layers.GlobalPresenters.ByID( MyObject.PresenterID, False )) do
      PointFont.FontAngle :=
				-Sign( MyObject.Centroid.iLongX ) * 900 + Round( ArcTan2( pt.x - pt1.x, pt.y - pt1.y ) * (180/LocalPI)) * 10;
	end;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.edtFindChange(Sender: TObject);
var
	iLayer, idx : integer;
	sTmp : string;
begin
	lbxFind.Items.Clear;
	sTmp := edtFind.Text;

	if Length( sTmp ) <> 0 then
		with Globe do
			for iLayer := 0 to Layers.Count - 1 do
				with Layers[iLayer] do
					for idx := 0 to Objects.Count - 1 do
						if comparetext( sTmp, Copy( Objects[idx].Title, 1, Length( sTmp ))) = 0 then
							lbxFind.Items.AddObject( Objects[idx].Title + #9 + IntToStr( iLayer ), Objects[idx]);
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.Find1Click(Sender: TObject);
begin
	lbxFind.Items.Clear;
	edtFind.text := '';

	if NoteBook1.ActivePage = 'Layers' then
	begin
		Find1.Caption := 'Layers';
		NoteBook1.ActivePage := 'Find';
	end
	else
	begin
		Find1.Caption := 'Find';
		NoteBook1.ActivePage := 'Layers';
	end;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.About1Click(Sender: TObject);
begin
	ShowMessage( 'TGlobe Demo Program'
		+ #10#10'Copyright (c) Graham Knight 1997 - 1999'
		+ #10#10'Email: gknight@helmstone.co.uk'
		+ #10'http://www3.mistral.co.uk/helmstone/tglobe' );
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.GlobeDblClick(Sender: TObject);
begin
//	GlobeEditor( Globe );
{	Clipboard.Assign( Globe.Bitmap );
	ShowMessage( 'Image Copied to Clipboard' );
}end;

{-------------------------------------------------------------------------}
procedure TfrmMain.AutomateGlobe1Click(Sender: TObject);
begin
	gbAutomate := not gbAutomate;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.View1Click(Sender: TObject);
begin
	AutomateGlobe1.Enabled := Globe.Projection.ProjectionClass = 'TSphericalPrj';
	AutomateGlobe1.Checked := gbAutomate;
	DayNightShadow1.Checked := gbShadow;
	Transparent1.Checked := goTransparentGlobe in Globe.GlobeOptions;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.DayNightShadow1Click(Sender: TObject);
begin
	gbShadow := not gbShadow;
end;

{-------------------------------------------------------------------------}
procedure TfrmMain.Transparent1Click(Sender: TObject);
begin
  if Transparent1.Checked then
  	Globe.GlobeOptions := Globe.GlobeOptions + [goTransparentGlobe]
  else
  	Globe.GlobeOptions := Globe.GlobeOptions - [goTransparentGlobe];
end;

end.
