//             ___________ __         ______
//            /_  __/ ___// /_  ___  / / / /_________  ___
//             / /  \__ \/ __ \/ _ \/ / / __/ ___/ _ \/ _ \
//            / /  ___/ / / / /  __/ / / /_/ /  /  __/  __/
//           /_/  /____/_/ /_/\___/_/_/\__/_/   \___/\___/

(*******************************************************************************
* TShellTree 1.17a                                                             *
********************************************************************************
*                                                                              *
* If you find bugs, has ideas for missing featurs, feel free to contact me     *
*               jpstotz@gmx.de                                                 *
*                                                                              *
* The latest version of TShellTree can be found at:                            *
*               http://jpstotz.com7.org/TShellTree                             *
********************************************************************************
* Date last modified:   Oct 01, 2001                                           *
*******************************************************************************)

unit ShellTree;

{$I DFS.INC}  { Standard defines for all Delphi Free Stuff components }

{$RANGECHECKS OFF}
{$OVERFLOWCHECKS OFF}
{$LOCALSYMBOLS ON}
{$OPTIMIZATION OFF}

interface

{$DEFINE DoRepaintProperty}
{$DEFINE CheckBoxes}

{.$DEFINE HideRootNode}
//If not remarked, the rootobject will not be visible

{.$DEFINE OVERSEER}
//External Debugger
{$IFDEF OVERSEER}{$DEFINE Debug}{$ENDIF}

{$IFDEF DFS_COMPILER_2}
   'Sorry, Delphi2 is not supported'
{$ENDIF}

uses
  //Standard Units that comes with Delphi
  ShlObj, ActiveX, Dialogs, Windows,  Messages,  SysUtils,  Classes,
  Graphics, Controls, Forms, StdCtrls, ComCtrls, CommCtrl, ShellApi,
  Menus,
  //Units that belong to TShellTree or are used by it.
  ItemProp, ShellTree_Dbt;

const
  CSIDL_USERDEFINED = -1;
  STD_DIR = 'C:\';
  TVIS_CHECKED  = $2000;
  TVIS_UNCHECKED = $1000;

  WM_SelectPidl     = WM_USER + 2;


// All-Attributes is passed to an IShellFolder when creating or updating a node.
// The remarked attributes cause on some systems an access to the drives A and B
  All_Attributes = {SFGAO_CANRENAME or SFGAO_CANDELETE or }SFGAO_HASPROPSHEET or
                   SFGAO_DROPTARGET or SFGAO_CAPABILITYMASK or SFGAO_LINK or
                   SFGAO_SHARE or {SFGAO_READONLY or} SFGAO_GHOSTED
                   or {SFGAO_DISPLAYATTRMASK or} SFGAO_FILESYSANCESTOR or
                   SFGAO_FOLDER or SFGAO_FILESYSTEM or SFGAO_HASSUBFOLDER or
                   SFGAO_CONTENTSMASK {or SFGAO_VALIDATE}
                   or SFGAO_REMOVABLE or SFGAO_COMPRESSED;

type
// TShellTree_Types declerations

  TFileSortOrder  = (so_Ascending,so_Descending);
  TFileSortType   = (so_Name,so_Size,so_Type,so_Date);
  TPopupMenuMode  = (pmm_System,pmm_PopupMenu);

  TSortData = record
    FileSortOrder : TFileSortOrder;
    FileSorttype  : TFileSortType;
  end;

  TLastAction = (la_None,la_RootPidl,la_RootDir,la_Dir);

  TPidls = (PIDL_DESKTOP,PIDL_PROGRAMS,PIDL_CONTROLS,PIDL_PRINTERS,PIDL_PERSONAL,
            PIDL_FAVORITES,PIDL_STARTUP,PIDL_RECENT,PIDL_SENDTO,PIDL_BITBUCKET,
            PIDL_STARTMENU,PIDL_DESKTOPDIRECTORY,PIDL_DRIVES,PIDL_NETWORK,PIDL_NETHOOD,
            PIDL_FONTS,PIDL_TEMPLATES,PIDL_USERDEFINED);
//The TPidls type is used by the RootPidl property of TShellTree.
  TAttribute = (Attr_CANCOPY, Attr_CANMOVE, Attr_CANLINK, Attr_CANRENAME,
                Attr_CANDELETE, Attr_HASPROPSHEET, Attr_DROPTARGET,
                Attr_CAPABILITYMASK, Attr_LINK, Attr_SHARE, Attr_READONLY,
                Attr_GHOSTED, Attr_DISPLAYATTRMASK, Attr_FILESYSANCESTOR,
                Attr_FOLDER, Attr_FILESYSTEM, Attr_HASSUBFOLDER,
                Attr_CONTENTSMASK, Attr_VALIDATE, Attr_REMOVABLE,
                Attr_COMPRESSED);

  TAttributes  = set of TAttribute;
//Contains a set of all attributes for folders or files.
  TVisOption  = (TF_FOLDERS,TF_NONFOLDERS,TF_HIDDEN);
//TF_FOLDERS    : Show all folders (except the hidden)
//TF_NONFOLDERS : Show all nonfolder elements (files, system entries...)
//TF_HIDDEN     : Shaw all hidden folder (Recycled) and files
  TVisOptions = set of TVisOption;

  PNodeInfo = ^TNodeInfo;
  TNodeInfo = record
    RelativeIDL       : PItemIDList;
    AbsoluteIDL       : PItemIDList;
    ParentShellFolder : IShellFolder;
    ShellFolder       : IShellFolder;
    Attributes        : UInt;
    Expanded          : Boolean;
    Updated           : Boolean;
  end;

  TShellNode = class;
  TShellTree = class;

  TDiscMonitorThread = class(TThread)
  private
    FShellNode : TShellNode;
    FDirectory : string;
    FFilters : integer;
    FDestroyEvent,
    FChangeEvent : THandle;
    FInvalid : Boolean;
    procedure InformChange;
    procedure SetDirectory (const Value : string);
    procedure SetFilters (Value : integer);
  protected
    procedure Execute; override;
    procedure Update;
  public
    constructor Create(aShellNode : TShellNode);
    destructor Destroy; override;
    property Invalid : Boolean read FInvalid;
    property Directory : string read FDirectory write SetDirectory;
    property Filters : integer read FFilters write SetFilters;
  end;


// TShellNode decleration

  TShellNode = class(TTreeNode)
  private
    FBinded : Boolean;
    FHasMonitorThread : Boolean;
    NodeInfo : TNodeInfo;
    {$IFDEF CheckBoxes}
    FChecked : Boolean;
    {$ENDIF}
    function GetAttributes : TAttributes;
    function GetIsFile : Boolean;
    function GetIsFolder : Boolean;
    function GetIsSysObject : Boolean;
    function GetShellFolder : IShellFolder;
    function GetNodePidlLevel : Integer;
    function GetNodePath  : String;
    function GetExtension : String;
    function GetFileName  : String;
    procedure FalseUpdated;
    procedure DeleteFalseUpdated;
    procedure UpdateNodeInfo(ParentISF : IShellFolder;Pidl : PItemIDList;Recursive : Boolean);
    {$IFDEF CheckBoxes}
    function GetChecked :Boolean;
    procedure SetChecked(value:Boolean);
    {$ENDIF}
  protected
    FMonitorThread : TDiscMonitorThread;
    procedure UpdateIcon; virtual;
    procedure CreateMonitorThread;
    procedure DestroyMonitorThread;
    procedure DestroySubMonitorThreads(Firstlevel : Boolean);
    procedure FolderChanged;
    procedure CreateShellFolder;
    procedure FreeShellfolder;
    procedure ChangeDiscMonitorFilter(NewFilters : Integer);
    function  GetNodeText : String; virtual;
 public
    destructor Destroy; override;
    constructor Create(AOwner: TTreeNodes); virtual;
    procedure PerformDefaultAction;
    procedure PerformVerb(const verb : string);
    procedure UpdateNodeText;
    function  GetSubPath(ItemIdList : PItemIDList) : STRING;
    function  HasAttribute(Attrib:  UInt) : Boolean;
    function  SetDisplayName(NewName : String) : Boolean; virtual;
    property  HasMonitorThread : Boolean read FHasMonitorThread;
    property  MonitorThread : TDiscMonitorThread read FMonitorThread;
    property  IsFile            : Boolean      read GetIsFile;
    property  IsFolder          : Boolean      read GetIsFolder;
    property  IsSysObject       : Boolean      read GetIsSysObject;
    property  ParentShellFolder : IShellFolder read Nodeinfo.ParentShellfolder;
    property  Shellfolder       : IShellFolder read GetShellFolder;
    property  AbsoluteIDL       : PItemIDList  read NodeInfo.AbsoluteIDL;
    property  RelativeIDL       : PItemIDList  read NodeInfo.RelativeIDL;
    property  Attributes        : TAttributes  read GetAttributes;
    property  Attr              : UInt         read NodeInfo.Attributes;
    property  PidlLevel         : Integer      read GetNodePidlLevel;
    //Returns the number of ItemId's in the AbsoluteIDL.
    //This value is independent from the current visible rootitem !
    //
    //For Example :
    //Desktop = 0
    //MyComputer = 1;
    //Drive C:\ = 2;
    //...
    property  Path        : String       read GetNodePath;
    //If the ShellNode is not a Systemobject, the path of the object
    //will be returned.
    //If the object is a Folder, thePath property returns the full Path of the Folder
    //If the object is a File, the complete Path inclusive Filename will be returned.
    property  Filename    : String       read GetFilename;
    //If the ShellNode is a file, it returns the filename
    //without path of the file.
    //If not, the result is an empty string
    property  Ext         : String       read GetExtension;
    //If the ShellNode is a file, it returns the extention
    //of the file. If not, the result is an empty string
    {$IFDEF CheckBoxes}
    property Checked : Boolean read GetChecked write SetChecked default False;
    {$ENDIF}
  end;

  TOnFileAddEvent         = procedure (const Filename : String;Attributes : UInt;Var Allow : Boolean) of object;
  TOnDirAddEvent          = procedure (const Dirname : String;Attributes : UInt;Var Allow : Boolean) of object;
  TOnPopupMenuEvent       = procedure (Node : TShellNode; Var Allow : Boolean) of object;
  TOnCheckBoxChangedEvent = procedure (Node : TShellNode; NewState : Boolean) of object;
  TSysCMenuPopupEvent     = procedure (Node : TShellNode; Var Menu : HMenu) of object;
  TSysCMenuIDEvent        = procedure (ID : Integer) of object;

// TShellTree decleration

  TShellTree = class(TCustomTreeView)
  private
{$IFNDEF DFS_COMPILER_4_UP}
    FToolTips : Boolean;
{$ENDIF}
    FTmpStartpath : String;

    FDeviceChangeHandle : HWnd;
    FDesktop        : IShellFolder;
    FDesktopPidl    : PItemIDList;
    FMyComputer     : IShellFolder;
    FMyComputerPidl : PItemIDList;
    {$IFDEF DoRepaintProperty}
      FDoRepaint : Boolean;
    {$ENDIF}
    FRootDir  : String;
    FRootPidl : TPidls;
    FOptions  : TVisOptions;
    FRootUsePidl   : Boolean;
    FDesignStart   : Boolean;
    FInitialized   : Boolean;
    FUpdAutoThread : Boolean;
    FUpdAuto       : Boolean;
    FUpdAutoRec    : Boolean;
    FShowIcons     : Boolean;
    FRootUserPidl  : PItemIDList;
    FUseShellCache : Boolean;
    FShowVirtualFolders  : Boolean;
    FFileMaskList : TStringList;
    FFileMaskUse : Boolean;
    FOnDirAdd  : TOnDirAddEvent;
    FOnFileAdd : TOnFileAddEvent;
    FFileSortType  : TFileSortType;
    FFileSortOrder : TFileSortOrder;
    FShowSharedOverlay : Boolean;
    FShowLinkOverlay : Boolean;
    FPopupMenuMode : TPopupMenuMode;
    FPopupMenu : TPopupMenu;
    FMessageHandle : HWnd;
    FShowMessages : Boolean;
    FOnPopupMenu : TOnPopupMenuEvent;
    FSysCMenu : TSysCMenuPopupEvent;
    {$IFDEF CheckBoxes}
    FCheckBoxes : Boolean;
    FOnCheckBoxChanged : TOnCheckBoxChangedEvent;
    procedure SetCheckBoxes(value:Boolean);
    {$ENDIF}
    procedure SetShowMessages(Value : Boolean);
    procedure SetShowSharedOverlay(Value : Boolean);
    procedure SetShowLinkOverlay(Value : Boolean);
    procedure SetFileSortType(Value : TFileSortType);
    procedure SetFileSortOrder(Value : TFileSortOrder);
    procedure SetFileMask(Value : String);
    procedure SetFileMaskUse(Value : Boolean);
    procedure SetPopupMenu   (value : TPopupMenu);
    procedure SetRootDir     (Value : String);
    procedure SetRootPidl    (Value : TPidls);
    procedure SetOptions     (Value : TVisOptions);
    procedure SetRootUsePidl (Value : Boolean);
    procedure SetPath        (Value : String);
    procedure SetShowIcons   (Value : Boolean);
    procedure SetRootUserPidl(Value : PItemIDList);
    procedure SetShowVirtualFolders(Value : Boolean);
    function  GetPath          : String;
    function  GetRelativePidl  : PItemIDList;
    function  GetAbsolutePidl  : PItemIDList;
    function  GetShellFolder   : IShellFolder;
    function  GetFileMask      : String;
    function  GetMessageHandle : HWnd;
{$IFNDEF DFS_COMPILER_4_UP}
    procedure SetToolTips(Value : Boolean);
{$ENDIF}
  protected
    DiscMonitorFilters : Integer;
    Allocator: IMalloc;
    DefaultSortProc : TTVCompare;
    SysImageListSmall : HImageList;
    FRootNode : TShellNode;

    procedure DiscMonitorFiltersChange;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure RetrieveDefaultPidls;
    procedure RetrieveSysImageList; virtual;
    procedure DeviceChangeWndProc(Var Message : TMessage);
    function  CreateNode : TTreeNode; override;
    procedure CreateWnd; override;
    procedure InitTree;
    procedure Loaded; override;
    function  CanExpand(Node: TTreeNode): Boolean; override;
    function  CanEdit(Node : TTreeNode) : Boolean; override;
    function  CanCollapse(Node: TTreeNode): Boolean; override;
    procedure Edit(const Item: TTVItem); override;
    procedure CreateRoot;

    procedure WndProc(Var Message : TMessage); override;
    procedure WMContextMenu(Var Message : TMessage); message WM_Contextmenu;
    procedure WMMouseWheel(Var Message: TMessage); message WM_Mousewheel;
    procedure WMSelectPidl(Var Message: TMessage); message WM_SelectPidl;
    procedure PopupSystemContextMenu(Node : TShellNode; Point : TPoint);
    function  AllowPopupMenu(Node : TShellNode) : Boolean; virtual;
    procedure RetrieveMessageHandle;

    procedure CreateRootPidl; virtual;
    //In derivations :
    //This method can be used to add one or more userdefined rootitems to the Tree.
    //The new rootitems appear before or after the rootitem specified by the
    //ShellTree properties.
    //For detailed information please contact the author.
    procedure MediaChangeDetected(Msg : Integer;HDR : PShellTree_Dev_Broadcast_Hdr);
    procedure UpdateMyComputer(MyComputerNode : TShellNode);
    procedure UpdateSpecialDrives(Msg : Integer;HDR : PShellTree_Dev_Broadcast_Volume;MyComputerNode : TShellNode);
    procedure UpdateSubNodes(Node : TTreeNode;Recursive : Boolean);
    procedure CreateRootNode(ItemIdList : PItemIDList);
    procedure EnumFolder(Node: TShellNode;GetAllEntries : Boolean);
    function  AddSubfolderToNode(Node: TShellNode; ItemIdList: PItemIDList) : Boolean;
    procedure UpdateTreeView(LastAction : TLastAction);
    procedure CheckOnDirAdd(ParentNode : TShellNode;const ItemIdList : PItemIDList;Attributes : UInt;Var DoAdd : Boolean);
    procedure CheckOnFileAdd(ParentNode : TShellNode;const ItemIdList : PItemIDList;Attributes : UInt;Var DoAdd : Boolean);
    procedure SortNode(Node : TTreeNode;Recursive : Boolean);
    function  GetPidlFromPath(ShellFolder : IShellFolder;Path : String;Checkpath : Boolean; var Pidl : PItemIDList) : Boolean;
    function  GetNodeFromIDList(IDList : PItemIDList) : TShellNode;
    procedure SelectPath(Path : String);
    function  FindPidl(IdList : PItemIDList;ChangeView,OnlyFullPath,AbsolutePidl : Boolean) : TShellNode;
    procedure CreateParams(var Params: TCreateParams); override;
    {$IFDEF CheckBoxes}
    procedure CNNotify(var Message : TWMNotify);message CN_NOTIFY;
    procedure CheckBoxChanged(Node : TShellNode; NewState : Boolean);
    function  GetNodeFromItem(const Item: TTVItem): TTreeNode;
    {$ENDIF}
    property  FileMaskList : TStringList read FFileMaskList write FFilemaskList stored false;
  public
    constructor Create(AOwner : TComponent); override;
    destructor  Destroy; override;
    procedure UpdateTree;
    //Updates the whole Tree with all expanded entries.
    //If something has changed in the filestructure it will be now
    //displayed correctly.
    procedure UpdateNode(Node : TTreeNode;Recursive : Boolean);
    //Updates only one specified Node of the ShellTree.
    //If Recursive=True and the specified Node is the
    //Rootnode of the Tree, you will get the same result
    //like the TShellTree.UpdateTree method.
    //If Recursive=False, only the direct Childnodes of the specified Node
    //will be updated.
    property Desktop    : IShellFolder read FDesktop;
    property MyComputer : IShellFolder read FMyComputer;

    property Selected;
    //Returns just like TTreeView the selected Node of the ShellTree.
    //Select return the Node as a TTreeNode, but it is
    //a TShellNode.
    property RelativePidl : PItemIDList read GetRelativePidl;
    //Returns the relative ItemIDList of the selected Node.
    //Relative means, this ItemIDList contains only one element that
    //indentifies an Systemelement relative from it's parentfolder.
    //TShellTree.RelativePidl is equal to TShellNode(TShellTree.Selected).RelativeIDL
    property AbsolutePidl : PItemIDList read GetAbsolutePidl;
    //Returns the absolute ItemIDList of the selected Node.
    //Absolute means, this ItemIDList contains all elements that
    //identifies an Systemelement from the Computer's Dektop.
    //TShellTree.AbsolutePidl is equal to TShellNode(TShellTree.Selected).AbsoluteIDL
    property ShellFolder  : IShellFolder read GetShellFolder;
    //Returns the IShellFolder interface of the selected Node.
    //If no node is selected or the selected node does not have
    //a IShellFolder interface the returnvalue will be nil.
    property RootUserPidl : PItemIDList read FRootUserPidl write SetRootUserPidl stored false;
    //This property allows the user to specify an ItemIdList
    //that will be displayed as the rootitem when RootUsePidl=True
    //and RootPidl=PIDL_USERDEFINED.
    //Be careful, TShellTree frees the given ItemIdList if a new ItemIdList
    //is set by RootUsePidl.
    //To reset to the Desktop (default value) set
    //RootUserPidl:=nil
    property RootNode : TShellNode read FRootNode;
    //Returns the TShellNode of the actual rootnode,
    //specified by the RootPidl, RootUsePidl and RootDir property.
    property MessageHandle : HWnd read GetMessageHandle;
    //Returns the window handle that should be used for system-errormessages.
    function SelectSpecialFolder(Node : TPidls) : TShellNode;
    //Selects a special systemfolder. If it is not visible
    //SelectSpecialFolder makes it visible. The return value is the
    //selected node.
    function SpecialFolder(Node : TPidls) : TShellNode;
    //Returns the node for a special systemfolder if it is visible
    //or nil when it is invisible
    property Items stored false;
    property Imagelist : HImageList read SysImageListSmall;
    //Returns the Handle of the SystemImageList that contains the small icons
    //(normally with a size of 16x16).
    //This handle can be used by SHGetFileInfo to recieve several
    //informations about a file or a directory (icon, attributes,...).
    {$IFDEF DoRepaintProperty}
      property DoRepaint : Boolean read FDoRepaint write FDoRepaint;
    {$ENDIF}
    function GetNodeFromPath(Path : String) : TShellNode;
  published
    property UseShellCache : Boolean read FUseShellCache write FUseShellCache default false;
    property ShowVirtualFolders: Boolean read FShowVirtualFolders write SetShowVirtualFolders;
    //ShowVirtualFolder allows you to enable or disable the displaying
    //of System-Folders like :
    //Controls folder
    //Dial Up Network (as Parent of the Desktop and of MyComputer)
    //Printers folder
    //Recycled (as Parent of the Desktop)
    property ShowSharedOverlay : Boolean read FShowSharedOverlay write SetShowSharedOverlay;
    //Enables or disables the "hand" of Windows that marks shared drives, folders and files.
    property ShowLinkOverlay   : Boolean read FShowLinkOverlay write SetShowLinkOverlay;
    //Enables or disables the "arrow" of Windows that marks links.
    property RootUsePIDL : Boolean read FRootUsePidl write SetRootUsePidl;
    //RootUsePidl can switch between two different Rootelements.
    //The one is defined by the RootPidl property. (RootUsePidl=False)
    //The second is userdefined by the RootDir property. (RootUsePidl=True)
    property RootPIDL    : TPidls read FRootPidl write SetRootPidl;
    //This property allows to specify one of the System's
    //default directories as the rootitem of TShellTree.
    //This property is only used when RootUsePidl=True.
    property RootDir     : String read FRootDir  write SetRootDir;
    //This property allows to set an individual RootDirectory.
    //If the directory exists and RootUsePIDL=FALSE, the specefied
    //directory will become the first visible item in the ShellTree.
    property Options     : TVisOptions read FOptions write SetOptions;
    //The options property is a set of three values that
    //let the user specify what type of folders or files
    //should be displayed in TShellTree. All three values
    //are of the Boolean type. When they are enabled(True)
    //the specified items will appear in the ShellTree, when
    //they are disabled(False) they will disappear.
    //
    //TF_FOLDERS : displays standard system and filesystem folders
    //TF_NONFOLDERS : displays files
    //TF_HIDDEN : displays hidden files and folder (Recycled,...)
    property Path        : String read GetPath write SetPath;
    //Returns the current path inclusive filename
    //If it is a shellobject Path returns an empty string.
    property UpdAutoThread : Boolean read FUpdAutoThread write FUpdAutoThread;
    //Enables or Diables the Automatic Updating by a thread
    //If Enabled a path of TShellTree updates itself directly
    //when changes in the filsystem are detected.
    property UpdAuto     : Boolean read FUpdAuto write FUpdAuto;
    //Enables or Diables the Automatic Updating
    //If Enabled everytime an item get expanded a second or third time, TShellTree
    //checks if all direct subdirectories are still valid.
    property UpdAutoRec  : Boolean read FUpdAutoRec write FUpdAutoRec;
    //Enables or Disables the Automatic Updateing
    //If Enabled everytime an item get expanded a second or third time, TShellTree
    //checks if all subdirectories are still valid.
    //If the reexpanded folder contains other visible (expanded)
    //subfolder, UpdAutoRec will recursively check them.
    property ShowIcons   : Boolean read FShowIcons write SetShowIcons;
    //If this property is disabled, TShellTree does not show any Icons
    //in front of the entries.
    property FileMask    : String read GetFileMask write SetFileMask;
    //Allows to specify one or more Filemasks.
    //If the Options property contains the TF_NONFOLDERS value
    //and FileMaskUse=True, files(only files, no folder/directories !)
    //only these files will be displayed that fits to the FileMaks(s).
    //Example :
    //Filemask:='*.exe;*.dll;*.com'
    property FileMaskUse : Boolean read FFileMaskUse write SetFileMaskUse;
    //Enables or Disables the use of the FileMask-property
    property FileSortType  : TFileSortType read FFileSortType write SetFileSortType;
    //Let you specify the sorttype of files.
    //
    //TFileSortType   = (so_Name,so_Size,so_Type,so_Date);
    //
    //Note : This property only changes the sort of files, not of
    //directories or systemelements.
    property FileSortOrder : TFileSortOrder read FFileSortOrder write SetFileSortOrder;
    //Let you specify the sorttype of files.
    //
    //TFileSortOrder  = (so_Ascending,so_Descending);
    //
    //Note : This property only changes the sort of files, not of
    //directories or systemelements.
    property OnDirAdd   : TOnDirAddEvent read FOnDirAdd write FOnDirAdd;
    //This event is fired everytime, TShellTree wants to add a file.
    //Use the allow : Boolean parameter to exclude a file from displaying.
    property OnFileAdd   : TOnFileAddEvent read FOnFileAdd write FOnFileAdd;
    //This event is fired everytime, TShellTree wants to add a file.
    //Use the allow : Boolean parameter to exclude a file from displaying.
    property PopupMenuMode : TPopupMenuMode read FPopupMenuMode write FPopupMenuMode;
    //This property lets you switch between the two Popupmenumodes :
    //pmm_System    : displays as context menu a real context menu like the explorer does
    //pmm_Popupmenu : displays the standard popupmenu, specefied by the Popupmenu property.
    //If the Popupmenu is empty, no menu will be displayed.
    property PopupMenu : TPopupMenu read FPopupMenu write SetPopupMenu;
    property ShowMessages : Boolean read FShowMessages write SetShowMessages;
    //When this property is disabled, you will not see any errormessageboxes
    //from TShellTree.
    property OnPopupMenu : TOnPopupMenuEvent read FOnPopupMenu write FOnPopupMenu;
    property OnSystemContextMenu : TSysCMenuPopupEvent read FSysCMenu write FSysCMenu;
{$IFNDEF DFS_COMPILER_4_UP}
    property ToolTips : boolean read FToolTips write SetToolTips;
{$ENDIF}
{$IFDEF Checkboxes}
    property CheckBoxes : boolean read FCheckBoxes write SetCheckBoxes default false;
    property OnCheckBoxChanged : TOnCheckBoxChangedEvent read FOnCheckBoxChanged write FOnCheckBoxChanged;
{$ENDIF}
    property ShowButtons;
    property ShowLines;
    property ShowRoot;
    property StateImages;
    property ReadOnly;
    property HideSelection default false;
    property Align;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property Indent;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnCollapsed;
    property OnCollapsing;
    property OnCompare;
    property OnDblClick;
    property OnDeletion;
    property OnDragDrop;
    property OnDragOver;
    property OnEdited;
    property OnEditing;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnExpanded;
    property OnExpanding;
    property OnGetImageIndex;
    property OnGetSelectedIndex;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property ShowHint;
    property TabOrder;
    property TabStop default True;
    property Visible;
{$IFDEF DFS_COMPILER_2}
    property ParentColor;
{$ELSE}
    property ParentColor default FALSE;
{$ENDIF}
{$IFDEF DFS_COMPILER_3_UP}
    property RightClickSelect default True;
{$ENDIF}
{$IFDEF DFS_COMPILER_4_UP}
    property Anchors;
    property AutoExpand;
    property BiDiMode;
    property BorderWidth;
    property ChangeDelay;
    property Constraints;
    property DragKind;
    property HotTrack;
    property OnCustomDraw;
    property OnCustomDrawItem;
    property OnEndDock;
    property OnStartDock;
    property ParentBiDiMode;
    property RowSelect;
    property ToolTips;
{$ENDIF}
{$IFDEF DFS_COMPILER_5_UP}
    property OnAdvancedCustomDraw;
    property OnAdvancedCustomDrawItem;
    property OnContextPopup;
{$ENDIF}
  end;

procedure Register;

// Sime useful functions when you have to concat a pathes and filename(s)
function AddEndSlashToDir(Dir:string):string;
function RemoveEndSlashFromDir(Dir:string):string;

function  GetPidlLen(Pidl : PItemIDList): Integer;
function  GetPIDLLevel(Path : PItemIDList) : Integer;
function  DeleteLastPidl(Var Pidl : PItemIDList) : Boolean;
function  GetLastPidl(Pidl : PItemIDList) : PItemIDList;
procedure FreePidl(Allocator : IMalloc;Var Pidl : PItemIDList);
function  GetPathFromPidl(IdList : PItemIDList) : String;
function  GetShellItemName(Folder: IShellFolder; ItemIdList: PItemIDList;Flags : DWord): String;

function ShellTree_cmpMultiMask(Path : String;Mask : TStringList) : Boolean;
function ShellTree_cmpmask(Path,Mask : string):boolean;

//Only have to be public for Debugging
{$IFDEF Debug}
function ConvertPidlToBin(FPidl : PItemIDList) : String;
procedure ReleaseCOM(Var ComInterface);
{$ENDIF}

implementation

{$IFDEF OVERSEER}
Uses
  uDbg;
{$ENDIF}

{$IFNDEF DFS_COMPILER_4_UP}
const
  TVS_NOTOOLTIPS = $0080;
{$ENDIF}

//**********************************************************************
//                           TShellTree_Procs
//**********************************************************************

// Added by Stephan Schneider -  sstephan@donau.de
procedure ReleaseCOM(Var ComInterface);
begin
  if Assigned(IUnknown(ComInterface)) then begin
    {$IFDEF DFS_COMPILER_2}
      IUnknown(ComInterface).Release;
    {$ENDIF}
    IUnknown(ComInterface) := nil;
  end;
end;

function GetShellItemName(Folder: IShellFolder; ItemIdList: PItemIDList;Flags : DWord): String;
// Flags :  SHGDN_NORMAL; or SHGDN_INFOLDER; or SHGDN_FORPARSING;
var
  StrResult: TStrRet;
Label Again;
begin
  Result:='';
Again:
  FillChar(StrResult,SizeOf(StrResult),#0);
  Folder.GetDisplayNameOf(ItemIdList,Flags, StrResult);
  case StrResult.uType of
    STRRET_WSTR   : begin
                      IF Assigned(StrResult.pOleStr) Then Result:=WideCharToString(StrResult.pOleStr)
                        else IF Flags=SHGDN_NORMAL Then begin
                          Flags:=SHGDN_FORPARSING;
                          GOTO Again;
                        end;
                    end;
    STRRET_OFFSET : Result:=StrPas(PChar(ItemIdList)+StrResult.uOffset);
    STRRET_CSTR   : Result:=Strpas(StrResult.cStr);
  else
    // If no normal name is available try to get the Parsing-name
    IF Flags=SHGDN_NORMAL Then begin
      Flags:=SHGDN_FORPARSING;
      GOTO Again;
    end;
  end;
end;

function SafeCheckDir(Dir : string):Boolean;
var
  ErrorMode:integer;
  Code: Integer;
begin
  ErrorMode:=SetErrorMode(SEM_FAILCRITICALERRORS);
  try
    Code := GetFileAttributes(PChar(Dir));
    Result := (Code <> -1) and (FILE_ATTRIBUTE_DIRECTORY and Code <> 0);
  finally
    SetErrorMode(ErrorMode);
  end;
end;

function ExtractRootFromDir(Dir:string):string;
const
  cnst1 : string=':';
  cnst2 : string='\';
  cnst3 : string='\\';
var Pos1:integer;
begin
Result:='';
if Length(Dir)=0 then exit;
if copy(Dir,1,2)=cnst3 then
   begin
   Result:=copy(Dir,3,Length(Dir));
   Pos1:=Pos(cnst2,Result);
   if Pos1=0 then
      begin
      Result:=cnst3+Result;
      exit;
      end else
      begin
      Result:=cnst3+copy(Result,1,Pos1);
      end;
   end else
   if copy(Dir,2,1)=cnst1 then Result:=copy(Dir,1,2);
end;

function RemoveEndSlashFromDir(Dir:string):string;
begin
  if Length(Dir)>1 then
    if Dir[Length(Dir)]='\' then Dir:=copy(Dir,1,Length(Dir)-1);
  Result:=Dir;
end;

function AddEndSlashToDir(Dir:string):string;
begin
  if Length(Dir)>1 then
    if Dir[Length(Dir)]<>'\' then Dir:=Dir+'\';
  Result:=Dir;
end;

function GetPidlLen(Pidl : PItemIDList): Integer;
begin
  Result:=0;
  if not Assigned(Pidl) then exit;
  while Pidl^.mkId.cb<>0 do begin
    inc(Result, Pidl^.mkId.cb);
    Pidl:=PItemIDList(@Pidl^.mkId.abID[Pidl^.mkId.cb-2]);
  end;
end;

function GetLastPidl(Pidl : PItemIDList) : PItemIDList;
begin
  Result:=Pidl;
  try
    while (Pidl^.mkId.cb<>0) AND NOT (IsBadReadPtr(@Pidl^.mkId.cb,SizeOf(Pidl^.mkId.cb))) do begin
      Pidl:=PItemIDList(@Pidl^.mkId.abID[Pidl^.mkId.cb-2]);
      IF (Pidl^.mkId.cb<>0) Then Result:=Pidl;
    end;
  finally
  end;
end;

function GetPIDLLevel(Path : PItemIDList) : Integer;
begin
  Result:=0;
  if not Assigned(Path) then exit;
  try
    while (Path^.mkId.cb<>0) AND NOT (IsBadReadPtr(@Path^.mkId.cb,SizeOf(Path^.mkId.cb))) do begin
      inc(Result);
      Path:=PItemIDList(@Path^.mkId.abID[Path^.mkId.cb-2]);
    end;
  finally
  end;
end;

procedure FreePidl(Allocator : IMalloc;Var Pidl : PItemIDList);
begin
  If (Assigned(Pidl)) AND (Allocator.DidAlloc(Pidl)>0) then
    Allocator.Free(Pidl);
  Pidl:=nil;
end;

{$IFDEF Debug}
function ConvertPidlToBin(FPidl : PItemIDList) : String;
Var
  X,Y : Integer;
  Pidl : PItemIDList;
  Level : Integer;
begin
  Pidl:=FPidl;
  Result:='';
  Level:=GetPidlLevel(Pidl);
  FOR Y:=1 TO Level DO begin
    FOR X:=1 TO Pidl^.mkid.cb-2 DO begin
      Result:=Result+InttoHex(Ord(Pidl^.mkid.abid[X-1]),2)+'.'
    end;
    Pidl:=PItemIDList(@Pidl^.mkid.abid[Pidl^.mkid.cb-2]);
    Result[Length(Result)]:='\';
    Result:=Result+#13;
  end;
end;
{$ENDIF}

{$IFDEF Overseer}

procedure LogPidl(Name : String; FPidl : PItemIDList);
Var
  X,Y : Integer;
  Pidl : PItemIDList;
  Level : Integer;
  S : String;
begin
  Pidl:=FPidl;
  Level:=GetPidlLevel(Pidl);
  FOR Y:=1 TO Level DO begin
    S:='';
    FOR X:=1 TO Pidl^.mkid.cb-2 DO begin
      S:=S+InttoHex(Ord(Pidl^.mkid.abid[X-1]),2)+'.'
    end;
    Pidl:=PItemIDList(@Pidl^.mkid.abid[Pidl^.mkid.cb-2]);
  end;
end;
{$ENDIF}

function  HasAttribute(Attributes,Test : UInt) : Boolean;
begin
  Result:=((Attributes or Test) = Attributes)
end;

function  DeleteLastPidl(Var Pidl : PItemIDList) : Boolean;
Var
  Level : Integer;
  I : Integer;
  TmpPidl : PItemIDList;
begin
  Result:=False;
  Level:=GetPidlLevel(Pidl);
  IF Level<=1 Then exit;
  TmpPidl:=Pidl;
  FOR I:=1 TO Level-1 DO
    TmpPidl:=PItemIDList(@TmpPidl^.mkId.abID[TmpPidl^.mkId.cb-2]);
  TmpPidl.mkid.cb:=0;
  Result:=True;
end;

procedure CopyPidl(Allocator : IMalloc;Var Source,Dest : PItemIDList);
Var
  Len : Integer;
begin
  IF NOT Assigned(Source) Then exit;
  Len:=GetPidlLen(Source);
  INC(len,2);
  Dest:=Allocator.Alloc(Len);
  if not Assigned(Dest) then begin
    OutOfMemoryError;
    Dest:=nil;
  end else begin
    system.move(Source^,Dest^,Len);
  end;
  IF Len=2 Then begin
    Dest^.mkid.cb:=0;
  end;
end;

function OptionsToFlags(Options : TVisOptions) : DWord;
begin
  Result:=0;
  IF (TF_Folders    in Options) Then Result:=Result + SHCONTF_FOLDERS;
  IF (TF_NonFolders in Options) Then Result:=Result + SHCONTF_NONFOLDERS;
  IF (TF_Hidden     in Options) Then Result:=Result + SHCONTF_INCLUDEHIDDEN;
end;

function ComparePidl(Level : Integer;LongPidl,Pidl : PItemIDList) : Boolean;
Var
  X : Integer;
begin
  Result:=False;
  IF Level>GetPidlLevel(LongPidl) Then exit;
  FOR X:=1 TO Level DO LongPidl:=PItemIDList(@LongPidl^.mkId.abID[LongPidl^.mkId.cb-2]);
  IF LongPidl^.mkid.cb=Pidl^.mkid.cb Then begin
    Result:=True;
    FOR X:=0 TO (LongPidl^.mkid.cb-3) DO begin
      IF LongPidl^.mkid.abid[X]<>Pidl^.mkid.abid[X] Then Result:=False;
    end;
  end;
end;

function  FindNodePidl(ParentNode : TShellNode;Pidl : PItemIDList) : TShellNode;
Var
  SN : TShellNode;
begin
  Result:=nil;
  SN:=TShellNode(ParentNode.GetFirstChild);
  While SN<>nil DO begin
    IF NOT SN.NodeInfo.Updated Then begin
      IF ComparePidl(0,Pidl,SN.NodeInfo.RelativeIdl) Then begin
        SN.NodeInfo.Updated:=True;
        Result:=SN;
        SN:=nil;
      end;
    end;
    SN:=TShellNode(ParentNode.GetNextChild(SN));
  end;
end;

function TestDirectory(Dir : String) : String;
var
  TmpDir : string;
begin
  Result:=Dir;
  TmpDir:=AddEndSlashToDir(ExtractRootFromDir(Dir));
  if (Result='')or(TmpDir='') then begin
    Result:='';
    exit;
    end;
  if not SafeCheckDir(Result) then begin
    while Assigned(StrRScan(PChar(Result),'\')) do begin
      if Result[Length(Result)]='\' then Result:=copy(Result,1,Length(Result)-1);
      Result:=copy(Result,1,StrRScan(PChar(Result),'\')-PChar(Result)+1);
      if (Length(Result)<=Length(TmpDir)) then break;
      if SafeCheckDir(Result) then break;
    end;
  end;
  if not SafeCheckDir(Result) then Result:='' else RemoveEndSlashFromDir(Result);
  if (Length(Result)>2)and(copy(Result,2,1)=':')and(copy(Result,3,1)<>'\') then begin
    delete(Result,1,2);
    insert(GetCurrentDir+'\',Result,1);
  end;
end;

function ConvertToCSIDL(P : TPidls) : Integer;
begin
  Case P OF
    PIDL_BITBUCKET                : Result := CSIDL_BITBUCKET;
    PIDL_CONTROLS                 : Result := CSIDL_CONTROLS;
    PIDL_DESKTOP                  : Result := CSIDL_DESKTOP;
    PIDL_DESKTOPDIRECTORY         : Result := CSIDL_DESKTOPDIRECTORY;
    PIDL_DRIVES                   : Result := CSIDL_DRIVES;
    PIDL_FONTS                    : Result := CSIDL_FONTS;
    PIDL_NETHOOD                  : Result := CSIDL_NETHOOD;
    PIDL_NETWORK                  : Result := CSIDL_NETWORK;
    PIDL_PERSONAL                 : Result := CSIDL_PERSONAL;
    PIDL_PRINTERS                 : Result := CSIDL_PRINTERS;
    PIDL_PROGRAMS                 : Result := CSIDL_PROGRAMS;
    PIDL_RECENT                   : Result := CSIDL_RECENT;
    PIDL_SENDTO                   : Result := CSIDL_SENDTO;
    PIDL_STARTMENU                : Result := CSIDL_STARTMENU;
    PIDL_STARTUP                  : Result := CSIDL_STARTUP;
    PIDL_TEMPLATES                : Result := CSIDL_TEMPLATES;
    PIDL_FAVORITES                : Result := CSIDL_FAVORITES;
    PIDL_USERDEFINED              : Result := CSIDL_USERDEFINED;
  else
    Result:=CSIDL_DESKTOP;
  end;
end;

function  ShellTreeSortProc(Node1, Node2: TTreeNode; Data: integer): Integer; stdcall;
Var
  ISF : IShellFolder;
  N1,N2 : TShellNode;
  Res : Integer;
  SortData : TSortData;
  SortType : Byte;
  SortOrder : Boolean;
begin
  SortData:=TSortdata(Pointer(Data)^);
  N1:=TShellNode(Node1);
  N2:=TShellNode(Node2);
  IF N1.IsFile and N2.IsFile Then begin
    Sorttype:=Byte(SortData.FileSortType);
    SortOrder:=Boolean(Byte(Sortdata.FileSortOrder));
  end else begin
    Sorttype:=0;
    SortOrder:=True;
  end;
  ISF:=TShellNode(Node1).ParentShellFolder;
  Res:=ISF.CompareIDs(SortType,N1.RelativeIDL,N2.RelativeIDL);
  IF (Res<>0) AND (Res<>1) Then Res:=-1;
  Result:=Res;
  IF NOT SortOrder then Result:=Result*(-1);
end;

function GetPathFromPidl(IdList : PItemIDList) : String;
Var
  S : String;
  I : Integer;
begin
  SetString(S,PChar(nil),MAX_PATH+1);
  SHGetPathFromIDList(IdList,Pointer(S));
  I:=Pos(#0,S);
  IF I>0 Then Result:=Copy(S,1,I-1)
    else Result:=S;
end;

function MakeAbsoluteIDL(Allocator : IMalloc;ParentNode: TShellNode; Child: PItemIDList): PItemIDList;
var
  PathLen,ChildLen: integer;
  ParentFolder: IShellFolder;
  ParentPath: PItemIDList;
  Pidl : PItemIDList;
begin
  ParentPath:=nil;
  ParentFolder:=nil;
  if Assigned(ParentNode) then begin
    ParentFolder:=ParentNode.ShellFolder;
    ParentPath  :=ParentNode.NodeInfo.AbsoluteIDL;
  end;
  ChildLen:=Child^.mkId.cb;
  IF ChildLen<=0 Then begin
    CopyPidl(Allocator,ParentPath,Pidl);
    Result:=Pidl;
    exit;
  end;
  if Assigned(ParentFolder) then PathLen:=GetPidlLen(ParentPath) else PathLen:=0;
  Result:=Allocator.Alloc(PathLen+ChildLen+2);
  IF Allocator.GetSize(Result)<(PathLen+ChildLen+2) then OutOfMemoryError;
  if PathLen<>0 then begin
    system.move(ParentPath^, Result^, PathLen);
    system.move(Child^, Result^.mkId.abID[PathLen-2], ChildLen);
  end else
    system.move(Child^, Result^, ChildLen);
  Result^.mkId.abID[PathLen+ChildLen-2]:=0;
  Result^.mkId.abID[PathLen+ChildLen-2+1]:=0;
end;

function ShellTree_cmpmask(Path,Mask : string):boolean;
// tests whether the string 'a' fits to the search mask in 'b'
var sr             : string;
    ps1,ps2,ps3    : integer;
    dontcare       : boolean;
    onechar        : char;
    tmp_list       : tstrings;
begin
  Path:=Ansilowercase(Path);
  Mask:=AnsiLowerCase(Mask);
     result := true;
     if Mask = '*' then exit; // fits always
     if Mask = '*.*' then if pos('.',Path) > 0 then exit; // fits, too
     if (pos('*',Mask) = 0) and (pos('?',Mask)=0) then
        if Path=Mask then exit;
         // searched file was found (searchstring IS text)

     result   := false;
     if Mask = '' then exit;
     tmp_list := tstringlist.create;
     try
        // divide partial strings ('?','*' or text) to tmp_list
        repeat
              onechar := Mask[1];
              if (onechar='*') or (onechar='?') then begin
                 tmp_list.add(onechar);
                 delete(Mask,1,1);
              end else begin
                  ps1 := pos('?',Mask);
                  if ps1 = 0 then ps1 := maxint;
                  ps2 := pos('*',Mask);
                  if ps2 = 0 then ps2 := maxint;
                  if ps2 > ps1 then ps2 := ps1;
                  tmp_list.add(copy(Mask,1,ps2-1));
                  Mask:=copy(Mask,ps2,maxint);
              end;
        until Mask = '';
        // now compare the string with the partial search masks
        dontcare := false;
        ps2      := 1;
        if tmp_list.count > 0 then for ps1 := 0 to pred(tmp_list.count) do begin
           sr := tmp_list[ps1];
           if sr = '?' then begin
              inc(ps2,1);
              if ps2 > length(Path) then exit;
           end else
           if sr = '*' then
              dontcare := true
           else begin
                if not dontcare then begin
                   if copy(Path,ps2,length(sr)) <> sr then exit;
                   dontcare := false;
                   ps2 := ps2+length(sr);
                end else begin
                   ps3:= pos(sr,copy(Path,ps2,maxint));
                   if ps3 = 0 then exit;
                   ps2 := ps3+length(sr);
                   dontcare := false;
                end;
           end;
        end;
        if not dontcare then if ps2 <> length(Path)+1 then exit;
        result := true;
     finally
            tmp_list.free;
     end;
end;

function ShellTree_cmpMultiMask(Path : String;Mask : TStringList) : Boolean;
Var
  X : Integer;
begin
  Result:=True;
  FOR X:=1 TO Mask.Count DO begin
    Result:=ShellTree_cmpMask(Path,Mask.Strings[X-1]);
    IF Result then exit;
  end;
end;

//**********************************************************************
//                           TShellNode
//**********************************************************************

constructor TShellNode.Create(AOwner: TTreeNodes);
begin
  inherited Create(AOwner);
  FHasMonitorThread:=False;
  FillChar(NodeInfo, sizeof(Nodeinfo), 0);
  FreeShellfolder;
end;

procedure TShellNode.CreateMonitorThread;
begin
  IF FHasMonitorThread Then exit;
  IF Length(Trim(Path))>0 Then begin
    FMonitorThread:=TDiscMonitorThread.Create(self);
    FMonitorThread.Resume;
  end;
  FHasMonitorThread:=True;
end;

procedure TShellNode.DestroySubMonitorThreads(Firstlevel : Boolean);
Var
  N : TTreeNode;
begin
  IF NOT FirstLevel then DestroyMonitorThread;
  IF NOT Expanded Then exit;
  N:=GetFirstChild;
  While Assigned(N) do begin
    TShellNode(N).DestroySubMonitorThreads(false);
    N:=GetNextChild(N);
  end;
end;

procedure TShellNode.DestroyMonitorThread;
begin
  IF NOT FHasMonitorThread Then exit;
  IF Assigned(FMonitorThread) then begin
    FHasMonitorThread:=False;
    FMonitorThread.Free;
    FMonitorThread:=nil;
  end;  
end;


procedure TShellNode.FolderChanged;
Var
  ST : TShellTree;
  MsgHandle : HWnd;
begin
  ST:=TShellTree(TreeView);
  IF NOT Assigned(ST) Then exit;
  IF {Expanded AND} ST.UpdAutoThread Then begin
    MsgHandle:=TShellTree(TreeView).FMessageHandle;
    try
      TShellTree(TreeView).FMessageHandle:=0;
      ST.UpdateSubNodes(self,false);
    finally
      TShellTree(TreeView).FMessageHandle:=MsgHandle;
    end;
  end;
end;

function  TShellNode.SetDisplayName(NewName : String) : Boolean;
Var
  SPNode : TTreeNode;
  Displayname : ARRAY[0..MAX_PATH] OF WideChar;
  NewPidl : PItemIDList;
  Pidl : PItemIDList;
  Flags : DWord;
  ST : TShellTree;
begin
  Result:=False;
  IF NewName<>Text Then begin
    ST:=TShellTree(TreeView);
    StringToWideChar(NewName,Displayname,MAX_PATH);
    Pidl:=RelativeIDL;
    NewPidl:=nil;
    SPNode:=Parent;
    Flags:=OptionsToFlags(ST.Options);
    IF (ParentShellFolder.SetNameOf(TShellTree(TreeView).MessageHandle,Pidl,Displayname,Flags,NewPidl)=NOERROR) Then begin
      UpdateNodeInfo(ParentShellFolder,NewPidl,True);
      IF Assigned(SPNode) Then ST.SortNode(SPNode,False);
      ST.Change(self);
      Result:=True;
    end else
       Result:=False;
  end;
end;

procedure TShellNode.UpdateNodeText;
begin
  Text:=GetNodeText;
end;

function TShellNode.GetNodeText : String;
begin
{$IFNDEF HideRootNode}
  IF NOT Assigned(Parent) AND (GetPidlLevel(RelativeIDL)>0) then begin
    Result:=GetShellItemName(ParentShellFolder,GetLastPidl(RelativeIDL),SHGDN_NORMAL);
    IF (Length(Result)=0) Then
      Result:=GetShellItemName(ParentShellFolder,RelativeIDL,SHGDN_NORMAL);
  end else
{$ENDIF}
    Result:=GetShellItemName(ParentShellFolder,RelativeIDL,SHGDN_NORMAL);
end;


procedure TShellNode.UpdateIcon;
Var
  FileInfo: TSHFileInfo;
begin
  SHGetFileInfo(Pointer(AbsoluteIDL),SFGAO_SHARE, FileInfo, sizeof(FileInfo),
  SHGFI_PIDL or SHGFI_SYSICONINDEX );
  ImageIndex:=FileInfo.iIcon;
  IF (HasAttribute(SFGAO_FOLDER)) Then
    SHGetFileInfo(Pointer(AbsoluteIDL), 0, FileInfo, sizeof(FileInfo),SHGFI_PIDL or SHGFI_SYSICONINDEX or SHGFI_OPENICON);
  SelectedIndex:=FileInfo.iIcon;
  IF TShellTree(TreeView).ShowSharedOverlay AND HasAttribute(SFGAO_SHARE) then begin
    OverlayIndex := 0;
  end else IF TShellTree(TreeView).FShowLinkOverlay AND HasAttribute(SFGAO_LINK) then begin
    OverlayIndex := 1;
  end else
    OverlayIndex:=-1;
end;

procedure TShellNode.UpdateNodeInfo(ParentISF : IShellFolder;Pidl : PItemIDList;Recursive : Boolean);
Var
  Node : TTreeNode;
  Attr : Cardinal;
begin
  Attr:=All_Attributes;
  IF Succeeded(ParentISF.GetAttributesOf(1,Pidl,Attr)) Then NodeInfo.Attributes:=Attr;
  Nodeinfo.ParentShellFolder:=ParentISF;
  IF Assigned(Pidl) Then begin
    FreeShellFolder;
    IF (Pidl<>NodeInfo.RelativeIDL) Then begin
      FreePidl(TShellTree(TreeView).Allocator,NodeInfo.RelativeIDL);
      FreePidl(TShellTree(TreeView).Allocator,Nodeinfo.AbsoluteIDL);
      NodeInfo.RelativeIDL:=Pidl;
    end;
  end;
  IF Assigned(Parent) AND Assigned(TreeView) Then
    Nodeinfo.AbsoluteIDL:=MakeAbsoluteIDL(TShellTree(TreeView).Allocator,TShellNode(Parent), Nodeinfo.RelativeIDL)
  else begin
{$IFNDEF HideRootNode}
    CopyPidl(TShellTree(TreeView).Allocator,Nodeinfo.RelativeIDL,Nodeinfo.AbsoluteIDL);
{$ELSE}
    IF Assigned(TShellTree(TreeView).RootNode) Then begin
      Nodeinfo.AbsoluteIDL:=MakeAbsoluteIDL(TShellTree(Treeview).Allocator,
        TShellTree(Treeview).RootNode, RelativeIDL);
//      CopyPidl(Nodeinfo.RelativeIDL,Nodeinfo.AbsoluteIDL);
    end else begin
      CopyPidl(TShellTree(Treeview).Allocator, Nodeinfo.RelativeIDL,Nodeinfo.AbsoluteIDL);
    end;
{$ENDIF}
  end;
  IF Recursive Then begin
    Node:=GetFirstChild;
    While Assigned(Node) DO begin
      TShellNode(Node).UpdateNodeInfo(ShellFolder,TShellNode(Node).RelativeIDL,True);
      Node:=GetNextChild(Node);
    end;
  end;
end;

procedure TShellNode.FreeShellfolder;
begin
  FBinded:=False;
  NodeInfo.ShellFolder:=nil;
end;

// Creates once an IShellFolder interface of a node when needed
procedure TShellNode.CreateShellFolder;
begin
  FBinded:=True;
  IF GetPidlLevel(RelativeIDL)>0 Then begin
    (ParentShellFolder.BindToObject(RelativeIdl, nil, IID_ISHELLFOLDER,pointer(NodeInfo.ShellFolder)))
  end else
    Nodeinfo.ShellFolder:=ParentShellFolder;
end;

// Returns filextension
Function TShellNode.GetExtension : String;
Begin
  Result:='';
  IF IsFile Then Result:=ExtractFileExt(Path);
End;

Function TShellNode.GetFileName : String;
Begin
  Result:='';
  IF IsFile Then Result:=ExtractFileName(Path);
End;

function TShellNode.GetNodePath : String;
begin
  Result:=GetPathFromPidl(AbsoluteIDL);
end;

function  TShellNode.GetSubPath(ItemIdList : PItemIDList) : String;
begin
  Result:=GetShellItemname(NodeInfo.ShellFolder,ItemIdList,SHGDN_FORPARSING)
end;

function TShellNode.GetNodePidlLevel : Integer;
begin
  Result:=GetPidlLevel(NodeInfo.AbsoluteIDL)
end;

//How many levels above the Desktop ?
function TShellNode.GetShellFolder : IShellFolder;
begin
  IF FBinded AND NOT Assigned(NodeInfo.ShellFolder) Then FBinded := False;
  IF NOT FBinded Then CreateShellFolder;
  Result:=NodeInfo.ShellFolder;
end;

procedure TShellNode.PerformDefaultAction;
Var
  Attr : ULONG;
  Pidl : PItemIDList;
begin
  Attr:=0;
  Pidl:=RelativeIDL;
{$IFDEF DFS_COMPILER_4_UP}
  itemprop.PerformDefaultAction(ParentShellFolder,Pidl,Attr,Handle,GetPidllevel(Pidl));
{$ELSE}
  itemprop.PerformDefaultActionPidl(ParentShellFolder,Pidl,Attr,Handle,GetPidllevel(Pidl));
{$ENDIF}
end;

procedure TShellNode.PerformVerb(const verb : string);
Var
  Attr : ULONG;
  Pidl : PItemIDList;
begin
  Attr:=0;
  Pidl:=RelativeIDL;
{$IFDEF DFS_COMPILER_4_UP}
  itemprop.PerformVerb(verb,ParentShellFolder,Pidl,Attr,Handle,GetPidllevel(Pidl));
{$ELSE}
  itemprop.PerformVerbPidl(verb,ParentShellFolder,Pidl,Attr,Handle,GetPidllevel(Pidl));
{$ENDIF}
end;

function TShellNode.GetIsFile : Boolean;
begin
  Result:=(HasAttribute(SFGAO_FILESYSTEM)) AND NOT HasAttribute(SFGAO_FOLDER);
end;

function TShellNode.GetIsFolder : Boolean;
begin
  Result:=(HasAttribute(SFGAO_FILESYSTEM)) AND HasAttribute(SFGAO_FOLDER);
end;

function TShellNode.GetIsSysObject : Boolean;
begin
  Result:=NOT (HasAttribute(SFGAO_FILESYSTEM){ OR HasAttribute(SFGAO_FILESYSANCESTOR)});
end;

function TShellNode.HasAttribute(Attrib : UInt) : Boolean;
begin
  Result:=((NodeInfo.Attributes or Attrib) = NodeInfo.Attributes)
end;

function TShellNode.GetAttributes;
begin
  Result:=[];
  With Nodeinfo DO begin
    IF (Attributes or SFGAO_CANCOPY)         = Attributes Then Result:=Result + [Attr_CANCOPY];
    IF (Attributes or SFGAO_CANMOVE)         = Attributes Then Result:=Result + [Attr_CANMOVE];
    IF (Attributes or SFGAO_CANLINK)         = Attributes Then Result:=Result + [Attr_CANLINK];
    IF (Attributes or SFGAO_CANRENAME)       = Attributes Then Result:=Result + [Attr_CANRENAME];
    IF (Attributes or SFGAO_CANDELETE)       = Attributes Then Result:=Result + [Attr_CANDELETE];
    IF (Attributes or SFGAO_HASPROPSHEET)    = Attributes Then Result:=Result + [Attr_HASPROPSHEET];
    IF (Attributes or SFGAO_DROPTARGET)      = Attributes Then Result:=Result + [Attr_DROPTARGET];
    IF (Attributes or SFGAO_CAPABILITYMASK)  = Attributes Then Result:=Result + [Attr_CAPABILITYMASK];
    IF (Attributes or SFGAO_LINK)            = Attributes Then Result:=Result + [Attr_LINK];
    IF (Attributes or SFGAO_SHARE)           = Attributes Then Result:=Result + [Attr_SHARE];
    IF (Attributes or SFGAO_READONLY)        = Attributes Then Result:=Result + [Attr_READONLY];
    IF (Attributes or SFGAO_GHOSTED)         = Attributes Then Result:=Result + [Attr_GHOSTED];
    IF (Attributes or SFGAO_DISPLAYATTRMASK) = Attributes Then Result:=Result + [Attr_DISPLAYATTRMASK];
    IF (Attributes or SFGAO_FILESYSANCESTOR) = Attributes Then Result:=Result + [Attr_FILESYSANCESTOR];
    IF (Attributes or SFGAO_FOLDER)          = Attributes Then Result:=Result + [Attr_FOLDER];
    IF (Attributes or SFGAO_FILESYSTEM)      = Attributes Then Result:=Result + [Attr_FILESYSTEM];
    IF (Attributes or SFGAO_HASSUBFOLDER)    = Attributes Then Result:=Result + [Attr_HASSUBFOLDER];
    IF (Attributes or SFGAO_CONTENTSMASK)    = Attributes Then Result:=Result + [Attr_CONTENTSMASK];
    IF (Attributes or SFGAO_VALIDATE)        = Attributes Then Result:=Result + [Attr_VALIDATE];
    IF (Attributes or SFGAO_REMOVABLE)       = Attributes Then Result:=Result + [Attr_REMOVABLE];
    IF (Attributes or SFGAO_COMPRESSED)      = Attributes Then Result:=Result + [Attr_COMPRESSED];
  end;
end;

destructor TShellNode.Destroy;
begin
  IF FHasMonitorThread AND Assigned(FMonitorThread) Then FMonitorThread.Free;
  FreePidl(TShellTree(TreeView).Allocator,NodeInfo.RelativeIDL);
  FreePidl(TShellTree(TreeView).Allocator,NodeInfo.AbsoluteIDL);
  FreeShellFolder;
  ReleaseCom(NodeInfo.ShellFolder);
  inherited Destroy;
end;

procedure TShellNode.FalseUpdated;
Var
  SN : TShellNode;
begin
  SN:=TShellNode(GetFirstChild);
  While(SN<>nil) DO begin
    SN.NodeInfo.Updated:=False;
    SN:=TShellNode(GetNextChild(SN));
  end;
end;

procedure TShellNode.DeleteFalseUpdated;
Var
  SN,Del : TTreeNode;
begin
  SN:=GetLastChild;
  While(SN<>nil) DO begin
    IF NOT TShellNode(SN).NodeInfo.Updated Then begin
      Del:=SN;
      SN:=GetPrevChild(SN);
      Del.Delete;
    end else
      SN:=GetPrevChild(SN);
  end;
end;

procedure TShellNode.ChangeDiscMonitorFilter(NewFilters : Integer);
Var
  SN : TShellNode;
begin
  IF FHasMonitorThread AND Assigned(FMonitorThread) Then begin
    IF NOT FMonitorThread.FInvalid Then begin
      FMonitorThread.Filters:=NewFilters;
    end;
  end;
  SN:=TShellNode(GetFirstChild);
  While Assigned(SN) do begin
    SN.ChangeDiscMonitorFilter(Newfilters);
    SN:=TShellNode(GetNextChild(SN));
  end;
end;

{$IFDEF Checkboxes}
function TShellNode.GetChecked :Boolean;
var
  Item: TTVItem;
begin
  Result := False;
  with Item do
  begin
    mask := TVIF_STATE;
    hItem := ItemId;
    if TreeView_GetItem(Handle, Item) then
      result:=((Item.State and TVIS_CHECKED) = TVIS_CHECKED);
  end;
end;

procedure TShellNode.SetChecked(value:Boolean);
var
  Item: TTVItem;
begin
  if FChecked<>Value then begin
    FChecked:=Value;
    with Item do begin
      mask := TVIF_STATE;
      hItem := ItemId;
      if TreeView_GetItem(Handle, Item) then begin
        if FChecked then
          Item.State:=(Item.State or TVIS_CHECKED)
        else
          Item.State:=(Item.State or TVIS_UNCHECKED);
        TreeView_SetItem(Handle, Item);
        TreeView.Selected:=self;
      end;
    end;
    TShellTree(Owner).CheckBoxChanged(self,Value);
  end;
end;
{$ENDIF}

//**********************************************************************
//                           TShellTree
//**********************************************************************

constructor TShellTree.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
{$IFNDEF DFS_COMPILER_4_UP}
   FToolTips:=True;
{$ENDIF}
  {$IFDEF DoRepaintProperty}
    FDoRepaint:=True;
  {$ENDIF}
  DiscMonitorFilters:=FILE_NOTIFY_CHANGE_ATTRIBUTES or FILE_NOTIFY_CHANGE_DIR_NAME;
  FDeviceChangeHandle:=AllocateHWnd(DeviceChangeWndProc);
  DefaultSortProc:=@ShellTreeSortProc;
  Height:=270;
  Width:=170;
  FTmpStartpath:='';
  FRootDir:=STD_DIR;
  FRootPidl:=PIDL_DESKTOP;
  FRootUsePidl:=True;
  FOptions:=[TF_FOLDERS];
  HideSelection:=False;
  FDesignStart:=False;
  FInitialized:=False;
  FUpdAutoThread:=True;
  FUpdAuto:=True;
  FShowIcons:=True;
  FShowVirtualFolders:=True;
  FShowMessages:=True;
  FOnFileAdd:=nil;
  FRootNode:=nil;
  FShowLinkOverlay:=True;
  FShowSharedOverlay:=True;
  FFileMaskList:=TStringList.Create;
  FFilemaskList.Add('*.*');
  FFileMaskUse:=False;
  FRootUserPidl:=nil;
  IF not((SHGetMalloc(Allocator)=NOERROR)and(SHGetDesktopFolder(FDesktop)=NOERROR)) then
    OutOfMemoryError;
end;

procedure TShellTree.CreateParams(var Params: TCreateParams);
begin
  inherited;
{$IFNDEF DFS_COMPILER_4_UP}
  IF NOT FToolTips then Params.Style:=Params.Style or TVS_NOTOOLTIPS;
{$ENDIF}
{$IFDEF CheckBoxes}
  if FCheckBoxes then Params.Style := Params.Style or TVS_CHECKBOXES;
{$ENDIF}
end;

{$IFDEF CheckBoxes}

function TShellTree.GetNodeFromItem(const Item: TTVItem): TTreeNode;
begin
  Result := nil;
  if Items <> nil then
    with Item do
      if (state and TVIF_PARAM) <> 0 then Result := Pointer(lParam)
      else Result := Items.GetNode(hItem);
end;

procedure TShellTree.CNNotify(var Message : TWMNotify);
var
  Node : TShellNode;
  Checked : Boolean;
begin
  if FCheckBoxes then begin
    with Message, PTVDispInfo(NMHdr)^ do begin
      IF (NMHdr^.code = TVN_GETDISPINFO) {AND (item.mask and $3 <> 0 )} then begin
//        debugger.LogFmtMsg('item %x mask %x state %x',[Integer(item.hitem),item.mask,item.state]);
        Node := TShellNode(GetNodeFromItem(item));
        Checked:=((item.state and  TVIS_Checked)<> 0);
        IF (Checked<>Node.FChecked) Then begin
          Node.FChecked:=Checked;
          CheckBoxChanged(Node,Checked);
        end;
      end;
    end;
  end;
  inherited;
end;

procedure TShellTree.CheckBoxChanged(Node : TShellNode; NewState : Boolean);
begin
  IF Assigned(FOnCheckBoxChanged) Then
    FOnCheckBoxChanged(Node,NewState);
end;

{$ENDIF}

procedure TShellTree.DeviceChangeWndProc(Var Message : TMessage);
begin
  With Message DO begin
    Result:=-1;
    IF (Msg = WM_DEVICEChange) Then begin
      IF (wParam = ShellTree_DBT_DEVICEARRIVAL) OR
         (wParam = ShellTree_DBT_DEVICEREMOVECOMPLETE) Then begin
//        Debugger.LogFmtMsg('DeviceChangeMessage %x',[wParam]);
        MediaChangeDetected(wParam,PShellTree_Dev_Broadcast_Hdr(lParam));
        Result:=1;
      end;
    end;
  end;
end;

procedure TShellTree.Loaded;
begin
  inherited;
  UpdateTreeView(la_None);
end;

procedure TShellTree.CreateWnd;
begin
  inherited CreateWnd;
  IF (FTmpStartpath<>'') Then begin
    Path:=FTmpStartPath;
    FTmpStartpath:='';
  end;
  RetrieveDefaultPidls;
  RetrieveMessageHandle;
  If NOT (csLoading in ComponentState) then UpdateTreeView(la_None);
end;

procedure TShellTree.WndProc(Var Message : TMessage);
begin
  inherited;
  IF (Message.Msg=CM_SHOWINGCHANGED) Then begin
   RetrieveMessageHandle;
    IF ShowIcons Then begin
      ShowIcons:=false;
      ShowIcons:=True;
    end;
    UpdateTree;
  end;
end;


procedure TShellTree.WMMouseWheel(Var Message: TMessage);
Var
  ScrollMsg: TMessage;
begin
  IF (csDesigning in ComponentState) Then    Exit;
  IF ShortInt(Message.WParamHi) = -120 Then ScrollMsg.WParamLo := SB_PAGEDOWN
    else if ShortInt(Message.WParamHi) = 120 Then  ScrollMsg.WParamLo := SB_PAGEUP;
  ScrollMsg.Msg := WM_VSCROLL;
  Dispatch(ScrollMsg);
end;

procedure TShellTree.WMContextMenu(Var Message : TMessage);
Var
  P : TPoint;
  R : TRect;
  NS,Node : TShellNode;
begin
  NS:=nil;
  IF Message.lparam=-1 Then begin
    Node:=TShellNode(Selected);
    IF Not Assigned(Node) Then exit;
    R:=Node.DisplayRect(True);
    P.X:=R.Left+((R.Right-R.Left  ) shr 2);
    P.Y:=R.Top- ((R.Top  -R.Bottom) shr 2);
  end else begin
    P.X:=LOWORD(Message.lParam);
    P.Y:=HIWORD(Message.lParam);
    P:=ScreenToClient(P);
    Node:=TShellNode(GetNodeAt(P.X,P.Y));
  end;
  IF NOT AllowPopupMenu(Node) Then exit;
  P:=ClientToScreen(P);
//  IF (Selected<>Node) Then NS:=TShellNode(Selected);
//  Selected:=Node;
  IF PopupMenuMode=pmm_PopupMenu Then begin
    IF Assigned(Popupmenu) Then begin
      PopupMenu.PopupComponent := self;
      PopupMenu.Popup(P.X,P.Y)
    end;
  end else begin
    PopupSystemContextmenu(Node,P);
  end;
  IF Assigned(NS) Then Selected:=NS;
end;

procedure TShellTree.PopupSystemContextMenu(Node : TShellNode; Point : TPoint);
Var
  PISF : IShellFolder;
  Pidl : PItemIDList;
  CanEdit,RenameSel : Boolean;
begin
  IF NOT Assigned(Node) then exit;
  PISF:=Node.ParentShellFolder;
  IF NOT Assigned(PISF) Then exit;
  Pidl:=Node.RelativeIDL;
  Canedit:=false;
{$IFDEF DFS_COMPILER_4_UP}
  ItemProp.DisplayContextMenu(PISF,Pidl,0,FMessageHandle,Point,1,CanEdit,RenameSel);
{$ELSE}
  {$IFDEF DFS_CPPB}
    ItemProp.DisplayContextMenuPIDL(PISF,Pidl,0,Pointer(FMessageHandle), Point, 1,CanEdit,RenameSel);
  {$ELSE}
    ItemProp.DisplayContextMenuPIDL(PISF,Pidl,0,FMessageHandle, Point, 1,CanEdit,RenameSel);
  {$ENDIF}  
{$ENDIF}
end;

function  TShellTree.AllowPopupMenu(Node : TShellNode) : Boolean;
begin
  IF NOT Assigned(Node) Then exit;
  Result:=True;
  IF Assigned(FOnPopupMenu) Then FOnPopupMenu(Node,Result);
end;

procedure TShellTree.UpdateTree;
begin
  {$IFDEF DoRepaintProperty}
    IF DoRepaint Then
  {$ENDIF}
    Items.BeginUpdate;
  try
    UpdateSubNodes(FRootNode,True);
  finally
  {$IFDEF DoRepaintProperty}
    IF DoRepaint Then
  {$ENDIF}
    Items.EndUpdate;
  end;
end;

procedure TShellTree.MediaChangeDetected(Msg : Integer;HDR : PShellTree_Dev_Broadcast_Hdr);
Var
  Attributes : DWord;
  Pidl : PItemIDList;
  HDRIsVolume : Boolean;
  Node : TShellNode;
  RootLevel : Integer;
  MyComputerNode : TShellNode;
begin
//  Debugger.EnterProc('MediaChageDetected');
  try
  IF FUseShellCache Then Attributes:=0
    else Attributes:=SFGAO_VALIDATE;
  Pidl:=nil;
  MyComputer.GetAttributesOf(0,Pidl,Attributes);
//Debugger.LogMsg('GetAttributes passed');
  HDRIsVolume:=(HDR^.dbch_devicetype = ShellTree_DBT_DEVTYP_VOLUME);
//Debugger.LogFmtMsg('HDRIsVolume          %d',[byte(HDRIsVolume)]);
//Debugger.LogFmtMsg('HDR^.dbch_size       %d',[HDR^.dbch_size]);
//Debugger.LogFmtMsg('HDR^.dbch_devicetype %x',[HDR^.dbch_devicetype]);
//Debugger.LogFmtMsg('HDR^.dbch_reserved   %x',[HDR^.dbch_reserved]);
  Node:=TShellNode(FRootNode);
  IF NOT Assigned(Node) Then exit;
  RootLevel:=GetPidllevel(Node.AbsoluteIDL);
  IF NOT Assigned(FMyComputerPidl) then exit;
//Debugger.LogFmtMsg('RootLevel is: %d',[RootLevel]);
  IF (RootLevel>1) Then begin
    IF NOT HDRIsVolume OR (RootLevel>2) Then exit;
    CopyPidl(Allocator,Node.NodeInfo.AbsoluteIDL,Pidl);
    try
      DeleteLastPidl(Pidl);
      IF NOT (Desktop.CompareIDs(0,FMyComputerPidl,Pidl)=0) Then exit;
    finally
      FreePidl(Allocator,Pidl);
    end;
//Debugger.LogMsg('UpdateSpecialDrives without MyComputer');
    UpdateSpecialDrives(Msg, PShellTree_Dev_Broadcast_Volume(HDR),nil);
    exit;
  end;
//Debugger.LogMsg('Second way');
  MyComputerNode:=FindPidl(FMyComputerPidl,false,True,True);
  IF HDRisVolume Then begin
//    Debugger.LogMsg('UpdateSpecialDrives with MyComputer');
    UpdateSpecialDrives(Msg, PShellTree_Dev_Broadcast_Volume(HDR),MyComputerNode)
  end else
    UpdateMyComputer(MyComputerNode);
  finally
//    Debugger.LeaveProc('MediaChageDetected');
  end;
end;

procedure TShellTree.UpdateMyComputer(MyComputerNode : TShellNode);
begin
  IF NOT Assigned(MyComputerNode) Then exit;
  IF (MyComputerNode.Count=0) Then exit;
  UpdateSubNodes(MyComputerNode,False);
end;

procedure TShellTree.UpdateSpecialDrives(Msg : Integer;HDR : PShellTree_Dev_Broadcast_Volume;MyComputerNode : TShellNode);
Var
  DriveBits: set of 0..25;
  CNode : TShellNode;
  Drive : String;
  DriveNum : Integer;
  ShowM : Boolean;
begin
//  Debugger.EnterProc('UpdateSpecialDrives');
  try
//  Debugger.LogFmtMsg('MsgValue = %x',[Msg]);
//  Debugger.LogFmtMsg('HDR^.dbcv_unitmask %x',[HDR^.dbcv_unitmask]);
//  Debugger.LogFmtMsg('HDR^.dbcv_flags    %x',[HDR^.dbcv_flags]);

  Integer(DriveBits):=HDR^.dbcv_unitmask;
  IF NOT Assigned(MyComputerNode) Then begin
    ShowM:=ShowMessages;
    ShowMessages:=false;
    try
      Drive:=FRootNode.Path;
//Debugger.LogMsg(Drive);
      IF (Length(Drive)=0) OR (Length(Drive)>3) Then exit;
      FOR DriveNum:=0 to 25 do begin
        IF DriveNum in Drivebits Then begin
          IF Drive[1]=Char(DriveNum+Ord('A')) Then begin
//Debugger.LogMsg('Update(Sub)Node(s) by Drive');
            UpdateNode(FRootNode,false);
            UpdateSubNodes(FRootNode,False);
          end;
        end;
      end;
    finally
      ShowMessages:=ShowM;
    end;
  end else begin
    CNode:=TShellNode(MyComputerNode.GetFirstChild);
    While Assigned(CNode) DO begin
      IF CNode.HasAttribute(SFGAO_FILESYSTEM) Then begin
        Drive:=CNode.Path;
        IF Length(Drive)>0 Then begin
          DriveNum:=Ord(Upcase(Drive[1]))-Ord('A');
          IF (DriveNum in DriveBits) then begin
//Debugger.LogMsg(Drive);
//Debugger.LogFmtMsg('Drivenum = %d',[DriveNum]);
            IF (Msg = ShellTree_DBT_DEVICEREMOVECOMPLETE) Then begin
              CNode.DeleteChildren;
              CNode.HasChildren:=True;
            end else if (Msg = ShellTree_DBT_DEVICEARRIVAL) Then
              CNode.HasChildren:=True;
            UpdateNode(CNode,False);
          end;
        end;
      end;
      CNode:=TShellNode(MyComputerNode.GetNextChild(CNode));
    end;
  end;
  finally
//    Debugger.LeaveProc('UpdateSpecialDrives');
  end;
end;

function TShellTree.SelectSpecialFolder(Node : TPidls) : TShellNode;
Var
  CSIDL : Integer;
  IdList : PItemIDList;
begin
  Result:=nil;
  CSIDL:=ConvertToCSIDL(Node);
  IF (CSIDL=CSIDL_USERDEFINED) Then exit;
  IF Succeeded(SHGetSpecialFolderLocation(MessageHandle,CSIDL, IdList)) then
  try
    Result:=FindPidl(IdList,True,True,True);
    IF Assigned(Result) Then begin
      Selected:=Result;
//      Node.Expand(False);
      Selected.MakeVisible;
    end;
  finally
    FreePidl(Allocator,IdList);
  end;
end;

function TShellTree.SpecialFolder(Node : TPidls) : TShellNode;
Var
  CSIDL : Integer;
  IdList : PItemIDList;
begin
  Result:=nil;
  CSIDL:=ConvertToCSIDL(Node);
  IF (CSIDL=CSIDL_USERDEFINED) Then exit;
  IF Succeeded(SHGetSpecialFolderLocation(MessageHandle,CSIDL, IdList)) then
  try
   Result:=FindPidl(IdList,false,True,True);
  finally
    FreePidl(Allocator,IdList);
  end;
end;

destructor TShellTree.Destroy;
begin
  DeallocateHWnd(FDeviceChangeHandle);
  FFilemaskList.Free;
  FreePidl(Allocator, FDesktopPidl);
  FreePidl(Allocator, FRootUserPidl);
  FreePidl(Allocator, FMyComputerPidl);
  ReleaseCom(FMyComputer);
  ReleaseCom(FDesktop);
  inherited destroy;
  ReleaseCom(Allocator);
end;

procedure TShellTree.UpdateNode(Node : TTreeNode;Recursive : Boolean);
Var
  SN : TShellNode;
  ParsePidl : PItemIDList;
begin
  IF NOT Assigned(Node) Then exit;
  {$IFDEF DoRepaintProperty}
    IF DoRepaint Then
  {$ENDIF}
  Items.BeginUpdate;
  try
    SN:=TShellNode(Node);
    GetPidlFromPath(SN.ParentShellFolder,SN.Path,False,ParsePidl);
//    IF ParsePidl=nil Then beep;
    SN.UpdateNodeinfo(SN.ParentShellFolder,ParsePidl,false);
    SN.UpdateNodeText;
    SN.UpdateIcon;
//    UpdateSubNodes(Node,Recursive);
  finally
  {$IFDEF DoRepaintProperty}
    IF DoRepaint Then
  {$ENDIF}
    Items.EndUpdate;
  end;
end;

procedure TShellTree.UpdateSubNodes(Node : TTreeNode;Recursive : Boolean);
Var
  TreeNode : TTreeNode;
begin
  IF NOT Assigned(Node) Then exit;
  EnumFolder(TShellNode(Node),True);
  IF Recursive Then begin
    TreeNode:=Node.GetFirstChild;
    While (TreeNode<>nil) DO begin
      IF  (TreeNode.Count>0) AND (TreeNode.Expanded) Then UpdateSubNodes(TreeNode,True)
        else TShellNode(TreeNode).Nodeinfo.Expanded:=False;
      TreeNode:=Node.GetNextChild(TreeNode);
    end;
  end;
end;



procedure TShellTree.WMSelectPidl(Var Message: TMessage);
Var
  P : PItemIDList;
  N : TShellNode;
begin
  // wparam : Boolean  - IsAbsoluteIDList
  // lparam : PItemIDList
  P:=PItemIDList(Message.lparam);
  N:=FindPidl(P,True,True,Boolean(Message.wparam));
  IF Assigned(N) Then begin
    Selected:=N;
    N.MakeVisible;
  end;
  Message.Result:=Integer(Assigned(N)); 
end;

function TShellTree.GetNodeFromPath(Path : String) : TShellNode;
Var
  ParsePidl : PItemIDList;
begin
  Result:=nil;
  //* 07.Jan 1998 Bug fixed
  IF NOT GetPidlFromPath(Desktop,Path,True,ParsePidl) Then exit;
  try
    Result:=GetNodeFromIDList(ParsePidl);
  finally
    FreePidl(Allocator,ParsePidl);
  end;
end;

function TShellTree.GetNodeFromIDList(IDList : PItemIDList) : TShellNode;
begin
  {$IFDEF DoRepaintProperty}
    IF DoRepaint Then
  {$ENDIF}
  Items.BeginUpdate;
  try
//    Result:=FindPidl(IDList,True,True,True);
    Result:=FindPidl(IDList,True,False,True);
  finally
    {$IFDEF DoRepaintProperty}
      IF DoRepaint Then
    {$ENDIF}
    Items.EndUpdate;
  end;
end;

procedure TShellTree.SelectPath(Path : String);
Var
  Node : TShellNode;
//  ParsePidl : PItemIDList;
begin
  Node:=GetNodeFromPath(Path);
  IF Assigned(Node) Then begin
    Node.Expand(False);
    Selected:=Node;
    Selected.MakeVisible;
  end;
end;

function TShellTree.FindPidl(IdList : PItemIDList;ChangeView,OnlyFullPath,AbsolutePidl : Boolean) : TShellNode;
Var
  Tmp,Pidl2,NodePidl : PItemIDList;
  PathFound : Boolean;
  ShellFolder : IShellFolder;
  ParentNode : TShellNode;
  ChildNode  : TShellNode;
  Save : Word;
  X,Y,Z : Integer;
begin
  //* 07.Jan 1998 Bug fixed
  Result:=nil;
  IF AbsolutePidl Then
    ParentNode:=FRootNode
  else
    ParentNode:=TShellNode(Selected);
  IF NOT Assigned(ParentNode) Then exit;
  X:=GetPidlLevel(IdList);
  Y:=GetPidlLevel(ParentNode.AbsoluteIDL);
  IF (X=Y) Then begin
    IF Desktop.CompareIDs(0,IdList,ParentNode.AbsoluteIDL)=0 Then Result:=ParentNode;
  end;
  IF AbsolutePidl Then begin
    IF (Y>=X) Then exit;
  end else
    Y:=0;  
  Pidl2:=IdList;
  FOR Z:=1 TO Y DO Pidl2:=PItemIDList(@Pidl2^.mkid.abid[Pidl2^.mkid.cb-2]);
  Tmp:=PItemIDList(@Pidl2^.mkid.abid[Pidl2^.mkid.cb-2]);
  Save:=Tmp^.mkid.cb;
  Tmp^.mkid.cb:=0;
  PathFound:=False;
  IF ChangeView AND NOT ParentNode.Expanded Then ParentNode.Expand(False);
  Childnode:=TShellNode(ParentNode.GetFirstChild);
  While NOT PathFound AND Assigned(ChildNode) DO begin
    ShellFolder:=ParentNode.ShellFolder;
    NodePidl:=ChildNode.RelativeIDL;
    IF (ShellFolder.CompareIDs(0,Pidl2,Nodepidl)=0) Then begin
      ParentNode:=ChildNode;
      Tmp^.mkid.cb:=Save;
      Pidl2:=Tmp;
      IF (GetPidlLevel(Pidl2)=0) Then PathFound:=True else begin
        IF ChangeView AND NOT ParentNode.Expanded Then ParentNode.Expand(False);
        Tmp:=PItemIDList(@Pidl2^.mkid.abid[Pidl2^.mkid.cb-2]);
        Save:=Tmp^.mkid.cb;
        Tmp^.mkid.cb:=0;
      end;
      ChildNode:=TShellNode(ParentNode.GetFirstChild);
      IF NOT PathFound AND NOT ParentNode.Expanded Then begin
        ChildNode:=nil;
        ParentNode:=nil;
      end;
    end else
      ChildNode:=TShellNode(ParentNode.GetNextChild(Childnode))
  end;
  IF (GetPidllevel(Tmp)>0) AND (OnlyFullPath) Then exit;
  Result:=ParentNode;
end;

procedure TShellTree.UpdateTreeView(LastAction : TLastAction);
Var
  OldDir,OldRoot : String;
begin
  {$IFDEF DoRepaintProperty}
    IF DoRepaint Then
  {$ENDIF}
  Items.BeginUpdate;
  try
    IF NOT FInitialized Then InitTree;
    IF (csloading in ComponentState) Then exit;
    OldDir:=Path;
    OldRoot:=RootDir;
    CreateRoot;
    IF (LastAction=la_RootDir) Then Path:=OldDir;
    IF (Length(OldRoot)>1) AND (LastAction<>la_RootDir) Then RootDir:=OldRoot;
  finally
    {$IFDEF DoRepaintProperty}
      IF DoRepaint Then
    {$ENDIF}
    Items.EndUpdate;
  end;
end;

function TShellTree.GetPidlFromPath(ShellFolder : IShellFolder;Path : String;Checkpath : Boolean; var Pidl : PItemIDList) : Boolean;
Var
  WCA : ARRAY[0..300] OF WideChar;
  S  : ULong;
  UL : ULong;
begin
  IF CheckPath then Path:=TestDirectory(Path);
  S:=Length(Path);
  Result:=False;
  IF S>0 Then begin
    StringToWideChar(Path,WCA,SizeOf(WCA));
    UL:=SFGAO_VALIDATE; //UL:=0;
    RESULT:=(ShellFolder.ParseDisplayName(MessageHandle,nil,@WCA,S,Pidl,UL)=NOERROR);
  end;
  IF NOT Result Then Pidl:=nil;
end;

//**********************************************************************
//                         Shell Edit Name
//**********************************************************************

function TShellTree.CanEdit(Node : TTreeNode) : Boolean;
begin
  Result:=inherited CanEdit(Node);
  IF Result Then
    Result:=TShellNode(Node).HasAttribute(SFGAO_CANRENAME);// AND Assigned(Node.Parent);
end;

procedure TShellTree.Edit(const Item: TTVItem);
var
  S: string;
  Node: TTreeNode;
begin
  with Item do
    if pszText <> nil then
    begin
      S := pszText;
      with Item do
       if (state and TVIF_PARAM) <> 0 then Node := Pointer(lParam)
         else Node := Items.GetNode(hItem);
      if Assigned(OnEdited) then OnEdited(Self, Node, S);
      if Node <> nil then begin
        IF TShellNode(Node).SetDisplayName(S) Then Node.Text := S;
      end;
    end;
end;


//**********************************************************************
//                         Shell Expand & Enumerator
//**********************************************************************

function TShellTree.CanExpand(Node: TTreeNode): Boolean;
Var
  SNode : TShellNode;
begin
  SNode:=TShellNode(Node);
  Result:=FDesignStart;
  if Assigned(OnExpanding) then OnExpanding(Self, Node, Result);
  IF Result AND (UpdAuto OR NOT (SNode.NodeInfo.Expanded)) Then begin
    UpdateSubNodes(SNode,UpdAutoRec);
    SNode.NodeInfo.Expanded:=True;
  end;
end;

procedure TShellTree.EnumFolder(Node: TShellNode;GetAllEntries : Boolean);
var
  ShellFolder: IShellFolder;
  Objects: IEnumIdList;
  ItemIdList: PItemIDList;
  DummyResult: ULONG;
  Count: integer;
  Flags : DWord;
  SaveCursor: TCursor;

begin
  IF NOT Assigned(Node) then exit;
  IF NOT Assigned(Node.ShellFolder) Then exit;
  {$IFDEF DoRepaintProperty}
    IF DoRepaint Then
  {$ENDIF}
  Items.BeginUpdate;
  SaveCursor:=Screen.Cursor;
  try
    Screen.Cursor:=crHourglass;
    Count:=0;
    ShellFolder:=Node.ShellFolder;
    Flags:=OptionsToFlags(Options);
    if not Assigned(Node.NodeInfo.RelativeIDL) then Flags:=Flags or SHCONTF_FOLDERS;
    Node.FalseUpdated;
    if ShellFolder.EnumObjects(MessageHandle, Flags, Objects)=NOERROR then begin
      Objects.Reset;
      DummyResult:=1;
      IF GetAllEntries Then begin
        while (DummyResult<=1) AND (Objects.Next(1, ItemIdList, DummyResult)=NOERROR) do begin
          AddSubfolderToNode(Node, ItemIdList);
        end;
      end else begin
        while (Count=0) AND (DummyResult<=1) AND (Objects.Next(1, ItemIdList, DummyResult)=NOERROR) do begin
          IF AddSubfolderToNode(Node, ItemIdList) Then inc(Count);
        end;
      end;
    end;
    Node.DeleteFalseUpdated;
    Node.HasChildren:=(Node.Count>0);
    SortNode(Node,False);
  finally
  {$IFDEF DoRepaintProperty}
    IF DoRepaint Then
  {$ENDIF}
    Items.EndUpdate;
    Screen.Cursor:=SaveCursor;
    ReleaseCom(Objects);
//    Application.ProcessMessages;
  end;
end;

function TShellTree.AddSubfolderToNode(Node: TShellNode; ItemIdList: PItemIDList) : Boolean;
var
  ShellFolder: IShellFolder;
  S: string;
  TreeNode: TShellNode;
  Attributes : UInt;
  HC : Boolean;
begin
  Result:=True;
  ShellFolder:=Node.ShellFolder;
  TreeNode:=FindNodePidl(Node,ItemIDList);
  Attributes:=All_Attributes;
  IF NOT (ShellFolder.GetAttributesOf(1,ItemIdList,Attributes)=NOERROR) Then Attributes:=0;
  IF Not ShowVirtualFolders Then begin
    IF ((Attributes and (SFGAO_FILESYSTEM or SFGAO_FILESYSANCESTOR))=0) then Result:=False;
  end;

  IF Result and (Attributes and (SFGAO_FILESYSTEM)<>0) then begin
    IF not HasAttribute(Attributes,SFGAO_FOLDER) and (TF_NONFOLDERS in Options) Then begin
      S:=GetShellItemName(ShellFolder,ItemIdList,SHGDN_FORPARSING);
      IF FileMaskUse AND NOT ShellTree_CmpMultiMask(ExtractFilename(S),FileMaskList) Then
        Result:=False
      else
        CheckOnFileAdd(Node,ItemIdList,Attributes,Result);
    end else begin
      CheckOnDirAdd(Node, ItemIdList, Attributes, Result);
    end;
  end;
  IF NOT Result Then begin
    FreePidl(Allocator,ItemIdList);
    IF Assigned(TreeNode) Then TreeNode.NodeInfo.Updated:=False;
    exit; // stop adding
  end;
  IF NOT Assigned(TreeNode) Then begin
    TreeNode:=TShellNode(Items.AddChild(Node,''));
    TreeNode.UpdateNodeInfo(ShellFolder,ItemIdList,False);
    TreeNode.UpdateNodeText;
    TreeNode.NodeInfo.Expanded:=False;
  end else begin
//    FreePidl(ItemIdList);
//    ItemIdList:=TreeNode.NodeInfo.RelativeIDL;
    TreeNode.UpdateNodeInfo(ShellFolder,ItemIdList,False);
    TreeNode.UpdateNodeText;
  end;
  Node.CreateMonitorThread;
  TreeNode.NodeInfo.Updated:=True;
  TreeNode.NodeInfo.Attributes:=Attributes;
  HC:=HasAttribute(Attributes,SFGAO_HASSUBFOLDER);
  IF (TF_NONFOLDERS in Options) AND NOT HC Then begin
    IF (TreeNode.HasAttribute(SFGAO_FOLDER)) AND (Assigned(TreeNode.ShellFolder))  Then begin
      IF NOT (TreeNode.Expanded) Then begin
        EnumFolder(TreeNode,False);
        HC:=TreeNode.Count>0;
      end else
        HC:=True;  
    end;
  end;
  IF NOT HC Then begin
    TreeNode.NodeInfo.Expanded:=false;
    TreeNode.DeleteChildren;
    IF NOT TreeNode.HasMonitorThread Then TreeNode.CreateMonitorThread;
  end;
  TreeNode.HasChildren:=HC;
  TreeNode.UpdateIcon;
end;

function TShellTree.CreateNode : TTreeNode;
begin
  Result:=TShellNode.Create(Items);
end;

function TShellTree.CanCollapse(Node: TTreeNode): Boolean;
begin
  Result:=inherited CanCollapse(Node);
  IF Result then
    TShellNode(Node).DestroySubMonitorThreads(True);
end;

procedure TShellTree.InitTree;
begin
  IF NOT FInitialized Then RetrieveSysImageList;
  FInitialized:=True;
end;

procedure TShellTree.CreateRoot;
begin
  Items.Clear;
  CreateRootPidl;
end;

procedure TShellTree.CreateRootPidl;
Var
  IdList : PItemIDList;
  CSIDL : Integer;
begin
  IF RootUsePidl Then begin
    CSIDL:=ConvertToCSIDL(RootPidl);
    IF (CSIDL<>CSIDL_USERDEFINED) Then begin
      IF NOT Succeeded(SHGetSpecialFolderLocation(MessageHandle,CSIDL, IdList)) then begin
        SHGetSpecialFolderLocation(MessageHandle,CSIDL_DESKTOP, IdList);
      end;
    end else begin
      CopyPidl(Allocator,FRootUserPidl,IdList);
    end;
  end else begin
    IF NOT GetPidlFromPath(Desktop,RootDir,True,IdList) Then
      GetPidlFromPath(Desktop,STD_DIR,True,IdList);
  end;
  CreateRootNode(IdList);
end;

procedure TShellTree.CreateRootNode(ItemIdList : PItemIDList);
Var
  Node : TShellNode;

begin
  FRootNode:=nil;
{$IFNDEF HideRootNode}
  Node:=TShellNode(Items. Add(Nil,''));
{$ELSE}
  Node:=TShellNode.Create(Items);
{$ENDIF}
  Node.UpdateNodeInfo(Desktop,ItemIdList,False);
  Node.UpdateNodeText;
  Node.HasChildren:=True;
  Node.UpdateIcon;
  FDesignStart:=True;
  Node.Expand(False);
  Selected:=Node;
  FRootNode:=Node;

  IF NOT RootUsePidl Then FRootDir:=GetPathFromPidl(ItemIDList);
{$IFDEF HideRootNode}
  CanExpand(Node);
  Selected:=Items.GetFirstNode;
  Changed(selected);
{$ENDIF}
end;

procedure TShellTree.RetrieveDefaultPidls;
begin
  SHGetSpecialFolderLocation(Handle,CSIDL_Desktop,FDesktopPidl);
  SHGetSpecialFolderLocation(Handle,CSIDL_Drives,FMyComputerPidl);
  Desktop.BindToObject(FMyComputerPidl,nil, IID_ISHELLFOLDER,pointer(FMyComputer));
  SetRootUserPidl(FDesktopPidl);
end;

procedure TShellTree.RetrieveMessageHandle;
Var
  Form : TCustomForm;
begin
  Form:=GetParentForm(self);
  IF Assigned(Form) Then begin
    FMessageHandle:=Form.Handle;
  end else begin
    IF ParentWindow<>0 Then FMessageHandle:=ParentWindow
      else FMessageHandle:=Handle;
  end;
end;

procedure TShellTree.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (AComponent = PopupMenu) and (Operation = opRemove) then PopupMenu := nil;
end;
{
procedure TShellTree.ProcessWinMessage;
var
  msgPeek: TMsg;
  bQueueEmpty: Boolean;
begin
  IF NOT FMultiTask OR (csDesigning in ComponentState) Then exit;
  repeat
    bQueueEmpty:=true;
    if PeekMessage(msgPeek,0,0,0,PM_REMOVE) then
      begin
        if (msgPeek.Message = WM_QUIT) then
          Halt(msgPeek.wParam);
        TranslateMessage(msgPeek);
        DispatchMessage(msgPeek);
        bQueueEmpty:=false;
      end;
  until bQueueEmpty;
end;
}

procedure TShellTree.SetRootDir(Value : String);
begin
  Value:=TestDirectory(Value);
  IF Length(Value)=0 Then exit;
  IF Value<>FRootDir Then begin
    FRootDir:=Value;
    IF HandleAllocated AND NOT RootUsePidl Then UpdateTreeView(la_RootDir);
  end;
end;

procedure TShellTree.SetRootPidl(Value : TPidls);
begin
  IF Value<>FRootPidl Then begin
    FRootPidl:=Value;
    IF HandleAllocated AND RootUsePidl Then UpdateTreeView(la_RootPidl);
  end;
end;



procedure TShellTree.SetOptions (Value : TVisOptions);
Var
  NewFilters : Integer;
begin
  NewFilters:=FILE_NOTIFY_CHANGE_ATTRIBUTES;
  IF (TF_FOLDERS in Value)    Then NewFilters:=NewFilters or FILE_NOTIFY_CHANGE_DIR_NAME;
  IF (TF_NONFOLDERS in Value) Then NewFilters:=NewFilters or FILE_NOTIFY_CHANGE_FILE_NAME;
  IF NewFilters<>DiscMonitorFilters Then begin
    DiscMonitorFilters:=NewFilters;
    DiscMonitorFiltersChange;
  end;
  IF Value <> FOptions Then begin
    FOptions :=Value;
    IF HandleAllocated Then UpdateTree;
  end;
end;

procedure TShellTree.SetPath(Value : String);
begin
{  Value:=TestDirectory(Value);
  IF Length(Value)=0 Then
    IF Items.Count>0 Then Selected:=FRootNode
      else exit;}
  IF Value<>GetPath Then begin
    IF HandleAllocated Then
      SelectPath(Value)
    else
      FTmpStartpath:=Value;
  end;
end;

function TShellTree.GetPath : String;
begin
  Result:='';
  IF Assigned(Selected) Then Result:=TShellNode(Selected).Path;
end;

procedure TShellTree.SetRootUsePidl(Value : Boolean);
begin
  IF Value<>FRootUsePidl Then begin
    FRootUsePidl:=Value;
    IF HandleAllocated Then UpdateTreeView(la_None);
  end;
end;

procedure TShellTree.CheckOnDirAdd(ParentNode : TShellNode;const ItemIdList : PItemIDList;Attributes : UInt;Var DoAdd : Boolean);
begin
  IF not (csDesigning in ComponentState) Then begin
    IF Assigned(FOnDirAdd) Then
      FOnDirAdd(ParentNode.GetSubpath(ItemIdList),Attributes,DoAdd);
  end;
end;

procedure TShellTree.CheckOnFileAdd(ParentNode : TShellNode;const ItemIdList : PItemIDList;Attributes : UInt;Var DoAdd : Boolean);
begin
  IF not (csDesigning in ComponentState) Then begin
    IF Assigned(FOnFileAdd) Then
      FOnFileAdd(ParentNode.GetSubpath(ItemIdList),Attributes,DoAdd);
  end;
end;

function TShellTree.GetRelativePidl : PItemIDList;
begin
  Result:=nil;
  IF Assigned(Selected) Then Result:=TShellNode(Selected).NodeInfo.RelativeIDL;
end;

function TShellTree.GetAbsolutePidl : PItemIDList;
begin
  Result:=nil;
  IF Assigned(Selected) Then Result:=TShellNode(Selected).NodeInfo.AbsoluteIDL;
end;

function TShellTree.GetShellFolder  : IShellFolder;
begin
  Result:=nil;
  IF Assigned(Selected) Then Result:=TShellNode(Selected).ShellFolder;
end;


procedure TShellTree.SetShowIcons  (Value : Boolean);
begin
  IF (Value<>FShowIcons) Then begin
    FShowIcons:=Value;
    IF not (csreading in ComponentState) Then begin
      IF Value Then
        TreeView_SetImageList(Handle, SysImageListSmall, TVSIL_NORMAL)
      else
        TreeView_SetImageList(Handle, 0, TVSIL_NORMAL);
    end;
  end;
end;

procedure TShellTree.SetShowVirtualFolders(Value : Boolean);
// Bug detected fixed 8.1.98
begin
  IF Value<>FShowVirtualFolders Then begin
    FShowVirtualFolders:=Value;
    IF not (csreading in ComponentState) Then UpdateTree;
  end;
end;

procedure TShellTree.RetrieveSysImageList;
Var
  FileInfo : TSHFileInfo;
  DesktoPItemIDList : PItemIDList;
begin
  SHGetSpecialFolderLocation(MessageHandle, CSIDL_DESKTOP, DesktoPItemIDList);
  SysImageListSmall:=SHGetFileInfo(PChar(DesktoPItemIDList), 0, FileInfo, sizeof(FileInfo),
                                   SHGFI_PIDL or SHGFI_SYSICONINDEX or SHGFI_SMALLICON);

  FreePidl(Allocator, DesktoPItemIDList);
  IF FShowIcons Then
      TreeView_SetImageList(Handle, SysImageListSmall, TVSIL_NORMAL)
end;

procedure TShellTree.SetRootUserPidl(Value : PItemIDList);
begin
  IF NOT Assigned(value) then Value:=FDesktopPidl;
  IF Assigned(FRootUserPidl) Then begin
    IF FDesktop.CompareIDs(0,Value,FRootUserPidl)=0 Then exit;
    FreePidl(Allocator, FRootUserPidl);
  end;
  CopyPidl(Allocator,Value,FRootUserPidl);
  IF (RootPidl=PIDL_USERDEFINED) AND (RootUsePidl) Then UpdateTreeView(la_RootPidl);
end;

function  TShellTree.GetFileMask     : String;
Var
  X : Integer;
begin
  Result:='';
  FOR X:=1 TO FileMaskList.Count DO begin
    Result:=Result+FileMaskList.Strings[X-1]+';';
  end;
  X:=Length(Result);
  IF X>0 Then DEC(X);
  Result:=Copy(Result,1,X);
end;

procedure TShellTree.SetFileMask(Value : String);

procedure Add(AddString : String);
begin
  AddString:=Trim(AddString);
  IF Length(AddString)>0 Then begin
    FileMaskList.Add(AddString);
  end;
end;

Var
  X : Integer;
  S : String;
begin
  IF Length(Trim(Value))=0 Then Value:='*.*';
  X:=Pos(';',Value);
  FilemaskList.Clear;
  While X>0 DO begin
    S:=Copy(Value,1,X-1);
    Add(S);
    System.Delete(Value,1,X);
    X:=Pos(';',Value);
  end;
  Trim(Value);
  Add(Value);
  IF (TF_NONFOLDERS in Options) Then UpdateTree;
end;

procedure TShellTree.SetFileMaskUse(Value : Boolean);
begin
  IF Value<>FFileMaskUse Then begin
    FFileMaskUse:=Value;
    IF (TF_NONFOLDERS in Options) Then UpdateTree;
  end;
end;

procedure TShellTree.SetFileSortType(Value : TFileSortType);
begin
  IF Value <> FFileSorttype Then begin
    FFileSorttype:=Value;
    SortNode(FRootNode,True);
  end;
end;

procedure TShellTree.SetFileSortOrder(Value : TFileSortOrder);
begin
  IF Value <> FFileSortOrder Then begin
    FFileSortOrder:=Value;
    SortNode(FRootNode,True);
  end;
end;

procedure TShellTree.SetPopupMenu(Value : TPopupMenu);
begin
  if FPopupMenu <> Value then begin
    FPopupMenu := Value;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

procedure TShellTree.SortNode(Node : TTreeNode;Recursive : Boolean);
Var
  Data : TSortdata;
  N : TTreeNode;
begin
  IF NOT Assigned(Node) Then exit;
  Data.FileSorttype:=FileSorttype;
  Data.FileSortOrder:=FileSortOrder;
  Node.CustomSort(DefaultSortProc,Integer(@Data));
  IF Recursive Then begin
    N:=Node.GetFirstChild;
    While Assigned(N) DO begin
      SortNode(N,Recursive);
      N:=Node.GetNextChild(N);
    end;
  end;
end;

procedure TShellTree.SetShowSharedOverlay(Value : Boolean);
begin
  IF (Value<>FShowSharedOverlay) Then begin
    FShowSharedOverlay:=Value;
    UpdateTree;
  end;
end;

procedure TShellTree.SetShowLinkOverlay(Value : Boolean);
begin
  IF (Value<>FShowLinkOverlay) Then begin
    FShowLinkOverlay:=Value;
    UpdateTree;
  end;
end;

procedure TShellTree.SetShowMessages(Value : Boolean);
begin
  IF Value<>FShowMessages Then begin
    FShowMessages:=Value;
  end;
end;

function  TShellTree.GetMessageHandle : HWnd;
begin
  IF FShowMessages Then
    Result:=FMessageHandle
  else
    Result:=0;
end;

{$IFNDEF DFS_COMPILER_4_UP}

procedure TShellTree.SetToolTips(Value : Boolean);
var
  Style: Integer;
begin
  IF Value<>FToolTips Then begin
    FToolTips:=Value;
    If HandleAllocated then begin
      Style := GetWindowLong(Handle, GWL_STYLE);
      if FToolTips then Style := Style or TVS_NOTOOLTIPS
        else Style := Style and not TVS_NOTOOLTIPS;
      SetWindowLong(Handle, GWL_STYLE, Style);
    end;
  end;
end;

{$ENDIF}

procedure TShellTree.DiscMonitorFiltersChange;
Var
  SN : TShellNode;
begin
  SN:=TShellNode(Items.GetFirstNode);
  IF NOT Assigned(SN) then exit;
  SN.ChangeDiscMonitorFilter(DiscMonitorFilters);
end;

//*******************************************************************
//                         DiscMonitor
//*******************************************************************

constructor TDiscMonitorThread.Create(aShellNode : TShellNode);
begin
  inherited Create (true);
  FInvalid:=False;
  FShellNode:=aShellNode;
  FDirectory:=FShellnode.Path;
  FFilters:=TShellTree(FShellNode.TreeView).DiscMonitorFilters;
  FDestroyEvent := CreateEvent (nil, false, false, nil);
  FChangeEvent := CreateEvent (nil, false, false, nil);
//  Debugger.LogMsg('New Thread '+FDirectory);
end;

destructor TDiscMonitorThread.Destroy;
begin
  SetEvent (FDestroyEvent);
//  Debugger.LogMsg('Destroying Thread'+FDirectory);
  inherited Destroy
end;

procedure TDiscMonitorThread.InformChange;
begin
  IF Assigned(FShellNode) Then
    FShellNode.FolderChanged;
end;

procedure TDiscMonitorThread.SetDirectory (const Value : string);
begin
  if Value <> FDirectory then
  begin
    FDirectory := Value;
    Update;
  end
end;

procedure TDiscMonitorThread.SetFilters (Value : integer);
begin
  if Value <> FFilters then
  begin
    FFilters := Value;
    Update;
  end
end;

procedure TDiscMonitorThread.Update;
begin
  if not Suspended then
    SetEvent (FChangeEvent);
end;

procedure TDiscMonitorThread.Execute;
const
  R : array [false..true] of BOOL = (BOOL (0), BOOL (1));
var
  A : array [0..2] of THandle;
  B : boolean;
begin
  B := false;
  A [0] := FDestroyEvent;
  A [1] := FChangeEvent;
  A [2] := FindFirstChangeNotification (PChar(FDirectory), R[false], FFilters);
  repeat
    if A [2] = INVALID_HANDLE_VALUE then
    begin
      FInvalid:=True;
{      case WaitForMultipleObjects (2, PWOHandleArray (@A), false, INFINITE) - WAIT_OBJECT_0 of
        0 : B := true;
        1 : A [2] := FindFirstChangeNotification (PChar(FDirectory), R[fSubTree], FFilters)
      end}
      B:=True;
    end else
      case WaitForMultipleObjects (3, PWOHandleArray (@A), false, INFINITE) - WAIT_OBJECT_0 of
        0 : begin
              FindCloseChangeNotification (A [2]);
              B := true
            end;
        1 : begin
              FindCloseChangeNotification (A [2]);
//              Debugger.LogMsg(Format('Thread %s updated',[FDirectory]));
              A [2] := FindFirstChangeNotification (PChar(FDirectory), R[false], FFilters)
            end;
        2 : begin
              Synchronize (InformChange);
//              Debugger.LogMsg(Format('Thread %s changed',[FDirectory]));
              FindNextChangeNotification (A [2])
            end;
      end
  until B;
  CloseHandle (FChangeEvent);
  CloseHandle (FDestroyEvent);
//  Debugger.LogMsg(Format('Thread %s closed',[FDirectory]));
end;

{$IFDEF CheckBoxes}
procedure TShellTree.SetCheckBoxes(value:Boolean);
begin
  if fCheckBoxes<>Value then begin
    fCheckBoxes:=Value;
    RecreateWnd;
  end;
end;
{$ENDIF}

procedure Register;
begin
  RegisterComponents('Shell', [TShellTree]);
end;

{$IFDEF OVERSEER}
initialization
  Debugger.Clear
{$ENDIF}
end.
