{
Copyright (c) 1998.  All rights reserved.
Orbital Decisions (Pty) Ltd
PO Box 1080, Milnerton, 7435, South Africa
sales@orbital.co.za
http://wwww.orbital.co.za/contents.htm

Demonstrates the functionality of the TODJobScheduler component
with a few simple rules:-
- The various item types are restricted to various job types,
    e.g. 'Item A1' is only allowed on a 'Job Type A'.
- Only on instance of an item type is allowed per job.
- A job's duration in days must be at least as long as the number
    of items it has.
- Job dates must follow each other in sequence with no gaps.
These rules are implemented in the schedule event handlers.
Some jobs and items are created on program startup by using the
  schedule component API.
}

unit DemoUnit;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, ODSched, StdCtrls, Buttons;

type
  TDemoForm = class(TForm)
    ODJobSchedule1: TODJobSchedule;
    ODJobSchedule2: TODJobSchedule;
    ODJobSchedule3: TODJobSchedule;
    Panel1: TPanel;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    Label7: TLabel;
    Label9: TLabel;
    Panel2: TPanel;
    AboutButton: TSpeedButton;
    ExitButton: TSpeedButton;
    SelectedText: TStaticText;
    Label8: TLabel;
    procedure InitJobDialog(Schedule: TODJobSchedule; Job: TODJob;
      UpperCaptionList, LowerCaptionList: TStrings);
    procedure InitItemDialog(Schedule: TODJobSchedule; Job: TODJob;
      Item: TODJobItem; CaptionList: TStrings);
    procedure AfterJobAdd(Schedule: TODJobSchedule; Job: TODJob);
    procedure FormShow(Sender: TObject);
    procedure ItemDragOver(Schedule: TODJobSchedule; Job: TODJob;
      Item: TODJobItem; Index: Integer; var Accept: Boolean);
    procedure AboutButtonClick(Sender: TObject);
    procedure ExitButtonClick(Sender: TObject);
    procedure ScheduleResize(Sender: TObject);
    procedure JobClick(Schedule: TODJobSchedule; Job: TODJob);
    procedure ItemClick(Schedule: TODJobSchedule; Job: TODJob;
      Item: TODJobItem);
    procedure AfterItemAdd(Schedule: TODJobSchedule; Job: TODJob;
      Item: TODJobItem);
    procedure BeforeItemFree(Schedule: TODJobSchedule; Job: TODJob;
      Item: TODJobItem);
    procedure BeforeJobFree(Schedule: TODJobSchedule; Job: TODJob);
    procedure AfterJobEdit(Schedule: TODJobSchedule; Job: TODJob);
    procedure AfterJobRemove(Schedule: TODJobSchedule);
    procedure AfterItemRemove(Schedule: TODJobSchedule; Job: TODJob);
    procedure BeforeJobRemove(Schedule: TODJobSchedule; Job: TODJob);
    procedure JobDragOver(Schedule: TODJobSchedule;
      Job: TODJob; Index: Integer; var Accept: Boolean);
    procedure JobDragDrop(Schedule: TODJobSchedule;
      Job: TODJob);
  private
    SelectedJob: TODJob;
    SelectedItem: TODJobItem;
    procedure UpdateDates(Schedule: TODJobSchedule);
  end;

var
  DemoForm: TDemoForm;

implementation

uses AboutUnit;

{$R *.DFM}

procedure TDemoForm.FormShow(Sender: TObject);
begin
  ShortDateFormat := 'dd/mm/yyyy';
  // Initialize schedules with some jobs and items
  with ODJobSchedule1 do
  begin
    with AddJobFrom(0, 'Job Type A', '',
      StrToDateTime('1/8/1998'), StrToDateTime('10/8/1998')) do
    begin
      AddItemFrom(0, 'Item A1');
      AddItemFrom(0, 'Item A2');
    end;
    with AddJobFrom(0, 'Job Type B', '',
      StrToDateTime('10/8/1998'), StrToDateTime('20/8/1998')) do
    begin
      AddItemFrom(0, 'Item B1');
      AddItemFrom(0, 'Item B2');
    end;
  end;
  with ODJobSchedule2.AddJobFrom(0, 'Job Type A', '',
    StrToDateTime('15/8/1998'), StrToDateTime('22/8/1998')) do
  begin
    AddItemFrom(0, 'Item A1');
    AddItemFrom(0, 'Item A3');
  end;
end;

procedure TDemoForm.InitJobDialog(Schedule: TODJobSchedule;
  Job: TODJob; UpperCaptionList, LowerCaptionList: TStrings);
begin
  if Job = nil then  //if adding a new job then list all available types
  begin
    UpperCaptionList.Add('Job Type A');
    UpperCaptionList.Add('Job Type B');
    UpperCaptionList.Add('Job Type C');
    UpperCaptionList.Add('Job Type D');
  end
  else  //else editing an existing job so prevent from changing the type
    UpperCaptionList.Add(Job.UpperCaption);
end;

procedure TDemoForm.InitItemDialog(Schedule: TODJobSchedule;
  Job: TODJob; Item: TODJobItem; CaptionList: TStrings);
begin
  if Job.UpperCaption = 'Job Type A' then
  begin
    if Job.ItemByCaption('Item A1') = nil then
      CaptionList.Add('Item A1');
    if Job.ItemByCaption('Item A2') = nil then
      CaptionList.Add('Item A2');
    if Job.ItemByCaption('Item A3') = nil then
      CaptionList.Add('Item A3');
  end
  else if Job.UpperCaption = 'Job Type B' then
  begin
    if Job.ItemByCaption('Item B1') = nil then
      CaptionList.Add('Item B1');
    if Job.ItemByCaption('Item B2') = nil then
      CaptionList.Add('Item B2');
    if Job.ItemByCaption('Item B3') = nil then
      CaptionList.Add('Item B3');
  end
  else if Job.UpperCaption = 'Job Type C' then
  begin
    if Job.ItemByCaption('Item C1') = nil then
      CaptionList.Add('Item C1');
    if Job.ItemByCaption('Item C2') = nil then
      CaptionList.Add('Item C2');
    if Job.ItemByCaption('Item C3') = nil then
      CaptionList.Add('Item C3');
  end
  else if Job.UpperCaption = 'Job Type D' then
  begin
    if Job.ItemByCaption('Item D1') = nil then
      CaptionList.Add('Item D1');
    if Job.ItemByCaption('Item D2') = nil then
      CaptionList.Add('Item D2');
    if Job.ItemByCaption('Item D3') = nil then
      CaptionList.Add('Item D3');
  end;
end;

procedure TDemoForm.AfterJobAdd(Schedule: TODJobSchedule; Job: TODJob);
begin
  with Job do
  begin
    if UpperCaption = 'Job Type A' then
      Color := clBlue
    else if UpperCaption = 'Job Type B' then
      Color := clGreen
    else if UpperCaption = 'Job Type C' then
      Color := clMaroon
    else if UpperCaption = 'Job Type D' then
      Color := clPurple;
    if FinishDate < StartDate then
      FinishDate := Startdate;
  end;
  UpdateDates(Schedule);
end;

procedure TDemoForm.ItemDragOver(Schedule: TODJobSchedule; Job: TODJob;
  Item: TODJobItem; Index: Integer; var Accept: Boolean);
begin
  Accept := (Job = Item.Job) or     //can rearrange on same job
    ((Job.Color = Item.Job.Color) and         //only accept items form the job type
     (Job.ItemByCaption(Item.Caption) = nil));   //and if not already has item type
end;

procedure TDemoForm.AboutButtonClick(Sender: TObject);
begin
  ODAboutForm.ShowModal;
end;

procedure TDemoForm.ExitButtonClick(Sender: TObject);
begin
  Close;
end;

procedure TDemoForm.JobClick(Schedule: TODJobSchedule; Job: TODJob);
begin
  if SelectedJob <> nil then
    SelectedJob.Font.Color := clWindow;
  SelectedJob := Job;
  SelectedJob.Font.Color := clYellow;
  if (SelectedItem <> nil) and (SelectedJob <> SelectedItem.Job) then
  begin
    SelectedItem.Color := clWindow;
    if Job.ItemCount > 0 then
      SelectedItem := SelectedJob.Items[0]
    else
      SelectedItem := nil;
  end;
  if SelectedItem <> nil then
  begin
    SelectedItem.Color := clYellow;
    SelectedText.Caption :=
      SelectedJob.UpperCaption + ' / ' + SelectedItem.Caption;
  end
  else SelectedText.Caption := SelectedJob.UpperCaption;
end;

procedure TDemoForm.ItemClick(Schedule: TODJobSchedule; Job: TODJob;
  Item: TODJobItem);
begin
  if SelectedJob <> nil then
    SelectedJob.Font.Color := clWindow;
  SelectedJob := Job;
  SelectedJob.Font.Color := clYellow;
  if SelectedItem <> nil then
    SelectedItem.Color := clWindow;
  SelectedItem := Item;
  SelectedItem.Color := clYellow;
  SelectedText.Caption :=
    SelectedJob.UpperCaption + ' / ' + SelectedItem.Caption;
end;

procedure TDemoForm.AfterItemAdd(Schedule: TODJobSchedule; Job: TODJob;
  Item: TODJobItem);
begin
  Job.FinishDate := Job.FinishDate + 1;  //auto increase duration
  UpdateDates(Schedule);
  if Item.Job <> SelectedJob then
    ItemClick(Schedule, Job, Item);
end;

procedure TDemoForm.BeforeItemFree(Schedule: TODJobSchedule;
  Job: TODJob; Item: TODJobItem);
begin
  if Item = SelectedItem then
  begin
    SelectedItem := nil;
    SelectedText.Caption := SelectedJob.UpperCaption;
  end;
end;

procedure TDemoForm.BeforeJobFree(Schedule: TODJobSchedule; Job: TODJob);
begin
  if Job = SelectedJob then
  begin
    SelectedJob := nil;
    SelectedItem := nil;
    SelectedText.Caption := '';
  end;
end;

procedure TDemoForm.AfterJobEdit(Schedule: TODJobSchedule; Job: TODJob);
begin
  with Job do     //enforce duration rule
    if FinishDate < StartDate + ItemCount then
    begin
      MessageDlg('The job duration in days cannot be negative or less ' +
        'than the number of items that it contains.', mtError, [mbOK], 0);
      FinishDate := StartDate + ItemCount;
    end;
  UpdateDates(Schedule);
end;

procedure TDemoForm.AfterJobRemove(Schedule: TODJobSchedule);
begin
  UpdateDates(Schedule);
end;

procedure TDemoForm.UpdateDates(Schedule: TODJobSchedule);
var
  PrevStartDate, LastFinishDate: TDateTime;
  ix: Integer;
begin
  if Schedule.JobCount > 0 then
  begin
    LastFinishDate := Schedule.Jobs[0].FinishDate;
    for ix := 1 to Schedule.JobCount-1 do
      with Schedule.Jobs[ix] do
      begin
        PrevStartDate := StartDate;
        StartDate := LastFinishDate;
        FinishDate := FinishDate + (StartDate - PrevStartDate);
        LastFinishDate := FinishDate;
      end;
    Schedule.Footer := 'Finish: ' + DateToStr(LastFinishDate);
  end
  else Schedule.Footer := '';
end;

procedure TDemoForm.AfterItemRemove(Schedule: TODJobSchedule; Job: TODJob);
begin
  Job.FinishDate := Job.FinishDate - 1;  //auto decrease duration
  UpdateDates(Schedule);
end;

procedure TDemoForm.BeforeJobRemove(Schedule: TODJobSchedule; Job: TODJob);
var
  PrevStartDate: TDateTime;
begin
  //if removing first job, make next job start on same day
  if (Job = Schedule.Jobs[0]) and (Schedule.JobCount > 1) then
    with Schedule.Jobs[1] do
    begin
      PrevStartDate := StartDate;
      StartDate := Job.StartDate;
      FinishDate := FinishDate + (StartDate - PrevStartDate);
    end;
end;

procedure TDemoForm.JobDragOver(Schedule: TODJobSchedule; Job: TODJob;
  Index: Integer; var Accept: Boolean);
var
  PrevStartDate: TDateTime;
begin
  //if dragged to first position then make it start on prev first job's start date
  if (Job.Schedule = Schedule) and (Index = 0) and (Schedule.JobCount > 1) then
    with Job do
    begin
      PrevStartDate := StartDate;
      StartDate := Schedule.Jobs[0].StartDate;
      FinishDate := FinishDate + (StartDate - PrevStartDate);
    end;
  Accept := True;
end;

procedure TDemoForm.JobDragDrop(Schedule: TODJobSchedule; Job: TODJob);
begin
  if Schedule = Job.Schedule then
    UpdateDates(Schedule);
end;

procedure TDemoForm.ScheduleResize(Sender: TObject);
begin
  ODJobSchedule2.Left := ODJobSchedule1.Left + ODJobSchedule1.Width;
  ODJobSchedule3.Left := ODJobSchedule2.Left + ODJobSchedule2.Width;
  Panel1.Left := ODJobSchedule3.Left + ODJobSchedule3.Width;
end;

initialization
  RegisterClasses([TODJobItem, TODJob, TODJobSchedule]);
end.
