
{*******************************************************}
{                                                       }
{       FuturaSoft Visual Component Library             }
{       Ruler Components                                }
{       Version 2.00 - 1. February 1998                 }
{       Unregistered (no TTabRuler source)              }
{       For Delphi 1.0, 2.0 and 3.0                     }
{                                                       }
{       Copyright (c) 1996-1998 Martin Krmer           }
{                                                       }
{       Martin Krmer                                   }
{       Adolf-Kolping-Str. 4                            }
{       D49179 Ostercappeln                            }
{       (Germany)                                       }
{                                                       }
{       E-mail: kraemer@mathematik.Uni-Osnabrueck.DE    }
{                                                       }
{*******************************************************}

unit Ruler;

{$B-,Z-}

{$IFDEF WIN32}
{$IFNDEF VER90}
{$DEFINE DELPHI3}  { Delphi Version >= 3.0 }
{$ENDIF}
{$ENDIF}

{$DEFINE AUTOSTYLE}
{$DEFINE UNREGISTERED}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Menus;

type
  TWindowsStyle = (wsAutoDetect, wsWin31, wsWin40);

const
{$IFDEF AUTOSTYLE}
  WindowsStyle : TWindowsStyle = wsAutoDetect;
{$ELSE}
{$IFDEF WIN32}
  WindowsStyle = wsWin40;
{$ELSE}
  WindowsStyle = wsWin31;
{$ENDIF}
{$ENDIF}

const
  beText = 0;
  beLoMetric = 254;
  beHiMetric = 2540;
  beLoEnglish = 100;
  beHiEnglish = 1000;
  beTWIPS = 1440;

type
  TRulerKind = (rbHorizontal, rbVertical);

  TRulerOption = (roShowMousePos, roShowCursorPos, roShowTabMarks,
    roTabEditing);
  TRulerOptions = set of TRulerOption;

  TRulerColumn = record
    Min : Integer;
    Max : Integer;
  end;

type
  TRulerTabKind = (tkDefault, tkLeftJustify, tkCenter, tkRightJustify,
    tkDecimal, tkLine);

  TRulerTabState = (tsNormal, tsEditing, tsFixed, tsDisabled);

  PRulerTab = ^TRulerTab;
  TRulerTab = record
    State : TRulerTabState;
    Kind : TRulerTabKind;
    Pos : Integer;
    PixelPos : Integer;
    {FillChar : Char;}
  end;

const
  MaxTabListSize = 65520 div SizeOf(TRulerTab);

type
  PRulerTabs = ^TRulerTabs;
  TRulerTabs = array[0..MaxTabListSize - 1] of TRulerTab;

  PTabList = ^TTabList;
  TTabList = record
    Visible : Boolean;
    FixedCapacity : Boolean;
    Capacity : Longint;
    Count : Longint;
    Tabs : PRulerTabs;
  end;

  {
                    |-----------------------------------------------|
  Gerteeinheiten:  Left/Top                                        Right/Bottom
                    |
                    |-----------------------|
  "                 0                       PixelOrigin
                                            |
                          |-----------------|
  Log. Einheiten:         0                 BaseOrigin
                          |
                          |-----|-----|-----------------|-----|
  "                       0     Min   ScaleOrigin       Max   Size
                          .           |                       .
                          '--|--'--'--#--'--'--|--'--'--#--'--'
                                      | ||              |
  Log. Skaleneinh.:                   0 |ScaleDivision  ScaleInterval
                                        ScaleUnit

  Mastabsverhltnisse:
  Gerteeinheiten = Log. Einheiten * PixelExtent/BaseExtent
                  = Log. Skaleneinh. * ScalePixelExtent/ScaleExtent
  }

  TRuler = class(TCustomControl)
  private
    { Private-Deklarationen }
    FTrueBaseExtent : Integer;
    FTruePixelExtent : Integer;
    FTrueScaleExtent : Integer;
    FTrueScalePixelExtent : Integer;
    FTextLeading : Integer;
    FTextOLAscent : Integer;
    FTextPixelsValid : Boolean;
    FRulerBitmap : TBitmap;
    FSavedCursorBitmap : TBitmap;
    FColumns : TList;
    FBaseExtent : Integer;
    FBaseOrigin : Integer;
    FBasePrecision : Integer;
    FCursorPos : Integer;
    FKind : TRulerKind;
    FMax : Integer;
    FMin : Integer;
    FMousePos : Integer;
    FOptions : TRulerOptions;
    FPixelExtent : Integer;
    FPixelOrigin : Integer;
    FScaleDivision : Integer;
    FScaleExtent : Integer;
    FScaleInterval : Integer;
    FScaleOrigin : Integer;
    FScalePixelExtent : Integer;
    FScalePrecision : Integer;
    FScaleUnit : Integer;
    FSize : Integer;
    FOnPaint : TNotifyEvent;
    procedure AdjustFontMetrics;
    procedure UpdateCursorPos(AShowing : Boolean);
    procedure UpdateMousePos(AShowing : Boolean);
    function GetColumn(Index : Integer) : TRulerColumn;
    function GetColumnCount : Integer;
    procedure SetBaseExtent(Value : Integer);
    procedure SetBaseOrigin(Value : Integer);
    function GetBorderStyle : TBorderStyle;
    procedure SetBorderStyle(Value : TBorderStyle);
    procedure SetCursorPos(Value : Integer);
    procedure SetKind(Value : TRulerKind);
    procedure SetMax(Value : Integer);
    procedure SetMin(Value : Integer);
    procedure SetMousePos(Value : Integer);
    procedure SetOptions(Value : TRulerOptions);
    procedure SetPixelExtent(Value : Integer);
    procedure SetPixelOrigin(Value : Integer);
    procedure SetScaleDivision(Value : Integer);
    procedure SetScaleExtent(Value : Integer);
    procedure SetScaleInterval(Value : Integer);
    procedure SetScaleOrigin(Value : Integer);
    procedure SetScalePixelExtent(Value : Integer);
    procedure SetScaleUnit(Value : Integer);
    procedure SetSize(Value : Integer);
    function GetTransparent : Boolean;
    procedure SetTransparent(Value : Boolean);
    procedure CMCtl3DChanged(var Message : TMessage); message CM_CTL3DCHANGED;
    procedure CMFontChanged(var Message : TMessage); message CM_FONTCHANGED;
    procedure CMColorChanged(var Message : TMessage); message CM_COLORCHANGED;
    procedure WMEraseBkgnd(var Message : TWMEraseBkgnd); message WM_ERASEBKGND;
  protected
    { Protected-Deklarationen }
    procedure CreateParams(var Params : TCreateParams); override;
    procedure Paint; override;
    procedure Painted; virtual;
    procedure SetParent(AParent : TWinControl); override;
  public
    { Public-Deklarationen }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure InsertColumn(AColumn : TRulerColumn);
    procedure RemoveColumn(AColumn : TRulerColumn);
    function GetBasePos(var PixelPos : Integer) : Integer;
    function GetPixelPos(Pos : Integer) : Integer;
    function GetScalePos(var PixelPos : Integer) : Integer;
    function GetScalePixelPos(Pos : Integer) : Integer;
    property Canvas;
    property Columns[Index : Integer] : TRulerColumn read GetColumn;
    property ColumnCount : Integer read GetColumnCount;
    property CursorPos : Integer read FCursorPos write SetCursorPos;
    property MousePos : Integer read FMousePos write SetMousePos;
  published
    { Published-Deklarationen }
    property BaseExtent : Integer read FBaseExtent write SetBaseExtent;
    property BaseOrigin : Integer read FBaseOrigin write SetBaseOrigin
      default 0;
    property BasePrecision : Integer read FBasePrecision write FBasePrecision
      default 0;
    property BorderStyle : TBorderStyle read GetBorderStyle write SetBorderStyle
      default bsSingle;
    property Kind : TRulerKind read FKind write SetKind default rbHorizontal;
    property Max : Integer read FMax write SetMax default High(Integer);
    property Min : Integer read FMin write SetMin default 0;
    property Options : TRulerOptions read FOptions write SetOptions default [];
    property PixelExtent : Integer read FPixelExtent write SetPixelExtent
      default 0;
    property PixelOrigin : Integer read FPixelOrigin write SetPixelOrigin
      default 0;
    property ScaleDivision : Integer read FScaleDivision write SetScaleDivision;
    property ScaleExtent : Integer read FScaleExtent write SetScaleExtent;
    property ScaleInterval : Integer read FScaleInterval write SetScaleInterval;
    property ScaleOrigin : Integer read FScaleOrigin write SetScaleOrigin
      default 0;
    property ScalePixelExtent : Integer read FScalePixelExtent
      write SetScalePixelExtent default 0;
    property ScalePrecision : Integer read FScalePrecision write FScalePrecision
      default 0;
    property ScaleUnit : Integer read FScaleUnit write SetScaleUnit;
    property Size : Integer read FSize write SetSize default -1;
    property Transparent : Boolean read GetTransparent write SetTransparent
      default False;

    property Align;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property ParentColor default False;
    property ParentCtl3D;
    property ParentFont default False;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnPaint : TNotifyEvent read FOnPaint write FOnPaint;
{$IFDEF WIN32}
    property OnStartDrag;
{$ENDIF}
  end;

  TTabRuler = class(TRuler)
  end;


function IMulDiv(Number : Integer; Numerator : Integer; Denominator : Integer;
  var Remainder : Integer) : Integer;
function IMulDivEx(Number : Integer; Numerator : Integer; Correction : Integer;
  Denominator : Integer; var Remainder : Integer) : Integer;
{function IMulDivAdjust(Numerator1 : Integer; Denominator1 : Integer;
  Numerator2 : Integer; Denominator2 : Integer) : Integer;}

procedure Register;

implementation

{$IFDEF WIN32}
  {$R *32}
{$ELSE}
  {$R *}
{$ENDIF}

const
  RulerBitmapWidth = 8;
  RulerBitmapHeight = 8;
  RulerBitmapVertHOffset = (Ord(High(TRulerTabKind)) + 1) * RulerBitmapWidth;
  RulerBitmapNormVOffset = 0;
  RulerBitmapGrayVOffset = RulerBitmapHeight;
  RulerBitmapSelectVOffset = 2 * RulerBitmapHeight;
  RulerBitmapMaskVOffset = 3 * RulerBitmapHeight;
  RulerBitmapHotSpots : array[Low(TRulerTabKind)..High(TRulerTabKind)] of Byte
    = (3, 1, 3, 5, 3, 3);

var
  RulerBitmap : TBitmap;

{$IFDEF UNREGISTERED}
function DelphiIsRunning : Boolean;
begin
  Result := (FindWindow('TAlignPalette', nil) <> 0)
    and (FindWindow('TPropertyInspector', nil) <> 0)
    and (FindWindow('TAppBuilder', nil) <> 0)
{$IFNDEF WIN32}
    and (GetModuleHandle('DELPHIED') <> 0)
    and (GetModuleHandle('DELPHIKB') <> 0)
{$ENDIF}
    ;
end;
{$ENDIF}

procedure FreeBitmaps; far;
begin
  if RulerBitmap <> nil then
  begin
    RulerBitmap.Free;
    RulerBitmap := nil;
  end;
end;

procedure ReadBitmaps;
{var
  BitmapMask : TBitmap;}
begin
  if RulerBitmap = nil then
  begin
{$IFNDEF WIN32}
    AddExitProc(FreeBitmaps);
{$ENDIF}
    RulerBitmap := TBitmap.Create;
    RulerBitmap.Handle := LoadBitmap(HInstance, 'RULERBITMAPS');
    (*BitmapMask := TBitmap.Create;
    try
      BitmapMask.Assign(RulerBitmap);
      { Color which will be transparent }
      BitmapMask.Canvas.Brush.Color := clFuchsia;
      BitmapMask.Monochrome := True;
      BitmapMask.Canvas.Brush.Color := clBlack;
      BitmapMask.Canvas.Font.Color := clWhite;
      BitmapMask.Monochrome := False;
      RulerBitmap.Height := RulerBitmap.Height * 2;
      RulerBitmap.Canvas.CopyMode := cmSrcAnd;
      RulerBitmap.Canvas.Draw(0, 0, BitmapMask);
      RulerBitmap.Canvas.CopyMode := cmNotSrcCopy;
      RulerBitmap.Canvas.Draw(0, BitmapMask.Height, BitmapMask);
    finally
      BitmapMask.Free;
    end;*)
  end;
end;

{ Implementationsteil von TRuler }

constructor TRuler.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  Width := 121;
  Height := 15;
  ControlStyle := [csCaptureMouse, csClickEvents, csFramed, csOpaque,
    csDoubleClicks];
  with Font do
  begin
    { Ein guter Compiler sollte die nachfolgende Anweisung
      so wie $IFDEF AUTOSTYLE optimieren. }
    if WindowsStyle >= wsWin40 then
    begin
      { Windows 95 Layout }
      Name := 'MS Sans Serif';
      Size := 8;
    end
    else begin
      Name := 'Small Fonts';
      Size := 7;
    end;
  end;
  ParentColor := False;
  ParentFont := False;
  BaseExtent := beLoMetric;
  FCursorPos := -1;
  FMax := High(FMax);
  FMousePos := -1;
  FScaleDivision := 25;
  ScaleExtent := beLoMetric;
  FScaleInterval := 100;
  FScaleUnit := 100;
  FSize := -1;
  ReadBitmaps;
  FRulerBitmap := RulerBitmap;
  FSavedCursorBitmap := TBitmap.Create;
  FSavedCursorBitmap.Width := RulerBitmapWidth;
  FSavedCursorBitmap.Height := RulerBitmapHeight;
end;

destructor TRuler.Destroy;
begin
  if FRulerBitmap <> RulerBitmap then
    FRulerBitmap.Free;
  FSavedCursorBitmap.Free;
  if FColumns <> nil then
    FColumns.Free;
  inherited Destroy;
end;

procedure TRuler.CreateParams(var Params : TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    {if csFramed in ControlStyle then
      Style := Style or WS_BORDER;}
    if not (csOpaque in ControlStyle) then
      ExStyle := ExStyle or WS_EX_TRANSPARENT;
  end;
end;

procedure TRuler.SetParent(AParent : TWinControl);
begin
  inherited SetParent(AParent);
  if AParent <> nil then
    AdjustFontMetrics;
end;

procedure TRuler.CMCtl3DChanged(var Message : TMessage);
begin
  inherited;
  Invalidate;
end;

procedure TRuler.CMFontChanged(var Message : TMessage);
begin
  AdjustFontMetrics;
  inherited;  { Impliziert Aufruf von Invalidate }
end;

procedure TRuler.CMColorChanged(var Message : TMessage);
begin
  if Color = clWhite then
  begin
    if FRulerBitmap <> RulerBitmap then
    begin
      FRulerBitmap.Free;
      FRulerBitmap := RulerBitmap;
    end;
  end
  else begin
    if FRulerBitmap = RulerBitmap then
    begin
      FRulerBitmap := TBitmap.Create;
      FRulerBitmap.Assign(RulerBitmap);
    end;
    with FRulerBitmap do
    begin
      Canvas.Brush.Color := Self.Color;
      Canvas.BrushCopy(Bounds(0, 0, Width, Height div 2),
        RulerBitmap, Bounds(0, 0, Width, Height div 2), clWhite);
    end;
  end;
  inherited;  { Impliziert Aufruf von Invalidate }
end;

procedure TRuler.AdjustFontMetrics;
var
  DC : HDC;
  SaveFont : HFont;
  SysMetrics, Metrics : TTextMetric;
  F : TForm;
  PixelsPerInch : Integer;
begin
  DC := GetDC(0);
  GetTextMetrics(DC, SysMetrics);
  SaveFont := SelectObject(DC, Font.Handle);
  GetTextMetrics(DC, Metrics);
  SelectObject(DC, SaveFont);
  ReleaseDC(0, DC);
  FTextLeading := Metrics.tmInternalLeading;
  FTextOLAscent := Metrics.tmAscent - FTextLeading;
  if (FPixelExtent = 0) or (FScalePixelExtent = 0) then
  begin
    F := TForm(GetParentForm(Self));
    if F = nil then
      PixelsPerInch := Screen.PixelsPerInch
    else
      PixelsPerInch := F.PixelsPerInch;
    if FPixelExtent = 0 then
    begin
      FTruePixelExtent := PixelsPerInch;
      if FBaseExtent = beText then
        FTrueBaseExtent := PixelsPerInch;
    end;
    if FScalePixelExtent = 0 then
    begin
      FTrueScalePixelExtent := PixelsPerInch;
      if FScaleExtent = beText then
        FTrueScaleExtent := PixelsPerInch;
    end;
    FTextPixelsValid := False;
  end;
end;

procedure TRuler.WMEraseBkgnd(var Message : TWMEraseBkgnd);
begin
  inherited;
  if not (csOpaque in ControlStyle) then
    Paint;
end;

procedure BrushOverlay(ACanvas : TCanvas; const Dest : TRect; Bitmap : TBitmap;
  const Source : TRect; Color : TColor);
const
  ROP_DSPDxax = $00E20746;
var
  SavedColor : TColor;
  crBack, crText : TColorRef;
begin
  if Bitmap = nil then Exit;
  with ACanvas do
  begin
    SavedColor := Brush.Color;
    Brush.Color := Color;
    if Bitmap.Monochrome then
    begin
      crText := SetTextColor(Handle, 0);
      crBack := SetBkColor(Handle, $FFFFFF);
    end
    else begin
      crText := 0;  { To avoid compiler warning }
      crBack := 0;  { To avoid compiler warning }
    end;
    StretchBlt(Handle, Dest.Left, Dest.Top, Dest.Right - Dest.Left,
      Dest.Bottom - Dest.Top, Bitmap.Canvas.Handle, Source.Left, Source.Top,
      Source.Right - Source.Left, Source.Bottom - Source.Top, ROP_DSPDxax);
    if Bitmap.Monochrome then
    begin
      SetTextColor(Handle, crText);
      SetBkColor(Handle, crBack);
    end;
    Brush.Color := SavedColor;
  end;
end;

procedure TRuler.Paint;
var
  R : TRect;
  TrueStartX, StartX, MinX, MaxX, EndX, TrueEndX : Integer;
  ScaleOriginX : Integer;
  ScaleRoundDiff, ScaleRoundCorrection : Integer;
  MarkPosition : Longint;
  ScaleIntervalMid : Integer;
  MarkPositionX, MarkPositionY, MarkTextOffsetX, MarkTextOffsetY : Integer;
  MarkText : string;
  XL : Longint;

  procedure DrawWindow(Left, Top, Right, Bottom : Integer; Color : TColor);
  var
    LTFColor, RBFColor : TColor;
    Framed3D : Boolean;
  begin
    with Canvas do
    begin
      Framed3D := csFramed in ControlStyle;
      if csOpaque in ControlStyle then
        Brush.Color := Color
      else if Framed3D then
        Brush.Style := bsClear
      else
        Exit;
      if Framed3D then
      begin
        Framed3D := Ctl3D;
        if not Framed3D then
        begin
          if Right < R.Right then Inc(Right);
          if Bottom < R.Bottom then Inc(Bottom);
        end;
        if Framed3D and (Color = clBtnFace) then
        begin
          RBFColor := clBtnShadow;
          LTFColor := clBtnHighlight;
        end
        else begin
          RBFColor := clBtnFace;
          LTFColor := clWindowFrame;
        end;
      end
      else begin
        RBFColor := 0;  { To avoid compiler warning }
        LTFColor := Color;
      end;
      Pen.Color := LTFColor;
      Rectangle(Left, Top, Right, Bottom);
      if Framed3D then
      begin
        Pen.Color := RBFColor;
        Dec(Right);
        Dec(Bottom);
        { Ein guter Compiler sollte die nachfolgende Anweisung
          so wie $IFDEF AUTOSTYLE optimieren. }
        if WindowsStyle >= wsWin40 then
          { Windows 95 Layout }
          Dec(Left);
        PolyLine([Point(Right, Top), Point(Right, Bottom), Point(Left, Bottom)]);
      end;
    end;
  end;

  procedure DrawHorzWindows;
  begin
    with R do
    begin
      if Left < StartX then
        DrawWindow(Left, Top, StartX, Bottom, clBtnShadow)
      else
        StartX := Left;
      if StartX < MinX then
        DrawWindow(StartX, Top, MinX, Bottom, clBtnFace)
      else
        MinX := StartX;
      if EndX < Right then
        DrawWindow(EndX, Top, Right, Bottom, clBtnShadow)
      else
        EndX := Right;
      if MaxX < EndX then
        DrawWindow(MaxX, Top, EndX, Bottom, clBtnFace)
      else
        MaxX := EndX;
      if MinX < MaxX then
        DrawWindow(MinX, Top, MaxX, Bottom, Color);
    end;
  end;

  procedure DrawVertWindows;
  begin
    with R do
    begin
      if Top < StartX then
        DrawWindow(Left, Top, Right, StartX, clBtnShadow)
      else
        StartX := Top;
      if StartX < MinX then
        DrawWindow(Left, StartX, Right, MinX, clBtnFace)
      else
        MinX := StartX;
      if EndX < Bottom then
        DrawWindow(Left, EndX, Right, Bottom, clBtnShadow)
      else
        EndX := Bottom;
      if MaxX < EndX then
        DrawWindow(Left, MaxX, Right, EndX, clBtnFace)
      else
        MaxX := EndX;
      if MinX < MaxX then
        DrawWindow(Left, MinX, Right, MaxX, Color);
    end;
  end;

  procedure DrawHorzMark;
  begin
    with Canvas do
    begin
      if MarkPosition mod FScaleInterval = 0 then
      begin
        MarkText := IntToStr(Abs(MarkPosition div FScaleUnit));
        MarkTextOffsetX := TextWidth(MarkText) div 2;
        if (MarkPositionX - MarkTextOffsetX > StartX)
          and (MarkPositionX + MarkTextOffsetX < EndX) then
        begin
          TextOut(MarkPositionX - MarkTextOffsetX, MarkTextOffsetY, MarkText);
          StartX := MarkPositionX + MarkTextOffsetX;
        end
        else if (MarkPositionX - MarkTextOffsetX > TrueStartX)
          and (MarkPositionX + MarkTextOffsetX < TrueEndX) then
        begin
          TextRect(R, MarkPositionX - MarkTextOffsetX, MarkTextOffsetY,
            MarkText);
          StartX := MarkPositionX + MarkTextOffsetX;
        end;
      end
      else if MarkPosition mod ScaleIntervalMid = 0 then
      begin
        MoveTo(MarkPositionX, MarkPositionY - 2);
        LineTo(MarkPositionX, MarkPositionY + 3);
      end
      else
      begin
        MoveTo(MarkPositionX, MarkPositionY - 1);
        LineTo(MarkPositionX, MarkPositionY + 1);
      end;
    end;
  end;

  procedure DrawVertMark;
  begin
    with Canvas do
    begin
      if MarkPosition mod FScaleInterval = 0 then
      begin
        MarkText := IntToStr(Abs(MarkPosition div FScaleUnit));
        MarkTextOffsetX := (TextWidth(MarkText) + 1) div 2;
        if (MarkPositionX - MarkTextOffsetY > StartX)
          and (MarkPositionX + MarkTextOffsetY < EndX) then
        begin
          TextOut(MarkPositionY - MarkTextOffsetX, MarkPositionX - MarkTextOffsetY,
            MarkText);
          StartX := MarkPositionX + MarkTextOffsetY - FTextLeading;
        end
        else if (MarkPositionX - MarkTextOffsetY > TrueStartX)
          and (MarkPositionX + MarkTextOffsetY < TrueEndX) then
        begin
          TextRect(R, MarkPositionY - MarkTextOffsetX, MarkPositionX - MarkTextOffsetY,
            MarkText);
          StartX := MarkPositionX + MarkTextOffsetY - FTextLeading;
        end;
      end
      else if MarkPosition mod ScaleIntervalMid = 0 then
      begin
        MoveTo(MarkPositionY - 2, MarkPositionX);
        LineTo(MarkPositionY + 3, MarkPositionX);
      end
      else
      begin
        MoveTo(MarkPositionY - 1, MarkPositionX);
        LineTo(MarkPositionY + 1, MarkPositionX);
      end;
    end;
  end;

begin
  StartX := MulDiv(-FBaseOrigin, FTruePixelExtent, FTrueBaseExtent)
    + FPixelOrigin;
  TrueStartX := StartX;
  if FMin = Low(FMin) then
    MinX := Low(MinX)
  else
    MinX := MulDiv(FMin - FBaseOrigin, FTruePixelExtent, FTrueBaseExtent)
      + FPixelOrigin;
  if FMax = High(FMax) then
    MaxX := High(MaxX)
  else
    MaxX := MulDiv(FMax - FBaseOrigin, FTruePixelExtent, FTrueBaseExtent)
      + FPixelOrigin;
  if (FSize < 0) or (FSize = High(FSize)) then
    EndX := High(EndX)
  else
    EndX := MulDiv(FSize - FBaseOrigin, FTruePixelExtent, FTrueBaseExtent)
      + FPixelOrigin;
  TrueEndX := EndX;
  R := ClientRect;
  Canvas.Pen.Mode := pmCopy;
  if FKind = rbHorizontal then
    DrawHorzWindows
  else
    DrawVertWindows;
  if Ctl3D and (csFramed in ControlStyle) then
  begin
    Dec(EndX);
    Dec(TrueEndX);
  end;
  if FKind = rbHorizontal then
  begin
    R.Left := StartX;
    R.Right := EndX;
  end
  else begin
    R.Top := StartX;
    R.Bottom := EndX;
  end;
  with Canvas do
  begin
    Brush.Style := bsClear;
    Font := Self.Font;
    Pen.Color := Self.Font.Color;
    if FKind = rbHorizontal then
    begin
      MarkTextOffsetY := (Height - FTextOLAscent) div 2 - FTextLeading;
      MarkPositionY := Height div 2;
    end
    else begin
      MarkTextOffsetY := FTextOLAscent div 2 + FTextLeading;
      MarkPositionY := Width div 2;
    end;
    ScaleIntervalMid := FScaleInterval div 2;

    ScaleOriginX := MulDiv(FScaleOrigin - FBaseOrigin, FTruePixelExtent,
      FTrueBaseExtent) + FPixelOrigin;
    ScaleRoundDiff := Longint(FScaleOrigin - FBaseOrigin) * FTruePixelExtent
      - Longint(ScaleOriginX - FPixelOrigin) * FTrueBaseExtent;
    ScaleRoundCorrection := MulDiv(ScaleRoundDiff, FTrueScaleExtent,
      FTrueBaseExtent);
    { Erste Teilungsmarke bestimmen }
    MarkPosition := Longint(StartX - ScaleOriginX) * FTrueScaleExtent;
    if MarkPosition >= 0 then
      Inc(MarkPosition, FTrueScalePixelExtent div 2)
    else
      Dec(MarkPosition, FTrueScalePixelExtent div 2);  { Windows: MulDiv ohne +/-1! }
      {Dec(MarkPosition, (FTrueScalePixelExtent + 1) div 2 - 1);}
{   alt.:
    Inc(MarkPosition, FTrueScalePixelExtent div 2)
    if MarkPosition < 0 then
      Dec(MarkPosition, FTrueScalePixelExtent - 1);
}
    MarkPosition := MarkPosition div FTrueScalePixelExtent;
    MarkPosition := (MarkPosition div FScaleDivision) * FScaleDivision;
    {MarkPosition := -FScaleOrigin mod FScaleDivision;
    if MarkPosition <= 0 then
      Inc(MarkPosition, FScaleDivision);
    Inc(MarkPosition, FScaleOrigin);}
    while True do
    begin
      XL := MarkPosition * FTrueScalePixelExtent + ScaleRoundCorrection;
      if XL >= 0 then
        Inc(XL, FTrueScaleExtent div 2)
      else
        Dec(XL, FTrueScaleExtent div 2);  { Windows: MulDiv ohne +/-1! }
        {Dec(XL, (FTrueScaleExtent + 1) div 2 - 1);}
      MarkPositionX := XL div FTrueScaleExtent + ScaleOriginX;
      if MarkPositionX >= EndX then
        Break;
      if MarkPositionX > StartX then
        if FKind = rbHorizontal then
          DrawHorzMark
        else
          DrawVertMark;
      Inc(MarkPosition, FScaleDivision);
    end;
  end;
  Painted;
  UpdateCursorPos(True);
  UpdateMousePos(True);
end;

procedure TRuler.Painted;
begin
  if Assigned(FOnPaint) then FOnPaint(Self);
end;

procedure TRuler.UpdateCursorPos(AShowing : Boolean);
var
  CursorPosX : Integer;
  CursorPosY : Integer;
  RulerBitmapHOffset : Integer;
  RulerBitmapVOffset : Integer;
begin
  if (roShowCursorPos in FOptions) and (FCursorPos >= 0) then
  begin
    CursorPosX := FPixelOrigin + FCursorPos - RulerBitmapHotSpots[tkDefault];
    if FKind = rbHorizontal then
    begin
      CursorPosY := Height - RulerBitmapHeight + 1;
      RulerBitmapHOffset := 0;
    end
    else begin
      CursorPosY := CursorPosX;
      CursorPosX := Width - RulerBitmapWidth + 1;
      RulerBitmapHOffset := RulerBitmapVertHOffset;
    end;
    RulerBitmapVOffset := RulerBitmapNormVOffset;
    with Canvas do
    begin
      if AShowing then
      begin
        FSavedCursorBitmap.Canvas.CopyRect(Bounds(0, 0, RulerBitmapWidth, RulerBitmapHeight),
          Canvas, Bounds(CursorPosX, CursorPosY, RulerBitmapWidth, RulerBitmapHeight));
        CopyMode := cmSrcAnd;
        CopyRect(Bounds(CursorPosX, CursorPosY, RulerBitmapWidth, RulerBitmapHeight),
          FRulerBitmap.Canvas, Bounds(RulerBitmapHOffset, RulerBitmapMaskVOffset,
          RulerBitmapWidth, RulerBitmapHeight));
        CopyMode := cmSrcPaint;
        CopyRect(Bounds(CursorPosX, CursorPosY, RulerBitmapWidth, RulerBitmapHeight),
          FRulerBitmap.Canvas, Bounds(RulerBitmapHOffset, RulerBitmapVOffset,
          RulerBitmapWidth, RulerBitmapHeight));
        CopyMode := cmSrcCopy;
      end
      else
        CopyRect(Bounds(CursorPosX, CursorPosY, RulerBitmapWidth, RulerBitmapHeight),
          FSavedCursorBitmap.Canvas, Bounds(0, 0, RulerBitmapWidth, RulerBitmapHeight));
    end;
  end;
end;

procedure TRuler.UpdateMousePos(AShowing : Boolean);
begin
  if (roShowMousePos in FOptions) and (FMousePos >= 0) then
    with Canvas do
    begin
      Pen.Color := clBtnFace;
      Pen.Mode := pmNotXor;
      if FKind = rbHorizontal then
      begin
        MoveTo(FMousePos, 1);
        LineTo(FMousePos, Height - 1);
      end
      else begin
        MoveTo(1, FMousePos);
        LineTo(Width - 1, FMousePos);
      end;
    end;
end;

procedure TRuler.InsertColumn(AColumn : TRulerColumn);
begin
  if FColumns = nil then
    FColumns := TList.Create;
  FColumns.Add(@AColumn);
end;

procedure TRuler.RemoveColumn(AColumn : TRulerColumn);
begin
  FColumns.Remove(@AColumn);
  if FColumns.Count = 0 then
  begin
    FColumns.Free;
    FColumns := nil;
  end;
end;

function TRuler.GetColumn(Index : Integer) : TRulerColumn;
begin
  Result := TRulerColumn(FColumns.Items[Index]^);
end;

function TRuler.GetColumnCount : Integer;
begin
  Result := FColumns.Count;
end;

function TRuler.GetBasePos(var PixelPos : Integer) : Integer;
var
  Remainder : Integer;
begin
  Result := IMulDiv(PixelPos - FPixelOrigin, FTrueBaseExtent, FTruePixelExtent,
    Remainder) + FBaseOrigin;
  if FBasePrecision > 0 then
  begin
    Result := IMulDiv(Result, 1, FBasePrecision, Remainder) * FBasePrecision;
    PixelPos := IMulDiv(Result - FBaseOrigin, FTruePixelExtent, FTrueBaseExtent,
      Remainder) + FPixelOrigin;
  end;
end;

function TRuler.GetPixelPos(Pos : Integer) : Integer;
var
  Remainder : Integer;
begin
  Result := IMulDiv(Pos - FBaseOrigin, FTruePixelExtent, FTrueBaseExtent,
    Remainder) + FPixelOrigin;
end;

function TRuler.GetScalePos(var PixelPos : Integer) : Integer;
var
  ScaleOriginX : Integer;
  Remainder : Integer;
begin
  ScaleOriginX := IMulDiv(FScaleOrigin - FBaseOrigin, FTruePixelExtent,
    FTrueBaseExtent, Remainder) + FPixelOrigin;
  Result := IMulDiv(PixelPos - ScaleOriginX, FTrueScaleExtent,
    FTrueScalePixelExtent, Remainder);
  if FScalePrecision > 0 then
  begin
    Result := IMulDiv(Result, 1, FScalePrecision, Remainder) * FScalePrecision;
    PixelPos := IMulDiv(Result, FTrueScalePixelExtent, FTrueScaleExtent,
      Remainder) + ScaleOriginX;
  end;
end;

function TRuler.GetScalePixelPos(Pos : Integer) : Integer;
var
  ScaleOriginX : Integer;
  Remainder : Integer;
begin
  ScaleOriginX := IMulDiv(FScaleOrigin - FBaseOrigin, FTruePixelExtent,
    FTrueBaseExtent, Remainder) + FPixelOrigin;
  Result := IMulDiv(Pos, FTrueScalePixelExtent, FTrueScaleExtent, Remainder)
    + ScaleOriginX;
end;

procedure TRuler.SetBaseExtent(Value : Integer);
begin
  if FBaseExtent <> Value then
  begin
    FBaseExtent := Value;
    if Value = beText then
      FTrueBaseExtent := FTruePixelExtent
    else
      FTrueBaseExtent := Value;
    FTextPixelsValid := False;
    Invalidate;
  end;
end;

procedure TRuler.SetBaseOrigin(Value : Integer);
begin
  if FBaseOrigin <> Value then
  begin
    FBaseOrigin := Value;
    FTextPixelsValid := False;
    Invalidate;
  end;
end;

function TRuler.GetBorderStyle : TBorderStyle;
begin
  if csFramed in ControlStyle then
    Result := bsSingle
  else
    Result := bsNone;
end;

procedure TRuler.SetBorderStyle(Value : TBorderStyle);
begin
  if BorderStyle <> Value then
  begin
    if Value = bsNone then
      ControlStyle := ControlStyle - [csFramed]
    else
      ControlStyle := ControlStyle + [csFramed];
    {RecreateWnd;}
    Invalidate;
  end;
end;

procedure TRuler.SetCursorPos(Value : Integer);
begin
  if FCursorPos <> Value then
  begin
    UpdateMousePos(False);
    UpdateCursorPos(False);
    FCursorPos := Value;
    UpdateCursorPos(True);
    UpdateMousePos(True);
  end;
end;

procedure TRuler.SetKind(Value : TRulerKind);
begin
  if FKind <> Value then
  begin
    FKind := Value;
    if not (csLoading in ComponentState) then
      SetBounds(Left, Top, Height, Width);
    Invalidate;
  end;
end;

procedure TRuler.SetMax(Value : Integer);
begin
  if FMax <> Value then
  begin
    FMax := Value;
    Invalidate;
  end;
end;

procedure TRuler.SetMin(Value : Integer);
begin
  if FMin <> Value then
  begin
    FMin := Value;
    FTextPixelsValid := False;
    Invalidate;
  end;
end;

procedure TRuler.SetMousePos(Value : Integer);
begin
  if FMousePos <> Value then
  begin
    UpdateMousePos(False);
    FMousePos := Value;
    UpdateMousePos(True);
  end;
end;

procedure TRuler.SetOptions(Value : TRulerOptions);
const
  LayoutOptions = [roShowMousePos, roShowCursorPos, roShowTabMarks];
var
  ChangedOptions : TRulerOptions;
begin
  if FOptions <> Value then
  begin
    ChangedOptions := (FOptions + Value) - (FOptions * Value);
    FOptions := Value;
    if ChangedOptions * LayoutOptions <> [] then
      Invalidate;
  end;
end;

procedure TRuler.SetPixelExtent(Value : Integer);
begin
  if FPixelExtent <> Value then
  begin
    FPixelExtent := Value;
    if Value = 0 then
      AdjustFontMetrics
    else begin
      FTruePixelExtent := Value;
      if FBaseExtent = beText then
        FTrueBaseExtent := Value;
    end;
    FTextPixelsValid := False;
    Invalidate;
  end;
end;

procedure TRuler.SetPixelOrigin(Value : Integer);
begin
  if FPixelOrigin <> Value then
  begin
    FPixelOrigin := Value;
    FTextPixelsValid := False;
    Invalidate;
  end;
end;

procedure TRuler.SetScaleDivision(Value : Integer);
begin
  if (Value > 0) and (FScaleDivision <> Value) then
  begin
    FScaleDivision := Value;
    Invalidate;
  end;
end;

procedure TRuler.SetScaleExtent(Value : Integer);
begin
  if FScaleExtent <> Value then
  begin
    FScaleExtent := Value;
    if Value = beText then
      FTrueScaleExtent := FTrueScalePixelExtent
    else
      FTrueScaleExtent := Value;
    Invalidate;
  end;
end;

procedure TRuler.SetScaleInterval(Value : Integer);
begin
  if (Value > 0) and (FScaleInterval <> Value) then
  begin
    FScaleInterval := Value;
    Invalidate;
  end;
end;

procedure TRuler.SetScaleOrigin(Value : Integer);
begin
  if FScaleOrigin <> Value then
  begin
    FScaleOrigin := Value;
    Invalidate;
  end;
end;

procedure TRuler.SetScalePixelExtent(Value : Integer);
begin
  if FScalePixelExtent <> Value then
  begin
    FScalePixelExtent := Value;
    if Value = 0 then
      AdjustFontMetrics
    else begin
      FTrueScalePixelExtent := Value;
      if FScaleExtent = beText then
        FTrueScaleExtent := Value;
    end;
    Invalidate;
  end;
end;

procedure TRuler.SetScaleUnit(Value : Integer);
begin
  if (Value > 0) and (FScaleUnit <> Value) then
  begin
    FScaleUnit := Value;
    Invalidate;
  end;
end;

procedure TRuler.SetSize(Value : Integer);
begin
  if FSize <> Value then
  begin
    FSize := Value;
    Invalidate;
  end;
end;

function TRuler.GetTransparent : Boolean;
begin
  Result := not (csOpaque in ControlStyle);
end;

procedure TRuler.SetTransparent(Value : Boolean);
begin
  if Transparent <> Value then
  begin
    if Value then
      ControlStyle := ControlStyle - [csOpaque]
    else
      ControlStyle := ControlStyle + [csOpaque];
    RecreateWnd;
  end;
end;


{ Miscellaneous functions }

{$S-}
function IMulDiv(Number : Integer; Numerator : Integer; Denominator : Integer;
  var Remainder : Integer) : Integer;
begin
  Result := IMulDivEx(Number, Numerator, 0, Denominator, Remainder);
end;

function IMulDivEx(Number : Integer; Numerator : Integer; Correction : Integer;
  Denominator : Integer; var Remainder : Integer) : Integer; assembler;
{ Remainder of -(MaxInt+1) indicates overflow }
{ Overflow indication occurs wrong for results of -(MaxInt+1) }
asm
{$IFDEF WIN32}
  push    ebx
  {mov     eax,Number}  {eax}
  imul    Numerator  {edx}
  {mov     ecx,Correction}  {ecx}
  add     eax,ecx
  adc     edx,0
  or      ecx,ecx
  jns     @@0
  dec     edx
@@0:
  mov     ecx,Denominator { Sets ebx:ecx to half of denominator ... }
  mov     ebx,edx         { Sets ebx:ecx sign equal to product sign ... }
  xor     ebx,ecx
  mov     ebx,0
  jns     @@1
  cmp     ecx,$80000000   { Checks for asymmetrical -(MaxInt+1) denominator }
  je      @@2             { If equal then carry flag isn't set }
  neg     ecx
@@1:
  or      ecx,ecx         { Clears carry flag }
  jns     @@2
  inc     ecx             { Doesn't affect carry flag }
  jns     @@2
  sub     ebx,1           { Sets carry flag }
@@2:
  rcr     ecx,1           { Carry flag is used to support +(MaxInt+1) }
  add     eax,ecx
  adc     edx,ebx
  js      @@4
  cmp     edx,ecx         { Tests for overflow of positive or zero product ... }
  jl      @@5
  jg      @@3
  or      eax,eax
  js      @@3
  test    Denominator,1
  jnz     @@5
@@3:
  xor     edx,Denominator { Sets overflow result ... }
  mov     eax,$80000000
  mov     edx,eax
  js      @@6
  dec     eax
  jmp     @@6
@@4:
  mov     ebx,edx         { Tests for overflow of negative product ... }
  inc     ebx
  cmp     ebx,ecx
  jg      @@5
  jl      @@3
  cmp     eax,$80000000
  jbe     @@3
  test    Denominator,1
  jz      @@3
@@5:
  idiv    Denominator
  sub     edx,ecx
@@6:
  mov     ecx,Remainder
  mov     [ecx],edx
  pop     ebx
{$ELSE}
  mov     ax,Number
  imul    Numerator
  mov     cx,Correction
  add     ax,cx
  adc     dx,0
  or      cx,cx
  jns     @@0
  dec     dx
@@0:
  mov     cx,Denominator  { Sets bx:cx to half of denominator ... }
  mov     bx,dx           { Sets bx:cx sign equal to product sign ... }
  xor     bx,cx
  mov     bx,0
  jns     @@1
  cmp     cx,$8000        { Checks for asymmetrical -(MaxInt+1) denominator }
  je      @@2             { If equal then carry flag isn't set }
  neg     cx
@@1:
  or      cx,cx           { Clears carry flag }
  jns     @@2
  inc     cx              { Doesn't affect carry flag }
  jns     @@2
  sub     bx,1            { Sets carry flag }
@@2:
  rcr     cx,1            { Carry flag is used to support +(MaxInt+1) }
  add     ax,cx
  adc     dx,bx
  js      @@4
  cmp     dx,cx           { Tests for overflow of positive or zero product ... }
  jl      @@5
  jg      @@3
  or      ax,ax
  js      @@3
  test    Denominator,1
  jnz     @@5
@@3:
  xor     dx,Denominator  { Sets overflow result ... }
  mov     ax,$8000
  mov     dx,ax
  js      @@6
  dec     ax
  jmp     @@6
@@4:
  mov     bx,dx           { Tests for overflow of negative product ... }
  inc     bx
  cmp     bx,cx
  jg      @@5
  jl      @@3
  cmp     ax,$8000
  jbe     @@3
  test    Denominator,1
  jz      @@3
@@5:
  idiv    Denominator
  sub     dx,cx
@@6:
  les     di,Remainder
  mov     es:[di],dx
{$ENDIF}
end;

(*function IMulDivAdjust(Numerator1 : Integer; Denominator1 : Integer;
  Numerator2 : Integer; Denominator2 : Integer) : Integer; assembler;
asm
{$IFDEF WIN32}
  push    ebx
  pop     ebx
{$ELSE}
  mov     ax,Numerator1
  imul    Denominator2
  mov     cx,ax
  mov     bx,dx
  mov     ax,Numerator2
  imul    Denominator1
  add     cx,ax
  adc     bx,dx
  shl     cx,1
  rcl     bx,1
  mov     ax,Denominator1
  imul    Denominator2
  push    bx
  xor     bx,dx
  jns     @@1
  not     dx
  neg     ax
  sbb     dx,-1
@@1:
  pop     bx
  push    bx
  sub     cx,ax
  sbb     bx,dx
  or      cx,bx
  jnz     @@3
  pop     bx
  push    bx
  {xor     bx,VZ}
@@3:
  xor     ax,ax
  xor     bx,dx
  pop     bx
  js      @@2
  inc     ax
  xor     bx,dx
  jns     @@2
  neg     ax
@@2:
{$ENDIF}
end;*)
{$S+}

{ Registrierung }

procedure Register;
begin
  RegisterComponents('Samples', [TRuler, TTabRuler]);
end;

{$IFDEF AUTOSTYLE}
initialization
  if WindowsStyle = wsAutoDetect then
    if NewStyleControls then
      WindowsStyle := wsWin40
    else
      WindowsStyle := wsWin31;
{$ENDIF}

{$IFDEF WIN32}
finalization
  FreeBitmaps;
{$ENDIF}
end.
