unit elReportsXDOImpl;

interface

uses
  SysUtils, ComObj, ComServ, ActiveX, Classes, elReportsXDO_TLB, elReportsEUI_TLB, elReports_TLB;

var
  elRecordSetsFactory,
  elRecordSetFactory,
  elRecordFactory,
  elFieldFactory,
  elReportXDOCallbackFactory: TAutoObjectFactory;

procedure Register(ComServer: TComServer);
procedure Unregister(ComServer: TComServer);

implementation

type
  TelRecordSets = class(TAutoObject
  , IelRecordSets
  )
  private
    fRecordSetNames: TStringList;
    fRecordSetObjects: TInterfaceList;
  protected
  public
    procedure AddRecordSet(const Name: WideString; const RecordSet: IelRecordSet); safecall;
    function  FindRecordSet(const Name: WideString): IelRecordSet; safecall;
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
  end;

  TelRecordSet = class(TAutoObject
  , IelRecordSet
  )
  private
    fRecords: TInterfaceList;
    fCursor: Integer;
  protected
    procedure AddRecord(const Record_: IelRecord); safecall;
    function  Get_Cursor: IelRecord; safecall;
    function  Get_RecordCount: Integer; safecall;
    function  Get_EOF: WordBool; safecall;
    procedure MoveNext; safecall;
  public
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
  end;

  TelRecord = class(TAutoObject
  , IelRecord
  )
  private
    fFields: TInterfaceList;
  protected
  public
    procedure AddField(const Field: IelField); safecall;
    function  FindField(const Name: WideString): IelField; safecall;
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
  end;

  TelField = class(TAutoObject
  , IelField
  )
  private
    fName,
    fValue: String;
    fRecordSet: IelRecordSet;
  protected
    function Get_Name: WideString; safecall;
    procedure Set_Name(const Value: WideString); safecall;
    function Get_Value: WideString; safecall;
    procedure Set_Value(const Value: WideString); safecall;
    function Get_RecordSet: IelRecordSet; safecall;
    procedure Set_RecordSet(const Value: IelRecordSet); safecall;
  public
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
    property Name: WideString read Get_Name write Set_Name;
    property Value: WideString read Get_Value write Set_Value;
    property RecordSet: IelRecordSet read Get_RecordSet write Set_RecordSet;
  end;

  TelReportXDOCallback = class(TAutoObject
  , IelReportCallback
  , IelReportXDOCallback
  )
  private
    fRecordsetNames: TStringList;
    fRecordsetObjects: TInterfaceList;
  protected
    procedure Generate(const Writer: IelReportWriter; const Value: WideString); safecall;
    function  FindRecordSet(const Name: WideString): IelRecordSet; safecall;
    procedure AddRecordset(const Name: WideString; const Recordset: IelRecordSet); safecall;
    procedure AddRecordsets(const RecordSets: IelRecordSets); safecall;
  public
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
  end;

{
uses
  StrUtilz;

}
function CopyAfter(const SubStr, Str: string): string;
var
  _: Integer;
begin
  _ := Pos(SubStr,Str);
  if _=0 then
    Result := ''
  else
    Result := Copy(Str,_+Length(SubStr),Length(Str));
end;

function CopyBefore(const SubStr,Str:string): string;
var
  _: Integer;
begin
  _ := Pos(SubStr,Str);
  if _=0 then
    Result := Str
  else
    Result := Copy(Str,1,_-1);
end;

procedure TelField.Set_Name(const Value: WideString);
begin
  FName := Value;
end;

function TelField.Get_Value: WideString;
begin
  Result := FValue;
end;

procedure TelField.Set_Value(const Value: WideString);
begin
  FValue := Value;
end;

function TelField.Get_RecordSet: IelRecordSet;
begin
  Result := FRecordSet;
end;

procedure TelField.Set_RecordSet(const Value: IelRecordSet);
begin
  FRecordSet := Value;
end;

procedure TelField.AfterConstruction;
begin
  inherited;
end;

procedure TelField.BeforeDestruction;
begin
  inherited;
end;

function TelField.Get_Name: WideString;
begin
  Result := FName;
end;

procedure TelRecord.AddField(const Field: IelField);
begin
  fFields.Add(Field);
end;

function TelRecord.FindField(const Name: WideString): IelField;
var
  i: Integer;
begin
  for i := 0 to fFields.Count - 1 do begin
    Result := IelField(fFields.Items[i]);
    if CompareText(Result.Name, Name) = 0 then
      Exit;
  end;
  Result := nil;
end;

procedure TelRecord.AfterConstruction;
begin
  inherited;
  fFields := TInterfaceList.Create;
end;

procedure TelRecord.BeforeDestruction;
begin
  fFields.Free;
  inherited;
end;

procedure TelRecordSet.AddRecord(const Record_: IelRecord);
begin
  fRecords.Add(Record_);
end;

function TelRecordSet.Get_Cursor: IelRecord;
begin
  if fCursor >= fRecords.Count then
    Result := nil
  else
    Result := IelRecord(fRecords.Items[fCursor]);
end;

function TelRecordSet.Get_RecordCount: Integer;
begin
  Result := fRecords.Count;
end;

function TelRecordSet.Get_EOF: WordBool;
begin
  Result := fCursor >= fRecords.Count;
end;

procedure TelRecordSet.MoveNext;
begin
  Inc(fCursor);
end;

procedure TelRecordSets.AddRecordSet(const Name: WideString; const RecordSet: IelRecordSet);
begin
  fRecordSetNames.Add(Name);
  fRecordSetObjects.Add(RecordSet);
end;

function TelRecordSets.FindRecordSet(const Name: WideString): IelRecordSet;
var
  i: Integer;
begin
  i := fRecordSetNames.IndexOf(Name);
  if i <> -1 then begin
    Result := IelRecordSet(fRecordSetObjects.Items[i]);
    Exit;
  end;
  Result := nil;
end;

procedure TelRecordSets.AfterConstruction;
begin
  inherited;
  fRecordSetNames := TStringList.Create;
  fRecordSetObjects := TInterfaceList.Create;
end;

procedure TelRecordSets.BeforeDestruction;
begin
  fRecordSetNames.Free;
  fRecordSetObjects.Free;
  inherited;
end;

procedure TelRecordSet.AfterConstruction;
begin
  inherited;
  fRecords := TInterfaceList.Create;
end;

procedure TelRecordSet.BeforeDestruction;
begin
  fRecords.Free;
  inherited;
end;

type
  EelReportXDOError = class(Exception);

procedure TelReportXDOCallback.Generate(const Writer: IelReportWriter; const Value: WideString);
var
  PrefixValue,
  SuffixValue,
  TableName,
  FieldName: String;
  RecordSet: IelRecordSet;
  Record_: IelRecord;
  Field: IelField;
begin
  SuffixValue := Value;
  repeat
    TableName := Trim(CopyBefore('.', SuffixValue));
    SuffixValue := Trim(CopyAfter('.', SuffixValue));
    if PrefixValue = '' then
      PrefixValue := TableName
    else
      PrefixValue := PrefixValue + '.' + TableName;
    if Assigned(RecordSet) then begin
      Record_ := RecordSet.Cursor;
      Field := Record_.FindField(TableName);
      if not Assigned(Field) then
        raise EelReportXDOError.CreateFmt('RecordSet %s not found', [PrefixValue]);
      RecordSet := Field.RecordSet;
    end
    else begin
      RecordSet := FindRecordSet(TableName);
      if not Assigned(RecordSet) then
        raise EelReportXDOError.CreateFmt('RecordSet %s not found', [PrefixValue]);
    end;
  until Pos('.', SuffixValue) = 0;

  FieldName := SuffixValue;
  if CompareText(FieldName, '#begin#') = 0 then begin
    if (RecordSet.RecordCount = 0) and
      Assigned(Writer.Report) and
      Assigned(Writer.Report.Collector) then
      Writer.Report.Collector.Dissapear := Writer.Report.Collector.Dissapear + 1;
    Writer.BeginRepeater(PrefixValue)
  end
  else if CompareText(FieldName, '#end#') = 0 then begin
    if (RecordSet.RecordCount = 0) and
      Assigned(Writer.Report) and
      Assigned(Writer.Report.Collector) then
      Writer.Report.Collector.Dissapear := Writer.Report.Collector.Dissapear - 1;
    RecordSet.MoveNext;
    Writer.RepeatNeeded := Writer.RepeatNeeded or not RecordSet.EOF;
    Writer.EndRepeater(PrefixValue);
  end
  else if not RecordSet.EOF then begin
    Record_ := RecordSet.Cursor;
    Field := Record_.FindField(FieldName);
    if not Assigned(Field) then
      raise EelReportXDOError.CreateFmt('Field %s not found', [Value]);
    Writer.WriteString(Field.Value);
  end;
end;

function TelReportXDOCallback.FindRecordSet(const Name: WideString): IelRecordSet;
var
  i: Integer;
begin
  i := fRecordSetNames.IndexOf(Name);
  if i <> -1 then begin
    Result := IelRecordSet(fRecordSetObjects.Items[i]);
    Exit;
  end;
  Result := nil;
end;


procedure TelReportXDOCallback.AddRecordset(const Name: WideString; const Recordset: IelRecordSet);
begin
  fRecordsetNames.Add(Name);
  fRecordsetObjects.Add(RecordSet);
end;

procedure TelReportXDOCallback.AddRecordsets(const RecordSets: IelRecordSets);
begin
  Assert(false);
end;

procedure TelReportXDOCallback.AfterConstruction;
begin
  inherited;
  fRecordsetNames := TStringList.Create;
  fRecordsetObjects := TInterfaceList.Create;
end;

procedure TelReportXDOCallback.BeforeDestruction;
begin
  fRecordsetNames.Free;
  fRecordsetObjects.Free;
  inherited;
end;

procedure Register(ComServer: TComServer);
begin
  elRecordSetsFactory := TAutoObjectFactory.Create(ComServer, TelRecordSets, Class_elRecordSets,
    ciMultiInstance, tmApartment);
  elRecordSetFactory := TAutoObjectFactory.Create(ComServer, TelRecordSet, Class_elRecordSet,
    ciMultiInstance, tmApartment);
  elRecordFactory := TAutoObjectFactory.Create(ComServer, TelRecord, Class_elRecord,
    ciMultiInstance, tmApartment);
  elFieldFactory := TAutoObjectFactory.Create(ComServer, TelField, Class_elField,
    ciMultiInstance, tmApartment);

  elReportXDOCallbackFactory := TAutoObjectFactory.Create(ComServer, TelReportXDOCallback, Class_elReportXDOCallback,
    ciMultiInstance, tmApartment);
end;

procedure Unregister(ComServer: TComServer);
begin
end;

initialization
{$IFNDEF elReports777}
  Register(ComServer);
{$ENDIF}

end.
