unit DsFramedImage;

// DsFramedImage Release 1.1 for Delphi 3/4
// Copyright (c) 1999-2000 by Djoko Susilo
// --------- Q14 Project Nov-99 --------------
// e-mail: djokos@cabi.net.id
// http://homepage.indo1.com/delphianhome/
// -------------------------------------------
// Files needed to install:
// DsFramedImage.pas (this file)
// DsFramedImage.dcr (resource for icon)
// -------------------------------------------
{
DsFramedImage is an enhancement of TImage so it capable to display image
in a 3D frame. This effect will greatly improve your image appearance.
Its frame is designed using gradient effect. The basic frame effects
available are: full gradient, inner gradient, outer gradient, simple 3D
and their combinations, including rounded corner.
See the sample picture included.

Note: Unlike with TImage, in this component the image is always centered.
      So, no property Center!.
}
// -----------------------------------------------------------------
// Update 06-May-00 R1.1 fix some known bugs:
// - now image always updated when loaded
// - fixed zero division error when image is cleared
// - fixed image blink in design time when property Align is set
//   other than alNone
// - fixed corner radius when FrameStyle=fsFlatGrad
// - added border to identify current position
// - fixed region deletion in GradientOuter, GradientInner and paint
//
// Known Issues:
// When AutoSize=True and Align=alClient, changing to other Align
// value does not change the component size. Just click the image,
// the component will resize correctly.
//
// *** Thanks to Pavel Kilovatiy for bugs report and suggestions ***
// -----------------------------------------------------------------

interface

uses
  Classes, Graphics, Controls, Windows, Messages;

type
  TFrameCorner = (fcSquare, fcRound);
  TFrameStyle = (fsNone, fsGradient, fsFlat, fsGradFlat, fsFlatGrad,
                 fsInGrad, fsOutGrad);
  TDsFramedImage = class(TGraphicControl)
  private
    FPicture: TPicture;
    FColor_High: TColor;
    FColor_Low: TColor;
    FFrameColor: TColor;
    FFrameCorner: TFrameCorner;
    FFrameStyle: TFrameStyle;
    FFrameWidth: Integer;
    FCornerRadius: Integer;
    FShadowColor: TColor;
    F, RgnF: HRgn;   {--region--}
    FActualSize: Boolean;
    Procedure WMSize (var message: TWMSize); message WM_Paint;
  protected
    procedure Paint; override;
    procedure SetPicture(Value: TPicture);
    procedure SetActualSize(Value: Boolean);
    procedure SetColor_High(Value: TColor);
    procedure SetColor_Low(Value: TColor);
    procedure SetCornerRadius(Value: Integer);
    procedure SetFrameColor(Value: TColor);
    procedure SetFrameCorner(Value: TFrameCorner);
    procedure SetFrameStyle(Value: TFrameStyle);
    procedure SetFrameWidth(Value: Integer);
    procedure SetShadowColor(Value: TColor);
    procedure GradientOuter(Rd: Integer);
    procedure GradientInner(Rd: Integer);
    procedure DrawPicture;
    procedure UpdatePicture(Sender: TObject);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Canvas; {--allow someone to draw the surface directly--}
  published
    {--redeclare property and event needed to support its functionality--}
    property Align;
    property ActualSize: Boolean read FActualSize write SetActualSize;
    property Picture: TPicture read FPicture write SetPicture;
    property Color_High: TColor
             read FColor_High write SetColor_High default $00E0E8E8;
    property Color_Low: TColor
             read FColor_Low write SetColor_Low default $00688DA2;
    property Enabled;
    property FrameColor: TColor
             read FFrameColor write SetFrameColor default clBtnFace;
    property FrameCorner: TFrameCorner
             read FFrameCorner write SetFrameCorner default fcRound;
    property FrameStyle: TFrameStyle
             read FFrameStyle write SetFrameStyle default fsGradient;
    property FrameWidth: Integer
             read FFrameWidth write SetFrameWidth default 8;
    property CornerRadius: Integer
             read FCornerRadius write SetCornerRadius default 10;
    property ShadowColor: TColor
             read FShadowColor write SetShadowColor default clGray;
    property Visible;
    {--choose and add other events if necessary--}
    property OnClick;
  end;

procedure Register;

implementation

var PicRect: TRect; {--picture area and position--}
    x0, x1, y0, y1: Integer;  {--frame area--}

constructor TDsFramedImage.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FColor_High:=$00E0E8E8;
  FColor_Low:=$00688DA2;
  FCornerRadius:=10;
  FFrameColor:=clBtnFace;
  FFrameCorner:=fcRound;
  FFrameStyle:=fsGradient;
  FFrameWidth:=8;
  FShadowColor:=clGray;
  FPicture:=TPicture.Create;
  FPicture.OnChange:=UpdatePicture;
  FActualSize:=False;
  Height:=89;
  Width:=89;
end;

destructor TDsFramedImage.Destroy;
begin
  FPicture.Free;
  inherited Destroy;
end;

procedure TDsFramedImage.UpdatePicture(Sender: TObject);
begin
  Invalidate;
end;

procedure TDsFramedImage.GradientOuter(Rd: Integer);
var
  FC, Warna: TColor;
  R, G, B: Byte;
  AwalR, AwalG, AwalB,
  AkhirR, AkhirG, AkhirB,
  y, t: Integer;
begin
  if (FFrameStyle=fsInGrad) or (FFrameStyle=fsOutGrad) then t:=FFrameWidth
  else if FFrameWidth mod 2=0 then t:=FFrameWidth div 2
  else t:=(FFrameWidth+1) div 2;
  Warna:=ColorToRGB(FFrameColor);
  FC:=ColorToRGB(FShadowColor);

  AwalR:=GetRValue(FC); AkhirR:=GetRValue(Warna);
  AwalG:=GetGValue(FC); AkhirG:=GetGValue(Warna);
  AwalB:=GetBValue(FC); AkhirB:=GetBValue(Warna);

  for y:=0 to t-1 do
  begin
    R:=AwalR+Round(Sqrt(t*t-Sqr(t-y))*(AkhirR-AwalR)/t);
    G:=AwalG+Round(Sqrt(t*t-Sqr(t-y))*(AkhirG-AwalG)/t);
    B:=AwalB+Round(Sqrt(t*t-Sqr(t-y))*(AkhirB-AwalB)/t);
    Canvas.Brush.Color:=RGB(R, G, B);
    if (Rd-y)>0 then
      RgnF:=CreateRoundRectRgn(x0+y, y0+y, x1+1-y, y1+1-y, 2*(Rd-y), 2*(Rd-y))
    else RgnF:=CreateRectRgn(x0+y, y0+y, x1-y, y1-y);
    try
      FrameRgn(Canvas.Handle, RgnF, Canvas.Brush.Handle, 1, 1);
    finally
      DeleteObject(RgnF);
    end;    
  end;
end;

procedure TDsFramedImage.GradientInner(Rd: Integer);
var
  FC, Warna: TColor;
  R, G, B: Byte;
  AwalR, AwalG, AwalB,
  AkhirR, AkhirG, AkhirB,
  y, t, n: Integer;
begin
  if (FFrameStyle=fsInGrad) or (FFrameStyle=fsOutGrad) then
    if FFrameWidth mod 2=0 then t:=FFrameWidth else t:=FFrameWidth+1
  else if FFrameWidth mod 2=0 then t:=FFrameWidth div 2
  else t:=(FFrameWidth+1) div 2;

  if FFrameStyle=fsInGrad then n:=0 else n:=t;

  Warna:=ColorToRGB(FFrameColor);
  FC:=ColorToRGB(FShadowColor);

  AwalR:=GetRValue(FC); AkhirR:=GetRValue(Warna);
  AwalG:=GetGValue(FC); AkhirG:=GetGValue(Warna);
  AwalB:=GetBValue(FC); AkhirB:=GetBValue(Warna);

  for y:=t downto 0 do
  begin
    R:=AwalR+Round(Sqrt(t*t-y*y)*(AkhirR-AwalR)/t);
    G:=AwalG+Round(Sqrt(t*t-y*y)*(AkhirG-AwalG)/t);
    B:=AwalB+Round(Sqrt(t*t-y*y)*(AkhirB-AwalB)/t);
    Canvas.Brush.Color:=RGB(R, G, B);
    if Rd-t-y>0 then
      RgnF:=CreateRoundRectRgn(x0+n+y-1, y0+n+y-1, x1-n-y+2, y1-n-y+2,
                               2*(Rd-y-t), 2*(Rd-y-t))
    else RgnF:=CreateRectRgn(x0+n+y-1, y0+n+y-1, x1-n-y+1, y1-n-y+1);
    try
      FrameRgn(Canvas.Handle, RgnF, Canvas.Brush.Handle, 1, 1);
    finally
      DeleteObject(RgnF);
    end;
  end;
end;

procedure TDsFramedImage.Paint;
var Rd, t: Integer;
begin
  {--check frame width and corner radius--}
  t:=FFrameWidth;
  if t mod 2<>0 then Inc(t);
  Rd:=2*FCornerRadius;

  {--display picture--}
  DrawPicture;

  {--draw component border to identify current position--}
  if csDesigning in ComponentState then
  begin
    Canvas.Brush.Style:=bsClear;
    Canvas.Pen.Color:=clBtnShadow;
    Canvas.Pen.Style:=psDash;
    Canvas.Rectangle(0, 0, Width, Height);
  end;

  {--create and fill frame area--}
  if FFrameStyle=fsNone then Exit;
  if Picture.Graphic=nil then
  begin
    x0:=0;      y0:=0;
    x1:=Width;  y1:=Height;
  end
  else if FActualSize then
  case Align of
    alLeft, alRight:
      begin
        x0:=0;      y0:=PicRect.Top-t;
        x1:=Width;  y1:=PicRect.Bottom+t;
      end;
    alTop, alBottom:
      begin
        x0:=PicRect.Left-t;  y0:=0;
        x1:=PicRect.Right+t; y1:=Height;
      end;
    alNone:
      begin
        x0:=0;      y0:=0;
        x1:=Width;  y1:=Height;
      end;
    alClient:
      begin
        x0:=PicRect.Left-t;
        x1:=PicRect.Right+t;
        y0:=PicRect.Top-t;
        y1:=PicRect.Bottom+t;
      end;
  end
  else
  begin
    x0:=PicRect.Left-t;   y0:=PicRect.Top-t;
    x1:=PicRect.Right+t;  y1:=PicRect.Bottom+t;
  end;

  if (FFrameStyle<>fsInGrad) and (FFrameStyle<>fsOutGrad) then
  begin
    {--outer region--}
    if (FFrameCorner=fcSquare) or (FCornerRadius=0) then
      F:=CreateRectRgn(x0, y0, x1-2, y1-2)
    else F:=CreateRoundRectRgn(x0, y0, x1-1, y1-1, Rd, Rd);
    {--inner region--}
    if (Rd-2*t)<1 then RgnF:=CreateRectRgn(x0+t-2, y0+t-2, x1-t, y1-t)
    else RgnF:=CreateRoundRectRgn(x0+t-2, y0+t-2, x1-t+1, y1-t+1, Rd-2*t, Rd-2*t);
    try
      CombineRgn(RgnF, RgnF, F, RGN_XOR);
      {--make 3D effect--}
      Canvas.Brush.Color:=FColor_High;
      FillRgn(Canvas.Handle, RgnF, Canvas.Brush.Handle);
      OffsetRgn(RgnF, 2, 2);
      Canvas.Brush.Color:=FColor_Low;
      FillRgn(Canvas.Handle, RgnF, Canvas.Brush.Handle);
      OffsetRgn(RgnF, -1, -1);
      Canvas.Brush.Color:=FFrameColor;
      FillRgn(Canvas.Handle, RgnF, Canvas.Brush.Handle);
    finally
      {--freed the memory used by regions--}
      DeleteObject(F);
      DeleteObject(RgnF);
    end;
  end;
                     
  case FFrameStyle of
    fsGradient:
       begin
         if FFrameCorner=fcRound then GradientOuter(FCornerRadius)
         else GradientOuter(0);
         GradientInner(FCornerRadius);
       end;
    fsGradFlat,  {--outer: gradient, inner: flat--}
    fsOutGrad:   {--outer: gradient, inner: none--}
       begin
         if FFrameCorner=fcRound then GradientOuter(FCornerRadius)
         else GradientOuter(0);
       end;
    fsFlatGrad:  {--outer: flat, inner: gradient--}
       begin
         GradientInner(FCornerRadius);
       end;
    fsInGrad:    {--outer: none, inner: gradient--}
       begin
         if FFrameCorner=fcSquare then GradientInner(0)
         else GradientInner(FCornerRadius+t);
       end;
  end; {case}
end;

procedure TDsFramedImage.DrawPicture;
var x, y: Integer;
    PicX, PicY: Integer;
    x1, y1: real;
    t: Integer;

begin
  t:=FFrameWidth;
  if t mod 2<>0 then Inc(t);
  if FFrameStyle=fsNone then t:=0;

  if Picture.Graphic<>nil then
  begin
    if FActualSize then
    begin
      case Align of {--check align value--}
        alTop, alBottom:
          begin
            Height:=Picture.Graphic.Height+2*t;
            PicX:=(Width-Picture.Graphic.Width) div 2;
            PicY:=t;
            PicRect.Top:=0+t;
            PicRect.Bottom:=Height-t;
            PicRect.Left:=PicX;
            PicRect.Right:=PicRect.Left+Picture.Graphic.Width;
            //Canvas.Draw(PicX, PicY, Picture.Graphic);
         end;
        alLeft, alRight:
          begin
            Width:=Picture.Graphic.Width+2*t;
            PicX:=t;
            PicY:=(Height-Picture.Graphic.Height) div 2;
            PicRect.Top:=PicY;
            PicRect.Bottom:=PicRect.Top+Picture.Graphic.Height;
            PicRect.Left:=0+t;
            PicRect.Right:=Width-t;
            //Canvas.Draw(PicX, PicY, Picture.Graphic);
          end;
        alClient:
          begin
            PicX:=(Width-Picture.Graphic.Width) div 2;
            PicY:=(Height-Picture.Graphic.Height) div 2;
            PicRect.Left:=PicX;
            PicRect.Right:=PicRect.Left+Picture.Graphic.Width;
            PicRect.Top:=PicY;
            PicRect.Bottom:=PicRect.Top+Picture.Graphic.Height;
            //Canvas.Draw(PicX, PicY, Picture.Graphic);
          end;
        alNone:
          begin
            Width:=Picture.Graphic.Width+2*t;
            Height:=Picture.Graphic.Height+2*t;
            PicX:=t;
            PicY:=t;
            //Canvas.Draw(PicX, PicY, Picture.Graphic);
          end;
      end;
      Canvas.Draw(PicX, PicY, Picture.Graphic);
    end
    else
    begin
      x1:=(Width-2*t)/Picture.Graphic.Width;
      y1:=(Height-2*t)/Picture.Graphic.Height;
      if x1>y1 then  {--fit to height--}
      begin
        PicRect.Top:=0+t;
        PicRect.Bottom:=Height-t;
        x:=Round(Picture.Graphic.Width*y1);
        PicRect.Left:=(Width-x) div 2;
        PicRect.Right:=PicRect.Left+x;
      end
      else  {--fit to width--}
      begin
        PicRect.Left:=0+t;
        PicRect.Right:=Width-t;
        y:=Round(Picture.Graphic.Height*x1);
        PicRect.Top:=(Height-y)div 2;
        PicRect.Bottom:=PicRect.Top+y;
      end;
      Canvas.StretchDraw(PicRect, Picture.Graphic);
    end;
  end;
end;

procedure TDsFramedImage.SetFrameCorner(Value: TFrameCorner);
begin
  if Value<>FFrameCorner then
    begin FFrameCorner:=Value; Invalidate; end;
end;

procedure TDsFramedImage.SetFrameStyle(Value: TFrameStyle);
begin
  if Value<>FFrameStyle then
    begin FFrameStyle:=Value; Invalidate; end;
end;

procedure TDsFramedImage.SetColor_High(Value: TColor);
begin
  if Value<>FColor_High then
    begin FColor_High:=Value; Invalidate; end;
end;

procedure TDsFramedImage.SetColor_Low(Value: TColor);
begin
  if Value<>FColor_Low then
    begin FColor_Low:=Value; Invalidate; end;
end;

procedure TDsFramedImage.SetFrameColor(Value: TColor);
begin
  if Value<>FFrameColor then
    begin FFrameColor:=Value; Invalidate; end;
end;

procedure TDsFramedImage.SetShadowColor(Value: TColor);
begin
  if Value<>FShadowColor then
    begin FShadowColor:=Value; Invalidate; end;
end;

procedure TDsFramedImage.SetCornerRadius(Value: Integer);
begin
  if Value<>FCornerRadius then
  begin
    if Value<0 then FCornerRadius:=0
    else if Value>30 then FCornerRadius:=30
    else FCornerRadius:=Value;
    if FCornerRadius>4*FFrameWidth then FCornerradius:=4*FFrameWidth-1;
    Invalidate;
  end;
end;

procedure TDsFramedImage.SetFrameWidth(Value: Integer);
begin
  if Value<>FFrameWidth then
  begin
    if Value<4 then FFrameWidth:=4         {--minimum value--}
    else if Value>20 then FFrameWidth:=20  {--maximum value--}
    else FFrameWidth:=Value;
    if FCornerRadius>4*FFrameWidth then FCornerradius:=4*FFrameWidth-1;
    Invalidate;
  end;
end;

procedure TDsFramedImage.SetPicture(Value: TPicture);
begin
  FPicture.Assign(Value);
  Invalidate;
end;

procedure TDsFramedImage.SetActualSize(Value: Boolean);
begin
  if (Value<>FActualSize) then
  begin
    FActualSize:=Value;
    Invalidate;
  end;
end;

procedure TDsFramedImage.WMSize(var message: TWMSize);
var x: Integer;
begin
  inherited;
  if Width>Height then x:=Height div 2
  else x:=Width div 2;
  if FFrameWidth>x then
    begin FFrameWidth:=x; Invalidate; end;
end;

procedure Register;
begin
  RegisterComponents('Trial', [TDsFramedImage]);
end;

end.

