unit bt_ddftp;

(*
     BTDragDropFTP 2.0
     -----------------
     Datum: 6.2.1999
     Author: Michael Justin
     All rights reserved

*)

interface
uses
  Windows, Messages, SysUtils, Classes, Controls, Menus,
  buttons,
  Forms, Dialogs, StdCtrls, Extctrls, ComCtrls, wininet;

const
 Revision   = '$Revision: 1.82 $';
 ccName     = 'Name';
 // Listview column indices
 ixSize  = 0; ccSize = 'Size';
 ixDate  = 1; ccDate = 'Date';
 ixType  = 2; ccType = 'Type';

type
 TOrderBy = (sbName, sbDate, sbSize, sbType);
 TDisplayMessageEvent = procedure (Sender: TObject;
                                   const Messagetext: string;
                                   const IsError: boolean) of object;
 TGetLocalFileEvent = procedure (Sender: TObject;
                                     const RemoteFilename: string;
                                     var LocalFileName: string) of object;
 TGetConfirmationEvent = procedure (Sender: TObject;
                                    const Messagetext: string;
                                    var Confirm: boolean) of object;

TExtraFileData = class (TObject)
    Size      : integer;
    IsFolder  : boolean;
    IsLink    : boolean;
    LinkTarget: string;
    function IsRoot(const Caption: string):    boolean;
    function CanDelete(const Caption: string): boolean;
    function CanRename(const Caption: string): boolean;
end;

TBTFTPConnection = class(TObject)
    hConnect : HInternet;
    function ConnectTo(const HostName, LoginName, Password, RemoteDir: string;
                       const Passive: boolean; const Port: integer): boolean;
    function Connected: boolean;
    function PutFile(const Filename: string): boolean;
    function GetFile(const RemoteFile, LocalFile: string):boolean;
    function SetCurrentDirectory(const RemoteDir: string): boolean;
    function DeleteFile(const Filename: string): boolean;
    function CreateDirectory(const DirName: string): boolean;
    function RemoveDirectory(const Filename: string): boolean;
    function RenameFile(const OldFile, Newfile: string): boolean;
    procedure Disconnect;

end;

TBTDragDropFTP = class(TPanel)

private

    Connection : TBTFTPConnection;

    stateDesc,
    nameDesc,
    ageDesc,
    sizeDesc : boolean;
    OrderBy: TOrderBy;
    ViewMenu, OrderMenu : TMenuItem;
    FirstItem  : integer;
    FileCount : integer;

    FActive   : boolean;
    FHostName : string;
    FLoginName: string;
    FPassive  :  boolean;
    FPassword : string;
    FPort     : integer;
    FRemoteDir: string;

    FUploading: boolean;
    FUsedDiskSpace: integer;

    FOnChange: TNotifyEvent;
    FOnConnect: TNotifyEvent;
    FOnDisconnect: TNotifyEvent;
    FOnDblClick: TNotifyEvent;

    FOnDisplayMessage: TDisplayMessageEvent;
    FOnGetConfirmation: TGetConfirmationEvent;
    FOnGetLocalFilename: TGetLocalFileEvent;

    procedure AddFile(lpFindFileData: TWin32FindData);
    procedure Change; dynamic;
    function  Confirm(const Messagetext: string): boolean;

    procedure DeleteOnClick(Sender: TObject);

    procedure DoDblClick(Sender: TObject);
    function  ShowLastResponseInfo(const wait: boolean): boolean;
    procedure MyPopupMenuOnPopup(Sender: TObject);
    procedure MakeDirOnClick(Sender: TObject);
    procedure RenameOnClick(Sender: TObject);
    procedure SetActive(Value: boolean);
    procedure Log(const Messagetext: string);
    procedure DoShowMessage(const Messagetext: string);
    procedure WMDropFiles(var msg: TMessage); message WM_DROPFILES;
    procedure ListViewCompare(Sender: TObject; Item1,
              Item2: TListItem; Data: Integer; var Compare: Integer);
    procedure ListViewColumnClick(Sender: TObject; Column: TListColumn);
    procedure ListviewEdited(Sender: TObject; Item: TListItem; var S: string);
    procedure RefreshFileList(Sender: TObject);
    procedure UploadComplete;

  public
    ListView: TListView;

    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Loaded; override;

    function  Connect: boolean;
    function  ConnectTo(HostName, LoginName, Password: string): boolean;
    function  Connected: boolean;

    function  DeleteFile(const FileName: string): boolean;
    function  DeleteFolder(const FolderName: string): boolean;

    procedure Disconnect;

    function  NewFolder(const DirName: string): boolean;
    function  Rename(const OldName, Newname: string): boolean;

    function  Upload(Files: string; wait: boolean): boolean;
    function  CanDownLoad: boolean;

    function  FileName: string;

    function  DownLoadSelected: boolean;
    function  VersionString: string;

    property  Uploading: boolean read FUploading;
    property  UsedDiskSpace: integer read FUsedDiskSpace;

    procedure ViewLargeIcons(Sender: TObject);
    procedure ViewSmallIcons(Sender: TObject);
    procedure ViewList(Sender: TObject);
    procedure ViewReport(Sender: TObject);

    procedure OrderByName(Sender: TObject);
    procedure OrderBySize(Sender: TObject);
    procedure OrderByDate(Sender: TObject);
    procedure OrderByType(Sender: TObject);

  published
    property Active: boolean read FActive write SetActive;
    property HostName: string read FHostName write FHostName;
    property LoginName: string read FLoginName write FLoginName;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnConnect: TNotifyEvent read FOnConnect write FOnConnect;
    property OnDisconnect: TNotifyEvent read FOnDisconnect write FOnDisconnect;
    property OnDblClick: TNotifyEvent read FOnDblClick write FOnDblClick;
    property OnGetConfirmation: TGetConfirmationEvent read FOnGetConfirmation write FOnGetConfirmation;
    property OnDisplayMessage: TDisplayMessageEvent read FOnDisplayMessage write FOnDisplayMessage;
    property OnGetLocalFile: TGetLocalFileEvent read FOnGetLocalFileName write FOnGetLocalFileName;
    property Passive: boolean read FPassive write FPassive;
    property Password: string read FPassword write FPassword;
    property Port: integer read FPort write FPort;
    property RemoteDir: string read FRemoteDir write FRemoteDir;

end;

TUploadThread = class(TThread)
    FTP: TBTDragDropFTP;
    UPhConnect : HInternet;
    FFileList: TStringlist;
    procedure Execute; override;
end;

TBTEasyFTPButton = class (TSpeedButton)

  private
    FHostName : string;
    FLocalFile: TFilename;
    FLoginName: string;
    FPassive  : boolean;
    FPassword : string;
    FPort     : integer;
    FRemoteDir: string;
    FBTDragDropFTP: TBTDragDropFTP;

    FOnDisplayMessage: TDisplayMessageEvent;

  public
    constructor Create(AOwner: TComponent); override;
    procedure Click; override;

  protected

  published

    property BTDragDropFTP: TBTDragDropFTP read FBTDragDropFTP write FBTDragDropFTP;
    property LocalFile: TFileName read FLocalFile write FLocalFile;
    property HostName: string read FHostName write FHostName;
    property LoginName: string read FLoginName write FLoginName;
    property OnDisplayMessage: TDisplayMessageEvent read FOnDisplayMessage write FOnDisplayMessage;
    // property Overwrite: boolean read FOverwrite write FOverwrite;
    property Passive: boolean read FPassive write FPassive;
    property Password: string read FPassword write FPassword;
    property Port: integer read FPort write FPort;
    property RemoteDir: string read FRemoteDir write FRemoteDir;


end;

procedure Register;

implementation
uses Registry, ShellApi;

function GetFileType(const FileName: string; const isFolder: boolean):string ;
var
TheKey, Ext: string;
begin
Result := '';
with TRegistry.Create do
begin
    RootKey := HKEY_CLASSES_ROOT;
    Ext := ExtractFileExt(Filename);
    if OpenKey(Ext, false)
    then
    begin
         TheKey := ReadString('');
         CloseKey;
         if isFolder then TheKey := 'Folder';
         if OpenKey(TheKey, false)
         then Result := ReadString('')
         else Result := '';
         CloseKey;
    end
    else
    begin
         system.Delete(Ext, 1, 1);
         Result := UpperCase(Ext)+' file';
    end;
    Free;
end;
end;

function GetIconIndex(const AFile: string; Attrs: DWORD): integer;
var
SFI: TSHFileInfo;
begin
SHGetFileInfo(PChar(AFile), Attrs, SFI, SizeOf(TSHFileInfo),
              SHGFI_SYSICONINDEX or SHGFI_USEFILEATTRIBUTES);
Result := SFI.iIcon;
end;

function TExtraFileData.IsRoot;
begin
Result := IsFolder and (Caption = '..');
end;

function TExtraFileData.CanDelete;
begin
Result := not IsRoot(Caption);
end;

function TExtraFileData.CanRename;
begin
Result := not IsRoot(Caption);
end;

function TBTFTPConnection.Connected;
begin
result := Assigned(hConnect)
end;

function TBTFTPConnection.ConnectTo;
var
hSession: HInternet;
PassiveMode: integer;
begin
result := false;
hSession := InternetOpen('DDFTP20',
                         INTERNET_OPEN_TYPE_PRECONFIG,
                         nil,
                         nil,
                         0);
if Assigned(hSession) then
begin
    if Passive
    then PassiveMode := INTERNET_FLAG_PASSIVE
    else PassiveMode := 0;
    hConnect := InternetConnect(
        hSession,
        PChar(Hostname),
        Port,
        PChar(LoginName),
        PChar(Password),
        INTERNET_SERVICE_FTP,
        PassiveMode,
        0);
    if Connected then
    begin
        if (RemoteDir<>'') and not SetCurrentDirectory(RemoteDir)
        then
        begin
             result := false;
             Exit;
        end
        else
        begin
            result := true;
            Exit;  // q&d
        end;
    end
end;
end;

function TBTFTPConnection.SetCurrentDirectory;
begin
result := FtpSetCurrentDirectory(hConnect, PChar(RemoteDir))
end;

function TBTFTPConnection.CreateDirectory;
begin
result := FTPCreateDirectory(hConnect, PChar(DirName))
end;

function TBTFTPConnection.GetFile;
begin
result := FTPGetFile(hConnect, PChar(RemoteFile), PChar(LocalFile), false, 0, 0, 0);
end;

function TBTFTPConnection.DeleteFile;
begin
result := FTPDeleteFile(hConnect, PChar(Filename));
end;

function TBTFTPConnection.RenameFile;
begin
result := FTPRenameFile(hConnect, PChar(OldFile), PChar(NewFile));
end;

function TBTFTPConnection.RemoveDirectory;
begin
result := FTPRemoveDirectory(hConnect, PChar(Filename))
end;

function TBTFTPConnection.PutFile;
begin
result := FTPPutFile(hConnect, PChar(Filename), PChar(ExtractFileName(Filename)), 0, 0 );
end;

procedure TBTFTPConnection.Disconnect;
begin
InternetCloseHandle(hConnect);
hConnect := nil;
end;

destructor TBTDragDropFTP.Destroy;
begin
if Connected then Disconnect;
Connection.Free;
inherited;
end;

constructor TBTDragDropFTP.Create;
var
SysIL: THandle;
SysSIL: THandle;
SFI: TSHFileInfo;
begin
    inherited Create(AOwner);
    Connection := TBTFTPConnection.Create;
    FHostName  := 'localhost';
    FLoginName := 'anonymous';
    FPort      := INTERNET_DEFAULT_FTP_PORT;
    FPassive   := true;

    parent := Owner As TWinControl;
    if csDesigning in ComponentState then
    begin
        Active := true;
        Borderwidth := 3;
    end;
    ListView := TListView.Create(self);
    with ListView do
    begin
        parent    := self;
        ReadOnly  := false;
        align     := alClient;
        viewstyle := vsReport;
        with Columns.Add do
        begin
            caption := ccName;
            width   := 150;
        end;
        with Columns.Add do
        begin
            caption := ccSize;
            width   := 70;
            Alignment := taRightJustify;
        end;
        with Columns.Add do
        begin
            caption := ccDate;
            width   := 112
        end;
        with Columns.Add do
        begin
            caption := ccType;
            width := 200;
        end;
        OnDblClick := DoDblClick;

        LargeImages := TImageList.Create(self);
        with LargeImages do
        begin
            Width  := 32;
            Height := 32;
            SysIL  := SHGetFileInfo('', 0, SFI, SizeOf(SFI),
                      SHGFI_SYSICONINDEX or SHGFI_LARGEICON);
            if SysIL <> 0
            then
            begin
                LargeImages.Handle := SysIL;
                ShareImages := TRUE;
            end;
        end;

        SmallImages := TImageList.Create(self);
        with SmallImages do
        begin
            Width  := 16;
            Height := 16;
            SysSIL := SHGetFileInfo('', 0, SFI, SizeOf(SFI),
                      SHGFI_SYSICONINDEX or SHGFI_SMALLICON);
            if SysSIL <> 0
            then
            begin
                SmallImages.Handle := SysSIL;
                ShareImages := TRUE;
            end;
        end;
    end;
end;

procedure TBTDragDropFTP.MyPopupMenuOnPopup;
begin

  // Don't "get" 'nil' or directories
  self.PopupMenu.items[FirstItem].Enabled := Connected and (ListView.Selected<>nil)
  and (not TExtraFileData(ListView.Selected.Data).IsFolder)
  and Assigned(FOnGetLocalFileName);
  // Don't delete 'nil'
  self.PopupMenu.items[FirstItem+2].Enabled := Connected and (ListView.Selected<>nil)
  and TExtraFileData(ListView.Selected.Data).CanDelete(ListView.Selected.Caption);
  // don't rename 'nil'
  self.PopupMenu.items[FirstItem+3].Enabled := Connected and (ListView.Selected<>nil)
  and TExtraFileData(ListView.Selected.Data).CanRename(ListView.Selected.Caption);

// New folder
PopupMenu.items[FirstItem+4].Enabled := Connected;
// View
PopupMenu.items[FirstItem+5].Enabled := Connected;
// Order
PopupMenu.items[FirstItem+7].Enabled := Connected;
// Refresh
PopupMenu.items[FirstItem+9].Enabled := Connected;
end;

procedure TBTDragDropFTP.Loaded;
begin
inherited;

ListView.OnClick := Self.OnClick;

if not (csDesigning in ComponentState)
then
begin
    if not Assigned(PopupMenu)
    then PopupMenu := TPopupMenu.Create(self);

    FirstItem := PopupMenu.items.count;

    PopupMenu.OnPopup := MyPopupMenuOnPopup;
    PopupMenu.items.Add(TMenuItem.Create(self));
    with PopupMenu.items[PopupMenu.items.count-1] do
    begin
         Caption := 'Get';
         OnClick := DoDblClick;
         Default := true;
    end;
    PopupMenu.items.Add(TMenuItem.Create(self));
    with PopupMenu.items[PopupMenu.items.count-1] do
    begin
         Caption := '-';
    end;
    PopupMenu.items.Add(TMenuItem.Create(self));
    with PopupMenu.items[PopupMenu.items.count-1] do
    begin
         Caption := 'Delete';
         OnClick := DeleteOnClick;
    end;
    PopupMenu.items.Add(TMenuItem.Create(self));
    with PopupMenu.items[PopupMenu.items.count-1] do
    begin
         Caption := 'Rename';
         OnClick := RenameOnClick;
    end;
    PopupMenu.items.Add(TMenuItem.Create(self));
    with PopupMenu.items[PopupMenu.items.count-1] do
    begin
         Caption := 'New Folder';
         OnClick := MakeDirOnClick;
    end;
    ViewMenu := TMenuItem.Create(self);
    PopupMenu.items.Add(ViewMenu);
    with PopupMenu.items[PopupMenu.items.count-1] do
    begin
         Caption := 'View';
         Add(TMenuItem.Create(self));
         items[0].Caption := 'Large icons';
         items[0].RadioItem := true;
         items[0].OnClick := ViewLargeIcons;
         Add(TMenuItem.Create(self));
         items[1].Caption := 'Small icons';
         items[1].RadioItem := true;
         items[1].OnClick := ViewSmallIcons;
         Add(TMenuItem.Create(self));
         items[2].Caption := 'List';
         items[2].RadioItem := true;
         items[2].OnClick := ViewList;
         Add(TMenuItem.Create(self));
         items[3].Caption := 'Details';
         items[3].RadioItem := true;
         items[3].OnClick := ViewReport;
         items[3].Checked := true;
    end;
    // Separator
    PopupMenu.items.Add(TMenuItem.Create(self));
    with PopupMenu.items[PopupMenu.items.count-1] do
    begin
         Caption := '-';
    end;
    OrderMenu := TMenuItem.Create(self);
    PopupMenu.items.Add(OrderMenu);
    with PopupMenu.items[PopupMenu.items.count-1] do
    begin
         Caption := 'Order';
         Add(TMenuItem.Create(self));
         items[0].Caption := 'by Name';
         items[0].RadioItem := true;
         items[0].Checked := true;
         items[0].OnClick := OrderByName;
         Add(TMenuItem.Create(self));
         items[1].Caption := 'by Size';
         items[1].RadioItem := true;
         items[1].OnClick := OrderBySize;
         Add(TMenuItem.Create(self));
         items[2].Caption := 'by Date';
         items[2].RadioItem := true;
         items[2].OnClick := OrderByDate;
         Add(TMenuItem.Create(self));
         items[3].Caption := 'by Type';
         items[3].RadioItem := true;
         items[3].OnClick := OrderByType;
    end;
    // Separator
    PopupMenu.items.Add(TMenuItem.Create(self));
    with PopupMenu.items[PopupMenu.items.count-1] do
    begin
         Caption := '-';
    end;
    // Refresh
    PopupMenu.items.Add(TMenuItem.Create(self));
    with PopupMenu.items[PopupMenu.items.count-1] do
    begin
         Caption := 'Refresh';
         OnClick := RefreshFileList;
    end;

    ListView.OnEdited      := ListviewEdited;
    ListView.OnColumnClick := ListViewColumnClick;
    ListView.OnCompare     := ListViewCompare;
    Log('Not connected');
end;
end;

function TBTDragDropFTP.ShowLastResponseInfo;
var
lpdwError: dword;
lpszBuffer: PChar;
lpdwBufferLength: dword;
sl: TStringlist;
begin
lpdwBufferLength := 256;
GetMem(lpszBuffer, lpdwBufferLength-1);
result := false;
if InternetGetLastResponseInfo(lpdwError,  lpszBuffer, lpdwBufferLength)
then
begin
  result := true;
  sl := TStringlist.Create;
  try
      sl.Text := Trim(String(lpszBuffer));
      if sl.Text<>''
      then
      begin
           DoShowMessage(sl.Strings[sl.count-1]);
           if wait then Sleep(1000)
      end;
  finally
      sl.Free;
  end;
end
end;

procedure TBTDragDropFTP.RenameOnClick;
begin
    if (ListView.Selected<>nil)
    then ListView.Selected.EditCaption;
end;

function TBTDragDropFTP.DeleteFile;
begin
    result := false;
    if Connection.DeleteFile(FileName)
    then begin RefreshFileList(self); result := true; end
    else if not ShowLastResponseInfo(true)
    then DoShowMessage('Could not delete '+FileName+' (Win32 Error '+IntToStr(GetLastError)+')');
end;

function TBTDragDropFTP.DeleteFolder;
begin
    result := false;
    if Connection.RemoveDirectory(FolderName)
    then begin RefreshFileList(self); result := true; end
    else if not ShowLastResponseInfo(true)
    then DoShowMessage('Could not delete folder '+FolderName+' (Win32 Error '+IntToStr(GetLastError)+')')
end;

procedure TBTDragDropFTP.DeleteOnClick;
begin
    if FileName<>'' then
    begin
        if not TExtraFileData(ListView.Selected.Data).IsFolder
        then
        begin
            if Confirm('Delete "'+ListView.Selected.Caption+'" ?')
            then DeleteFile(FileName);
        end
        else
        begin
            if Confirm('Delete folder "'+ListView.Selected.Caption+'" ?')
            then DeleteFolder(FileName);
        end
    end;
end;

procedure TBTDragDropFTP.NewFolder;
begin
    result := false;
    if not Connected then DoShowMessage('No connection to host.');
    if (not Connected ) or (Trim(DirName)='') then exit;
    if Connection.CreateDirectory(DirName)
    then begin RefreshFileList(self); result := true; end
    else ShowLastResponseInfo(true);
end;

procedure TBTDragDropFTP.MakeDirOnClick;
var
DirName: string;
begin
InputQuery('New folder', 'Folder name:', DirName);
NewFolder(DirName);
end;

function TBTDragDropFTP.ConnectTo;
begin
self.HostName := HostName;
self.LoginName := LoginName;
self.Password := Password;
result := Connect;
end;

function TBTDragDropFTP.Connect;
var
hSession: HInternet;
PassiveMode: integer;
begin
result := false;
if Connected then Disconnect;
if HostName  = '' then raise Exception.Create('HostName must be specified');
if LoginName = '' then raise Exception.Create('LoginName must be specified');
if Password  = '' then raise Exception.Create('Password must be specified');
Log('Connecting to '+Hostname);
hSession := InternetOpen('DDFTP20',
                         INTERNET_OPEN_TYPE_PRECONFIG,
                         nil,
                         nil,
                         0);
if Assigned(hSession) then
begin
    if FPassive
    then PassiveMode := INTERNET_FLAG_PASSIVE
    else PassiveMode := 0;
    Connection.hConnect := InternetConnect(
        hSession,
        PChar(Hostname),
        FPort,
        PChar(LoginName),
        PChar(Password),
        INTERNET_SERVICE_FTP,
        PassiveMode,
        0);
    if Connected then
    begin
        Log('Connected to '+Hostname);
        if Assigned(FOnConnect) then FOnConnect(Self);
        if (RemoteDir<>'') and not Connection.SetCurrentDirectory(RemoteDir)
        then
        begin
             if not ShowLastResponseInfo(true)
             then DoShowMessage('Could not change to '+RemoteDir);
             result := false;
             Exit;
        end
        else
        begin
            RefreshFileList(self);
            result := true;
            Exit;  // q&d
        end;

    end
end;
if not ShowLastResponseInfo(true)
then DoShowMessage('Could not connect to '+HostName);
Sleep(1000);
Disconnect;
end;

function TBTDragDropFTP.Connected;
begin
result := Connection.Connected;
end;

procedure TBTDragDropFTP.Disconnect;
begin
InternetCloseHandle(Connection.hConnect);
Connection.hConnect := nil;
ListView.items.Clear;
Log('Not connected');
if Assigned(FOnDisconnect) then FOnDisconnect(Self);
end;

procedure TBTDragDropFTP.AddFile(lpFindFileData: TWin32FindData);
var
Seconds, Minutes, Hours, Day, Month, Year: word;
Systemtime: TSystemtime;
i, Pos: integer;
extra: TExtraFileData;
cLinkTarget: array[0..259] of char;
Filesize: extended;
begin
Application.ProcessMessages;
with Listview.items.add do
begin
    extra := TExtraFileData.Create;
    Data  := extra;

    Pos := StrLen(lpFindFileData.cFilename)+1;;

    extra.isLink   := (lpFindFileData.cFilename[Pos]='-')
                      and (lpFindFileData.cFilename[Pos+1]='>')
                      and (lpFindFileData.cFilename[Pos+2]=' ')
                      and (lpFindFileData.cFilename[Pos+3]<>#0) ;

    if extra.islink
    then
    begin
         for i:=0 to 259-Pos-3
         do cLinkTarget[i] := lpFindFileData.cFilename[i+Pos+3];
         extra.LinkTarget := string(cLinkTarget);
    end;

    Extra.isFolder := ((lpFindFileData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY)
                    = FILE_ATTRIBUTE_DIRECTORY)
                    or extra.IsLink;

    if Extra.isFolder then
    begin
        caption:=PChar(String(lpFindFileData.cFilename));
        ImageIndex := GetIconIndex('.', FILE_ATTRIBUTE_NORMAL or FILE_ATTRIBUTE_DIRECTORY);
        subitems.add('');
        Extra.Size := -1;
    end
    else
    begin
        caption:=lpFindFileData.cFilename;
        Inc(FileCount);
        ImageIndex := GetIconIndex(caption, FILE_ATTRIBUTE_NORMAL);
        Filesize := lpFindFileData.nFileSizeLow/1024+0.0;
        if Filesize<1 then Filesize:=1;
        Extra.Size := Trunc(Filesize);
        subitems.add(Format('%.0n', [Filesize])+' KB');
        Inc(FUsedDiskSpace, lpFindFileData.nFileSizeLow);
    end;

    FileTimeToSystemTime(lpFindFileData.ftLastWriteTime, systemtime);

    Day := systemtime.wDay;
    Month := systemtime.wMonth;
    Year  := systemTime.wYear;
    Seconds := systemTime.wSecond;
    Minutes := systemTime.wMinute;
    Hours   := systemTime.wHour;

    SubItems.Add(DateTimeToStr(
                  EncodeDate(Year, Month, Day)+
                  EncodeTime(Hours, Minutes, Seconds, 0)));

    SubItems.Add(GetFileType(lpFindFileData.cFilename, Extra.isFolder));
end;
end;

procedure TBTDragDropFTP.RefreshFileList;
var
lpFindFileData: TWin32FindData;
hFind: HInternet;
CurDirName: PAnsiChar;
leng: DWord;
begin
if not Connected then Exit;
ListView.items.Clear;
FUsedDiskSpace := 0;
FileCount  := 0;
GetMem(CurDirName, INTERNET_MAX_PATH_LENGTH+1);
Leng := INTERNET_MAX_PATH_LENGTH;
if FtpGetCurrentDirectory(Connection.hConnect, CurDirName, Leng)
then begin
    Log('Transferring data for '+CurDirName);
    LockWindowUpdate(ListView.handle);
    ListView.items.clear;
    if CurDirName<>'/'
    then with ListView.items.add do
    begin
         Caption := '..';
         ImageIndex := GetIconIndex('.', FILE_ATTRIBUTE_NORMAL or FILE_ATTRIBUTE_DIRECTORY);
         SubItems.Add(''); // Size
         SubItems.Add(''); // Date
         SubItems.Add(''); // Type
         Data := TExtraFileData.Create;
         TExtraFileData(Data).IsFolder := true;
    end;
    hFind := FtpFindFirstFile(Connection.hConnect, nil, lpFindFileData, INTERNET_FLAG_RELOAD, 0);
    if Assigned(hFind) then
    begin
        if GetLastError<>ERROR_NO_MORE_FILES then
        begin
            AddFile(lpFindFileData);
            while InternetFindNextFile(hFind, @lpFindFileData)
            do
            begin
            Log('Transferring data for '+CurDirName+' '+String(lpFindFileData.cFileName));
            AddFile(lpFindFileData);
            end;
        end;
        InternetCloseHandle(hFind);
    end;
    Log('Transfer completed ('+FormatFloat('#,##0', UsedDiskSpace)+' bytes in '+IntToStr(FileCount)+' files)');
    ListView.SortType  := stBoth;
    LockWindowUpdate(0);
    Change;
end
else // FtpGetCurrentDirectory failed
begin
    Log('Could not get current directory');
    DisConnect;
end;
FreeMem(CurDirName);
end;

procedure TBTDragDropFTP.Change;
begin
if Assigned(FOnChange) then FOnChange(Self);
end;

function TBTDragDropFTP.Confirm;
begin
if Assigned(FOnGetConfirmation)
then FOnGetConfirmation(self, MessageText, result)
else Confirm := mrYes=MessageDlg(Messagetext,mtConfirmation,[mbYes,mbNo],0);
end;

function TBTDragDropFTP.Rename;
begin
    result := false;
    if Connection.RenameFile(OldName, NewName)
    then
    with ListView.Selected do
    begin
        result := true;
        Caption := NewName;
        ImageIndex := GetIconIndex(NewName, FILE_ATTRIBUTE_NORMAL);
        SubItems[ixType] := GetFileType(NewName, false);
    end
    else if not ShowLastResponseInfo(true)
    then DoShowMessage('Could not rename '+OldName+' (Win32 Error '+IntToStr(GetLastError)+')');
end;

procedure TBTDragDropFTP.ListviewEdited;
begin
    if ListView.Selected.Caption=s then Exit;
    Rename(ListView.Selected.Caption, s);
end;

procedure TBTDragDropFTP.SetActive;
begin
If Value<>FActive then
begin
    FActive := Value;
    if not (csDesigning in ComponentState)
    then DragAcceptFiles(Handle, Value);
end;
end;

procedure TBTDragDropFTP.WMDropFiles;
var
FileName: PChar;
Files: string;
i, count,size,Drop: integer;
begin
  FileName := '';
  Files := '';
  Application.BringToFront;
  Drop  := msg.WParam;
  count := DragQueryFile(Drop, $FFFFFFFF, '', 0);
  for i:=1 to count do
  begin
      size := DragQueryFile(Drop, i-1, nil, 1);
      GetMem(filename, size+1);
      DragQueryFile(Drop, i-1, FileName, size+1);
      Files := Files+FileName+#13#10;
  end;
  FreeMem(filename);
  Upload(Files, false);
  while Uploading do Application.ProcessMessages;
  DragFinish(Drop);
end;

procedure TBTDragDropFTP.Log;
begin
if csDestroying in ComponentState then exit;
if Assigned (FOnDisplayMessage)
then FOnDisplayMessage(self, Messagetext, false);
end;

procedure TBTDragDropFTP.DoShowMessage;
begin
if Assigned (FOnDisplayMessage)
then FOnDisplayMessage(self, Messagetext, true)
else MessageDlg(MessageText, mtError, [mbOK], 0);
end;

procedure TBTDragDropFTP.DoDblClick;
var
RemoteFile, LocalFileName: string;
begin
if (ListView.Selected<>nil) then
begin
    RemoteFile := ListView.Selected.Caption;
    if not TExtraFileData(ListView.Selected.Data).IsFolder then
    begin
        if Assigned(FOnGetLocalFileName) then
        begin
             LocalFileName := '';
             FOnGetLocalFileName(self, RemoteFile, LocalFileName);
             if LocalFileName<>''
             then Connection.GetFile(RemoteFile, LocalFileName);
        end;
    end
    else
    begin
        {if TExtraFileData(ListView.Selected.Data).IsLink
        then RemoteFile := TExtraFileData(ListView.Selected.Data).LinkTarget;}
        if Connection.SetCurrentDirectory(RemoteFile)
        then RefreshFileList(self)
        else if not ShowLastResponseInfo(true)
        then DoShowMessage('Could not change to directory '+RemoteFile);
    end;
end;
if Assigned(FOnDblClick) then FOnDblClick(Sender)
end;

procedure TBTDragDropFTP.UploadComplete;
begin
    FUploading := false;
    DragAcceptFiles(Handle, Active);
    Log('');
    RefreshFileList(self);
end;

function TBTDragDropFTP.CanDownload;
begin
     if ListView.Selected<>nil
     then result := not TExtraFileData(ListView.Selected.Data).IsFolder
     else result := false;
end;

function TBTDragDropFTP.FileName;
begin
     if ListView.Selected<>nil
     then result := ListView.Selected.Caption
     else result := '';
end;

function TBTDragDropFTP.DownloadSelected;
var
RemoteFile, LocalFileName: string;
begin
    result := false;
    if Uploading then
    begin
        Log('File upload in progress!');
        exit;
    end;
    if CanDownload then
    begin
        RemoteFile := ListView.Selected.Caption;
        if Assigned(FOnGetLocalFileName) then
        begin
             LocalFileName := '';
             FOnGetLocalFileName(self, RemoteFile, LocalFileName);
        end
        else
        with TSaveDialog.Create(self) do
        try
            FileName := RemoteFile;
            if Execute then LocalFileName := Filename;
        finally
            Free;
        end;
        if LocalFileName<>''
        then FTPGetFile(Connection.hConnect, PChar(RemoteFile), PChar(LocalFileName),
                        false, 0, 0, 0);
        result := true;
    end;
end;

function TBTDragDropFTP.Upload;
begin
    result := false;
    if Uploading then
    begin
        Log('File upload in progress!');
        exit;
    end;
    if not Connected then Connect;
    FUploading := true;
    DragAcceptFiles(Handle, false);
    with TUploadThread.Create(true) do
    begin
         FTP            := self;
         UPhConnect     := Connection.hConnect;
         FFilelist      := TStringlist.Create;
         FFilelist.Text := Files;
         Resume;
         if Wait then WaitFor;
    end;
    result := true; // Thread started
end;

procedure TUploadThread.Execute;
var i: integer;
begin
for i:=0 to FFilelist.Count-1 do
begin
    FTP.Log('Uploading "'+ExtractFileName(FFilelist[i])+'"');
    FTPPutFile(UPhConnect, PChar(FFilelist[i]),
               PChar(ExtractFileName(FFilelist[i])), 0, 0 );
end;
Synchronize(FFilelist.Free);
Synchronize(FTP.UploadComplete);
end;

procedure TBTDragDropFTP.ViewLargeIcons;
begin
if not Connected then exit;
ListView.ViewStyle := vsIcon;
ViewMenu.items[0].Checked := true;
end;

procedure TBTDragDropFTP.ViewSmallIcons;
begin
if not Connected then exit;
ListView.ViewStyle := vsSmallIcon;
ViewMenu.items[1].Checked := true;
end;

procedure TBTDragDropFTP.ViewList;
begin
if not Connected then exit;
ListView.ViewStyle := vsList;
ViewMenu.items[2].Checked := true;
end;

procedure TBTDragDropFTP.ViewReport;
begin
if not Connected then exit;
ListView.ViewStyle := vsReport;
ViewMenu.items[3].Checked := true;
end;

procedure TBTDragDropFTP.OrderByName;
begin
if not Connected then exit;
OrderBy := sbName;
OrderMenu.items[0].Checked := true;
ListView.AlphaSort;
end;

procedure TBTDragDropFTP.OrderBySize;
begin
if not Connected then exit;
OrderBy := sbSize;
OrderMenu.items[1].Checked := true;
ListView.AlphaSort;
end;

procedure TBTDragDropFTP.OrderByDate;
begin
if not Connected then exit;
OrderBy := sbDate;
OrderMenu.items[2].Checked := true;
ListView.AlphaSort;
end;

procedure TBTDragDropFTP.OrderByType;
begin
if not Connected then exit;
OrderBy := sbType;
OrderMenu.items[3].Checked := true;
ListView.AlphaSort;
end;

procedure TBTDragDropFTP.ListViewCompare(Sender: TObject; Item1,
  Item2: TListItem; Data: Integer; var Compare: Integer);
var
ci1, ci2: string;
desc: boolean;
begin
try
    desc := false;
    Compare := 0;
    case OrderBy of
    sbName: begin
                ci1 := UpperCase(Item1.Caption);
                ci2 := UpperCase(Item2.Caption);
                if TExtraFileData(Item1.Data).IsFolder
                then ci1 := ' '+Item1.Caption;
                if TExtraFileData(Item2.Data).IsFolder
                then ci2 := ' '+Item2.Caption;
                desc := nameDesc;
                if ci1 < ci2 then Compare := -1 else
                   if ci1 > ci2 then Compare := 1;
            end;
    sbDate:
            begin
                ci1 := Item1.SubItems[ixDate];
                ci2 := Item2.SubItems[ixDate];
                desc := ageDesc;
                if StrToDateTime(ci1) < StrToDateTime(ci2) then Compare := -1 else
                   if StrToDateTime(ci1) > StrToDateTime(ci2) then Compare := 1;
            end;
    sbSize:
            begin
                if TExtraFileData(Item1.Data).Size < TExtraFileData(Item2.Data).Size
                then Compare := -1
                else if TExtraFileData(Item1.Data).Size > TExtraFileData(Item2.Data).Size
                then Compare := 1;
                desc := sizeDesc;
            end;
    sbType:
            begin
                ci1 := Item1.SubItems[ixType];
                ci2 := Item2.SubItems[ixType];
                if Item1.SubItems[ixSize]=''
                then ci1 := ' '+Item1.Caption;
                if Item2.SubItems[ixSize]=''
                then ci2 := ' '+Item2.Caption;
                desc := stateDesc;
                if ci1 < ci2 then Compare := -1 else
                   if ci1 > ci2 then Compare := 1;
            end;
    else
    //
    end;
    if desc then Compare := Compare*-1;
except
    on EStringlistError do Compare := 0;
end;
end;

procedure TBTDragDropFTP.ListViewColumnClick(Sender: TObject;
  Column: TListColumn);
begin
if Column.Caption = ccName then
begin
   if OrderBy = sbName
   then NameDesc := not NameDesc
   else OrderBy := sbName;
end;

if Column.Caption = ccDate then
begin
   if OrderBy = sbDate
   then ageDesc := not ageDesc
   else OrderBy := sbDate;
end;

if Column.Caption = ccSize then
begin
   if OrderBy = sbSize
   then sizeDesc := not sizeDesc
   else OrderBy := sbSize;
end;

if Column.Caption = ccType then
begin
   if OrderBy = sbType
   then stateDesc := not stateDesc
   else OrderBy := sbType;
end;
ListView.AlphaSort;
end;

function TBTDragDropFTP.VersionString;
begin
result := Copy (Revision, 12, Length(Revision)-12);
end;

constructor TBTEasyFTPButton.Create;
begin
    inherited Create(AOwner);
    FHostName  := 'localhost';
    FLoginName := 'anonymous';
    FPort      := INTERNET_DEFAULT_FTP_PORT;
    FPassive   := true;
end;

procedure TBTEasyFTPButton.Click;
begin
inherited;
if (LocalFile='') or not FileExists(LocalFile)
then raise EFOpenError.Create('Localfile '''+Localfile+''' not found');
with TBTFTPConnection.Create do
try
    ConnectTo(HostName, LoginName, Password, RemoteDir, Passive, Port);
    if Connected then PutFile(LocalFile);
    if Assigned(BTDragDropFTP) then BTDragDropFTP.RefreshFileList(self);
    Disconnect;
finally;
    Free
end;
end;

procedure Register;
begin
RegisterComponents('BetaSoft', [TBTDragDropFTP]);
RegisterComponents('BetaSoft', [TBTEasyFTPButton]);
end;

end.
