unit AllColorsBox;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, MaxMin;

type
  TAllColorsBox = class(TCustomComboBox)
  private
    FColorValue: TColor;
    FDisplayNames: Boolean;
    FOnChange: TNotifyEvent;
    procedure SetColorValue(NewValue: TColor);
    procedure SetDisplayNames(Value: Boolean);
    procedure ResetItemHeight;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
  protected
    procedure CreateWnd; override;
    procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
    procedure Click; override;
    procedure BuildList; virtual;
    procedure Change; override;
    procedure GetColorStr(const S : string); virtual;
    procedure MItem(Control: TWinControl; Index: Integer;
      var Height: Integer);
  public
    constructor Create(AOwner: TComponent); override;
    property Text;
  published
    property ColorValue: TColor read FColorValue write SetColorValue
      default clBlack;
    property DisplayNames: Boolean read FDisplayNames write SetDisplayNames
      default True;
    property Color;
    property Ctl3D;
    property DragMode;
    property DragCursor;
    property Enabled;
    property Font;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDropDown;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnStartDrag;
  end;

procedure Register;

implementation

function GetItemHeight(Font: TFont): Integer;
var
  DC: HDC;
  SaveFont: HFont;
  Metrics: TTextMetric;
begin
  DC := GetDC(0);
  try
    SaveFont := SelectObject(DC, Font.Handle);
    GetTextMetrics(DC, Metrics);
    SelectObject(DC, SaveFont);
  finally
    ReleaseDC(0, DC);
  end;
  Result := Metrics.tmHeight + 1;
end;

constructor TAllColorsBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  OnMeasureItem := MItem;
  ItemHeight := 18;
  Style := csOwnerDrawFixed;
  FColorValue := clBlack;  { make default color selected }
  FDisplayNames := True;
end;

procedure TAllColorsBox.GetColorStr(const S : string);
//var
//  I: Integer;
//  ColorName: string;
begin
  Items.AddObject(S, TObject(StringToColor(S)));
end;

procedure TAllColorsBox.BuildList;
begin
  Clear;
  GetColorValues(GetColorStr);
end;

procedure TAllColorsBox.MItem(Control: TWinControl; Index: Integer;
      var Height: Integer);
begin
 Height := Perform(CB_GETITEMHEIGHT, 0, 0);
end;

procedure TAllColorsBox.SetDisplayNames(Value: Boolean);
begin
  if DisplayNames <> Value then begin
    FDisplayNames := Value;
    Invalidate;
  end;
end;

procedure TAllColorsBox.SetColorValue(NewValue: TColor);
var
  Item: Integer;
  CurrentColor: TColor;
begin
  if (ItemIndex < 0) or (NewValue <> FColorValue) then
    { change selected item }
    for Item := 0 to Pred(Items.Count) do begin
      CurrentColor := TColor(Items.Objects[Item]);
      if CurrentColor = NewValue then begin
        FColorValue := NewValue;
        if ItemIndex <> Item then ItemIndex := Item;
        Change;
        Break;
      end;
    end;
end;

procedure TAllColorsBox.CreateWnd;
begin
  inherited CreateWnd;
  BuildList;
  SetColorValue(FColorValue);
end;

procedure TAllColorsBox.DrawItem(Index: Integer; Rect: TRect;
  State: TOwnerDrawState);
const
  ColorWidth = 22;
var
  ARect: TRect;
  Text: array[0..255] of Char;
  Safer: TColor;
begin
  ARect := Rect;
  Inc(ARect.Top, 2);
  Inc(ARect.Left, 2);
  Dec(ARect.Bottom, 2);
  if FDisplayNames then ARect.Right := ARect.Left + ColorWidth
  else Dec(ARect.Right, 3);
  with Canvas do begin
    FillRect(Rect);
    Safer := Brush.Color;
    Pen.Color := clBlack;
    Rectangle(ARect.Left, ARect.Top, ARect.Right, ARect.Bottom);
    Brush.Color := TColor(Items.Objects[Index]);
    try
      InflateRect(ARect, -1, -1);
      FillRect(ARect);
    finally
      Brush.Color := Safer;
    end;
    if FDisplayNames then begin
      StrPCopy(Text, Items[Index]);
      Rect.Left := Rect.Left + ColorWidth + 6;
      DrawText(Canvas.Handle, Text, StrLen(Text), Rect, DT_SINGLELINE or
        DT_VCENTER or DT_NOPREFIX);
    end;
  end;
end;

procedure TAllColorsBox.Click;
begin
  if ItemIndex >= 0 then ColorValue := TColor(Items.Objects[ItemIndex]);
  inherited Click;
end;

procedure TAllColorsBox.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ResetItemHeight;
  RecreateWnd;
end;

procedure TAllColorsBox.ResetItemHeight;
begin
  ItemHeight := Max(GetItemHeight(Font) + 1, 9);
end;

procedure TAllColorsBox.Change;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure Register;
begin
  RegisterComponents('Genesis', [TAllColorsBox]);  
end;

end.
