unit MainModule2;

interface

uses
  Windows, Messages, SysUtils, Classes, HTTPApp, WebTracker, DataObj, DBTables;

type
  TWebModule1 = class(TWebModule)
    ppClientDetails: TPageProducer;
    ppStockDetails: TPageProducer;
    procedure WebModule1Create(Sender: TObject);
    procedure WebModule1AfterDispatch(Sender: TObject;
      Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
    procedure WebModule1BeforeDispatch(Sender: TObject;
      Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
    procedure WebModule1Destroy(Sender: TObject);
    procedure WebModule1waLoginAction(Sender: TObject;
      Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
    procedure ppClientDetailsHTMLTag(Sender: TObject; Tag: TTag;
      const TagString: String; TagParams: TStrings;
      var ReplaceText: String);
    procedure WebModule1waViewAccountAction(Sender: TObject;
      Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
    procedure WebModule1waViewStockAction(Sender: TObject;
      Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
    procedure ppStockDetailsHTMLTag(Sender: TObject; Tag: TTag;
      const TagString: String; TagParams: TStrings;
      var ReplaceText: String);
    procedure WebModule1waSellAction(Sender: TObject; Request: TWebRequest;
      Response: TWebResponse; var Handled: Boolean);
  private
    { Private declarations }
  public
    { Public declarations }
    UserProfile : TUserProfile;
    WebSession : TWebSession;
  end;

var
  WebModule1: TWebModule1;

implementation

{$R *.DFM}

{------------------------------------------------------------------------------
Name    : TWebModule1.WebModule1Create
Purpose : OnCreate event handler for the web module
          - initializes the WebSession variable
------------------------------------------------------------------------------}
procedure TWebModule1.WebModule1Create(Sender: TObject);
begin
  WebSession := nil;
end;

{------------------------------------------------------------------------------
Name    : TWebModule1.WebModule1Destroy
Purpose : OnDestroy event handler for the web module
          - free's the WebSession variable
------------------------------------------------------------------------------}
procedure TWebModule1.WebModule1Destroy(Sender: TObject);
begin
  if Assigned(WebSession) then
    WebSession.Free;
end;

{------------------------------------------------------------------------------
Name    : TWebModule1.WebModule1AfterDispatch
Purpose : OnAfterDispatch event handler for the web module
          - Saves the web session in memory and adds the Session Id to all
            references in the Response HTML going back to the user
          - Encrypts all sensitive data going back to the user that could be
            sent back in by further actions (POSTs and GETs)
------------------------------------------------------------------------------}
procedure TWebModule1.WebModule1AfterDispatch(Sender: TObject;
  Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
begin
  if Assigned(WebSession) then
    begin
    WebTracker.SaveWebSession(WebSession, Response);
    WebSession.EncryptParameters(Response);
    end;
end;

{------------------------------------------------------------------------------
Name    : TWebModule1.WebModule1BeforeDispatch
Purpose : OnBeforeDispatch event handler for the web module
          - if not working with the LOGIN web page, get the Session Id from
            the web request URL and get the session
          - check if session is expired
          - Decrypt all sensitive data that was previously encrypted
------------------------------------------------------------------------------}
procedure TWebModule1.WebModule1BeforeDispatch(Sender: TObject;
  Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
begin
  try
    if Request.PathInfo <> '/Login' then
      begin
      WebSession := WebTracker.FindWebSession(Request);
      WebSession.DecryptParameters(Request);
      end;
  except
    on EUnknownWebSessionError do
      begin
      Response.Content := 'Error:  Unknown Session.';
      Handled := true;
      end;
    on EExpiredWebSessionError do
      begin
      Response.Content := 'Error:  Session has expired.';
      Handled := true;
      end;
    end;
end;

{------------------------------------------------------------------------------
Name    : TWebModule1.WebModule1waLoginAction
Purpose : /Login Action handler
          - validate the user
          - create a UserProfile object for them
          - locate all stock purchases by this user
------------------------------------------------------------------------------}
procedure TWebModule1.WebModule1waLoginAction(Sender: TObject;
  Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
var
  UserId, Password : string;
  Query : TQuery;
  Account : TAccount;
begin
  UserId := Request.ContentFields.Values['UserId'];
  Password := Request.ContentFields.Values['Password'];

  { ... Validate the User ... }

  {create a web user object to track this user}
  WebSession := TWebSession.Create;

  {create an object to track some data about this user}
  UserProfile := TUserProfile.Create;
  UserProfile.UserId := UserId;
  UserProfile.Password := Password;
  UserProfile.IPAddress := Request.RemoteAddr;
  UserProfile.FirstName := 'Dave';
  UserProfile.LastName := 'Sawyer';
  WebSession.AddData('Profile', UserProfile);

  {I prefer this method of copying data into my own objects rather and freeing the Query}
  Query := TQuery.Create(nil);
  Query.DatabaseName := 'DBDemos';
  Query.Sql.Add('select * from clients where (LAST_NAME = "' + UserProfile.LastName + '")');
  Query.Sql.Add('  and (FIRST_NAME = "' + UserProfile.FirstName + '");');
  Query.Open;
  UserProfile.Address := Query.FieldByName('ADDRESS_1').AsString + #13#10 +
                         Query.FieldByName('CITY').AsString + #13#10 +
                         Query.FieldByName('STATE').AsString + '   ' +
                         Query.FieldByName('ZIP').AsString;
  UserProfile.Telephone := Query.FieldByName('TELEPHONE').AsString;
  UserProfile.AccountOpenDate := Query.FieldByName('DATE_OPEN').AsDateTime;

  {I pretend the user has more then 1 account number for this demo}
  Account := TAccount.Create;
  Account.Number := Query.FieldByName('ACCT_NBR').AsString;
  UserProfile.Accounts.Add(Account);

  Account := TAccount.Create;
  Account.Number := Query.FieldByName('ACCT_NBR').AsString;
  UserProfile.Accounts.Add(Account);

  {free up the query to conserve database licenses}
  Query.Close;
  Query.Free;

  Response.Content := ppClientDetails.Content;
  Handled := true;
end;

{------------------------------------------------------------------------------
Name    : TWebModule1.ppClientDetailsHTMLTag
Purpose : Client Details Web Page HTML Tag handler
------------------------------------------------------------------------------}
procedure TWebModule1.ppClientDetailsHTMLTag(Sender: TObject; Tag: TTag;
  const TagString: String; TagParams: TStrings; var ReplaceText: String);
var
  ChPos, i : integer;
  Account : TAccount;
begin
  if TagString = 'Name' then
    ReplaceText := UserProfile.FirstName + ' ' + UserProfile.LastName;

  if TagString = 'Telephone' then
    ReplaceText := UserProfile.Telephone;

  if TagString = 'Address' then
    begin
    {change the CR LF to HTML <BR>}
    ReplaceText := UserProfile.Address;
    ChPos := Pos(#13#10, ReplaceText);
    while ChPos <> 0 do
      begin
      ReplaceText := copy(ReplaceText, 1, ChPos-1) + '<BR>' + copy(ReplaceText, ChPos+2, length(ReplaceText));
      ChPos := Pos(#13#10, ReplaceText);
      end;
    end;

  if TagString = 'Accounts' then
    begin
    {in this sample, I use the index of the Account List as the Value - not the Account Number}
    ReplaceText := '<SELECT NAME="Account" SIZE="1">';
    for i := 0 to UserProfile.Accounts.Count-1 do
      begin
      Account := TAccount(UserProfile.Accounts[i]);
      ReplaceText := ReplaceText + '<OPTION VALUE="' + IntToStr(i) + '">' + Account.Number + #13#10;
      end;
    ReplaceText := ReplaceText + '</SELECT>' + #13#10;
    end;
end;

{------------------------------------------------------------------------------
Name    : TWebModule1.WebModule1waViewAccountAction
Purpose : /ViewAccount Action handler
          - validate the user
          - create a UserProfile object for them
          - locate all stock purchases by this user
------------------------------------------------------------------------------}
procedure TWebModule1.WebModule1waViewAccountAction(Sender: TObject;
  Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
var
  Query : TQuery;
  Account : TAccount;
  Index : integer;
begin
  {find the account to look at}
  UserProfile := TUserProfile(WebSession.GetData('Profile'));
  Index := StrToInt(Request.ContentFields.Values['Account']);
  Account := UserProfile.Accounts[Index];

  {this time around, I will leave the TQuery open}
  Query := TQuery.Create(nil);
  Query.DatabaseName := 'DBDemos';
  Query.Sql.Add('select * from holdings where ACCT_NBR = "' + Account.Number + '";');
  Query.Open;
  WebSession.AddData('Holdings', Query);

  Response.Content := ppStockDetails.Content;
  Handled := true;
end;

{------------------------------------------------------------------------------
Name    : TWebModule1.ppStockDetailsHTMLTag
Purpose : Stock Details Web Page HTML Tag handler
------------------------------------------------------------------------------}
procedure TWebModule1.ppStockDetailsHTMLTag(Sender: TObject; Tag: TTag;
  const TagString: String; TagParams: TStrings; var ReplaceText: String);
var
  Query : TQuery;
  Account : TAccount;
  Index : integer;
begin
  if TagString = 'Account' then
    begin
    {knowing that I would need the Account again, I normally would store the "active" account in the data object}
    UserProfile := TUserProfile(WebSession.GetData('Profile'));
    Index := StrToInt(Request.ContentFields.Values['Account']);
    Account := UserProfile.Accounts[Index];
    ReplaceText := Account.Number;
    end;

  if TagString = 'Portfolio' then
    begin
    {in this case, the stock symbol is passed back as the Value}
    Query := TQuery(WebSession.GetData('Holdings'));
    Query.First;
    ReplaceText := '';
    while not Query.Eof do
      begin
      ReplaceText := ReplaceText + '<tr><td><INPUT TYPE="RADIO" NAME="Stock" VALUE="' +
                     Query.FieldByName('SYMBOL').AsString + '"></td>'#13#10;
      ReplaceText := ReplaceText + '<td><a href="/scripts/Demo2.dll/ViewStock?Stock=' +
                     Query.FieldByName('SYMBOL').AsString + '">' +
                     Query.FieldByName('SYMBOL').AsString + '</a></td>' + #13#10;
      ReplaceText := ReplaceText + '<td>' + Query.FieldByName('SHARES').AsString + '</td></tr>';
      Query.Next;
      end;
    end;
end;

{------------------------------------------------------------------------------
Name    : TWebModule1.WebModule1waViewStockAction
Purpose : /ViewStock Action handler
          - get the Stock Symbol
          - search through the list of stocks for this Web User
          - display the details of that stock
------------------------------------------------------------------------------}
procedure TWebModule1.WebModule1waViewStockAction(Sender: TObject;
  Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
var
  Query : TQuery;
  StockSymbol : string;
begin
  {get the stock symbol from the URL}
  StockSymbol := Request.QueryFields.Values['Stock'];

  {locate the symbol in the still open Query from the user's last request}
  Query := TQuery(WebSession.GetData('Holdings'));
  Query.First;
  while (Query.FieldByName('SYMBOL').AsString <> StockSymbol) and (not Query.Eof) do
    Query.Next;

  {display the stock details}
  Response.Content := 'Symbol : ' + Query.FieldByName('SYMBOL').AsString + '<BR>' +
                      'Purchase Price : ' + Format('%8.2m', [Query.FieldByName('PUR_PRICE').AsFloat]) + '<BR>' +
                      'Purchase Date : ' + FormatDateTime('dd-mmm-yyyy', Query.FieldByName('PUR_Date').AsDateTime) + '<BR>';
end;

{------------------------------------------------------------------------------
Name    : TWebModule1.WebModule1waSellAction
Purpose : /Sell Action handler
          - get the Stock Symbol
          - search through the list of stocks for this Web User
          - display the details of that stock
          - pretty much the same as the /ViewStock actions except input is from
            a form's POST
------------------------------------------------------------------------------}
procedure TWebModule1.WebModule1waSellAction(Sender: TObject;
  Request: TWebRequest; Response: TWebResponse; var Handled: Boolean);
var
  Query : TQuery;
  StockSymbol : string;
  HiddenField : string;
begin
  {get the stock symbol indicated by the radio buttons}
  StockSymbol := Request.ContentFields.Values['Stock'];
  HiddenField := Request.ContentFields.Values['HiddenField'];
  
  {again, locate the symbol in the still open Query from the user's last request}
  Query := TQuery(WebSession.GetData('Holdings'));
  Query.First;
  while (Query.FieldByName('SYMBOL').AsString <> StockSymbol) and (not Query.Eof) do
    Query.Next;

  {display the stock details}
  Response.Content := 'Symbol : ' + Query.FieldByName('SYMBOL').AsString + '<BR>' +
                      'Purchase Price : ' + Format('%8.2m', [Query.FieldByName('PUR_PRICE').AsFloat]) + '<BR>' +
                      'Purchase Date : ' + FormatDateTime('dd-mmm-yyyy', Query.FieldByName('PUR_Date').AsDateTime) + '<BR>' +
                      'Hidden Field : ' + HiddenField;
end;


{------------------------------------------------------------------------------
  change the WebTracker defaults
------------------------------------------------------------------------------}
initialization
  WebTracker.ClientTimeOut := (1.00/24/60) * 10; //10 minutes
  WebTracker.CleanUpTimeInterval := (1.00/6);    //4 hours

end.
