{***************************************************************
 *
 * Unit Name : Btrieve
 * Purpose   : Btrieve Interface
 * Author    : Eugen Mihailescu
 * Company   : EM Quicksoft Romania SRL
 * Copyright : 1998,2002  All rights reserved.
 * Web Site  : http://www.em-quicksoft.com
 * Support   : support@em-quicksoft.com
 * History   : 03/19/2000
 *
 * Disclaimer: The objectives of this program is only educational.
 *             You MUST NOT use it without getting any material benefit on it
 *             You have the right to share this code with others without removing
 *             the copyright and/or disclaimer notice.
 *             You are allowed to include its code in your application
 *             only if it was designed for non commercial purpose,
 *             otherwise we claim $10 for each copy of your
 *             application which contains this code inside.
 ****************************************************************}

Unit Btrieve;

Interface

Uses
  Windows,
  Messages,
  SysUtils,
  Classes,
  Graphics,
  Controls,
  Forms,
  Dialogs,
  DsgnIntf,
  FileCtrl,
  Btr,
  DBTables,
  Db,
  BtrLib,
  BtrTypes;
Type
  TBuff = Array[0..9999] Of char;
  TBtrAbout = Class(TPropertyEditor)
  public
    Procedure Edit;
      override;
    Function GetAttributes: TPropertyAttributes;
      override;
    Function GetValue: String;
      override;
  End;
  TDateFormat = (ShortStyle, LongStyle);
  TBtrieveTypes = (AsString, AsInteger, asFloat, asDate, asTime, asDecimal, asMoney, AsBoolean, asNumeric, asBFloat, asLString, asZString, asUnknown, asY, asChar, asAutoIncrement);
  TBtrieveDict = ^TBtrieveDictionary;
  TBtrieveDictionary = Record
    FileHeader: String;
    FieldName: String;
    FieldAlias: String;
    Offset: Integer;
    Size: Integer;
    FieldType: TBtrieveTypes;
    Value: Variant;
    AllowNULL: Boolean;
    NextRec: TBtrieveDict;
  End;

  BtrRecInfo = Record
    FileHeader: String;
    FieldName: String;
    FieldAlias: String;
    FieldType: TBtrieveTypes;
    AllowNULL: Boolean;
    Offset: Integer;
    Size: Integer;
  End;

  TBtrGridStr254 = String[254];
  TBtrGridFilename = Type TBtrGridStr254;
  TBtrGridFileNameProperty = Class(TStringProperty)
  public
    Procedure Edit; override;
    Function GetAttributes: TPropertyAttributes; override;
  End;

  TBtrieveNotifyEvent = Procedure(Sender: TObject) Of Object;

  TBtrieve = Class(TComponent)
  private
    fAbout: TBtrAbout;
    fDateFormat: TDateFormat;
    dPath: TBtrGridFileName;
    fActive: Boolean;
    FFile: String;
    fField: String;
    BtrDB: TBtr;
    FileDict: TTable;
    FieldDict: TTable;
    fDataSource: TDataSource;
    fFilesList: TStringList;
    fIndex: Integer;
    fHeadDefList: TBtrieveDict;
    fStartDefList: TBtrieveDict;
    FBeforeOpen: TBtrieveNotifyEvent;
    //    fOtherList : TStringList;

    Procedure SetDPath(AFilename: TBtrGridFileName);
    Procedure SetActive(AACtive: Boolean);
    Procedure SetBtrSource(Value: TBtr);
    Procedure SetFileDictName(Value: String);
    Procedure SetFieldDictName(Value: String);
    Procedure SetFilesList(Value: TStringList);
    Procedure SetFileIndex(Value: Integer);

    { Private declarations }
  protected
    Procedure CreateDefList;
    Procedure DestroyDefList;
    { Protected declarations }
  public
    Constructor Create(AOwner: TComponent); override;
    Destructor Destroy; override;

    Function FindKey(FileNo, FieldCode: String; Value: Variant): Boolean;
    Function GetFieldValue(FileNo, FieldCode: String; asType: TBtrieveTypes): Variant;
    Function SetFieldValue(FileNo, FieldCode: String; Value: Variant; asType: TBtrieveTypes): Integer;
    Function GetFieldInfo(FileNo, FieldCode: String): BtrRecInfo;
    Function setbuff(B: TBuff; S: String; p, L: Integer): TBuff;
    Function GetFieldList(fHeader: String): TStringList;
    { Public declarations }
  published
    Property DictionaryPath: TBtrGridFileName read DPath write SetdPath;
    Property Active: Boolean read fActive write SetActive default False;
    Property FileDictName: String read FFile write SetFileDictName;
    Property FieldDictName: String read fField write SetFieldDictName;
    Property BtrSource: TBtr read BtrDB write SetBtrSource;
    Property FilesList: TStringList read fFilesList write SetFilesList;
    Property FileListIndex: Integer read fIndex write SetFileIndex;
    Property About: TBtrAbout read fAbout;
    Property BeforeOpen: TBtrieveNotifyEvent read FBeforeOpen write FBeforeOpen;
    Property DateFormat: TDateFormat read fDateFormat write fDateFormat;
    { Published declarations }
  End;

Procedure Register;

Const
  CRLF              = #13#10;
  {$IFDEF SCALA_RV6}
  Version           = 0;
  {$ELSE}
  Version           = 1;
  {$ENDIF}

Implementation

//******************* TBtrAbout.Edit *************************

Procedure TBtrAbout.Edit;
Begin
  Application.MessageBox('Btrieve for Borland Delphi v3.0,v4.0' + CRLF +
    'Freeware Component and Source Code' + CRLF +
    'You have rights to use it or distribute it as freeware' + CRLF + CRLF +
    'For updates and technical support, ' + CRLF +
    'write to eugen.mihailescu@scala.ro' + CRLF + CRLF +
    'Source code in this package may be distributed ' + CRLF +
    'or modified without written consent.' + CRLF + CRLF +
    'This component requires TBtr component and ScalaLib unit.',
    'About Btrieve Component...', mb_OK + MB_ICONINFORMATION);
End;
//******************* TBtrAbout.GetAttributes *************************

Function TBtrAbout.GetAttributes: TPropertyAttributes;
Begin
  Result := [paMultiSelect, paDialog, paReadOnly];
End;

Function TBtrAbout.GetValue: String;
Begin
  Result := '[ About... ]';
End;
//******************* TBtrieve.Create *************************

Constructor TBtrieve.Create(AOwner: TComponent);
Begin
  fFilesList := TStringList.Create;
  //  fOtherList := TStringList.Create;
  fHeadDefList := Nil;
  fStartDefList := Nil;
  Inherited Create(AOwner);
  RegisterClass(TTable);
  RegisterClass(TDataSource);
  FileDict := TTable.Create(Nil);
  FieldDict := TTable.Create(Nil);
  fDataSource := TDataSource.Create(Nil);
  fDataSource.DataSet := FileDict;
End;
//******************* TBtrieve.Destroy *************************

Destructor TBtrieve.Destroy;
Begin
  If fActive Then Active := False;
  fDataSource.Free;
  FieldDict.Free;
  FileDict.Free;
  Inherited Destroy;
End;
//******************* TBtrieve.setbuff *************************

Function TBtrieve.setbuff(B: TBuff; S: String; p, L: Integer): TBuff;
Var
  ix                : Integer;
Begin
  S := Format('%-' + IntToStr(L) + 's', [S]);
  For ix := p To p + L - 1 Do
    B[ix] := S[ix - p + 1];
  Result := B;
End;
//******************* TBtrieve.DestroyDefList *************************

Procedure TBtrieve.DestroyDefList;
Var
  aux               : TBtrieveDict;
Begin
  fHeadDefList := fStartDefList;
  While fHeadDefList <> Nil Do
    Begin
      aux := fHeadDefList;
      fHeadDefList := fHeadDefList^.NextRec;
      Dispose(aux);
    End;
  fStartDefList := Nil;
  fHeadDefList := Nil;
End;
//******************* TBtrieve.CreateDefList *************************

Procedure TBtrieve.CreateDefList;
Var
  I                 : Integer;

  Procedure AddNode(FileName, FieldName, FieldAlias: String; FieldType: TBtrieveTypes; AllowNULL: Boolean; Offset, Size: Integer);
  Var
    aux             : TBtrieveDict;
  Begin
    If fHeadDefList = Nil Then
      Begin
        New(aux);
        aux^.FileHeader := FileName;
        aux^.FieldName := FieldName;
        aux^.Offset := Offset;
        aux^.Size := Size;
        aux^.FieldAlias := FieldAlias;
        aux^.AllowNULL := AllowNULL;
        aux^.FieldType := FieldType;
        aux^.NextRec := Nil;
        fHeadDefList := aux;
        fStartDefList := aux;
      End
    Else
      Begin
        New(aux);
        aux^.FileHeader := FileName;
        aux^.FieldName := FieldName;
        aux^.Offset := Offset;
        aux^.Size := Size;
        aux^.FieldAlias := FieldAlias;
        aux^.AllowNULL := AllowNULL;
        aux^.FieldType := FieldType;
        aux^.NextRec := Nil;
        fHeadDefList^.NextRec := aux;
        fHeadDefList := aux;
      End;
  End;

Begin
  If Not fActive Then
    MessageDlg('Dictionary is not open. You must open it then you can create Dictionary Definition list.', mtInformation, [mbOK], 0)
  Else
    If fFilesList.Count = 0 Then
    MessageDlg('You have no file(s) in File''s list. Dictionary Definition list is empty.', mtInformation, [mbOK], 0)
  Else
    Begin
      fHeadDefList := Nil;
      fStartDefList := Nil;
      For I := 1 To fFilesList.Count Do
        Begin
          If FileDict.FindKey([fFilesList.Strings[I - 1]]) Then
            Begin
              FieldDict.First;
              While Not FieldDict.Eof Do
                Begin
                  AddNode(FieldDict.FieldByName('FileNo').AsString, FieldDict.FieldByName('FieldCode').AsString, FieldDict.FieldByName('Name').AsString,
                    TBtrieveTypes(FieldDict.FieldByName('DataType').AsInteger), Not FieldDict.FieldByName('IsNotNull').AsBoolean,
                    FieldDict.FieldByName('Offset').AsInteger, FieldDict.FieldByName('Size').AsInteger);
                  FieldDict.Next;
                End;
            End;
        End;
    End;
End;
//******************* TBtrieve.SetFileDictName *************************

Procedure TBtrieve.SetFileDictName(Value: String);
Begin
  If fActive Then
    If MessageDlg('You cannot change the dictionary File name while it is still Active. Do you want to make it inactive then change the name?', mtConfirmation, [mbYes, mbCancel], 0) = mrYes Then
      Begin
        fActive := False;
        FFile := Value;
        fActive := True;
      End
    Else
  Else
    FFile := Value;
End;
//******************* TBtrieve.SetFieldDictName *************************

Procedure TBtrieve.SetFieldDictName(Value: String);
Begin
  If fActive Then
    If MessageDlg('You cannot change the dictionary Field name while it is still Active. Do you want to make it inactive then change the name?', mtConfirmation, [mbYes, mbCancel], 0) = mrYes Then
      Begin
        fActive := False;
        fField := Value;
        fActive := True;
      End
    Else
  Else
    fField := Value;
End;
//******************* TBtrieve.SetDPath *************************

Procedure TBtrieve.SetDPath(AFilename: TBtrGridFileName);
Begin
  If AFileName[Length(AFileName)] = '\' Then
    dPath := AFilename
  Else
    dPath := AFilename + '\';
End;
//******************* TBtrieve.SetBtrSource *************************

Procedure TBtrieve.SetBtrSource(Value: TBtr);
Begin
  If Value <> Nil Then
    BtrDB := Value
  Else
    BtrDB := Nil;
End;
//******************* TBtrieve.SetActive *************************

Procedure TBtrieve.SetActive(AActive: Boolean);
Var
  isError           : Boolean;
Begin
  If Assigned(FBeforeOpen) Then FBeforeOpen(Nil);
  isError := False;
  If Not fActive Then
    Begin
      FileDict.TableName := DictionaryPath + FileDictName;
      FileDict.IndexFieldNames := 'FileNo;ScalaModule';
      FieldDict.TableName := DictionaryPath + FieldDictName;
      FieldDict.IndexFieldNames := 'FileNo;FieldCode';
      FieldDict.MasterSource := fDataSource;
      FieldDict.MasterFields := 'FileNo';
      Try
        FileDict.Active := True;
      Except
        On E: Exception Do
          Begin
            MessageDlg('Cannot open File dictionary.'#13#10 + E.Message, mtError, [mbOK], 0);
            isError := True;
          End;
      End;
      Try
        FieldDict.Active := True;
      Except
        On E: Exception Do
          Begin
            MessageDlg('Cannot open Field dictionary.'#13#10 + E.Message, mtError, [mbOK], 0);
            isError := True;
          End;
      End;
    End
  Else
    Begin
      Try
        FileDict.Active := False;
      Except
        On E: Exception Do
          Begin
            MessageDlg('Cannot close File dictionary.'#13#10 + E.Message, mtError, [mbOK], 0);
            isError := True;
          End;
      End;
      Try
        FieldDict.Active := False;
      Except
        On E: Exception Do
          Begin
            MessageDlg('Cannot close Field dictionary.'#13#10 + E.Message, mtError, [mbOK], 0);
            isError := True;
          End;
      End;
    End;
  fActive := (AActive) And (Not isError);
  If fActive Then
    CreateDefList
  Else
    DestroyDefList;
End;
//******************* TBtrieve.SetFilesList *************************

Procedure TBtrieve.SetFilesList(Value: TStringList);
Begin
  fFilesList.CommaText := Value.CommaText;
End;
//******************* TBtrieve.SetFileIndex *************************

Procedure TBtrieve.SetFileIndex(Value: Integer);
Begin
  If (Value <= fFilesList.Count) And (Value >= 0) Then
    fIndex := Value
  Else
    fIndex := 0;
End;
//******************* TBtrieve.GetFieldInfo *************************

Function TBtrieve.GetFieldInfo(FileNo, FieldCode: String): BtrRecInfo;
Var
  Found             : Boolean;
  R                 : BtrRecInfo;
Begin
  fHeadDefList := fStartDefList;
  Found := False;
  R.FileHeader := '';
  R.FieldName := '';
  R.FieldAlias := '';
  R.AllowNULL := False;
  R.FieldType := asUnknown;
  R.Offset := -1;
  R.Size := 0;
  Try
    While (Not Found) And (fHeadDefList <> Nil) Do
      Begin
        If (fHeadDefList^.FileHeader = FileNo) And (fHeadDefList^.FieldName = FieldCode) Then
          Begin
            Found := True;
            R.FileHeader := FileNo;
            R.FieldName := FieldCode;
            R.Offset := fHeadDefList^.Offset;
            R.Size := fHeadDefList^.Size;
            R.FieldAlias := fHeadDefList^.FieldAlias;
            R.AllowNULL := fHeadDefList^.AllowNULL;
            R.FieldType := fHeadDefList^.FieldType;
          End
        Else
          fHeadDefList := fHeadDefList^.NextRec;
      End;
  Except
  End;
  Result := R;
End;
//******************* TBtrieve.GetFieldValue *************************

Function TBtrieve.GetFieldValue(FileNo, FieldCode: String; asType: TBtrieveTypes): Variant;
Var
  S                 : String;
  R                 : BtrRecInfo;
  V                 : Variant;
Begin
  V := ' ';
  Try
    If (fHeadDefList = Nil) Or (Not Assigned(BtrDB)) Or (Not BtrDb.Active) Then
      V := ' '
    Else
      Begin
        If (BtrDB.DataBuffer <> Nil) And (BtrDB.RecordCount > 0) Then
          S := strpas(BtrDB.DataBuffer)
        Else
          S := '';

        R := GetFieldInfo(FileNo, FieldCode);
        If R.Offset = -1 Then
          V := ' '
        Else
          Case asType Of
            AsString, asUnknown:
              Try
                V := Copy(S, R.Offset + 1, R.Size);
              Except
                V := ' ';
              End;
            AsInteger:
              Try
                V := StrToInt(Trim((Copy(S, R.Offset + 1, R.Size))));
              Except
                V := 0;
              End;
            asFloat:
              Try
                V := FloatAsSystem(Trim(Copy(S, R.Offset + 1, R.Size)));
              Except
                V := 0;
              End;
            asDate, asTime:
              Try
                V := DateAsSystem(Copy(S, R.Offset + 1, R.Size), Version);
              Except
                V := EncodeDate(1900, 1, 1);
              End;
            AsBoolean:
              Try
                V := BooleanAsSystem(Copy(S, R.Offset + 1, R.Size));
              Except
                V := False;
              End;
            asChar:
              Try
                V := Copy(S, R.Offset + 1, 1);
              Except
                V := ' ';
              End;
            Else
              V := ' ';
          End;
      End;
  Except
  End;
  Result := V;
End;
//******************* TBtrieve.SetFieldValue *************************

Function TBtrieve.SetFieldValue(FileNo, FieldCode: String; Value: Variant; asType: TBtrieveTypes): Integer;
Var
  R                 : BtrRecInfo;
  V, W              : TBuff;
Begin
  Result := -1;
  Try
    If (fHeadDefList = Nil) Or
      (Not Assigned(BtrDB)) Or
      (Not BtrDB.Active) Then
      Result := -1
    Else
      Begin
        R := GetFieldInfo(FileNo, FieldCode);
        If R.Offset = -1 Then
          V := ''
        Else
          Begin
            StrCopy(W, BtrDB.DataBuffer);
            Case asType Of
              asUnknown, AsString: V := SetBuff(W, Value, R.Offset, R.Size);
              AsInteger: V := SetBuff(W, IntToStr(Value), R.Offset, R.Size);
              AsBoolean:
                If Value Then
                  V := SetBuff(W, '1', R.Offset, R.Size)
                Else
                  V := SetBuff(W, '0', R.Offset, R.Size);
              asFloat: V := SetBuff(W, String(CurrAsScala(Value, 8)), R.Offset, R.Size);
              asDate: V := SetBuff(W, DateAsScala(Value, Ord(fDateFormat)), R.Offset, R.Size);
              asChar: V := SetBuff(W, Copy(String(Value), 1, 1), R.Offset, R.Size);
              Else
                V := '';
            End
          End;
      End;
    StrCopy(BtrDB.DataBuffer, V);
    BtrDB.Update;
    V := '';
    W := '';
    Result := 0;
  Except
  End;
End;
//******************* TBtrieve.FindKey *************************

Function TBtrieve.FindKey(FileNo, FieldCode: String; Value: Variant): Boolean;
Var
  KEYB              : CHARARRAY;
  R, R1, R2         : BtrRecInfo;
  S, S1, S2         : String;
  L                 : TStringList;
Begin
  If Not BtrDB.Active Then
    Result := False
  Else
    Begin
      If Pos(';', FieldCode) = 0 Then
        Begin
          R := GetFieldInfo(FileNo, FieldCode);
          StrPCopy(KEYB, Format('%-' + IntToStr(R.Size) + 's', [String(Value)]));
        End
      Else
        If Pos(';', Copy(FieldCode, Pos(';', FieldCode) + 1, Length(FieldCode) - Pos(';', FieldCode))) = 0 Then
        Begin
          Value := '"' + StringReplace(Value, ';', '","', [rfReplaceAll]) + '"';
          L := TStringList.Create;
          Try
            L.CommaText := Value;
            R := GetFieldInfo(FileNo, Copy(FieldCode, 1, 7));
            R1 := GetFieldInfo(FileNo, Copy(FieldCode, 9, 7));
            S := L.Strings[0];
            S1 := L.Strings[1];
          Finally
            L.Free;
          End;
          StrPCopy(KEYB, Format('%-' + IntToStr(R.Size) + 's%-' + IntToStr(R1.Size) + 's', [S, S1]));
        End
      Else
        Begin
          Value := '"' + StringReplace(Value, ';', '","', [rfReplaceAll]) + '"';
          L := TStringList.Create;
          Try
            L.CommaText := Value;
            R := GetFieldInfo(FileNo, Copy(FieldCode, 1, 7));
            R1 := GetFieldInfo(FileNo, Copy(FieldCode, 9, 7));
            R2 := GetFieldInfo(FileNo, Copy(FieldCode, 17, 7));
            S := L.Strings[0];
            S1 := L.Strings[1];
            S2 := L.Strings[2];
          Finally
            L.Free;
          End;
          StrPCopy(KEYB, Format('%-' + IntToStr(R.Size) + 's%-' + IntToStr(R1.Size) + 's%-' + IntToStr(R2.Size) + 's', [S, S1, S2]));
        End;
      StrCopy(BtrDB.KeyBuffer, KEYB);
      Try
        BtrDB.Equal;
        Result := True;
      Except
        On E: Exception Do
          Result := False;
      End;
    End;
End;
//******************* TBtrieve.GetFieldList *************************

Function TBtrieve.GetFieldList(fHeader: String): TStringList;
Var
  L                 : TStringList;
Begin
  L := TStringList.Create;
  Try
    If fHeadDefList = Nil Then
      Result.CommaText := L.CommaText
    Else
      Begin
        fHeadDefList := fStartDefList;
        While fHeadDefList <> Nil Do
          Begin
            If fHeadDefList^.FileHeader = fHeader Then
              L.Add(fHeadDefList^.FieldName);
            fHeadDefList := fHeadDefList^.NextRec;
          End;
        Result.CommaText := L.CommaText;
        fHeadDefList := fStartDefList;
      End;
  Finally
    L.Free;
  End;
End;
//******************* TBtrGridFileNameProperty.GetAttributes *************************

Function TBtrGridFileNameProperty.GetAttributes: TPropertyAttributes;
Begin
  Result := [paDialog, paMultiSelect, paAutoUpdate];
End;
//******************* TBtrGridFileNameProperty.Edit *************************

Procedure TBtrGridFileNameProperty.Edit;
Var
  Dir               : String;
Begin
  Dir := GetStrValue;
  If SelectDirectory(Dir, [sdAllowCreate, sdPerformCreate, sdPrompt], 0) Then
    SetStrValue(Dir);
End;

Procedure Register;
Begin
  RegisterComponents('EM-Quicksoft', [TBtrieve]);
  RegisterPropertyEditor(TypeInfo(TBtrGridFileName), Nil, '', TBtrGridFileNameProperty);
  RegisterPropertyEditor(TypeInfo(TBtrAbout), Nil, '', TBtrAbout);
End;

End.

