{***************************************************************
 *
 * Unit Name : BtrLib
 * Purpose   : Btrieve Utils Library
 * Author    : Eugen Mihailescu
 * Company   : EM Quicksoft Romania SRL
 * Copyright : 1998,2002  All rights reserved.
 * Web Site  : http://www.em-quicksoft.com
 * Support   : support@em-quicksoft.com
 * History   : 03/19/2000
 *
 * Disclaimer: The objectives of this program is only educational.
 *             You MUST NOT use it without getting any material benefit on it
 *             You have the right to share this code with others without removing
 *             the copyright and/or disclaimer notice.
 *             You are allowed to include its code in your application
 *             only if it was designed for non commercial purpose,
 *             otherwise we claim $10 for each copy of your
 *             application which contains this code inside.
 ****************************************************************}

Unit BtrLib;
Interface
Uses Graphics,
  ExtCtrls,
  Forms,
  Btr,
  SysUtils,
  Classes,
  Windows,
  IniFiles,
  Registry,
  Dialogs,
  DBTables,
  Db,
  BtrExcep,
  Controls,
  DbiProcs,
  DbiTypes,
  DbiErrs,
  Shellapi,
  Printers,
  Math,
  richedit;
Type
  TReport = Procedure(ReportName: String); stdcall;
  TEditReport = Procedure(ReportName, DatabaseName: String); stdcall;
  FldInfo = Record
    Position: Integer;
    Len: Integer;
    Info: PChar;
  End;
  TLicenseStr = Record
    Index: Integer;
    StrOption: String;
    ValidDate: TDateTime
  End;
Const
  SCALA_DEC_SEPARATOR: char = '.';

Function ppIconFromResource(aBitmapName: String): THandle;
Function myMessageDlg(Const Caption, Msg: String; AType: TMsgDlgType; AButtons: TMsgDlgButtons): Word;
Function myConfirmDelete(Const Caption, Msg: String; AType: TMsgDlgType; AButtons: TMsgDlgButtons): Word;
Function myConfirmRemove(Const Caption, Msg: String; AType: TMsgDlgType; AButtons: TMsgDlgButtons): Word;

Procedure ShowAbout;
Function GetRecordNumber(Table: TTable): Longint;
Function StrToNr(Nr: String): Longint;  // Converts string like: "000234..." to longint as 234...
Function StrToNrFloat(Nr: String): Real; // Converts string like: "000234.134..." to float as 234.123...
Function CurrWithoutSep(S: String): String; //Converts a string like 123,456,345.12 to 123456345.12
Function GetScalaPath: String;
Function GetScalaVerInfo: String;
Function GetCompInfo(BtrComp: TBtr): PChar;
Function SetBuffer(Buffer: PChar; Info: Variant; Pos, Len: Integer):
  PChar;
Function DateAsScala(vDate: TDate; ScalaVer: Byte): String;
Function CurrAsScala(vValue: Real; vDecimals: Byte): String;
Function DateAsSystem(vDate: String; ScalaVer: Byte): TDate;
Function CurrAsSystem(vValue: String): Currency;
Function FloatAsSystem(vValue: String): Real;
Function IntAsSystem(vValue: String): Longint;
Function HexaAsSystem(vValue: String): String;
Function HexaAsScala(vValue: String): String;
Function BooleanAsSystem(vValue: String): Boolean;
Function BooleanAsScala(vValue: Boolean): String;
Function GetScalaFieldValue(FileNo, FieldCode: PChar; BtrFile: TBtr):
  FldInfo;
Function SetScalaFieldValue(FileNo, FieldCode: PChar; BtrFile: TBtr;
  Value: Variant): Integer;
Procedure ApplySign;
Procedure ReplaceSC(Var Source: AnsiString; Const Target, Replace:
  AnsiString; CFlg: Boolean);
Function GetCommaList(Lista: String): String;
Function LicEnCrypt(Index: Integer; StrOption: String; ValidDate: TDateTime): String;
Function LicDeCrypt(LicenseCode: String; OptionCount: Integer): TLicenseStr;
Function HexToInt(Value: String): Integer;
Function IntToHex(Value: Integer): String;
Function ChangeAliasPath(AliasName, Path: String; canCreate: Boolean): Boolean;
Function SendToRecycleBin(Title: String; FileName: String): Boolean;
Function myRenameFile(Title: String; OldFileName, NewFileName: String): Boolean;
Function myMoveFile(Title: String; OldLocation, NewLocation: String): Boolean;
Function myCopyFile(Title: String; FromLocation, ToLocation: String): Boolean;
Function Soundex(OriginalWord: String): String;
Function SoundAlike(Word1, Word2: String): Boolean;
Function ReplaceSub(Str, sub1, sub2: String): String;
Function GetLongFileName(Const FileName: String): String;
Function GetShortFileName(Const FileName: String): String;
Function GetLastAccessedTime(FileName: String): TDateTime;
Function CreateMultipleVolume(FileName: String; VolumeSize: Integer): String;
Function ExtractMultipleVolume(FileName: String): String;
Function FilesCreated(Files: String): Boolean;
Procedure PrintStrings(Strings: TStrings);
Procedure PrintDocument(Lines: TStringList; Const Caption: String);
Function GetHDSerialNo: pdword;
Function GetDateDiff(MaxDate, MinDate: TDate): Integer;
Function LoadReport(ReportName: String): Boolean;
Function EditReport(ReportName, DatabaseName: String): Boolean;
Implementation
{$R resScalaLib.res}

{This function will returns a string list contains Scala RV6 & Scala 5 PATH
 separated by comma}
//******************* PrintDocument *************************

Procedure PrintDocument(Lines: TStringList; Const Caption: String);
Var
  Range             : TFormatRange;
  PageRect          : TRect;
  LastChar, MaxLen, LogX, LogY, OldMap: Integer;
Begin
  PageRect := Rect(0, 0, 0, 0);
  FillChar(Range, SizeOf(TFormatRange), 0);
  With Printer, Range Do
    Begin
      Title := Caption;
      BeginDoc;
      hdc := Handle;
      hdcTarget := hdc;
      LogX := GetDeviceCaps(Handle, LOGPIXELSX);
      LogY := GetDeviceCaps(Handle, LOGPIXELSY);
      If IsRectEmpty(PageRect) Then
        Begin
          rc.Right := PageWidth * 1440 Div LogX;
          rc.Bottom := PageHeight * 1440 Div LogY;
        End
      Else
        Begin
          rc.Left := PageRect.Left * 1440 Div LogX;
          rc.Top := PageRect.Top * 1440 Div LogY;
          rc.Right := PageRect.Right * 1440 Div LogX;
          rc.Bottom := PageRect.Bottom * 1440 Div LogY;
        End;
      rcPage := rc;
      LastChar := 0;
      MaxLen := Length(Lines.Text);
      chrg.cpMax := -1;
      // ensure printer DC is in text map mode
      OldMap := SetMapMode(hdc, MM_TEXT);
      SendMessage(Handle, EM_FORMATRANGE, 0, 0); // flush buffer
      Try
        Repeat
          chrg.cpMin := LastChar;
          LastChar := SendMessage(Handle, EM_FORMATRANGE, 1, Longint(@Range));
          If (LastChar < MaxLen) And (LastChar <> -1) Then NewPage;
          Application.ProcessMessages;
        Until (LastChar >= MaxLen) Or (LastChar <= 0);
        EndDoc;
      Finally
        SendMessage(Handle, EM_FORMATRANGE, 0, 0); // flush buffer
        SetMapMode(hdc, OldMap);        // restore previous map mode
      End;
    End;
End;
//******************* PrintStrings *************************

Procedure PrintStrings(Strings: TStrings);
Var
  Prn               : TextFile;
  I                 : Word;
Begin
  AssignPrn(Prn);
  Try
    Rewrite(Prn);
    Try
      For I := 0 To Strings.Count - 1 Do
        writeln(Prn, Strings.Strings[I]);
    Finally
      CloseFile(Prn);
    End;
  Except
    On EInOutError Do
      MessageDlg('Error Printing text.', mtError, [mbOK], 0);
  End;
End;
//******************* GetScalaPath *************************

Function GetScalaPath: String;
Var
  WinDir            : ShortString;
  SFWFile, SCALAFile: TIniFile;
  Path              : String;
  S51               : TRegistry;
Begin
  Path := '';
  GetWindowsDirectory(@WinDir[1], SizeOf(WinDir) - 1);
  WinDir[0] := Chr(StrLen(@WinDir[1]));
  If Length(WinDir) > 3 Then
    WinDir := UpperCase(WinDir + '\');

  S51 := TRegistry.Create;
  Try
    S51.RootKey := HKEY_LOCAL_MACHINE;
    If S51.OpenKey('SOFTWARE\Classes\Sfw.Document\protocol\StdFileEditing\server', False) Then
      Path := ExtractFilePath(S51.ReadString(''))
    Else
      Begin
        If fileexists(WinDir + 'SFW.INI') Then
          Begin
            SFWFile := TIniFile.Create(WinDir + 'SFW.INI');
            Path := SFWFile.ReadString('SFWDir', 'BTR', '');
          End
        Else
          If fileexists(WinDir + 'SCALA.INI') Then
          Begin
            SCALAFile := TIniFile.Create(WinDir + 'SCALA.INI');
            Path := ExtractFilePath(SCALAFile.ReadString('Tools',
              'SpoolProg', ''));
          End;
      End;
    Result := Path;
  Finally
    S51.Free;
  End;
End;
{This function will returns a string list contains Info about current Scala's version(s)}
//******************* GetScalaVerInfo *************************

Function GetScalaVerInfo: String;
Var
  W95Reg            : TRegistry;
  Info              : TStringList;
  S                 : String;
Begin
  Info := TStringList.Create;
  W95Reg := TRegistry.Create;
  Try
    W95Reg.RootKey := HKEY_LOCAL_MACHINE;
    If W95Reg.OpenKey('SOFTWARE\Scala Business Solutions NV', False
      ) Then
      W95Reg.GetKeyNames(Info);
  Finally
    W95Reg.Free;
  End;
  S := Info.CommaText;
  Info.Free;
  Result := S;
End;
//******************* GetCompInfo *************************
{This function will returns a string list contains the info structure list bellow:

POSITION  LENGTH  DESCRIPTION
----------------------------------
1         2       Company Code
3         30      Company Name
33        2       Financial Year 1
35        2       Financial Year 2
37        2       Financial Year 3
39        2       Financial Year 4
41        2       Financial Year 5

Each string info will be terminated by CR/LF pair}

Function GetCompInfo(BtrComp: TBtr): PChar;
Var
  Info              : String;
  RecStr            : String;
  I                 : Integer;
  //  Path              : String;
Begin
  Info := '';
  For I := 1 To BtrComp.RecordCount Do
    Begin
      RecStr := String(BtrComp.DataBuffer);
      Info := Info + Copy(RecStr, 1, 2) + ',' + Copy(RecStr,
        3, 30) + ',' + Copy(RecStr, 174, 2) + ',' + Copy(RecStr, 176,
        2) + ',' + Copy(RecStr, 178, 2) + ',' + Copy(RecStr, 180, 2)
        + ',' + Copy(RecStr, 182, 2) + #13#10;
      Try
        BtrComp.Next;
      Except
      End;
    End;
  Result := PChar(Info);
End;
//******************* SetBuffer *************************
{This function will returns a PChar result modified with new Info chars from position Pos,
with length Len}

Function SetBuffer(Buffer: PChar; Info: Variant; Pos, Len: Integer):
  PChar;
Begin
  Result := PChar(Copy(Buffer, 1, Pos - 1) + Format('%-s' + IntToStr(Len
    ), [Info]) + Copy(Buffer, Pos + Len, StrLen(Buffer) - Pos + Len - 1
    ));
End;
//******************* DateAsScala *************************
{This function will returns a String result converted from vDate variable base on
ScalaVer constant; ScalaVer=0 if Scala RV6; ScalaVer=1 if Scala 5}

Function DateAsScala(vDate: TDate; ScalaVer: Byte): String;
Var
  Y, M, D           : Word;
Begin
  DecodeDate(vdate, Y, M, D);
  Case ScalaVer Of
    0:
      If ((Y = 1900) And (M = 1) And (D = 1)) Then
        Result := '999999'
      Else
        Result := FormatDateTime('yymmdd', vDate);
    1:
      If ((Y = 1900) And (M = 1) And (D = 1)) Then
        Result := '99999999'
      Else
        Result := FormatDateTime('yyyymmdd', vDate);
    Else
      myMessageDlg('Application Error',
        'Invalid version of vDate parameter in library ScalaLib.dll for function DateAsScala.', mtError, [mbOK]);
  End;
End;
//******************* CurrAsScala *************************
{This function will returns a String result converted from vValue with vDecimals decimals,
using current SCALA_DEC_SEPARATOR decimal separator}

Function CurrAsScala(vValue: Real; vDecimals: Byte): String;
Begin
  Result := FormatFloat('0.########', vValue);
End;
//******************* DateAsSystem *************************
{This function will returns a TDate value obtains from vDate using current ScalaVer
Scala's version; ScalaVer=0 if Scala RV6; ScalaVer=1 if Scala 5}

Function DateAsSystem(vDate: String; ScalaVer: Byte): TDate;

Var
  Y, M, D           : Word;
Begin
  vDate := Trim(vDate);
  Try
    Case ScalaVer Of
      0:
        Begin
          Y := 1900 + StrToInt(Copy(vDate, 1, 2));
          M := StrToInt(Copy(vDate, 3, 2));
          D := StrToInt(Copy(vDate, 5, 2));
          Result := EncodeDate(Y, M, D);
        End;
      1:
        Begin
          If Length(vDate) >= 4 Then
            Y := StrToInt(Copy(vDate, 1, 4))
          Else
            Y := 1900;
          If Length(vDate) >= 6 Then
            M := StrToInt(Copy(vDate, 5, 2))
          Else
            M := 1;
          If Length(vDate) >= 8 Then
            D := StrToInt(Copy(vDate, 7, 2))
          Else
            D := 1;
          Result := EncodeDate(Y, M, D);
        End;
      Else
        Result := EncodeDate(1900, 1, 1);
    End;
  Except
    Result := EncodeDate(1900, 1, 1);
  End;
End;
//******************* CurrAsSystem *************************
{This function will returns a Real value obtains from vValue using current
SCALA_DEC_SEPARATOR decimal separator. If not valid separator, RESULT=0}

Function CurrAsSystem(vValue: String): Currency;
Begin
  If Trim(vValue) = '' Then vValue := '0';
  If Pos(SCALA_DEC_SEPARATOR, vValue) > 0 Then
    Begin
      CurrencyDecimals := Length(vValue) - Pos(SCALA_DEC_SEPARATOR, vValue
        );
      Result := StrToCurr(Copy(vValue, 1, Pos(SCALA_DEC_SEPARATOR, vValue)
        - 1) + DecimalSeparator + Copy(vValue, Pos(SCALA_DEC_SEPARATOR,
        vValue) + 1, Length(vValue) - Pos(SCALA_DEC_SEPARATOR, vValue)))
    End
  Else
    Result := StrToCurr(vValue);
End;
//******************* IntAsSystem *************************

Function IntAsSystem(vValue: String): Longint;
Begin
  vValue := Trim(vValue);
  If Trim(vValue) = '' Then vValue := '0';
  If Pos(SCALA_DEC_SEPARATOR, vValue) > 0 Then
    Begin
      CurrencyDecimals := Length(vValue) - Pos(SCALA_DEC_SEPARATOR, vValue);
      Result := StrToInt(Copy(vValue, 1, Pos(SCALA_DEC_SEPARATOR, vValue)
        - 1) + DecimalSeparator + Copy(vValue, Pos(SCALA_DEC_SEPARATOR,
        vValue) + 1, Length(vValue) - Pos(SCALA_DEC_SEPARATOR, vValue)))
    End
  Else
    Result := StrToInt(vValue);
End;
//******************* FloatAsSystem *************************

Function FloatAsSystem(vValue: String): Real;
Begin
  If Trim(vValue) = '' Then vValue := '0';
  If Pos(SCALA_DEC_SEPARATOR, vValue) > 0 Then
    Begin
      CurrencyDecimals := Length(vValue) - Pos(SCALA_DEC_SEPARATOR, vValue
        );
      Result := StrToFloat(Copy(vValue, 1, Pos(SCALA_DEC_SEPARATOR, vValue)
        - 1) + DecimalSeparator + Copy(vValue, Pos(SCALA_DEC_SEPARATOR,
        vValue) + 1, Length(vValue) - Pos(SCALA_DEC_SEPARATOR, vValue)))
    End
  Else
    Result := StrToFloat(vValue);
End;
//******************* ShowAbout *************************

Procedure ShowAbout;
Begin
  myMessageDlg('About Library Author', 'Scala Library for Delphi 3'#13#10'Author: Eugen Mihailescu'#13#10'Company: Scala Business Solutions SRL'#13#10'Last update: July 10, 1998'#13#10'Copyright  1998, 2020 Eugen Mihailescu'#13#10'All rights reserved.',
    mtInformation, [mbOK]);
End;
//******************* GetScalaFieldValue *************************

Function GetScalaFieldValue(FileNo, FieldCode: PChar; BtrFile: TBtr):
  FldInfo;
Var
  DbFile, DbField   : TTable;
  DbDefinitions     : TDatabase;
  Session           : TSession;
  DSFile            : TDataSource;
  Position, Len     : Integer;
  Found             : Boolean;
  Info              : Pointer;
  Fld               : FldInfo;
Begin
  Session := TSession.Create(Nil);
  DbDefinitions := TDatabase.Create(Nil);
  DbFile := TTable.Create(Nil);
  DbField := TTable.Create(Nil);
  DSFile := TDataSource.Create(Nil);
  Fld.Position := 0;
  Fld.Len := 0;
  Fld.Info := Nil;
  Try
    Session.AutoSessionName := True;
    If Session.IsAlias('ScalaDEF') Then
      Begin
        DSFile.DataSet := DbFile;
        DbDefinitions.AliasName := 'ScalaDEF';
        DbDefinitions.DatabaseName := 'TableDEF';
        DbFile.DatabaseName := 'TableDEF';
        DbField.DatabaseName := 'TableDEF';
        DbField.TableName := 'DBFIELD.db';
        DbFile.TableName := 'DBFILE.db';
        DbField.IndexFieldNames := 'FILENO;FIELDCODE';
        DbFile.IndexFieldNames := 'FILENO;SCALAMODULE';
        DbField.MasterSource := DSFile;
        DbField.MasterFields := 'FILENO';
        DbField.Active := True;
        DbFile.Active := True;
        Found := DBFile.FindKey([FileNo]);
        If Found Then
          Begin
            Found := DbField.FindKey([FileNo, FieldCode]);
            If Found Then
              Begin
                Position := DBField['Offset'];
                Len := DbField['Size'];
                Info := BtrFile.DataBuffer;
                Fld.Position := Position;
                Fld.Len := Len;
                Fld.Info := PChar(Copy(String(Info), Position, Len));
              End
          End;
        DbField.Active := False;
        DbFile.Active := False;
      End;
    Result := Fld;
  Finally
    Session.Free;
    DbDefinitions.Free;
    DbField.Free;
    DbFile.Free;
    DSFile.Free;
  End;
End;
//******************* SetScalaFieldValue *************************

Function SetScalaFieldValue(FileNo, FieldCode: PChar; BtrFile: TBtr;
  Value: Variant): Integer;
Var
  DbFile, DbField   : TTable;
  DbDefinitions     : TDatabase;
  Session           : TSession;
  DSFile            : TDataSource;
  Position, Len     : Integer;
  Found             : Boolean;
Begin
  Session := TSession.Create(Nil);
  DbDefinitions := TDatabase.Create(Nil);
  DbFile := TTable.Create(Nil);
  DbField := TTable.Create(Nil);
  DSFile := TDataSource.Create(Nil);
  Try
    Session.AutoSessionName := True;
    If Session.IsAlias('ScalaDEF') Then
      Begin
        DSFile.DataSet := DbFile;
        DbDefinitions.AliasName := 'ScalaDEF';
        DbDefinitions.DatabaseName := 'TableDEF';
        DbFile.DatabaseName := 'TableDEF';
        DbField.DatabaseName := 'TableDEF';
        DbField.TableName := 'DBFIELD.db';
        DbFile.TableName := 'DBFILE.db';
        DbField.IndexFieldNames := 'FILENO;FIELDCODE';
        DbFile.IndexFieldNames := 'FILENO;SCALAMODULE';
        DbField.MasterSource := DSFile;
        DbField.MasterFields := 'FILENO';
        DbField.Active := True;
        DbFile.Active := True;
        Found := DBFile.FindKey([FileNo]);
        If Found Then
          Begin
            Found := DbField.FindKey([FileNo, FieldCode]);
            If Found Then
              Begin
                Position := DBField['Offset'];
                Len := DbField['Size'];
                StrCopy(BtrFile.DataBuffer, SetBuffer(PChar(
                  BtrFile.DataBuffer), Value, Position, Len));
                Try
                  BtrFile.Update;
                  Result := 0;
                Except
                  On E: BtrException Do
                    Result := E.StatusId;
                End;
              End
            Else
              Result := -1;
            //Meaning that Dictionary Field has no key for [FileNo,FieldCode]
          End
        Else
          Result := -2;
        //Meaning that Dictionary File has no key for [FileNo]
        DbField.Active := False;
        DbFile.Active := False;
      End
    Else
      Result := -3;
    //Meaning that BDE has no alias called 'ScalaDEF'
  Finally
    Session.Free;
    DbDefinitions.Free;
    DbField.Free;
    DbFile.Free;
    DSFile.Free;
  End;
End;
//******************* StrToNr *************************

Function StrToNr(Nr: String): Longint;
Var
  I                 : Integer;
  not_zero          : Boolean;
Begin
  not_zero := False;
  For I := 1 To Length(nr) Do
    If nr[I] <> '0' Then
      Begin
        not_zero := True;
        Break;
      End;
  If Not not_zero Then
    Result := 0
  Else
    Begin
      If (nr = '0') Or (nr = '00') Then
        Result := 0
      Else
        If (nr <> '') And (nr <> '00') Then
        While (nr[1] = '0') And (Length(nr) > 0) Do
          nr := Copy(nr, 2, Length(nr) - 1);
      If Trim(nr) <> '' Then
        Result := StrToInt(nr)
      Else
        Result := 0;
    End;
End;
//******************* StrToNrFloat *************************

Function StrToNrFloat(Nr: String): Real;
Begin
  If nr = '0' Then
    Result := 0
  Else
    If (nr <> '') And (nr <> '00') Then
    While (nr[1] = '0') And (Length(nr) > 0) Do
      nr := Copy(nr, 2, Length(nr) - 1);

  If Trim(nr) <> '' Then
    Result := FloatAsSystem(nr)
  Else
    Result := 0;
End;
//******************* CurrWithoutSep *************************

Function CurrWithoutSep(S: String): String;
Var
  I                 : Byte;
Begin
  I := 1;
  While I <= Length(S) Do
    Begin
      I := Pos(ThousandSeparator, S);
      If I = 0 Then Break;
      S := Copy(S, 1, I - 1) + Copy(S, I + 1, Length(S) - I + 1);
    End;
End;
//******************* GetRecordNumber *************************

Function GetRecordNumber(Table: TTable): Longint;
Var
  CursorProps       : CURProps;
  RecordProps       : RECProps;
Begin                                   { Return 0 if dataset is not Paradox or dBASE }
  Result := 0;
  With Table Do
    Begin                               { Is the dataset active? }
      If State = dsInactive Then
        Raise EDatabaseError.Create('Cannot perform this operation ' +
          'on a closed dataset');
      { We need to make this call to grab the cursor's iSeqNums }

      Check(DbiGetCursorProps(Handle, CursorProps));
      { Synchronize the BDE cursor with the Dataset's cursor }
      UpdateCursorPos;
      { Fill RecordProps with the current record's properties }
      Check(DbiGetRecord(Handle, dbiNOLOCK, Nil, @RecordProps));
      { What kind of dataset are we looking at? } Case CursorProps.iSeqNums Of
        0: Result := RecordProps.iPhyRecNum; { dBASE   }
        1: Result := RecordProps.iSeqNum; { Paradox }
      End;
    End;
End;
//******************* ApplySign *************************

Procedure ApplySign;
Var
  T                 : TSignFrm;
  A, L, Z           : Word;
Begin
  A := 1998;
  L := 12;
  Z := 31;
  If Now > EncodeDate(A, L, Z) Then
    Begin
      T := TSignFrm.Create(Nil);
      Try
        T.ShowModal;
        If T.TotDays.Progress = 0 Then
          While 1 = 1 Do
            Begin
              T.ShowModal;
            End;
      Finally
        T.Free;
      End;
    End;
End;
//******************* ReplaceSC *************************

Procedure ReplaceSC(Var Source: AnsiString; Const Target, Replace:
  AnsiString; CFlg: Boolean);
Var
  I                 : Integer;
Begin
  I := Pos(Target, Source);
  While I > 0 Do
    Begin
      If I > 0 Then
        Source := Copy(Source, 1, I - 1) + Replace + Copy(Source, I + Length
          (Target), Length(Source) - I + 1 - Length(Target));
      I := Pos(Target, Source);
    End;
End;
//******************* GetCommaList *************************

Function GetCommaList(Lista: String): String;
Var
  L                 : TStringList;
  S                 : String;
  I                 : Integer;
Begin
  S := '(';
  L := TStringList.Create;
  Try
    L.CommaText := Lista;
    For I := 0 To L.Count - 2 Do
      S := S + '''' + L.Strings[I] + ''',';
    If L.Count > 0 Then S := S + '''' + L.Strings[L.Count - 1] + '''';
    S := S + ')';
  Finally
    L.Free;
  End;
  Result := S;
End;
//******************* HexaAsSystem *************************

Function HexaAsSystem(vValue: String): String;
Var
  I                 : Integer;
  Nr                : Real;
Begin
  Nr := 0;
  For I := 1 To Length(vValue) Do
    Nr := Nr + Power(16, (I - 1)) * Ord(vValue[I]);
  Result := FloatToStr(Nr);
End;
//******************* HexaAsScala *************************

Function HexaAsScala(vValue: String): String;
Var
  V1, V2            : Integer;
Begin
  vValue := Trim(vValue);
  V2 := StrToInt(vValue) Div 16;
  V1 := StrToInt(vValue) Mod 16;
  Result := char(V1) + char(V2);
End;
//******************* BooleanAsSystem *************************

Function BooleanAsSystem(vValue: String): Boolean;
Begin
  Result := vValue = '1';
End;
//******************* BooleanAsScala *************************

Function BooleanAsScala(vValue: Boolean): String;
Begin
  If vValue Then
    Result := '1'
  Else
    Result := '0';
End;
//******************* ppIconFromResource *************************

Function ppIconFromResource(aBitmapName: String): THandle;
Var
  lpLibModule       : PLibModule;

Begin

  Result := 0;

  lpLibModule := LibModuleList;

  While (lpLibModule <> Nil) And (Result = 0) Do
    Begin

      Result := LoadIcon(lpLibModule.ResInstance, PChar(aBitmapName));
      lpLibModule := lpLibModule.Next;
    End;

End;
//******************* myMessageDlg *************************

Function myMessageDlg(Const Caption, Msg: String; AType: TMsgDlgType; AButtons: TMsgDlgButtons): Word;
Var
  lDialog           : TForm;
  lImage            : TComponent;
  aIconHandle       : Longint;
Begin

  If (AType = mtCustom) Then
    AType := mtError;

  lDialog := CreateMessageDialog(Msg, AType, AButtons);

  If (Caption <> '') Then
    lDialog.Caption := Caption;

  aIconHandle := ppIconFromResource('SCALA5');

  If (aIconHandle <> 0) Then
    Begin
      lImage := lDialog.FindComponent('Image');

      If (lImage <> Nil) Then
        TImage(lImage).Picture.Icon.Handle := aIconHandle;
    End;

  Result := lDialog.ShowModal;

  lDialog.Free;

End;                                    {function, myMessageDlg}
//******************* myConfirmDelete *************************

Function myConfirmDelete(Const Caption, Msg: String; AType: TMsgDlgType; AButtons: TMsgDlgButtons): Word;
Var
  lDialog           : TForm;
  lImage            : TComponent;
  aIconHandle       : Longint;
Begin

  If (AType = mtCustom) Then
    AType := mtError;

  lDialog := CreateMessageDialog(Msg, AType, AButtons);

  If (Caption <> '') Then
    lDialog.Caption := Caption;

  aIconHandle := ppIconFromResource('REMOVE');

  If (aIconHandle <> 0) Then
    Begin
      lImage := lDialog.FindComponent('Image');

      If (lImage <> Nil) Then
        TImage(lImage).Picture.Icon.Handle := aIconHandle;
    End;

  Result := lDialog.ShowModal;

  lDialog.Free;

End;                                    {function, myMessageDlg}
//******************* myConfirmRemove *************************

Function myConfirmRemove(Const Caption, Msg: String; AType: TMsgDlgType; AButtons: TMsgDlgButtons): Word;
Var
  lDialog           : TForm;
  lImage            : TComponent;
  aIconHandle       : Longint;
Begin

  If (AType = mtCustom) Then
    AType := mtError;

  lDialog := CreateMessageDialog(Msg, AType, AButtons);

  If (Caption <> '') Then
    lDialog.Caption := Caption;

  aIconHandle := ppIconFromResource('RECYCLE');

  If (aIconHandle <> 0) Then
    Begin
      lImage := lDialog.FindComponent('Image');

      If (lImage <> Nil) Then
        TImage(lImage).Picture.Icon.Handle := aIconHandle;
    End;

  Result := lDialog.ShowModal;

  lDialog.Free;

End;                                    {function, myMessageDlg}
//******************* IntToHex *************************

Function IntToHex(Value: Integer): String;
Begin
  Result := Format('%0x', [Value]);
End;
//******************* HexToInt *************************

Function HexToInt(Value: String): Integer;
Const
  HEX               : Array['A'..'F'] Of Integer = (10, 11, 12, 13, 14, 15);
Var
  vInt, I           : Integer;
Begin
  vInt := 0;
  For I := 1 To Length(Value) Do
    If Value[I] < 'A' Then
      vInt := vInt * 16 + Ord(Value[I]) - 48
    Else
      vInt := vInt * 16 + HEX[Value[I]];
  Result := vInt;
End;
//******************* LicEnCrypt *************************

Function LicEnCrypt(Index: Integer; StrOption: String; ValidDate: TDateTime): String;
Var
  S                 : String;
  A, L, Z           : Word;
  I                 : Integer;
  V                 : Real;
Begin
  S := IntToHex(Index) + '-';
  V := 0;
  For I := 1 To Length(StrOption) Do
    V := V + StrToInt(StrOption[I]) * IntPower(32, Length(StrOption) - I);
  S := S + IntToHex(Round(V)) + '-';
  DecodeDate(ValidDate, A, L, Z);
  V := A * IntPower(32, 2) + L * IntPower(32, 1) + Z * IntPower(32, 0);
  S := S + IntToHex(Round(V));
  Result := S;
End;
//******************* LicDeCrypt *************************

Function LicDeCrypt(LicenseCode: String; OptionCount: Integer): TLicenseStr;
Var
  R                 : TLicenseStr;
  S1, S2, S3        : String;
  I                 : Integer;
  n                 : Longint;
  A, L, Z           : Word;
Begin
  S1 := Copy(LicenseCode, 1, Pos('-', LicenseCode) - 1);
  S2 := Copy(LicenseCode, Pos('-', LicenseCode) + 1, Length(LicenseCode) - Pos('-', LicenseCode));
  S3 := Copy(S2, Pos('-', S2) + 1, Length(S2) - Pos('-', S2));
  S2 := Copy(S2, 1, Pos('-', S2) - 1);
  R.Index := HexToInt(S1);
  n := HexToInt(S2);
  R.StrOption := '';
  For I := OptionCount Downto 1 Do
    Begin
      If n Div Round(intpower(32, I - 1)) > 0 Then
        Begin
          n := n Mod Round(intpower(32, I - 1));
          R.StrOption := R.StrOption + '1';
        End
      Else
        R.StrOption := R.StrOption + '0';
    End;
  A := HexToInt(S3) Div Trunc(IntPower(32, 2));
  L := (HexToInt(S3) - A * Trunc(IntPower(32, 2))) Div Trunc(IntPower(32, 1));
  Z := (HexToInt(S3) - A * Trunc(IntPower(32, 2)) - L * Trunc(IntPower(32, 1))) Div Trunc(IntPower(32, 0));
  R.ValidDate := EncodeDate(A, L, Z);
  Result := R;
End;
//******************* ChangeAliasPath *************************

Function ChangeAliasPath(AliasName, Path: String; canCreate: Boolean): Boolean;
Var
  AParams           : TStringList;
Begin
  If Not Session.IsAlias(AliasName) Then
    If canCreate Then
      Session.AddStandardAlias(AliasName, Path, 'PARADOX')
    Else
      Begin
        Result := False;
        Exit;
      End;
  AParams := TStringList.Create;
  Try
    AParams.Clear;
    AParams.Add('PATH=' + Path);
    Session.ModifyAlias(AliasName, AParams);
    Session.SaveConfigFile;
    Result := True;
  Finally
    AParams.Free;
  End;
End;
//******************* SendToRecycleBin *************************

Function SendToRecycleBin(Title: String; FileName: String): Boolean;
Var
  T                 : TSHFileOpStruct;
Begin
  With T Do
    Begin
      Wnd := 0;
      wFunc := FO_DELETE;
      pFrom := Pansichar(FileName);
      pTo := PChar('');
      fFlags := FOF_ALLOWUNDO Or FOF_NOCONFIRMATION Or FOF_SIMPLEPROGRESS Or FOF_NOERRORUI;
      lpszProgressTitle := PChar(Title);
    End;
  Try
    SHFileOperation(T);
  Except
  End;
End;
//******************* myRenameFile *************************

Function myRenameFile(Title: String; OldFileName, NewFileName: String): Boolean;
Var
  T                 : TSHFileOpStruct;
Begin
  With T Do
    Begin
      Wnd := 0;
      wFunc := FO_RENAME;
      pFrom := Pansichar(OldFilename);
      pTo := pansichar(NewFileName);
      fFlags := FOF_ALLOWUNDO Or FOF_NOCONFIRMATION Or FOF_SIMPLEPROGRESS Or FOF_NOERRORUI;
      lpszProgressTitle := PChar(Title);
    End;
  Try
    SHFileOperation(T);
  Except
  End;
End;
//******************* myMoveFile *************************

Function myMoveFile(Title: String; OldLocation, NewLocation: String): Boolean;
Var
  T                 : TSHFileOpStruct;
Begin
  With T Do
    Begin
      Wnd := 0;
      wFunc := FO_MOVE;
      pFrom := Pansichar(OldLocation);
      pTo := pansichar(NewLocation);
      fFlags := FOF_ALLOWUNDO Or FOF_NOCONFIRMATION Or FOF_SIMPLEPROGRESS Or FOF_NOERRORUI;
      lpszProgressTitle := PChar(Title);
    End;
  Try
    SHFileOperation(T);
  Except
  End;
End;
//******************* Soundex *************************

Function Soundex(OriginalWord: String): String;
Var
  Tempstring1, Tempstring2: String;
  Count             : Integer;
Begin
  Tempstring1 := '';
  Tempstring2 := '';
  OriginalWord := UpperCase(OriginalWord); {Make original word uppercase}
  Appendstr(Tempstring1, OriginalWord[1]); {Use the first letter of the word}
  For Count := 2 To Length(OriginalWord) Do
    {Assign a numeric value to each letter, except the first}

    Case OriginalWord[Count] Of
      'B', 'F', 'P', 'V': Appendstr(Tempstring1, '1');
      'C', 'G', 'J', 'K', 'Q', 'S', 'X', 'Z': Appendstr(Tempstring1, '2');
      'D', 'T': Appendstr(Tempstring1, '3');
      'L': Appendstr(Tempstring1, '4');
      'M', 'N': Appendstr(Tempstring1, '5');
      'R': Appendstr(Tempstring1, '6');
      {All other letters, punctuation and numbers are ignored}
    End;
  Appendstr(Tempstring2, OriginalWord[1]);
  {Go through the result removing any consecutive duplicate numeric values.}

  For Count := 2 To Length(Tempstring1) Do
    If Tempstring1[Count - 1] <> Tempstring1[Count] Then
      Appendstr(Tempstring2, Tempstring1[Count]);
  Soundex := Tempstring2;               {This is the soundex value}
End;
//******************* SoundAlike *************************

Function SoundAlike(Word1, Word2: String): Boolean;
Begin
  If (Word1 = '') And (Word2 = '') Then
    Result := True
  Else
    If (Word1 = '') Or (Word2 = '') Then
    Result := False
  Else
    If (Soundex(Word1) = Soundex(Word2)) Then
    Result := True
  Else
    Result := False;
End;
//******************* ReplaceSub *************************

Function ReplaceSub(Str, sub1, sub2: String): String;
Var
  aPos              : Integer;
  rslt              : String;
Begin
  aPos := Pos(sub1, Str);
  rslt := '';
  While (aPos <> 0) Do
    Begin
      rslt := rslt + Copy(Str, 1, aPos - 1) + sub2;
      Delete(Str, 1, aPos + Length(sub1));
      aPos := Pos(sub1, Str);
    End;
  Result := rslt + Str;
End;
//******************* GetShortFileName *************************

Function GetShortFileName(Const FileName: String): String;
Var
  aTmp              : Array[0..255] Of char;
Begin
  If GetShortPathName(PChar(FileName), aTmp, SizeOf(aTmp) - 1) = 0 Then
    Result := FileName
  Else
    Result := StrPas(aTmp);
End;
//******************* GetLongFileName *************************

Function GetLongFileName(Const FileName: String): String;
Var
  aInfo             : TSHFileInfo;
Begin
  If SHGetFileInfo(PChar(FileName), 0, aInfo, SizeOf(aInfo), SHGFI_DISPLAYNAME) <> 0 Then
    Result := String(aInfo.szDisplayName)
  Else
    Result := FileName;
End;
//******************* GetLastAccessedTime *************************

Function GetLastAccessedTime(FileName: String): TDateTime;
Var
  filehandle        : THandle;
  LocalFileTime     : TFileTime;
  DosFileTime       : DWORD;
  LastAccessedTime  : TDateTime;
  FindData          : TWin32FindData;
Begin
  Result := EncodeDate(1900, 1, 1);
  filehandle := FindFirstFile(PChar(FileName), FindData);
  If filehandle <> INVALID_HANDLE_VALUE Then
    Begin
      Windows.FindClose(filehandle);
      If (FindData.dwFileAttributes And FILE_ATTRIBUTE_DIRECTORY) = 0 Then
        Begin
          FileTimeToLocalFileTime(FindData.ftLastWriteTime, LocalFileTime);
          FileTimeToDosDateTime(LocalFileTime,
            LongRec(DosFileTime).Hi, LongRec(DosFileTime).Lo);
          LastAccessedTime := FileDateToDateTime(DosFileTime);
          Result := LastAccessedTime;
        End;
    End;
End;
//******************* myCopyFile *************************

Function myCopyFile(Title: String; FromLocation, ToLocation: String): Boolean;
Var
  T                 : TSHFileOpStructA;
Begin
  If Not fileexists(FromLocation) Then Exit;
  With T Do
    Begin
      Wnd := 0;
      wFunc := FO_COPY;
      pFrom := pansichar(FromLocation);
      pTo := pansichar(ToLocation);
      fFlags := FOF_ALLOWUNDO Or FOF_NOCONFIRMATION Or FOF_SIMPLEPROGRESS Or FOF_NOERRORUI;
      lpszProgressTitle := PansiChar(Title);
    End;
  Try
    SHFileOperationA(T);
  Except
  End;
End;
//******************* CreateMultipleVolume *************************

Function CreateMultipleVolume(FileName: String; VolumeSize: Integer): String;
Var
  I, j              : Integer;
  FromF, ToF        : File;
  NumRead, NumWritten: Integer;
  Buf               : Array[1..2048] Of char;
  VolCount          : Integer;
  CurrVol           : Integer;
  fSize             : Integer;
  fName             : String;
  fExt              : String;
  fPath             : String;
  Files             : TStringList;
  vSize             : Integer;
  vBool             : Boolean;
Begin
  Files := TStringList.Create;
  Try
    AssignFile(FromF, FileName);
    Reset(FromF, 1);                    { Record size = 1 }
    fSize := FileSize(FromF);
    VolCount := fSize Div VolumeSize;
    If (VolCount < 1) Or (fSize = VolumeSize) Then
      Begin
        CloseFile(FromF);
        Result := '';
        Exit;
      End;
    If fSize Mod VolumeSize <> 0 Then Inc(VolCount);
    fName := Copy(ExtractFileName(FileName), 1, Pos('.', ExtractFileName(FileName)) - 1);
    fPath := ExtractFilePath(FileName);
    fExt := Copy(extractfileext(FileName), 2, Length(extractfileext(FileName)) - 1);
    If fPAth[Length(fPath)] <> '\' Then
      fPath := fPAth + '\';

    For I := 0 To VolCount - 1 Do
      Begin
        If I = 0 Then
          Begin
            AssignFile(ToF, fPath + fName + '@' + IntToStr(VolCount - 1) + '@' + fExt + '.vol');
            Files.Add(fPath + fName + '@' + IntToStr(VolCount - 1) + '@' + fExt + '.vol');
          End
        Else
          Begin
            AssignFile(ToF, fPath + fName + '@' + fExt + '.' + FormatFloat('000', I - 1)); { Open output file }
            Files.Add(fPath + fName + '@' + fExt + '.' + FormatFloat('000', I - 1));
          End;

        Rewrite(ToF, 1);                { Record size = 1 }
        Repeat
          BlockRead(FromF, Buf, SizeOf(Buf), NumRead);
          BlockWrite(ToF, Buf, NumRead, NumWritten);
          vSize := FileSize(tof);
          If I <> VolCount - 1 Then
            vBool := volumesize - vSize < SizeOf(Buf)
          Else
            vBool := NumRead < NumWritten;
        Until (NumRead = 0) Or (vBool);
        If I <> VolCount - 1 Then
          Begin
            BlockRead(FromF, Buf, volumesize - FileSize(tof), NumRead);
            BlockWrite(ToF, Buf, NumRead, NumWritten);
          End;
        CloseFile(ToF);
      End;
    CloseFile(FromF);
    Result := Files.CommaText;
  Finally
    Files.Free
  End;
End;
//******************* ExtractMultipleVolume *************************

Function ExtractMultipleVolume(FileName: String): String;
Var
  I, j              : Integer;
  S, T              : String;
  FromF, ToF        : File;
  NumRead, NumWritten: Integer;
  Buf               : Array[1..2048] Of char;
  fFound            : Boolean;
Begin
  fFound := True;
  S := Copy(FileName, 1 + Pos('@', FileName), Length(FileName) - Pos('@', FileName));
  S := Copy(S, 1, Pos('@', S) - 1);
  I := StrToInt(S);
  For j := 0 To I - 1 Do
    Begin
      T := FileName;
      replacesc(T, '@' + IntToStr(I), '', False);
      replacesc(T, extractfileext(T), '.' + FormatFloat('000', j), False);
      If Not fileexists(T) Then
        Begin
          fFound := False;
          mymessagedlg('Eroare', 'Fisierul de volum ' + T + ' nu exista pe disc. Executie anulata.', mtWarning, [mbOK]);
          Break;
        End;
    End;
  If Not fFound Then Exit;
  S := Copy(FileName, 1 + Pos('@', FileName), Length(FileName) - Pos('@', FileName));
  S := Copy(S, 1, Pos('@', S) - 1);
  I := StrToInt(S);
  T := extractfileext(FileName);
  S := FileName;
  Result := S;
  replacesc(S, '@' + IntToStr(I) + '@', '.', False);
  replacesc(S, T, '', False);
  AssignFile(FromF, FileName);
  Reset(FromF, 1);
  AssignFile(ToF, S);

  Rewrite(ToF, 1);
  Repeat
    BlockRead(FromF, Buf, SizeOf(Buf), NumRead);
    BlockWrite(ToF, Buf, NumRead, NumWritten);
  Until (NumRead = 0) Or (NumWritten <> NumRead);
  CloseFile(FromF);

  For j := 0 To I - 1 Do
    Begin
      T := FileName;
      replacesc(T, '@' + IntToStr(I), '', False);
      replacesc(T, extractfileext(T), '.' + FormatFloat('000', j), False);
      AssignFile(FromF, T);
      Reset(FromF, 1);
      Repeat
        BlockRead(FromF, Buf, SizeOf(Buf), NumRead);
        BlockWrite(ToF, Buf, NumRead, NumWritten);
      Until (NumRead = 0) Or (NumWritten <> NumRead);
      CloseFile(FromF);
    End;
  CloseFile(ToF);
End;
//******************* FilesCreated *************************

Function FilesCreated(Files: String): Boolean;
Var
  L                 : TStringList;
  I                 : Integer;
  R                 : Boolean;
Begin
  R := True;
  L := TStringList.Create;
  Try
    L.CommaText := Files;
    For I := 0 To L.Count - 1 Do
      Begin
        If (Trim(L.Strings[I]) <> '') Then
          If (Not fileexists(L.Strings[I])) Then
            Begin
              R := False;
              Break;
            End;
      End;
  Finally
    L.Free;
  End;
  Result := R;
End;
//******************* GetHDSerialNo *************************

Function GetHDSerialNo: pdword;
Var
  SerialNum         : pdword;
  A, B              : DWORD;
  Buffer            : Array[0..255] Of char;
Begin
  If GetVolumeInformation('c:\', Buffer, SizeOf(Buffer), SerialNum, A, B, Nil, 0) Then
    Result := SerialNum;
End;
//******************* GetDateDiff *************************

Function GetDateDiff(MaxDate, MinDate: TDate): Integer;
Var
  y1, y2            : Word;
  m1, m2            : Word;
  D1, D2            : Word;
  m_diff, d_diff, yr_diff: Integer;
  I, N1, N2         : Integer;
Const
  DV                = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
Begin
  DecodeDate(maxdate, y1, m1, D1);
  DecodeDate(mindate, y2, m2, D2);
  yr_diff := y1 - y2 - 1;
  If m1 - m2 >= 0 Then
    Begin
      N1 := 0;
      For I := 1 To m1 Do
        ;
    End
  Else
    Begin
    End;
  If D1 - D2 >= 0 Then
    Begin
    End
  Else
    Begin
    End;
End;
//******************* LoadReport *************************

Function LoadReport(ReportName: String): Boolean;
Var
  H                 : THandle;
  R                 : TReport;
Begin
  H := LoadLibrary(PChar('Report.dll'));
  Try
    If H >= HINSTANCE_ERROR Then
      Begin
        @R := getprocaddress(H, 'LoadReport');
        If @R <> Nil Then R(ReportName);
      End;
  Finally
    FreeLibrary(H);
  End;
End;
//******************* EditReport *************************

Function EditReport(ReportName, DatabaseName: String): Boolean;
Var
  H                 : THandle;
  R                 : TEditReport;
Begin
  H := LoadLibrary(PChar('ReportEditor.dll'));
  Try
    If H >= HINSTANCE_ERROR Then
      Begin
        @R := getprocaddress(H, 'EditReport');
        If @R <> Nil Then R(ReportName, DatabaseName);
      End;
  Finally
    FreeLibrary(H);
  End;
End;
End.

