// -----------------------------------------------------------------------------
unit XAVar;

// -----------------------------------------------------------------------------
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls, Forms,
  Dialogs;

// -----------------------------------------------------------------------------
type
// -----------------------------------------------------------------------------
  TVariableType = (vtNone, vtBoolean, vtDateTime, vtFloat, vtInteger, vtString);

// -----------------------------------------------------------------------------
  TXAVariable = class(TObject)
    private      (* Private declarations   *)
      FItemID                : Integer;       // ID Number: Used in Paging
      FName                  : string;        // Name of Variable
      FType                  : TVariableType; // Type of Variable
      FValue                 : string;        // Value of Variable
      FValid                 : Boolean;       // Specifies whether VAR is Valid
      procedure SetName(Name : string);
    protected    (* Protected declarations *)
      procedure SetAsBoolean(Value: Boolean);
      procedure SetAsFloat(Value: Double);
      procedure SetAsInteger(Value: Longint);
      procedure SetAsString(const Value: string);
      procedure SetAsDateTime(Value: TDateTime);
      function GetAsBoolean: Boolean;
      function GetAsFloat: Double;
      function GetAsInteger: Longint;
      function GetAsString: string;
      function GetAsDateTime: TDateTime;
    public       (* Public declarations    *)
      constructor Create(ItemID : Integer; Name : string; VType : TVariableType);
      destructor Destroy; override;
      property ItemID             : Integer      read FItemID         write FItemID;
      property Name               : string       read FName           write SetName;
      property AsBoolean          : Boolean      read GetAsBoolean    write SetAsBoolean;
      property AsFloat            : Double       read GetAsFloat      write SetAsFloat;
      property AsInteger          : LongInt      read GetAsInteger    write SetAsInteger;
      property AsString           : string       read GetAsString     write SetAsString;
      property AsDateTime         : TDateTime    read GetAsDateTime   write SetAsDateTime;
      property Valid              : Boolean      read FValid;
      property IsNull             : Boolean      read FValid;
   end;

// -----------------------------------------------------------------------------
  TXAVariableManager = class(TComponent)
    private      (* Private declarations   *)
      FActiveVersion         : Double;
      FGroup                 : TList;       // List of Variable Items
      FItemIndex             : Integer;     // Position within the FGroup List
      FBlankVariable         : TXAVariable; // BLANK Variable to prevent errors
      FSafety                : Boolean;     // FLAGS use of Blank Variable
    protected    (* Protected declarations *)
      procedure SetActiveVersion(DValue: Double);
      function GetIndex : Integer;
      procedure SetIndex(DIndex : Integer);
      function GetVariable(Index : Word): TXAVariable;
      function GetCount : Integer;
    public      (* Public declarations    *)
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      property Items[Index: Word] : TXAVariable  read GetVariable;                           default;
      property ItemIndex          : Integer      read GetIndex        write SetIndex;
    published    (* Published declarations *)
      property ActiveVersion      : Double       read FActiveVersion  write SetActiveVersion;
      property Count              : Integer      read GetCount;
      property ErrorSafety        : Boolean      read FSafety         write FSafety;
      function AddVariable(Value : TXAVariable) : Integer;
      procedure RemoveVariable(Value : TXAVariable);
      procedure DeleteVariable(Index : Integer);
      function LocateVariable(Name : string) : Integer;
      function VariableByName(Name : string) : TXAVariable;
      procedure Clear;
      procedure CleanUp;
   end;

// -----------------------------------------------------------------------------
  procedure Register;

// -----------------------------------------------------------------------------
implementation

// -----------------------------------------------------------------------------
procedure Register;
begin
  RegisterComponents('Samples', [TXAVariableManager]);
end;

// -----------------------------------------------------------------------------
// Variable Object
// -----------------------------------------------------------------------------
constructor TXAVariable.Create(ItemID : Integer; Name : string; VType : TVariableType);
begin
  inherited Create;
  FItemID := ItemID;
  FName := '<#NONE#>';
  FType := vtNone;
  FValue := '';
  if ((Name <> '') and (VType <> vtNone)) then
  begin
    FName := UpperCase(Name);
    FType := VType;
    case FType of
    vtBoolean, vtInteger, vtFloat, vtDateTime : FValue := '0';
                                           else FValue := '';
    end;
  end;
  FValid := ((FName <> '<#NONE#>') and (FType <> vtNone));
end;

destructor TXAVariable.Destroy;
begin
  inherited Destroy;
end;

// -----------------------------------------------------------------------------
procedure TXAVariable.SetName(Name : string);
begin
  FName := UpperCase(Name);
end;

procedure TXAVariable.SetAsBoolean(Value: Boolean);
begin
  FType := vtBoolean;
  FValue := IntToStr(Trunc(Ord(Value)));
end;

procedure TXAVariable.SetAsFloat(Value: Double);
begin
  FType := vtFloat;
  FValue := FloatToStr(Value);
end;

procedure TXAVariable.SetAsInteger(Value: Longint);
begin
  FType := vtInteger;
  FValue := IntToStr(Trunc(Value));
end;

procedure TXAVariable.SetAsString(const Value: string);
begin
  FType := vtString;
  FValue := Value;
end;

procedure TXAVariable.SetAsDateTime(Value: TDateTime);
begin
  FType := vtDateTime;
  FValue := DateToStr(Value);
end;

// -----------------------------------------------------------------------------
function TXAVariable.GetAsBoolean: Boolean;
begin
  case FType of
  vtBoolean, vtInteger : Result := Boolean(StrToIntDef(FValue,0));
               vtFloat : Result := Boolean(Trunc(StrToFloat(FValue)));
                    else Result := False;
  end;
end;

function TXAVariable.GetAsFloat: Double;
begin
  case FType of
  vtBoolean, vtInteger : Result := StrToFloat(FValue);
   vtFloat, vtDateTime : Result := StrToFloat(FValue);
                    else Result := 0;
  end;
end;

function TXAVariable.GetAsInteger: Longint;
begin
  case FType of
  vtBoolean, vtInteger : Result := StrToIntDef(FValue,0);
   vtFloat, vtDateTime : Result := Trunc(StrToFloat(FValue));
                    else Result := 0;
  end;
end;

function TXAVariable.GetAsString: string;
begin
  Result := FValue;
end;

function TXAVariable.GetAsDateTime: TDateTime;
begin
  case FType of
  vtDateTime : Result := StrToDateTime(FValue);
          else Result := 0;
  end;
end;

// -----------------------------------------------------------------------------
// Variable manager Object
// -----------------------------------------------------------------------------
constructor TXAVariableManager.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FActiveVersion := 1.2;
  FGroup := TList.Create;
  FGroup.Clear;
  FItemIndex := -1;
  FBlankVariable := TXAVariable.Create(0,'<#BLANK#>',vtString);
  FBlankVariable.AsString := '0';
  FSafety := True;
end;

destructor TXAVariableManager.Destroy;
var
  ovLoop                     : Integer;
begin
  FBlankVariable.Free;
  for ovLoop := (FGroup.Count - 1) downto 0 do
    TXAVariable(FGroup.Items[ovLoop]).Free;
  FGroup.Clear;
  FGroup.Free;
  inherited Destroy;
end;

// -----------------------------------------------------------------------------
procedure TXAVariableManager.SetActiveVersion(DValue: Double);
begin
// -- Place Holder so that Version Number can be a visual Property
end;

procedure TXAVariableManager.SetIndex(DIndex : Integer);
begin
  if (FGroup.Count <> 0) and (DIndex >= -1) and (DIndex < (FGroup.Count)) then
       FItemIndex := DIndex
  else FItemIndex := -1;
end;

function TXAVariableManager.GetIndex : Integer;
begin
  Result := FItemIndex;
end;

// -----------------------------------------------------------------------------
function TXAVariableManager.GetVariable(Index : Word): TXAVariable;
begin
  if (Index > -1) and (Index < FGroup.Count) and (FGroup.Count <> 0) then
       Result := TXAVariable(FGroup.Items[Index])
  else if (FSafety) then Result := FBlankVariable
                    else Result := nil;
end;

function TXAVariableManager.GetCount : Integer;
begin
  Result := FGroup.Count;
end;

// -----------------------------------------------------------------------------
function TXAVariableManager.AddVariable(Value : TXAVariable) : Integer;
begin
  if (LocateVariable(Value.Name) = -1) then Result := FGroup.Add(Value)
                                       else Result := -1;
end;

procedure TXAVariableManager.RemoveVariable(Value : TXAVariable);
var
  ovLoop                     : Integer;
begin
  for ovLoop := 0 to (FGroup.Count - 1) do
  if (Value = FGroup.Items[ovLoop]) then FGroup.Remove(Value);
end;

procedure TXAVariableManager.DeleteVariable(Index : Integer);
begin
  if (FGroup.Count <> 0) and (Index > -1) and (Index < (FGroup.Count)) then
    FGroup.Delete(Index);
end;

function TXAVariableManager.LocateVariable(Name : string) : Integer;
var
  ovLoop                     : Integer;
begin
  Result := -1;
  Name := UpperCase(Name);
  for ovLoop := 0 to (FGroup.Count - 1) do
  if (Name = TXAVariable(FGroup.Items[ovLoop]).Name) then Result := ovLoop;
end;

function TXAVariableManager.VariableByName(Name : string) : TXAVariable;
var
  ovLoop                     : Integer;
begin
  Result := nil;
  Name := UpperCase(Name);
  for ovLoop := 0 to (FGroup.Count - 1) do
  if (Name = TXAVariable(FGroup.Items[ovLoop]).Name) then
    Result := TXAVariable(FGroup.Items[ovLoop]);
  if FSafety and (Result = nil) then Result := FBlankVariable;
end;

procedure TXAVariableManager.Clear;
var
  ovLoop                     : Integer;
begin
  for ovLoop := (FGroup.Count - 1) downto 0 do
    TXAVariable(FGroup.Items[ovLoop]).Free;
  FGroup.Clear;
end;

procedure TXAVariableManager.CleanUp;
var
  ovLoop                     : Integer;
begin
  for ovLoop := (FGroup.Count - 1) downto 0 do
  if (not TXAVariable(FGroup.Items[ovLoop]).Valid) then
    TXAVariable(FGroup.Items[ovLoop]).Free;
end;

// -----------------------------------------------------------------------------

end.
