// xAction Package  1998, Deks
// Version info in readme.txt and help
//
// DeksSite: http://deks.webjump.com/
// E-mail: Deks@online.nsk.su
// ============================================================================
// Design-time stuff for component.
unit uxActionEditor;

interface

// Need to install RxLib (http://rx.demo.ru)
{$I RX.INC}

// some copiler options:
{$B-} {- Complete Boolean Evaluation }
{$R-} {- Range-Checking }
{$V-} {- Var-String Checking }
{$T-} {- Typed @ operator }
{$X+} {- Extended syntax }
{$P+} {- Open string params }
{$J+} {- Writeable structured consts }
{$H+} {- Use long strings by default }

uses
  Windows, SysUtils, Messages, Classes, Graphics, Controls, Forms, Dialogs,
  Grids, DsgnIntf, DsgnWnds, LibIntf, StdCtrls, ExtCtrls, VCLUtils,
  uxAction, Placemnt, Menus;

type
  // ==============================
  // TxControlLinksProperty
  // ==============================
  TxControlLinksProperty = class(TPropertyEditor)
    function GetAttributes: TPropertyAttributes; override;
    function GetValue: string; override;
    procedure Edit; override;
  end;

  // ==============================
  // TxActionEditor
  // ==============================
  TxActionEditor = class(TComponentEditor)
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
  end;

  // ==============================
  // TFormActionEditor
  // ==============================
  TFormActionEditor = class(TDesignWindow)
    ActionGrid: TDrawGrid;
    BtnPanel: TPanel;
    CloseBtn: TButton;
    BtnDelete: TButton;
    BtnAdd: TButton;
    FormStorage: TFormStorage;
    PanelOptions: TPanel;
    CbConfirmDelete: TCheckBox;
    BtnClear: TButton;
    DialogMenu: TPopupMenu;
    miDialogMenuClose: TMenuItem;
    procedure FormClose(Sender: TObject; var ControlLink: TCloseAction);
{$IFDEF RX_D3}
    procedure ActionGridDrawCell(Sender: TObject; Col, Row: LongInt;
      Rect: TRect; State: TGridDrawState);
    procedure ActionGridSelectCell(Sender: TObject; Col, Row: LongInt;
      var CanSelect: Boolean);
{$ELSE}
    procedure ActionGridDrawCell(Sender: TObject; Col, Row: Integer;
      Rect: TRect; State: TGridDrawState);
    procedure ActionGridSelectCell(Sender: TObject; Col, Row: Integer;
      var CanSelect: Boolean);
{$ENDIF}
    procedure CloseBtnClick(Sender: TObject);
    procedure BtnDeleteClick(Sender: TObject);
    procedure ActionGridKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure BtnAddClick(Sender: TObject);
    procedure BtnClearClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
  private
    FAction: TxAction;
    FDeleting: Boolean;
    procedure SetAction(Value: TxAction);
    function GetForm: TCustomForm;
    procedure UpdateData;
    function CheckAction: Boolean;
    procedure SelectLink(ControlLink: TxControlLink);
    function LinkByRow(Row: Integer): TxControlLink;
  protected
    function UniqueName(Component: TComponent): string; override;
    procedure Activated; override;
  public
    procedure FormModified; override;
    procedure FormClosed(Form: TCustomForm); override;
    function GetEditState: TEditState; override;
    procedure ComponentDeleted(Component: TComponent); override;
    property Action: TxAction read FAction write SetAction;
    property OwnerForm: TCustomForm read GetForm;
  end;

var
  FormActionEditor: TFormActionEditor;

implementation

{$R *.DFM}

uses uxAboutActionForm, uxActionConstants;

// ============================================================================
function FindEditor(Action: TxAction): TFormActionEditor;
var
  I: Integer;
begin
  Result := nil;
  for I := 0 to Screen.FormCount - 1 do begin
    if Screen.Forms[I] is TFormActionEditor then begin
      if TFormActionEditor(Screen.Forms[I]).Action = Action then
      begin
        Result := TFormActionEditor(Screen.Forms[I]);
        Break;
      end;
    end;
  end;
end;

// ============================================================================
procedure ShowActionListEditor(Designer: TDesigner; Action: TxAction);
var
  Editor: TFormActionEditor;
begin
  if Action = nil then Exit;
  Editor := FindEditor(Action);
  if Editor <> nil then begin
    Editor.Show;
    if Editor.WindowState = wsMinimized then Editor.WindowState := wsNormal;
  end
  else begin
    Editor := TFormActionEditor.Create(Application);
    try
      Editor.Designer := TFormDesigner(Designer);
      Editor.Action := Action;
      Editor.Show;
    except
      Editor.Free;
      raise;
    end;
  end;
end;

// ============================================================================
// TxControlLinksProperty
function TxControlLinksProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paReadOnly];
end;

function TxControlLinksProperty.GetValue: string;
begin
  //Result := 'Control Links';
  FmtStr(Result, '(%s)', [GetPropType^.Name]);
end;

procedure TxControlLinksProperty.Edit;
begin
  ShowActionListEditor(Designer, TxAction(GetComponent(0)));
end;

// ============================================================================
// TxActionEditor
procedure TxActionEditor.ExecuteVerb(Index: Integer);
begin
  case Index of
    0,1,2: begin
      with TxAboutActionform.Create(Application) do begin
        try
          ShowModal;
        finally
          Free;
        end; // try
      end; // with
    end;
    3: ShowActionListEditor(Designer, TxAction(Component));
  end;
end;

function TxActionEditor.GetVerb(Index: Integer): string;
begin
  case Index of
    0: Result := 'xAction Package v1.2';
    1: Result := ' 1998, Deks';
    2: Result := '-';
    3: Result := 'XAction Editor ...';
  end;
end;

function TxActionEditor.GetVerbCount: Integer;
begin
  Result := 4;
end;

// ============================================================================
// TFormActionEditor
procedure TFormActionEditor.SetAction(Value: TxAction);
begin
  if FAction <> Value then begin
    FAction := Value;
    UpdateData;
  end;
end;

// ----------------------------------------------------------------------------
function TFormActionEditor.UniqueName(Component: TComponent): string;
var
  Temp: string;
begin
  Result := '';

  if (Component <> nil)
  then Temp := Component.ClassName
  else Temp := 'TComponent';

  if (UpCase(Temp[1]) = 'T') and (Length(Temp) > 1) then
    System.Delete(Temp, 1, 1);

  Result := Designer.UniqueName(Temp);
end;

// ----------------------------------------------------------------------------
function TFormActionEditor.GetEditState: TEditState;
begin
  Result := [];
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.FormClosed(Form: TCustomForm);
begin
  if Form = OwnerForm then Close;
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.FormModified;
begin
  if not (csDestroying in ComponentState) then UpdateData;
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.Activated;
begin
  SelectLink(LinkByRow(ActionGrid.Row - 1));
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.ComponentDeleted(Component: TComponent);
begin
  if Component = FAction then begin
    FAction := nil;
    Close;
  end;
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.UpdateData;
var
  aCount: Integer;
begin
  //   :

  if CheckAction then begin
    //   :
    aCount := FAction.ControlLinks.Count;
    if aCount = 0 then begin
      ActionGrid.RowCount := 2;
      ActionGrid.FixedRows := 1;
      SelectLink(nil);
    end else begin
      ActionGrid.RowCount := aCount+1;
      ActionGrid.FixedRows := 1;
    end;

    //    :
    BtnDelete.Enabled := aCount > 0;

    //  :
    ActionGrid.Invalidate;
  end;
end;

// ----------------------------------------------------------------------------
function TFormActionEditor.GetForm: TCustomForm;
begin
  Result := Designer.Form;
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.FormClose(Sender: TObject;
  var ControlLink: TCloseAction);
begin
  ControlLink := caFree;
end;

// ----------------------------------------------------------------------------
function TFormActionEditor.CheckAction: Boolean;
begin
  Result := (FAction <> nil) and (FAction.Owner <> nil) and
    (Designer.Form <> nil);
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.SelectLink(ControlLink: TxControlLink);
var
  FComponents: TComponentList;
begin
  if CheckAction and Active then begin
    FComponents := TComponentList.Create;
    if ControlLink <> nil then
      FComponents.Add(ControlLink)
    else
      FComponents.Add(FAction);
    SetSelection(FComponents);
  end;
end;

// ----------------------------------------------------------------------------
function TFormActionEditor.LinkByRow(Row: Integer): TxControlLink;
begin
  Result := nil;
  if CheckAction and (Row >= 0) and
    (Row < FAction.ControlLinks.Count) then
  begin
    Result := FAction.ControlLinks.Items[Row];
  end;
end;

// ----------------------------------------------------------------------------
{$IFDEF RX_D3}
procedure TFormActionEditor.ActionGridDrawCell(Sender: TObject; Col,
  Row: LongInt; Rect: TRect; State: TGridDrawState);
{$ELSE}
procedure TFormActionEditor.ActionGridDrawCell(Sender: TObject; Col,
  Row: Integer; Rect: TRect; State: TGridDrawState);
{$ENDIF}
var
  CellText: string;
  ControlLink: TxControlLink;
begin
  CellText := '';
  if (gdFixed in State) then begin
    case Col of
      0: CellText := 'ControlLink name';
    end;
  end else begin
    ControlLink := LinkByRow(Row - 1);
    if ControlLink <> nil then begin
      case Col of
        0: CellText := ControlLink.Name;
      end;
    end;
  end;
  DrawCellText(ActionGrid, Col, Row, CellText, Rect, taLeftJustify, vaCenter);
end;

// ----------------------------------------------------------------------------
{$IFDEF RX_D3}
procedure TFormActionEditor.ActionGridSelectCell(Sender: TObject; Col,
  Row: LongInt; var CanSelect: Boolean);
{$ELSE}
procedure TFormActionEditor.ActionGridSelectCell(Sender: TObject; Col,
  Row: Integer; var CanSelect: Boolean);
{$ENDIF}
begin
  SelectLink(LinkByRow(Row - 1));
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.CloseBtnClick(Sender: TObject);
begin
  // Close designer window.
  Close;
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.BtnDeleteClick(Sender: TObject);
var
  ControlLink: TxControlLink;
begin
  // Handle click on "Delete" button.

  // ask for confirmation from user:
  if (cbConfirmDelete.Checked = True)
  and (Application.MessageBox('Delete item ?', 'Confirmation',
    mb_YesNo + mb_IconQuestion ) <> ID_YES)
  then Exit;

  // delete item:
  ControlLink := LinkByRow(ActionGrid.Row - 1);
  if ControlLink <> nil then begin
    Designer.ValidateRename(ControlLink, ControlLink.Name, '');
    FDeleting := True;
    try
      ControlLink.Free;
      Designer.Modified;
    finally
      FDeleting := False;
    end;
  end;
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.ActionGridKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  // Handle "key down" event for item's grid.
  if Shift = [] then begin
    case Key of
      VK_RETURN:
        // select item when user press ENTER key:
        if LinkByRow(ActionGrid.Row - 1) <> nil then begin
          ActivateInspector(#0);
        end;
      VK_DELETE:
        // delete item (call Delete's button event handler), when
        // user click DEL key:
        BtnDeleteClick(nil);
    end;
  end;
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.BtnAddClick(Sender: TObject);
var
  ControlLink: TxControlLink;
begin
  // Handle ADD's button click: add new item to list.
  if not CheckAction then Exit;

  // create new item:
  ControlLink := TxControlLink.Create( Action.Owner );

  // init new item's properties:
  if controlLink <> nil then
    ControlLink.Name := UniqueName( ControlLink );

  // add item to list:
  Action.Add(ControlLink);

  // display new version of list:
  UpdateData;

  // select that item:
  SelectLink(ControlLink);
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.BtnClearClick(Sender: TObject);
begin
  // Handle click event for "Clear" button.

  // ask confirmation from user:
  if (Application.MessageBox('Delete all items ?', 'Confirmation',
    mb_YesNo + mb_IconQuestion ) <> ID_YES)
  then Exit;

  // delete all items from action list:
  if Action <> nil then begin
    Action.Clear;
    Designer.Modified; // let designer to notify some minor changes :)
    UpdateData; // update list of items in component editor
  end;
end;

// ----------------------------------------------------------------------------
procedure TFormActionEditor.FormCreate(Sender: TObject);
begin
  // Init strings from language resources:
  Caption := SxAD_Caption;
  CloseBtn.Caption := SxAD_Close;
  BtnAdd.Caption := SxAD_Add;
  BtnDelete.Caption := SxAD_Delete;
  BtnClear.Caption := SxAD_Clear;
  CbConfirmDelete.Caption := SxAD_ConfirmDelete;  
end;

end.

