// xAction Package  1998, Deks
// Version info in readme.txt and help
//
// DeksSite: http://deks.webjump.com/
// E-mail: Deks@online.nsk.su
// ============================================================================
// xAction main unit

unit uxAction;

interface

// Need to install RxLib (http://rx.demo.ru)
{$I RX.INC}

uses
  Classes, Controls, ExtCtrls, Menus;

type
  TxAction = class;

  // ==================
  // TxControlLink
  // ==================
  TxControlLink = class(TComponent)
  protected
    FControl: TControl;
    FMenuItem: TMenuItem;
    FAction: TxAction;

    procedure SetAction(Value: TxAction);

    procedure SetParentComponent(Value: TComponent); override;
    procedure Notification(AComponent: TComponent; AOperation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function HasParent: Boolean; override;
    function GetParentComponent: TComponent; override;

    property Action: TxAction read FAction write SetAction;
  published
    property Control: TControl read FControl write FControl;
    property MenuItem: TMenuItem read FMenuItem write FMenuItem;
  end;

  // ==================
  // TxAction
  // ==================
  TxAction = class(TComponent)
  protected
    FControlLinks: TList;
    FEnabled: Boolean;
    FVisible: Boolean;
    FCaption: String;
    FHint: String;
    FOnExecute: TNotifyEvent;

    procedure Loaded; override;
    procedure Notification(AComponent: TComponent; AOperation: TOperation); override;
    {$IFDEF RX_D3}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
    {$ELSE}
    procedure GetChildren(Proc: TGetChildProc); override;
    {$ENDIF}

    procedure SetControlLinks( Value: TList);
    procedure SetEnabled(Value: Boolean);
    procedure SetVisible(Value: Boolean);
    procedure SetCaption(Value: String);
    procedure SetHint(Value: String);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure Add( ControlLink: TxControlLink);
    procedure Remove( ControlLink: TxControlLink);
    procedure Clear;

    procedure Execute; virtual;
  published
    property Enabled: Boolean read FEnabled write SetEnabled default True;
    property Visible: Boolean read FVisible write SetVisible default True;
    property Caption: String read FCaption write SetCaption;
    property Hint: String read Fhint write SetHint;
    property ControlLinks: TList read FControlLinks write SetControlLinks;

    property OnExecute: TNotifyEvent read FOnExecute write FOnExecute;
  end;

implementation

uses SysUtils, Forms, StdCtrls, TypInfo;

const
  Registered: Boolean = False;

// ============================================================================
// TxControlLink
// ============================================================================
constructor TxControlLink.Create(AOwner: TComponent);
begin
  // Constructor.

  //   :
  inherited Create(AOwner);

  //   :
  FControl := nil;
  FMenuItem := nil;
  FAction := nil;
end;

// ----------------------------------------------------------------------------
destructor TxControlLink.Destroy;
begin
  // .

  //        :
  if FAction <> nil then
    FAction.Remove(Self);

  //     :
  Action := nil;

  //   :
  inherited Destroy;
end;

// ----------------------------------------------------------------------------
procedure TxControlLink.Notification(AComponent: TComponent; AOperation: TOperation);
begin
  //   -   .

  //   .
  inherited Notification(AComponent, AOperation);

  //      (  ,  
  //    ):
  if AOperation = opRemove then begin
    if AComponent = Control then Control := nil;
    if AComponent = MenuItem then MenuItem := nil;
  end;
end;

// ----------------------------------------------------------------------------
procedure TxControlLink.SetAction(Value: TxAction);
begin
  //   SetAction.

  // ;   ,      
  // :
  if Value = FAction then Exit;

  //     :
  if FAction <> nil then
    FAction.Remove(Self);

  //   :
  FAction := Value;

  //     :
  if FAction <> nil then
    FAction.Add(Self);
end;

// ----------------------------------------------------------------------------
function TxControlLink.HasParent: Boolean;
begin
  //     ?  !
  Result := True;
end;

// ----------------------------------------------------------------------------
function TxControlLink.GetParentComponent: TComponent;
begin
  //    -    ...    !
  Result := FAction;
end;

// ----------------------------------------------------------------------------
procedure TxControlLink.SetParentComponent(Value: TComponent);
begin
  //    (  
  //  GetParentcomponent)
  if FAction <> nil then
    FAction.Remove(Self);
  if (Value <> nil) and (Value is TxAction) then
    Action := TxAction(Value);
end;

// ============================================================================
// TxAction
// ============================================================================
constructor TxAction.Create(AOwner: TComponent);
begin
  // Constructor.

  // call inherited constructor:
  inherited Create(AOwner);

  // create ControlLink's "storage":
  FControlLinks := TList.Create;

  // register ControlLink class when necessary:
  if not Registered then begin
    RegisterClasses([TxControlLink]);
    Registered := True;
  end;

  // init internal fields of class:
  FEnabled := True;
  FVisible := True;
  FCaption := '';
end;

// ----------------------------------------------------------------------------
destructor TxAction.Destroy;
begin
  // Destructor.

  // clear ControlLink's:
  Clear;

  // destroy ControlLinks object:
  if FControlLinks <> nil then
    FControlLinks.Free;

  // call inherited destructor:
  inherited Destroy;
end;

// ----------------------------------------------------------------------------
procedure TxAction.Loaded;
//var Loading: Boolean;
begin
// No action yet (in current version :)
//  Loading := csLoading in ComponentState;
  inherited Loaded;
end;

// ----------------------------------------------------------------------------
procedure TxAction.Notification(AComponent: TComponent; AOperation: TOperation);
begin
// No action yet (in current version :)
  inherited Notification(AComponent, AOperation);
//  if AOperation = opRemove then begin
//  end;
end;

// ----------------------------------------------------------------------------
{$IFDEF RX_D3}
procedure TxAction.GetChildren(Proc: TGetChildProc; Root: TComponent);
{$ELSE}
procedure TxAction.GetChildren(Proc: TGetChildProc);
{$ENDIF}
var i: Integer;
begin
{$IFDEF RX_D3}
  inherited GetChildren(Proc, Root);
{$ELSE}
  inherited GetChildren(Proc);
{$ENDIF}
  for i := 0 to FControlLinks.Count - 1 do begin
    Proc(TxControlLink(FControlLinks.Items[i]));
  end;
end;

// ----------------------------------------------------------------------------
procedure TxAction.SetControlLinks(Value: TList);
begin
  // no action!
end;

// ----------------------------------------------------------------------------
procedure TxAction.SetEnabled(Value: Boolean);
var
  i: Integer;
  ControlLink: TxControlLink;
begin
  // Set ENABLED property's value.
  FEnabled := Value;
  if FControlLinks = nil then Exit;
  for i:=0 to FControlLinks.Count - 1 do begin
    ControlLink := TxControlLink(FControlLinks[i]);
    if ControlLink.Control <> nil then
      ControlLink.Control.Enabled := Value;
    if ControlLink.MenuItem <> nil then
      ControlLink.MenuItem.Enabled := Value;
  end;
end;

// ----------------------------------------------------------------------------
procedure TxAction.SetVisible(Value: Boolean);
var
  i: Integer;
  ControlLink: TxControlLink;
begin
  FVisible := Value;
  if FControlLinks = nil then Exit;
  for i:=0 to FControlLinks.Count - 1 do begin
    ControlLink := TxControlLink(FControlLinks[i]);
    if ControlLink.Control <> nil then
      ControlLink.Control.Visible := Value;
    if ControlLink.MenuItem <> nil then
      ControlLink.MenuItem.Visible := Value;
  end;
end;

// ----------------------------------------------------------------------------
procedure TxAction.SetCaption(Value: String);
var
  i: Integer;
  PropInfo: PPropInfo;
  ControlLink: TxControlLink;
begin
  // Set CAPTION property value in all linked components.

  // set new property value:
  FCaption := Value;

  if FControlLinks = nil then Exit;
  for i:=0 to FControlLinks.Count - 1 do begin
    ControlLink := TxControlLink(FControlLinks[i]);
    // use RTTI to set Caption value:
    if ControlLink.Control <> nil then begin
      // get property's information:
      PropInfo := GetPropInfo(ControlLink.Control.ClassInfo, 'Caption');
      if PropInfo <> nil then
        SetStrProp(ControlLink.Control, PropInfo, Value); // set CAPTION property using system RTTI function
    end;

    // for MenuItem situation is much easie: just set new value for caption:
    if ControlLink.MenuItem <> nil then
      ControlLink.MenuItem.Caption := Value;
  end;
end;

// ----------------------------------------------------------------------------
procedure TxAction.SetHint(Value: String);
var
  i: Integer;
  PropInfo: PPropInfo;
  ControlLink: TxControlLink;
begin
  // Set HINT property for linked controls/menu items.

  // set new property value:
  FHint := Value;

  if FControlLinks = nil then Exit;

  for i:=0 to FControlLinks.Count - 1 do begin
    ControlLink := TxControlLink(FControlLinks[i]);

    // Set TControl.Hint property using RTTI:
    if ControlLink.Control <> nil then begin
      // use RTTI to set property, because not all TControls
      // have HINT property:
      PropInfo := GetPropInfo(ControlLink.Control.ClassInfo, 'Hint');
      if PropInfo <> nil then
        SetStrProp(ControlLink.Control, PropInfo, Value);
    end;

    // set TMenuItem.Hint property (very simple, yeah ?):
    if ControlLink.MenuItem <> nil then
      ControlLink.MenuItem.Hint := Value;
  end;
end;

// ----------------------------------------------------------------------------
procedure TxAction.Add(ControlLink: TxControlLink);
var
  i: Integer;
  isFound: Boolean;
begin
  // Add item to list.

  // test variables/parameters used in this method:
  if (ControlLink = nil) or (FControlLinks = nil) then Exit;

  // if this item is already in list:
  isFound := False;
  for i:=0 to FControlLinks.Count - 1  do begin
    if FControlLinks[i] = ControlLink then
      isFound := True;
  end;

  // if an item not in list yet, just add it to list:
  if isFound = False then begin
    FControlLinks.Add(ControlLink);
    ControlLink.Action := Self;
  end;
end;

// ----------------------------------------------------------------------------
procedure TxAction.Remove(ControlLink: TxControlLink);
var
  i: Integer;
  isFound: Boolean;
begin
  // Remove item from list.

  // test variables/parameters used in this method:
  if (ControlLink.Action <> Self) or (ControlLink = nil)  then Exit;

  // test if this item is already in list:
  isFound := False;
  for i:=0 to FControlLinks.Count - 1  do begin
    if FControlLinks[i] = ControlLink then
      isFound := True;
  end;

  // Remove item from list (only if this item is in list):
  if isFound = True then begin
    FControlLinks.Remove(ControlLink);
    ControlLink.Action := nil; // drop link to parent's action component
  end;
end;

// ----------------------------------------------------------------------------
procedure TxAction.Clear;
var
  ControlLink: TxControlLink;
begin
  // Remove (and delete) all items from list.

  // While some item is in list ...
  while FControlLinks.Count > 0 do begin
    // get last item:
    ControlLink := FControlLinks.Last;

    // remove it from list:
    Remove(ControlLink);

    // delete item:
    ControlLink.Free;
  end;
end;

// ----------------------------------------------------------------------------
procedure TxAction.Execute;
begin
  // Execute action.
  if Assigned(FOnExecute) then
    FOnExecute(Self);
end;

end.
