{******************************************************************
*  (c)copyrights Corona Ltd. Donetsk 1999
*  Project: Zeos Library
*  Module: TMySQLQuery class for direct MySQL access (version 2.0)
*  Author: Sergey Seroukhov   E-Mail: voland@cm.dongu.donetsk.ua
*  Date: 27/01/99
*
*  List of changes:
*  20/03/99 - Added functions Lookup, Locate
*  21/03/99 - Changed edit complex queries principe,
*             added function FormKeyValues, fix bugs in functions SetRecNo,
*             Locate, ReverseSort, ClearSort
*  23/03/99 - Component splited into TMySQLQuery, TMySQLDatabase
*             and TMySQLTransact
*  27/03/99 - Added filtration by equation
*  01/04/99 - Fixed bug in close query, added Master-Detail links
*  02/04/99 - Added update/delete records filtration
*  17/04/99 - Added mySQL error messages to exceptions
*  21/04/99 - Fixed bugs with field indexing, reopen, etc.
*  22/04/99 - QueryRecord optimized
*  25/06/99 - Added BLOB-fields
*  11/07/99 - Redefined func CompareBookmarks, add Boolean type as enum("Y","N")
*  06/08/99 - Added CommitUpdates function, changed ApplyUpdates
*  09/08/99 - Fixed auto_increment fields update
*  11/08/99 - Fixed refresh auto_increment in controls after ApplyUpdates,
*             There is no auto ApplyUpdates on close query
***************************************************************************}

unit ZMySQLQuery;

interface

{$R *.DCR}

uses
  SysUtils, Windows, DB, Classes, Forms, Math, ZDirSql, ZDirMySql, DbCommon,
  ZMySQLCon, ZMySQLTr, ZToken, LibMySQL, ZMySQLExtra, Dialogs, ZParser;
                         
{$I ..\Zeos.inc}

const
  MAX_FIELD_LINKS = 10;
  MAX_FIELDS = 255;

type

TApplyUpdatesEvent = procedure(Sender: TObject; Old, New: TFieldValues;
                          Status: TFieldStatus) of object;

TShowRecordTypes = set of (rtModified, rtInserted, rtDeleted, rtUnmodified);

{******************** TMySQLQuery definition *****************}

// Direct MySQL query with descendant of TDataSet
TMySQLQuery = class(TDataSet)
private
  Records: TDBRecords;        // All and updates query records
  SaveRecords: TDBRecords;    // Stored old query records (cache)
  FieldOffSets: PIntArray;
  FRecBufSize: Integer;
  FCurRec: Integer;           // Current record number
  FLastBookmark: Integer;
  FRecordSize: Integer;       // Size of record buffer
  FFilterBuffer: PRecordData;
  FirstFetch: Boolean;
  FQueryAllRecords: Boolean;  // Is fetch all records?
  FFetchCount: LongInt;       // Already fetched records quantity
  FRequestLive: Boolean;      // Is query updated?
  FReadOnly: Boolean;         // Is query read only???
  FCachedUpdates: Boolean;    // Is use cached updates?
  FUseGen: Boolean;           // Is use zeos generators?
  FSQL: TStringList;          // SQL query strings
  FConnection: TMySQLDatabase;// Connect to database component
  FTransact: TMySQLTransact;  // Connect to transact-server component
  FFieldDescs: TCollection;
  FTables, FAliases: TStringList; // Tables and table aliase of a query
  FApplyUpdates: TApplyUpdatesEvent; // On Apply Updates event
  FUpdateCmds: TStringList;   // Updates command list
  FStoreResult: Boolean;      // Is use store or use result mode
  FUpdateRecord: TShowRecordTypes;// Showed status records
  FLinkRequery: Boolean;      // Is master-detail links requery
  FParser, FSaveParser: TParser;

  FLinkFields: String;
  FMasterLink: TMasterDataLink;
  FMasterFields: array[0..MAX_FIELD_LINKS] of String;
  FDetailFields: array[0..MAX_FIELD_LINKS] of String;
  FLinkCount: Integer;

  FDb, FDbUse: TDirMySqlConnect;      // Connect to database
  FQuery, FExecQuery: TDirMySqlQuery; // Queries to database
  FFieldDescNums: array[0..MAX_FIELDS] of Integer;
  FQueryFieldNums: array[0..MAX_FIELDS] of Integer;
  FRealFields: array[0..MAX_FIELDS] of TField;

{**************** Internal methods *********************}
  function GetActiveRecBuf(var RecBuf: PRecordData): Boolean;
  function GetRecBufSize: Integer;
  procedure QueryRecords;
  procedure QueryRecord;
  procedure FieldToRecord(Field: TField; FieldIsNull: Boolean;
    FieldData: Pointer; RecordData: PRecordData);
  function FieldOffset(Field: TField): Integer;
  procedure InternalUpdate;
  procedure SaveRecordData(Buffer: PRecordData; Index: Integer);
  procedure SetSQL(Value: TStrings);
  function GetSQL: TStrings;
  procedure SetDatabase(Value: TMySQLDatabase);
  procedure SetTransact(Value: TMySQLTransact);
  function GetAffectedRows: LongInt;

// Define query fields
  procedure DefineFieldDefs;
// Define all fields of query
  procedure FillAllFields(TableName: String);
// Define all index fields of query
//  procedure FillAllIndexFields(TableName: String);
// Define the field
  procedure FillField(TableName, FieldName, AliasName: String);
// Put field description into collection
  procedure PutFieldDesc(Table, Field, Alias, TypeName, Null, Key,
    Def, Extra: String);
// Find field description in collection
  function FindFieldDesc(Field, Alias: String): TMySQLField;
// Normalize field names in collection
  procedure NormalizeFieldDescs;
// Check if record is hided by filter
  function CheckRecordByFilter(RecNo: LongInt): Boolean;
// Get field index in FieldDefs
  function GetFieldDefNo(Field: TField): Integer;
// Create optimize array for fields find
  procedure OptimizeFindFields;
protected
{**************** Overriding standart methods ********************}
// Close blobs of field
  procedure CloseBlob(Field: TField); override;
  function AllocRecordBuffer: PChar; override;
  procedure FreeRecordBuffer(var Buffer: PChar); override;
  procedure GetBookmarkData(Buffer: PChar; Data: Pointer); override;
  function GetBookmarkFlag(Buffer: PChar): TBookmarkFlag; override;
  function GetRecord(Buffer: PChar; GetMode: TGetMode; DoCheck: Boolean):
    TGetResult; override;
  function GetRecordSize: Word; override;
  procedure InternalAddRecord(Buffer: Pointer; Append: Boolean); override;
  procedure InternalClose; override;
  procedure InternalDelete; override;
  procedure InternalFirst; override;
  procedure InternalGotoBookmark(Bookmark: Pointer); override;
  procedure InternalHandleException; override;
  procedure InternalInitFieldDefs; override;
  procedure InternalInitRecord(Buffer: PChar); override;
  procedure InternalLast; override;
  procedure InternalOpen; override;
  procedure InternalEdit; override;
  procedure InternalPost; override;
  procedure InternalRefresh; override;
  procedure InternalSetToRecord(Buffer: PChar); override;
  function IsCursorOpen: Boolean; override;
  procedure SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag); override;
  procedure SetBookmarkData(Buffer: PChar; Data: Pointer); override;
  procedure SetFieldData(Field: TField; Buffer: Pointer); override;
  function GetCanModify: Boolean; override;
// Internal locate with fields equal to values
  function InternalLocate(KeyFields: string; KeyValues: Variant;
    Options: TLocateOptions): LongInt;
// Find record in filtered query
  function FindRecord(Restart, GoForward: Boolean): Boolean; override;
// Turn on/off filter
  procedure SetFiltered(Value: Boolean); override;
// Set filter equation
  procedure SetFilterText(const Value: string); override;
protected
{***************** Extra overriding functions *********************}
  function GetRecordCount: Integer; override;
  function GetRecNo: Integer; override;
  procedure SetRecNo(Value: Integer); override;
// Get field value from buffer
  function GetRealFieldValue(FieldNo: Integer; Buffer: PRecordData): String;
// Get string field value from buffer
  function GetFieldValue(FieldNo: Integer; Buffer: PRecordData): String;
// Set string field value to buffer
  function SetRealFieldValue(FieldNo: Integer; Value: String;
                         Buffer: PRecordData): String;
// Fill field structure
  procedure FillFieldValues(Buffer: PRecordData; FieldValues: TFieldValues);
// Automatic keys update using zeos generators
  procedure ApplyGens(Buffer: PRecordData);
// Auto create update queries
  procedure FormSQLQuery(Old,New: TFieldValues; Status: TFieldStatus);
// Create all demanded connections
  procedure CreateConnections;
// Post updates from cache buffer
  procedure FlushBuffer;
// Clear cache buffer
  procedure ClearBuffer;
// Internal commit updates
  procedure Flush;

// Get auto_increment field of table
// TableName - table name
  function GetAutoIncField(TableName: String): String;
// Get primary key field name of table
  function GetPrimaryKey(TableName: String): String;
// Get primary key field alias of table
  function GetPrimaryKeyAlias(TableName: String): String;
// Get field number in a query
  function GetRealFieldNo(FieldName: String): Integer;

// Get linked fields names
  function GetLinkFields: String;
// Set linked fields names
  procedure SetLinkFields(const Value: String);
// Get master link datasource
  function GetMasterDataSource: TDataSource;
// Set master link datasource
  procedure SetMasterDataSource(Value: TDataSource);
// Master dataset change event
  procedure MasterChanged(Sender: TObject);
// Master dataset disable event
  procedure MasterDisabled(Sender: TObject);
// Requery when master dataset was changed
  procedure MasterRequery;
// Set default field values
  procedure InitDefaults;
// Check if cache have updates
  function GetUpdatesPending: Boolean;
  procedure SetUpdateRecord(Value: TShowRecordTypes);
public
// Create blob stream
  function CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream; override;
  function GetFieldData(Field: TField; Buffer: Pointer): Boolean; override;

// Class constructor
  constructor Create(AOwner: TComponent); Override;
// Class destructor
  destructor Destroy; override;

// Check if query os sequenced?
  function IsSequenced: Boolean; override;
// Invert sort order
  procedure SortInverse;
// Clear record sorting
  procedure SortClear;
// Inc sort records by field
  procedure SortByField(FieldName: String);
// Desc sort records by field
  procedure SortDescByField(FieldName: String);
// Compare two bookmarks
  function CompareBookmarks(Bookmark1, Bookmark2: TBookmark): Integer; override;

// Quantity affected rows
  property AffectedRows: LongInt read GetAffectedRows;

// Post cached updates
  procedure ApplyUpdates;
// Clear updates buffer
  procedure CommitUpdates;
// Cancel cached updates
  procedure CancelUpdates;
// Get updates status
  function UpdateStatus: TUpdateStatus; {$IFNDEF VER100} override; {$ENDIF}
// Clear field updates
  procedure RevertRecord;

// Autofrom primary key for current record (compatable with locate)
// KeyFields - key field list
// KeyValues - key field values array
  procedure FormKeyValues(var KeyFields: String; var KeyValues: Variant);

// Locate record
  function Locate(const KeyFields: string; const KeyValues: Variant;
    Options: TLocateOptions): Boolean; override;
// Find for lookup fields
  function Lookup(const KeyFields: string; const KeyValues: Variant;
   const ResultFields: string): Variant; override;

// Check if made any updates in cache
  property UpdatesPending: Boolean read GetUpdatesPending;

published
// Is cached updates?
  property CachedUpdates: Boolean read FCachedUpdates write FCachedUpdates;
// Is query editable?
  property RequestLive: Boolean read FRequestLive write FRequestLive;
// Is fetch all records?
  property QueryAllRecords: Boolean read FQueryAllRecords write FQueryAllRecords;
// SQL query strings
  property SQL: TStrings Read GetSQL write SetSQL;
// Connect to sql database
  property Database: TMySQLDatabase read FConnection write SetDatabase;
// Connect to transact-server
  property Transaction: TMySQLTransact read FTransact write SetTransact;
// Query open status
  property Active;
// Is use zeos generators?
  property UseGen: Boolean read FUseGen write FUseGen;
// Is store or use updates?
  property StoreResult: Boolean read FStoreResult write FStoreResult;
// What kind of fields is showed
  property ShowRecordTypes: TShowRecordTypes read FUpdateRecord
    write SetUpdateRecord;

// Fields for master-detail links as '<Master field>=<Detail field>'
  property LinkFields: String read GetLinkFields write SetLinkFields;
// Master datasource
  property MasterSource: TDataSource read GetMasterDataSource
    write SetMasterDataSource;
// Is reopen query or use client filtering?
  property LinkRequery: Boolean read FLinkRequery write FLinkRequery;

// Filter equation
  property Filter;
// Turn on/off a filter
  property Filtered;
// Filtered options
  property FilterOptions;
// Is autoevalute calc fields?

  property AutoCalcFields;
  property BeforeOpen;
  property AfterOpen;
  property BeforeClose;
  property AfterClose;
  property BeforeInsert;
  property AfterInsert;
  property BeforeEdit;
  property AfterEdit;
  property BeforePost;
  property AfterPost;
  property BeforeCancel;
  property AfterCancel;
  property BeforeDelete;
  property AfterDelete;
  property BeforeScroll;
  property AfterScroll;
  property OnCalcFields;
  property OnDeleteError;
  property OnEditError;
  property OnFilterRecord;
  property OnNewRecord;
  property OnPostError;
// On Apply updates from cache event
  property OnApplyUpdates: TApplyUpdatesEvent
    read FApplyUpdates write FApplyUpdates;
end;

procedure Register;

implementation

uses ZExtra;

{$I ZMySQLInc1.inc}
{$I ZMySQLInc2.inc}
{$I ZMySQLInc3.inc}
{$I ZMySQLInc4.inc}
{$I ZMySQLInc5.inc}

{***************** TMySQLQuery implemantation *******************}

// Class constructor
constructor TMySQLQuery.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FTables := TStringList.Create;
  FAliases := TStringList.Create;
  FQuery := TDirMySqlQuery.Create;
  FSQL := TStringList.Create;
  FFieldDescs := TCollection.Create(TMySQLField);
  FStoreResult := true;
  FRequestLive := false;
  FDbUse := NIL;
  FUpdateRecord := [rtModified, rtInserted, rtUnmodified];

  FMasterLink := TMasterDataLink.Create(Self);
  FMasterLink.OnMasterChange := MasterChanged;
  FMasterLink.OnMasterDisable := MasterDisabled;
end;

// Class destructor
destructor TMySQLQuery.Destroy;
begin
  inherited Destroy;
  FMasterLink.Free;

  FAliases.Free;
  FTables.Free;
  FFieldDescs.Free;
  FSQL.Free;
  FQuery.Free;
end;

// Set new sql query
procedure TMySQLQuery.SetSQL(Value: TStrings);
begin
  Close;
  FSQL.Assign(Value);
end;

// Get an sql query
function TMySQLQuery.GetSQL: TStrings;
begin
  Result := FSQL;
end;

// Get affected rows quantity
function TMySQLQuery.GetAffectedRows: LongInt;
begin
  Result := 0;
end;

// Set connect to database component
procedure TMySQLQuery.SetDatabase(Value: TMySQLDatabase);
begin
  Close;
  FConnection := Value;
  FDb := FConnection.Handle;
end;

// Set connect to transact-server component
procedure TMySQLQuery.SetTransact(Value: TMySQLTransact);
begin
  Close;
  FTransact := Value;
end;

// Calculate record buffer length
function TMySQLQuery.GetRecBufSize: Integer;
var I: Integer;
begin
  GetMem(FieldOffsets, (FieldCount+1) * SizeOf(Integer));

  for I := 0 to MAX_FIELDS do FRealFields[I] := NIL;

  Result := 0;
  FRecordSize := 0;
  for I := 0 to FieldCount - 1 do with Fields[I] do begin
    if Index >= 0 then begin
      FieldOffsets^[Index] := FRecordSize;
      FRealFields[Index] := Fields[I];
    end;
    Inc(Result, MaxIntValue([DataSize,4]) + 1);
    Inc(FRecordSize, MaxIntValue([DataSize,4]) + 1);
  end;
  Inc(Result, RecInfoSize);
end;

// Read query from server to internal buffer
procedure TMySQLQuery.QueryRecord;
var
  I, N, BlobCount: Integer;
  Buffer: Pointer;
  ps: array[0..2000] of char;
  Accept: Boolean;
  FieldDesc: TMySQLField;
  TempTime: TDateTime;
  TimeStamp: TTimeStamp;
  TempInt: LongInt;
  TempDouble: Double;
  TempBool: Boolean;
begin
  BlobCount := 0;
  if FQuery.EOF then Exit;

  repeat
    if not FirstFetch then FQuery.Next;
    if FQuery.EOF then Exit;

    FirstFetch := False;
    FFilterBuffer := PRecordData(AllocRecordBuffer);
    for I := 0 to FieldCount - 1 do begin
      if Fields[I].FieldKind in [fkData] then begin
        Buffer := nil;
// To optimize performance, fields are accessed as binary data
        N := FFieldDescNums[Fields[I].FieldNo];
        if N>=0 then begin
          FieldDesc := TMySQLField(FFieldDescs.Items[N]);
          N := FQueryFieldNums[Fields[I].FieldNo];
          case FieldDesc.FieldType of
            ftString: begin
                Buffer := FQuery.GetFieldBuffer(N);
              end;
            ftInteger: begin
                TempInt := LongInt(StrToIntDef(FQuery.Fields[N],0));
                Move(TempInt, ps, SizeOf(LongInt));
                Buffer := @ps;
              end;
            ftFloat: begin
                TempDouble := StrToFloatEx(FQuery.Fields[N]);
                Move(TempDouble, ps, SizeOf(Double));
                Buffer := @ps;
              end;
            ftBoolean: begin
                if FQuery.Fields[N]='' then TempBool := false
                else TempBool := UpperCase(FQuery.Fields[N])[1]='Y';
                Move(TempBool, ps, SizeOf(Boolean));
                Buffer := @ps;
              end;
            ftTime: begin
                TimeStamp := DateTimeToTimeStamp(
                  SqlDateToDateTime(FQuery.Fields[N]));
                Buffer := PChar(@TimeStamp);
              end;
            ftDateTime: begin
                TimeStamp := DateTimeToTimeStamp(
                  SqlDateToDateTime(FQuery.Fields[N]));
                TempTime := TimeStampToMSecs(TimeStamp);
                Buffer := PChar(@TempTime);
              end;
            ftDate: begin
                TempTime := SqlDateToDateTime(FQuery.Fields[N]);
                TempInt := DateTimeToTimeStamp(TempTime).Date;
                Buffer := PChar(@TempInt);
              end;
            ftBlob, ftMemo: begin
                FFilterBuffer^.Blobs[BlobCount].BlobSize :=
                  StrLen(FQuery.GetFieldBuffer(N))+1;
                GetMem(Buffer, FFilterBuffer^.Blobs[BlobCount].BlobSize);
                Move(FQuery.GetFieldBuffer(N)^,Buffer^,
                  FFilterBuffer^.Blobs[BlobCount].BlobSize);
                FFilterBuffer^.Blobs[BlobCount].BlobData := Buffer;
                FFilterBuffer^.Blobs[BlobCount].FieldNum := Fields[I].FieldNo;
                Inc(BlobCount);
              end;
          end;
        end else begin
          N := FQueryFieldNums[Fields[I].FieldNo];
          Buffer := FQuery.GetFieldBuffer(N);
        end;
        if (Buffer <> nil) then
          FieldToRecord(Fields[I], false, Buffer, FFilterBuffer)
        else DatabaseError(' : ' + Fields[I].FieldName);
      end;
    end;

    Accept := True;
    with TDBRecord.Create(Records) do begin
      Inc(FFetchCount);
      Index := FFetchCount;
      ArrangeIndex := FFetchCount;
      Data := FFilterBuffer;
      Inc(FLastBookMark);
      Data^.Bookmark := FLastBookMark;
      Data^.BookmarkFlag := bfCurrent;
    end;
  until Accept or FQuery.EOF;
end;

// Read all records from server
procedure TMySQLQuery.QueryRecords;
begin
  Records.Clear;
  FQuery.Open;
  FQuery.First;
  FirstFetch := True;
  while (not FQuery.EOF) and QueryAllRecords do QueryRecord;
end;

// Copy TField to internal buffer
procedure TMySQLQuery.FieldToRecord(Field: TField; FieldIsNull: Boolean;
  FieldData: Pointer; RecordData: PRecordData);
var OffSet, DataSize: Integer;
begin
  OffSet := FieldOffset(Field);
  DataSize := Field.DataSize;
  if FieldIsNull then RecordData^.Bytes[Offset] := 1
  else begin
    RecordData^.Bytes[Offset] := 0;
    if RecordData.ArraySize < Offset + 1 + DataSize then
      MessageBeep($FFFF);
    Move(FieldData^, RecordData^.Bytes[OffSet + 1], DataSize);
  end;
end;

// Get field number in FieldDefs
function TMySQLQuery.GetFieldDefNo(Field: TField): Integer;
var I: Integer;
begin
  Result := -1;
  for I := 0 to MAX_FIELDS do begin
    if not Assigned(FRealFields[I]) then continue;
    if Field.FieldName=FRealFields[I].FieldName then begin
      Result := I;
      exit;
    end;
  end;
end;

// Get field offset in internal buffer
function TMySQLQuery.FieldOffset(Field: TField): Integer;
begin
  Result := FieldOffsets[GetFieldDefNo(Field)];
end;

// Internal open query
procedure TMySQLQuery.InternalOpen;
begin
// Create collections for store records
  Records := TDBRecords.Create;
  SaveRecords := TDBRecords.Create;
  FUpdateCmds := TStringList.Create;

  FReadOnly := True;
  FFetchCount := 0;
  FieldOffsets := nil;
// Set initial pos (-1 - records not fetched)
  FCurRec := -1;
  BookmarkSize := SizeOf(Integer);

// Create all demanded connects
  CreateConnections;

  MasterRequery;
  FQuery.Open;

  if not FQuery.Active then
    DatabaseError(FQuery.Dataset.Error);

  InternalInitFieldDefs;
// Create TField components when no persistent fields have been created
  if DefaultFields then CreateFields;
// Bind the TField components to the physical fields
  BindFields(True);
  OptimizeFindFields;

// Calculate the size of the record buffers.
// Note: This is NOT the same as the RecordSize property which
// only gets the size of the data in the record buffer
  FRecBufSize := GetRecBufSize;
// Check to see if we have a unique key...
  FReadOnly := not FRequestLive;

// Set field default values
  InitDefaults;
// Fetch records from server
  QueryRecords;
end;

// Internal close qiery
procedure TMySQLQuery.InternalClose;
begin
//  ApplyUpdates;
  FQuery.Close;
// Close lowerlevel connect to database
  if Assigned(FDbUse) then begin
    FDbUse.Free;
    FDbUse := NIL;
  end;

// Clear field definition and tables lists
  FFieldDescs.Clear;
  FAliases.Clear;
  FTables.Clear;

// Free the record collection
  Records.Free; Records := nil;
  SaveRecords.Free; SaveRecords := nil;
  FUpdateCmds.Free; FUpdateCmds := nil;

// Destroy the TField components if no persistent fields
  if DefaultFields then DestroyFields;
// Reset these internal flags
  FLastBookmark := 0;
  FCurRec := -1;
// Free memory for Field offset array
  if FieldOffsets <> nil then
    FreeMem(FieldOffsets, FieldCount * SizeOf(Integer));
  FieldOffsets := nil;
end;

// Check if cursor open (??)
function TMySQLQuery.IsCursorOpen: Boolean;
begin
  Result := Assigned(Records);
end;

// Define all fields in a query
procedure TMySQLQuery.InternalInitFieldDefs;
var
  I: Integer;
  FieldName: String;
  FieldRequired: Boolean;
  FieldSize: Integer;
  FieldType: TFieldType;
  FieldNo: Integer;
  FieldPrecision : Integer;
  FieldDesc: TMySQLField;
begin
  FieldDefs.Clear;
  FieldNo := 1;

// Create TField for every query field
  if FQuery.Active then begin
    DefineFieldDefs;

    for I := 0 to FQuery.FieldCount - 1 do begin
      FieldRequired := False;
      FieldPrecision := 30;
      FieldDesc := FindFieldDesc(FQuery.FieldName(I), FQuery.FieldName(I));
      if Assigned(FieldDesc) then begin
        FieldName := FieldDesc.Alias;
        FieldType := FieldDesc.FieldType;
        FieldSize := FieldDesc.Length;
      end else begin
        FieldName := FQuery.FieldName(I);
        FieldType := ftString;
        FieldSize := MaxIntValue([FQuery.FieldMaxSize(I), FQuery.FieldSize(I)]);
      end;

      with TFieldDef.Create( FieldDefs, FieldName, FieldType, FieldSize,
        FieldRequired, FieldNo) do begin
        Precision := FieldPrecision;
      end;
      Inc(FieldNo);
    end;
  end;
end;

// Internal exception processing
procedure TMySQLQuery.InternalHandleException;
begin
  Application.HandleException(Self);
end;

// Internal go to bookmark
procedure TMySQLQuery.InternalGotoBookmark(Bookmark: Pointer);
var I, B: Integer;
begin
  B := PInteger(Bookmark)^;
  if (B - 1 > 0) and (B - 1 < Records.Count) then begin
    if B = TDBRecord(Records.Items[B - 1]).Data^.Bookmark then begin
      FCurRec := B - 1;
      exit;
    end;
  end;
  for I := 0 to Records.Count - 1 do begin
    if PInteger(Bookmark)^ = TDBRecord(Records.Items[I]).Data^.Bookmark then begin
      FCurRec := I;
      exit;
    end;
  end;
{$IFDEF RUSSIAN}
  DatabaseError('  ');
{$ELSE}
  DatabaseError('Bookmark not found');
{$ENDIF}
end;

// Internal go to defined record
procedure TMySQLQuery.InternalSetToRecord(Buffer: PChar);
begin
  InternalGotoBookmark(@PRecordData(Buffer).Bookmark);
end;

// Get bookmark flag
function TMySQLQuery.GetBookmarkFlag(Buffer: PChar): TBookmarkFlag;
begin
  Result := PRecordData(Buffer).BookmarkFlag;
end;

// Set bookmark flag
procedure TMySQLQuery.SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag);
begin
  PRecordData(Buffer).BookmarkFlag := Value;
end;

// Get bookmark data
procedure TMySQLQuery.GetBookmarkData(Buffer: PChar; Data: Pointer);
begin
  PInteger(Data)^ := PRecordData(Buffer).Bookmark;
end;

// Set boomark data
procedure TMySQLQuery.SetBookmarkData(Buffer: PChar; Data: Pointer);
begin
  PRecordData(Buffer).Bookmark := PInteger(Data)^;
end;

// Compare two bookmarks
function TMySQLQuery.CompareBookmarks(Bookmark1, Bookmark2: TBookmark): Integer;
var Book1, Book2: Integer;
begin
  Result := 0;
  if not Assigned(Bookmark1) or not Assigned(Bookmark2) then exit;
  Book1 := PInteger(Bookmark1)^;
  Book2 := PInteger(Bookmark2)^;
  if Book1<Book2 then Result := -1
  else if Book1>Book2 then Result := 1;
end;

// Get record buffer size
function TMySQLQuery.GetRecordSize: Word;
begin
  Result := FRecordSize;
end;

// Allocate record buffer in memory
function TMySQLQuery.AllocRecordBuffer: PChar;
var
  B: PRecordData;
  Idx: Integer;
begin
  GetMem(Result, FRecBufSize);
  B := PRecordData(Result);
  B^.ArraySize := FRecBufSize - RecInfoSize;
  for Idx:=0 to 15 do begin
    B^.Blobs[Idx].BlobData := NIL;
    B^.Blobs[Idx].BlobSize := 0;
    B^.Blobs[Idx].FieldNum := 0;
  end;
end;

// Free allocated buffer
procedure TMySQLQuery.FreeRecordBuffer(var Buffer: PChar);
begin
  FreeMem(Buffer);
end;

// Switch between records
function TMySQLQuery.GetRecord(Buffer: PChar; GetMode: TGetMode;
  DoCheck: Boolean): TGetResult;
var
  TempRec: LongInt;
  CanFetch: Boolean;
begin
  CanFetch := (not CachedUpdates) or (rtUnmodified in FUpdateRecord);
  Result := grOK;
  case GetMode of
    gmNext: begin
      TempRec := FCurRec;
      while Result<>grEOF do begin
        if TempRec < Records.Count - 1 then Inc(TempRec)
        else
          if (FQuery.EOF)or(not CanFetch) then Result := grEOF
          else begin
            QueryRecord;
            if FQuery.EOF then Result := grEOF
           else Inc(TempRec);
          end;
        if Result=grEOF then break;
        if CheckRecordByFilter(TempRec) then break;
      end;
      if Result=grOk then FCurRec := TempRec;
    end;

    gmPrior: begin
      TempRec := FCurRec;
      while Result<>grBOF do begin
        if TempRec <= 0 then Result := grBOF
        else Dec(TempRec);
        if Result=grBOF then break;
        if CheckRecordByFilter(TempRec) then break;
      end;
      if Result=grOk then FCurRec := TempRec;
    end;

    gmCurrent: begin
      TempRec := FCurRec;
      while Result<>grError do begin
        if (TempRec < 0) or (TempRec >= Records.Count) then begin
          if (FQuery.EOF)or(not CanFetch) then Result := grError
          else begin
            QueryRecord;
            if FQuery.EOF then Result := grError;
          end;
        end;
        if Result=grError then break;
        if CheckRecordByFilter(TempRec) then break;
        Inc(TempRec);
      end;
      if Result=grOk then FCurRec := TempRec;
    end;
  end;

  if Result = grOK then begin
    Move(TDBRecord(Records.Items[FCurRec]).Data^, Buffer^, FRecBufSize);
    with PRecordData(Buffer)^ do BookmarkFlag := bfCurrent;
    GetCalcFields(Buffer);
  end else if (Result = grError) and DoCheck then begin
    DatabaseError('  .');
  end;
end;

// Clear and initialize new record buffer
procedure TMySQLQuery.InternalInitRecord(Buffer: PChar);
var
  I, N: Integer;
  MasterField: TField;
begin
  FillChar(PRecordData(Buffer)^.Bytes[0], FRecordSize, 1);
  for I := 0 to 15 do begin
    PRecordData(Buffer)^.Blobs[I].BlobData := NIL;
    PRecordData(Buffer)^.Blobs[I].BlobSize := 0;
    PRecordData(Buffer)^.Blobs[I].FieldNum := 0;
  end;

  for I := 0 to FieldCount-1 do begin
    MasterField := Fields[I];
    N := GetRealFieldNo(MasterField.FieldName);
    if N<0 then continue;

    SetRealFieldValue(N, MasterField.DefaultExpression, PRecordData(Buffer));
  end;

  if FMasterLink.Active and (FMasterLink.Fields.Count > 0) then begin
    for I := 0 to FLinkCount-1 do begin
      N := 0; MasterField := NIL;
      while N<FMasterLink.Fields.Count do begin
        MasterField := TField(FMasterLink.Fields[N]);
        if MasterField.FieldName = FMasterFields[I] then break;
        MasterField := NIL;
        Inc(N);
      end;

      if not Assigned(MasterField) then
{$IFDEF RUSSIAN}
        DatabaseError('   "'+FMasterFields[I]+'"');
{$ELSE}
        DatabaseError('Incorrect field name "'+FMasterFields[I]+'"');
{$ENDIF}

      N := GetRealFieldNo(FDetailFields[I]);
      if N<0 then
{$IFDEF RUSSIAN}
        DatabaseError('   "'+FDetailFields[I]+'"');
{$ELSE}
        DatabaseError('Incorrect field name "'+FDetailFields[I]+'"');
{$ENDIF}
      SetRealFieldValue(N, MasterField.AsString, PRecordData(Buffer));
    end;
  end;
end;

// Get current record buffer
function TMySQLQuery.GetActiveRecBuf(var RecBuf: PRecordData): Boolean;
var I: Integer;
begin
  case State of
    dsBrowse:
      if IsEmpty then RecBuf := nil
      else RecBuf := PRecordData(ActiveBuffer);
    dsEdit, dsInsert:
      RecBuf := PRecordData(ActiveBuffer);
    dsCalcFields:
      RecBuf := PRecordData(CalcBuffer);
    dsNewValue, dsCurValue:
      RecBuf := PRecordData(ActiveBuffer);
    dsOldValue:
      begin
        I := FCurRec;
        if I < 0 then I := 0;
        if I < Records.Count then RecBuf := TDBRecord(Records.Items[I]).Data
        else RecBuf := nil;
      end;
  else
    RecBuf := nil;
  end;
  Result := RecBuf <> nil;
end;

// Store record buffer into TField
function TMySQLQuery.GetFieldData(Field: TField; Buffer: Pointer): Boolean;
var B: PRecordData;
begin
  Result := False;
  if not GetActiveRecBuf(b) then exit;

  if B^.Bytes[FieldOffset(Field)] = 0 then begin
    if Buffer <> nil then
      Move(B^.Bytes[FieldOffset(Field) + 1], Buffer^, Field.DataSize);
    Result := True;
  end;
end;

// Retrive data from TField into record buffer
procedure TMySQLQuery.SetFieldData(Field: TField; Buffer: Pointer);
var B: PRecordData;
begin
  if not GetActiveRecBuf(B) then exit;

  if State in [dsEdit, dsInsert] then Field.Validate(Buffer);

  if Buffer = nil then B^.Bytes[FieldOffset(Field)] := 1
  else begin
    B^.Bytes[FieldOffset(Field)] := 0;
    Move(Buffer^, B^.Bytes[FieldOffset(Field) + 1], Field.DataSize);
  end;

  if not (State in [dsCalcFields, dsFilter, dsNewValue]) then
    DataEvent(deFieldChange, Longint(Field));
end;

// Internal go to first record
procedure TMySQLQuery.InternalFirst;
begin
  FCurRec := -1;
end;

// Internal go to last record
procedure TMySQLQuery.InternalLast;
begin
  while (not FQuery.EOF) do QueryRecord;
  FCurRec := Records.Count;
end;

// Save record buffer
procedure TMySQLQuery.SaveRecordData(Buffer: PRecordData; Index: Integer);
var
  B: PRecordData;
  Idx: Integer;
begin
  B := TDBRecord(Records.Items[Index]).Data;
//  FreeBlob(B);
  for Idx:=0 to 15 do
    if Buffer^.Blobs[Idx].BlobData<>B^.Blobs[Idx].BlobData then
      FreeMem(B^.Blobs[Idx].BlobData, B^.Blobs[Idx].BlobSize);

  Move(Buffer^, B^, FRecBufSize);
{
  for Idx:=0 to 15 do begin
    if Buffer^.Blobs[Idx].BlobData=B^.Blobs[Idx].BlobData then continue;

    if Buffer^.Blobs[Idx].BlobData<>NIL then begin
      FreeMem(B^.Blobs[Idx].BlobData, B^.Blobs[Idx].BlobSize);
      GetMem(B^.Blobs[Idx].BlobData, Buffer^.Blobs[Idx].BlobSize);
      Move(Buffer^.Blobs[Idx].BlobData^, B^.Blobs[Idx].BlobData^,
        Buffer^.Blobs[Idx].BlobSize);
      B^.Blobs[Idx].BlobSize := Buffer^.Blobs[Idx].BlobSize;
      B^.Blobs[Idx].FieldNum := Buffer^.Blobs[Idx].FieldNum;
    end;
  end;
}
end;

// Internal edit mode setting
procedure TMySQLQuery.InternalEdit;
var B: PRecordData;
begin
  if not GetActiveRecBuf(B) then Exit;
end;

// Internal updates store
procedure TMySQLQuery.InternalUpdate;
var
  B: PRecordData;
  TempRecord: TDBRecord;
begin
  if not GetActiveRecBuf(B) then Exit;

  if TDBRecord(Records.Items[FCurRec]).FieldStatus=fsNormal then
    TDBRecord(Records.Items[FCurRec]).FieldStatus := fsUpdated;
  if SaveRecords.FindRecord(TDBRecord(Records.Items[FCurRec]).Index)<0 then begin
    TempRecord := TDBRecord(SaveRecords.Add);
    TempRecord.Copy(TDBRecord(Records.Items[FCurRec]));
  end;

// Update the record in the collection
  SaveRecordData(B, FCurRec);
  if not FCachedUpdates then Flush;
end;

// Internal post updates
procedure TMySQLQuery.InternalPost;
 var B: PRecordData;
begin
  if State = dsEdit then InternalUpdate
  else begin
    GetActiveRecBuf(B);
    InternalAddRecord(B, False);
  end;
end;

// Internal add new record
procedure TMySQLQuery.InternalAddRecord(Buffer: Pointer; Append: Boolean);
var
  B: PRecordData;
  AddRecord, TempRecord: TDBRecord;
begin
  if not GetActiveRecBuf(B) then Exit;

  if B <> Buffer then
{$IFDEF RUSSIAN}
    DatabaseError('    InternalAddRecord');
{$ELSE}
    DatabaseError('Internal error in InternalAddRecord');
{$ENDIF}

  if Append then InternalLast;

  AddRecord := TDBRecord(Records.Add);
  if FCurRec >= 0 then AddRecord.Index := FCurRec;
  Inc(FFetchCount);
  AddRecord.Index := FFetchCount;
  AddRecord.ArrangeIndex := FFetchCount;
  AddRecord.Data := PRecordData(AllocRecordBuffer);
  SaveRecordData(B, Records.Count-1);
  Inc(FLastBookmark);
  AddRecord.Data^.Bookmark := FLastBookmark;
  AddRecord.Data^.BookmarkFlag := bfCurrent;

//  ApplyDefaults(AddRecord.Data);
  if FUseGen then  ApplyGens(AddRecord.Data);

  if Append then AddRecord.FieldStatus := fsAppend
  else AddRecord.FieldStatus := fsInserted;

  if SaveRecords.FindRecord(AddRecord.Index)<0 then begin
    TempRecord := TDBRecord(SaveRecords.Add);
    TempRecord.Copy(AddRecord);
  end;

  if not FCachedUpdates then Flush;
//  Resync([]);
end;

// Internal delete record
procedure TMySQLQuery.InternalDelete;
var
  B: PRecordData;
  TempRecord: TDBRecord;
  N: Integer;
begin
  if not GetActiveRecBuf(B) then Exit;
//  if State in [dsEdit, dsInsert] then Post;

  if TDBRecord(Records.Items[FCurRec]).FieldStatus in [fsInserted, fsAppend] then begin
    N := SaveRecords.FindRecord(TDBRecord(Records.Items[FCurRec]).Index);
    Records.Items[FCurRec].Free;
    SaveRecords.Items[N].Free
  end else begin
    TDBRecord(Records.Items[FCurRec]).FieldStatus := fsDeleted;

    N := SaveRecords.FindRecord(TDBRecord(Records.Items[FCurRec]).Index);
    if N<0 then begin
      TempRecord := TDBRecord(SaveRecords.Add);
      TempRecord.Copy(TDBRecord(Records.Items[FCurRec]));
    end else
      TDBRecord(SaveRecords.Items[N]).FieldStatus := fsDeleted;
  end;

  if not FCachedUpdates then Flush;
end;

// Get records quantity
function TMySQLQuery.GetRecordCount: Longint;
begin
  if not FQuery.EOF then Result := FQuery.RecordCount
  else Result := Records.Count;
end;

// Get current record number
function TMySQLQuery.GetRecNo: Longint;
begin
  UpdateCursorPos;
  if (FCurRec = -1) and (RecordCount > 0) then Result := 1
  else Result := FCurRec + 1;
end;

// Set currenct record number
procedure TMySQLQuery.SetRecNo(Value: Integer);
begin
  Value := MaxIntValue([1, Value]);
  while not FQuery.EOF and (Value>Records.Count) do
    QueryRecord;
  if (Value <= Records.Count) then FCurRec := Value - 1
  else FCurRec := Records.Count - 1;
  if not (State in [dsInactive]) then Resync([]);
end;

// Define is query editable?
function TMySQLQuery.GetCanModify: Boolean;
begin
  Result := FRequestLive and not FReadOnly;
end;

// Internal locate record
function TMySQLQuery.InternalLocate(KeyFields: string; KeyValues: Variant;
  Options: TLocateOptions): LongInt;
var
  Count: Integer;
  FieldIndexes: array[0..MAX_FIELDS] of Integer;
  Buffer, Token, Temp1, Temp2: String;
  TokenType: TTokenType;
  I, J: LongInt;
  RecBuffer: PRecordData;
  IsFound: Boolean;
  Value1, Value2: Variant;
begin
  Result := -1;
  Count := 0;
  Buffer := KeyFields;
  while Buffer<>'' do begin
    TokenType := ExtractToken(Buffer, Token);
    case TokenType of
      ttDigit: begin
          FieldIndexes[Count] := StrToInt(Token);
          Inc(Count);
        end;
      ttAlpha: begin
          FieldIndexes[Count] := GetRealFieldNo(Token);
          if FieldIndexes[Count]<0 then
{$IFDEF RUSSIAN}
            DatabaseErrorFmt('   "%s"',[Token]);
{$ELSE}
            DatabaseErrorFmt('Incorrect field name "%s"',[Token]);
{$ENDIF}
          Inc(Count);
        end;
      ttString: begin
          DeleteQuotes(Token);
          FieldIndexes[Count] := GetRealFieldNo(Token);
          if FieldIndexes[Count]<0 then
{$IFDEF RUSSIAN}
            DatabaseErrorFmt('   "%s"',[Token]);
{$ELSE}
            DatabaseErrorFmt('Incorrect field name "%s"',[Token]);
{$ENDIF}
          Inc(Count);
        end;
    end;
  end;

  if Count=1 then KeyValues := VarArrayOf([KeyValues]);
  if VarArrayHighBound(KeyValues,1)-VarArrayLowBound(KeyValues,1)<>Count-1 then
{$IFDEF RUSSIAN}
    DatabaseError('  ');
{$ELSE}
    DatabaseError('Incorrect arguments number');
{$ENDIF}

  I := 0;
  while I<Records.Count do begin
    if Filtered and not CheckRecordByFilter(I) then begin
      Inc(I);
      continue;
    end;
    RecBuffer := TDBRecord(Records.Items[I]).Data;
    IsFound := true;
    for J := 0 to Count-1 do begin
      Value1 := GetRealFieldValue(FieldIndexes[J], RecBuffer);
      Value2 := KeyValues[J];
      if Options=[] then begin
        if Value1<>Value2 then IsFound := false;
      end else begin
        if loCaseInsensitive in Options then begin
          Temp1 := UpperCase(VarToStr(Value1));
          Temp2 := UpperCase(VarToStr(Value2));
        end else begin
          Temp1 := VarToStr(Value1);
          Temp2 := VarToStr(Value2);
        end;
        if loPartialKey in Options then begin
          if not StrCmpBegin(Temp1, Temp2) then IsFound := false;
        end else begin
          if Temp1<>Temp2 then IsFound := false;
        end;
      end;
      if not IsFound then break;
    end;
    if IsFound then begin
      Result := I;
      break;
    end;
    if (I=Records.Count-1) and (not FQuery.EOF) then QueryRecord;
    Inc(I);
  end;
end;

// Locate a record
function TMySQLQuery.Locate(const KeyFields: string; const KeyValues: Variant;
  Options: TLocateOptions): Boolean;
var I: Integer;
begin
  DoBeforeScroll;
  I := InternalLocate(KeyFields, KeyValues, Options);
  if I>=0 then begin
    RecNo := I+1;
    Result := true;
    DoAfterScroll;
  end else Result := false;
  SetFound(Result);
end;

// For lookup fields...
function TMySQLQuery.Lookup(const KeyFields: string; const KeyValues: Variant;
  const ResultFields: string): Variant;
var
  I, Count: Integer;
  RecBuffer: PRecordData;
  ResFields: array[0..50] of Integer;
  ResValues: Variant;
  TokenType: TTokenType;
  Buffer, Token: String;
begin
  Result := False;

  Count := 0;
  Buffer := ResultFields;
  while Buffer<>'' do begin
    TokenType := ExtractToken(Buffer, Token);
    case TokenType of
      ttDigit: begin
          ResFields[Count] := StrToInt(Token);
          Inc(Count);
        end;
      ttAlpha: begin
          ResFields[Count] := GetRealFieldNo(Token);
          if ResFields[Count]<0 then
{$IFDEF RUSSIAN}
            DatabaseErrorFmt('   "%s"',[Token]);
{$ELSE}
            DatabaseErrorFmt('Incorrect field name "%s"',[Token]);
{$ENDIF}
          Inc(Count);
        end;
      ttString: begin
          DeleteQuotes(Token);
          ResFields[Count] := GetRealFieldNo(Token);
          if ResFields[Count]<0 then
{$IFDEF RUSSIAN}
            DatabaseErrorFmt('   "%s"',[Token]);
{$ELSE}
            DatabaseErrorFmt('Incorrect field name "%s"',[Token]);
{$ENDIF}
          Inc(Count);
        end;
    end;
  end;

  I := InternalLocate(KeyFields, KeyValues, []);
  if I<0 then exit;

  RecBuffer := TDBRecord(Records.Items[I]).Data;
  if Count<2 then ResValues := GetRealFieldValue(ResFields[0], RecBuffer)
  else begin
    ResValues := VarArrayCreate([0, Count-1], varVariant);
    for I := 0 to Count-1 do begin
      ResValues[I] := GetRealFieldValue(ResFields[I], RecBuffer);
    end;
  end;
  Result := ResValues;
end;

// Internal refresh query
procedure TMySQLQuery.InternalRefresh;
var
  KeyFields: String;
  KeyValues: Variant;
  Count: Integer;
begin
  Count := RecordCount;
  FormKeyValues(KeyFields, KeyValues);
  ApplyUpdates;

  FQuery.Close;
  Records.Clear;
  FQuery.Open;
  if not FQuery.Active then
{$IFDEF RUSSIAN}
    DatabaseError('  ');
{$ELSE}
    DatabaseError('Error refreshing query');
{$ENDIF}

  FLastBookmark := 0;
  FFetchCount := 0;
  FCurRec := -1;

  FirstFetch := True;
  while (not FQuery.EOF) and (Count>0) do begin
    QueryRecord;
    Dec(Count);
  end;

  if KeyFields<>'' then Locate(KeyFields, KeyValues, []);

// Reread visible records
  if not (State in [dsInactive]) then Resync([]);
end;

// Find record in a filtered query
function TMySQLQuery.FindRecord(Restart, GoForward: Boolean): Boolean;
var
  N: Integer;
  TempParser: TParser;
begin
  CheckBrowseMode;
  DoBeforeScroll;

  TempParser := FParser;
  FParser := FSaveParser;

  N := FCurRec;
  if Restart and GoForward then N := 0
  else if Restart and not GoForward then begin
    while not FQuery.EOF do QueryRecord;
    N := Records.Count-1;
  end else if not Restart and GoForward then Inc(N)
  else if not Restart and not GoForward then Dec(N);
  Result := false;

  while (N>=0) and (N<Records.Count) do begin
    if CheckRecordByFilter(N) then begin
      Result := true;
      break;
    end;
    if not GoForward then Dec(N)
    else begin
      Inc(N);
      if (N>=Records.Count) and not FQuery.EOF then
        QueryRecord;
    end;
  end;

  FParser := TempParser;
  SetFound(Result);
  if Result then begin
    RecNo := N+1;
    DoAfterScroll;
  end;
end;

// Turn off/on filtering
procedure TMySQLQuery.SetFiltered(Value: Boolean);
var
  SavePlace: TBookmark;
begin
  SavePlace := GetBookmark;
  if Value=Filtered then exit;
  inherited SetFiltered(Value);
  if Value then FParser := FSaveParser
  else FParser := NIL;

  if not (State in [dsInactive]) then Resync([]);
  First;
  if not Assigned(FParser) then GotoBookmark(SavePlace);
end;

// Set filter equation
procedure TMySQLQuery.SetFilterText(const Value: String);
var
  TempParser: TParser;
  Changed: Boolean;
  SavePlace: TBookmark;
begin
  Changed := false;
  TempParser := FParser;
  SavePlace := GetBookmark;
  inherited SetFilterText(Value);
  if Value='' then begin
    FSaveParser.Free;
    FSaveParser := NIL;
  end else begin
    if not Assigned(FSaveParser) then FSaveParser := TParser.Create(Self);
    if FSaveParser.Equation<>Value then begin
      FSaveParser.Equation := Value;
      Changed := true;
    end;
  end;
  if Filtered then FParser := FSaveParser
  else FParser := NIL;

  if (TempParser<>FParser)or(Filtered and Changed) then begin
    if not (State in [dsInactive]) then Resync([]);
    First;
    if not Assigned(FParser) then GotoBookmark(SavePlace);
  end;
end;

// Check is query sequensed?
function TMySQLQuery.IsSequenced: Boolean;
begin
  Result := (not Filtered);
end;

{*********************************************************}

// Register component in component palette
procedure Register;
begin
  RegisterComponents(ZEOS_PALETTE, [TMySQLQuery]);
end;

end.
