{******************************************************************
*  (c)copyrights Corona Ltd. Donetsk 1999
*  Project: Zeos Library
*  Module: Extra functions for TMySQLQuery (version 1.0)
*  Author: Sergey Seroukhov   E-Mail: voland@cm.dongu.donetsk.ua
*  Date: 03/02/1999
*
*  List of changes:
******************************************************************}

unit ZMySQLExtra;

interface

uses SysUtils, Classes, Db, ZToken;

{$I ..\Zeos.inc}

type

TIntArray = array[0..1000000] of Integer;
PIntArray = ^TIntArray;
TByteArray = array[0..1000000000] of Byte;
PByteArray = ^TByteArray;

TRecordBlob = packed record
  BlobData: PByteArray;
  BlobSize: Integer;
  FieldNum: Integer;
end;

TBlobArray = array[0..15] of TRecordBlob;

TRecordData = packed record
  Bookmark: Integer;
  BookmarkFlag: TBookmarkFlag;
  ArraySize: Integer;
  Blobs: TBlobArray;
  Bytes: TByteArray;
end;
PRecordData = ^TRecordData;

// Field buffer status
TFieldStatus = (fsNormal, fsInserted, fsAppend, fsDeleted, fsUpdated, fsSaved);

{********************* TDBRecord definition **********************}

// Class for records storage
TDBRecord = class(TCollectionItem)
public
  constructor Create(Collection: TCollection); override;
  destructor Destroy; override;

  procedure Assign(Source: TPersistent); override;
  procedure Copy(Source: TDBRecord);
public
  Index, ArrangeIndex: LongInt; // Field number and its sort index
  Data: PRecordData;            // Buffer pointer
  FieldStatus: TFieldStatus;    // Record status
end;

// Blobs clear
procedure FreeBlob(Buffer: PRecordData);

{********************   TMySQLField *********************}
type

// Key Mode
TKeyMode = (kmNone, kmPrimary, kmIndex, kmUnique);
// Autoupdated values type
TAutoMode = (amNone, amAutoInc, amTimeStamp);

// Field of MySQL Query
TMySQLField = class(TCollectionItem)
public
  Table, Field, Alias: String; // Table name, field name, field alias
  FieldType: TFieldType;       // Field type
  Length: Integer;             // Field length
  KeyMode: TKeyMode;           // Key mode
  AutoMode: TAutoMode;         // Autoupdate type
  IsNull: Boolean;             // Is null
  Def: String;                 // Default value
end;

{******************** TDBRecords definition *******************}

// Buffer records class
TDBRecords = class (TCollection)
public
// Class constructor
  constructor Create;

// Find record by index
  function FindRecord(Index: LongInt): LongInt;
end;

{******************* TFieldValues definition *********************}

// Internal fields processing class
TFieldValues = class (TObject)
private
  FNames, FValues, FAliases: TStringList;

// Get field value by it number
// Index - field number
  function GetField(Index: Integer): String;
// Get field value by it name
// Index - field name
  function GetFieldByName(Index: String): String;
// Get field value by it alias
// Index - field alias
  function GetFieldByAlias(Index: String): String;
// Get field quaitity
  function GetFieldCount: Integer;
// Get field name by it index
// Index - field number
  function GetFieldName(Index: Integer): String;
// Get field alias by it number
// Index - field number
  function GetFieldAlias(Index: Integer): String;
public
// Class constructor
  constructor Create;
// Class destructor
  destructor Destroy; override;

// Add new field
// Name - field name
// Alias - field alias
// Values - field value
  procedure AddField(Name, Alias, Value: String);
// Clear fields
  procedure Clear;

// Field values by it numbers
  property Fields[Index: Integer]: String read GetField;
// Field values by if names
  property FieldsByName[Index: String]: String read GetFieldByName;
// Field values by it aliases
  property FieldsByAlias[Index: String]: String read GetFieldByAlias;
// Fields quantity
  property Count: Integer read GetFieldCount;
// Field names by it numbers
  property FieldNames[Index: Integer]: String read GetFieldName;
// Field aliases by it numbers
  property FieldAliases[Index: Integer]: String read GetFieldAlias;
end;

const
  RecInfoSize = SizeOf(TRecordData) - SizeOf(TByteArray);

{**************** Extra functions definition ****************}

// Extract field from string as Db.Table.Field
function ExtractField(Value: String): String;

// Calc maximum length of enum type value
// Value - type definition as 'enum(val1,val2...valN)'
function EnumMaxLength(Value: String): Integer;

// Split query string to several parts
function SplitQuery(SQL: String;
  var Select, From, Where, Group, OrderBy, Other: String): Boolean;

// Join query string from splitted parts
function FormQuery(const Select, From: String; const Where: String;
                   const Group: String; const OrderBy: String;
                   const Other: String): String;

// Convert string to Ansi SQL escaped string
function StringToSQL(Value: String): String;

type
  TZFieldDataType = (zftUnknown, zftNumeric, zftString, zftDate, zftBoolean, zftMemo);

function CreateField(Sender: TComponent; DataType: TFieldType): TField;
function GetDataType(Field: TField): TZFieldDataType;

implementation

{***************** TDBRecord implementation ******************}

// Free blobs
procedure FreeBlob(Buffer: PRecordData);
var Idx: Integer;
begin
  for Idx:=0 to 15 do begin
    with Buffer^.Blobs[Idx] do begin
      if (BlobData<>NIL){ and (BlobSize>0)} then
        FreeMem(BlobData, BlobSize);
      BlobData := NIL;
      BlobSize := 0;
      FieldNum := 0;
    end;
  end;
end;

// Class constructor
constructor TDBRecord.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  Data := nil;
//  OldData := nil;
  FieldStatus := fsNormal;
end;

// Class destructor
destructor TDBRecord.Destroy;
begin
  if Data <> nil then begin
    FreeBlob(Data);
    FreeMem(Data, Data^.ArraySize + RecInfoSize);
  end;
  inherited;
end;

// Assign another DBRecord object with the same values
// Source - object
procedure TDBRecord.Assign(Source: TPersistent);
begin
  if Source is TDBRecord then begin
    Data := TDBRecord(Source).Data;
    Index := TDBRecord(Source).Index;
    ArrangeIndex := TDBRecord(Source).ArrangeIndex;
    FieldStatus := TDBRecord(Source).FieldStatus;
  end else inherited Assign(Source);
end;

// Copy one DBRecord object values to another
// Source - copy object
// BufLen - buffer length
procedure TDBRecord.Copy(Source: TDBRecord);
var
  FRecBufSize, Idx: Integer;
begin
  FRecBufSize := Source.Data^.ArraySize + RecInfoSize;
  if Data=NIL then begin
    GetMem(Data, FRecBufSize);
    Data^.ArraySize := FRecBufSize - RecInfoSize;
    for Idx:=0 to 15 do begin
      Data^.Blobs[Idx].BlobData := NIL;
      Data^.Blobs[Idx].BlobSize := 0;
      Data^.Blobs[Idx].FieldNum := 0;
    end;
  end;

  Move(TDBRecord(Source).Data^, Data^, FRecBufSize);
  for Idx:=0 to 15 do with TDBRecord(Source).Data^.Blobs[Idx] do begin
    if (BlobData<>NIL) and (BlobSize<>0) then begin
      GetMem(Data^.Blobs[Idx].BlobData, BlobSize);
      Move(BlobData^, Data^.Blobs[Idx].BlobData^, BlobSize);
      Data^.Blobs[Idx].BlobSize := BlobSize;
      Data^.Blobs[Idx].FieldNum := FieldNum;
    end else begin
      Data^.Blobs[Idx].BlobData := NIL;
      Data^.Blobs[Idx].BlobSize := 0;
      Data^.Blobs[Idx].FieldNum := 0;
    end;
  end;
  Index := TDBRecord(Source).Index;
  ArrangeIndex := TDBRecord(Source).ArrangeIndex;
  FieldStatus := TDBRecord(Source).FieldStatus;
end;

{************** TDBRecords implementation ********************}

// Class constructor
constructor TDBRecords.Create;
begin
  inherited Create(TDBRecord);
end;

// Find a record by it index
function TDBRecords.FindRecord(Index: LongInt): LongInt;
var I: LongInt;
begin
  Result := -1;
  for I:=0 to Count-1 do
    if TDBRecord(Items[I]).Index=Index then begin
      Result := I;
      break;
    end;
end;

{*************** TFieldValues implementation ****************}

// Class constructor
constructor TFieldValues.Create;
begin
  FNames := TStringList.Create;
  FValues := TStringList.Create;
  FAliases := TStringList.Create;
end;

// Class destructor
destructor TFieldValues.Destroy;
begin
  FNames.Free;
  FValues.Free;
  FAliases.Free;
end;

// Get field value by it number
// Index - field number
function TFieldValues.GetField(Index: Integer): String;
begin
  Result := '';
  if (Index<0) or (Index>=FValues.Count) then
{$IFDEF RUSSIAN}
    DatabaseError('  ')
{$ELSE}
    DatabaseError('Incorrect field number');
{$ENDIF}
  else
    Result := FValues[Index];
end;

// Get field value by name
// Index - field name
function TFieldValues.GetFieldByName(Index: String): String;
var I: Integer;
begin
  Result := '';
  I := 0;
  while I<FNames.Count do begin
    if FNames[I]=Index then begin
      Result := FValues[I];
      break;
    end;
    Inc(I);
  end;
  if I>=FNames.Count then
{$IFDEF RUSSIAN}
    DatabaseError('  ');
{$ELSE}
    DatabaseError('Incorrect field name');
{$ENDIF}
end;

// Get field value by alias
// Index - field alias
function TFieldValues.GetFieldByAlias(Index: String): String;
var I: Integer;
begin
  Result := '';
  I := 0;
  while I<FAliases.Count do begin
    if FAliases[I]=Index then begin
      Result := FValues[I];
      break;
    end;
    Inc(I);
  end;
  if I>=FNames.Count then
{$IFDEF RUSSIAN}
    DatabaseError('  ');
{$ELSE}
    DatabaseError('Incorrect field alias');
{$ENDIF}
end;

//   
// Name -  
// Alias -  
// Values -  
procedure TFieldValues.AddField(Name, Alias, Value: String);
begin
  if (FNames.IndexOf(Name)>=0) then exit;
  FNames.Add(Name);
  FValues.Add(Value);
  FAliases.Add(Alias);
  if FNames.Count<>FValues.Count then
{$IFDEF RUSSIAN}
    DatabaseError('  TFieldValues');
{$ELSE}
    DatabaseError('TFieldValues internal error');
{$ENDIF}
end;

// Clear fields
procedure TFieldValues.Clear;
begin
  FNames.Clear;
  FValues.Clear;
  FAliases.Clear;
end;

// Get fields quantity
function TFieldValues.GetFieldCount: Integer;
begin
  Result := FNames.Count;
end;

// Get field name by it number
// Index - field number
function TFieldValues.GetFieldName(Index: Integer): String;
begin
  Result := '';
  if (Index<0) or (Index>=FNames.Count) then
{$IFDEF RUSSIAN}
    DatabaseError('  ')
{$ELSE}
    DatabaseError('Incorrect field number')
{$ENDIF}
  else
    Result := FNames[Index];
end;

// Get field alias by it number
// Index - field number
function TFieldValues.GetFieldAlias(Index: Integer): String;
begin
  Result := '';
  if (Index<0) or (Index>=FNames.Count) then
{$IFDEF RUSSIAN}
    DatabaseError('  ')
{$ELSE}
    DatabaseError('Incorrect field number')
{$ENDIF}
  else
    Result := FAliases[Index];
end;

{************** Extra functions implementation ***************}

// Extract field name from string as Db.Table.Field
function ExtractField(Value: String): String;
var P: Integer;
begin
  Result := Value;
  P := Pos('.',Result);
  while (P>0) do begin
    Result := Copy(Result, P+1, Length(Result)-P);
    P := Pos('.',Result);
  end;
end;

// Calc maximum value len of enum type
// Value - type definition as 'enum(val1,val2...valN)'
function EnumMaxLength(Value: String): Integer;
var
  Buffer, Token: String;
  TokenType: TTokenType;
begin
  Buffer := Copy(Value,6,Length(Value)-6);
  Result := 0;
  while Buffer<>'' do begin
    TokenType := ExtractToken(Buffer, Token);
    if (TokenType = ttDelim) or (TokenType = ttUnknown) then continue;
    DeleteQuotes(Token);
    if Length(Token) > Result then Result := Length(Token);
  end;
end;

// Split SELECT query string to several parts
function SplitQuery(SQL: String;
  var Select, From, Where, Group, OrderBy, Other: String): Boolean;
var
  Buffer, Token, Add, Temp: String;
  N: Integer;
begin
  Select := ''; From := ''; Where := '';
  Group := ''; OrderBy := ''; Other := '';
  Buffer := SQL;
  Result := false;

  ExtractToken(Buffer, Token);
  if UpperCase(Token)<>'SELECT' then exit;
  N := 0;

  while Buffer<>'' do begin
    if Buffer[1] in [' ',#9,#10,#13] then Add := ' '
    else Add := '';
    ExtractToken(Buffer, Token);

    Temp := UpperCase(Token);
    if Temp='FROM' then begin
      N := 1; continue;
    end;
    if Temp='WHERE' then begin
      N := 2; continue;
    end;
    if Temp='GROUP' then begin
      N := 3;
      ExtractToken(Buffer, Token);
      if UpperCase(Token)<>'BY' then PutbackToken(Buffer, Token);
      continue;
    end;
    if Temp='ORDER' then begin
      N := 4;
      ExtractToken(Buffer, Token);
      if UpperCase(Token)<>'BY' then PutbackToken(Buffer, Token);
      continue;
    end;
    if (Temp='HAVING')or(Temp='LIMIT')or(Temp='PROCEDURE')or
      (Temp='INTO') then begin
      N := 5;
    end;

    case N of
      0: Select := Select + Add + Token;
      1: From := From + Add + Token;
      2: Where := Where + Add + Token;
      3: Group := Group + Add + Token;
      4: OrderBy := OrderBy + Add + Token;
      else Other := Other + Add + Token;
    end;
  end;

  Result := true;
end;

// Join query string from splitted parts
function FormQuery(const Select, From: String; const Where: String;
                   const Group: String; const OrderBy: String;
                   const Other: String): String;
begin
  Result := 'SELECT ' + Select;
  if From<>'' then Result := Result + ' FROM ' + From;
  if Where<>'' then Result := Result + ' WHERE ' + Where;
  if Group<>'' then Result := Result + ' GROUP BY ' + Group;
  if OrderBy<>'' then Result := Result + ' ORDER BY ' + OrderBy;
  if Other<>'' then Result := Result + ' ' + Other;
end;

// Convert string to escaped Ansi SQL string
function StringToSQL(Value: String): String;
var I: Integer;
begin
  Result := '';
  for I := 1 to Length(Value) do
    case Value[I] of
      #39,'"','\': Result := Result + '\' + Value[I];
      #10: continue;
      #13: Result := Result + '\n';
      #0: Result := Result + '\0';
      else Result := Result + Value[I];
    end;
end;

// Creates new field by type
function CreateField(Sender: TComponent; DataType: TFieldType): TField;
begin
  case DataType of
    ftInteger:  Result := TIntegerField.Create(Sender);
    ftSmallint: Result := TSmallIntField.Create(Sender);
    ftWord:     Result := TWordField.Create(Sender);
    ftBoolean:  Result := TBooleanField.Create(Sender);
    ftFloat:    Result := TFloatField.Create(Sender);
    ftCurrency: Result := TCurrencyField.Create(Sender);
    ftDate:     Result := TDateField.Create(Sender);
    ftTime:     Result := TTimeField.Create(Sender);
    ftDateTime: Result := TDateTimeField.Create(Sender);
    ftString:   Result := TStringField.Create(Sender);
    ftMemo:     Result := TMemoField.Create(Sender);
    ftBlob:     Result := TBlobField.Create(Sender);
    else Result := TField.Create(Sender);
  end;
end;

// Convert one field type to another local type
// Field - field object
function GetDataType(Field: TField): TZFieldDataType;
begin
  case Field.DataType of
    ftSmallint,ftInteger,ftWord,ftFloat,ftCurrency,ftAutoInc :
      Result := zftNumeric;
    ftDate, ftTime, ftDateTime :
      Result := zftDate;
    ftString :
      Result := zftString;
    ftBoolean:
      Result := zftBoolean;
    ftMemo:
      Result := zftMemo;
    else
      Result  := zftUnknown;
  end;
end;

end.
