{******************************************************************
*  (c)copyrights Corona Ltd. Donetsk 1999
*  Project: Zeos Library
*  Module: Abstract class API for direct Transact-server access
*  Author: Sergey Seroukhov   E-Mail: voland@cm.dongu.donetsk.ua
*
*  List of changes:
***************************************************************}

unit ZDirTrSql;

interface
uses SysUtils, ZDirSql, ZSocket, ZConvert;

const MAX_BUFFER_SIZE=1024*8;

type

{**************** TDirTrSQLConnect definition ***************}

// Class for direct connect to MySQL Transact-server
TDirTrSQLConnect = class (TDirConnect)
private
  FSocket: TInetClientSocket;         // Socket descriptior
  FBuffer: array[0..MAX_BUFFER_SIZE] of Char; // Commands buffer
protected
// Get connect handle
  function GetHandle: TInetClientSocket;
// Get exchange buffer
  function GetBuffer: PChar;
public
// Class constructor
  constructor Create;
// Class destructor
  destructor Destroy; override;

// Get a connect status
  function Status: Integer; override;
// Get an error message
  function GetErrorMsg: String; override;

// Connect to transact-server
  function Connect: Integer; override;
// Disconnect from transact server
  procedure Disconnect; override;

// Receive a server message
  function GetRequest: Integer;

// Send a message to transact-server
  function PutMessage(const Frm: string; const Args: array of const): Integer;

// Check correct answer
  function IsOk: Boolean;

// Internal buffer
  property Buffer: PChar read GetBuffer;
// Connect handle
  property Handle: TInetClientSocket read GetHandle;
end;

{**************** TDirTrSqlQuery definition ******************}

// Class for direct query to transact-server
TDirTrSqlQuery = class(TDirQuery)
public
// Class constructors
  constructor Create;
  constructor CreateDb(NewDb: TDirTrSQLConnect);

// Execute a query
  function Exec: Integer; override;
// Commit transaction
  procedure Commit; virtual;
// Rollback transaction
  procedure Rollback; virtual;

// Get a query status
  function Status: Integer; override;
end;


implementation

{***************** TDirTrSqlConnect implementation ****************}

// Class consructor
constructor TDirTrSQLConnect.Create;
begin
  InitClass;
  FHost := 'localhost';
  FPort := '19700';
  FSocket := NIL;
end;

// Class destructor
destructor TDirTrSQLConnect.Destroy;
begin
  Disconnect;
  inherited;
end;

// Get connect handle
function TDirTrSQLConnect.GetHandle: TInetClientSocket;
begin
  Result := FSocket;
end;

// Get internal buffer pointer
function TDirTrSQLConnect.GetBuffer: PChar;
begin
  Result := @FBuffer;
end;

// Receive a server message
function TDirTrSQLConnect.GetRequest: Integer;
begin
  Result := 0;
  if Assigned(FSocket) then
    Result := FSocket.Read(FBuffer, MAX_BUFFER_SIZE-1,0);
  FBuffer[Result] := #0;
end;

// Send a message to server
// Frm - format string
// Args - constants array
function TDirTrSQLConnect.PutMessage(const Frm: string;
  const Args: array of const): Integer;
var Temp: array[0..MAX_BUFFER_SIZE] of Char;
begin
  if not Assigned(FSocket) then Result := 0
  else begin
    StrPCopy(Temp, Format(Frm, Args));
    Result := FSocket.Write(Temp, strlen(Temp),0);
  end;
end;

// Check correct answer
function TDirTrSQLConnect.IsOk: Boolean;
begin
  Result := false;
  if(FBuffer[0]='O') and (FBuffer[1]='K') then Result := true;
end;

// Get a connect status
function TDirTrSQLConnect.Status;
begin
  if not FActive then Result := DB_CONNECTION_NONE
  else Result := DB_CONNECTION_OK;
end;

// Get an error message
function TDirTrSQLConnect.GetErrorMsg: String;
begin
  if(not IsOk) then FError := StrPas(FBuffer);
  Result := ConvKoi2Win(FError);
end;

// Compare two strings from end
function StrEndComp(Str1, Str2: PChar): Integer;
var P1, P2: Integer;
begin
  P1 := StrLen(Str1);
  P2 := StrLen(Str2);
  Result := 0;
  while (P1>0) and (P2>0) do begin
    Result := Ord(Str1[P1]) - Ord(Str2[P2]);
    if Result<>0 then break;
    Dec(P1);
    Dec(P2);
  end;
end;

// Connect to transact-server
function TDirTrSQLConnect.Connect;
label ErrLabel;
begin
  Disconnect();

  FSocket := TInetClientSocket.CreateByName('localhost', StrToIntDef(FPort,19700));
  if FSocket.ConnectSocket(FHost, StrToIntDef(FPort,19700))<>0 then begin
ErrLabel:
    StrPCopy(FBuffer,'Connect to transact-server error');
    FSocket.Free;
    FSocket := NIL;
    Result := DB_ERROR;
    exit;
  end;

  GetRequest;
  if(StrEndComp(FBuffer,'login: ')<>0) then begin
    GetRequest;
    if(StrEndComp(FBuffer,'login: ')<>0) then goto ErrLabel;
  end;
  PutMessage('%s'+#10,[FLogin]);
  GetRequest;
  if(StrEndComp(FBuffer,'password: ')<>0) then goto ErrLabel;
  PutMessage('%s'+#10,[FPasswd]);
  GetRequest;
  if(StrComp(FBuffer,#10+#13)<>0) then goto ErrLabel;

  PutMessage('CONNECT %s;'+#10,[FDb]);
  GetRequest;
  if not IsOk then goto ErrLabel;

  FActive := true;
  Result := DB_CONNECTION_OK;
end;

// Disconnect from transact-server
procedure TDirTrSQLConnect.Disconnect;
begin
  if Assigned(FSocket) then begin
    PutMessage('QUIT;'+#10,[0]);
    FSocket.Free;
    FSocket := NIL;
  end;
  FActive := false;
end;

{******************* TDirTrSqlQuery implementation **********************}

// Class constructor
constructor TDirTrSqlQuery.Create;
begin
  InitClass;
end;

// Class destructor
constructor TDirTrSqlQuery.CreateDb(NewDb: TDirTrSQLConnect);
begin
  InitClass;
  FDb := NewDb;
end;

// Post a command to transaction buffer
function TDirTrSqlQuery.Exec: Integer;
var Trd: TDirTrSQLConnect;
begin
  Trd := TDirTrSQLConnect(FDb);
  FHaveError := false;
  Result := -1;
  if not Assigned(FDb) or not FDb.Active then begin
    FHaveError := true;
    exit;
  end;
  Trd.PutMessage('%s;'+#10,[FSQL]);
  Trd.GetRequest;
  if not Trd.IsOk then begin
    FHaveError := true;
    exit;
  end;
  Result := 0;
end;

// Commit transaction
procedure TDirTrSqlQuery.Commit;
var Trd: TDirTrSQLConnect;
begin
  Trd := TDirTrSQLConnect(FDb);
  FHaveError := false;
  if not Assigned(FDb) or not FDb.Active then begin
    FHaveError := true;
    exit;
  end;
  Trd.PutMessage('COMMIT;'+#10,[0]);
  Trd.GetRequest;
  if not Trd.IsOk then FHaveError := true;
end;

// Rollback transaction
procedure TDirTrSqlQuery.Rollback;
var Trd: TDirTrSQLConnect;
begin
  Trd := TDirTrSQLConnect(FDb);
  FHaveError := false;
  if not Assigned(FDb) or not FDb.Active then begin
    FHaveError := true;
    exit;
  end;
  Trd.PutMessage('ROLLBACK;'+#10,[0]);
  Trd.GetRequest;
  if not Trd.IsOk then FHaveError := true;
end;

// Get a query status
function TDirTrSqlQuery.Status: Integer;
begin
  if FHaveError then Result := DB_ERROR
  else Result := DB_COMMAND_OK;
end;

end.
