{******************************************************************
*  (c)copyrights Corona Ltd. Donetsk 1999
*  Project: Zeos Library
*  Module: Abstract class API for direct database access (version 1.2)
*  Author: Sergey Seroukhov   E-Mail: voland@cm.dongu.donetsk.ua
*  Date: 01/12/98
*
*  List of changes:
******************************************************************}

unit ZDirSql;

interface
uses SysUtils, Classes, ZToken;

const
  DB_BUFF_MAX = 1024;      // Max buffer size

  DB_CONNECTION_NONE = 0;
  DB_CONNECTION_OK = 1;
  DB_CONNECTION_BAD = 2;

  DB_COMMAND_OK	= 0;	// Command execute correctly
  DB_EMPTY_QUERY = 1;	// Query isn't return rows
  DB_TUPLES_OK = 2;	// Query return rows
  DB_ERROR = 5;         // Error was returned
  DB_BAD_RESPONSE = 6;  // Bad server response
  DB_UNEXPECTED = 7;	// Unknown error

type

{******************** TDirConnect definition **********************}

// Abstract class for direct database connection
TDirConnect = class(TObject)
protected
  FActive: Boolean;
  FError: String;
  FHost, FPort, FDb, FLogin, FPasswd: String;

// Get error message
  function GetErrorMsg : String; virtual;
// Class initialization
  procedure InitClass;
public
// Class constructor
  constructor Create;
// Class destructor
  destructor Destroy; override;

// Initialization
  function Init: Integer; virtual;
// Get connect status
  function Status: Integer; virtual;

// Connect to database
  function Connect: Integer; virtual;
// Connect to database with full params
  function ConnectFull(NewHost, NewPort, NewDb, NewLogin,
                   NewPasswd : String) : Integer; virtual;
// Break connection
  procedure Disconnect; virtual;
//
  function Reset: Integer; virtual;
// Create a new database
  function CreateNew: Integer; virtual;
// Drop the current database
  function Drop: Integer; virtual;

// Is connect active?
  property Active: Boolean read FActive;
// Host name of database server
  property HostName: String read FHost write FHost;
  property Port: String read FPort write FPort;
  property Db: String read FDb write FDb;
  property Login: String read FLogin write FLogin;
  property Passwd: String read FPasswd write FPasswd;
  property Error: String read GetErrorMsg;
end;

{******************** TDirQuery definition *********************}

// Abstract class for database query
TDirQuery = class
protected
  FActive, FHaveError, FReadOnly: Boolean;
  FDb: TDirConnect;
  FRecno: Integer;
  FBOF, FEOF: Boolean;
  FSQL: String;
  FFldNames, FFldValues: TStringList;

// Class initialization
  procedure InitClass;
// Is begin of rows
  function GetBOF: Boolean; virtual;
// Is end of rows
  function GetEOF: Boolean; virtual;
public
// Class constructor
  constructor Create;
// Class constructor with Connect handle
  constructor CreateDb(NewDb: TDirConnect);
// Class destructor
  destructor Destroy; override;

  procedure SetSqlFormat(const Frm: string; const Args: array of const);

// Execute a query without rows returning
  function Exec: Integer; virtual;
// Open a query with row retuning
  procedure Open; virtual;
// Close an opened query
  procedure Close; virtual;

// Go to first row
  procedure First; virtual;
// Go to last row
  procedure Last; virtual;
// Go to prev row
  procedure Prev; virtual;
// Go to next row
  procedure Next; virtual;
// Go to row with number N
  procedure Go(N: Integer); virtual;
// Locate the row
  function  Locate(Params: String): Boolean;
// Find next row
  function  FindNext: Boolean;

// Get a query status
  function Status: Integer; virtual;

// Get a field count
  function FieldCount: Integer; virtual;
// Get a record count
  function RecordCount: Integer; virtual;

  function FieldName(N: Integer): String; virtual;
  function FieldIndex(FN: String): Integer; virtual;
  function FieldSize(N: Integer): Integer; virtual;

  function GetField(N: Integer): String; virtual;
  function GetFieldByName(FN: String): String; virtual;
  procedure SetField(N: Integer; Value: String); virtual;
  procedure SetFieldByName(FN: String; Value: String); virtual;

  property Active: Boolean read FActive;
  property Dataset: TDirConnect read FDb write FDb;
  property SQL: String read FSQL write FSQL;
  property BOF: Boolean read GetBOF;
  property EOF: Boolean read GetEOF;
  property RecordNo: Integer read FRecno;
  property ReadOnly: Boolean read FReadOnly;
  property Fields[Index: Integer]: String read GetField write SetField; default;
  property FieldByName[Index: String]: String
             read GetFieldByName write SetFieldByName;
end;

implementation

{********************* TDirConnect implementation *********************}

function TDirConnect.GetErrorMsg: String;
begin
  Result := FError;
end;

//   
procedure TDirConnect.InitClass;
begin
  FActive := false;
  FError := 'Not connected';
  FHost := 'localhost';
  FPort := '';
  FDB := '';
  FLogin := '';
  FPasswd := '';
end;

constructor TDirConnect.Create;
begin
  InitClass;
end;

destructor TDirConnect.Destroy;
begin
  Disconnect;
end;

function TDirConnect.Init: Integer;
begin
  Result := DB_UNEXPECTED;
end;

function TDirConnect.Status;
begin
  Result := DB_CONNECTION_NONE;
end;

function TDirConnect.Connect: Integer;
begin
  Result := DB_UNEXPECTED;
end;

function TDirConnect.ConnectFull(NewHost, NewPort, NewDb, NewLogin,
  NewPasswd: String): Integer;
begin
  FHost := NewHost;
  FPort := NewPort;
  FDb := NewDb;
  FLogin := NewLogin;
  FPasswd := NewPasswd;
  Result := Connect;
end;

procedure TDirConnect.Disconnect;
begin
  Init;
  FActive := false;
end;

function TDirConnect.Reset:Integer;
begin
  Result := DB_UNEXPECTED;
end;

function TDirConnect.CreateNew:Integer;
begin
  Result := DB_UNEXPECTED;
end;

function TDirConnect.Drop:Integer;
begin
  Result := DB_UNEXPECTED;
end;

{******************** TDirQuery implementation ******************}

function BIF(A, B, C: Boolean): Boolean;
begin
  if A then Result := B else Result := C;
end;

function IIF(A: Boolean; B, C: Integer): Integer;
begin
  if A then Result := B else Result := C;
end;

// Class initialization
procedure TDirQuery.InitClass;
begin
  FHaveError  := false;
  FActive := false;
  FDB := NIL;
  FRecno := 0;
  FBOF := true;
  FEOF := true;
  FSQL := '';
  FReadOnly := true;
  FFldNames := TStringList.Create;
  FFldValues := TStringList.Create;
end;

// Class constructor
constructor TDirQuery.Create;
begin
  InitClass;
end;

// Class constructor
// NewDB - database connect handle
constructor TDirQuery.CreateDB(NewDb: TDirConnect);
begin
  InitClass;
  FDB := NewDb;
end;

// Class destructor
destructor TDirQuery.Destroy;
begin
  Close;
  FFldNames.Free;
  FFldValues.Free;
end;

// Execute a query without rows returning
function TDirQuery.Exec: Integer;
begin
  FFldNames.Clear;
  FFldValues.Clear;
  Result := 0;
end;

// Open a query
procedure TDirQuery.Open;
begin
  FFldNames.Clear;
  FFldValues.Clear;
end;

// Set sql query with format string
procedure TDirQuery.SetSqlFormat(const Frm: string; const Args: array of const);
begin
  FSQL := Format(Frm, Args);
end;

// Close an open query
procedure TDirQuery.Close;
begin
  FFldNames.Clear;
  FFldValues.Clear;
  FActive := false;
  FHaveError := false;
  FRecno := 0;
  FBOF := true;
  FEOF := true;
end;

// Go to first row
procedure TDirQuery.First;
begin
  FRecno := 0;
  FBOF := BIF(RecordCount>0, false, true);
  FEOF := FBOF;
end;

// Go to last row
procedure TDirQuery.Last;
begin
  FRecno := IIF(RecordCount>0, RecordCount - 1, 0);
  FBOF := BIF(RecordCount>0, false, true);
  FEOF := FBOF;
end;

// Go to prior row
procedure TDirQuery.Prev;
begin
  FEOF := false;
  if (FRecno>0) then begin
    Dec(FRecno);
    FBOF := false;
  end else FBOF := true;
  if (RecordCount <= 0) then begin
    FBOF := true;
    FEOF := true;
  end;
end;

// Go to next row
procedure TDirQuery.Next;
begin
  FBOF := false;
  if (FRecno < (RecordCount-1)) then begin
    Inc(FRecno);
    FEOF := false;
  end else FEOF := true;
  if (RecordCount<=0) then begin
    FBOF := true;
    FEOF := true;
  end;
end;

// Go to row with N number
// N - row number
procedure TDirQuery.Go(N: Integer);
begin
  FRecno := IIF(N<(RecordCount-1), N, RecordCount-1);
  FRecno := IIF(FRecno<0, 0, FRecno);
  FBOF := BIF(FRecno<=0, true, false);
  FEOF := BIF(FRecno<(RecordCount-1), false, true);
end;

// Get a query status
function TDirQuery.Status: Integer;
begin
  Result := DB_COMMAND_OK;
end;

// Get field quantity in a query
function TDirQuery.FieldCount: Integer;
begin
  Result := 0;
end;

// Get a record quantity in a query
function TDirQuery.RecordCount: Integer;
begin
  Result := 0;
end;

// Get a field name by it number
// N - field number
function TDirQuery.FieldName(N: Integer): String;
begin
  Result := '';
end;

// Get field number by it name
// FN - field name
function TDirQuery.FieldIndex(FN: String): Integer;
begin
  Result := 0;
end;

// Get a field size
// N - field number
function TDirQuery.FieldSize(N: Integer): Integer;
begin
  Result := 0;
end;

// Get a field value by it number
// N - field number
function TDirQuery.GetField(N: Integer): String;
begin
  Result := '';
end;

// Get a field value by it name
// FN - field name
function TDirQuery.GetFieldByName(FN: String): String;
begin
  Result := GetField(FieldIndex(FN));
end;

// Set a field value by it number
// N - field number
// Value - new field value
procedure TDirQuery.SetField(N: Integer; Value: String);
begin
end;

// Set a field value by it name
// FN - field name
// Value - new field value
procedure TDirQuery.SetFieldByName(FN: String; Value: String);
begin
  SetField(FieldIndex(FN), Value);
end;

// Find a first row equal to params
// Params - params string as "field=value..."
function TDirQuery.Locate(Params: String): Boolean;
var I, N: Integer;
begin
  Result := false;
  SplitParams(Params, FFldNames, FFldValues);
  if FFldValues.Count=0 then exit;

  for I:=FFldValues.Count-1 downto 0 do begin
    if IsDigit(FFldNames[I][1]) then N := StrToIntDef(FFldNames[I], -1)
    else N := FieldIndex(FFldNames[I]);
    if (N<0) or (N>=RecordCount) then begin
      FFldNames.Delete(I);
      FFldValues.Delete(I);
    end else FFldValues.Objects[I] := TObject(N);
  end;

  First;
  while not EOF do begin
    Result := true;
    for I:=0 to FFldValues.Count-1 do
      if GetField(Integer(FFldValues.Objects[I]))<>FFldValues[I] then begin
        Result := false;
        break;
      end;
    if Result then exit;
    Next;
  end;
end;

// Find a next by locate row
function TDirQuery.FindNext: Boolean;
var I: Integer;
begin
  Result := false;
  if FFldValues.Count=0 then exit;

  Next;
  while not EOF do begin
    Result := true;
    for I:=0 to FFldValues.Count-1 do
      if GetField(Integer(FFldValues.Objects[I]))<>FFldValues[I] then
        Result := false;
    if Result then exit;
    Next;
  end;
end;

// Is begin of rows?
function TDirQuery.GetBOF: Boolean;
begin
  Result := FBOF;
end;

// Is end of rows?
function TDirQuery.GetEOF: Boolean;
begin
  Result := FEOF;
end;

end.
