{******************************************************************
*  (c)copyrights Corona Ltd. Donetsk 1999
*  Project: Zeos Library
*  Module: Abstract class API for direct MySQL access (version 1.2)
*  Author: Sergey Seroukhov   E-Mail: voland@cm.dongu.donetsk.ua
*
*  List of changes:
*  1.1  Ported from C++ unix Zeos Library
*  1.2  Added several functions
*  04/03/99 - Clear virtual constructor...
*  13/03/99 - Add use_result mode
******************************************************************}

unit ZDirMySql;

interface
uses SysUtils, ZDirSql, LibMySQL, Dialogs;

type

{****************** TDirMySQLConnect definition ********************}

// Direct connect class to MySQL
TDirMySQLConnect = class(TDirConnect)
private
  FMysqlRec: MYSQL; // Connect handle
protected
// Get an error message
  function GetErrorMsg: String; override;
// Get a connect handle
  function GetHandle: PMYSQL;
public
// Class constructor
  constructor Create;
// Class destructor
  destructor Destroy; override;

// Get a connect status
  function Status: Integer; override;

// Connect to database server
  function Connect: Integer; override;
// Break connection to server
  procedure Disconnect; override;
// Create a new database
  function CreateNew: Integer; override;
// Drop the database
  function Drop: Integer; override;

// Connect database handle
  property Handle: PMYSQL read GetHandle;
end;

{****************** TDirMysqlQuery definition ********************}

// Open mode
TOpenMode = (omUse, omStore);

// Direct query to MySQL
TDirMySqlQuery = class (TDirQuery)
protected
  FResult: PMYSQL_RES; // Result rows handle
  FOpenMode: TOpenMode;
  FRow: PMYSQL_ROW;

// Is end of query
  function GetEOF: Boolean; override;
public
// Class constructors
  constructor Create;
  constructor CreateDb(NewDb: TDirMySQLConnect);

// Execute a query
  function Exec: Integer; override;
// Open a query
  procedure Open; override;
// Close an open query
  procedure Close; override;

// Get a query status
  function Status: Integer; override;

// Go to next row
  procedure Next; override;

// Get fields quantity
  function FieldCount: Integer; override;
// Get a row quantity
  function RecordCount: Integer; override;

// Get a field name by it number
  function FieldName(N: Integer): String; override;
// Get a field number by it name
  function FieldIndex(FN: String): Integer; override;
// Get a field size
  function FieldSize(N: Integer): Integer; override;
// Get a maximum size of field
  function FieldMaxSize(N: Integer): Integer;
// Get a field type
  function FieldType(N: Integer): ENUM_FIELD_TYPES;
// Get a field flags
  function FieldFlags(N: Integer): Integer;
// Get a field decimals
  function FieldDecimals(N: Integer): Integer;

// Get a field value
  function GetField(N: Integer): String; override;
// Get a field value's buffer
  function GetFieldBuffer(N: Integer): PChar;
// Set open mode - use or store results
  procedure SetOpenMode(Value: TOpenMode);
// Get an open mode
  function GetOpenMode: TOpenMode;
end;

implementation

uses ZConvert;

{*************** TDirMySQLConnect implementation *****************}

// Class constructor
constructor TDirMySQLConnect.Create;
begin
  InitClass;
  Port := '3306';
end;

// Class destructor
destructor TDirMySQLConnect.Destroy;
begin
  inherited;
end;

// Get a connect status
function TDirMySQLConnect.Status: Integer;
begin
  if not Active then Result := DB_CONNECTION_NONE
  else Result := DB_CONNECTION_OK;
end;

// Get an error message
function TDirMySQLConnect.GetErrorMsg: String;
begin
  if (not Active) then Result := FError
  else begin
    FError := Trim(StrPas(@FMysqlRec._net.last_error));
    Result := ConvKoi2Win(FError);
  end;
end;

// Get a connect handle
function TDirMySQLConnect.GetHandle: PMYSQL;
begin
  Result := @FMysqlRec;
end;

// Connect to database server
function TDirMySQLConnect.Connect: Integer;
begin
  Disconnect;

  mysql_init(@FMysqlRec);

  if (mysql_real_connect(@FMysqlRec, PChar(FHost), PChar(FLogin), PChar(FPasswd),
     PChar(FDB), MYSQL_PORT, NIL, _CLIENT_CONNECT_WITH_DB) = NIL) then begin
    FError := 'Connect to server error';
    Result := DB_ERROR;
    exit;
  end;

  FActive := true;
  Result := DB_CONNECTION_OK;
end;

// Break connection to server
procedure TDirMySQLConnect.Disconnect;
begin
  if FActive then begin
    mysql_close(@FMysqlRec);
    FActive := false;
  end;
end;

// Create a new database
function TDirMySQLConnect.CreateNew;
begin
  FActive := false;
  if (mysql_connect(@FMysqlRec,PChar(FHost),PChar(FLogin),PChar(FPasswd))=NIL) then
  begin
    FError := 'Connect to server error';
    Result := DB_ERROR;
    exit;
  end;

  if (mysql_create_db(@FMysqlRec, PChar(FDB))<>0) then begin
    mysql_close(@FMysqlRec);
    FError := 'Database creating error';
    Result := DB_ERROR;
    exit;
  end;

  FActive := true;
  Result := DB_CONNECTION_OK;
end;

// Drop the database
function TDirMySQLConnect.Drop: Integer;
begin
  if (not FActive) then Result := DB_ERROR
  else begin
    mysql_drop_db(@FMysqlRec, PChar(FDB));
    mysql_close(@FMysqlRec);
    FActive := false;
    Result := DB_COMMAND_OK;
  end;
end;

{****************** TDirMySqlQuery implementation ******************}

// Class constructor
constructor TDirMySqlQuery.Create;
begin
  InitClass;
  FResult := NIL;
  FHaveError := false;
  FDB := NIL;
  FOpenMode := omStore;
  FRow := NIL;
end;

// Class constructor
// NewDB - connect to database
constructor TDirMySqlQuery.CreateDB(NewDB: TDirMySQLConnect);
begin
  InitClass;
  FResult := NIL;
  FHaveError := false;
  FDB := NewDB;
  FOpenMode := omStore;
  FRow := NIL;
end;

// Close an open query
procedure TDirMySqlQuery.Close;
begin
  inherited Close;
  if (FResult <> NIL) then
    mysql_free_result(FResult);
  FResult := NIL;
  FActive := false;
  FRow := NIL;
end;

// Execute a query without rows returning
function TDirMySqlQuery.Exec: Integer;
var FMysqlRec: PMYSQL;
begin
  inherited Exec;
  FResult := NIL;

  if (FDB <> NIL) then begin
    FMysqlRec := TDirMySQLConnect(FDB).Handle;

    if (mysql_query(FMysqlRec, PChar(FSQL)) = 0) then begin
      FHaveError := false;
      Result := DB_COMMAND_OK;
      exit;
    end;
  end;

  FHaveError := true;
  Result := DB_ERROR;
end;

// Open a query
procedure TDirMySqlQuery.Open;
var FMysqlRec: PMYSQL;
begin
  if not Assigned(FDB) or Active then exit;
  inherited Open;
  FMysqlRec := TDirMySQLConnect(FDB).Handle;
  FRow := NIL;

  if (mysql_query(FMysqlRec, PChar(FSQL)) = 0) then begin
    if FOpenMode=omStore then FResult := mysql_store_result(FMysqlRec)
    else FResult := mysql_use_result(FMysqlRec);

    if Assigned(FResult) then begin
      if (FResult^.field_count <> 0) then begin
        if FOpenMode=omUse then FRow := mysql_fetch_row(FResult);

        FActive := true;
        FHaveError := false
      end else FHaveError := true;
    end else FHaveError := true;
  end else FHaveError := true;
  First;
end;

// Get a query status
function TDirMySqlQuery.Status: Integer;
begin
  if (FHaveError) then Result := DB_ERROR
  else
    if (FResult = NIL) then Result := DB_COMMAND_OK
    else Result := DB_TUPLES_OK;
end;

// Get a rows quantity
function TDirMySqlQuery.RecordCount: Integer;
begin
  if (FResult = NIL) then Result := 0
  else  Result := FResult^.row_count;
end;

// Get a fields quantity
function TDirMySqlQuery.FieldCount: Integer;
begin
  if (FResult = NIL) then Result := 0
  else Result := FResult^.field_count;
end;

// Get a field name by it number
// N - field number
function TDirMySqlQuery.FieldName(N: Integer): String;
var
  Field: PMYSQL_FIELD;
  I, P: Integer;
begin
  if (FResult = NIL) then Result := ''
  else begin
    mysql_field_seek(FResult, N);
    Field := mysql_fetch_field(FResult);
    if (Field = NIL) then Result := ''
    else Result := Field.name;

    P := 0;
    for I := 0 to N-1 do begin
      mysql_field_seek(FResult, I);
      Field := mysql_fetch_field(FResult);
      if (Field<>NIL) and (Field.name=Result) then Inc(P);
    end;
    if P<>0 then
      Result := Result + '_' + IntToStr(P);
  end;
end;

// Get a field size
// N - field number
function TDirMySqlQuery.FieldSize(N: Integer): Integer;
var Field : PMYSQL_FIELD;
begin
  if (FResult = NIL) then Result := 0
  else begin
    mysql_field_seek(FResult, N);
    Field := mysql_fetch_field(FResult);
    if (Field = NIL) then Result := 0
    else Result := Field.length;
  end;
end;

// Get a field number by it name
// FN - field name
function TDirMySqlQuery.FieldIndex(FN: String): Integer;
var
  I, P: Integer;
  Name, Num: String;
begin
  Result := -1;
  if (FResult <> NIL) then begin
    for I := 0 to FieldCount-1 do
      if (FN = FieldName(I)) then begin
        Result := I;
        break;
      end;
  end else exit;
  if Result<>-1 then exit;
  Name := ''; Num := ''; P := 1;
  for I := Length(FN) downto 1 do
    if P=1 then begin
      if FN[I]='_' then P := 0 else Num := FN[I] + Num;
    end else Name := FN[I] + Name;
  if Name='' then exit;
  P := StrToIntDef(Num,0)+1;
  if P<=1 then exit;
  for I := 0 to FieldCount-1 do begin
    if Name = FieldName(I) then Dec(P);
    if P=0 then begin
      Result := I;
      exit;
    end;
  end;
end;

// Get a maxumum field size
// N - field number
function TDirMySqlQuery.FieldMaxSize(N: Integer): Integer;
var Field: PMYSQL_FIELD;
begin
  if (FResult = NIL) then Result := 0
  else begin
    mysql_field_seek(FResult, N);
    Field := mysql_fetch_field(FResult);
    if (Field = NIL) then Result := 0
    else Result := Field.max_length;
  end;
end;

// Get a field type
// N - field number
function TDirMySqlQuery.FieldType(N: Integer): ENUM_FIELD_TYPES;
var Field: PMYSQL_FIELD;
begin
  if (FResult = NIL) then Result := 0
  else begin
    mysql_field_seek(FResult, N);
    Field := mysql_fetch_field(FResult);
    if (Field = NIL) then Result := 0
    else Result := Field._type;
  end;
end;

// Get a field flags
// N - field number
function TDirMySqlQuery.FieldFlags(N: Integer): Integer;
var Field: PMYSQL_FIELD;
begin
  if (FResult = NIL) then Result := 0
  else begin
    mysql_field_seek(FResult, N);
    Field := mysql_fetch_field(FResult);
    if (Field = NIL) then Result := 0
    else Result := Field.flags;
  end;
end;

// Get a field decimals
// N - field number
function TDirMySqlQuery.FieldDecimals(N: Integer): Integer;
var Field: PMYSQL_FIELD;
begin
  if (FResult = NIL) then Result := 0
  else begin
    mysql_field_seek(FResult, N);
    Field := mysql_fetch_field(FResult);
    if (Field = NIL) then Result := 0
    else Result := Field.decimals;
  end;
end;

// Get a field value
// N - field number
function TDirMySqlQuery.GetField(N: Integer): String;
var Row: PMYSQL_ROW;
begin
  Result := '';
  if FResult=NIL then exit;
  if FOpenMode=omStore then begin
    mysql_data_seek(FResult, FRecno);
    Row := mysql_fetch_row(FResult);
  end else Row := FRow;
  Inc(Row, N);
  if Assigned(Row) then Result := StrPas(PChar(Row^))
  else Result := '';
end;

// Get a field value's buffer
// N - field number
function TDirMySqlQuery.GetFieldBuffer(N: Integer): PChar;
var Row: PMYSQL_ROW;
begin
  Result := '';
  if ((FResult = NIL) or EOF or BOF) then exit;
  if FOpenMode=omStore then begin
    mysql_data_seek(FResult, FRecno);
    Row := mysql_fetch_row(FResult);
  end else Row := FRow;
  Inc(Row, N);
  Result := PChar(Row^);
  if Result=NIL then Result := '';
end;

// Set open query mode
// Value - open mode
procedure TDirMySqlQuery.SetOpenMode(Value: TOpenMode);
begin
  FOpenMode := Value;
end;

// Get a query open mode
function TDirMySqlQuery.GetOpenMode: TOpenMode;
begin
  Result := FOpenMode;
end;

// Is end of rows?
function TDirMySqlQuery.GetEOF: Boolean;
begin
  if FOpenMode=omStore then Result := FEOF
  else Result := FRow=NIL;
end;

// Go to next row
procedure TDirMySqlQuery.Next;
begin
  inherited;
  if FOpenMode=omUse then FRow := mysql_fetch_row(FResult);
end;

end.
