{******************************************************************
*  (c)copyrights Corona Ltd. Donetsk 1999
*  Project: Zeos Library
*  Module: Classes for Winsock API (version 1.0)
*  Author: Sergey Seroukhov   E-Mail: voland@cm.dongu.donetsk.ua
*  Date: 01/12/98
*
*  List of changes:
*  12/03/99 - delete Exception in Winsock closed
******************************************************************}

unit ZSocket;

interface
uses Classes, Windows, Winsock, SysUtils;

{$I ..\Zeos.inc}

const
  BACKLOG_NUM = 5;

type

EWinsockError = class(Exception);

{******************* TURL definition *****************}

// Class to incapsulate Universal Resource Locator
TURL = class
private
  FHostName: String;
  FFileName, FPath: String;
  FPort: Integer;
  FIP: u_long;

// Convert an IP address to a host name
  function IP2Name(IP:u_long): String;
  function Name2IP(const HostNm: String): u_long;

  procedure FillUrlByName(const NewHost: String; NewPort: Integer;
    const NewFile: String);
  procedure FillUrlByIP(NewIP: u_long; NewPort: Integer; const NewFile: String);

public
// Class constructors
  constructor Create;
  constructor CreateByName(const NewHost: String; NewPort: Integer;
    NewFile: String);
  constructor CreateByIP(NewIP: u_long; NewPort: Integer; NewFile: String);

  function FillAddr(var Addr: TSockAddrIn): Integer;
  function GetHostName: String;
  procedure SetHostName(const NewHost: String);

  function GetIP: u_long;
  procedure SetIP(NewIP: u_long);
  procedure Assign(NewURL: TURL );
  function GetUrl: String;

  property FileName: String read FFileName write FFileName;
  property Port: Integer read FPort write FPort;
  property URL: String read GetUrl;
  property HostName: String read GetHostName write SetHostName;
  property IP: u_long read GetIP write SetIP;
end;

{**************** TInetSocket definition *****************}

// TCP-IP socket abstract class
TInetSocket=class
private
  FServer, FClient: TURL;
  FSid: TSocket; 	     	// socket descriptor
  FRc: Integer;	 	    	// member function return status code
public
// Class constructors
  constructor Create;
  constructor CreateByHandle( NewSid:Integer );
// Class destructor
  destructor Destroy; override;

  function IsGood: Boolean; 	// check sock object status
  function QueueSize: LongInt;
  procedure SetOptions(Cmd: Longint; var Arg: u_long);
  procedure SetEvents(Handle: HWND; Msg: u_int; Event: Longint);

  procedure CloseConnect;

  property Handle: TSocket read FSid;
  property Server: TURL read FServer;
  property Client: TURL read FClient;
end;

{************* TInetClientSocket definition **************}

// TCP-IP client socket class
TInetClientSocket=class(TInetSocket)
public
// Class constructors
  constructor Create;
  constructor CreateByName(const Name: String; Port: Integer);
  constructor CreateByIP(IP: u_long; Port: Integer);
  constructor CreateByURL(NewURL: TURL);
  constructor CreateByHandle(NewSid: Integer);

  function IsArrive: Boolean;
  function ConnectSocket(const HostNm: String; Port: Integer): Integer;
  function ConnectSocketByURL(Url: TURL): Integer;

  function Write(var Buf; Len, Flag: Integer): Integer;
  function Read( var Buf; Len, Flag: Integer): Integer;
end;

{**************** TInetServerSocket definition **************}

// TCP-IP server socket class
TInetServerSocket=class(TInetSocket)
private
  function BindSocket: Integer;
public
// Class constructors
  constructor Create;
  constructor CreateByName(const Name: String; Port: Integer);
  constructor CreateByIP(IP: u_long; Port: Integer);
  constructor CreateByURL(NewURL: TURL);
  constructor CreateByHandle(NewSid: Integer);

  function ListenConnect: Integer;
  function AcceptConnect: TInetClientSocket;
  function ShutdownConnect(Mode: Integer): Integer;
end;

// Initialize Winsock library
function WinSocketStartup: Boolean;
// Deinitialize Winsock library
procedure WinSocketCleanup;
// Define host name by IP-address
function IP2Str(IP: u_long): String;
// Define IP-address by host name
function Str2IP(Buff: String): u_long;
// Define host name of the local comp
function GetLocalHost: string;
// Process Winsock errors
procedure WinSocketCheckError;

implementation

{*********************    ************************}

// Initialize Winsock library
function WinSocketStartup: Boolean;
var
  wVersionRequested: Word;
  wsaData: TWSADATA;
begin
  wVersionRequested := $0101;
  Result := WSAStartup(wVersionRequested, wsaData) = 0;

  if not Result then
{$IFDEF RUSSIAN}
    raise EWinsockError.Create('  Winsock');
{$ELSE}
    raise EWinsockError.Create('Unable to load Winsock');
{$ENDIF}
end;

// Deinialize Winsock labrary
procedure WinSocketCleanup;
begin
  WSACleanup;
end;

// Process Winsock errors
procedure WinSocketCheckError;
var ErrorCode: Integer;
begin
  ErrorCode := WSAGetLastError;
  if (ErrorCode <> 0) and (ErrorCode <> WSAEWOULDBLOCK) then
    raise EWinsockError.CreateFmt('Winsocket error: %s No %d.',
      [SysErrorMessage(ErrorCode), ErrorCode]);
end;

// Get localhost name
function GetLocalHost: string;
var PHostName: PChar;
begin
  GetMem(PHostName, 100);
  try
    if GetHostName(PHostName, 100) <> -1 then Result := PHostName
    else Result := '';
  finally
    FreeMem(PHostName)
  end;
end;

// Invert 4 bytes number
function RevertInt(Value: u_long): u_long;
begin
  Result:= ((Value shr 24) or ((Value and $ff0000)shr 8) or
           ((Value and $ff00)shl 8) or ((Value and $ff)shl 24) );
end;

// IP-Address to string
function IP2Str(IP: u_long): String;
begin
  Result:=Format('%d.%d.%d.%d', [(IP shr 24)and $ff, (IP shr 16)and $ff ,
          (IP shr 8)and $ff, IP and $ff]);
end;

// String to IP-address
function Str2IP(Buff: String): u_long;
var addr:array[0..3] of u_long;
begin
//    sscanf(buff,"%d.%d.%d.%d",&addr[0],&addr[1],&addr[2],&addr[3]);
    Result := ((addr[0] shl 24)or(addr[1] shl 16)or(addr[2] shl 8)or addr[3]);
end;

{*****************  TURL implementation *******************}

// Class constructor
constructor TURL.Create;
begin
  FHostName := '';
  FFileName := '/';
  FIP := 0;
  FPort := 0;
end;

constructor TURL.CreateByName(const NewHost: String; NewPort: Integer;
  NewFile: String);
begin
  if NewFile<>'' then
    FillUrlByName(NewHost, newPort, newFile)
  else
    FillUrlByName(NewHost, NewPort, '/');
end;

constructor TURL.CreateByIP(NewIP: u_long; NewPort: Integer; NewFile: String);
begin
  if NewFile<>'' then
    FillUrlByIP(NewIP, NewPort, NewFile)
  else
    FillUrlByIP(NewIP, NewPort, '/');
end;

function TURL.GetIP: u_long;
begin
  Result:= RevertInt(FIP);
end;

// Convert IP-address to host name
function TURL.IP2Name(IP: u_long): String;
var
  inaddr: TInAddr;
  laddr: u_long;
  hp: PHostEnt;
  p: ^PChar;
begin
  inaddr.S_addr := IP;
  Result := '';
  laddr := inet_addr(inet_ntoa(inaddr));
  if (laddr = -1) then Exit;
  hp := gethostbyaddr(@laddr, SizeOf (laddr), AF_INET);
  WinSocketCheckError;
  if not Assigned(hp) then Exit;
  p := @hp^.h_addr_list;
  while Assigned(p^) do begin
    Move(p^, inaddr.s_addr, SizeOf (inaddr.s_addr));
    if Assigned(hp^.h_name) then begin
      Result := StrPas(hp^.h_name);
      break;
    end;
    Inc(p);
  end;
end;

function TURL.Name2IP(const HostNm: String ): u_long;
var
  hp:PHostEnt;
  buff:array[0..255] of Char;
begin
  Result := 0;
  hp := gethostbyname(StrPCopy(buff,HostNm));
  if Assigned(hp) then
    Move(hp^.h_addr^^,Result,hp^.h_length);
  WinSocketCheckError;
end;

procedure TURL.FillUrlByName(const NewHost: String; NewPort: Integer;
  const NewFile: String);
begin
  FHostName := NewHost;
  FIP := Name2IP(NewHost);
  FPort := NewPort;
  FFileName := NewFile;
end;

procedure TURL.FillUrlByIp(NewIP: u_long; NewPort: Integer; const NewFile: String);
begin
  FIP := RevertInt(NewIP);
  FHostName := IP2Name(FIP);
  FPort := NewPort;
  FFileName := NewFile;
end;

function TURL.FillAddr(var addr: TSockAddrIn): Integer;
begin
  addr.sin_family := AF_INET;
  addr.sin_addr.s_addr := FIP;
  addr.sin_port := htons(FPort);
  Result := SizeOf(addr);
end;

function TURL.GetHostName: String;
begin
  if FHostName='' then
    FHostName := IP2Str(RevertInt(FIP));
  Result := FHostName;
end;

procedure TURL.SetHostName(const NewHost: String);
begin
  FHostName := NewHost;
  FIP := Name2IP(FHostName);
end;

procedure TURL.SetIP(NewIP: u_long);
begin
  FIP := RevertInt(NewIP);
  FHostName := '';
  if FIP=$100007F then
    FHostName := 'localhost'
  else
    try
      FHostName := IP2Name(FIP);
    except
      FHostName := IP2Str(FIP);
    end;
end;

function TURL.GetURL: String;
begin
  FPath:=HostName+':'+IntToStr(FPort)+FileName;
  Result:=FPath;
end;

procedure TURL.Assign(NewURL: TURL);
begin
  FPort := NewURL.FPort;
  FHostName := NewURL.FHostName;
  FFileName := NewURL.FFileName;
  FIP := NewURL.FIP;
end;

{******************* TInetSocket implementation ***********************}

// Check socket status
function TInetSocket.IsGood: Boolean;
begin
  Result := (FSid >= 0);
end;

function TInetSocket.QueueSize: LongInt;
begin
  Result:=0;
  ioctlsocket(FSid, FIONREAD, Result);
end;

procedure TInetSocket.SetEvents(Handle: HWND; Msg: u_int; Event: LongInt);
begin
  WSAAsyncSelect(FSid, Handle, Msg, Event);
  WinSocketCheckError;
end;

procedure TInetSocket.SetOptions(Cmd: LongInt; var Arg: u_long);
begin
  ioctlsocket(FSid, cmd, arg);
end;

constructor TInetSocket.Create;
begin
  FServer := TURL.Create;
  FClient := TURL.Create;
  FSid := 0;
end;

constructor TInetSocket.CreateByHandle(NewSid: Integer);
begin
  FServer := TURL.CreateByName('localhost', 0, '/');
  FClient := TURL.CreateByIP( 0, 0, '/');
  FSid := NewSid;
end;

destructor TInetSocket.Destroy;
begin
  try
    CloseConnect;
  except
  end;
  FServer.Free;
  FClient.Free;
end;

procedure TInetSocket.CloseConnect;
begin
  shutdown (FSid,2);
//  WinSocketCheckError;
  closesocket(FSid);
//  WinSocketCheckError;
end;

{************** TInetClientSocket implementation ***************}

function TInetClientSocket.IsArrive: Boolean;
begin
  Result := (QueueSize > 0);
end;

function TInetClientSocket.Write(var Buf; Len,Flag: Integer): Integer;
begin
  Result := Send(FSid, Buf, Len, Flag );
  if Result = SOCKET_ERROR then WinSocketCheckError;
end;

function TInetClientSocket.Read(var Buf; Len,Flag: Integer ): Integer;
begin
  Result := Recv(FSid, Buf, Len, Flag);
  if Result = SOCKET_ERROR then WinSocketCheckError;
end;

constructor TInetClientSocket.Create;
begin
  FServer := TURL.Create;
  FClient := TURL.Create;
  FSid := 0;
end;

constructor TInetClientSocket.CreateByName(const Name: String; Port: Integer);
begin
  FServer := TURL.CreateByName(Name, Port, '/');
  FClient := TURL.Create;
  FSid := Socket(PF_INET, SOCK_STREAM, IPPROTO_IP);
  if FSid = INVALID_SOCKET then WinSocketCheckError;
end;

constructor TInetClientSocket.CreateByIP(IP: u_long; Port: Integer);
begin
  FServer := TURL.CreateByIP(IP, Port, '/');
  FClient := TURL.Create;
  FSid := Socket(PF_INET, SOCK_STREAM, IPPROTO_IP);
  if FSid = INVALID_SOCKET then WinSocketCheckError;
end;

constructor TInetClientSocket.CreateByUrl(NewURL: TURL);
begin
  FServer := TURL.Create;
  FServer.Assign(newURL);
  FClient := TURL.Create;
  FSid := Socket(PF_INET, SOCK_STREAM, IPPROTO_IP);
  if FSid = INVALID_SOCKET then WinSocketCheckError;
end;

constructor TInetClientSocket.CreateByHandle(NewSid: Integer);
begin
  FServer := TURL.CreateByName('localhost', 0, '/');
  FClient := TURL.CreateByIP( 0, 0, '/');
  FSid := NewSid;
end;

function TInetClientSocket.ConnectSocket(const HostNm: String; Port: Integer): Integer;
var url: TURL;
begin
  url := TURL.CreateByName(HostNm, Port, '/');
  Result := ConnectSocketByURL(url);
  url.Free;
end;

function TInetClientSocket.ConnectSocketByURL(Url: TURL): Integer;
var
  Addr: TSockAddrIn;
  Len: Integer;
begin
  Len := url.FillAddr(Addr);
  FRc := Connect(FSid, Addr, Len);
  if FRc=SOCKET_ERROR then WinSocketCheckError;
  Result := FRc;
end;

{**************** TInetServerSocket implementation ***************}

function TInetServerSocket.BindSocket: Integer;
var
  addr: TSockAddrIn;
  len: u_long;
begin
  FSid := socket(AF_INET, SOCK_STREAM, 0);
  WinSocketCheckError;
  len := FServer.FillAddr( addr );
  FRc := bind(FSid, addr, len);
  WinSocketCheckError;
  if FRc<0 then begin
    FRc := getsockname(FSid, addr, len);
    if FRc<0 then begin
      CloseConnect;
    end;
    FSid := -1;
  end;
  Result := FRc;
end;

constructor TInetServerSocket.Create;
begin
  FServer := TURL.Create;
  FClient := TURL.Create;
end;

constructor TInetServerSocket.CreateByName(const Name: String; Port: Integer);
begin
  FServer := TURL.CreateByName(Name, Port, '/');
  FClient := TURL.Create;
  BindSocket;
end;

constructor  TInetServerSocket.CreateByIP(IP: u_long; Port: Integer);
begin
  FServer := TURL.CreateByIP(IP, Port, '/');
  FClient := TURL.Create;
  BindSocket;
end;

constructor TInetServerSocket.CreateByURL(NewURL: TURL);
begin
  FServer := TURL.Create;
  FServer.Assign(NewURL);
  FClient := TURL.Create;
  BindSocket;
end;

constructor TInetServerSocket.CreateByHandle(NewSid: Integer);
begin
  FServer := TURL.CreateByName('localhost', 0, '/');
  FClient := TURL.Create;
  FSid := NewSid;
end;

function TInetServerSocket.ListenConnect: Integer;
begin
  Result := listen(FSid,BACKLOG_NUM);
  WinSocketCheckError;
end;

function TInetServerSocket.AcceptConnect: TInetClientSocket;
var
  sock: TInetClientSocket;
  addr: TSockAddrIn;
  len: u_long;
begin
  len := SizeOf( addr );
  FRc := accept(FSid, @addr, @len);
//  WinSocketCheckError;
  if( FRc >0 ) then begin
    FClient.IP := RevertInt(addr.sin_addr.s_addr);
    FClient.Port := addr.sin_port;
    sock := TInetClientSocket.CreateByHandle(FRc);
    sock.Server.Assign(FServer);
    sock.Client.Assign(FClient);
    Result := sock;
  end else
    Result := NIL;
end;

function TInetServerSocket.ShutdownConnect(Mode: Integer): Integer;
begin
  Result := shutdown(FSid,Mode);
  WinSocketCheckError;
end;

initialization
  WinSocketStartup;
finalization
  WinSocketCleanup;
end.
