{******************************************************************
*  (c)copyrights Corona Ltd. Donetsk 1999
*  Project: Zeos Library
*  Module: Formula parser component
*  Author: Sergey Seroukhov   E-Mail: voland@cm.dongu.donetsk.ua
*  Date: 26/03/99
*
*  List of changes:
*  27/03/99 - Class convert to component, add vars
*  16/04/99 - Add some functions, operators LIKE, XOR
******************************************************************}

unit ZParser;

{$R *.DCR}

interface
uses SysUtils, Classes, ZToken, ZMatch;

{$I ..\Zeos.inc}

const
  MAX_PARSE_ITEMS = 100;
  MAX_PARSE_STACK = 100;
  MAX_PARSE_VARS = 20;
  MAX_PARSE_FUNCS = 20;

type

TParseItemType=(ptFunction, ptVariable, ptDelim, ptString, ptInteger, ptFloat,
  ptBoolean);

TParseItem = record
  ItemValue: Variant;
  ItemType: TParseItemType;
end;

TParseStack = array[0..MAX_PARSE_STACK] of Variant;

TParseVar = record
  VarName: String;
  VarValue: Variant;
end;

TParser = class;

TParseFunc = function(Sender: TParser): Variant;

TParseFuncRec = record
  FuncName: String;
  FuncPtr: TParseFunc;
end;

EParseException = class(Exception);

{*************** TParser implementation *************}

TParser = class(TComponent)
private
  FParseItems: array[0..MAX_PARSE_ITEMS] of TParseItem;
  FParseCount: Integer;
  FErrCheck: Integer;
  FEquation: String;
  FParseStack: TParseStack;
  FStackCount: Integer;
  FVars: array[0..MAX_PARSE_VARS] of TParseVar;
  FVarCount: Integer;
  FFuncs: array[0..MAX_PARSE_FUNCS] of TParseFuncRec;
  FFuncCount: Integer;

// Extract highlevel lexems
  function ExtractTokenEx(var Buffer, Token: String): TParseItemType;
// Define priority level of equation
  function OpLevel(Operat: String): Integer;
// Internal function to convert infix equation to postfix
  function Parse(Level: Integer; var Buffer: String): Integer;
// Split equation to stack for calculate
  procedure SetEquation(Value: String);
// Get variable value
  function GetVar(VarName: String): Variant;
// Set new variable value
  procedure SetVar(VarName: String; VarValue: Variant);
// Get varname by it index
  function GetVarName(VarIndex: Integer): String;
// Get function handle
  function GetFunc(FuncName: String): TParseFunc;
// Set new function handle
  procedure SetFunc(FuncName: String; FuncPtr: TParseFunc);
// Define function name by it handle
  function GetFuncName(FuncIndex: Integer): String;
// Convert types of two values
  procedure CheckTypes(Value1: Variant; var Value2: Variant);
// Convert variant type value
  function ConvType(Value: Variant): Variant;
// Check function
  function CheckFunc(var Buffer: String): Boolean;
public
// Class constructor
  constructor Create(AOwner: TComponent); override;
// Class destructor
  destructor Destroy; override;
// Calculate an equation
  function Evalute: Variant;
// Clear variables and equation
  procedure Clear;
// Push value to stack
  procedure Push(Value: Variant);
// Pop value from stack
  function Pop: Variant;

// Variables
  property Variables[Index: String]: Variant read GetVar write SetVar;
// Quantity of varibles
  property VarCount: Integer read FVarCount;
// Variable names
  property VarNames[Index: Integer]: String read GetVarName;
// Functions
  property Functions[Index: String]: TParseFunc read GetFunc write SetFunc;
// Quantity of functions
  property FuncCount: Integer read FFuncCount;
// Function names
  property FuncNames[Index: Integer]: String read GetFuncName;
published
// Equantion
  property Equation: String read FEquation write SetEquation;
end;

//  
procedure Register;

implementation

uses ZExtra;

{************** User functions implementation *************}

// Get current date and time
function FuncNow(Sender: TParser): Variant; forward;

// Define minimal value
function FuncMin(Sender: TParser): Variant; forward;

// Define maximum value
function FuncMax(Sender: TParser): Variant; forward;

// Define result by value
function FuncIIf(Sender: TParser): Variant; forward;

// Calculate sum of values
function FuncSum(Sender: TParser): Variant; forward;

{******************* TParser implementation ****************}

// Class constructor
constructor TParser.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FErrCheck := 0;
  FStackCount := 0;
  FVarCount := 0;
  FFuncCount := 0;
  SetFunc('NOW',FuncNow);
  SetFunc('MAX',FuncMax);
  SetFunc('MIN',FuncMin);
  SetFunc('IIF',FuncIIf);
  SetFunc('SUM',FuncSum);
end;

// Class destructor
destructor TParser.Destroy;
begin
  inherited;
end;

// Extract highlevel lexem
function TParser.ExtractTokenEx(var Buffer, Token: String): TParseItemType;
var
  P: Integer;
  Temp: String;
  TokenType: TTokenType;
begin
  repeat
    TokenType := ExtractToken(Buffer, Token);
  until (Token<>#10)and(Token<>#13);
  if Token='[' then begin
    TokenType := ttAlpha;
    P := Pos(']',Buffer); Token := '';
    if P>0 then begin
      Token := Copy(Buffer,1,P-1);
      Buffer := Copy(Buffer,P+1,Length(Buffer)-P);
    end;
  end;
  if (Buffer<>'')and(Token='>')and(Buffer[1]='=') then begin
    ExtractToken(Buffer, Temp);
    Token := Token + Temp;
  end;
  if (Buffer<>'')and(Token='<')and((Buffer[1]='=')or(Buffer[1]='>')) then begin
    ExtractToken(Buffer, Temp);
    Token := Token + Temp;
  end;
  Temp := UpperCase(Token);
  if (Temp='AND') or (Temp='NOT') or (Temp='OR') or (Temp='XOR') or
    (Temp='LIKE') then begin
    Token := Temp;
    Result := ptDelim;
    exit;
  end;
  if (Temp='TRUE') or (Temp='FALSE') then begin
    Token := Temp;
    Result := ptBoolean;
    exit;
  end;

  Result := ptString;
  case TokenType of
    ttAlpha: Result := ptVariable;
    ttDelim: Result := ptDelim;
    ttDigit: begin
        if (Buffer<>'') and (Buffer[1]='.') then begin
          ExtractToken(Buffer, Temp);
          Token := Token + '.';
          if (Buffer<>'')and(Buffer[1]>='0')and(Buffer[1]<='9') then begin
            ExtractToken(Buffer,Temp);
            Token := Token + Temp;
          end;
          Result := ptFloat;
        end else Result := ptInteger;
      end;
  end;
end;

// Get priority level of operation
function TParser.OpLevel(Operat: String): Integer;
var Temp: String;
begin
  Result := 7;
  Temp := UpperCase(Operat);
  if (Temp='AND') or (Temp='OR') or (Temp='XOR') then Result := 1;
  if (Temp='NOT') then Result := 2;
  if (Temp='<')or(Temp='>')or(Temp='=')or(Temp='>=')or(Temp='<=')
    or(Temp='<>') then Result := 3;
  if (Temp[1]='+') or (Temp[1]='-') or (Temp='LIKE') then Result := 4;
  if (Temp[1]='/') or (Temp[1]='*') or (Temp[1]='%') then Result := 5;
  if (Temp[1]='^') then Result := 6;
end;

// Internal convert equation from infix form to postfix
function TParser.Parse(Level: Integer; var Buffer: String): Integer;
var
  ParseType: TParseItemType;
  Token, FuncName: String;
  Temp: Char;
  NewLevel, Params, SaveCount: Integer;
begin
  Result := 0;
  while Buffer<>'' do begin
    ParseType := ExtractTokenEx(Buffer, Token);
    if Token='' then exit;
    if (Token=')') or (Token=',') then begin
      PutbackToken(Buffer, Token);
      exit;
    end;
    if Token='(' then begin
      FErrCheck := 0;
      Parse(0,Buffer);
      ExtractTokenEx(Buffer, Token);
      if Token<>')' then
{$IFDEF RUSSIAN}
        raise EParseException.Create(' ');
{$ELSE}
        raise EParseException.Create('Syntax error');
{$ENDIF}
      FErrCheck := 1;
      continue;
    end;

    if ParseType=ptDelim then begin
      NewLevel := OpLevel(Token);

      if (FErrCheck=2)and(Token<>'NOT') then
{$IFDEF RUSSIAN}
        raise EParseException.Create(' ');
{$ELSE}
        raise EParseException.Create('Syntax error');
{$ENDIF}
      if FErrCheck=0 then
        if (Token<>'NOT')and(Token<>'+')and(Token<>'-') then
{$IFDEF RUSSIAN}
          raise EParseException.Create(' ')
{$ELSE}
          raise EParseException.Create('Syntax error')
{$ENDIF}
        else if Token<>'NOT' then NewLevel := 6;

      if (Token<>'NOT')and(NewLevel<=Level) then begin
        PutbackToken(Buffer, Token);
        Result := NewLevel;
        exit;
      end else if (Token='NOT')and(NewLevel<Level) then begin
        PutbackToken(Buffer, Token);
        Result := NewLevel;
        exit;
      end;

      if (FErrCheck=0) and (Token='+') then continue;
      if (FErrCheck=0) and (Token='-') then Token := '~';
      FErrCheck := 2;

      while (Buffer<>'')and(Buffer[1]<>')')and(Parse(NewLevel, Buffer)>NewLevel) do;
      FParseItems[FParseCount].ItemValue := Token;
      FParseItems[FParseCount].ItemType := ptDelim;
      Inc(FParseCount);
      Result := NewLevel;
      continue;
    end;

    if FErrCheck=1 then
{$IFDEF RUSSIAN}
      raise EParseException.Create(' ');
{$ELSE}
      raise EParseException.Create('Syntax error');
{$ENDIF}
    FErrCheck := 1;

    case ParseType of
      ptVariable: begin
          FParseItems[FParseCount].ItemValue := Token;
          if CheckFunc(Buffer) then ParseType := ptFunction
          else SetVar(Token,NULL);
        end;
      ptInteger: FParseItems[FParseCount].ItemValue := StrToInt(Token);
      ptFloat: begin
          Temp := DecimalSeparator;
          DecimalSeparator := '.';
          FParseItems[FParseCount].ItemValue := StrToFloat(Token);
          DecimalSeparator := Temp;
        end;
      ptString: begin
          DeleteQuotes(Token);
          FParseItems[FParseCount].ItemValue := Token;
        end;
      ptBoolean:
        if Token='TRUE' then FParseItems[FParseCount].ItemValue := true
        else FParseItems[FParseCount].ItemValue := false;
    end;

// Process function params
    if ParseType=ptFunction then begin
      FuncName := UpperCase(Token);
      SaveCount := FParseCount;
      Params := 0;
      repeat
        FErrCheck := 0;
        Parse(0,Buffer);
        ExtractTokenEx(Buffer, Token);
        case Token[1] of
          ',': begin
              Inc(Params);
              continue;
            end;
          ')': begin
              if SaveCount<FParseCount then Inc(Params);
              FParseItems[FParseCount].ItemValue := ConvType(Params);
              FParseItems[FParseCount].ItemType := ptInteger;
              Inc(FParseCount);
              break;
            end;
          else
{$IFDEF RUSSIAN}
            raise EParseException.Create(' ');
{$ELSE}
            raise EParseException.Create('Syntax error');
{$ENDIF}
        end;
      until Buffer='';
      FParseItems[FParseCount].ItemValue := FuncName;
    end;

    FParseItems[FParseCount].ItemValue :=
      ConvType(FParseItems[FParseCount].ItemValue);
    FParseItems[FParseCount].ItemType := ParseType;
    Inc(FParseCount);

  end;
end;

// Split equation to stack
// Value - equation buffer
procedure TParser.SetEquation(Value: String);
begin
  FParseCount := 0; FErrCheck := 0;
  FEquation := Value;
  while Value<>'' do Parse(0, Value);
end;

// Get variable name by it index
function TParser.GetVarName(VarIndex: Integer): String;
begin
  if VarIndex>=FVarCount then
{$IFDEF RUSSIAN}
    raise EParseException.Create('  ');
{$ELSE}
    raise EParseException.Create('Incorrect variable index');
{$ENDIF}
  Result := FVars[VarIndex].VarName;
end;

// Get variable value
function TParser.GetVar(VarName: String): Variant;
var I: Integer;
begin
  I := 0;
  while I<FVarCount do begin
    if FVars[I].VarName=VarName then begin
      Result := FVars[I].VarValue;
      exit;
    end;
    Inc(I);
  end;
  Result := NULL;
end;

// Set new value to variable
procedure TParser.SetVar(VarName: String; VarValue: Variant);
var I: Integer;
begin
  I := 0;
  while I<FVarCount do begin
    if FVars[I].VarName=VarName then begin
      if VarType(VarValue)<>varNull then
        FVars[I].VarValue := ConvType(VarValue);
      exit;
    end;
    Inc(I);
  end;
  if I>=MAX_PARSE_VARS then exit;
  FVars[I].VarName := VarName;
  FVars[I].VarValue := ConvType(VarValue);
  Inc(FVarCount);
end;

// Get function name by it handle
function TParser.GetFuncName(FuncIndex: Integer): String;
begin
  if FuncIndex>=FFuncCount then
{$IFDEF RUSSIAN}
    raise EParseException.Create('  ');
{$ELSE}
    raise EParseException.Create('Incorrect function index');
{$ENDIF}
  Result := FFuncs[FuncIndex].FuncName;
end;

// Get function handle
function TParser.GetFunc(FuncName: String): TParseFunc;
var I: Integer;
begin
  I := 0;
  FuncName := UpperCase(FuncName);
  while I<FFuncCount do begin
    if UpperCase(FFuncs[I].FuncName)=FuncName then begin
      Result := FFuncs[I].FuncPtr;
      exit;
    end;
    Inc(I);
  end;
  Result := NIL;
end;

// Set new function handle
procedure TParser.SetFunc(FuncName: String; FuncPtr: TParseFunc);
var I: Integer;
begin
  I := 0;
  while I<FFuncCount do begin
    if FFuncs[I].FuncName=FuncName then begin
      if Assigned(FuncPtr) then
        FFuncs[I].FuncPtr := FuncPtr;
      exit;
    end;
    Inc(I);
  end;
  if I>=MAX_PARSE_FUNCS then exit;
  FFuncs[I].FuncName := FuncName;
  FFuncs[I].FuncPtr := FuncPtr;
  Inc(FFuncCount);
end;

// Convert types of two variant values
function TParser.ConvType(Value: Variant): Variant;
begin
  case VarType(Value) of
    varByte, varSmallint, varInteger:
      Result := VarAsType(Value, varInteger);
    varSingle, varDouble, varCurrency:
      Result := VarAsType(Value, varDouble);
    varDate, varOleStr, varString, varVariant:
      Result := VarAsType(Value, varString);
    varBoolean:
      Result := Value;
    varNull:
      Result := VarAsType(0, varInteger);
//    varDispatch, varError, varUnknown, varTypeMask, varArray, varByRef, varEmpty,
//    varNull:
    else
{$IFDEF RUSSIAN}
      raise EParseException.Create(' ');
{$ELSE}
      raise EParseException.Create('Types mismatch');
{$ENDIF}
  end;
end;

// Convert types of two variant values
procedure TParser.CheckTypes(Value1: Variant; var Value2: Variant);
begin
  case VarType(Value1) of
    varInteger:
      if VarType(Value2)=varString then Value2 := StrToFloatEx(Value2)
      else Value2 := VarAsType(Value2, varInteger);
    varString: Value2 := VarAsType(Value2, varString);
    varDouble:
      if VarType(Value2)=varString then Value2 := StrToFloatEx(Value2)
      else Value2 := VarAsType(Value2, varDouble);
    varBoolean:
      case VarType(Value2) of
        varInteger, varDouble:
          Value2 := Value2<>0;
        varString:
          Value2 := StrToFloatEx(Value2)<>0;
        varBoolean:
        else
{$IFDEF RUSSIAN}
          raise EParseException.Create(' ');
{$ELSE}
          raise EParseException.Create('Types mismatch');
{$ENDIF}
      end;
    else
{$IFDEF RUSSIAN}
      raise EParseException.Create(' ');
{$ELSE}
      raise EParseException.Create('Types mismatch');
{$ENDIF}
  end;
end;

// Calculate an equation
function TParser.Evalute: Variant;
var
  I: Integer;
  Value1, Value2: Variant;
  Op: String;
  FuncPtr: TParseFunc;
begin
  FStackCount := 0;
  for I := 0 to FParseCount-1 do begin
    case FParseItems[I].ItemType of
      ptFunction: begin
          FuncPtr := GetFunc(FParseItems[I].ItemValue);
          if Assigned(FuncPtr) then Push(FuncPtr(Self))
          else
{$IFDEF RUSSIAN}
            raise EParseException.Create('   "'+FParseItems[I].ItemValue+'"');
{$ELSE}
            raise EParseException.Create('Function "'+FParseItems[I].ItemValue+'" not found');
{$ENDIF}
        end;
      ptVariable: begin
          Value1 := GetVar(FParseItems[I].ItemValue);
          if VarType(Value1)=varNull then
{$IFDEF RUSSIAN}
            raise EParseException.Create('   "'+FParseItems[I].ItemValue+'"');
{$ELSE}
            raise EParseException.Create('Variable "'+FParseItems[I].ItemValue+'" not found');
{$ENDIF}
          Push(Value1);
        end;
      ptFloat, ptInteger, ptString, ptBoolean: Push(FParseItems[I].ItemValue);
      ptDelim: begin
          Op := VarAsType(FParseItems[I].ItemValue, varString);

          if Op[1] in ['+','-','*','/','%'] then begin
            Value2 := Pop; Value1 := Pop;
            CheckTypes(Value1, Value2);
            case Op[1] of
              '+': Push(Value1 + Value2);
              '-': Push(Value1 - Value2);
              '*': Push(Value1 * Value2);
              '/': Push(Value1 / Value2);
              '%': Push(Value1 mod Value2);
            end;
            continue;
          end;

          if (Op='=')or(Op='<')or(Op='>') then begin
            Value2 := Pop; Value1 := Pop;
            CheckTypes(Value1, Value2);
            case Op[1] of
              '=': Push(Value1 = Value2);
              '<': Push(Value1 < Value2);
              '>': Push(Value1 > Value2);
            end;
            continue;
          end;

          if (Op='>=')or(Op='<=')or(Op='<>') then begin
            Value2 := Pop; Value1 := Pop;
            CheckTypes(Value1, Value2);
            if Op='>=' then Push(Value1 >= Value2);
            if Op='<=' then Push(Value1 <= Value2);
            if Op='<>' then Push(Value1 <> Value2);
            continue;
          end;

          if (Op='AND')or(Op='OR')or(Op='XOR') then begin
//            Value2 := VarAsType(Pop,varBoolean);
//            Value1 := VarAsType(Pop,varBoolean);
            Value1 := Pop; Value2 := Pop;
            if Op='AND' then Push(Value1 and Value2);
            if Op='OR' then Push(Value1 or Value2);
            if Op='XOR' then
              Push((not Value1 and Value2) or (Value1 and not Value2));
            continue;
          end;

          if Op='~' then begin
            Value1 := Pop;
//            CheckTypes(VarAsType(0.1,varDouble), Value1);
            Push(-Value1);
            continue;
          end;

          if Op='NOT' then begin
            Value1 := Pop;
            CheckTypes(true, Value1);
            Push(not Value1);
            continue;
          end;

          if (Op='^') then begin
            Value2 := VarAsType(Pop,varDouble);
            Value1 := VarAsType(Pop,varDouble);
            Push(exp(Value2*ln(Value1)));
            continue;
          end;

          if (Op='LIKE') then begin
            Value2 := VarAsType(Pop, varString);
            Value1 := VarAsType(Pop, varString);
            Push(IsMatch(Value2,Value1));
            continue;
          end;

{$IFDEF RUSSIAN}
          raise EParseException.Create(' ');
{$ELSE}
          raise EParseException.Create('Incorrect operation');
{$ENDIF}
        end;
    end;
  end;
  Result := Pop;
  if FStackCount>0 then
{$IFDEF RUSSIAN}
    raise EParseException.Create(' ');
{$ELSE}
    raise EParseException.Create('Evalution error');
{$ENDIF}
end;

// Push value to stack
procedure TParser.Push(Value: Variant);
begin
  if FStackCount>=MAX_PARSE_STACK then
{$IFDEF RUSSIAN}
    raise EParseException.Create(' ');
{$ELSE}
    raise EParseException.Create('Stack is full');
{$ENDIF}
  FParseStack[FStackCount] := Value;
  Inc(FStackCount);
end;

// Pop value from stack
function TParser.Pop: Variant;
begin
  if FStackCount=0 then
{$IFDEF RUSSIAN}
    raise EParseException.Create(' ');
{$ELSE}
    raise EParseException.Create('Stack is empty');
{$ENDIF}
  Dec(FStackCount);
  Result := FParseStack[FStackCount];
end;

// Clear all variables and equation
procedure TParser.Clear;
begin
  FStackCount := 0;
  FParseCount := 0;
  FVarCount := 0;
  FEquation := '';
end;

// Define function
function TParser.CheckFunc(var Buffer: String): Boolean;
var
  I: Integer;
  Token: String;
begin
  I := 1;
  while (I<=Length(Buffer)) and (Buffer[I] in [' ',#9,#10,#13]) do
    Inc(I);
  if (Buffer[I]='(') then begin
    Result := true;
    ExtractToken(Buffer, Token);
  end else Result := false;
end;

{****************************************************}

// Register component in component palette
procedure Register;
begin
  RegisterComponents(ZEOS_PALETTE, [TParser]);
end;

{************** User functions implementation **************}

// Get current date and time
function FuncNow(Sender: TParser): Variant;
begin
  Result := Sender.Pop;
  if Result<>0 then
{$IFDEF RUSSIAN}
    EParseException.Create('   NOW()');
{$ELSE}
    EParseException.Create('Incorrect params in function NOW()');
{$ENDIF}
  Result := FormatDateTime('yyyy-mm-dd hh:nn:ss', Now());
end;

// Get maximum value
function FuncMax(Sender: TParser): Variant;
var
  Count: Integer;
  Temp: Variant;
begin
  Count := Sender.Pop;
  if Count=0 then
{$IFDEF RUSSIAN}
    EParseException.Create('   MAX()');
{$ELSE}
    EParseException.Create('Incorrect params in function MAX()');
{$ENDIF}
  Result := Sender.Pop;
  Dec(Count);
  while Count>0 do begin
    Temp := Sender.Pop;
    if Temp>Result then Result := Temp;
    Dec(Count);
  end;
end;

// Get minimum value
function FuncMin(Sender: TParser): Variant;
var
  Count: Integer;
  Temp: Variant;
begin
  Count := Sender.Pop;
  if Count=0 then
{$IFDEF RUSSIAN}
    EParseException.Create('   MIN()');
{$ELSE}
    EParseException.Create('Incorrect params in function MIN()');
{$ENDIF}
  Result := Sender.Pop;
  Dec(Count);
  while Count>0 do begin
    Temp := Sender.Pop;
    if Temp<Result then Result := Temp;
    Dec(Count);
  end;
end;

// Calculate sum of values
function FuncSum(Sender: TParser): Variant;
var Count: Integer;
begin
  Count := Sender.Pop;
  if Count=0 then
{$IFDEF RUSSIAN}
    EParseException.Create('   SUM()');
{$ELSE}
    EParseException.Create('Incorrect params in function SUM()');
{$ENDIF}
  Result := Sender.Pop;
  Dec(Count);
  while Count>0 do begin
    Result := Result + Sender.Pop;
    Dec(Count);
  end;
end;

// Get result by value
function FuncIIf(Sender: TParser): Variant;
var
  Count: Integer;
  Temp, Temp1, Temp2: Variant;
begin
  Count := Sender.Pop;
  if Count<>3 then
{$IFDEF RUSSIAN}
    EParseException.Create('   IIF()');
{$ELSE}
    EParseException.Create('Incorrect params in function IIF()');
{$ENDIF}
  Temp2 := Sender.Pop;
  Temp1 := Sender.Pop;
  Temp := VarAsType(Sender.Pop, varBoolean);
  if Temp then Result := Temp1
  else Result := Temp2;
end;

end.
