{******************************************************************
*  (c)copyrights Corona Ltd. Donetsk 1999
*  Project: Zeos Library
*  Module: Incapsulate classes for Macro-Capella API
*  Author: Sergey Seroukhov   E-Mail: voland@cm.dongu.donetsk.ua
*  Date: 26/07/99
*
*  List of changes:
******************************************************************}

unit ZMacro;

interface
uses Classes, SysUtils, MacroLib;

{$I ..\Zeos.inc}

type

// Macro-Capella exception type
EMacroError = Exception;

// Internal Macro-Capella Variant Class
TVarValue = class
private
  FHandle: PChar;

// Internal methods
  function GetAsPointer: Pointer;
  function GetAsInteger: LongInt;
  function GetAsString: String;
  function GetAsFloat: Double;
  function GetAsBoolean: Boolean;
  function GetVarType: Integer;
  procedure SetAsPointer(Value: Pointer);
  procedure SetAsInteger(Value: LongInt);
  procedure SetAsString(Value: String);
  procedure SetAsFloat(Value: Double);
  procedure SetAsBoolean(Value: Boolean);
public
// Class constructor
  constructor Create;
// Class destructor
  destructor Destroy; override;

// Values handle
  property Handle: PChar read FHandle write FHandle;
// Pointer value
  property AsPointer: Pointer read GetAsPointer write SetAsPointer;
// Integer value
  property AsInteger: Integer read GetAsInteger write SetAsInteger;
// String value
  property AsString: String read GetAsString write SetAsString;
// Float value
  property AsFloat: Double read GetAsFloat write SetAsFloat;
// Logical value
  property AsBoolean: Boolean read GetAsBoolean write SetAsBoolean;
// Variant type
  property VarType: Integer read GetVarType;
end;

// Counter stack class
TCountStack = class
private
  FHandle: PChar;         // Stack handle

// Internal functions
  function GetError: String;
  function GetCount: LongInt;
  function GetProcessId: LongInt;
public
// Class constructor
  constructor Create;
// Class destructor
  destructor Destroy; override;

// Push value to stack
  procedure Push(Value: TVarValue);
  procedure PushValue(Value: Variant);
// Pop value from stack
  procedure Pop(Value: TVarValue);
  function PopValue: Variant;

// Stack handle
  property Handle: PChar read FHandle write FHandle;
// Error message
  property Error: String read GetError;
// Values quantity in stack
  property Count: LongInt read GetCount;
// Process index
  property ProcessId: LongInt read GetProcessId;
end;

// Macro-Capella class
TMacro = class
  FStack: TCountStack;        // Counter stack
public
// Class constructor
  constructor Create;
// Class destructor
  destructor Destroy; override;

// Create new function
  procedure CreateFunc(FuncName, Code: String; ProcId: LongInt);
// Check if function exists
  function IsExistFunc(FuncName: String; ProcId: LongInt): Boolean;
// Drop function
  procedure DestroyFunc(FuncName: String; ProcId: LongInt);
// Drop all process functions
  procedure DestroyAllFuncs(ProcId: LongInt);
// Execute a code
  procedure ExecCode(Code: String);
// Execute a function
  procedure ExecFunc(FuncName: String);

// Create new variable
  procedure CreateVar(VarName: String; VarType, OptNum, Cmd: Integer;
    ProcId: LongInt);
// Check if variable exists
  function IsExistVar(VarName: String; ProcId: LongInt): Boolean;
// Drop a variable
  procedure DestroyVar(VarName: String; ProcId: LongInt);
// Drop all process variables
  procedure DestroyAllVars(ProcId: LongInt);
// Get variable value
  procedure GetVar(VarName: String; OptNum: Integer; ProcId: LongInt);
// Set new variable value
  procedure SetVar(VarName: String; OptNum: Integer; ProcId: LongInt);
// Execute object method
  procedure ExecMethod(Method: String; ProcId: LongInt);

// Counter stack
  property Stack: TCountStack read FStack write FStack;
end;

implementation

uses ZConvert, ZExtra;

{*************** Functions for convert data types ***********}

// Convert Zero string to Pascal string
function Str2Pas(Value: PChar): String;
begin
{$IFDEF RUSSIAN}
  Result := ConvKoi2Win(StrPas(Value));
{$ELSE}
  Result := StrPas(Value);
{$ENDIF}
end;

// Convert Pascal string to Zeos string
function Pas2Str(Value: String): PChar;
begin
  Result := StrAlloc(Length(Value)+1);
{$IFDEF RUSSIAN}
  StrPCopy(Result, ConvWin2Koi(Value));
{$ELSE}
  StrPCopy(Result, Value);
{$ENDIF}
end;

{*************** TVarValue implementation ****************}
// Class constructor
constructor TVarValue.Create;
var Temp: PChar;
begin
  Temp := macro_create_value(NIL);
  FHandle := Temp;
end;

// Class destructor
destructor TVarValue.Destroy;
begin
  macro_free_value(FHandle);
end;

// Get variant type of value
function TVarValue.GetVarType: Integer;
begin
  Result := macro_value_type(FHandle);
end;

// Get pointer value
function TVarValue.GetAsPointer: Pointer;
begin
  Result := macro_get_pointer(FHandle);
end;

// Get integer value
function TVarValue.GetAsInteger: LongInt;
begin
  Result := macro_get_int(FHandle);
end;

// Get string value
function TVarValue.GetAsString: String;
begin
  Result := Str2Pas(macro_get_string(FHandle));
end;

// Get float value
function TVarValue.GetAsFloat: Double;
begin
  Result := macro_get_real(FHandle);
end;

// Get logical value
function TVarValue.GetAsBoolean: Boolean;
begin
  Result := macro_get_bool(FHandle)<>0;
end;

// Set pointer value
procedure TVarValue.SetAsPointer(Value: Pointer);
begin
  macro_set_pointer(FHandle, Value);
end;

// Set integer value
procedure TVarValue.SetAsInteger(Value: LongInt);
begin
  macro_set_int(FHandle, Value);
end;

// Set string value
procedure TVarValue.SetAsString(Value: String);
var PArray: PChar;
begin
  PArray := Pas2Str(Value);
  macro_set_string(FHandle, PArray);
  StrDispose(PArray);
end;

// Set float value
procedure TVarValue.SetAsFloat(Value: Double);
begin
  macro_set_real(FHandle, Value);
end;

// Set logical value
procedure TVarValue.SetAsBoolean(Value: Boolean);
begin
  if Value then macro_set_int(FHandle, -1)
  else macro_set_int(FHandle, 0);
end;

{*************** TCountStack implementation ****************}
// Class constructor
constructor TCountStack.Create;
var Temp: PChar;
begin
  Temp := macro_create_stack(NIL);
  FHandle := Temp;
end;

// Class destructor
destructor TCountStack.Destroy;
begin
  macro_free_stack(FHandle);
end;

// Get an error message
function TCountStack.GetError: String;
begin
  Result := Str2Pas(macro_error(FHandle));
end;

// Get values quantity in stack
function TCountStack.GetCount: LongInt;
begin
  Result := macro_stack_count(FHandle);
end;

// Get process index
function TCountStack.GetProcessId: LongInt;
begin
  Result := macro_process_id(FHandle);
end;

// Push value to stack
procedure TCountStack.Push(Value: TVarValue);
begin
  if macro_stack_push(FHandle, Value.Handle)=0 then
    raise EMacroError.Create(Error);
end;

// Pop value from stack
procedure TCountStack.Pop(Value: TVarValue);
begin
  if macro_stack_pop(FHandle, Value.Handle)=0 then
    raise EMacroError.Create(Error);
end;

// Push value to stack
procedure TCountStack.PushValue(Value: Variant);
var TempVar: TVarValue;
begin
  TempVar := TVarValue.Create;
  case VarType(Value) of
    varSmallint,varInteger,varByte: TempVar.AsInteger := Value;
    varSingle,varDouble,varCurrency: TempVar.AsFloat := Value;
    varDate: TempVar.AsString := DateTimeToSQLDate(Value);
    varBoolean: TempVar.AsBoolean := Value;
    else TempVar.AsString := Value;
  end;
  Push(TempVar);
  TempVar.Free;
end;

// Pop value from stack
function TCountStack.PopValue: Variant;
var TempVar: TVarValue;
begin
  TempVar := TVarValue.Create;
  Pop(TempVar);
  case TempVar.VarType of
    vrPointer: Result := LongInt(TempVar.AsPointer);
    vrInteger: Result := TempVar.AsInteger;
    vrDouble: Result := TempVar.AsFloat;
    vrBoolean: Result := TempVar.AsBoolean;
    else Result := TempVar.AsString;
  end;
  TempVar.Free;
end;

{***************** TMacro implementation *******************}

// Class constructor
constructor TMacro.Create;
begin
  FStack := TCountStack.Create;
end;

// Class destructor
destructor TMacro.Destroy;
begin
  FStack.Free;
end;

// Create new function
procedure TMacro.CreateFunc(FuncName, Code: String; ProcId: LongInt);
var PFuncName, PCode: PChar;
begin
  if ProcId<0 then ProcId := FStack.ProcessId;
  PFuncName := Pas2Str(FuncName); PCode := Pas2Str(Code);
  macro_create_func(FStack.Handle, PFuncName, PCode, ProcId);
  StrDispose(PFuncName); StrDispose(PCode);
  if FStack.Error<>'' then
    raise EMacroError.Create(FStack.Error);
end;

// Check if function exists
function TMacro.IsExistFunc(FuncName: String; ProcId: LongInt): Boolean;
var PFuncName: PChar;
begin
  if ProcId<0 then ProcId := FStack.ProcessId;
  PFuncName := Pas2Str(FuncName);
  Result := macro_find_func(FStack.Handle, PFuncName, ProcId)<>0;
  StrDispose(PFuncName);
end;

// Drop function
procedure TMacro.DestroyFunc(FuncName: String; ProcId: LongInt);
var PFuncName: PChar;
begin
  if ProcId<0 then ProcId := FStack.ProcessId;
  PFuncName := Pas2Str(FuncName);
  macro_drop_func(FStack.Handle, PFuncName, ProcId);
  StrDispose(PFuncName);
end;

// Drop all process functions
procedure TMacro.DestroyAllFuncs(ProcId: LongInt);
begin
  if ProcId<0 then ProcId := FStack.ProcessId;
  macro_drop_all_funcs(FStack.Handle, ProcId);
end;

// Execute a code
procedure TMacro.ExecCode(Code: String);
var PCode: PChar;
begin
  PCode := Pas2Str(Code);
  macro_exec_code(FStack.Handle, PCode);
  StrDispose(PCode);
  if FStack.Error<>'' then
    raise EMacroError.Create(FStack.Error);
end;

// Execute a function
procedure TMacro.ExecFunc(FuncName: String);
var PFuncName: PChar;
begin
  PFuncName := Pas2Str(FuncName);
  macro_exec_func(FStack.Handle, PFuncName);
  StrDispose(PFuncName);
  if FStack.Error<>'' then
    raise EMacroError.Create(FStack.Error);
end;

// Create a new variable
procedure TMacro.CreateVar(VarName: String; VarType, OptNum, Cmd: Integer;
 ProcId: LongInt);
var PVarName: PChar;
begin
  if ProcId<0 then ProcId := FStack.ProcessId;
  PVarName := Pas2Str(VarName);
  macro_create_var(FStack.Handle, PVarName, VarType, OptNum, Cmd, ProcId);
  StrDispose(PVarName);
  if FStack.Error<>'' then
    raise EMacroError.Create(FStack.Error);
end;

// Check if variable exists
function TMacro.IsExistVar(VarName: String; ProcId: LongInt): Boolean;
var PVarName: PChar;
begin
  if ProcId<0 then ProcId := FStack.ProcessId;
  PVarName := Pas2Str(VarName);
  Result := macro_find_var(FStack.Handle, PVarName, ProcId)<>0;
  StrDispose(PVarName);
end;

// Drop variable
procedure TMacro.DestroyVar(VarName: String; ProcId: LongInt);
var PVarName: PChar;
begin
  if ProcId<0 then ProcId := FStack.ProcessId;
  PVarName := Pas2Str(VarName);
  macro_drop_var(FStack.Handle, PVarName, ProcId);
  StrDispose(PVarName);
end;

// Drop all process variables
procedure TMacro.DestroyAllVars(ProcId: LongInt);
begin
  if ProcId<0 then ProcId := FStack.ProcessId;
  macro_drop_all_vars(FStack.Handle, ProcId);
end;

// Get variable value
procedure TMacro.GetVar(VarName: String; OptNum: Integer; ProcId: LongInt);
var PVarName: PChar;
begin
  if ProcId<0 then ProcId := FStack.ProcessId;
  PVarName := Pas2Str(VarName);
  macro_get_var(FStack.Handle, PVarName, OptNum, ProcId);
  StrDispose(PVarName);
  if FStack.Error<>'' then
    raise EMacroError.Create(FStack.Error);
end;

// Set new variable value
procedure TMacro.SetVar(VarName: String; OptNum: Integer; ProcId: LongInt);
var PVarName: PChar;
begin
  if ProcId<0 then ProcId := FStack.ProcessId;
  PVarName := Pas2Str(VarName);
  macro_set_var(FStack.Handle, PVarName, OptNum, ProcId);
  StrDispose(PVarName);
  if FStack.Error<>'' then
    raise EMacroError.Create(FStack.Error);
end;

// Execute an object method
procedure TMacro.ExecMethod(Method: String; ProcId: LongInt);
var PMethod: PChar;
begin
  if ProcId<0 then ProcId := FStack.ProcessId;
  PMethod := Pas2Str(Method);
  macro_exec_method(FStack.Handle, PMethod, ProcId);
  StrDispose(PMethod);
  if FStack.Error<>'' then
    raise EMacroError.Create(FStack.Error);
end;

end.
